/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    Copyright (C) 2009-2012 Martin Brehm

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/

#include "2df.h"
#include "travis.h"


C2DF::C2DF()
{
	m_sLabelX = NULL;
	m_sLabelY = NULL;
	m_sLabelZ = NULL;
	m_pBin = NULL;
	m_pStepsY = NULL;
	m_fCountX = NULL;
	m_fCountY = NULL;
	m_iHistogramRes = 0;

	m_iPlotType = 1;
	m_iSmoothGrade = 1;
	m_iInterpolationOrder = 2;
	m_fPlotExp = 0.5;
	m_iExpLegend = 1;
}


C2DF::~C2DF()
{
	if (m_sLabelX != NULL)
	{
		delete[] m_sLabelX;
		m_sLabelX = NULL;
	}
	if (m_sLabelY != NULL)
	{
		delete[] m_sLabelY;
		m_sLabelY = NULL;
	}
	if (m_sLabelZ != NULL)
	{
		delete[] m_sLabelZ;
		m_sLabelZ = NULL;
	}
	if (m_pBin != NULL)
	{
		delete[] m_pBin;
		m_pBin = NULL;
	}
	if (m_pStepsY != NULL)
	{
		delete[] m_pStepsY;
		m_pStepsY = NULL;
	}
	if (m_fCountX != NULL)
	{
		delete[] m_fCountX;
		m_fCountX = NULL;
	}
	if (m_fCountY != NULL)
	{
		delete[] m_fCountY;
		m_fCountY = NULL;
	}
}

void C2DF::CorrectAngle(int channel)
{
	BTIN;
	int z, z2;
	double d;

	if (channel == 0)
	{
		for (z=0;z<m_iRes[0];z++)
		{
			d = cos((m_fMinVal[0]+(double)z*(m_fMaxVal[0]-m_fMinVal[0])/m_iRes[0])*Pi/180.0) - cos((m_fMinVal[0]+(double)(z+1)*(m_fMaxVal[0]-m_fMinVal[0])/m_iRes[0])*Pi/180.0);
			for (z2=0;z2<m_iRes[1];z2++)
			{
				m_pBin[z2*m_iRes[0]+z] /= d;
			}
		}
	} else
	{
		for (z2=0;z2<m_iRes[1];z2++)
		{
			d = cos((m_fMinVal[1]+(double)z2*(m_fMaxVal[1]-m_fMinVal[1])/m_iRes[1])*Pi/180.0) - cos((m_fMinVal[1]+(double)(z2+1)*(m_fMaxVal[1]-m_fMinVal[1])/m_iRes[1])*Pi/180.0);
			for (z=0;z<m_iRes[0];z++)
			{
				m_pBin[z2*m_iRes[0]+z] /= d;
			}
		}
	}
	BTOUT;
}

void C2DF::UnCorrectAngle(int channel)
{
	BTIN;
	int z, z2;

	if (channel == 0)
	{
		for (z=0;z<m_iRes[0];z++)
			for (z2=0;z2<m_iRes[1];z2++)
				m_pBin[z2*m_iRes[0]+z] *= cos((m_fMinVal[0]+(double)z*(m_fMaxVal[0]-m_fMinVal[0])/m_iRes[0])*Pi/180.0) - cos((m_fMinVal[0]+(double)(z+1)*(m_fMaxVal[0]-m_fMinVal[0])/m_iRes[0])*Pi/180.0);
	} else
	{
		for (z=0;z<m_iRes[0];z++)
			for (z2=0;z2<m_iRes[1];z2++)
				m_pBin[z2*m_iRes[0]+z] *= cos((m_fMinVal[1]+(double)z2*(m_fMaxVal[1]-m_fMinVal[1])/m_iRes[1])*Pi/180.0) - cos((m_fMinVal[1]+(double)(z2+1)*(m_fMaxVal[1]-m_fMinVal[1])/m_iRes[1])*Pi/180.0);
	}
	BTOUT;
}

void C2DF::CorrectRadialDist(int channel)
{
	BTIN;
	int z, z2;
	double d;

	if (channel == 0)
	{
		for (z=0;z<m_iRes[0];z++)
		{
			d = pow(m_fMinVal[0]+(z+1.0)/m_iRes[0]*(m_fMaxVal[0]-m_fMinVal[0]),3) - pow(m_fMinVal[0]+((double)z)/m_iRes[0]*(m_fMaxVal[0]-m_fMinVal[0]),3);
			for (z2=0;z2<m_iRes[1];z2++)
			{
				m_pBin[z2*m_iRes[0]+z] /= d;
			}
		}
	} else
	{
		for (z2=0;z2<m_iRes[1];z2++)
		{
			d = pow(m_fMinVal[1]+(z2+1.0)/m_iRes[1]*(m_fMaxVal[1]-m_fMinVal[1]),3) - pow(m_fMinVal[1]+((double)z2)/m_iRes[1]*(m_fMaxVal[1]-m_fMinVal[1]),3);
			for (z=0;z<m_iRes[0];z++)
			{
				m_pBin[z2*m_iRes[0]+z] /= d;
			}
		}
	}
	BTOUT;
}

void C2DF::UnCorrectRadialDist(int channel)
{
	BTIN;
	int z, z2;

	if (channel == 0)
	{
		for (z=0;z<m_iRes[0];z++)
			for (z2=0;z2<m_iRes[1];z2++)
				m_pBin[z2*m_iRes[0]+z] *= (double)(pow(m_fMinVal[0]+(z+1.0)/m_iRes[0]*(m_fMaxVal[0]-m_fMinVal[0]),3) - pow(m_fMinVal[0]+((double)z)/m_iRes[0]*(m_fMaxVal[0]-m_fMinVal[0]),3));
	} else
	{
		for (z=0;z<m_iRes[0];z++)
			for (z2=0;z2<m_iRes[1];z2++)
				m_pBin[z2*m_iRes[0]+z] *= (double)(pow(m_fMinVal[1]+(z2+1.0)/m_iRes[1]*(m_fMaxVal[1]-m_fMinVal[1]),3) - pow(m_fMinVal[1]+((double)z2)/m_iRes[1]*(m_fMaxVal[1]-m_fMinVal[1]),3));
	}
	BTOUT;
}

void C2DF::AddToBin(double x, double y)
{
	BXIN;
	double rx, ry;
	int ix, iy;

	if ((x < m_fMinVal[0]) || (y < m_fMinVal[1]) || (x > m_fMaxVal[0]) || (y > m_fMaxVal[1]))
	{
		m_fSkipEntries++;
		BXOUT;
		return;
	}
	m_fBinEntries++;

	rx = (x-m_fMinVal[0])*m_fFac[0] - 0.5;
	ix = (int)floor(rx);
	if (ix < 0)
	{
		ix = 0;
		rx = 0;
	} else if (ix > m_iRes[0]-2)
	{
		ix = m_iRes[0]-2;
		rx = 1.0;
	} else
		rx -= ix;

	ry = (y-m_fMinVal[1])*m_fFac[1] - 0.5;
	iy = (int)floor(ry);
	if (iy < 0)
	{
		iy = 0;
		ry = 0;
	} else if (iy > m_iRes[1]-2)
	{
		iy = m_iRes[1]-2;
		ry = 1.0;
	} else
		ry -= iy;

	m_pBin[ iy    * m_iRes[0] + ix    ] += (1.0-rx) * (1.0-ry);
	m_pBin[ iy    * m_iRes[0] + ix + 1] +=      rx  * (1.0-ry);
	m_pBin[(iy+1) * m_iRes[0] + ix    ] += (1.0-rx) *      ry;
	m_pBin[(iy+1) * m_iRes[0] + ix + 1] +=      rx  *      ry;

	m_fCountX[ix  ] += (1.0-rx);
	m_fCountX[ix+1] +=      rx;
	m_fCountY[iy  ] += (1.0-ry);
	m_fCountY[iy+1] +=      ry;
	BXOUT;
}

void C2DF::AddToBin(double x, double y, double val)
{ 
	BXIN;
	double rx, ry;
	int ix, iy;

	if ((x < m_fMinVal[0]) || (y < m_fMinVal[1]) || (x > m_fMaxVal[0]) || (y > m_fMaxVal[1]))
	{
		m_fSkipEntries++;
		BXOUT;
		return;
	}
	m_fBinEntries++;

	rx = (x-m_fMinVal[0])*m_fFac[0] - 0.5;
	ix = (int)floor(rx);
	if (ix < 0)
	{
		ix = 0;
		rx = 0;
	} else if (ix > m_iRes[0]-2)
	{
		ix = m_iRes[0]-2;
		rx = 1.0;
	} else
		rx -= ix;

	ry = (y-m_fMinVal[1])*m_fFac[1] - 0.5;
	iy = (int)floor(ry);
	if (iy < 0)
	{
		iy = 0;
		ry = 0;
	} else if (iy > m_iRes[1]-2)
	{
		iy = m_iRes[1]-2;
		ry = 1.0;
	} else
		ry -= iy;

	m_pBin[ iy    * m_iRes[0] + ix    ] += (1.0-rx) * (1.0-ry) * val;
	m_pBin[ iy    * m_iRes[0] + ix + 1] +=      rx  * (1.0-ry) * val;
	m_pBin[(iy+1) * m_iRes[0] + ix    ] += (1.0-rx) *      ry  * val;
	m_pBin[(iy+1) * m_iRes[0] + ix + 1] +=      rx  *      ry  * val;

	m_fCountX[ix  ] += (1.0-rx);
	m_fCountX[ix+1] +=      rx;
	m_fCountY[iy  ] += (1.0-ry);
	m_fCountY[iy+1] +=      ry;
	BXOUT;
}


void C2DF::AddToBin_IntX(int x, double y, double val)
{ 
	BXIN;
	double ry;
	int iy;

	if ((y < m_fMinVal[1]) || (y > m_fMaxVal[1]))
	{
		m_fSkipEntries++;
		BXOUT;
		return;
	}
	m_fBinEntries++;

	ry = (y-m_fMinVal[1])*m_fFac[1] - 0.5;
	iy = (int)floor(ry);
	if (iy < 0)
	{
		iy = 0;
		ry = 0;
	} else if (iy > m_iRes[1]-2)
	{
		iy = m_iRes[1]-2;
		ry = 1.0;
	} else
		ry -= iy;

	m_pBin[ iy    * m_iRes[0] + x] += (1.0-ry) * val;
	m_pBin[(iy+1) * m_iRes[0] + x] +=      ry  * val;

	m_fCountX[x]++;
	m_fCountY[iy  ] += (1.0-ry);
	m_fCountY[iy+1] +=      ry;
	BXOUT;
}


void C2DF::AddToBin_IntX_fast(int x, double y)
{ 
	double ry;
	int iy;

	if (y > m_fMaxVal[1])
		return;

	ry = (y-m_fMinVal[1])*m_fFac[1] - 0.5;
	iy = (int)floor(ry);
	if (iy < 0)
	{
		iy = 0;
		ry = 0;
	} else if (iy > m_iRes[1]-2)
	{
		iy = m_iRes[1]-2;
		ry = 1.0;
	} else
		ry -= iy;

	m_pBin[ iy    * m_iRes[0] + x] += (1.0-ry);
	m_pBin[(iy+1) * m_iRes[0] + x] +=      ry ;
}


/*void C2DF::AddToSingleBin(double x, double y, double val)
{ 
	BXIN;
	double rx, ry;
	int ix, iy;

	if ((x < m_fMinVal[0]) || (y < m_fMinVal[1]) || (x > m_fMaxVal[0]) || (y > m_fMaxVal[1]))
	{
		m_fSkipEntries++;
		BXOUT;
		return;
	}
	m_fBinEntries++;
	rx = ((x-m_fMinVal[0])/(m_fMaxVal[0]-m_fMinVal[0]))*((double)m_iRes[0]-1);
	ry = ((y-m_fMinVal[1])/(m_fMaxVal[1]-m_fMinVal[1]))*((double)m_iRes[1]-1);
	ix = (int)(rx+0.5);
	iy = (int)(ry+0.5);
	m_pBin[iy*(m_iRes[0]+1) + ix] += val;
	BXOUT;
}*/

//DEL void C2DF::CorrectRadialDist(bool xdim)
//DEL {
//DEL 	BTIN;
//DEL 	int x, y;
//DEL 	double f;
//DEL 
//DEL 	if (xdim)
//DEL 	{
//DEL 		for (x=0;x<m_iRes[0];x++)
//DEL 		{
//DEL 			f = (double)(1.0f/(pow(m_fMinVal[0]+(x+1.0f)/m_iRes[0]*(m_fMaxVal[0]-m_fMinVal[0]),3)-pow(m_fMinVal[0]+((double)x)/m_iRes[0]*(m_fMaxVal[0]-m_fMinVal[0]),3)));
//DEL 			for (y=0;y<m_iRes[1];y++)
//DEL 				m_pBin[x+y*(m_iRes[0]+1)] *= f;
//DEL 		}
//DEL 	} else
//DEL 	{
//DEL 		for (y=0;y<m_iRes[1];y++)
//DEL 		{
//DEL 			f = (double)(1.0f/(pow(m_fMinVal[1]+(y+1.0f)/m_iRes[1]*(m_fMaxVal[1]-m_fMinVal[1]),3)-pow(m_fMinVal[1]+((double)y)/m_iRes[1]*(m_fMaxVal[1]-m_fMinVal[1]),3)));
//DEL 			for (x=0;x<m_iRes[0];y++)
//DEL 				m_pBin[x+y*(m_iRes[0]+1)] *= f;
//DEL 		}
//DEL 	}
//DEL 	BTOUT;
//DEL }

void C2DF::MultiplyBin(double m)
{
	BTIN;
	int z;

	for (z=0;z<m_iRes[0]*m_iRes[1];z++)
		m_pBin[z] *= m;
	BTOUT;
}

double C2DF::PPMBin()
{
	BTIN;
	double f;
	int z;

	f = 0;
	for (z=0;z<m_iRes[0]*m_iRes[1];z++)
	{
		m_pBin[z] *= 1000000.0f / (double)g_iSteps / ((CMolecule*)g_oaMolecules[g_iFixMol])->m_laSingleMolIndex.GetSize();
		if (f < m_pBin[z])
			f = m_pBin[z];
	}
	BTOUT;
	return f;
}

void C2DF::NormalizeBin(double mi, double ma)
{
	BTIN;
	int z;
	double tmi, tma, d, td;

	tmi = 99999999.0f;
	tma = 0.0f;
	for (z=0;z<m_iRes[0]*m_iRes[1];z++)
	{
		if (m_pBin[z] < tmi)
			tmi = m_pBin[z];
		if (m_pBin[z] > tma)
			tma = m_pBin[z];
	}
	if (tma-tmi < 1E-20f)
		tma += 0.00001f;
	d = ma - mi;
	td = tma - tmi;
	for (z=0;z<m_iRes[0]*m_iRes[1];z++)
		m_pBin[z] = ((m_pBin[z]-tmi)/td*d)+mi;
	BTOUT;
}

double C2DF::NormalizeBinIntegral(double val)
{
	BTIN;
	int z;
	double d;

	d = 0;
	for (z=0;z<m_iRes[0]*m_iRes[1];z++)
		d += m_pBin[z];
	if (d == 0)
	{
		BTOUT;
		return 0.0;
	}
	d = val/d;
	for (z=0;z<m_iRes[0]*m_iRes[1];z++)
		m_pBin[z]*=d;
	return d;
	BTOUT;
}

void C2DF::Create()
{
	BTIN;
	int z;
	m_fBinEntries = 0;
	m_fSkipEntries = 0;

	try { m_pStepsY = new unsigned long[m_iRes[1]]; } catch(...) { m_pStepsY = NULL; }
	if (m_pStepsY == NULL) NewException((double)m_iRes[1]*sizeof(unsigned long),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	try { m_fCountX = new double[m_iRes[0]]; } catch(...) { m_fCountX = NULL; }
	if (m_fCountX == NULL) NewException((double)m_iRes[0]*sizeof(double),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	try { m_fCountY = new double[m_iRes[1]]; } catch(...) { m_fCountY = NULL; }
	if (m_fCountY == NULL) NewException((double)m_iRes[1]*sizeof(double),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<m_iRes[0];z++)
		m_fCountX[z] = 0;

	for (z=0;z<m_iRes[1];z++)
	{
		m_pStepsY[z] = 0;
		m_fCountY[z] = 0;
	}

	try { m_pBin = new double[m_iRes[0]*m_iRes[1]]; } catch(...) { m_fCountY = NULL; }
	if (m_fCountY == NULL) NewException((double)m_iRes[0]*m_iRes[1]*sizeof(double),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<m_iRes[0]*m_iRes[1];z++)
		m_pBin[z] = 0;
	m_fFac[0] = (double)m_iRes[0] / (m_fMaxVal[0]-m_fMinVal[0]);
	m_fFac[1] = (double)m_iRes[1] / (m_fMaxVal[1]-m_fMinVal[1]);
	BTOUT;
}

void C2DF::Write(const char *prefix, const char *name, const char *suffix)
{
	BTIN;
	FILE *a;
	int z, z2;
	char buf[256];
	
	strcpy(buf,prefix);
	strcat(buf,name);
	strcat(buf,suffix);

	a = OpenFileWrite(buf,true);
	if (m_sLabelX == NULL)
		fprintf(a,"# (no label); ");
			else fprintf(a,"# %s; ",m_sLabelX);
	if (m_sLabelY == NULL)
		fprintf(a,"(no label); ");
			else fprintf(a,"%s; ",m_sLabelY);
	if (m_sLabelZ == NULL)
		fprintf(a,"Occurrence\n");
			else fprintf(a,"%s\n",m_sLabelZ);
	for (z=0;z<m_iRes[1];z++)
		for (z2=0;z2<m_iRes[0];z2++)
			fprintf(a,"%#.10G;  %#.10G;  %#.10G\n",m_fMinVal[0]+(z2+0.5)*(m_fMaxVal[0]-m_fMinVal[0])/m_iRes[0],m_fMinVal[1]+(z+0.5)*(m_fMaxVal[1]-m_fMinVal[1])/m_iRes[1],m_pBin[z*m_iRes[0]+z2]);
	fclose(a);
	
	BTOUT;
}

void C2DF::WriteGraceBunch(int channel, int graphs, float fac, const char *prefix, const char *name, const char *suffix)
{
	BTIN;
	int x, y, t;
	char buf[256];
	CGrace *g;

	try { g = new CGrace(); } catch(...) { g = NULL; }
	if (g == NULL) NewException((double)sizeof(CGrace),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	g->CurrentGraph()->m_bLegend = true;

	if (channel == 0)
	{
		g->SetLabelX(m_sLabelY);
		if (m_sLabelZ == NULL)
			g->SetLabelY("Occurrence");
				else g->SetLabelY(m_sLabelZ);
		CalcMaxEntry();
		g->SetRangeX(m_fMinVal[1],m_fMaxVal[1]);
		g->SetRangeY(0,m_fMaxEntry*1.1*fac);
		g->MakeTicks();
		sprintf(buf,"Legend unit is %s",m_sLabelX);
		g->SetSubTitle(buf);

		for (x=0;x<graphs;x++)
		{
			t = (int)((double)x / (graphs-1) * m_iRes[0]);
			g->AddDataset();
			sprintf(buf,"%.2f",m_fMinVal[0]+t*(m_fMaxVal[0]-m_fMinVal[0])/m_iRes[0]);
			g->SetDatasetName(buf);
			g->CurrentGraph()->CurrentDataset()->m_faValues.SetMaxSize(m_iRes[1]*2);
			for (y=0;y<m_iRes[1]-1;y++)
				g->AddXYTupel(m_fMinVal[1]+(y+0.5)*(m_fMaxVal[1]-m_fMinVal[1])/m_iRes[1],m_pBin[y*m_iRes[0]+t]*fac);
		}
	} else
	{
		g->SetLabelX(m_sLabelX);
		if (m_sLabelZ == NULL)
			g->SetLabelY("Occurrence");
				else g->SetLabelY(m_sLabelZ);
		CalcMaxEntry();
		g->SetRangeX(m_fMinVal[0],m_fMaxVal[0]);
		g->SetRangeY(0,m_fMaxEntry*1.1*fac);
		g->MakeTicks();
		sprintf(buf,"Legend unit is %s",m_sLabelY);
		g->SetSubTitle(buf);

		for (y=0;y<graphs;y++)
		{
			t = (int)((double)y / (graphs-1) * m_iRes[1]);
			g->AddDataset();
			sprintf(buf,"%.2f",m_fMinVal[1]+t*(m_fMaxVal[1]-m_fMinVal[1])/m_iRes[1]);
			g->SetDatasetName(buf);
			g->CurrentGraph()->CurrentDataset()->m_faValues.SetMaxSize(m_iRes[0]*2);
			for (x=0;x<m_iRes[0]-1;x++)
				g->AddXYTupel(m_fMinVal[0]+(x+0.5)*(m_fMaxVal[0]-m_fMinVal[0])/m_iRes[0],m_pBin[t*m_iRes[0]+x]*fac);
		}
	}
	strcpy(buf,prefix);
	strcat(buf,name);
	strcat(buf,suffix);
	g->WriteAgr(buf,false);
	*strrchr(buf,'.') = 0;
	strcat(buf,".csv");
	g->WriteCSV(buf);

	BTOUT;
}

void C2DF::WriteCSV(const char *prefix, const char *name, const char *suffix)
{
	BTIN;
	FILE *a;
	int z, z2;
	char buf[256];
	
	strcpy(buf,prefix);
	strcat(buf,name);
	strcat(buf,suffix);

	a = OpenFileWrite(buf,true);
	if (m_sLabelX == NULL)
		fprintf(a,"(no label) \\ ");
			else fprintf(a,"%s \\ ",m_sLabelX);
	if (m_sLabelY == NULL)
		fprintf(a,"(no label)");
			else fprintf(a,"%s",m_sLabelY);
	for (z=0;z<m_iRes[1];z++)
			fprintf(a,"; %#.10G",m_fMinVal[1]+(z+0.5)*(m_fMaxVal[1]-m_fMinVal[1])/m_iRes[1]);
	fprintf(a,"\n");
	for (z2=0;z2<m_iRes[0];z2++)
	{
		fprintf(a,"%#.10G",m_fMinVal[0]+(z2+0.5)*(m_fMaxVal[0]-m_fMinVal[0])/m_iRes[0]);
		for (z=0;z<m_iRes[1];z++)
			fprintf(a,"; %#.10G",m_pBin[z*m_iRes[0]+z2]);
		fprintf(a,"\n");
	}
	fclose(a);
	BTOUT;
}

void C2DF::WriteXProjection(const char *prefix, const char *name, const char *suffix)
{
	BTIN;
	FILE *a;
	int z, z2;
	char buf[256];
	double d;
	double *tb;

	try { tb = new double[m_iRes[0]]; } catch(...) { tb = NULL; }
	if (tb == NULL) NewException((double)m_iRes[0]*sizeof(double),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<m_iRes[0];z++)
	{
		d = 0;
		for (z2=0;z2<m_iRes[1];z2++)
			d += m_pBin[z2*m_iRes[0]+z];
		d /= m_iRes[1];
		tb[z] = d;
	}
	
	strcpy(buf,prefix);
	strcat(buf,name);
	strcat(buf,suffix);

	a = OpenFileWrite(buf,true);
	for (z=0;z<m_iRes[0];z++)
		fprintf(a,"%f;  %f\n",m_fMinVal[0]+(z+0.5)*(m_fMaxVal[0]-m_fMinVal[0])/m_iRes[0],tb[z]);
	fclose(a);
	delete[] tb;
	BTOUT;
}

void C2DF::WriteYProjection(const char *prefix, const char *name, const char *suffix)
{
	BTIN;
	FILE *a;
	int z, z2;
	char buf[256];
	double d;
	double *tb;

	try { tb = new double[m_iRes[1]]; } catch(...) { tb = NULL; }
	if (tb == NULL) NewException((double)m_iRes[1]*sizeof(double),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<m_iRes[1];z++)
	{
		d = 0;
		for (z2=0;z2<m_iRes[0];z2++)
			d += m_pBin[z*m_iRes[0]+z2];
		d /= m_iRes[1];
		tb[z] = d;
	}
	
	strcpy(buf,prefix);
	strcat(buf,name);
	strcat(buf,suffix);

	a = OpenFileWrite(buf,true);
	for (z=0;z<m_iRes[1];z++)
		fprintf(a,"%f;  %f\n",m_fMinVal[1]+(z+0.5)*(m_fMaxVal[1]-m_fMinVal[1])/m_iRes[1],tb[z]);
	fclose(a);
	delete[] tb;
	BTOUT;
}

void C2DF::WriteMathematica(const char *prefix, const char *name, const char *suffix)
{
	BTIN;
	FILE *a;
	int z, z2, i;
	char buf[256];
	
	strcpy(buf,prefix);
	strcat(buf,name);
	strcat(buf,suffix);

	a = OpenFileWrite(buf,true);
	fprintf(a,"{ ");
	i = 0;
	for (z=0;z<m_iRes[1];z++)
		for (z2=0;z2<m_iRes[0];z2++)
		{
			fprintf(a,"{ %#.10f,  %#.10f,  %#.10f } ",m_fMinVal[0]+(z2+0.5)*(m_fMaxVal[0]-m_fMinVal[0])/m_iRes[0],m_fMinVal[1]+(z+0.5)*(m_fMaxVal[1]-m_fMinVal[1])/m_iRes[1],m_pBin[z*m_iRes[0]+z2]);
			i++;
			if ((z2+1 < m_iRes[0]) || (z+1 < m_iRes[1]))
				fprintf(a,", ");
			if (i % 5 == 0)
				fprintf(a,"\n");
		}
	fprintf(a," }\n");
	fclose(a);
	BTOUT;
}

void C2DF::WriteMathematicaNb(const char *prefix, const char *name, const char *suffix, bool manrange)
{
	BTIN;
	FILE *a;
	int z, z2;
	char buf[256];
	int minorx, majorx, minory, majory;
	
	strcpy(buf,prefix);
	strcat(buf,name);
	strcat(buf,suffix);

	if (!manrange)
		CalcMaxEntry();

	CreateTicks(m_fMinVal[0],m_fMaxVal[0],majorx,minorx);
	CreateTicks(m_fMinVal[1],m_fMaxVal[1],majory,minory);

	a = OpenFileWrite(buf,true);

	fprintf(a,"\n");
	fprintf(a,"Notebook[{\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," StyleBox[\n");
	fprintf(a,"  RowBox[{\"TRAVIS\", \" \", \"Analysis\"}], \"Section\",\n");
	fprintf(a,"  Evaluatable->False]], \"Input\",\n");
	fprintf(a," Evaluatable->False],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"Cell[\"Input Data\", \"Section\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = False}, \n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell], \n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]], \n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"dat\", \"=\",\n");
	fprintf(a,"   RowBox[{\"dat2\", \"=\", \n");
	fprintf(a,"    RowBox[{\"{\", \n");
	fprintf(a,"     RowBox[{\n");

	for (z=0;z<m_iRes[1];z++)
		for (z2=0;z2<m_iRes[0];z2++)
		{
			fprintf(a,"     RowBox[{\"{\", \n");
			fprintf(a,"      RowBox[{\"%#.10f\", \",\", \"%#.10f\", \",\", \"%#.10f\"}], \"}\"}]",m_fMinVal[0]+(z2+0.5)*(m_fMaxVal[0]-m_fMinVal[0])/m_iRes[0],m_fMinVal[1]+(z+0.5)*(m_fMaxVal[1]-m_fMinVal[1])/m_iRes[1],m_pBin[z*m_iRes[0]+z2]);
			if ((z < m_iRes[1]-1) || (z2 < m_iRes[0]-1))
				fprintf(a,", \",\", \n");
		}
	
	fprintf(a,"}], \"}\"}]}]}], \n");
	fprintf(a,"  \";\"}]], \"Input\"]\n");
	fprintf(a,"}, Closed]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"\n");
	
	fprintf(a,"Cell[\"Internal Routines\", \"Section\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = False}, \n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell], \n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]], \n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[{\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"prim\", \"=\", \n");
	fprintf(a,"   RowBox[{\"{\", \"}\"}]}], \";\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"funcX\", \"=\", \n");
	fprintf(a,"   RowBox[{\"{\", \"}\"}]}], \";\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"funcY\", \"=\", \n");
	fprintf(a,"   RowBox[{\"{\", \"}\"}]}], \";\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"x\", \"=.\"}], \";\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"s\", \"=.\"}], \";\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\n");

	fprintf(a,"   RowBox[{\"maxsig\", \"[\",\n");
	fprintf(a,"    RowBox[{\"x_\", \",\", \"s_\"}], \"]\"}], \"=\",\n");
	fprintf(a,"   RowBox[{\"N\", \"[\",\n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"     RowBox[{\"If\", \"[\", \n");
	fprintf(a,"      RowBox[{\n");
	fprintf(a,"       RowBox[{\"x\", \"\\[Equal]\", \"0\"}], \",\", \"0\", \",\", \n");
	fprintf(a,"       RowBox[{\n");
	fprintf(a,"        RowBox[{\"Ceiling\", \"[\", \n");
	fprintf(a,"         RowBox[{\"x\", \"/\", \n");
	fprintf(a,"          RowBox[{\"10\", \"^\", \n");
	fprintf(a,"           RowBox[{\"Floor\", \"[\", \n");
	fprintf(a,"            RowBox[{\"N\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"Log\", \"[\", \n");
	fprintf(a,"               RowBox[{\"10\", \",\", \n");
	fprintf(a,"                RowBox[{\"Abs\", \"[\", \"x\", \"]\"}]}], \"]\"}], \"+\", \"1\", \"-\", \"s\"}],\n");
	fprintf(a,"              \"]\"}], \"]\"}]}]}], \"]\"}], \"*\", \n");
	fprintf(a,"        RowBox[{\"10\", \"^\", \n");
	fprintf(a,"         RowBox[{\"Floor\", \"[\", \n");
	fprintf(a,"          RowBox[{\"N\", \"[\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"Log\", \"[\", \n");
	fprintf(a,"             RowBox[{\"10\", \",\", \n");
	fprintf(a,"              RowBox[{\"Abs\", \"[\", \"x\", \"]\"}]}], \"]\"}], \"+\", \"1\", \"-\", \"s\"}], \n");
	fprintf(a,"           \"]\"}], \"]\"}]}]}]}], \"]\"}], \"+\", \n");
	fprintf(a,"     RowBox[{\n");
	fprintf(a,"      RowBox[{\"Abs\", \"[\", \"x\", \"]\"}], \"/\", \"1000000\"}]}], \"]\"}]}], \n");
	fprintf(a,"  \";\"}], \"\\n\", \n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"minsig\", \"[\", \n");
	fprintf(a,"    RowBox[{\"x_\", \",\", \"s_\"}], \"]\"}], \"=\", \n");
	fprintf(a,"   RowBox[{\"N\", \"[\", \n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"     RowBox[{\"If\", \"[\", \n");
	fprintf(a,"      RowBox[{\n");
	fprintf(a,"       RowBox[{\"x\", \"\\[Equal]\", \"0\"}], \",\", \"0\", \",\", \n");
	fprintf(a,"       RowBox[{\n");
	fprintf(a,"        RowBox[{\"Floor\", \"[\", \n");
	fprintf(a,"         RowBox[{\"x\", \"/\", \n");
	fprintf(a,"          RowBox[{\"10\", \"^\", \n");
	fprintf(a,"           RowBox[{\"Floor\", \"[\", \n");
	fprintf(a,"            RowBox[{\"N\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"Log\", \"[\", \n");
	fprintf(a,"               RowBox[{\"10\", \",\", \n");
	fprintf(a,"                RowBox[{\"Abs\", \"[\", \"x\", \"]\"}]}], \"]\"}], \"+\", \"1\", \"-\", \"s\"}],\n");
	fprintf(a,"              \"]\"}], \"]\"}]}]}], \"]\"}], \"*\", \n");
	fprintf(a,"        RowBox[{\"10\", \"^\", \n");
	fprintf(a,"         RowBox[{\"Floor\", \"[\", \n");
	fprintf(a,"          RowBox[{\"N\", \"[\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"Log\", \"[\", \n");
	fprintf(a,"             RowBox[{\"10\", \",\", \n");
	fprintf(a,"              RowBox[{\"Abs\", \"[\", \"x\", \"]\"}]}], \"]\"}], \"+\", \"1\", \"-\", \"s\"}], \n");
	fprintf(a,"           \"]\"}], \"]\"}]}]}]}], \"]\"}], \"-\", \n");
	fprintf(a,"     RowBox[{\n");
	fprintf(a,"      RowBox[{\"Abs\", \"[\", \"x\", \"]\"}], \"/\", \"1000000\"}]}], \"]\"}]}], \n");
	fprintf(a,"  \";\"}], \"\\n\", \n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"prec\", \"[\", \"x_\", \"]\"}], \"=\", \n");
	fprintf(a,"   RowBox[{\"Max\", \"[\", \n");
	fprintf(a,"    RowBox[{\"0\", \",\", \n");
	fprintf(a,"     RowBox[{\n");
	fprintf(a,"      RowBox[{\"Floor\", \"[\", \n");
	fprintf(a,"       RowBox[{\"N\", \"[\", \n");
	fprintf(a,"        RowBox[{\"Log\", \"[\", \n");
	fprintf(a,"         RowBox[{\"10\", \",\", \n");
	fprintf(a,"          RowBox[{\"1\", \"/\", \n");
	fprintf(a,"           RowBox[{\"Abs\", \"[\", \"x\", \"]\"}]}]}], \"]\"}], \"]\"}], \"]\"}], \"+\", \n");
	fprintf(a,"      \"2\"}]}], \"]\"}]}], \";\"}], \"\\[IndentingNewLine]\", \n");


/*	fprintf(a,"   RowBox[{\"maxsig\", \"[\", \n");
	fprintf(a,"    RowBox[{\"x_\", \",\", \"s_\"}], \"]\"}], \"=\", \n");
	fprintf(a,"   RowBox[{\"N\", \"[\", \n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"     RowBox[{\"If\", \"[\", \n");
	fprintf(a,"      RowBox[{\n");
	fprintf(a,"       RowBox[{\"x\", \"\\[Equal]\", \"0\"}], \",\", \"0\", \",\", \n");
	fprintf(a,"       RowBox[{\n");
	fprintf(a,"        RowBox[{\"Ceiling\", \"[\", \n");
	fprintf(a,"         RowBox[{\"x\", \"/\", \n");
	fprintf(a,"          RowBox[{\"10\", \"^\", \n");
	fprintf(a,"           RowBox[{\"Floor\", \"[\", \n");
	fprintf(a,"            RowBox[{\n");
	fprintf(a,"             RowBox[{\"Log\", \"[\", \n");
	fprintf(a,"              RowBox[{\"10\", \",\", \n");
	fprintf(a,"               RowBox[{\"Abs\", \"[\", \"x\", \"]\"}]}], \"]\"}], \"+\", \"1\", \"-\", \"s\"}], \n");
	fprintf(a,"            \"]\"}]}]}], \"]\"}], \"*\", \n");
	fprintf(a,"        RowBox[{\"10\", \"^\", \n");
	fprintf(a,"         RowBox[{\"Floor\", \"[\", \n");
	fprintf(a,"          RowBox[{\n");
	fprintf(a,"           RowBox[{\"Log\", \"[\", \n");
	fprintf(a,"            RowBox[{\"10\", \",\", \n");
	fprintf(a,"             RowBox[{\"Abs\", \"[\", \"x\", \"]\"}]}], \"]\"}], \"+\", \"1\", \"-\", \"s\"}], \n");
	fprintf(a,"          \"]\"}]}]}]}], \"]\"}], \"+\", \n");
	fprintf(a,"     RowBox[{\n");
	fprintf(a,"      RowBox[{\"Abs\", \"[\", \"x\", \"]\"}], \"/\", \"1000000\"}]}], \"]\"}]}], \n");
	fprintf(a,"  \";\"}], \"\\n\", \n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"minsig\", \"[\", \n");
	fprintf(a,"    RowBox[{\"x_\", \",\", \"s_\"}], \"]\"}], \"=\", \n");
	fprintf(a,"   RowBox[{\"N\", \"[\", \n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"     RowBox[{\"If\", \"[\", \n");
	fprintf(a,"      RowBox[{\n");
	fprintf(a,"       RowBox[{\"x\", \"\\[Equal]\", \"0\"}], \",\", \"0\", \",\", \n");
	fprintf(a,"       RowBox[{\n");
	fprintf(a,"        RowBox[{\"Floor\", \"[\", \n");
	fprintf(a,"         RowBox[{\"x\", \"/\", \n");
	fprintf(a,"          RowBox[{\"10\", \"^\", \n");
	fprintf(a,"           RowBox[{\"Floor\", \"[\", \n");
	fprintf(a,"            RowBox[{\n");
	fprintf(a,"             RowBox[{\"Log\", \"[\", \n");
	fprintf(a,"              RowBox[{\"10\", \",\", \n");
	fprintf(a,"               RowBox[{\"Abs\", \"[\", \"x\", \"]\"}]}], \"]\"}], \"+\", \"1\", \"-\", \"s\"}], \n");
	fprintf(a,"            \"]\"}]}]}], \"]\"}], \"*\", \n");
	fprintf(a,"        RowBox[{\"10\", \"^\", \n");
	fprintf(a,"         RowBox[{\"Floor\", \"[\", \n");
	fprintf(a,"          RowBox[{\n");
	fprintf(a,"           RowBox[{\"Log\", \"[\", \n");
	fprintf(a,"            RowBox[{\"10\", \",\", \n");
	fprintf(a,"             RowBox[{\"Abs\", \"[\", \"x\", \"]\"}]}], \"]\"}], \"+\", \"1\", \"-\", \"s\"}], \n");
	fprintf(a,"          \"]\"}]}]}]}], \"]\"}], \"-\", \n");
	fprintf(a,"     RowBox[{\n");
	fprintf(a,"      RowBox[{\"Abs\", \"[\", \"x\", \"]\"}], \"/\", \"1000000\"}]}], \"]\"}]}], \n");
	fprintf(a,"  \";\"}], \"\\n\", \n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"prec\", \"[\", \"x_\", \"]\"}], \"=\", \n");
	fprintf(a,"   RowBox[{\"Max\", \"[\", \n");
	fprintf(a,"    RowBox[{\"0\", \",\", \n");
	fprintf(a,"     RowBox[{\n");
	fprintf(a,"      RowBox[{\"Floor\", \"[\", \n");
	fprintf(a,"       RowBox[{\"Log\", \"[\", \n");
	fprintf(a,"        RowBox[{\"10\", \",\", \n");
	fprintf(a,"         RowBox[{\"1\", \"/\", \n");
	fprintf(a,"          RowBox[{\"Abs\", \"[\", \"x\", \"]\"}]}]}], \"]\"}], \"]\"}], \"+\", \"2\"}]}], \n");
	fprintf(a,"    \"]\"}]}], \";\"}], \"\\[IndentingNewLine]\", \n");*/


	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"plot\", \":=\", \n");
	fprintf(a,"    RowBox[{\"(\", \n");
	fprintf(a,"     RowBox[{\n");
	fprintf(a,"      RowBox[{\"x\", \"=.\"}], \";\", \n");
	fprintf(a,"      RowBox[{\"y\", \"=.\"}], \";\", \"\\[IndentingNewLine]\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"(*\", \" \", \n");
	fprintf(a,"       RowBox[{\"Select\", \" \", \"Color\", \" \", \"Function\"}], \" \", \"*)\"}], \n");
	fprintf(a,"      \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"If\", \"[\", \n");
	fprintf(a,"       RowBox[{\n");
	fprintf(a,"        RowBox[{\"coloring\", \"\\[Equal]\", \"1\"}], \",\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\"ColFunc\", \"[\", \"x_\", \"]\"}], \"=\", \n");
	fprintf(a,"         RowBox[{\"If\", \"[\", \n");
	fprintf(a,"          RowBox[{\n");
	fprintf(a,"           RowBox[{\"x\", \"\\[LessEqual]\", \"0\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"            RowBox[{\"1\", \",\", \"1\", \",\", \"1\"}], \"]\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"If\", \"[\", \n");
	fprintf(a,"            RowBox[{\n");
	fprintf(a,"             RowBox[{\"x\", \"<\", \n");
	fprintf(a,"              RowBox[{\"1\", \"/\", \"5\"}]}], \",\", \n");
	fprintf(a,"             RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"              RowBox[{\n");
	fprintf(a,"               RowBox[{\"1\", \"-\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\"5\", \"^\", \"1.5\"}], \"*\", \n");
	fprintf(a,"                   RowBox[{\"x\", \"^\", \"1.5\"}]}], \")\"}], \"*\", \"0.8\"}]}], \",\", \n");
	fprintf(a,"               RowBox[{\"1\", \"-\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\"5\", \"^\", \"1.5\"}], \"*\", \n");
	fprintf(a,"                   RowBox[{\"x\", \"^\", \"1.5\"}]}], \")\"}], \"*\", \"0.8\"}]}], \",\", \n");
	fprintf(a,"               \"1\"}], \"]\"}], \",\", \n");
	fprintf(a,"             RowBox[{\"If\", \"[\", \n");
	fprintf(a,"              RowBox[{\n");
	fprintf(a,"               RowBox[{\"x\", \"<\", \n");
	fprintf(a,"                RowBox[{\"2\", \"/\", \"5\"}]}], \",\", \n");
	fprintf(a,"               RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"0.2\", \"-\", \n");
	fprintf(a,"                  RowBox[{\"(\", \n");
	fprintf(a,"                   RowBox[{\"x\", \"-\", \n");
	fprintf(a,"                    RowBox[{\"1\", \"/\", \"5\"}]}], \")\"}]}], \",\", \n");
	fprintf(a,"                 RowBox[{\"0.2\", \"+\", \n");
	fprintf(a,"                  RowBox[{\"0.8\", \"*\", \n");
	fprintf(a,"                   RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"5\", \"^\", \"0.75\"}], \"*\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\"x\", \"-\", \n");
	fprintf(a,"                    RowBox[{\"1\", \"/\", \"5\"}]}], \")\"}], \"^\", \"0.75\"}]}], \n");
	fprintf(a,"                    \")\"}]}]}], \",\", \n");
	fprintf(a,"                 RowBox[{\"1\", \"-\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\"5\", \"^\", \"1.33\"}], \"*\", \n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\"x\", \"-\", \n");
	fprintf(a,"                    RowBox[{\"1\", \"/\", \"5\"}]}], \")\"}], \"^\", \"1.33\"}]}]}]}], \n");
	fprintf(a,"                \"]\"}], \",\", \n");
	fprintf(a,"               RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"x\", \"<\", \n");
	fprintf(a,"                  RowBox[{\"3\", \"/\", \"5\"}]}], \",\", \n");
	fprintf(a,"                 RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\"5\", \"^\", \"0.5\"}], \"*\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\"x\", \"-\", \n");
	fprintf(a,"                    RowBox[{\"2\", \"/\", \"5\"}]}], \")\"}], \"^\", \"0.5\"}]}], \",\", \n");
	fprintf(a,"                   \"1\", \",\", \"0\"}], \"]\"}], \",\", \n");
	fprintf(a,"                 RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\"x\", \"<\", \n");
	fprintf(a,"                    RowBox[{\"4\", \"/\", \"5\"}]}], \",\", \n");
	fprintf(a,"                   RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"1\", \",\", \n");
	fprintf(a,"                    RowBox[{\"1\", \"-\", \n");
	fprintf(a,"                    RowBox[{\"5\", \"*\", \n");
	fprintf(a,"                    RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\"x\", \"-\", \n");
	fprintf(a,"                    RowBox[{\"3\", \"/\", \"5\"}]}], \")\"}]}]}], \",\", \"0\"}], \"]\"}], \n");
	fprintf(a,"                   \",\", \n");
	fprintf(a,"                   RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"x\", \"<\", \"1\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"1\", \",\", \"0\", \",\", \n");
	fprintf(a,"                    RowBox[{\"5\", \"*\", \n");
	fprintf(a,"                    RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\"x\", \"-\", \n");
	fprintf(a,"                    RowBox[{\"4\", \"/\", \"5\"}]}], \")\"}]}]}], \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"1\", \",\", \"0\", \",\", \"1\"}], \"]\"}]}], \"]\"}]}], \n");
	fprintf(a,"                  \"]\"}]}], \"]\"}]}], \"]\"}]}], \"]\"}]}], \"]\"}]}], \",\", \n");
	fprintf(a,"        RowBox[{\"If\", \"[\", \n");
	fprintf(a,"         RowBox[{\n");

	fprintf(a,"          RowBox[{\"coloring\", \"\\[Equal]\", \"2\"}], \",\",\n");
	fprintf(a,"          RowBox[{\n");
	fprintf(a,"           RowBox[{\"ColFunc\", \"[\", \"x_\", \"]\"}], \"=\", \n");
	fprintf(a,"           RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"            RowBox[{\n");
	fprintf(a,"             RowBox[{\"1\", \"-\", \"x\"}], \",\", \n");
	fprintf(a,"             RowBox[{\"1\", \"-\", \"x\"}], \",\", \n");
	fprintf(a,"             RowBox[{\"1\", \"-\", \"x\"}]}], \"]\"}]}], \",\", \n");
	fprintf(a,"          RowBox[{\n");
	fprintf(a,"           RowBox[{\"ColFunc\", \"[\", \"x_\", \"]\"}], \"=\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"ColorData\", \"[\", \"\\\"\\<TemperatureMap\\>\\\"\", \"]\"}], \"[\", \n");

	fprintf(a,"            \"x\", \"]\"}]}]}], \"]\"}]}], \"]\"}], \";\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      \"\\[IndentingNewLine]\", \n");

	fprintf(a,"      RowBox[{\"oplotrangeX1\", \"=\", \"%f\"}], \";\", \"\\[IndentingNewLine]\", \n",m_fMinVal[0]);
	fprintf(a,"      RowBox[{\"oplotrangeY1\", \"=\", \"%f\"}], \";\", \"\\[IndentingNewLine]\", \n",m_fMinVal[1]);
	fprintf(a,"      RowBox[{\"oplotrangeX2\", \"=\", \"%f\"}], \";\", \"\\[IndentingNewLine]\", \n",m_fMaxVal[0]);
	fprintf(a,"      RowBox[{\"oplotrangeY2\", \"=\", \"%f\"}], \";\", \"\\[IndentingNewLine]\", \n",m_fMaxVal[1]);
	fprintf(a,"      RowBox[{\"oplotresX\", \"=\", \"%d\"}], \";\", \"\\[IndentingNewLine]\", \n",m_iRes[0]);
	fprintf(a,"      RowBox[{\"oplotresY\", \"=\", \"%d\"}], \";\", \"\\[IndentingNewLine]\", \n",m_iRes[1]);
	
	fprintf(a,"      \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"plotresX\", \"=\", \n");
	fprintf(a,"       RowBox[{\"Ceiling\", \"[\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"(\", \n");
	fprintf(a,"           RowBox[{\"plotrangeX2\", \"-\", \"plotrangeX1\"}], \")\"}], \"/\", \n");
	fprintf(a,"          RowBox[{\"(\", \n");
	fprintf(a,"           RowBox[{\"oplotrangeX2\", \"-\", \"oplotrangeX1\"}], \")\"}]}], \"*\", \n");
	fprintf(a,"         \"oplotresX\"}], \"]\"}]}], \";\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"plotresY\", \"=\", \n");
	fprintf(a,"       RowBox[{\"Ceiling\", \"[\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"(\", \n");
	fprintf(a,"           RowBox[{\"plotrangeY2\", \"-\", \"plotrangeY1\"}], \")\"}], \"/\", \n");
	fprintf(a,"          RowBox[{\"(\", \n");
	fprintf(a,"           RowBox[{\"oplotrangeY2\", \"-\", \"oplotrangeY1\"}], \")\"}]}], \"*\", \n");
	fprintf(a,"         \"oplotresY\"}], \"]\"}]}], \";\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"plotresIndX\", \"=\", \n");
	fprintf(a,"       RowBox[{\"Ceiling\", \"[\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"(\", \n");
	fprintf(a,"           RowBox[{\"plotrangeX1\", \"-\", \"oplotrangeX1\"}], \")\"}], \"/\", \n");
	fprintf(a,"          RowBox[{\"(\", \n");
	fprintf(a,"           RowBox[{\"oplotrangeX2\", \"-\", \"oplotrangeX1\"}], \")\"}]}], \"*\", \n");
	fprintf(a,"         \"oplotresX\"}], \"]\"}]}], \";\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"plotresIndY\", \"=\", \n");
	fprintf(a,"       RowBox[{\"Ceiling\", \"[\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"(\", \n");
	fprintf(a,"           RowBox[{\"plotrangeY1\", \"-\", \"oplotrangeY1\"}], \")\"}], \"/\", \n");
	fprintf(a,"          RowBox[{\"(\", \n");
	fprintf(a,"           RowBox[{\"oplotrangeY2\", \"-\", \"oplotrangeY1\"}], \")\"}]}], \"*\", \n");
	fprintf(a,"         \"oplotresY\"}], \"]\"}]}], \";\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"(*\", \" \", \n");
	fprintf(a,"       RowBox[{\"Smoothen\", \" \", \"Data\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"dat\", \"=\", \n");
	fprintf(a,"       RowBox[{\"Table\", \"[\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"ox\", \"=\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"Mod\", \"[\", \n");
	fprintf(a,"             RowBox[{\"i\", \",\", \"plotresX\"}], \"]\"}], \"+\", \"plotresIndX\"}]}], \n");
	fprintf(a,"          \";\", \n");
	fprintf(a,"          RowBox[{\"oy\", \"=\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"Quotient\", \"[\", \n");
	fprintf(a,"             RowBox[{\"i\", \",\", \"plotresX\"}], \"]\"}], \"+\", \"plotresIndY\"}]}], \n");
	fprintf(a,"          \";\", \n");
	fprintf(a,"          RowBox[{\"x\", \"=\", \n");
	fprintf(a,"           RowBox[{\"Mod\", \"[\", \n");
	fprintf(a,"            RowBox[{\"i\", \",\", \"plotresX\"}], \"]\"}]}], \";\", \n");
	fprintf(a,"          RowBox[{\"y\", \"=\", \n");
	fprintf(a,"           RowBox[{\"Quotient\", \"[\", \n");
	fprintf(a,"            RowBox[{\"i\", \",\", \"plotresX\"}], \"]\"}]}], \";\", \n");
	fprintf(a,"          RowBox[{\"{\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"plotrangeX1\", \"+\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"(\", \n");
	fprintf(a,"               RowBox[{\"plotrangeX2\", \"-\", \"plotrangeX1\"}], \")\"}], \"*\", \n");
	fprintf(a,"              RowBox[{\"x\", \"/\", \n");
	fprintf(a,"               RowBox[{\"(\", \n");
	fprintf(a,"                RowBox[{\"plotresX\", \"-\", \"1\"}], \")\"}]}]}]}], \",\", \n");
	fprintf(a,"            RowBox[{\"plotrangeY1\", \"+\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"(\", \n");
	fprintf(a,"               RowBox[{\"plotrangeY2\", \"-\", \"plotrangeY1\"}], \")\"}], \"*\", \n");
	fprintf(a,"              RowBox[{\"y\", \"/\", \n");
	fprintf(a,"               RowBox[{\"(\", \n");
	fprintf(a,"                RowBox[{\"plotresY\", \"-\", \"1\"}], \")\"}]}]}]}], \",\", \n");
	fprintf(a,"            RowBox[{\n");
	fprintf(a,"             RowBox[{\"s\", \"=\", \"0\"}], \";\", \n");
	fprintf(a,"             RowBox[{\"t\", \"=\", \"0\"}], \";\", \n");
	fprintf(a,"             RowBox[{\"For\", \"[\", \n");
	fprintf(a,"              RowBox[{\n");
	fprintf(a,"               RowBox[{\"ty\", \"=\", \n");
	fprintf(a,"                RowBox[{\"-\", \"smoothgrade\"}]}], \",\", \n");
	fprintf(a,"               RowBox[{\"ty\", \"\\[LessEqual]\", \"smoothgrade\"}], \",\", \n");
	fprintf(a,"               RowBox[{\"ty\", \"++\"}], \",\", \n");
	fprintf(a,"               RowBox[{\"For\", \"[\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"tx\", \"=\", \n");
	fprintf(a,"                  RowBox[{\"-\", \"smoothgrade\"}]}], \",\", \n");
	fprintf(a,"                 RowBox[{\"tx\", \"\\[LessEqual]\", \"smoothgrade\"}], \",\", \n");
	fprintf(a,"                 RowBox[{\"tx\", \"++\"}], \",\", \n");
	fprintf(a,"                 RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"tx\", \"^\", \"2\"}], \"+\", \n");
	fprintf(a,"                    RowBox[{\"ty\", \"^\", \"2\"}]}], \"\\[LessEqual]\", \n");
	fprintf(a,"                    RowBox[{\"smoothgrade\", \"^\", \"2\"}]}], \",\", \n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\"r\", \"=\", \n");
	fprintf(a,"                    RowBox[{\"1\", \"/\", \n");
	fprintf(a,"                    RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"tx\", \"^\", \"2\"}], \"+\", \n");
	fprintf(a,"                    RowBox[{\"ty\", \"^\", \"2\"}], \"+\", \"1\"}], \")\"}]}]}], \";\", \n");
	fprintf(a,"                    RowBox[{\"t\", \"+=\", \"r\"}], \";\", \n");
	fprintf(a,"                    RowBox[{\"s\", \"+=\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"dat2\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Max\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Min\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ox\", \"+\", \"tx\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"oplotresX\", \"-\", \"1\"}]}], \"]\"}], \",\", \"0\"}], \n");
	fprintf(a,"                    \"]\"}], \"+\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Max\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Min\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"oy\", \"+\", \"ty\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"oplotresY\", \"-\", \"1\"}]}], \"]\"}], \",\", \"0\"}], \n");
	fprintf(a,"                    \"]\"}], \"*\", \"oplotresX\"}], \"+\", \"1\"}], \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"3\", \"]\"}], \"]\"}], \"*\", \"r\"}]}]}], \",\"}], \n");
	fprintf(a,"                  \"]\"}]}], \"]\"}]}], \"]\"}], \";\", \n");
	fprintf(a,"             RowBox[{\"s\", \"/\", \"t\"}]}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\"i\", \",\", \"0\", \",\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"plotresX\", \"*\", \"plotresY\"}], \"-\", \"1\"}]}], \"}\"}]}], \n");
	fprintf(a,"        \"]\"}]}], \";\", \"\\[IndentingNewLine]\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"(*\", \" \", \n");
	fprintf(a,"       RowBox[{\n");
	fprintf(a,"       \"Add\", \" \", \"Functions\", \" \", \"to\", \" \", \"primitive\", \" \", \"list\"}], \n");
	fprintf(a,"       \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"For\", \"[\", \n");
	fprintf(a,"       RowBox[{\n");
	fprintf(a,"        RowBox[{\"i\", \"=\", \"1\"}], \",\", \n");
	fprintf(a,"        RowBox[{\"i\", \"<=\", \n");
	fprintf(a,"         RowBox[{\"Length\", \"[\", \"funcX\", \"]\"}]}], \",\", \n");
	fprintf(a,"        RowBox[{\"i\", \"++\"}], \",\", \n");
	fprintf(a,"        RowBox[{\"AppendTo\", \"[\", \n");
	fprintf(a,"         RowBox[{\"prim\", \",\", \n");
	fprintf(a,"          RowBox[{\"ListLinePlot\", \"[\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"Table\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\"x\", \",\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"funcX\", \"[\", \n");
	fprintf(a,"                  RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                 RowBox[{\"[\", \"1\", \"]\"}], \"]\"}]}], \"}\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\"x\", \",\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\n");
	fprintf(a,"                  RowBox[{\"funcX\", \"[\", \n");
	fprintf(a,"                   RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                  RowBox[{\"[\", \"2\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                 RowBox[{\"[\", \"1\", \"]\"}], \"]\"}], \",\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\n");
	fprintf(a,"                  RowBox[{\"funcX\", \"[\", \n");
	fprintf(a,"                   RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                  RowBox[{\"[\", \"2\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                 RowBox[{\"[\", \"2\", \"]\"}], \"]\"}], \",\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"funcX\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"2\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"2\", \"]\"}], \"]\"}], \"-\", \n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"funcX\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"2\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"1\", \"]\"}], \"]\"}]}], \")\"}], \"/\", \"1000\"}]}], \n");
	fprintf(a,"               \"}\"}]}], \"]\"}], \",\", \n");
	fprintf(a,"            RowBox[{\"PlotStyle\", \"\\[Rule]\", \n");
	fprintf(a,"             RowBox[{\"Evaluate\", \"[\", \n");
	fprintf(a,"              RowBox[{\n");
	fprintf(a,"               RowBox[{\"funcX\", \"[\", \n");
	fprintf(a,"                RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"               RowBox[{\"[\", \"3\", \"]\"}], \"]\"}], \"]\"}]}]}], \"]\"}]}], \"]\"}]}], \n");
	fprintf(a,"       \"]\"}], \";\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"For\", \"[\", \n");
	fprintf(a,"       RowBox[{\n");
	fprintf(a,"        RowBox[{\"i\", \"=\", \"1\"}], \",\", \n");
	fprintf(a,"        RowBox[{\"i\", \"<=\", \n");
	fprintf(a,"         RowBox[{\"Length\", \"[\", \"funcY\", \"]\"}]}], \",\", \n");
	fprintf(a,"        RowBox[{\"i\", \"++\"}], \",\", \n");
	fprintf(a,"        RowBox[{\"AppendTo\", \"[\", \n");
	fprintf(a,"         RowBox[{\"prim\", \",\", \n");
	fprintf(a,"          RowBox[{\"ListLinePlot\", \"[\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"Table\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"funcY\", \"[\", \n");
	fprintf(a,"                  RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                 RowBox[{\"[\", \"1\", \"]\"}], \"]\"}], \",\", \"y\"}], \"}\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\"y\", \",\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\n");
	fprintf(a,"                  RowBox[{\"funcY\", \"[\", \n");
	fprintf(a,"                   RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                  RowBox[{\"[\", \"2\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                 RowBox[{\"[\", \"1\", \"]\"}], \"]\"}], \",\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\n");
	fprintf(a,"                  RowBox[{\"funcY\", \"[\", \n");
	fprintf(a,"                   RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                  RowBox[{\"[\", \"2\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                 RowBox[{\"[\", \"2\", \"]\"}], \"]\"}], \",\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"funcY\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"2\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"2\", \"]\"}], \"]\"}], \"-\", \n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"funcY\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"2\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"1\", \"]\"}], \"]\"}]}], \")\"}], \"/\", \"1000\"}]}], \n");
	fprintf(a,"               \"}\"}]}], \"]\"}], \",\", \n");
	fprintf(a,"            RowBox[{\"PlotStyle\", \"\\[Rule]\", \n");
	fprintf(a,"             RowBox[{\"Evaluate\", \"[\", \n");
	fprintf(a,"              RowBox[{\n");
	fprintf(a,"               RowBox[{\"funcY\", \"[\", \n");
	fprintf(a,"                RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"               RowBox[{\"[\", \"3\", \"]\"}], \"]\"}], \"]\"}]}]}], \"]\"}]}], \"]\"}]}], \n");
	fprintf(a,"       \"]\"}], \";\", \"\\[IndentingNewLine]\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"tfx\", \":=\", \n");
	fprintf(a,"       RowBox[{\"Table\", \"[\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"i2\", \"=\", \n");
	fprintf(a,"           RowBox[{\"tickminX\", \"+\", \n");
	fprintf(a,"            RowBox[{\n");
	fprintf(a,"             RowBox[{\"i\", \"/\", \n");
	fprintf(a,"              RowBox[{\"(\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"(\", \n");
	fprintf(a,"                 RowBox[{\"majorticksX\", \"-\", \"1\"}], \")\"}], \"*\", \n");
	fprintf(a,"                RowBox[{\"(\", \n");
	fprintf(a,"                 RowBox[{\"minorticksX\", \"+\", \"1\"}], \")\"}]}], \")\"}]}], \"*\", \n");
	fprintf(a,"             RowBox[{\"(\", \n");
	fprintf(a,"              RowBox[{\"tickmaxX\", \"-\", \"tickminX\"}], \")\"}]}]}]}], \";\", \n");
	fprintf(a,"          RowBox[{\"{\", \n");
	fprintf(a,"           RowBox[{\"i2\", \",\", \n");
	fprintf(a,"            RowBox[{\"If\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\n");
	fprintf(a,"               RowBox[{\"Mod\", \"[\", \n");
	fprintf(a,"                RowBox[{\"i\", \",\", \n");
	fprintf(a,"                 RowBox[{\"minorticksX\", \"+\", \"1\"}]}], \"]\"}], \"\\[Equal]\", \n");
	fprintf(a,"               \"0\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"ticklabelprefixX\", \"<>\", \n");
	fprintf(a,"               RowBox[{\"ToString\", \"[\", \n");
	fprintf(a,"                RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                 RowBox[{\n");
	fprintf(a,"                  RowBox[{\"tickprecX\", \"\\[Equal]\", \"0\"}], \",\", \n");
	fprintf(a,"                  RowBox[{\"Round\", \"[\", \n");
	fprintf(a,"                   RowBox[{\"i2\", \",\", \"1\"}], \"]\"}], \",\", \n");
	fprintf(a,"                  RowBox[{\"PaddedForm\", \"[\", \n");
	fprintf(a,"                   RowBox[{\"i2\", \",\", \n");
	fprintf(a,"                    RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\"20\", \",\", \"tickprecX\"}], \"}\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"NumberPadding\", \"\\[Rule]\", \n");
	fprintf(a,"                    RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\"\\\"\\<\\>\\\"\", \",\", \"\\\"\\<0\\>\\\"\"}], \"}\"}]}]}], \n");
	fprintf(a,"                   \"]\"}]}], \"]\"}], \"]\"}], \"<>\", \"ticklabelsuffixX\"}], \",\", \n");
	fprintf(a,"              \"\\\"\\<\\>\\\"\"}], \"]\"}], \",\", \n");
	fprintf(a,"            RowBox[{\"If\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\n");
	fprintf(a,"               RowBox[{\"Mod\", \"[\", \n");
	fprintf(a,"                RowBox[{\"i\", \",\", \n");
	fprintf(a,"                 RowBox[{\"minorticksX\", \"+\", \"1\"}]}], \"]\"}], \"\\[Equal]\", \n");
	fprintf(a,"               \"0\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"majorticklength\", \"*\", \n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\"1\", \"-\", \"majortickshift\"}], \")\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"majorticklength\", \"*\", \"majortickshift\"}]}], \"}\"}], \n");
	fprintf(a,"              \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"minorticklength\", \"*\", \n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\"1\", \"-\", \"minortickshift\"}], \")\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"minorticklength\", \"*\", \"minortickshift\"}]}], \"}\"}]}],\n");
	fprintf(a,"              \"]\"}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\"i\", \",\", \"0\", \",\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"(\", \n");
	fprintf(a,"             RowBox[{\"majorticksX\", \"-\", \"1\"}], \")\"}], \"*\", \n");
	fprintf(a,"            RowBox[{\"(\", \n");
	fprintf(a,"             RowBox[{\"minorticksX\", \"+\", \"1\"}], \")\"}]}]}], \"}\"}]}], \"]\"}]}], \n");
	fprintf(a,"      \";\", \"\\[IndentingNewLine]\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"tfxnolabel\", \":=\", \n");
	fprintf(a,"       RowBox[{\"Table\", \"[\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"i2\", \"=\", \n");
	fprintf(a,"           RowBox[{\"tickminX\", \"+\", \n");
	fprintf(a,"            RowBox[{\n");
	fprintf(a,"             RowBox[{\"i\", \"/\", \n");
	fprintf(a,"              RowBox[{\"(\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"(\", \n");
	fprintf(a,"                 RowBox[{\"majorticksX\", \"-\", \"1\"}], \")\"}], \"*\", \n");
	fprintf(a,"                RowBox[{\"(\", \n");
	fprintf(a,"                 RowBox[{\"minorticksX\", \"+\", \"1\"}], \")\"}]}], \")\"}]}], \"*\", \n");
	fprintf(a,"             RowBox[{\"(\", \n");
	fprintf(a,"              RowBox[{\"tickmaxX\", \"-\", \"tickminX\"}], \")\"}]}]}]}], \";\", \n");
	fprintf(a,"          RowBox[{\"{\", \n");
	fprintf(a,"           RowBox[{\"i2\", \",\", \"\\\"\\<\\>\\\"\", \",\", \n");
	fprintf(a,"            RowBox[{\"If\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\n");
	fprintf(a,"               RowBox[{\"Mod\", \"[\", \n");
	fprintf(a,"                RowBox[{\"i\", \",\", \n");
	fprintf(a,"                 RowBox[{\"minorticksX\", \"+\", \"1\"}]}], \"]\"}], \"\\[Equal]\", \n");
	fprintf(a,"               \"0\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"majorticklength\", \"*\", \n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\"1\", \"-\", \"majortickshift\"}], \")\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"majorticklength\", \"*\", \"majortickshift\"}]}], \"}\"}], \n");
	fprintf(a,"              \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"minorticklength\", \"*\", \n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\"1\", \"-\", \"minortickshift\"}], \")\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"minorticklength\", \"*\", \"minortickshift\"}]}], \"}\"}]}],\n");
	fprintf(a,"              \"]\"}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\"i\", \",\", \"0\", \",\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"(\", \n");
	fprintf(a,"             RowBox[{\"majorticksX\", \"-\", \"1\"}], \")\"}], \"*\", \n");
	fprintf(a,"            RowBox[{\"(\", \n");
	fprintf(a,"             RowBox[{\"minorticksX\", \"+\", \"1\"}], \")\"}]}]}], \"}\"}]}], \"]\"}]}], \n");
	fprintf(a,"      \";\", \"\\[IndentingNewLine]\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"tfy\", \":=\", \n");
	fprintf(a,"       RowBox[{\"Table\", \"[\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"i2\", \"=\", \n");
	fprintf(a,"           RowBox[{\"tickminY\", \"+\", \n");
	fprintf(a,"            RowBox[{\n");
	fprintf(a,"             RowBox[{\"i\", \"/\", \n");
	fprintf(a,"              RowBox[{\"(\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"(\", \n");
	fprintf(a,"                 RowBox[{\"majorticksY\", \"-\", \"1\"}], \")\"}], \"*\", \n");
	fprintf(a,"                RowBox[{\"(\", \n");
	fprintf(a,"                 RowBox[{\"minorticksY\", \"+\", \"1\"}], \")\"}]}], \")\"}]}], \"*\", \n");
	fprintf(a,"             RowBox[{\"(\", \n");
	fprintf(a,"              RowBox[{\"tickmaxY\", \"-\", \"tickminY\"}], \")\"}]}]}]}], \";\", \n");
	fprintf(a,"          RowBox[{\"{\", \n");
	fprintf(a,"           RowBox[{\"i2\", \",\", \n");
	fprintf(a,"            RowBox[{\"If\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\n");
	fprintf(a,"               RowBox[{\"Mod\", \"[\", \n");
	fprintf(a,"                RowBox[{\"i\", \",\", \n");
	fprintf(a,"                 RowBox[{\"minorticksY\", \"+\", \"1\"}]}], \"]\"}], \"\\[Equal]\", \n");
	fprintf(a,"               \"0\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"ticklabelprefixY\", \"<>\", \n");
	fprintf(a,"               RowBox[{\"ToString\", \"[\", \n");
	fprintf(a,"                RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                 RowBox[{\n");
	fprintf(a,"                  RowBox[{\"tickprecY\", \"\\[Equal]\", \"0\"}], \",\", \n");
	fprintf(a,"                  RowBox[{\"Round\", \"[\", \n");
	fprintf(a,"                   RowBox[{\"i2\", \",\", \"1\"}], \"]\"}], \",\", \n");
	fprintf(a,"                  RowBox[{\"PaddedForm\", \"[\", \n");
	fprintf(a,"                   RowBox[{\"i2\", \",\", \n");
	fprintf(a,"                    RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\"20\", \",\", \"tickprecY\"}], \"}\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"NumberPadding\", \"\\[Rule]\", \n");
	fprintf(a,"                    RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\"\\\"\\<\\>\\\"\", \",\", \"\\\"\\<0\\>\\\"\"}], \"}\"}]}]}], \n");
	fprintf(a,"                   \"]\"}]}], \"]\"}], \"]\"}], \"<>\", \"ticklabelsuffixY\"}], \",\", \n");
	fprintf(a,"              \"\\\"\\<\\>\\\"\"}], \"]\"}], \",\", \n");
	fprintf(a,"            RowBox[{\"If\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\n");
	fprintf(a,"               RowBox[{\"Mod\", \"[\", \n");
	fprintf(a,"                RowBox[{\"i\", \",\", \n");
	fprintf(a,"                 RowBox[{\"minorticksY\", \"+\", \"1\"}]}], \"]\"}], \"\\[Equal]\", \n");
	fprintf(a,"               \"0\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"majorticklength\", \"*\", \n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\"1\", \"-\", \"majortickshift\"}], \")\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"majorticklength\", \"*\", \"majortickshift\"}]}], \"}\"}], \n");
	fprintf(a,"              \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"minorticklength\", \"*\", \n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\"1\", \"-\", \"minortickshift\"}], \")\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"minorticklength\", \"*\", \"minortickshift\"}]}], \"}\"}]}],\n");
	fprintf(a,"              \"]\"}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\"i\", \",\", \"0\", \",\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"(\", \n");
	fprintf(a,"             RowBox[{\"majorticksY\", \"-\", \"1\"}], \")\"}], \"*\", \n");
	fprintf(a,"            RowBox[{\"(\", \n");
	fprintf(a,"             RowBox[{\"minorticksY\", \"+\", \"1\"}], \")\"}]}]}], \"}\"}]}], \"]\"}]}], \n");
	fprintf(a,"      \";\", \"\\[IndentingNewLine]\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"tfynolabel\", \":=\", \n");
	fprintf(a,"       RowBox[{\"Table\", \"[\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"i2\", \"=\", \n");
	fprintf(a,"           RowBox[{\"tickminY\", \"+\", \n");
	fprintf(a,"            RowBox[{\n");
	fprintf(a,"             RowBox[{\"i\", \"/\", \n");
	fprintf(a,"              RowBox[{\"(\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"(\", \n");
	fprintf(a,"                 RowBox[{\"majorticksY\", \"-\", \"1\"}], \")\"}], \"*\", \n");
	fprintf(a,"                RowBox[{\"(\", \n");
	fprintf(a,"                 RowBox[{\"minorticksY\", \"+\", \"1\"}], \")\"}]}], \")\"}]}], \"*\", \n");
	fprintf(a,"             RowBox[{\"(\", \n");
	fprintf(a,"              RowBox[{\"tickmaxY\", \"-\", \"tickminY\"}], \")\"}]}]}]}], \";\", \n");
	fprintf(a,"          RowBox[{\"{\", \n");
	fprintf(a,"           RowBox[{\"i2\", \",\", \"\\\"\\<\\>\\\"\", \",\", \n");
	fprintf(a,"            RowBox[{\"If\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\n");
	fprintf(a,"               RowBox[{\"Mod\", \"[\", \n");
	fprintf(a,"                RowBox[{\"i\", \",\", \n");
	fprintf(a,"                 RowBox[{\"minorticksY\", \"+\", \"1\"}]}], \"]\"}], \"\\[Equal]\", \n");
	fprintf(a,"               \"0\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"majorticklength\", \"*\", \n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\"1\", \"-\", \"majortickshift\"}], \")\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"majorticklength\", \"*\", \"majortickshift\"}]}], \"}\"}], \n");
	fprintf(a,"              \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"minorticklength\", \"*\", \n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\"1\", \"-\", \"minortickshift\"}], \")\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"minorticklength\", \"*\", \"minortickshift\"}]}], \"}\"}]}],\n");
	fprintf(a,"              \"]\"}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\"i\", \",\", \"0\", \",\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"(\", \n");
	fprintf(a,"             RowBox[{\"majorticksY\", \"-\", \"1\"}], \")\"}], \"*\", \n");
	fprintf(a,"            RowBox[{\"(\", \n");
	fprintf(a,"             RowBox[{\"minorticksY\", \"+\", \"1\"}], \")\"}]}]}], \"}\"}]}], \"]\"}]}], \n");
	fprintf(a,"      \";\", \"\\[IndentingNewLine]\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"tfz\", \":=\", \n");
	fprintf(a,"       RowBox[{\"Table\", \"[\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"i2\", \"=\", \n");
	fprintf(a,"           RowBox[{\"tickminZ\", \"+\", \n");
	fprintf(a,"            RowBox[{\n");
	fprintf(a,"             RowBox[{\"i\", \"/\", \n");
	fprintf(a,"              RowBox[{\"(\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"(\", \n");
	fprintf(a,"                 RowBox[{\"majorticksZ\", \"-\", \"1\"}], \")\"}], \"*\", \n");
	fprintf(a,"                RowBox[{\"(\", \n");
	fprintf(a,"                 RowBox[{\"minorticksZ\", \"+\", \"1\"}], \")\"}]}], \")\"}]}], \"*\", \n");
	fprintf(a,"             RowBox[{\"(\", \n");
	fprintf(a,"              RowBox[{\"tickmaxZ\", \"-\", \"tickminZ\"}], \")\"}]}]}]}], \";\", \n");
	fprintf(a,"          RowBox[{\"{\", \n");
	fprintf(a,"           RowBox[{\"i2\", \",\", \n");
	fprintf(a,"            RowBox[{\"If\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\n");
	fprintf(a,"               RowBox[{\"Mod\", \"[\", \n");
	fprintf(a,"                RowBox[{\"i\", \",\", \n");
	fprintf(a,"                 RowBox[{\"minorticksZ\", \"+\", \"1\"}]}], \"]\"}], \"\\[Equal]\", \n");
	fprintf(a,"               \"0\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"ticklabelprefixZ\", \"<>\", \n");
	fprintf(a,"               RowBox[{\"ToString\", \"[\", \n");
	fprintf(a,"                RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                 RowBox[{\n");
	fprintf(a,"                  RowBox[{\"tickprecZ\", \"\\[Equal]\", \"0\"}], \",\", \n");
	fprintf(a,"                  RowBox[{\"Round\", \"[\", \n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\"i2\", \"^\", \n");
	fprintf(a,"                    RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\"1\", \"/\", \"exp\"}], \")\"}]}], \",\", \"1\"}], \"]\"}], \n");
	fprintf(a,"                  \",\", \n");
	fprintf(a,"                  RowBox[{\"PaddedForm\", \"[\", \n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\"i2\", \"^\", \n");
	fprintf(a,"                    RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\"1\", \"/\", \"exp\"}], \")\"}]}], \",\", \n");
	fprintf(a,"                    RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\"20\", \",\", \"tickprecZ\"}], \"}\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"NumberPadding\", \"\\[Rule]\", \n");
	fprintf(a,"                    RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\"\\\"\\<\\>\\\"\", \",\", \"\\\"\\<0\\>\\\"\"}], \"}\"}]}]}], \n");
	fprintf(a,"                   \"]\"}]}], \"]\"}], \"]\"}], \"<>\", \"ticklabelsuffixZ\"}], \",\", \n");
	fprintf(a,"              \"\\\"\\<\\>\\\"\"}], \"]\"}], \",\", \n");
	fprintf(a,"            RowBox[{\"If\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\n");
	fprintf(a,"               RowBox[{\"Mod\", \"[\", \n");
	fprintf(a,"                RowBox[{\"i\", \",\", \n");
	fprintf(a,"                 RowBox[{\"minorticksZ\", \"+\", \"1\"}]}], \"]\"}], \"\\[Equal]\", \n");
	fprintf(a,"               \"0\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"majorticklength\", \"*\", \n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\"1\", \"-\", \"majortickshift\"}], \")\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"majorticklength\", \"*\", \"majortickshift\"}]}], \"}\"}], \n");
	fprintf(a,"              \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"minorticklength\", \"*\", \n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\"1\", \"-\", \"minortickshift\"}], \")\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"minorticklength\", \"*\", \"minortickshift\"}]}], \"}\"}]}],\n");
	fprintf(a,"              \"]\"}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\"i\", \",\", \"0\", \",\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"(\", \n");
	fprintf(a,"             RowBox[{\"majorticksZ\", \"-\", \"1\"}], \")\"}], \"*\", \n");
	fprintf(a,"            RowBox[{\"(\", \n");
	fprintf(a,"             RowBox[{\"minorticksZ\", \"+\", \"1\"}], \")\"}]}]}], \"}\"}]}], \"]\"}]}], \n");
	fprintf(a,"      \";\", \"\\[IndentingNewLine]\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"tfznolabel\", \":=\", \n");
	fprintf(a,"       RowBox[{\"Table\", \"[\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"i2\", \"=\", \n");
	fprintf(a,"           RowBox[{\"tickminZ\", \"+\", \n");
	fprintf(a,"            RowBox[{\n");
	fprintf(a,"             RowBox[{\"i\", \"/\", \n");
	fprintf(a,"              RowBox[{\"(\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"(\", \n");
	fprintf(a,"                 RowBox[{\"majorticksZ\", \"-\", \"1\"}], \")\"}], \"*\", \n");
	fprintf(a,"                RowBox[{\"(\", \n");
	fprintf(a,"                 RowBox[{\"minorticksZ\", \"+\", \"1\"}], \")\"}]}], \")\"}]}], \"*\", \n");
	fprintf(a,"             RowBox[{\"(\", \n");
	fprintf(a,"              RowBox[{\"tickmaxZ\", \"-\", \"tickminZ\"}], \")\"}]}]}]}], \";\", \n");
	fprintf(a,"          RowBox[{\"{\", \n");
	fprintf(a,"           RowBox[{\"i2\", \",\", \"\\\"\\<\\>\\\"\", \",\", \n");
	fprintf(a,"            RowBox[{\"If\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\n");
	fprintf(a,"               RowBox[{\"Mod\", \"[\", \n");
	fprintf(a,"                RowBox[{\"i\", \",\", \n");
	fprintf(a,"                 RowBox[{\"minorticksZ\", \"+\", \"1\"}]}], \"]\"}], \"\\[Equal]\", \n");
	fprintf(a,"               \"0\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"majorticklength\", \"*\", \n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\"1\", \"-\", \"majortickshift\"}], \")\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"majorticklength\", \"*\", \"majortickshift\"}]}], \"}\"}], \n");
	fprintf(a,"              \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"minorticklength\", \"*\", \n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\"1\", \"-\", \"minortickshift\"}], \")\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"minorticklength\", \"*\", \"minortickshift\"}]}], \"}\"}]}],\n");
	fprintf(a,"              \"]\"}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\"i\", \",\", \"0\", \",\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"(\", \n");
	fprintf(a,"             RowBox[{\"majorticksZ\", \"-\", \"1\"}], \")\"}], \"*\", \n");
	fprintf(a,"            RowBox[{\"(\", \n");
	fprintf(a,"             RowBox[{\"minorticksZ\", \"+\", \"1\"}], \")\"}]}]}], \"}\"}]}], \"]\"}]}], \n");
	fprintf(a,"      \";\", \"\\[IndentingNewLine]\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"tfl\", \":=\", \n");
	fprintf(a,"       RowBox[{\"Table\", \"[\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"i2\", \"=\", \n");
	fprintf(a,"           RowBox[{\"tickminL\", \"+\", \n");
	fprintf(a,"            RowBox[{\n");
	fprintf(a,"             RowBox[{\"i\", \"/\", \n");
	fprintf(a,"              RowBox[{\"(\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"(\", \n");
	fprintf(a,"                 RowBox[{\"majorticksL\", \"-\", \"1\"}], \")\"}], \"*\", \n");
	fprintf(a,"                RowBox[{\"(\", \n");
	fprintf(a,"                 RowBox[{\"minorticksL\", \"+\", \"1\"}], \")\"}]}], \")\"}]}], \"*\", \n");
	fprintf(a,"             RowBox[{\"(\", \n");
	fprintf(a,"              RowBox[{\"tickmaxL\", \"-\", \"tickminL\"}], \")\"}]}]}]}], \";\", \n");
	fprintf(a,"          RowBox[{\"{\", \n");
	fprintf(a,"           RowBox[{\"i2\", \",\", \n");
	fprintf(a,"            RowBox[{\"If\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\n");
	fprintf(a,"               RowBox[{\"Mod\", \"[\", \n");
	fprintf(a,"                RowBox[{\"i\", \",\", \n");
	fprintf(a,"                 RowBox[{\"minorticksL\", \"+\", \"1\"}]}], \"]\"}], \"\\[Equal]\", \n");
	fprintf(a,"               \"0\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"ToString\", \"[\", \n");
	fprintf(a,"               RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"tickprecL\", \"\\[Equal]\", \"0\"}], \",\", \n");
	fprintf(a,"                 RowBox[{\"Round\", \"[\", \n");
	fprintf(a,"                  RowBox[{\"i2\", \",\", \"1\"}], \"]\"}], \",\", \n");
	fprintf(a,"                 RowBox[{\"PaddedForm\", \"[\", \n");
	fprintf(a,"                  RowBox[{\"i2\", \",\", \n");
	fprintf(a,"                   RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\"20\", \",\", \"tickprecL\"}], \"}\"}], \",\", \n");
	fprintf(a,"                   RowBox[{\"NumberPadding\", \"\\[Rule]\", \n");
	fprintf(a,"                    RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\"\\\"\\<\\>\\\"\", \",\", \"\\\"\\<0\\>\\\"\"}], \"}\"}]}]}], \n");
	fprintf(a,"                  \"]\"}]}], \"]\"}], \"]\"}], \",\", \"\\\"\\<\\>\\\"\"}], \"]\"}], \",\", \n");
	fprintf(a,"            RowBox[{\"If\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\n");
	fprintf(a,"               RowBox[{\"Mod\", \"[\", \n");
	fprintf(a,"                RowBox[{\"i\", \",\", \n");
	fprintf(a,"                 RowBox[{\"minorticksL\", \"+\", \"1\"}]}], \"]\"}], \"\\[Equal]\", \n");
	fprintf(a,"               \"0\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"majorticklengthL\", \"*\", \n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\"1\", \"-\", \"majortickshiftL\"}], \")\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"majorticklengthL\", \"*\", \"majortickshiftL\"}]}], \"}\"}],\n");
	fprintf(a,"               \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"minorticklengthL\", \"*\", \n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\"1\", \"-\", \"minortickshiftL\"}], \")\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"minorticklengthL\", \"*\", \"minortickshiftL\"}]}], \n");
	fprintf(a,"               \"}\"}]}], \"]\"}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\"i\", \",\", \"0\", \",\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"(\", \n");
	fprintf(a,"             RowBox[{\"majorticksL\", \"-\", \"1\"}], \")\"}], \"*\", \n");
	fprintf(a,"            RowBox[{\"(\", \n");
	fprintf(a,"             RowBox[{\"minorticksL\", \"+\", \"1\"}], \")\"}]}]}], \"}\"}]}], \"]\"}]}], \n");
	fprintf(a,"      \";\", \"\\[IndentingNewLine]\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"tflnolabel\", \":=\", \n");
	fprintf(a,"       RowBox[{\"Table\", \"[\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"i2\", \"=\", \n");
	fprintf(a,"           RowBox[{\"tickminL\", \"+\", \n");
	fprintf(a,"            RowBox[{\n");
	fprintf(a,"             RowBox[{\"i\", \"/\", \n");
	fprintf(a,"              RowBox[{\"(\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"(\", \n");
	fprintf(a,"                 RowBox[{\"majorticksL\", \"-\", \"1\"}], \")\"}], \"*\", \n");
	fprintf(a,"                RowBox[{\"(\", \n");
	fprintf(a,"                 RowBox[{\"minorticksL\", \"+\", \"1\"}], \")\"}]}], \")\"}]}], \"*\", \n");
	fprintf(a,"             RowBox[{\"(\", \n");
	fprintf(a,"              RowBox[{\"tickmaxL\", \"-\", \"tickminL\"}], \")\"}]}]}]}], \";\", \n");
	fprintf(a,"          RowBox[{\"{\", \n");
	fprintf(a,"           RowBox[{\"i2\", \",\", \"\\\"\\<\\>\\\"\", \",\", \n");
	fprintf(a,"            RowBox[{\"If\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\n");
	fprintf(a,"               RowBox[{\"Mod\", \"[\", \n");
	fprintf(a,"                RowBox[{\"i\", \",\", \n");
	fprintf(a,"                 RowBox[{\"minorticksL\", \"+\", \"1\"}]}], \"]\"}], \"\\[Equal]\", \n");
	fprintf(a,"               \"0\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"majorticklengthL\", \"*\", \n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\"1\", \"-\", \"majortickshiftL\"}], \")\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"majorticklengthL\", \"*\", \"majortickshiftL\"}]}], \"}\"}],\n");
	fprintf(a,"               \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"minorticklengthL\", \"*\", \n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\"1\", \"-\", \"minortickshiftL\"}], \")\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"minorticklengthL\", \"*\", \"minortickshiftL\"}]}], \n");
	fprintf(a,"               \"}\"}]}], \"]\"}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\"i\", \",\", \"0\", \",\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"(\", \n");
	fprintf(a,"             RowBox[{\"majorticksL\", \"-\", \"1\"}], \")\"}], \"*\", \n");
	fprintf(a,"            RowBox[{\"(\", \n");
	fprintf(a,"             RowBox[{\"minorticksL\", \"+\", \"1\"}], \")\"}]}]}], \"}\"}]}], \"]\"}]}], \n");
	fprintf(a,"      \";\", \"\\[IndentingNewLine]\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"(*\", \" \", \n");
	fprintf(a,"       RowBox[{\"Create\", \" \", \"Contour\", \" \", \"Plot\"}], \" \", \"*)\"}], \n");
	fprintf(a,"      \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"datmin\", \"=\", \n");
	fprintf(a,"       RowBox[{\"If\", \"[\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\"ValueQ\", \"[\", \"mincolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"         \"mincolorvalue\", \",\", \n");
	fprintf(a,"         RowBox[{\"Min\", \"[\", \n");
	fprintf(a,"          RowBox[{\"Table\", \"[\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\n");
	fprintf(a,"             RowBox[{\"dat\", \"[\", \n");
	fprintf(a,"              RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"             RowBox[{\"[\", \"3\", \"]\"}], \"]\"}], \",\", \n");
	fprintf(a,"            RowBox[{\"{\", \n");
	fprintf(a,"             RowBox[{\"i\", \",\", \"1\", \",\", \n");
	fprintf(a,"              RowBox[{\"plotresX\", \"*\", \"plotresY\"}]}], \"}\"}]}], \"]\"}], \n");
	fprintf(a,"          \"]\"}]}], \"]\"}]}], \";\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"datminexp\", \"=\", \n");
	fprintf(a,"       RowBox[{\"If\", \"[\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\"ValueQ\", \"[\", \"mincolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"Sign\", \"[\", \"mincolorvalue\", \"]\"}], \"*\", \n");
	fprintf(a,"          RowBox[{\"Abs\", \"[\", \n");
	fprintf(a,"           RowBox[{\"mincolorvalue\", \"^\", \"exp\"}], \"]\"}]}], \",\", \n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"Sign\", \"[\", \"datmin\", \"]\"}], \"*\", \n");
	fprintf(a,"          RowBox[{\"Abs\", \"[\", \n");
	fprintf(a,"           RowBox[{\"datmin\", \"^\", \"exp\"}], \"]\"}]}]}], \"]\"}]}], \";\", \n");
	fprintf(a,"      \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"datmax\", \"=\", \n");
	fprintf(a,"       RowBox[{\"If\", \"[\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\"ValueQ\", \"[\", \"maxcolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"         \"maxcolorvalue\", \",\", \n");
	fprintf(a,"         RowBox[{\"Max\", \"[\", \n");
	fprintf(a,"          RowBox[{\"Table\", \"[\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\n");
	fprintf(a,"             RowBox[{\"dat\", \"[\", \n");
	fprintf(a,"              RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"             RowBox[{\"[\", \"3\", \"]\"}], \"]\"}], \",\", \n");
	fprintf(a,"            RowBox[{\"{\", \n");
	fprintf(a,"             RowBox[{\"i\", \",\", \"1\", \",\", \n");
	fprintf(a,"              RowBox[{\"plotresX\", \"*\", \"plotresY\"}]}], \"}\"}]}], \"]\"}], \n");
	fprintf(a,"          \"]\"}]}], \"]\"}]}], \";\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"datmaxexp\", \"=\", \n");
	fprintf(a,"       RowBox[{\"If\", \"[\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\"ValueQ\", \"[\", \"maxcolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"Sign\", \"[\", \"maxcolorvalue\", \"]\"}], \"*\", \n");
	fprintf(a,"          RowBox[{\"Abs\", \"[\", \n");
	fprintf(a,"           RowBox[{\"maxcolorvalue\", \"^\", \"exp\"}], \"]\"}]}], \",\", \n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"Sign\", \"[\", \"datmax\", \"]\"}], \"*\", \n");
	fprintf(a,"          RowBox[{\"Abs\", \"[\", \n");
	fprintf(a,"           RowBox[{\"datmax\", \"^\", \"exp\"}], \"]\"}]}]}], \"]\"}]}], \";\", \n");
	fprintf(a,"      \"\\[IndentingNewLine]\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"      RowBox[{\"If\", \"[\", \n");
	fprintf(a,"       RowBox[{\n");
	fprintf(a,"        RowBox[{\"plottype\", \"\\[Equal]\", \"1\"}], \",\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"        RowBox[{\"Show\", \"[\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"         RowBox[{\n");


	fprintf(a,"          RowBox[{\"ListContourPlot\", \"[\",\n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"Table\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"dat\", \"[\", \n");
	fprintf(a,"                  RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                 RowBox[{\"[\", \"1\", \"]\"}], \"]\"}], \",\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"dat\", \"[\", \n");
	fprintf(a,"                  RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                 RowBox[{\"[\", \"2\", \"]\"}], \"]\"}], \",\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"Sign\", \"[\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\"dat\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                   RowBox[{\"[\", \"3\", \"]\"}], \"]\"}], \"]\"}], \"*\", \n");
	fprintf(a,"                 RowBox[{\"Abs\", \"[\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\"dat\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"3\", \"]\"}], \"]\"}], \"^\", \"exp\"}], \"]\"}]}]}], \n");
	fprintf(a,"               \"}\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\"i\", \",\", \"1\", \",\", \n");
	fprintf(a,"                RowBox[{\"plotresX\", \"*\", \"plotresY\"}]}], \"}\"}]}], \"]\"}], \",\", \n");
	fprintf(a,"            RowBox[{\"{\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"PlotRange\", \"\\[Rule]\", \n");
	fprintf(a,"               RowBox[{\"{\", \n");
	fprintf(a,"                RowBox[{\"datminexp\", \",\", \"datmaxexp\"}], \"}\"}]}], \",\", \n");
	fprintf(a,"              RowBox[{\"ColorFunction\", \"\\[Rule]\", \n");
	fprintf(a,"               RowBox[{\"Function\", \"[\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"{\", \"z\", \"}\"}], \",\", \n");
	fprintf(a,"                 RowBox[{\"ColFunc\", \"[\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\"z\", \"+\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"mincolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"-\", \n");
	fprintf(a,"                    RowBox[{\"Sign\", \"[\", \"mincolorvalue\", \"]\"}]}], \"*\", \n");
	fprintf(a,"                    RowBox[{\"Abs\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"mincolorvalue\", \"^\", \"exp\"}], \"]\"}]}], \",\", \n");
	fprintf(a,"                    RowBox[{\"-\", \"datminexp\"}]}], \"]\"}]}], \")\"}], \"/\", \n");
	fprintf(a,"                   RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"maxcolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Sign\", \"[\", \"maxcolorvalue\", \"]\"}], \"*\", \n");
	fprintf(a,"                    RowBox[{\"Abs\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"maxcolorvalue\", \"^\", \"exp\"}], \"]\"}]}], \"-\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Sign\", \"[\", \"mincolorvalue\", \"]\"}], \"*\", \n");
	fprintf(a,"                    RowBox[{\"Abs\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"mincolorvalue\", \"^\", \"exp\"}], \"]\"}]}]}], \",\", \n");
	fprintf(a,"                    RowBox[{\"datmaxexp\", \"-\", \"datminexp\"}]}], \"]\"}], \"/\", \n");
	fprintf(a,"                    \"colorscale\"}], \")\"}]}], \"]\"}]}], \"]\"}]}], \",\", \n");
	fprintf(a,"              RowBox[{\"ColorFunctionScaling\", \"\\[Rule]\", \"False\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"Contours\", \"\\[Rule]\", \"contours\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"PlotRangePadding\", \"\\[Rule]\", \"framemargins\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"FrameStyle\", \"\\[Rule]\", \n");
	fprintf(a,"               RowBox[{\"Directive\", \"[\", \n");
	fprintf(a,"                RowBox[{\"Thickness\", \"[\", \"framethickness\", \"]\"}], \"]\"}]}], \n");
	fprintf(a,"              \",\", \n");
	fprintf(a,"              RowBox[{\"FrameTicks\", \"\\[Rule]\", \n");
	fprintf(a,"               RowBox[{\"{\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"{\", \n");
	fprintf(a,"                  RowBox[{\"tfy\", \",\", \n");
	fprintf(a,"                   RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"ticksbothsidesY\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"labelsbothsidesY\", \"]\"}], \",\", \n");
	fprintf(a,"                    \"tfy\", \",\", \"tfynolabel\"}], \"]\"}], \",\", \"None\"}], \"]\"}]}],\n");
	fprintf(a,"                   \"}\"}], \",\", \n");
	fprintf(a,"                 RowBox[{\"{\", \n");
	fprintf(a,"                  RowBox[{\"tfx\", \",\", \n");
	fprintf(a,"                   RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"ticksbothsidesX\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"labelsbothsidesX\", \"]\"}], \",\", \n");
	fprintf(a,"                    \"tfx\", \",\", \"tfxnolabel\"}], \"]\"}], \",\", \"None\"}], \"]\"}]}],\n");
	fprintf(a,"                   \"}\"}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"              RowBox[{\"FrameLabel\", \"\\[Rule]\", \n");
	fprintf(a,"               RowBox[{\"{\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"Style\", \"[\", \n");
	fprintf(a,"                  RowBox[{\"xlabel\", \",\", \n");
	fprintf(a,"                   RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"                    RowBox[{\"labelsize\", \"*\", \n");
	fprintf(a,"                    RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"]\"}], \",\", \n");
	fprintf(a,"                 RowBox[{\"Style\", \"[\", \n");
	fprintf(a,"                  RowBox[{\"ylabel\", \",\", \n");
	fprintf(a,"                   RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"                    RowBox[{\"labelsize\", \"*\", \n");
	fprintf(a,"                    RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"]\"}], \",\", \n");
	fprintf(a,"                 \"\\\"\\<\\>\\\"\"}], \"}\"}]}], \",\", \n");
	fprintf(a,"              RowBox[{\"ContourLabels\", \"\\[Rule]\", \n");
	fprintf(a,"               RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"ValueQ\", \"[\", \"usecontourlabels\", \"]\"}], \",\", \n");
	fprintf(a,"                 RowBox[{\"(\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\"Text\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Style\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"#3\", \"^\", \n");
	fprintf(a,"                    RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\"1\", \"/\", \"exp\"}], \")\"}]}], \",\", \n");
	fprintf(a,"                    RowBox[{\"FontFamily\", \"\\[Rule]\", \"\\\"\\<Arial\\>\\\"\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"                    RowBox[{\"contourlabelsize\", \"*\", \n");
	fprintf(a,"                    RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\"#1\", \",\", \"#2\"}], \"}\"}]}], \"]\"}], \"&\"}], \")\"}], \n");
	fprintf(a,"                 \",\", \"None\"}], \"]\"}]}], \",\", \n");
	fprintf(a,"              RowBox[{\"InterpolationOrder\", \"\\[Rule]\", \"interpolationorder\"}],\n");
	fprintf(a,"               \",\", \n");
	fprintf(a,"              RowBox[{\"ContourStyle\", \"\\[Rule]\", \n");
	fprintf(a,"               RowBox[{\"Directive\", \"[\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"Thickness\", \"[\", \"contourthickness\", \"]\"}], \",\", \n");
	fprintf(a,"                 \"contourcolor\", \",\", \n");
	fprintf(a,"                 RowBox[{\"Opacity\", \"[\", \"contouropacity\", \"]\"}], \",\", \n");
	fprintf(a,"                 RowBox[{\"Dashing\", \"[\", \"contourdash\", \"]\"}]}], \"]\"}]}], \",\", \n");
	fprintf(a,"              RowBox[{\"TextStyle\", \"\\[Rule]\", \n");
	fprintf(a,"               RowBox[{\"{\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"FontFamily\", \"\\[Rule]\", \"\\\"\\<Arial\\>\\\"\"}], \",\", \n");
	fprintf(a,"                 RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"                  RowBox[{\"ticklabelsize\", \"*\", \n");
	fprintf(a,"                   RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"              RowBox[{\"AspectRatio\", \"\\[Rule]\", \"aspectratio\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"ImageSize\", \"\\[Rule]\", \"plotpixels\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"If\", \"[\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"drawmesh\", \"\\[Equal]\", \"1\"}], \",\", \n");
	fprintf(a,"                RowBox[{\"{\", \n");
	fprintf(a,"                 RowBox[{\n");
	fprintf(a,"                  RowBox[{\"Mesh\", \"\\[Rule]\", \n");
	fprintf(a,"                   RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"majorticksX\", \"-\", \"2\", \"+\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\"majorticksX\", \"-\", \"1\"}], \")\"}], \"*\", \n");
	fprintf(a,"                    \"minorticksX\"}]}], \",\", \n");
	fprintf(a,"                    RowBox[{\"majorticksY\", \"-\", \"2\", \"+\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\"majorticksY\", \"-\", \"1\"}], \")\"}], \"*\", \n");
	fprintf(a,"                    \"minorticksY\"}]}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"                  RowBox[{\"MeshFunctions\", \"\\[Rule]\", \n");
	fprintf(a,"                   RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"#1\", \"&\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"#2\", \"&\"}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"                  RowBox[{\"MeshStyle\", \"\\[Rule]\", \n");
	fprintf(a,"                   RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Directive\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Thickness\", \"[\", \"meshthicknessX\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"Opacity\", \"[\", \"meshopacityX\", \"]\"}], \",\", \n");
	fprintf(a,"                    \"meshcolorX\"}], \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"Directive\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Thickness\", \"[\", \"meshthicknessY\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"Opacity\", \"[\", \"meshopacityY\", \"]\"}], \",\", \n");
	fprintf(a,"                    \"meshcolorY\"}], \"]\"}]}], \"}\"}]}]}], \"}\"}], \",\", \n");
	fprintf(a,"                RowBox[{\"Mesh\", \"\\[Rule]\", \"None\"}]}], \"]\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"PerformanceGoal\", \"\\[Rule]\", \"\\\"\\<Quality\\>\\\"\"}]}], \n");
	fprintf(a,"             \"}\"}]}], \"]\"}], \",\", \"\\[IndentingNewLine]\", \"prim\"}], \"]\"}], \",\",\n");
	fprintf(a,"         \"\\[IndentingNewLine]\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"        RowBox[{\"If\", \"[\", \n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"plottype\", \"\\[Equal]\", \"2\"}], \",\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"          RowBox[{\"Show\", \"[\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"ListDensityPlot\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"Table\", \"[\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"{\", \n");
	fprintf(a,"                 RowBox[{\n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\"dat\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                   RowBox[{\"[\", \"1\", \"]\"}], \"]\"}], \",\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\"dat\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                   RowBox[{\"[\", \"2\", \"]\"}], \"]\"}], \",\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\"Sign\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"dat\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"3\", \"]\"}], \"]\"}], \"]\"}], \"*\", \n");
	fprintf(a,"                   RowBox[{\"Abs\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"dat\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"3\", \"]\"}], \"]\"}], \"^\", \"exp\"}], \"]\"}]}]}], \n");
	fprintf(a,"                 \"}\"}], \",\", \n");
	fprintf(a,"                RowBox[{\"{\", \n");
	fprintf(a,"                 RowBox[{\"i\", \",\", \"1\", \",\", \n");
	fprintf(a,"                  RowBox[{\"plotresX\", \"*\", \"plotresY\"}]}], \"}\"}]}], \"]\"}], \n");
	fprintf(a,"              \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"PlotRange\", \"\\[Rule]\", \n");
	fprintf(a,"                 RowBox[{\"{\", \n");
	fprintf(a,"                  RowBox[{\"datminexp\", \",\", \"datmaxexp\"}], \"}\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"ColorFunction\", \"\\[Rule]\", \n");
	fprintf(a,"                 RowBox[{\"Function\", \"[\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\"{\", \"z\", \"}\"}], \",\", \n");
	fprintf(a,"                   RowBox[{\"ColFunc\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\"z\", \"+\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"mincolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"-\", \n");
	fprintf(a,"                    RowBox[{\"Sign\", \"[\", \"mincolorvalue\", \"]\"}]}], \"*\", \n");
	fprintf(a,"                    RowBox[{\"Abs\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"mincolorvalue\", \"^\", \"exp\"}], \"]\"}]}], \",\", \n");
	fprintf(a,"                    RowBox[{\"-\", \"datminexp\"}]}], \"]\"}]}], \")\"}], \"/\", \n");
	fprintf(a,"                    RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"maxcolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Sign\", \"[\", \"maxcolorvalue\", \"]\"}], \"*\", \n");
	fprintf(a,"                    RowBox[{\"Abs\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"maxcolorvalue\", \"^\", \"exp\"}], \"]\"}]}], \"-\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Sign\", \"[\", \"mincolorvalue\", \"]\"}], \"*\", \n");
	fprintf(a,"                    RowBox[{\"Abs\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"mincolorvalue\", \"^\", \"exp\"}], \"]\"}]}]}], \",\", \n");
	fprintf(a,"                    RowBox[{\"datmaxexp\", \"-\", \"datminexp\"}]}], \"]\"}], \"/\", \n");
	fprintf(a,"                    \"colorscale\"}], \")\"}]}], \"]\"}]}], \"]\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"ColorFunctionScaling\", \"\\[Rule]\", \"False\"}], \",\", \n");
	fprintf(a,"                RowBox[{\"PlotRangePadding\", \"\\[Rule]\", \"framemargins\"}], \",\", \n");
	fprintf(a,"                RowBox[{\"FrameStyle\", \"\\[Rule]\", \n");
	fprintf(a,"                 RowBox[{\"Directive\", \"[\", \n");
	fprintf(a,"                  RowBox[{\"Thickness\", \"[\", \"framethickness\", \"]\"}], \"]\"}]}], \n");
	fprintf(a,"                \",\", \n");
	fprintf(a,"                RowBox[{\"FrameTicks\", \"\\[Rule]\", \n");
	fprintf(a,"                 RowBox[{\"{\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\"tfy\", \",\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"ticksbothsidesY\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"labelsbothsidesY\", \"]\"}], \",\", \n");
	fprintf(a,"                    \"tfy\", \",\", \"tfynolabel\"}], \"]\"}], \",\", \"None\"}], \"]\"}]}],\n");
	fprintf(a,"                     \"}\"}], \",\", \n");
	fprintf(a,"                   RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\"tfx\", \",\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"ticksbothsidesX\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"labelsbothsidesX\", \"]\"}], \",\", \n");
	fprintf(a,"                    \"tfx\", \",\", \"tfxnolabel\"}], \"]\"}], \",\", \"None\"}], \"]\"}]}],\n");
	fprintf(a,"                     \"}\"}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"FrameLabel\", \"\\[Rule]\", \n");
	fprintf(a,"                 RowBox[{\"{\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\"Style\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"xlabel\", \",\", \n");
	fprintf(a,"                    RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"                    RowBox[{\"labelsize\", \"*\", \n");
	fprintf(a,"                    RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"]\"}], \",\", \n");
	fprintf(a,"                   RowBox[{\"Style\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"ylabel\", \",\", \n");
	fprintf(a,"                    RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"                    RowBox[{\"labelsize\", \"*\", \n");
	fprintf(a,"                    RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"]\"}], \",\", \n");
	fprintf(a,"                   \"\\\"\\<\\>\\\"\"}], \"}\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                \"InterpolationOrder\", \"\\[Rule]\", \"interpolationorder\"}], \",\", \n");
	fprintf(a,"                RowBox[{\"TextStyle\", \"\\[Rule]\", \n");
	fprintf(a,"                 RowBox[{\"{\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\"FontFamily\", \"\\[Rule]\", \"\\\"\\<Arial\\>\\\"\"}], \",\", \n");
	fprintf(a,"                   RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"                    RowBox[{\"ticklabelsize\", \"*\", \n");
	fprintf(a,"                    RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"AspectRatio\", \"\\[Rule]\", \"aspectratio\"}], \",\", \n");
	fprintf(a,"                RowBox[{\"ImageSize\", \"\\[Rule]\", \"plotpixels\"}], \",\", \n");
	fprintf(a,"                RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                 RowBox[{\n");
	fprintf(a,"                  RowBox[{\"drawmesh\", \"\\[Equal]\", \"1\"}], \",\", \n");
	fprintf(a,"                  RowBox[{\"{\", \n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Mesh\", \"\\[Rule]\", \n");
	fprintf(a,"                    RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"majorticksX\", \"-\", \"2\", \"+\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\"majorticksX\", \"-\", \"1\"}], \")\"}], \"*\", \n");
	fprintf(a,"                    \"minorticksX\"}]}], \",\", \n");
	fprintf(a,"                    RowBox[{\"majorticksY\", \"-\", \"2\", \"+\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\"majorticksY\", \"-\", \"1\"}], \")\"}], \"*\", \n");
	fprintf(a,"                    \"minorticksY\"}]}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"                    RowBox[{\"MeshFunctions\", \"\\[Rule]\", \n");
	fprintf(a,"                    RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"#1\", \"&\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"#2\", \"&\"}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"                    RowBox[{\"MeshStyle\", \"\\[Rule]\", \n");
	fprintf(a,"                    RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Directive\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Thickness\", \"[\", \"meshthicknessX\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"Opacity\", \"[\", \"meshopacityX\", \"]\"}], \",\", \n");
	fprintf(a,"                    \"meshcolorX\"}], \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"Directive\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Thickness\", \"[\", \"meshthicknessY\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"Opacity\", \"[\", \"meshopacityY\", \"]\"}], \",\", \n");
	fprintf(a,"                    \"meshcolorY\"}], \"]\"}]}], \"}\"}]}]}], \"}\"}], \",\", \n");
	fprintf(a,"                  RowBox[{\"Mesh\", \"\\[Rule]\", \"None\"}]}], \"]\"}], \",\", \n");
	fprintf(a,"                RowBox[{\"PerformanceGoal\", \"\\[Rule]\", \"\\\"\\<Quality\\>\\\"\"}]}], \n");
	fprintf(a,"               \"}\"}]}], \"]\"}], \",\", \"\\[IndentingNewLine]\", \"prim\"}], \"]\"}], \n");
	fprintf(a,"          \",\", \"\\[IndentingNewLine]\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"          RowBox[{\"If\", \"[\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"plottype\", \"\\[Equal]\", \"3\"}], \",\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"            RowBox[{\"ReliefPlot\", \"[\", \n");


	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"Table\", \"[\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"Sign\", \"[\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\"dat\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"i\", \"*\", \"plotresX\"}], \"+\", \"j\"}], \"]\"}], \"]\"}], \n");
	fprintf(a,"                   \"[\", \n");
	fprintf(a,"                   RowBox[{\"[\", \"3\", \"]\"}], \"]\"}], \"]\"}], \"*\", \n");
	fprintf(a,"                 RowBox[{\"Abs\", \"[\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\"dat\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"i\", \"*\", \"plotresX\"}], \"+\", \"j\"}], \"]\"}], \"]\"}], \n");
	fprintf(a,"                    \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"3\", \"]\"}], \"]\"}], \"^\", \"exp\"}], \"]\"}]}], \n");
	fprintf(a,"                \",\", \n");
	fprintf(a,"                RowBox[{\"{\", \n");
	fprintf(a,"                 RowBox[{\"i\", \",\", \"1\", \",\", \n");
	fprintf(a,"                  RowBox[{\"plotresY\", \"-\", \"1\"}]}], \"}\"}], \",\", \n");
	fprintf(a,"                RowBox[{\"{\", \n");
	fprintf(a,"                 RowBox[{\"j\", \",\", \"1\", \",\", \n");
	fprintf(a,"                  RowBox[{\"plotresX\", \"-\", \"1\"}]}], \"}\"}]}], \"]\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"PlotRange\", \"\\[Rule]\", \n");
	fprintf(a,"                 RowBox[{\"{\", \n");
	fprintf(a,"                  RowBox[{\"datminexp\", \",\", \"datmaxexp\"}], \"}\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"ColorFunction\", \"\\[Rule]\", \n");
	fprintf(a,"                 RowBox[{\"Function\", \"[\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\"{\", \"z\", \"}\"}], \",\", \n");
	fprintf(a,"                   RowBox[{\"ColFunc\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\"z\", \"+\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"mincolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"-\", \n");
	fprintf(a,"                    RowBox[{\"Sign\", \"[\", \"mincolorvalue\", \"]\"}]}], \"*\", \n");
	fprintf(a,"                    RowBox[{\"Abs\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"mincolorvalue\", \"^\", \"exp\"}], \"]\"}]}], \",\", \n");
	fprintf(a,"                    RowBox[{\"-\", \"datminexp\"}]}], \"]\"}]}], \")\"}], \"/\", \n");
	fprintf(a,"                    RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"maxcolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Sign\", \"[\", \"maxcolorvalue\", \"]\"}], \"*\", \n");
	fprintf(a,"                    RowBox[{\"Abs\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"maxcolorvalue\", \"^\", \"exp\"}], \"]\"}]}], \"-\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Sign\", \"[\", \"mincolorvalue\", \"]\"}], \"*\", \n");
	fprintf(a,"                    RowBox[{\"Abs\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"mincolorvalue\", \"^\", \"exp\"}], \"]\"}]}]}], \",\", \n");
	fprintf(a,"                    RowBox[{\"datmaxexp\", \"-\", \"datminexp\"}]}], \"]\"}], \"/\", \n");
	fprintf(a,"                    \"colorscale\"}], \")\"}]}], \"]\"}]}], \"]\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"ColorFunctionScaling\", \"\\[Rule]\", \"False\"}], \",\", \n");
	fprintf(a,"                RowBox[{\"PlotRangePadding\", \"\\[Rule]\", \"framemargins\"}], \",\", \n");
	fprintf(a,"                RowBox[{\"FrameStyle\", \"\\[Rule]\", \n");
	fprintf(a,"                 RowBox[{\"Directive\", \"[\", \n");
	fprintf(a,"                  RowBox[{\"Thickness\", \"[\", \"framethickness\", \"]\"}], \"]\"}]}], \n");
	fprintf(a,"                \",\", \n");
	fprintf(a,"                RowBox[{\"FrameTicks\", \"\\[Rule]\", \n");
	fprintf(a,"                 RowBox[{\"{\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\"tfy\", \",\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"ticksbothsidesY\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"labelsbothsidesY\", \"]\"}], \",\", \n");
	fprintf(a,"                    \"tfy\", \",\", \"tfynolabel\"}], \"]\"}], \",\", \"None\"}], \"]\"}]}],\n");
	fprintf(a,"                     \"}\"}], \",\", \n");
	fprintf(a,"                   RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\"tfx\", \",\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"ticksbothsidesX\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"labelsbothsidesX\", \"]\"}], \",\", \n");
	fprintf(a,"                    \"tfx\", \",\", \"tfxnolabel\"}], \"]\"}], \",\", \"None\"}], \"]\"}]}],\n");
	fprintf(a,"                     \"}\"}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"FrameLabel\", \"\\[Rule]\", \n");
	fprintf(a,"                 RowBox[{\"{\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\"Style\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"xlabel\", \",\", \n");
	fprintf(a,"                    RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"                    RowBox[{\"labelsize\", \"*\", \n");
	fprintf(a,"                    RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"]\"}], \",\", \n");
	fprintf(a,"                   RowBox[{\"Style\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"ylabel\", \",\", \n");
	fprintf(a,"                    RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"                    RowBox[{\"labelsize\", \"*\", \n");
	fprintf(a,"                    RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"]\"}], \",\", \n");
	fprintf(a,"                   \"\\\"\\<\\>\\\"\"}], \"}\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"AspectRatio\", \"\\[Rule]\", \"aspectratio\"}], \",\", \n");
	fprintf(a,"                RowBox[{\"ImageSize\", \"\\[Rule]\", \"plotpixels\"}], \",\", \n");
	fprintf(a,"                RowBox[{\"DataRange\", \"\\[Rule]\", \n");
	fprintf(a,"                 RowBox[{\"{\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\"plotrangeX1\", \",\", \"plotrangeX2\"}], \"}\"}], \",\", \n");
	fprintf(a,"                   RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\"plotrangeY1\", \",\", \"plotrangeY2\"}], \"}\"}]}], \n");
	fprintf(a,"                  \"}\"}]}], \",\", \n");
	fprintf(a,"                RowBox[{\"LabelStyle\", \"\\[Rule]\", \n");
	fprintf(a,"                 RowBox[{\"{\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                   RowBox[{\"FontFamily\", \"\\[Rule]\", \"\\\"\\<Arial\\>\\\"\"}], \",\", \n");
	fprintf(a,"                   RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"                    RowBox[{\"ticklabelsize\", \"*\", \n");
	fprintf(a,"                    RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"}\"}]}]}], \n");
	fprintf(a,"               \"}\"}]}], \"]\"}], \",\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"            \"\\[IndentingNewLine]\", \n");
	fprintf(a,"            RowBox[{\"If\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"plottype\", \"\\[Equal]\", \"4\"}], \",\", \n");
	fprintf(a,"              \"\\[IndentingNewLine]\", \n");
	fprintf(a,"              RowBox[{\"ListPlot3D\", \"[\", \n");
	fprintf(a,"               RowBox[{\n");
	fprintf(a,"                RowBox[{\"Table\", \"[\", \n");
	fprintf(a,"                 RowBox[{\n");
	fprintf(a,"                  RowBox[{\"{\", \n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"dat\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"1\", \"]\"}], \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"dat\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"2\", \"]\"}], \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Sign\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"dat\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"3\", \"]\"}], \"]\"}], \"]\"}], \"*\", \n");
	fprintf(a,"                    RowBox[{\"Abs\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"dat\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"i\", \"]\"}], \"]\"}], \"[\", \n");
	fprintf(a,"                    RowBox[{\"[\", \"3\", \"]\"}], \"]\"}], \"^\", \"exp\"}], \"]\"}]}]}], \n");
	fprintf(a,"                   \"}\"}], \",\", \n");
	fprintf(a,"                  RowBox[{\"{\", \n");
	fprintf(a,"                   RowBox[{\"i\", \",\", \"1\", \",\", \n");
	fprintf(a,"                    RowBox[{\"plotresX\", \"*\", \"plotresY\"}]}], \"}\"}]}], \"]\"}], \n");
	fprintf(a,"                \",\", \n");
	fprintf(a,"                RowBox[{\"{\", \n");
	fprintf(a,"                 RowBox[{\n");
	fprintf(a,"                  RowBox[{\"PlotRange\", \"\\[Rule]\", \n");
	fprintf(a,"                   RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\"datminexp\", \",\", \"datmaxexp\"}], \"}\"}]}], \",\", \n");
	fprintf(a,"                  RowBox[{\"ColorFunction\", \"\\[Rule]\", \n");
	fprintf(a,"                   RowBox[{\"Function\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\"x\", \",\", \"y\", \",\", \"z\"}], \"}\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"ColFunc\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\"z\", \"+\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"mincolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"-\", \n");
	fprintf(a,"                    RowBox[{\"Sign\", \"[\", \"mincolorvalue\", \"]\"}]}], \"*\", \n");
	fprintf(a,"                    RowBox[{\"Abs\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"mincolorvalue\", \"^\", \"exp\"}], \"]\"}]}], \",\", \n");
	fprintf(a,"                    RowBox[{\"-\", \"datminexp\"}]}], \"]\"}]}], \")\"}], \"/\", \n");
	fprintf(a,"                    RowBox[{\"(\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"maxcolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Sign\", \"[\", \"maxcolorvalue\", \"]\"}], \"*\", \n");
	fprintf(a,"                    RowBox[{\"Abs\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"maxcolorvalue\", \"^\", \"exp\"}], \"]\"}]}], \"-\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Sign\", \"[\", \"mincolorvalue\", \"]\"}], \"*\", \n");
	fprintf(a,"                    RowBox[{\"Abs\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"mincolorvalue\", \"^\", \"exp\"}], \"]\"}]}]}], \",\", \n");
	fprintf(a,"                    RowBox[{\"datmaxexp\", \"-\", \"datminexp\"}]}], \"]\"}], \"/\", \n");
	fprintf(a,"                    \"colorscale\"}], \")\"}]}], \"]\"}]}], \"]\"}]}], \",\", \n");
	fprintf(a,"                  RowBox[{\"ColorFunctionScaling\", \"\\[Rule]\", \"False\"}], \",\", \n");
	fprintf(a,"                  RowBox[{\"PlotRangePadding\", \"\\[Rule]\", \"framemargins\"}], \n");
	fprintf(a,"                  \",\", \n");
	fprintf(a,"                  RowBox[{\n");
	fprintf(a,"                  \"InterpolationOrder\", \"\\[Rule]\", \"2\"}], \n");
	fprintf(a,"                  \",\", \n");
	fprintf(a,"                  RowBox[{\"TextStyle\", \"\\[Rule]\", \n");
	fprintf(a,"                   RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"FontFamily\", \"\\[Rule]\", \"\\\"\\<Arial\\>\\\"\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"FontSize\", \"\\[Rule]\", \"ticklabelsize\"}]}], \n");
	fprintf(a,"                    \"}\"}]}], \",\", \n");
	fprintf(a,"                  RowBox[{\"AspectRatio\", \"\\[Rule]\", \"aspectratio\"}], \",\", \n");
	fprintf(a,"                  RowBox[{\"ImageSize\", \"\\[Rule]\", \"plotpixels\"}], \",\", \n");
	fprintf(a,"                  RowBox[{\"Mesh\", \"\\[Rule]\", \n");
	fprintf(a,"                   RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\"gridxcount\", \",\", \"gridycount\"}], \"}\"}]}], \",\", \n");
	fprintf(a,"                  RowBox[{\"MeshStyle\", \"\\[Rule]\", \n");
	fprintf(a,"                   RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Directive\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Thickness\", \"[\", \"gridxthickness\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"Opacity\", \"[\", \"gridxopacity\", \"]\"}], \",\", \n");
	fprintf(a,"                    \"gridxcolor\"}], \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"Directive\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Thickness\", \"[\", \"gridythickness\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"Opacity\", \"[\", \"gridyopacity\", \"]\"}], \",\", \n");
	fprintf(a,"                    \"gridycolor\"}], \"]\"}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"                  RowBox[{\"Boxed\", \"\\[Rule]\", \n");
	fprintf(a,"                   RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"boxed\", \"]\"}], \",\", \"True\", \",\", \n");
	fprintf(a,"                    \"False\"}], \"]\"}]}], \",\", \n");
	fprintf(a,"                  RowBox[{\"BoxStyle\", \"\\[Rule]\", \n");
	fprintf(a,"                   RowBox[{\"Directive\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"boxcolor\", \",\", \n");
	fprintf(a,"                    RowBox[{\"Thickness\", \"[\", \"boxthickness\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"Opacity\", \"[\", \"boxopacity\", \"]\"}]}], \"]\"}]}], \n");
	fprintf(a,"                  \",\", \n");
	fprintf(a,"                  RowBox[{\"Ticks\", \"\\[Rule]\", \n");
	fprintf(a,"                   RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\"tfx\", \",\", \"tfy\", \",\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"labelsz\", \"]\"}], \",\", \"tfz\", \",\", \n");
	fprintf(a,"                    \"tfznolabel\"}], \"]\"}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"                  RowBox[{\"TicksStyle\", \"\\[Rule]\", \n");
	fprintf(a,"                   RowBox[{\"Directive\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"Thickness\", \"[\", \"axesthickness\", \"]\"}], \"]\"}]}],\n");
	fprintf(a,"                   \",\", \n");
	fprintf(a,"                  RowBox[{\"AxesStyle\", \"\\[Rule]\", \n");
	fprintf(a,"                   RowBox[{\"Thickness\", \"[\", \"axesthickness\", \"]\"}]}], \",\", \n");
	fprintf(a,"                  RowBox[{\"AxesEdge\", \"\\[Rule]\", \n");
	fprintf(a,"                   RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    \"xaxisposition\", \",\", \"yaxisposition\", \",\", \n");
	fprintf(a,"                    \"zaxisposition\"}], \"}\"}]}], \",\", \n");
	fprintf(a,"                  RowBox[{\"AxesLabel\", \"\\[Rule]\", \n");
	fprintf(a,"                   RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"Style\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"xlabel\", \",\", \n");
	fprintf(a,"                    RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"                    RowBox[{\"labelsize\", \"*\", \n");
	fprintf(a,"                    RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"Style\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"ylabel\", \",\", \n");
	fprintf(a,"                    RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"                    RowBox[{\"labelsize\", \"*\", \n");
	fprintf(a,"                    RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"Style\", \"[\", \n");
	fprintf(a,"                    RowBox[{\"zlabel\", \",\", \n");
	fprintf(a,"                    RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"                    RowBox[{\"labelsize\", \"*\", \n");
	fprintf(a,"                    RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"]\"}]}], \n");
	fprintf(a,"                    \"}\"}]}], \",\", \n");
	fprintf(a,"                  RowBox[{\"TextStyle\", \"\\[Rule]\", \n");
	fprintf(a,"                   RowBox[{\"{\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"FontFamily\", \"\\[Rule]\", \"\\\"\\<Arial\\>\\\"\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"                    RowBox[{\"ticklabelsize\", \"*\", \n");
	fprintf(a,"                    RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"}\"}]}]}], \n");
	fprintf(a,"                 \"}\"}]}], \"]\"}], \"\\[IndentingNewLine]\", \",\"}], \"]\"}]}], \n");
	fprintf(a,"           \"]\"}]}], \"]\"}]}], \"]\"}]}], \"\\[IndentingNewLine]\", \")\"}]}], \";\"}], \n");
	fprintf(a,"  \"\\[IndentingNewLine]\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\"Create\", \" \", \"Legend\"}], \" \", \"*)\"}]}], \"\\[IndentingNewLine]\", \n");
	fprintf(a," RowBox[{\n");

	fprintf(a,"  RowBox[{\"plotlegend\", \":=\",\n");
	fprintf(a,"   RowBox[{\"(\",\n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"     RowBox[{\"x\", \"=.\"}], \";\", \n");
	fprintf(a,"     RowBox[{\"y\", \"=.\"}], \";\", \n");
	fprintf(a,"     RowBox[{\"If\", \"[\", \n");
	fprintf(a,"      RowBox[{\n");
	fprintf(a,"       RowBox[{\"plottype\", \">\", \"1\"}], \",\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"       \"\\[IndentingNewLine]\", \n");
	fprintf(a,"       RowBox[{\"DensityPlot\", \"[\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\"If\", \"[\", \n");
	fprintf(a,"          RowBox[{\n");
	fprintf(a,"           RowBox[{\"explegend\", \"==\", \"1\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"x\", \"^\", \"exp\"}], \",\", \"x\"}], \"]\"}], \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\"x\", \",\", \"datmin\", \",\", \"datmax\"}], \"}\"}], \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\"y\", \",\", \"0\", \",\", \"1\"}], \"}\"}], \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\n");
	fprintf(a,"           RowBox[{\"PlotRange\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"If\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"explegend\", \"==\", \"1\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\"datminexp\", \",\", \"datmaxexp\"}], \"}\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\"datmin\", \",\", \"datmax\"}], \"}\"}]}], \"]\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"ColorFunction\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"Function\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"{\", \"z\", \"}\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"ColFunc\", \"[\", \n");
	fprintf(a,"               RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"explegend\", \"==\", \"1\"}], \",\", \n");
	fprintf(a,"                 RowBox[{\n");
	fprintf(a,"                  RowBox[{\"(\", \n");
	fprintf(a,"                   RowBox[{\"z\", \" \", \"-\", \"datminexp\"}], \")\"}], \"/\", \n");
	fprintf(a,"                  RowBox[{\"(\", \n");
	fprintf(a,"                   RowBox[{\"datmaxexp\", \"-\", \"datminexp\"}], \")\"}]}], \",\", \n");
	fprintf(a,"                 RowBox[{\n");
	fprintf(a,"                  RowBox[{\"(\", \n");
	fprintf(a,"                   RowBox[{\"z\", \"+\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"mincolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"-\", \"mincolorvalue\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"-\", \"datmin\"}]}], \"]\"}]}], \")\"}], \"/\", \n");
	fprintf(a,"                  RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"maxcolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"maxcolorvalue\", \"-\", \"mincolorvalue\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"datmax\", \"-\", \"datmin\"}]}], \"]\"}]}]}], \"]\"}], \n");
	fprintf(a,"               \"]\"}]}], \"]\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"ColorFunctionScaling\", \"\\[Rule]\", \"False\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"Axes\", \"\\[Rule]\", \"False\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"Frame\", \"\\[Rule]\", \"True\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"FrameStyle\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"Directive\", \"[\", \n");
	fprintf(a,"             RowBox[{\"Thickness\", \"[\", \"framethickness\", \"]\"}], \"]\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"FrameTicks\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"{\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\"None\", \",\", \"None\"}], \"}\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\"tfl\", \",\", \"None\"}], \"}\"}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"PlotRangePadding\", \"\\[Rule]\", \"None\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"FrameLabel\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"{\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"Style\", \"[\", \n");
	fprintf(a,"               RowBox[{\"zlabel\", \",\", \n");
	fprintf(a,"                RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"                 RowBox[{\"labelsize\", \"*\", \n");
	fprintf(a,"                  RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"]\"}], \",\", \n");
	fprintf(a,"              \"\\\"\\<\\>\\\"\", \",\", \"\\\"\\<\\>\\\"\"}], \"}\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"TextStyle\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"{\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"FontFamily\", \"\\[Rule]\", \"\\\"\\<Arial\\>\\\"\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"               RowBox[{\"ticklabelsize\", \"*\", \n");
	fprintf(a,"                RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"AspectRatio\", \"\\[Rule]\", \"0.04\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"ImageSize\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"plotpixels\", \"*\", \n");
	fprintf(a,"             RowBox[{\"7\", \"/\", \"8\"}]}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"PlotPoints\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"{\", \n");
	fprintf(a,"             RowBox[{\"2000\", \",\", \"2\"}], \"}\"}]}]}], \"}\"}]}], \"]\"}], \",\", \n");
	fprintf(a,"       \"\\[IndentingNewLine]\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"       RowBox[{\"ContourPlot\", \"[\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\"If\", \"[\", \n");
	fprintf(a,"          RowBox[{\n");
	fprintf(a,"           RowBox[{\"explegend\", \"==\", \"1\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"x\", \"^\", \"exp\"}], \",\", \"x\"}], \"]\"}], \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\"x\", \",\", \"datmin\", \",\", \"datmax\"}], \"}\"}], \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\"y\", \",\", \"0\", \",\", \"1\"}], \"}\"}], \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\n");
	fprintf(a,"           RowBox[{\"PlotRange\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"If\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"explegend\", \"==\", \"1\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\"datminexp\", \",\", \"datmaxexp\"}], \"}\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\"datmin\", \",\", \"datmax\"}], \"}\"}]}], \"]\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"ColorFunction\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"Function\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"{\", \"z\", \"}\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"ColFunc\", \"[\", \n");
	fprintf(a,"               RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"explegend\", \"==\", \"1\"}], \",\", \n");
	fprintf(a,"                 RowBox[{\n");
	fprintf(a,"                  RowBox[{\"(\", \n");
	fprintf(a,"                   RowBox[{\"z\", \" \", \"-\", \"datminexp\"}], \")\"}], \"/\", \n");
	fprintf(a,"                  RowBox[{\"(\", \n");
	fprintf(a,"                   RowBox[{\"datmaxexp\", \"-\", \"datminexp\"}], \")\"}]}], \",\", \n");
	fprintf(a,"                 RowBox[{\n");
	fprintf(a,"                  RowBox[{\"(\", \n");
	fprintf(a,"                   RowBox[{\"z\", \"+\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"mincolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"-\", \"mincolorvalue\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"-\", \"datmin\"}]}], \"]\"}]}], \")\"}], \"/\", \n");
	fprintf(a,"                  RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"maxcolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"maxcolorvalue\", \"-\", \"mincolorvalue\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"datmax\", \"-\", \"datmin\"}]}], \"]\"}]}]}], \"]\"}], \n");
	fprintf(a,"               \"]\"}]}], \"]\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"ColorFunctionScaling\", \"\\[Rule]\", \"False\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"Axes\", \"\\[Rule]\", \"False\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"Frame\", \"\\[Rule]\", \"True\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"FrameStyle\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"Directive\", \"[\", \n");
	fprintf(a,"             RowBox[{\"Thickness\", \"[\", \"framethickness\", \"]\"}], \"]\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"Contours\", \"\\[Rule]\", \"contours\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"FrameTicks\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"{\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\"None\", \",\", \"None\"}], \"}\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\"tfl\", \",\", \"None\"}], \"}\"}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"PlotRangePadding\", \"\\[Rule]\", \"None\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"ContourStyle\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"Directive\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"Thickness\", \"[\", \"contourthickness\", \"]\"}], \",\", \n");
	fprintf(a,"              \"contourcolor\", \",\", \n");
	fprintf(a,"              RowBox[{\"Opacity\", \"[\", \"contouropacity\", \"]\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"Dashing\", \"[\", \"contourdash\", \"]\"}]}], \"]\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"FrameLabel\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"{\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"Style\", \"[\", \n");
	fprintf(a,"               RowBox[{\"zlabel\", \",\", \n");
	fprintf(a,"                RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"                 RowBox[{\"labelsize\", \"*\", \n");
	fprintf(a,"                  RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"]\"}], \",\", \n");
	fprintf(a,"              \"\\\"\\<\\>\\\"\", \",\", \"\\\"\\<\\>\\\"\"}], \"}\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"TextStyle\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"{\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"FontFamily\", \"\\[Rule]\", \"\\\"\\<Arial\\>\\\"\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"               RowBox[{\"ticklabelsize\", \"*\", \n");
	fprintf(a,"                RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"AspectRatio\", \"\\[Rule]\", \"0.04\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"ImageSize\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"plotpixels\", \"*\", \n");
	fprintf(a,"             RowBox[{\"7\", \"/\", \"8\"}]}]}]}], \"}\"}]}], \"]\"}]}], \"]\"}]}], \n");
	fprintf(a,"    \")\"}]}], \";\"}]}], \"Input\"]\n");
	fprintf(a,"}, Closed]],\n");


/*	fprintf(a,"  RowBox[{\"plotlegend\", \":=\", \n");
	fprintf(a,"   RowBox[{\"(\", \n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"     RowBox[{\"x\", \"=.\"}], \";\", \n");
	fprintf(a,"     RowBox[{\"y\", \"=.\"}], \";\", \n");
	fprintf(a,"     RowBox[{\"If\", \"[\", \n");
	fprintf(a,"      RowBox[{\n");
	fprintf(a,"       RowBox[{\"plottype\", \">\", \"1\"}], \",\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"       \"\\[IndentingNewLine]\", \n");
	fprintf(a,"       RowBox[{\"DensityPlot\", \"[\", \n");
	fprintf(a,"        RowBox[{\"x\", \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\"x\", \",\", \n");
	fprintf(a,"           RowBox[{\"If\", \"[\", \n");
	fprintf(a,"            RowBox[{\n");
	fprintf(a,"             RowBox[{\"ValueQ\", \"[\", \"mincolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"             \"mincolorvalue\", \",\", \"datmin\"}], \"]\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"If\", \"[\", \n");
	fprintf(a,"            RowBox[{\n");
	fprintf(a,"             RowBox[{\"ValueQ\", \"[\", \"maxcolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"             \"maxcolorvalue\", \",\", \"datmax\"}], \"]\"}]}], \"}\"}], \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\"y\", \",\", \"0\", \",\", \"1\"}], \"}\"}], \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\n");
	fprintf(a,"           RowBox[{\"PlotRange\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"{\", \n");
	fprintf(a,"             RowBox[{\"datmin\", \",\", \"datmax\"}], \"}\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"ColorFunction\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"Function\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"{\", \"z\", \"}\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"ColFunc\", \"[\", \n");
	fprintf(a,"               RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"ValueQ\", \"[\", \"explegend\", \"]\"}], \",\", \n");
	fprintf(a,"                 RowBox[{\n");
	fprintf(a,"                  RowBox[{\"(\", \n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\"z\", \" \", \"^\", \"exp\"}], \"+\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"mincolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"-\", \"mincolorvalue\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"-\", \"datminexp\"}]}], \"]\"}]}], \")\"}], \"/\", \n");
	fprintf(a,"                  RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"maxcolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"maxcolorvalue\", \"-\", \"mincolorvalue\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"datmaxexp\", \"-\", \"datminexp\"}]}], \"]\"}]}], \",\", \n");
	fprintf(a,"                 RowBox[{\n");
	fprintf(a,"                  RowBox[{\"(\", \n");
	fprintf(a,"                   RowBox[{\"z\", \"+\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"mincolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"-\", \"mincolorvalue\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"-\", \"datmin\"}]}], \"]\"}]}], \")\"}], \"/\", \n");
	fprintf(a,"                  RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"maxcolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"maxcolorvalue\", \"-\", \"mincolorvalue\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"datmax\", \"-\", \"datmin\"}]}], \"]\"}]}]}], \"]\"}], \n");
	fprintf(a,"               \"]\"}]}], \"]\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"ColorFunctionScaling\", \"\\[Rule]\", \"False\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"Axes\", \"\\[Rule]\", \"False\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"Frame\", \"\\[Rule]\", \"True\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"FrameStyle\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"Directive\", \"[\", \n");
	fprintf(a,"             RowBox[{\"Thickness\", \"[\", \"framethickness\", \"]\"}], \"]\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"FrameTicks\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"{\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\"None\", \",\", \"None\"}], \"}\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\"tfl\", \",\", \"None\"}], \"}\"}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"PlotRangePadding\", \"\\[Rule]\", \"None\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"FrameLabel\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"{\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"Style\", \"[\", \n");
	fprintf(a,"               RowBox[{\"zlabel\", \",\", \n");
	fprintf(a,"                RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"                 RowBox[{\"labelsize\", \"*\", \n");
	fprintf(a,"                  RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"]\"}], \",\", \n");
	fprintf(a,"              \"\\\"\\<\\>\\\"\", \",\", \"\\\"\\<\\>\\\"\"}], \"}\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"TextStyle\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"{\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"FontFamily\", \"\\[Rule]\", \"\\\"\\<Arial\\>\\\"\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"               RowBox[{\"ticklabelsize\", \"*\", \n");
	fprintf(a,"                RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"AspectRatio\", \"\\[Rule]\", \"0.04\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"ImageSize\", \"\\[Rule]\", \n");

	fprintf(a,"            RowBox[{\"plotpixels\", \"*\",\n");
	fprintf(a,"             RowBox[{\"7\", \"/\", \"8\"}]}]}], \",\",\n");
	fprintf(a,"           RowBox[{\"PlotPoints\", \"\\[Rule]\",\n");
	fprintf(a,"            RowBox[{\"{\",\n");
	fprintf(a,"             RowBox[{\"2000\", \",\", \"2\"}], \"}\"}]}]}], \"}\"}]}], \"]\"}], \",\", \n");
	fprintf(a,"       \"\\[IndentingNewLine]\", \"\\[IndentingNewLine]\", \n");


	fprintf(a,"       RowBox[{\"ContourPlot\", \"[\",\n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\"If\", \"[\", \n");
	fprintf(a,"          RowBox[{\n");
	fprintf(a,"           RowBox[{\"explegend\", \"==\", \"1\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"x\", \"^\", \"exp\"}], \",\", \"x\"}], \"]\"}], \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\"x\", \",\", \n");
	fprintf(a,"           RowBox[{\"If\", \"[\", \n");
	fprintf(a,"            RowBox[{\n");
	fprintf(a,"             RowBox[{\"ValueQ\", \"[\", \"mincolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"             \"mincolorvalue\", \",\", \"datmin\"}], \"]\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"If\", \"[\", \n");
	fprintf(a,"            RowBox[{\n");
	fprintf(a,"             RowBox[{\"ValueQ\", \"[\", \"maxcolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"             \"maxcolorvalue\", \",\", \"datmax\"}], \"]\"}]}], \"}\"}], \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\"y\", \",\", \"0\", \",\", \"1\"}], \"}\"}], \",\", \n");
	fprintf(a,"         RowBox[{\"{\", \n");
	fprintf(a,"          RowBox[{\n");
	fprintf(a,"           RowBox[{\"PlotRange\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"If\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"explegend\", \"==\", \"1\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\"datminexp\", \",\", \"datmaxexp\"}], \"}\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\"datmin\", \",\", \"datmax\"}], \"}\"}]}], \"]\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"ColorFunction\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"Function\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"{\", \"z\", \"}\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"ColFunc\", \"[\", \n");
	fprintf(a,"               RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                RowBox[{\n");
	fprintf(a,"                 RowBox[{\"explegend\", \"==\", \"1\"}], \",\", \n");
	fprintf(a,"                 RowBox[{\n");
	fprintf(a,"                  RowBox[{\"(\", \n");
	fprintf(a,"                   RowBox[{\"z\", \" \", \"+\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"mincolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"-\", \"mincolorvalue\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"-\", \"datminexp\"}]}], \"]\"}]}], \")\"}], \"/\", \n");
	fprintf(a,"                  RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"maxcolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"maxcolorvalue\", \"-\", \"mincolorvalue\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"datmaxexp\", \"-\", \"datminexp\"}]}], \"]\"}]}], \",\", \n");
	fprintf(a,"                 RowBox[{\n");
	fprintf(a,"                  RowBox[{\"(\", \n");
	fprintf(a,"                   RowBox[{\"z\", \"+\", \n");
	fprintf(a,"                    RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                    RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"mincolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"-\", \"mincolorvalue\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"-\", \"datmin\"}]}], \"]\"}]}], \")\"}], \"/\", \n");
	fprintf(a,"                  RowBox[{\"If\", \"[\", \n");
	fprintf(a,"                   RowBox[{\n");
	fprintf(a,"                    RowBox[{\"ValueQ\", \"[\", \"maxcolorvalue\", \"]\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"maxcolorvalue\", \"-\", \"mincolorvalue\"}], \",\", \n");
	fprintf(a,"                    RowBox[{\"datmax\", \"-\", \"datmin\"}]}], \"]\"}]}]}], \"]\"}], \n");
	fprintf(a,"               \"]\"}]}], \"]\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"ColorFunctionScaling\", \"\\[Rule]\", \"False\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"Axes\", \"\\[Rule]\", \"False\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"Frame\", \"\\[Rule]\", \"True\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"FrameStyle\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"Directive\", \"[\", \n");
	fprintf(a,"             RowBox[{\"Thickness\", \"[\", \"framethickness\", \"]\"}], \"]\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"Contours\", \"\\[Rule]\", \"contours\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"FrameTicks\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"{\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\"None\", \",\", \"None\"}], \"}\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"{\", \n");
	fprintf(a,"               RowBox[{\"tfl\", \",\", \"None\"}], \"}\"}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"PlotRangePadding\", \"\\[Rule]\", \"None\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"ContourStyle\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"Directive\", \"[\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"Thickness\", \"[\", \"contourthickness\", \"]\"}], \",\", \n");
	fprintf(a,"              \"contourcolor\", \",\", \n");
	fprintf(a,"              RowBox[{\"Opacity\", \"[\", \"contouropacity\", \"]\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"Dashing\", \"[\", \"contourdash\", \"]\"}]}], \"]\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"FrameLabel\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"{\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"Style\", \"[\", \n");
	fprintf(a,"               RowBox[{\"zlabel\", \",\", \n");
	fprintf(a,"                RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"                 RowBox[{\"labelsize\", \"*\", \n");
	fprintf(a,"                  RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"]\"}], \",\", \n");
	fprintf(a,"              \"\\\"\\<\\>\\\"\", \",\", \"\\\"\\<\\>\\\"\"}], \"}\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"TextStyle\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"{\", \n");
	fprintf(a,"             RowBox[{\n");
	fprintf(a,"              RowBox[{\"FontFamily\", \"\\[Rule]\", \"\\\"\\<Arial\\>\\\"\"}], \",\", \n");
	fprintf(a,"              RowBox[{\"FontSize\", \"\\[Rule]\", \n");
	fprintf(a,"               RowBox[{\"ticklabelsize\", \"*\", \n");
	fprintf(a,"                RowBox[{\"plotpixels\", \"/\", \"800\"}]}]}]}], \"}\"}]}], \",\", \n");
	fprintf(a,"           RowBox[{\"AspectRatio\", \"\\[Rule]\", \"0.04\"}], \",\", \n");
	fprintf(a,"           RowBox[{\"ImageSize\", \"\\[Rule]\", \n");
	fprintf(a,"            RowBox[{\"plotpixels\", \"*\", \n");
	fprintf(a,"             RowBox[{\"7\", \"/\", \"8\"}]}]}]}], \"}\"}]}], \"]\"}]}], \"]\"}]}], \n");
	fprintf(a,"    \")\"}]}], \";\"}]}], \"Input\"]\n");
	fprintf(a,"}, Closed  ]],\n");*/


	fprintf(a,"\n");
	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[\"Plotting Parameters\", \"Section\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = False}, \n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell],\n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]], \n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[\"Misc. Properties\", \"Subsection\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = True}, \n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell], \n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]], \n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"     RowBox[{\"Choose\", \" \", \"the\", \" \", \"plot\", \" \", \n");
	fprintf(a,"      RowBox[{\"type\", \".\", \" \", \"1\"}]}], \" \", \"=\", \" \", \n");
	fprintf(a,"     RowBox[{\"Contour\", \" \", \"Plot\"}]}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"2\", \" \", \"=\", \" \", \n");
	fprintf(a,"     RowBox[{\"Density\", \" \", \"Plot\"}]}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"3\", \" \", \"=\", \" \", \n");
	fprintf(a,"     RowBox[{\"Relief\", \" \", \"Plot\"}]}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"4\", \" \", \"=\", \" \", \n");
	fprintf(a,"     RowBox[{\"3\", \"D\", \" \", \"Plot\"}]}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"plottype\", \"=\", \"%d\"}], \";\"}]}]], \"Input\"],\n",m_iPlotType);
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"     RowBox[{\"Choose\", \" \", \"the\", \" \", \"color\", \" \", \n");
	fprintf(a,"      RowBox[{\"scale\", \":\", \"1\"}]}], \"=\", \n");
	fprintf(a,"     RowBox[{\"Travis\", \" \", \"Scale\"}]}], \",\", \n");
	fprintf(a,"    RowBox[{\"2\", \"=\", \n");
	fprintf(a,"     RowBox[{\"black\", \" \", \"and\", \" \", \"white\"}]}], \",\", \n");
	fprintf(a,"    RowBox[{\"3\", \"=\", \n");
	fprintf(a,"     RowBox[{\"Temperature\", \" \", \"scale\"}]}]}], \" \", \"*)\"}], \n");
	fprintf(a,"  \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"coloring\", \"=\", \"1\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Choose\", \" \", \"the\", \" \", \"exponent\", \" \", \"your\", \" \", \"data\", \" \", \n");
	fprintf(a,"    \"will\", \" \", \"be\", \" \", \"processed\", \" \", \n");
	fprintf(a,"    RowBox[{\"with\", \".\", \" \", \"Try\"}], \" \", \"values\", \" \", \"between\", \" \", \n");
	fprintf(a,"    \"0.2\", \" \", \"and\", \" \", \"1.\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"exp\", \"=\", \"%.1f\"}], \";\"}]}]], \"Input\"],\n",m_fPlotExp);
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Also\", \" \", \"scale\", \" \", \"the\", \" \", \"legend\", \" \", \"with\", \" \", \"the\", \n");
	fprintf(a,"    \" \", \"exponent\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"explegend\", \"=\", \"%d\"}], \";\"}]}]], \"Input\"],\n",m_iExpLegend);
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"    \"Choose\", \" \", \"the\", \" \", \"smoothing\", \" \", \"grade\", \" \", \"for\", \" \", \n");
	fprintf(a,"     \"your\", \" \", \n");
	fprintf(a,"     RowBox[{\"data\", \".\", \" \", \"The\"}], \" \", \"higher\", \" \", \"it\", \" \", \"is\"}],\n");
	fprintf(a,"     \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"the\", \" \", \"longer\", \" \", \"it\", \" \", \n");
	fprintf(a,"     RowBox[{\"takes\", \".\", \" \", \"0\"}], \" \", \"means\", \" \", \"no\", \" \", \n");
	fprintf(a,"     RowBox[{\"smoothing\", \".\"}]}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"smoothgrade\", \"=\", \"%d\"}], \";\"}]}]], \"Input\"],\n",m_iSmoothGrade);
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"interpolation\", \" \", \"order\", \" \", \"mathematica\", \" \", \"uses\",\n");
	fprintf(a,"     \" \", \"between\", \" \", \"the\", \" \", \"data\", \" \", \n");
	fprintf(a,"    RowBox[{\"points\", \".\", \" \", \"0\"}], \" \", \"shows\", \" \", \"the\", \" \", \"data\", \n");
	fprintf(a,"    \" \", \"points\", \" \", \"like\", \" \", \"they\", \" \", \n");
	fprintf(a,"    RowBox[{\"are\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"interpolationorder\", \"=\", \"%d\"}], \";\"}]}]], \"Input\"],\n",m_iInterpolationOrder);
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\"Choose\", \" \", \"the\", \" \", \"color\", \" \", \n");
	fprintf(a,"    RowBox[{\"scaling\", \".\", \" \", \"1\"}], \" \", \"means\", \" \", \"that\", \" \", \"the\",\n");
	fprintf(a,"     \" \", \"highest\", \" \", \"peak\", \" \", \"only\", \" \", \"gets\", \" \", \"the\", \" \", \n");
	fprintf(a,"    \"highest\", \" \", \n");
	fprintf(a,"    RowBox[{\"color\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"colorscale\", \"=\", \"1.0\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"width\", \" \", \"of\", \" \", \"the\", \" \", \"Plot\", \" \", \"in\", \" \", \n");
	fprintf(a,"    \"pixels\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"plotpixels\", \"=\", \"1000\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\"The\", \" \", \"aspect\", \" \", \"ratio\", \" \", \"of\", \" \", \"the\", \" \", \n");
	fprintf(a,"    RowBox[{\"plot\", \".\", \" \", \"1\"}], \" \", \"means\", \" \", \n");
	fprintf(a,"    RowBox[{\"quadratic\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"aspectratio\", \"=\", \"1\"}], \";\"}]}]], \"Input\"],\n");

	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"     RowBox[{\n");
	fprintf(a,"     \"Controls\", \" \", \"if\", \" \", \"a\", \" \", \"mesh\", \" \", \"is\", \" \", \"drawn\", \n");
	fprintf(a,"      \" \", \"across\", \" \", \"the\", \" \", \n");
	fprintf(a,"      RowBox[{\"plot\", \".\", \" \", \"1\"}]}], \"=\", \"enable\"}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"0\", \"=\", \n");
	fprintf(a,"     RowBox[{\n");
	fprintf(a,"      RowBox[{\n");
	fprintf(a,"       RowBox[{\"disable\", \".\", \" \", \"For\"}], \" \", \"mesh\", \" \", \"appearance\", \n");
	fprintf(a,"       \" \", \"see\", \" \", \"\\\"\\<Axis Properties\\>\\\"\"}], \" \", \"\\[Rule]\", \" \", \n");
	fprintf(a,"      \"\\\"\\<Mesh Properties\\>\\\"\"}]}]}], \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"drawmesh\", \"=\", \"1\"}], \";\"}]}]], \"Input\"]\n");

	fprintf(a,"}, Open  ]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[\"Contour Properties\", \"Subsection\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = False}, \n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell], \n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]], \n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Choose\", \" \", \"the\", \" \", \"number\", \" \", \"of\", \" \", \"contours\", \" \", \n");
	fprintf(a,"    \"that\", \" \", \"will\", \" \", \"be\", \" \", \"displayed\"}], \" \", \"*)\"}], \n");
	fprintf(a,"  \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"contours\", \"=\", \"30\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"    \"Print\", \" \", \"the\", \" \", \"height\", \" \", \"value\", \" \", \"onto\", \" \", \n");
	fprintf(a,"     \"each\", \" \", \"contour\", \" \", \n");
	fprintf(a,"     RowBox[{\"line\", \"?\", \" \", \"If\"}], \" \", \"desired\"}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"replace\", \" \", \"the\", \" \", \"dot\", \" \", \"by\", \" \", \n");
	fprintf(a,"     RowBox[{\"\\\"\\<1\\>\\\"\", \".\"}]}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"usecontourlabels\", \"=.\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\"If\", \" \", \"contour\", \" \", \"labels\", \" \", \"are\", \" \", \n");
	fprintf(a,"    RowBox[{\"used\", \":\", \" \", \n");
	fprintf(a,"     RowBox[{\"The\", \" \", \"font\", \" \", \"size\", \" \", \"of\", \" \", \"these\", \" \", \n");
	fprintf(a,"      RowBox[{\"labels\", \".\"}]}]}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"contourlabelsize\", \"=\", \"10\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"thickness\", \" \", \"of\", \" \", \"the\", \" \", \"contour\", \" \", \n");
	fprintf(a,"    \"lines\", \" \", \n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"     RowBox[{\"(\", \n");
	fprintf(a,"      RowBox[{\n");
	fprintf(a,"      \"relative\", \" \", \"to\", \" \", \"the\", \" \", \"overall\", \" \", \"plot\", \" \", \n");
	fprintf(a,"       \"width\"}], \")\"}], \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"contourthickness\", \"=\", \"0.001\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\"The\", \" \", \"color\", \" \", \"of\", \" \", \"the\", \" \", \"contour\", \" \", \n");
	fprintf(a,"     RowBox[{\"lines\", \".\", \" \", \"Enter\"}], \" \", \"three\", \" \", \"values\", \" \", \n");
	fprintf(a,"     \"between\", \" \", \"0\", \" \", \"and\", \" \", \"1\", \" \", \"for\", \" \", \"red\"}], \",\",\n");
	fprintf(a,"     \" \", \n");
	fprintf(a,"    RowBox[{\"green\", \" \", \"and\", \" \", \"blue\", \" \", \"color\", \" \", \n");
	fprintf(a,"     RowBox[{\"component\", \".\"}]}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"contourcolor\", \"=\", \n");
	fprintf(a,"    RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"     RowBox[{\"0\", \",\", \"0\", \",\", \"0\"}], \"]\"}]}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\"The\", \" \", \"opacity\", \" \", \"of\", \" \", \"the\", \" \", \"contour\", \" \", \n");
	fprintf(a,"     RowBox[{\"lines\", \".\", \" \", \"1\"}], \" \", \"means\", \" \", \"fully\", \" \", \n");
	fprintf(a,"     \"opaque\"}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"0\", \" \", \"means\", \" \", \"totally\", \" \", \"transparent\", \" \", \n");
	fprintf(a,"     RowBox[{\n");
	fprintf(a,"      RowBox[{\"(\", \"invisible\", \")\"}], \".\"}]}]}], \" \", \"*)\"}], \n");
	fprintf(a,"  \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"contouropacity\", \"=\", \"0.7\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\"The\", \" \", \"pattern\", \" \", \"of\", \" \", \"the\", \" \", \"contour\", \" \", \n");
	fprintf(a,"    RowBox[{\"lines\", \".\", \" \", \"Large\"}], \" \", \"values\", \" \", \"like\", \" \", \n");
	fprintf(a,"    \"1000\", \" \", \"mean\", \" \", \"no\", \" \", \n");
	fprintf(a,"    RowBox[{\"dashing\", \".\", \" \", \"e\", \".\", \"g\", \".\", \" \", \"0.01\"}], \" \", \n");
	fprintf(a,"    \"results\", \" \", \"in\", \" \", \"dashed\", \" \", \n");
	fprintf(a,"    RowBox[{\"lines\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"contourdash\", \"=\", \"1000\"}], \";\"}]}]], \"Input\"]\n");
	fprintf(a,"}, Closed]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[\"3D Plot Properties\", \"Subsection\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = False}, \n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell], \n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]], \n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Draw\", \" \", \"Tick\", \" \", \"labels\", \" \", \"on\", \" \", \"the\", \" \", \"Z\", \" \", \n");
	fprintf(a,"    \"axis\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"labelsz\", \"=\", \".\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Draw\", \" \", \"a\", \" \", \"rectangular\", \" \", \"box\", \" \", \"around\", \" \", \n");
	fprintf(a,"    \"the\", \" \", \"3\", \"D\", \" \", \"plot\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"boxed\", \"=\", \".\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"line\", \" \", \"width\", \" \", \"of\", \" \", \"this\", \" \", \"box\"}], \n");
	fprintf(a,"   \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"boxthickness\", \"=\", \"0.002\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\"The\", \" \", \"color\", \" \", \"of\", \" \", \"this\", \" \", \"box\"}], \" \", \n");
	fprintf(a,"   \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"boxcolor\", \"=\", \n");
	fprintf(a,"    RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"     RowBox[{\"0\", \",\", \"0\", \",\", \"0\"}], \"]\"}]}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\"The\", \" \", \"opacity\", \" \", \"of\", \" \", \"this\", \" \", \"box\"}], \" \", \n");
	fprintf(a,"   \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"boxopacity\", \"=\", \"1\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\"May\", \" \", \"be\", \" \", \"either\", \" \", \n");
	fprintf(a,"     RowBox[{\"{\", \n");
	fprintf(a,"      RowBox[{\"y\", \",\", \"z\"}], \"}\"}], \" \", \"with\", \" \", \"y\"}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"z\", \" \", \"=\", \" \", \n");
	fprintf(a,"     RowBox[{\"1\", \" \", \"/\", \" \", \n");
	fprintf(a,"      RowBox[{\"-\", \"1\"}]}]}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"or\", \" \", \"Automatic\"}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"or\", \" \", \"None\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"xaxisposition\", \"=\", \n");
	fprintf(a,"    RowBox[{\"{\", \n");
	fprintf(a,"     RowBox[{\n");
	fprintf(a,"      RowBox[{\"-\", \"1\"}], \",\", \n");
	fprintf(a,"      RowBox[{\"-\", \"1\"}]}], \"}\"}]}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\"May\", \" \", \"be\", \" \", \"either\", \" \", \n");
	fprintf(a,"     RowBox[{\"{\", \n");
	fprintf(a,"      RowBox[{\"x\", \",\", \"z\"}], \"}\"}], \" \", \"with\", \" \", \"x\"}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"z\", \" \", \"=\", \" \", \n");
	fprintf(a,"     RowBox[{\"1\", \" \", \"/\", \" \", \n");
	fprintf(a,"      RowBox[{\"-\", \"1\"}]}]}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"or\", \" \", \"Automatic\"}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"or\", \" \", \"None\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"yaxisposition\", \"=\", \n");
	fprintf(a,"    RowBox[{\"{\", \n");
	fprintf(a,"     RowBox[{\n");
	fprintf(a,"      RowBox[{\"-\", \"1\"}], \",\", \n");
	fprintf(a,"      RowBox[{\"-\", \"1\"}]}], \"}\"}]}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");

	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\"May\", \" \", \"be\", \" \", \"either\", \" \", \n");
	fprintf(a,"     RowBox[{\"{\", \n");
	fprintf(a,"      RowBox[{\"x\", \",\", \"y\"}], \"}\"}], \" \", \"with\", \" \", \"x\"}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"y\", \" \", \"=\", \" \", \n");
	fprintf(a,"     RowBox[{\"1\", \" \", \"/\", \" \", \n");
	fprintf(a,"      RowBox[{\"-\", \"1\"}]}]}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"or\", \" \", \"Automatic\"}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"or\", \" \", \"None\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"zaxisposition\", \"=\", \"None\"}], \";\"}]}]], \"Input\"],\n");

	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"thickness\", \" \", \"of\", \" \", \"the\", \" \", \"axes\", \" \", \"and\", \n");
	fprintf(a,"    \" \", \"tick\", \" \", \"marks\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"axesthickness\", \"=\", \"0.005\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"label\", \" \", \"to\", \" \", \"plot\", \" \", \"on\", \" \", \"the\", \" \", \n");
	fprintf(a,"    \"Z\", \" \", \"axis\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");

	if (m_sLabelZ == NULL)
		fprintf(a,"   RowBox[{\"zlabel\", \"=\", \"\\\"\\<Occurrence\\>\\\"\"}], \";\"}]}]], \"Input\"],\n");
			else fprintf(a,"   RowBox[{\"zlabel\", \"=\", \"\\\"\\<%s\\>\\\"\"}], \";\"}]}]], \"Input\"],\n",m_sLabelZ);

	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"How\", \" \", \"many\", \" \", \"grid\", \" \", \"lines\", \" \", \"to\", \" \", \"draw\", \" \",\n");
	fprintf(a,"     \"in\", \" \", \"X\", \" \", \"direction\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"gridxcount\", \"=\", \"25\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Color\", \" \", \"for\", \" \", \"the\", \" \", \"grid\", \" \", \"lines\", \" \", \"in\", \" \",\n");
	fprintf(a,"     \"X\", \" \", \"direction\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"gridxcolor\", \"=\", \n");
	fprintf(a,"    RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"     RowBox[{\"0\", \",\", \"0\", \",\", \"0\"}], \"]\"}]}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Thickness\", \" \", \"of\", \" \", \"the\", \" \", \"grid\", \" \", \"lines\", \" \", \"in\", \n");
	fprintf(a,"    \" \", \"X\", \" \", \"direction\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"gridxthickness\", \"=\", \"0.001\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Opacity\", \" \", \"of\", \" \", \"the\", \" \", \"grid\", \" \", \"lines\", \" \", \"in\", \n");
	fprintf(a,"    \" \", \"X\", \" \", \"direction\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"gridxopacity\", \"=\", \"1\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"How\", \" \", \"many\", \" \", \"grid\", \" \", \"lines\", \" \", \"to\", \" \", \"draw\", \" \",\n");
	fprintf(a,"     \"in\", \" \", \"Y\", \" \", \"direction\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"gridycount\", \"=\", \"25\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Color\", \" \", \"for\", \" \", \"the\", \" \", \"grid\", \" \", \"lines\", \" \", \"in\", \" \",\n");
	fprintf(a,"     \"Y\", \" \", \"direction\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"gridycolor\", \"=\", \n");
	fprintf(a,"    RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"     RowBox[{\"0\", \",\", \"0\", \",\", \"0\"}], \"]\"}]}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Thickness\", \" \", \"of\", \" \", \"the\", \" \", \"grid\", \" \", \"lines\", \" \", \"in\", \n");
	fprintf(a,"    \" \", \"Y\", \" \", \"direction\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"gridythickness\", \"=\", \"0.001\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Opacity\", \" \", \"of\", \" \", \"the\", \" \", \"grid\", \" \", \"lines\", \" \", \"in\", \n");
	fprintf(a,"    \" \", \"Y\", \" \", \"direction\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"gridyopacity\", \"=\", \"1\"}], \";\"}]}]], \"Input\"]\n");
	fprintf(a,"}, Closed]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[\"Plot Range\", \"Subsection\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = False}, \n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell], \n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]], \n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"\n");

	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\"(*\", \" \",\n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"    \"Here\", \" \", \"you\", \" \", \"can\", \" \", \"\\\"\\<zoom in\\>\\\"\", \" \", \"into\", \" \", \n");
	fprintf(a,"     \"a\", \" \", \"specified\", \" \", \"area\", \" \", \"of\", \" \", \"your\", \" \", \n");
	fprintf(a,"     RowBox[{\"plot\", \".\", \" \", \"The\"}], \" \", \"range\", \" \", \"of\", \" \", \"your\", \n");
	fprintf(a,"     \" \", \"data\", \" \", \"was\", \" \", \"originally\", \" \", \"X1\"}], \"=\", \"%f\"}], \n",m_fMinVal[0]);
	fprintf(a,"   \",\", \" \", \n");
	fprintf(a,"   RowBox[{\"Y1\", \"=\", \"%f\"}], \",\", \" \", \n",m_fMinVal[1]);
	fprintf(a,"   RowBox[{\"X2\", \"=\", \"%f\"}], \",\", \" \", \n",m_fMaxVal[0]);
	fprintf(a,"   RowBox[{\"Y2\", \"=\", \"%f.\"}]}], \" \", \"*)\"}]], \"Input\"],\n",m_fMaxVal[1]);
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"plotrangeX1\", \"=\", \"%f\"}], \";\"}]], \"Input\"],\n",m_fMinVal[0]);
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"plotrangeY1\", \"=\", \"%f\"}], \";\"}]], \"Input\"],\n",m_fMinVal[1]);
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"plotrangeX2\", \"=\", \"%f\"}], \";\"}]], \"Input\"],\n",m_fMaxVal[0]);
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"plotrangeY2\", \"=\", \"%f\"}], \";\"}]], \"Input\"]\n",m_fMaxVal[1]);
	fprintf(a,"}, Closed]],\n");

	fprintf(a,"\n");
	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[\"Manual Color/Contour Scale\", \"Subsection\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = False}, \n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell], \n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]], \n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"\n");

	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"     RowBox[{\n");
	fprintf(a,"     \"Enter\", \" \", \"a\", \" \", \"value\", \" \", \"here\", \" \", \"ONLY\", \" \", \"if\", \n");
	fprintf(a,"      \" \", \"you\", \" \", \"want\", \" \", \"to\", \" \", \"create\", \" \", \"different\", \n");
	fprintf(a,"      \" \", \"plots\", \" \", \"with\", \" \", \"the\", \" \", \"same\", \" \", \"color\", \" \", \n");
	fprintf(a,"      RowBox[{\"scale\", \".\", \"\\[IndentingNewLine]\", \"Then\"}], \" \", \"no\", \" \", \n");
	fprintf(a,"      \"rescaling\", \" \", \"for\", \" \", \"smoothened\", \" \", \"data\", \" \", \"will\", \n");
	fprintf(a,"      \" \", \n");
	fprintf(a,"      RowBox[{\"occur\", \".\", \" \", \"Data\"}], \" \", \"for\", \" \", \"this\", \" \", \n");
	fprintf(a,"      RowBox[{\"plot\", \":\", \" \", \"maxcolorvalue\"}]}], \" \", \"=\", \" \", \n");
	fprintf(a,"     \"%f\"}], \";\", \" \", \n",m_fMaxEntry);
	fprintf(a,"    RowBox[{\"mincolorvalue\", \" \", \"=\", \" \", \n");
	fprintf(a,"     RowBox[{\"%c\", \"%f\"}]}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n",(m_fMinEntry<0)?'-':'+',m_fMinEntry);
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\n");

	if (manrange)
		fprintf(a,"    RowBox[{\"mincolorvalue\", \"=\", \"%f\"}], \";\"}], \"\\[IndentingNewLine]\", \n",0);
			else fprintf(a,"    RowBox[{\"mincolorvalue\", \"=.\"}], \";\"}], \"\\[IndentingNewLine]\", \n");

	fprintf(a,"   RowBox[{\n");

	if (manrange)
		fprintf(a,"    RowBox[{\"maxcolorvalue\", \"=\", \"%f\"}], \";\"}]}]}]], \"Input\"],\n",m_fMaxEntry);
			else fprintf(a,"    RowBox[{\"maxcolorvalue\", \"=.\"}], \";\"}]}]}]], \"Input\"]\n");

	fprintf(a,"\n");

/*	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\"Enter\", \" \", \"both\", \" \", \n");
	fprintf(a,"    RowBox[{\"(\", \"!!\", \")\"}], \" \", \"value\", \" \", \"here\", \" \", \"ONLY\", \" \", \n");
	fprintf(a,"    \"if\", \" \", \"you\", \" \", \"want\", \" \", \"to\", \" \", \"create\", \" \", \"different\",\n");
	fprintf(a,"     \" \", \"plots\", \" \", \"with\", \" \", \"the\", \" \", \"same\", \" \", \"contour\", \" \", \n");
	fprintf(a,"    RowBox[{\"lines\", \".\", \"\\[IndentingNewLine]\", \"This\"}], \" \", \"one\", \" \", \n");
	fprintf(a,"    \"also\", \" \", \"rescales\", \" \", \"the\", \" \", \"coloring\", \" \", \"if\", \" \", \n");
	fprintf(a,"    \"no\", \" \", \n");
	fprintf(a,"    RowBox[{\"min\", \"/\", \"maxcolorvalues\"}], \" \", \"are\", \" \", \n");
	fprintf(a,"    RowBox[{\"specified\", \".\", \" \", \"Data\"}], \" \", \"for\", \" \", \"this\", \" \", \n");
	fprintf(a,"    \"plot\", \" \", \"see\", \" \", \n");
	fprintf(a,"    RowBox[{\"above\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\"mincolorvalue\", \"=.\"}], \";\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\"maxcolorvalue\", \"=.\"}], \";\"}]}]}]], \"Input\"]\n");*/

	fprintf(a,"}, Closed]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[\"Frame Thickness / Font Sizes\", \"Subsection\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = False}, \n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell], \n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]], \n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Choose\", \" \", \"the\", \" \", \"font\", \" \", \"size\", \" \", \"for\", \" \", \"the\", \n");
	fprintf(a,"    \" \", \"tick\", \" \", \"labels\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"ticklabelsize\", \"=\", \"28\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Choose\", \" \", \"the\", \" \", \"font\", \" \", \"size\", \" \", \"for\", \" \", \"the\", \n");
	fprintf(a,"    \" \", \"axis\", \" \", \"labels\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"labelsize\", \"=\", \"32\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Choose\", \" \", \"the\", \" \", \"thickness\", \" \", \"of\", \" \", \"the\", \" \", \n");
	fprintf(a,"    \"frame\", \" \", \"and\", \" \", \"the\", \" \", \"axis\", \" \", \"ticks\"}], \" \", \"*)\"}],\n");
	fprintf(a,"   \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"framethickness\", \"=\", \"0.005\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Define\", \" \", \"if\", \" \", \"space\", \" \", \"should\", \" \", \"be\", \" \", \"left\", \n");
	fprintf(a,"    \" \", \"between\", \" \", \"the\", \" \", \"plot\", \" \", \"and\", \" \", \"the\", \" \", \n");
	fprintf(a,"    RowBox[{\"axes\", \".\", \" \", \"Set\"}], \" \", \"this\", \" \", \"to\", \" \", \n");
	fprintf(a,"    \"\\\"\\<Automatic\\>\\\"\", \" \", \"or\", \" \", \"to\", \" \", \n");
	fprintf(a,"    RowBox[{\"\\\"\\<0\\>\\\"\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"framemargins\", \"=\", \"0\"}], \";\"}]}]], \"Input\"]\n");
	fprintf(a,"}, Closed]]\n");
	fprintf(a,"}, Closed]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[\"Axis Properties\", \"Section\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = False}, \n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell], \n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]], \n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[\"General Properties\", \"Subsection\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = False}, \n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell], \n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]], \n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"length\", \" \", \"of\", \" \", \"the\", \" \", \"major\", \" \", \"ticks\", \n");
	fprintf(a,"    \" \", \"on\", \" \", \"the\", \" \", \n");
	fprintf(a,"    RowBox[{\"axes\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"majorticklength\", \"=\", \"0.03\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"length\", \" \", \"of\", \" \", \"the\", \" \", \"minor\", \" \", \"ticks\", \n");
	fprintf(a,"    \" \", \n");
	fprintf(a,"    RowBox[{\"(\", \n");
	fprintf(a,"     RowBox[{\"the\", \" \", \"ones\", \" \", \"without\", \" \", \"tick\", \" \", \"labels\"}],\n");
	fprintf(a,"      \")\"}], \" \", \"on\", \" \", \"the\", \" \", \n");
	fprintf(a,"    RowBox[{\"axes\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"minorticklength\", \"=\", \"0.015\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"    \"Controls\", \" \", \"if\", \" \", \"the\", \" \", \"major\", \" \", \"ticks\", \" \", \n");
	fprintf(a,"     \"point\", \" \", \"to\", \" \", \"the\", \" \", \"inside\", \" \", \"or\", \" \", \n");
	fprintf(a,"     RowBox[{\"outside\", \".\", \" \", \"0\"}], \" \", \"is\", \" \", \"inside\"}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"0.5\", \" \", \"is\", \" \", \"symmetrical\"}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"1\", \" \", \"is\", \" \", \n");
	fprintf(a,"     RowBox[{\"outside\", \".\"}]}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"majortickshift\", \"=\", \"0.5\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"    \"Controls\", \" \", \"if\", \" \", \"the\", \" \", \"minor\", \" \", \"ticks\", \" \", \n");
	fprintf(a,"     \"point\", \" \", \"to\", \" \", \"the\", \" \", \"inside\", \" \", \"or\", \" \", \n");
	fprintf(a,"     RowBox[{\"outside\", \".\", \" \", \"0\"}], \" \", \"is\", \" \", \"inside\"}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"0.5\", \" \", \"is\", \" \", \"symmetrical\"}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"1\", \" \", \"is\", \" \", \n");
	fprintf(a,"     RowBox[{\"outside\", \".\"}]}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"minortickshift\", \"=\", \"0.5\"}], \";\"}]}]], \"Input\"]\n");
	fprintf(a,"}, Closed]],\n");
	fprintf(a,"\n");

	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[\"Mesh Properties\", \"Subsection\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = False}, \n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell], \n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]],\n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"thickness\", \" \", \"of\", \" \", \"the\", \" \", \"mesh\", \" \", \n");
	fprintf(a,"    \"perpendicular\", \" \", \"to\", \" \", \"the\", \" \", \"X\", \" \", \"axis\", \" \", \n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"     RowBox[{\"(\", \n");
	fprintf(a,"      RowBox[{\n");
	fprintf(a,"      \"relative\", \" \", \"to\", \" \", \"the\", \" \", \"overall\", \" \", \"plot\", \" \", \n");
	fprintf(a,"       \"width\"}], \")\"}], \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"meshthicknessX\", \"=\", \"0.002\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \",\n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"    \"The\", \" \", \"color\", \" \", \"of\", \" \", \"the\", \" \", \"mesh\", \" \", \n");
	fprintf(a,"     \"perpendicular\", \" \", \"to\", \" \", \"the\", \" \", \"X\", \" \", \n");
	fprintf(a,"     RowBox[{\"axis\", \".\", \" \", \"Enter\"}], \" \", \"three\", \" \", \"values\", \" \", \n");
	fprintf(a,"     \"between\", \" \", \"0\", \" \", \"and\", \" \", \"1\", \" \", \"for\", \" \", \"red\"}], \",\",\n");
	fprintf(a,"     \" \", \n");
	fprintf(a,"    RowBox[{\"green\", \" \", \"and\", \" \", \"blue\", \" \", \"color\", \" \", \n");
	fprintf(a,"     RowBox[{\"component\", \".\"}]}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"meshcolorX\", \"=\", \n");
	fprintf(a,"    RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"     RowBox[{\"0\", \",\", \"0\", \",\", \"0\"}], \"]\"}]}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"    \"The\", \" \", \"opacity\", \" \", \"of\", \" \", \"the\", \" \", \"mesh\", \" \", \n");
	fprintf(a,"     \"perpendicular\", \" \", \"to\", \" \", \"the\", \" \", \"X\", \" \", \n");
	fprintf(a,"     RowBox[{\"axis\", \".\", \" \", \"1\"}], \" \", \"means\", \" \", \"fully\", \" \", \n");
	fprintf(a,"     \"opaque\"}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"0\", \" \", \"means\", \" \", \"totally\", \" \", \"transparent\", \" \", \n");
	fprintf(a,"     RowBox[{\n");
	fprintf(a,"      RowBox[{\"(\", \"invisible\", \")\"}], \".\"}]}]}], \" \", \"*)\"}], \n");
	fprintf(a,"  \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"meshopacityX\", \"=\", \"0.5\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"thickness\", \" \", \"of\", \" \", \"the\", \" \", \"mesh\", \" \", \n");
	fprintf(a,"    \"perpendicular\", \" \", \"to\", \" \", \"the\", \" \", \"Y\", \" \", \"axis\", \" \", \n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"     RowBox[{\"(\", \n");
	fprintf(a,"      RowBox[{\n");
	fprintf(a,"      \"relative\", \" \", \"to\", \" \", \"the\", \" \", \"overall\", \" \", \"plot\", \" \", \n");
	fprintf(a,"       \"width\"}], \")\"}], \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"meshthicknessY\", \"=\", \"0.002\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"    \"The\", \" \", \"color\", \" \", \"of\", \" \", \"the\", \" \", \"mesh\", \" \", \n");
	fprintf(a,"     \"perpendicular\", \" \", \"to\", \" \", \"the\", \" \", \"Y\", \" \", \n");
	fprintf(a,"     RowBox[{\"axis\", \".\", \" \", \"Enter\"}], \" \", \"three\", \" \", \"values\", \" \", \n");
	fprintf(a,"     \"between\", \" \", \"0\", \" \", \"and\", \" \", \"1\", \" \", \"for\", \" \", \"red\"}], \",\",\n");
	fprintf(a,"     \" \", \n");
	fprintf(a,"    RowBox[{\"green\", \" \", \"and\", \" \", \"blue\", \" \", \"color\", \" \", \n");
	fprintf(a,"     RowBox[{\"component\", \".\"}]}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"meshcolorY\", \"=\", \n");
	fprintf(a,"    RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"     RowBox[{\"0\", \",\", \"0\", \",\", \"0\"}], \"]\"}]}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"    \"The\", \" \", \"opacity\", \" \", \"of\", \" \", \"the\", \" \", \"mesh\", \" \", \n");
	fprintf(a,"     \"perpendicular\", \" \", \"to\", \" \", \"the\", \" \", \"Y\", \" \", \n");
	fprintf(a,"     RowBox[{\"axis\", \".\", \" \", \"1\"}], \" \", \"means\", \" \", \"fully\", \" \", \n");
	fprintf(a,"     \"opaque\"}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"0\", \" \", \"means\", \" \", \"totally\", \" \", \"transparent\", \" \", \n");
	fprintf(a,"     RowBox[{\n");
	fprintf(a,"      RowBox[{\"(\", \"invisible\", \")\"}], \".\"}]}]}], \" \", \"*)\"}], \n");
	fprintf(a,"  \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"meshopacityY\", \"=\", \"0.5\"}], \";\"}]}]], \"Input\"]\n");
	fprintf(a,"}, Closed  ]],\n");

	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[\"X Axis\", \"Subsection\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = False}, \n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell], \n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]], \n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"label\", \" \", \"to\", \" \", \"appear\", \" \", \"on\", \" \", \"the\", \" \", \n");
	fprintf(a,"    \"x\", \" \", \"axis\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"xlabel\", \"=\", \"\\\"\\<%s\\>\\\"\"}], \n",m_sLabelX);
	fprintf(a,"   \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"lowest\", \" \", \"tick\", \" \", \"value\", \" \", \"for\", \" \", \"the\", \n");
	fprintf(a,"    \" \", \"x\", \" \", \n");
	fprintf(a,"    RowBox[{\"axis\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"tickminX\", \":=\", \n");
	fprintf(a,"    RowBox[{\"maxsig\", \"[\", \n");
	fprintf(a,"     RowBox[{\"plotrangeX1\", \",\", \"2\"}], \"]\"}]}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"highest\", \" \", \"tick\", \" \", \"value\", \" \", \"for\", \" \", \"the\", \n");
	fprintf(a,"    \" \", \"x\", \" \", \n");
	fprintf(a,"    RowBox[{\"axis\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"tickmaxX\", \":=\", \n");
	fprintf(a,"    RowBox[{\"minsig\", \"[\", \n");
	fprintf(a,"     RowBox[{\"plotrangeX2\", \",\", \"2\"}], \"]\"}]}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Print\", \" \", \"this\", \" \", \"number\", \" \", \"of\", \" \", \"digits\", \" \", \n");
	fprintf(a,"    \"after\", \" \", \"the\", \" \", \"decimal\", \" \", \"point\", \" \", \"of\", \" \", \"the\", \n");
	fprintf(a,"    \" \", \"tick\", \" \", \n");
	fprintf(a,"    RowBox[{\"labels\", \".\", \" \", \"0\"}], \" \", \"means\", \" \", \"no\", \" \", \n");
	fprintf(a,"    \"decimal\", \" \", \n");
	fprintf(a,"    RowBox[{\"point\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"tickprecX\", \":=\", \n");
	fprintf(a,"    RowBox[{\"prec\", \"[\", \"plotrangeX2\", \"]\"}]}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Use\", \" \", \"this\", \" \", \"count\", \" \", \"of\", \" \", \"major\", \" \", \"ticks\", \n");
	fprintf(a,"    \" \", \n");
	fprintf(a,"    RowBox[{\"(\", \n");
	fprintf(a,"     RowBox[{\"with\", \" \", \"tick\", \" \", \"label\"}], \")\"}], \" \", \"on\", \" \", \n");
	fprintf(a,"    \"the\", \" \", \"x\", \" \", \n");
	fprintf(a,"    RowBox[{\"axis\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"majorticksX\", \"=\", \"%d\"}], \";\"}]}]], \"Input\"],\n",majorx);
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\"How\", \" \", \"many\", \" \", \"minor\", \" \", \"ticks\", \" \", \n");
	fprintf(a,"    RowBox[{\"(\", \n");
	fprintf(a,"     RowBox[{\"without\", \" \", \"tick\", \" \", \"label\"}], \")\"}], \" \", \"to\", \" \", \n");
	fprintf(a,"    \"show\", \" \", \"PER\", \" \", \"MAJOR\", \" \", \n");
	fprintf(a,"    RowBox[{\"TICK\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"minorticksX\", \"=\", \"%d\"}], \";\"}]}]], \"Input\"],\n",minorx);
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"A\", \" \", \"character\", \" \", \"string\", \" \", \"to\", \" \", \"stand\", \" \", \n");
	fprintf(a,"    \"before\", \" \", \"each\", \" \", \"tick\", \" \", \n");
	fprintf(a,"    RowBox[{\"label\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"ticklabelprefixX\", \"=\", \"\\\"\\<\\>\\\"\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"A\", \" \", \"character\", \" \", \"string\", \" \", \"to\", \" \", \"stand\", \" \", \n");
	fprintf(a,"    \"after\", \" \", \"each\", \" \", \"tick\", \" \", \n");
	fprintf(a,"    RowBox[{\"label\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"ticklabelsuffixX\", \"=\", \"\\\"\\<\\>\\\"\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Show\", \" \", \"tick\", \" \", \"marks\", \" \", \"on\", \" \", \"both\", \" \", \"the\", \" \",\n");
	fprintf(a,"     \"top\", \" \", \"X\", \" \", \"axis\", \" \", \"and\", \" \", \"the\", \" \", \"bottom\", \" \",\n");
	fprintf(a,"     \"X\", \" \", \n");
	fprintf(a,"    RowBox[{\"axis\", \"?\", \" \", \"Set\"}], \" \", \"to\", \" \", \"\\\"\\<1\\>\\\"\", \" \", \"if\",\n");
	fprintf(a,"     \" \", \"desired\", \" \", \"or\", \" \", \"to\", \" \", \"\\\"\\<.\\>\\\"\", \" \", \"if\", \" \", \n");
	fprintf(a,"    RowBox[{\"not\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"ticksbothsidesX\", \"=\", \"1\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Show\", \" \", \"tick\", \" \", \"labels\", \" \", \"on\", \" \", \"both\", \" \", \"the\", \n");
	fprintf(a,"    \" \", \"top\", \" \", \"X\", \" \", \"axis\", \" \", \"and\", \" \", \"the\", \" \", \"bottom\", \n");
	fprintf(a,"    \" \", \"X\", \" \", \n");
	fprintf(a,"    RowBox[{\"axis\", \"?\", \" \", \"Set\"}], \" \", \"to\", \" \", \"\\\"\\<1\\>\\\"\", \" \", \"if\",\n");
	fprintf(a,"     \" \", \"desired\", \" \", \"or\", \" \", \"to\", \" \", \"\\\"\\<.\\>\\\"\", \" \", \"if\", \" \", \n");
	fprintf(a,"    RowBox[{\"not\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"labelsbothsidesX\", \"=.\"}], \";\"}]}]], \"Input\"]\n");
	fprintf(a,"}, Closed]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[\"Y Axis\", \"Subsection\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = False}, \n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell], \n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]], \n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"label\", \" \", \"to\", \" \", \"appear\", \" \", \"on\", \" \", \"the\", \" \", \n");
	fprintf(a,"    \"Y\", \" \", \"axis\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"ylabel\", \"=\", \"\\\"\\<%s\\>\\\"\"}], \n",m_sLabelY);
	fprintf(a,"   \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"lowest\", \" \", \"tick\", \" \", \"value\", \" \", \"for\", \" \", \"the\", \n");
	fprintf(a,"    \" \", \"Y\", \" \", \n");
	fprintf(a,"    RowBox[{\"axis\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"tickminY\", \":=\", \n");
	fprintf(a,"    RowBox[{\"maxsig\", \"[\", \n");
	fprintf(a,"     RowBox[{\"plotrangeY1\", \",\", \"2\"}], \"]\"}]}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"highest\", \" \", \"tick\", \" \", \"value\", \" \", \"for\", \" \", \"the\", \n");
	fprintf(a,"    \" \", \"Y\", \" \", \n");
	fprintf(a,"    RowBox[{\"axis\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"tickmaxY\", \":=\", \n");
	fprintf(a,"    RowBox[{\"minsig\", \"[\", \n");
	fprintf(a,"     RowBox[{\"plotrangeY2\", \",\", \"2\"}], \"]\"}]}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Print\", \" \", \"this\", \" \", \"number\", \" \", \"of\", \" \", \"digits\", \" \", \n");
	fprintf(a,"    \"after\", \" \", \"the\", \" \", \"decimal\", \" \", \"point\", \" \", \"of\", \" \", \"the\", \n");
	fprintf(a,"    \" \", \"tick\", \" \", \n");
	fprintf(a,"    RowBox[{\"labels\", \".\", \" \", \"0\"}], \" \", \"means\", \" \", \"no\", \" \", \n");
	fprintf(a,"    \"decimal\", \" \", \n");
	fprintf(a,"    RowBox[{\"point\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"tickprecY\", \":=\", \n");
	fprintf(a,"    RowBox[{\"prec\", \"[\", \"plotrangeY2\", \"]\"}]}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Use\", \" \", \"this\", \" \", \"count\", \" \", \"of\", \" \", \"major\", \" \", \"ticks\", \n");
	fprintf(a,"    \" \", \n");
	fprintf(a,"    RowBox[{\"(\", \n");
	fprintf(a,"     RowBox[{\"with\", \" \", \"tick\", \" \", \"label\"}], \")\"}], \" \", \"on\", \" \", \n");
	fprintf(a,"    \"the\", \" \", \"Y\", \" \", \n");
	fprintf(a,"    RowBox[{\"axis\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"majorticksY\", \"=\", \"%d\"}], \";\"}]}]], \"Input\"],\n",majory);
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\"How\", \" \", \"many\", \" \", \"minor\", \" \", \"ticks\", \" \", \n");
	fprintf(a,"    RowBox[{\"(\", \n");
	fprintf(a,"     RowBox[{\"without\", \" \", \"tick\", \" \", \"label\"}], \")\"}], \" \", \"to\", \" \", \n");
	fprintf(a,"    \"show\", \" \", \"PER\", \" \", \"MAJOR\", \" \", \n");
	fprintf(a,"    RowBox[{\"TICK\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"minorticksY\", \"=\", \"%d\"}], \";\"}]}]], \"Input\"],\n",minory);
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"A\", \" \", \"character\", \" \", \"string\", \" \", \"to\", \" \", \"stand\", \" \", \n");
	fprintf(a,"    \"before\", \" \", \"each\", \" \", \"tick\", \" \", \n");
	fprintf(a,"    RowBox[{\"label\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"ticklabelprefixY\", \"=\", \"\\\"\\<\\>\\\"\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"A\", \" \", \"character\", \" \", \"string\", \" \", \"to\", \" \", \"stand\", \" \", \n");
	fprintf(a,"    \"after\", \" \", \"each\", \" \", \"tick\", \" \", \n");
	fprintf(a,"    RowBox[{\"label\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"ticklabelsuffixY\", \"=\", \"\\\"\\<\\>\\\"\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Show\", \" \", \"tick\", \" \", \"marks\", \" \", \"on\", \" \", \"both\", \" \", \"the\", \" \",\n");
	fprintf(a,"     \"left\", \" \", \"Y\", \" \", \"axis\", \" \", \"and\", \" \", \"the\", \" \", \"right\", \" \",\n");
	fprintf(a,"     \"Y\", \" \", \n");
	fprintf(a,"    RowBox[{\"axis\", \"?\", \" \", \"Set\"}], \" \", \"to\", \" \", \"\\\"\\<1\\>\\\"\", \" \", \"if\",\n");
	fprintf(a,"     \" \", \"desired\", \" \", \"or\", \" \", \"to\", \" \", \"\\\"\\<.\\>\\\"\", \" \", \"if\", \" \", \n");
	fprintf(a,"    RowBox[{\"not\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"ticksbothsidesY\", \"=\", \"1\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Show\", \" \", \"tick\", \" \", \"labels\", \" \", \"on\", \" \", \"both\", \" \", \"the\", \n");
	fprintf(a,"    \" \", \"left\", \" \", \"Y\", \" \", \"axis\", \" \", \"and\", \" \", \"the\", \" \", \"right\", \n");
	fprintf(a,"    \" \", \"Y\", \" \", \n");
	fprintf(a,"    RowBox[{\"axis\", \"?\", \" \", \"Set\"}], \" \", \"to\", \" \", \"\\\"\\<1\\>\\\"\", \" \", \"if\",\n");
	fprintf(a,"     \" \", \"desired\", \" \", \"or\", \" \", \"to\", \" \", \"\\\"\\<.\\>\\\"\", \" \", \"if\", \" \", \n");
	fprintf(a,"    RowBox[{\"not\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"labelsbothsidesY\", \"=.\"}], \";\"}]}]], \"Input\"]\n");
	fprintf(a,"}, Closed]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[\"Z Axis\", \"Subsection\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = False},\n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell], \n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]], \n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"label\", \" \", \"to\", \" \", \"appear\", \" \", \"on\", \" \", \"the\", \" \", \n");
	fprintf(a,"    \"Z\", \" \", \"axis\"}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	if (m_sLabelZ == NULL)
		fprintf(a,"   RowBox[{\"zlabel\", \"=\", \"\\\"\\<Occurrence\\>\\\"\"}], \n");
			else fprintf(a,"   RowBox[{\"zlabel\", \"=\", \"\\\"\\<%s\\>\\\"\"}], \n",m_sLabelZ);
	fprintf(a,"   \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"lowest\", \" \", \"tick\", \" \", \"value\", \" \", \"for\", \" \", \"the\", \n");
	fprintf(a,"    \" \", \"Z\", \" \", \n");
	fprintf(a,"    RowBox[{\"axis\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"tickminZ\", \":=\", \n");
	fprintf(a,"    RowBox[{\"maxsig\", \"[\", \n");
	fprintf(a,"     RowBox[{\"datminexp\", \",\", \"2\"}], \"]\"}]}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"highest\", \" \", \"tick\", \" \", \"value\", \" \", \"for\", \" \", \"the\", \n");
	fprintf(a,"    \" \", \"Z\", \" \", \n");
	fprintf(a,"    RowBox[{\"axis\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"tickmaxZ\", \":=\", \n");
	fprintf(a,"    RowBox[{\"minsig\", \"[\", \n");
	fprintf(a,"     RowBox[{\"datmaxexp\", \",\", \"2\"}], \"]\"}]}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Print\", \" \", \"this\", \" \", \"number\", \" \", \"of\", \" \", \"digits\", \" \", \n");
	fprintf(a,"    \"after\", \" \", \"the\", \" \", \"decimal\", \" \", \"point\", \" \", \"of\", \" \", \"the\", \n");
	fprintf(a,"    \" \", \"tick\", \" \", \n");
	fprintf(a,"    RowBox[{\"labels\", \".\", \" \", \"0\"}], \" \", \"means\", \" \", \"no\", \" \", \n");
	fprintf(a,"    \"decimal\", \" \", \n");
	fprintf(a,"    RowBox[{\"point\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"tickprecZ\", \":=\", \n");
	fprintf(a,"    RowBox[{\"prec\", \"[\", \"datmaxexp\", \"]\"}]}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Use\", \" \", \"this\", \" \", \"count\", \" \", \"of\", \" \", \"major\", \" \", \"ticks\", \n");
	fprintf(a,"    \" \", \n");
	fprintf(a,"    RowBox[{\"(\", \n");
	fprintf(a,"     RowBox[{\"with\", \" \", \"tick\", \" \", \"label\"}], \")\"}], \" \", \"on\", \" \", \n");
	fprintf(a,"    \"the\", \" \", \"Z\", \" \", \n");
	fprintf(a,"    RowBox[{\"axis\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"majorticksZ\", \"=\", \"5\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\"How\", \" \", \"many\", \" \", \"minor\", \" \", \"ticks\", \" \", \n");
	fprintf(a,"    RowBox[{\"(\", \n");
	fprintf(a,"     RowBox[{\"without\", \" \", \"tick\", \" \", \"label\"}], \")\"}], \" \", \"to\", \" \", \n");
	fprintf(a,"    \"show\", \" \", \"PER\", \" \", \"MAJOR\", \" \", \n");
	fprintf(a,"    RowBox[{\"TICK\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"minorticksZ\", \"=\", \"3\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"A\", \" \", \"character\", \" \", \"string\", \" \", \"to\", \" \", \"stand\", \" \", \n");
	fprintf(a,"    \"before\", \" \", \"each\", \" \", \"tick\", \" \", \n");
	fprintf(a,"    RowBox[{\"label\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"ticklabelprefixZ\", \"=\", \"\\\"\\<\\>\\\"\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"A\", \" \", \"character\", \" \", \"string\", \" \", \"to\", \" \", \"stand\", \" \", \n");
	fprintf(a,"    \"after\", \" \", \"each\", \" \", \"tick\", \" \", \n");
	fprintf(a,"    RowBox[{\"label\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"ticklabelsuffixZ\", \"=\", \"\\\"\\<\\>\\\"\"}], \";\"}]}]], \"Input\"]\n");
	fprintf(a,"}, Closed]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[\"Legend Axis\", \"Subsection\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = False}, \n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell], \n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]], \n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"lowest\", \" \", \"tick\", \" \", \"value\", \" \", \"for\", \" \", \"the\", \n");
	fprintf(a,"    \" \", \"legend\", \" \", \"X\", \" \", \n");
	fprintf(a,"    RowBox[{\"axis\", \".\", \" \", \"\\\"\\<datmin\\>\\\"\"}], \" \", \"sets\", \" \", \"this\", \n");
	fprintf(a,"    \" \", \"automatically\", \" \", \"to\", \" \", \"the\", \" \", \"lowest\", \" \", \"plot\", \n");
	fprintf(a,"    \" \", \n");
	fprintf(a,"    RowBox[{\"element\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"tickminL\", \":=\", \n");
	fprintf(a,"    RowBox[{\"maxsig\", \"[\", \n");
	fprintf(a,"     RowBox[{\"datmin\", \",\", \"3\"}], \"]\"}]}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"highest\", \" \", \"tick\", \" \", \"value\", \" \", \"for\", \" \", \"the\", \n");
	fprintf(a,"    \" \", \"legend\", \" \", \"Y\", \" \", \n");
	fprintf(a,"    RowBox[{\"axis\", \".\", \" \", \"\\\"\\<datmax\\>\\\"\"}], \" \", \"sets\", \" \", \"this\", \n");
	fprintf(a,"    \" \", \"automatically\", \" \", \"to\", \" \", \"the\", \" \", \"highest\", \" \", \"plot\", \n");
	fprintf(a,"    \" \", \n");
	fprintf(a,"    RowBox[{\"element\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"tickmaxL\", \":=\", \n");
	fprintf(a,"    RowBox[{\"minsig\", \"[\", \n");
	fprintf(a,"     RowBox[{\"datmax\", \",\", \"3\"}], \"]\"}]}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Print\", \" \", \"this\", \" \", \"number\", \" \", \"of\", \" \", \"digits\", \" \", \n");
	fprintf(a,"    \"after\", \" \", \"the\", \" \", \"decimal\", \" \", \"point\", \" \", \"of\", \" \", \"the\", \n");
	fprintf(a,"    \" \", \"tick\", \" \", \n");
	fprintf(a,"    RowBox[{\"labels\", \".\", \" \", \"0\"}], \" \", \"means\", \" \", \"no\", \" \", \n");
	fprintf(a,"    \"decimal\", \" \", \n");
	fprintf(a,"    RowBox[{\"point\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"tickprecL\", \":=\", \n");
	fprintf(a,"    RowBox[{\"prec\", \"[\", \"datmax\", \"]\"}]}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"Use\", \" \", \"this\", \" \", \"count\", \" \", \"of\", \" \", \"major\", \" \", \"ticks\", \n");
	fprintf(a,"    \" \", \n");
	fprintf(a,"    RowBox[{\"(\", \n");
	fprintf(a,"     RowBox[{\"with\", \" \", \"tick\", \" \", \"label\"}], \")\"}], \" \", \"on\", \" \", \n");
	fprintf(a,"    \"the\", \" \", \"legend\", \" \", \"X\", \" \", \n");
	fprintf(a,"    RowBox[{\"axis\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"majorticksL\", \"=\", \"5\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\"How\", \" \", \"many\", \" \", \"minor\", \" \", \"ticks\", \" \", \n");
	fprintf(a,"    RowBox[{\"(\", \n");
	fprintf(a,"     RowBox[{\"without\", \" \", \"tick\", \" \", \"label\"}], \")\"}], \" \", \"to\", \" \", \n");
	fprintf(a,"    \"show\", \" \", \"PER\", \" \", \"MAJOR\", \" \", \n");
	fprintf(a,"    RowBox[{\"TICK\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"minorticksL\", \"=\", \"3\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"length\", \" \", \"of\", \" \", \"the\", \" \", \"major\", \" \", \"ticks\", \n");
	fprintf(a,"    \" \", \"on\", \" \", \"the\", \" \", \"legend\", \" \", \"X\", \" \", \n");
	fprintf(a,"    RowBox[{\"axis\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"majorticklengthL\", \"=\", \"0.015\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"   \"The\", \" \", \"length\", \" \", \"of\", \" \", \"the\", \" \", \"minor\", \" \", \"ticks\", \n");
	fprintf(a,"    \" \", \n");
	fprintf(a,"    RowBox[{\"(\", \n");
	fprintf(a,"     RowBox[{\"the\", \" \", \"ones\", \" \", \"without\", \" \", \"tick\", \" \", \"labels\"}],\n");
	fprintf(a,"      \")\"}], \" \", \"on\", \" \", \"the\", \" \", \"legend\", \" \", \"X\", \" \", \n");
	fprintf(a,"    RowBox[{\"axis\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"minorticklengthL\", \"=\", \"0.008\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"    \"Controls\", \" \", \"if\", \" \", \"the\", \" \", \"major\", \" \", \"ticks\", \" \", \n");
	fprintf(a,"     \"point\", \" \", \"to\", \" \", \"the\", \" \", \"inside\", \" \", \"or\", \" \", \n");
	fprintf(a,"     RowBox[{\"outside\", \".\", \" \", \"0\"}], \" \", \"is\", \" \", \"inside\"}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"0.5\", \" \", \"is\", \" \", \"symmetrical\"}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"1\", \" \", \"is\", \" \", \n");
	fprintf(a,"     RowBox[{\"outside\", \".\"}]}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"majortickshiftL\", \"=\", \"0\"}], \";\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"    \"Controls\", \" \", \"if\", \" \", \"the\", \" \", \"minor\", \" \", \"ticks\", \" \", \n");
	fprintf(a,"     \"point\", \" \", \"to\", \" \", \"the\", \" \", \"inside\", \" \", \"or\", \" \", \n");
	fprintf(a,"     RowBox[{\"outside\", \".\", \" \", \"0\"}], \" \", \"is\", \" \", \"inside\"}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"0.5\", \" \", \"is\", \" \", \"symmetrical\"}], \",\", \" \", \n");
	fprintf(a,"    RowBox[{\"1\", \" \", \"is\", \" \", \n");
	fprintf(a,"     RowBox[{\"outside\", \".\"}]}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\n");
	fprintf(a,"   RowBox[{\"minortickshiftL\", \"=\", \"0\"}], \";\"}]}]], \"Input\"]\n");
	fprintf(a,"}, Closed]]\n");
	fprintf(a,"}, Closed]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[\"Lines / Rectangles / Functions\", \"Section\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = False}, \n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell], \n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]], \n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[\"Function Drawing\", \"Subsection\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = False}, \n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell], \n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]], \n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"\n");


	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"    \"Here\", \" \", \"some\", \" \", \"functions\", \" \", \"of\", \" \", \"the\", \" \", \"form\",\n");
	fprintf(a,"      \" \", \"f\", \n");
	fprintf(a,"     RowBox[{\"(\", \"x\", \")\"}]}], \" \", \"=\", \" \", \n");
	fprintf(a,"    RowBox[{\"y\", \" \", \"are\", \" \", \n");
	fprintf(a,"     RowBox[{\"defined\", \".\", \" \", \"Uncomment\"}], \" \", \"them\", \" \", \"for\", \" \",\n");
	fprintf(a,"      \"testing\", \" \", \"and\", \" \", \"define\", \" \", \"your\", \" \", \"own\", \" \", \n");
	fprintf(a,"     RowBox[{\"ones\", \".\"}]}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\"(*\", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\"AppendTo\", \"[\", \n");
	fprintf(a,"     RowBox[{\"funcX\", \",\", \n");
	fprintf(a,"      RowBox[{\"{\", \n");
	fprintf(a,"       RowBox[{\n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"(\", \n");
	fprintf(a,"           RowBox[{\"x\", \"/\", \"%f\"}], \")\"}], \"^\", \"2\"}], \"*\", \n",m_fMaxVal[0]);
	fprintf(a,"         \"%f\"}], \",\", \n",m_fMaxVal[1]);
	fprintf(a,"        RowBox[{\"{\", \n");
	fprintf(a,"         RowBox[{\"%f\", \",\", \"%f\"}], \"}\"}], \",\", \n",m_fMinVal[0],m_fMaxVal[0]);
	fprintf(a,"        RowBox[{\"{\", \n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"Thickness\", \"[\", \"0.01\", \"]\"}], \",\", \n");
	fprintf(a,"          RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"           RowBox[{\"1\", \",\", \"1\", \",\", \"0\"}], \"]\"}], \",\", \n");
	fprintf(a,"          RowBox[{\"Dashing\", \"[\", \"0.02\", \"]\"}]}], \"}\"}]}], \"}\"}]}], \"]\"}], \n");
	fprintf(a,"    \";\"}], \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\"(*\", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\"AppendTo\", \"[\", \n");
	fprintf(a,"     RowBox[{\"funcX\", \",\", \n");
	fprintf(a,"      RowBox[{\"{\", \n");
	fprintf(a,"       RowBox[{\n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"(\", \n");
	fprintf(a,"           RowBox[{\"x\", \"/\", \"%f\"}], \")\"}], \"^\", \"3\"}], \"*\", \n",m_fMaxVal[0]);
	fprintf(a,"         \"%f\"}], \",\", \n",m_fMaxVal[1]);
	fprintf(a,"        RowBox[{\"{\", \n");
	fprintf(a,"         RowBox[{\"%f\", \",\", \"%f\"}], \"}\"}], \",\", \n",m_fMinVal[0]+(m_fMaxVal[0]-m_fMinVal[0])*0.2,m_fMaxVal[0]-(m_fMaxVal[0]-m_fMinVal[0])*0.2);
	fprintf(a,"        RowBox[{\"{\", \n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"Thickness\", \"[\", \"0.02\", \"]\"}], \",\", \n");
	fprintf(a,"          RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"           RowBox[{\"0\", \",\", \"1\", \",\", \"1\"}], \"]\"}]}], \"}\"}]}], \"}\"}]}], \n");
	fprintf(a,"     \"]\"}], \";\"}], \"*)\"}]}]], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\n");
	fprintf(a,"    \"Here\", \" \", \"some\", \" \", \"functions\", \" \", \"of\", \" \", \"the\", \" \", \"form\",\n");
	fprintf(a,"      \" \", \"f\", \n");
	fprintf(a,"     RowBox[{\"(\", \"y\", \")\"}]}], \" \", \"=\", \" \", \n");
	fprintf(a,"    RowBox[{\"x\", \" \", \"are\", \" \", \n");
	fprintf(a,"     RowBox[{\"defined\", \".\", \" \", \"Uncomment\"}], \" \", \"them\", \" \", \"for\", \" \",\n");
	fprintf(a,"      \"testing\", \" \", \"and\", \" \", \"define\", \" \", \"your\", \" \", \"own\", \" \", \n");
	fprintf(a,"     RowBox[{\"ones\", \".\"}]}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\"(*\", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\"AppendTo\", \"[\", \n");
	fprintf(a,"     RowBox[{\"funcY\", \",\", \n");
	fprintf(a,"      RowBox[{\"{\", \n");
	fprintf(a,"       RowBox[{\n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\"Sin\", \"[\", \n");
	fprintf(a,"          RowBox[{\n");
	fprintf(a,"           RowBox[{\"y\", \"/\", \"%f\"}], \"*\", \"10\", \"Pi\"}], \"]\"}], \"*\", \n",m_fMaxVal[1]);
	fprintf(a,"         \"%f\"}], \",\", \n",m_fMaxVal[0]*0.5);
	fprintf(a,"        RowBox[{\"{\", \n");
	fprintf(a,"         RowBox[{\"%f\", \",\", \"%f\"}], \"}\"}], \",\", \n",m_fMinVal[1],m_fMaxVal[1]);
	fprintf(a,"        RowBox[{\"{\", \n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"Thickness\", \"[\", \"0.01\", \"]\"}], \",\", \n");
	fprintf(a,"          RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"           RowBox[{\"0\", \",\", \"0\", \",\", \"0\"}], \"]\"}]}], \"}\"}]}], \"}\"}]}], \n");
	fprintf(a,"     \"]\"}], \";\"}], \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\"(*\", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\"AppendTo\", \"[\", \n");
	fprintf(a,"     RowBox[{\"funcY\", \",\", \n");
	fprintf(a,"      RowBox[{\"{\", \n");
	fprintf(a,"       RowBox[{\n");
	fprintf(a,"        RowBox[{\"y\", \"*\", \"%f\"}], \",\", \n",m_fMaxVal[0]/m_fMaxVal[1]);
	fprintf(a,"        RowBox[{\"{\", \n");
	fprintf(a,"         RowBox[{\"%f\", \",\", \"%f\"}], \"}\"}], \",\", \n",m_fMinVal[1],m_fMaxVal[1]*0.5);
	fprintf(a,"        RowBox[{\"{\", \n");
	fprintf(a,"         RowBox[{\n");
	fprintf(a,"          RowBox[{\"Thickness\", \"[\", \"0.02\", \"]\"}], \",\", \n");
	fprintf(a,"          RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"           RowBox[{\"1\", \",\", \"0\", \",\", \"0\"}], \"]\"}]}], \"}\"}]}], \"}\"}]}], \n");
	fprintf(a,"     \"]\"}], \";\"}], \"*)\"}]}]], \"Input\"]\n");
	fprintf(a,"}, Closed]],\n");
	fprintf(a,"\n");

	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"Cell[\"Rectangle and Circle Drawing\", \"Subsection\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = False}, \n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell], \n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]], \n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\"Draw\", \" \", \"Rectangles\", \" \", \"and\", \" \", \n");
	fprintf(a,"    RowBox[{\"Circles\", \".\", \" \", \"Uncomment\"}], \" \", \"the\", \" \", \"examples\", \n");
	fprintf(a,"    \" \", \"below\", \" \", \"and\", \" \", \"try\", \" \", \"it\", \" \", \n");
	fprintf(a,"    RowBox[{\"out\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\"(*\", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\"AppendTo\", \"[\", \n");
	fprintf(a,"     RowBox[{\"prim\", \",\", \n");
	fprintf(a,"      RowBox[{\"Graphics\", \"[\", \n");
	fprintf(a,"       RowBox[{\"{\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\"EdgeForm\", \"[\", \n");
	fprintf(a,"          RowBox[{\"{\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"Thickness\", \"[\", \"0.006\", \"]\"}], \",\", \n");
	fprintf(a,"            RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"             RowBox[{\"1\", \",\", \"0\", \",\", \"1\"}], \"]\"}], \",\", \n");
	fprintf(a,"            RowBox[{\"Dashing\", \"[\", \"0.02\", \"]\"}]}], \"}\"}], \"]\"}], \",\", \n");
	fprintf(a,"         RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"          RowBox[{\"1\", \",\", \"0\", \",\", \"0\"}], \"]\"}], \",\", \n");
	fprintf(a,"         RowBox[{\"Opacity\", \"[\", \"0\", \"]\"}], \",\", \n");


	fprintf(a,"         RowBox[{\"Rectangle\", \"[\", \n");
	fprintf(a,"          RowBox[{\n");
	fprintf(a,"           RowBox[{\"{\", \n");
	fprintf(a,"            RowBox[{\n");
	fprintf(a,"             RowBox[{\"%f\"}], \",\", \n",m_fMinVal[0]);
	fprintf(a,"             RowBox[{\"%f\"}]}], \"}\"}], \",\", \n",m_fMinVal[1]);
	fprintf(a,"           RowBox[{\"{\", \n");
	fprintf(a,"            RowBox[{\"%f\", \",\", \"%f\"}], \"}\"}]}], \"]\"}]}], \n",m_fMaxVal[0]*0.7,m_fMaxVal[1]*0.3);

	fprintf(a,"        \"}\"}], \"]\"}]}], \"]\"}], \";\", \"\\[IndentingNewLine]\", \n");
	fprintf(a,"    RowBox[{\"AppendTo\", \"[\", \n");
	fprintf(a,"     RowBox[{\"prim\", \",\", \n");
	fprintf(a,"      RowBox[{\"Graphics\", \"[\", \n");
	fprintf(a,"       RowBox[{\"{\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\"EdgeForm\", \"[\", \n");
	fprintf(a,"          RowBox[{\"{\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"Thickness\", \"[\", \"0.004\", \"]\"}], \",\", \n");
	fprintf(a,"            RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"             RowBox[{\"0\", \",\", \"0\", \",\", \"0\"}], \"]\"}]}], \"}\"}], \"]\"}], \",\", \n");
	fprintf(a,"         RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"          RowBox[{\"0\", \",\", \"1\", \",\", \"0\"}], \"]\"}], \",\", \n");
	fprintf(a,"         RowBox[{\"Opacity\", \"[\", \"1\", \"]\"}], \",\", \n");
	fprintf(a,"         RowBox[{\"Disk\", \"[\", \n");
	fprintf(a,"          RowBox[{\n");
	fprintf(a,"           RowBox[{\"{\", \n");
	fprintf(a,"            RowBox[{\"%f\", \",\", \"%f\"}], \"}\"}], \",\", RowBox[{\"{\", RowBox[{\"%f\", \",\", \"%f\"}], \"}\"}]}], \"]\"}]}], \"}\"}], \n",m_fMinVal[0]+(m_fMaxVal[0]-m_fMinVal[0])*0.41666,(m_fMaxVal[0]-m_fMinVal[0])*0.0625,(m_fMaxVal[0]-m_fMinVal[0])*0.0625,(m_fMaxVal[1]-m_fMinVal[1])*0.0625);
	fprintf(a,"       \"]\"}]}], \"]\"}], \";\"}], \"*)\"}]\n");

	if (m_oaCircles.GetSize() > 0)
	{
		fprintf(a,",\"\\[IndentingNewLine]\", \n");
		fprintf(a,"RowBox[{\n");
		for (z=0;z<m_oaCircles.GetSize();z++)
		{
			fprintf(a,"   RowBox[{\n");
			fprintf(a,"    RowBox[{\"AppendTo\", \"[\", \n");
			fprintf(a,"     RowBox[{\"prim\", \",\", \n");
			fprintf(a,"      RowBox[{\"Graphics\", \"[\", \n");
			fprintf(a,"       RowBox[{\"{\", \n");
			fprintf(a,"        RowBox[{\n");
			fprintf(a,"         RowBox[{\"EdgeForm\", \"[\", \n");
			fprintf(a,"          RowBox[{\"{\", \n");
			fprintf(a,"           RowBox[{\n");
			fprintf(a,"            RowBox[{\"Thickness\", \"[\", \"0.002\", \"]\"}], \",\", \n");
			fprintf(a,"            RowBox[{\"RGBColor\", \"[\", \n");
			fprintf(a,"             RowBox[{\"0\", \",\", \"0\", \",\", \"0\"}], \"]\"}]}], \"}\"}], \"]\"}], \",\", \n");
			fprintf(a,"         RowBox[{\"RGBColor\", \"[\", \n");
			fprintf(a,"          RowBox[{\"%f\", \",\", \"%f\", \",\", \"%f\"}], \"]\"}], \",\", \n",((CMathematicaCircle*)m_oaCircles[z])->m_fColorR,((CMathematicaCircle*)m_oaCircles[z])->m_fColorG,((CMathematicaCircle*)m_oaCircles[z])->m_fColorB);
			fprintf(a,"         RowBox[{\"Opacity\", \"[\", \"1\", \"]\"}], \",\", \n");
			fprintf(a,"         RowBox[{\"Disk\", \"[\", \n");
			fprintf(a,"          RowBox[{\n");
			fprintf(a,"           RowBox[{\"{\", \n");
			fprintf(a,"            RowBox[{\"%f\", \",\", \"%f\"}], \"}\"}], \",\", \"%f\"}], \"]\"}]}], \"}\"}], \n",((CMathematicaCircle*)m_oaCircles[z])->m_fPosX,((CMathematicaCircle*)m_oaCircles[z])->m_fPosY,((CMathematicaCircle*)m_oaCircles[z])->m_fRadius);
			fprintf(a,"       \"]\"}]}], \"]\"}], \";\"}]");
			if (z < m_oaCircles.GetSize()-1)
				fprintf(a,", \"\\[IndentingNewLine]\", ");
			fprintf(a,"\n");
		}
		fprintf(a,"}]\n");
	}
	fprintf(a,"}]], \"Input\"]\n");
	fprintf(a,"}, Closed  ]],\n");

	fprintf(a,"\n");
	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[\"Line Drawing\", \"Subsection\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = False},\n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell], \n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]], \n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\n");
	fprintf(a," RowBox[{\n");
	fprintf(a,"  RowBox[{\"(*\", \" \", \n");
	fprintf(a,"   RowBox[{\"Draw\", \" \", \n");
	fprintf(a,"    RowBox[{\"Lines\", \".\", \" \", \"Uncomment\"}], \" \", \"the\", \" \", \"example\", \" \",\n");
	fprintf(a,"     \"below\", \" \", \"and\", \" \", \"try\", \" \", \"it\", \" \", \n");
	fprintf(a,"    RowBox[{\"out\", \".\"}]}], \" \", \"*)\"}], \"\\[IndentingNewLine]\", \n");
	fprintf(a,"  RowBox[{\"(*\", \n");
	fprintf(a,"   RowBox[{\n");
	fprintf(a,"    RowBox[{\"AppendTo\", \"[\", \n");
	fprintf(a,"     RowBox[{\"prim\", \",\", \n");
	fprintf(a,"      RowBox[{\"Graphics\", \"[\", \n");
	fprintf(a,"       RowBox[{\"{\", \n");
	fprintf(a,"        RowBox[{\n");
	fprintf(a,"         RowBox[{\"Thickness\", \"[\", \"0.01\", \"]\"}], \",\", \n");
	fprintf(a,"         RowBox[{\"RGBColor\", \"[\", \n");
	fprintf(a,"          RowBox[{\"0\", \",\", \"0\", \",\", \"1\"}], \"]\"}], \",\", \n");
	fprintf(a,"         RowBox[{\"Dashing\", \"[\", \"1000\", \"]\"}], \",\", \n");
	fprintf(a,"         RowBox[{\"Line\", \"[\", \n");
	fprintf(a,"          RowBox[{\"{\", \n");
	fprintf(a,"           RowBox[{\n");
	fprintf(a,"            RowBox[{\"{\", \n");
	fprintf(a,"             RowBox[{\"%f\", \",\", \"%f\"}], \"}\"}], \",\", \n",m_fMinVal[0],m_fMinVal[1]);
	fprintf(a,"            RowBox[{\"{\", \n");
	fprintf(a,"             RowBox[{\"%f\", \",\", \"%f\"}], \"}\"}], \",\", \n",m_fMaxVal[0]*0.4,m_fMaxVal[1]*0.8);
	fprintf(a,"            RowBox[{\"{\", \n");
	fprintf(a,"             RowBox[{\"%f\", \",\", \"%f\"}], \"}\"}]}], \"}\"}], \"]\"}]}],\n",m_fMaxVal[0]*0.4,m_fMinVal[1]);
	fprintf(a,"         \"}\"}], \"]\"}]}], \"]\"}], \";\"}], \"*)\"}]}]], \"Input\"]\n");
	fprintf(a,"}, Closed]]\n");
	fprintf(a,"}, Closed]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[CellGroupData[{\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[\"\\<\\\n");
	fprintf(a,"Plot output. Click \\\"Evaluation -> Evaluate Notebook\\\" to draw/redraw!\\\n");
	fprintf(a,"\\>\", \"Section\",\n");
	fprintf(a," CellDingbat->DynamicModuleBox[{$CellContext`state$$ = True}, \n");
	fprintf(a,"   OpenerBox[\n");
	fprintf(a,"    Dynamic[$CellContext`state$$, (FrontEndExecute[{\n");
	fprintf(a,"        FrontEnd`SelectionMove[\n");
	fprintf(a,"         FrontEnd`ButtonNotebook[], All, ButtonCell], \n");
	fprintf(a,"        FrontEndToken[\"OpenCloseGroup\"]}]; $CellContext`state$$ = #)& ]], \n");
	fprintf(a,"   DynamicModuleValues :> {}]],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\"plot\"], \"Input\"],\n");
	fprintf(a,"\n");
	fprintf(a,"Cell[BoxData[\"plotlegend\"], \"Input\"]\n");
	fprintf(a,"}, Open  ]]\n");
	fprintf(a,"}\n");
	fprintf(a,"]\n");

	fclose(a);
	BTOUT;
}

void C2DF::NormRDF(double n)
{
	BTIN;
	int x, y;
	double f;

	for (y=0;y<m_iRes[1];y++)
	{
		f = n/(double)m_pStepsY[y];
		for (x=0;x<m_iRes[0];x++)
			m_pBin[x+y*m_iRes[0]] *= f;
	}
	BTOUT;
}


void C2DF::StepY(int y)
{
	BXIN;
	m_pStepsY[y]++;
	BXOUT;
}


void C2DF::CalcMaxEntry()
{
	BTIN;
	int z, z2;
	
	m_fMaxEntry = -99999999.0;
	m_fMinEntry =  99999999.0;
	for (z=0;z<m_iRes[1];z++)
	{
		for (z2=0;z2<m_iRes[0];z2++)
		{
			if (m_pBin[z*m_iRes[0]+z2] > m_fMaxEntry)
				m_fMaxEntry = m_pBin[z*m_iRes[0]+z2];
			if (m_pBin[z*m_iRes[0]+z2] < m_fMinEntry)
				m_fMinEntry = m_pBin[z*m_iRes[0]+z2];
		}
	}
	BTOUT;
}


void C2DF::SetLabelX(const char *s)
{
	if (m_sLabelX != NULL)
		delete[] m_sLabelX;

	try { m_sLabelX = new char[strlen(s)+1]; } catch(...) { m_sLabelX = NULL; }
	if (m_sLabelX == NULL) NewException((double)(strlen(s)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	strcpy(m_sLabelX,s);
}

void C2DF::SetLabelY(const char *s)
{
	if (m_sLabelY != NULL)
		delete[] m_sLabelY;

	try { m_sLabelY = new char[strlen(s)+1]; } catch(...) { m_sLabelY = NULL; }
	if (m_sLabelY == NULL) NewException((double)(strlen(s)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	strcpy(m_sLabelY,s);
}

void C2DF::SetLabelZ(const char *s)
{
	if (m_sLabelZ != NULL)
		delete[] m_sLabelZ;

	try { m_sLabelZ = new char[strlen(s)+1]; } catch(...) { m_sLabelZ = NULL; }
	if (m_sLabelZ == NULL) NewException((double)(strlen(s)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	strcpy(m_sLabelZ,s);
}

void C2DF::MakeTensorProduct(C2DF *inp)
{
	int x, y;
	double tf, *px, *py;

	try { px = new double[m_iRes[0]]; } catch(...) { px = NULL; }
	if (px == NULL) NewException((double)m_iRes[0]*sizeof(double),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	try { py = new double[m_iRes[1]]; } catch(...) { py = NULL; }
	if (py == NULL) NewException((double)m_iRes[1]*sizeof(double),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (y=0;y<m_iRes[1];y++)
	{
		tf = 0;
		for (x=0;x<m_iRes[0];x++)
			tf += inp->m_pBin[y*m_iRes[0]+x];
		tf /= (double)m_iRes[0];
		py[y] = tf;
	}
	for (x=0;x<m_iRes[0];x++)
	{
		tf = 0;
		for (y=0;y<m_iRes[1];y++)
			tf += inp->m_pBin[y*m_iRes[0]+x];
		tf /= (double)m_iRes[1];
		px[x] = tf;
	}
	for (x=0;x<m_iRes[0];x++)
	{
		for (y=0;y<m_iRes[1];y++)
			m_pBin[y*m_iRes[0]+x] = px[x] * py[y];
	}
	delete[] py;
	delete[] px;
}

void C2DF::CopyFrom(C2DF *df)
{
	long z;

	if (df->m_sLabelX != NULL)
	{
		try { m_sLabelX = new char[strlen(df->m_sLabelX)+1]; } catch(...) { m_sLabelX = NULL; }
		if (m_sLabelX == NULL) NewException((double)(strlen(df->m_sLabelX)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		strcpy(m_sLabelX,df->m_sLabelX);
	}
	if (df->m_sLabelY != NULL)
	{
		try { m_sLabelY = new char[strlen(df->m_sLabelY)+1]; } catch(...) { m_sLabelY = NULL; }
		if (m_sLabelY == NULL) NewException((double)(strlen(df->m_sLabelY)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);

		strcpy(m_sLabelY,df->m_sLabelY);
	}
	if (df->m_sLabelZ != NULL)
	{
		try { m_sLabelZ = new char[strlen(df->m_sLabelZ)+1]; } catch(...) { m_sLabelZ = NULL; }
		if (m_sLabelZ == NULL) NewException((double)(strlen(df->m_sLabelZ)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);

		strcpy(m_sLabelZ,df->m_sLabelZ);
	}
	m_iRes[0] = df->m_iRes[0];
	m_iRes[1] = df->m_iRes[1];
	m_fMinVal[0] = df->m_fMinVal[0];
	m_fMinVal[1] = df->m_fMinVal[1];
	m_fMaxVal[0] = df->m_fMaxVal[0];
	m_fMaxVal[1] = df->m_fMaxVal[1];

	try { m_fCountX = new double[m_iRes[0]]; } catch(...) { m_fCountX = NULL; }
	if (m_fCountX == NULL) NewException((double)m_iRes[0]*sizeof(double),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	try { m_fCountY = new double[m_iRes[1]]; } catch(...) { m_fCountY = NULL; }
	if (m_fCountY == NULL) NewException((double)m_iRes[1]*sizeof(double),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	try { m_pStepsY = new unsigned long[m_iRes[1]]; } catch(...) { m_pStepsY = NULL; }
	if (m_pStepsY == NULL) NewException((double)m_iRes[0]*sizeof(unsigned long),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<m_iRes[0];z++)
		m_fCountX[z] = df->m_fCountX[z];

	for (z=0;z<m_iRes[1];z++)
	{
		m_fCountY[z] = df->m_fCountY[z];
		m_pStepsY[z] = df->m_pStepsY[z];
	}

	try { m_pBin = new double[m_iRes[0]*m_iRes[1]]; } catch(...) { m_pBin = NULL; }
	if (m_pBin == NULL) NewException((double)m_iRes[0]*m_iRes[1]*sizeof(double),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<m_iRes[0]*m_iRes[1];z++)
		m_pBin[z] = df->m_pBin[z];
	m_fFac[0] = m_iRes[0] / (m_fMaxVal[0]-m_fMinVal[0]);
	m_fFac[1] = m_iRes[1] / (m_fMaxVal[1]-m_fMinVal[1]);
	m_fBinEntries = df->m_fBinEntries;
	m_fSkipEntries = df->m_fSkipEntries;
//	m_fMathematicaColorScale = df->m_fMathematicaColorScale;
//	m_fMathematicaColorOffset = df->m_fMathematicaColorOffset;
	m_pChannels[0] = df->m_pChannels[0];
	m_pChannels[1] = df->m_pChannels[1];
}

void C2DF::Subtract(C2DF *df)
{
	int z;

	if ((m_iRes[0] != df->m_iRes[0]) || (m_iRes[1] != df->m_iRes[1]))
	{
		eprintf("C2DF::Subtract(C2DF*): Resolution of CDFs must be identical.\n");
		return;
	}
	for (z=0;z<m_iRes[0]*m_iRes[1];z++)
		m_pBin[z] -= df->m_pBin[z];
}


void C2DF::WriteCombinedPlot(const char *prefix, const char *name, const char *suffix)
{
	CGrace *comb;
	char buf[256];
	int z;

	try { comb = new CGrace(); } catch(...) { comb = NULL; }
	if (comb == NULL) NewException((double)sizeof(CGrace),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	m_pChannels[0]->CalcMinMax();
	m_pChannels[1]->CalcMinMax();

	comb->SetViewport(0.15f,0.8f,0.8f,0.95f);
	comb->CurrentGraph()->m_bTicks = false;
	comb->CurrentGraph()->m_bTickLabels = false;
	comb->CurrentGraph()->m_fFrameWidth = 0.5;
	comb->SetRangeX(m_pChannels[0]->m_fMinVal,m_pChannels[0]->m_fMaxVal);
	comb->SetRangeY(/*m_pChannels[0]->m_fMinEntry*/0.0,m_pChannels[0]->m_fMaxEntry+(m_pChannels[0]->m_fMaxEntry/*-m_pChannels[0]->m_fMinEntry*/)*0.1);
	comb->MakeTicks();
	if (g_iObsChannel[0] == 0) // RDF
		comb->AddLine(m_pChannels[0]->m_fMinVal,1,m_pChannels[0]->m_fMaxVal,1,1,1);
	comb->AddDataset();
	comb->CurrentGraph()->CurrentDataset()->m_faValues.SetMaxSize(m_pChannels[0]->m_iResolution*2);
	for (z=0;z<m_pChannels[0]->m_iResolution-1;z++)
		comb->AddXYTupel(0,m_pChannels[0]->m_fMinVal+(z+0.5)*(m_pChannels[0]->m_fMaxVal-m_pChannels[0]->m_fMinVal)/m_pChannels[0]->m_iResolution,m_pChannels[0]->m_pBin[z]);

	comb->AddGraph();
	comb->SetViewport(0.8f, 0.15f, 0.95f, 0.8f);
	comb->CurrentGraph()->m_bTicks = false;
	comb->CurrentGraph()->m_bTickLabels = false;
	comb->CurrentGraph()->m_fFrameWidth = 0.5;
	comb->SetRangeX(/*m_pChannels[1]->m_fMinEntry*/0.0,m_pChannels[1]->m_fMaxEntry+(m_pChannels[1]->m_fMaxEntry/*-m_pChannels[1]->m_fMinEntry*/)*0.1);
	comb->SetRangeY(m_pChannels[1]->m_fMinVal,m_pChannels[1]->m_fMaxVal);
	comb->MakeTicks();
	if (g_iObsChannel[1] == 0) // RDF
		comb->AddLine(1,m_pChannels[1]->m_fMinVal,1,m_pChannels[1]->m_fMaxVal,1,1);
	comb->AddDataset();
	comb->CurrentGraph()->CurrentDataset()->m_faValues.SetMaxSize(m_pChannels[1]->m_iResolution*2);
	for (z=0;z<m_pChannels[1]->m_iResolution-1;z++)
		comb->AddXYTupel(0,m_pChannels[1]->m_pBin[z],m_pChannels[1]->m_fMinVal+(z+0.5)*(m_pChannels[1]->m_fMaxVal-m_pChannels[1]->m_fMinVal)/m_pChannels[1]->m_iResolution);

	comb->AddGraph();
	comb->SetViewport(0.15f, 0.15f, 0.8f, 0.8f);
	comb->CurrentGraph()->m_bTicksBothSidesX = true;
	comb->CurrentGraph()->m_bTicksBothSidesY = true;
	comb->CurrentGraph()->m_bTickInX = false;
	comb->CurrentGraph()->m_bTickInY = false;
	comb->SetLabelX(m_sLabelX);
	comb->SetLabelY(m_sLabelY);
	comb->SetRangeX(m_pChannels[0]->m_fMinVal,m_pChannels[0]->m_fMaxVal);
	comb->SetRangeY(m_pChannels[1]->m_fMinVal,m_pChannels[1]->m_fMaxVal);
	comb->MakeTicks();
	comb->AddDataset();

	strcpy(buf,prefix);
	strcat(buf,name);
	strcat(buf,suffix);
	comb->WriteAgr(buf,false);
}


void C2DF::Mirror(float plane, int channel)
{
	int x, y, t;
	double *tbin, p;

	try { tbin = new double[m_iRes[0]*m_iRes[1]]; } catch(...) { tbin = NULL; }
	if (tbin == NULL) NewException((double)m_iRes[0]*m_iRes[1]*sizeof(double),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	if (channel == 0)
	{
		p = (plane-m_fMinVal[0])*m_fFac[0];
		for (x=0;x<m_iRes[0];x++)
		{
			for (y=0;y<m_iRes[0];y++)
			{
				tbin[y*m_iRes[0]+x] = m_pBin[y*m_iRes[0]+x];
				t = (int)(2.0*p - x - 1);
				if ((t >= 0) && (t < m_iRes[0]))
				{
					tbin[y*m_iRes[0]+x] += m_pBin[y*m_iRes[0]+t];
					tbin[y*m_iRes[0]+x] /= 2.0;
				}
			}
		}
	} else if (channel == 1)
	{
		p = (plane-m_fMinVal[1])*m_fFac[1];
		for (x=0;x<m_iRes[0];x++)
		{
			for (y=0;y<m_iRes[0];y++)
			{
				tbin[y*m_iRes[0]+x] = m_pBin[y*m_iRes[0]+x];
				t = (int)(2*p - y - 1);
				if ((t >= 0) && (t < m_iRes[1]))
				{
					tbin[y*m_iRes[0]+x] += m_pBin[t*m_iRes[0]+x];
					tbin[y*m_iRes[0]+x] /= 2.0;
				}
			}
		}
	}

	delete[] m_pBin;
	m_pBin = tbin;
}

void C2DF::SwapAxes()
{
	char *t;
	int x, y;
	double *pd;
	double d;
	int i;
	CDF *df;

	t = m_sLabelX;
	m_sLabelX = m_sLabelY;
	m_sLabelY = t;

	try { pd = new double[m_iRes[0]*m_iRes[1]]; } catch(...) { pd = NULL; }
	if (pd == NULL) NewException((double)m_iRes[0]*m_iRes[1]*sizeof(double),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (y=0;y<m_iRes[1];y++)
		for (x=0;x<m_iRes[0];x++)
			pd[x*m_iRes[1]+y] = m_pBin[y*m_iRes[0]+x];
	delete[] m_pBin;
	m_pBin = pd;
	d = m_fMinVal[0];
	m_fMinVal[0] = m_fMinVal[1];
	m_fMinVal[1] = d;
	d = m_fMaxVal[0];
	m_fMaxVal[0] = m_fMaxVal[1];
	m_fMaxVal[1] = d;
	i = m_iRes[0];
	m_iRes[0] = m_iRes[1];
	m_iRes[1] = i;
	df = m_pChannels[0];
	m_pChannels[0] = m_pChannels[1];
	m_pChannels[1] = df;
	d = m_fFac[0];
	m_fFac[0] = m_fFac[1];
	m_fFac[1] = d;
}

void C2DF::NormalizeXCount()
{
	int x, y;

	for (x=0;x<m_iRes[0];x++)
		for (y=0;y<m_iRes[1];y++)
			if (m_fCountX[x] != 0)
				m_pBin[y*m_iRes[0]+x] /= m_fCountX[x];
}

void C2DF::NormalizeYCount()
{
	int x, y;

	for (y=0;y<m_iRes[1];y++)
		for (x=0;x<m_iRes[0];x++)
			if (m_fCountY[y] != 0)
				m_pBin[y*m_iRes[0]+x] /= m_fCountY[y];
}

void C2DF::AddCircle(double x, double y, double r, double cr, double cg, double cb)
{
	CMathematicaCircle *c;

	try { c = new CMathematicaCircle(); } catch(...) { c = NULL; }
	if (c == NULL) NewException((double)sizeof(CMathematicaCircle),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	c->m_fPosX = x;
	c->m_fPosY = y;
	c->m_fRadius = r;
	c->m_fColorR = cr;
	c->m_fColorG = cg;
	c->m_fColorB = cb;
	m_oaCircles.Add(c);
}

double C2DF::GetValue(double x, double y)
{
	BXIN;
	double rx, ry, r;
	int ix, iy;

	if ((x < m_fMinVal[0]) || (y < m_fMinVal[1]) || (x > m_fMaxVal[0]) || (y > m_fMaxVal[1]))
	{
		BXOUT;
		return 0;
	}

	rx = (x-m_fMinVal[0])*m_fFac[0] - 0.5;
	ix = (int)floor(rx);
	if (ix < 0)
	{
		ix = 0;
		rx = 0;
	} else if (ix > m_iRes[0]-2)
	{
		ix = m_iRes[0]-2;
		rx = 1.0;
	} else
		rx -= ix;

	ry = (y-m_fMinVal[1])*m_fFac[1] - 0.5;
	iy = (int)floor(ry);
	if (iy < 0)
	{
		iy = 0;
		ry = 0;
	} else if (iy > m_iRes[1]-2)
	{
		iy = m_iRes[1]-2;
		ry = 1.0;
	} else
		ry -= iy;

	r  = m_pBin[ iy    * m_iRes[0] + ix    ] * (1.0-rx) * (1.0-ry);
	r += m_pBin[ iy    * m_iRes[0] + ix + 1] *      rx  * (1.0-ry);
	r += m_pBin[(iy+1) * m_iRes[0] + ix    ] * (1.0-rx) *      ry;
	r += m_pBin[(iy+1) * m_iRes[0] + ix + 1] *      rx  *      ry;

	BXOUT;
	return r;
}

void C2DF::AddToBin(int x, int y, double val)
{
	BXIN;

	m_fBinEntries++;
	m_pBin[y*m_iRes[0] + x] += val;
	m_fCountX[x]++;
	m_fCountY[y]++;
	BXOUT;
}

void C2DF::CalcHistogram()
{
	int z, z2, t;

	CalcMaxEntry();

	try { m_pHistogram = new double[m_iHistogramRes]; } catch(...) { m_pHistogram = NULL; }
	if (m_pHistogram == NULL) NewException((double)m_iHistogramRes*sizeof(double),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<m_iHistogramRes;z++)
		m_pHistogram[z] = 0;
	for (z=0;z<m_iRes[0]*m_iRes[1];z++)
	{
		t = (int)floor((m_pBin[z]-m_fMinEntry)/(m_fMaxEntry-m_fMinEntry)*m_iHistogramRes);
		for (z2=0;z2<t;z2++)
			m_pHistogram[z2]++;
	}
	for (z=0;z<m_iHistogramRes;z++)
		m_pHistogram[z] /= ((double)m_iRes[0])*m_iRes[1];
}

void C2DF::WriteHistogram(const char *prefix, const char *name, const char *suffix)
{
	BTIN;
	FILE *a;
	int z;
	char buf[256];
	
	strcpy(buf,prefix);
	strcat(buf,name);
	strcat(buf,suffix);
	
	a = OpenFileWrite(buf,true);

	for (z=0;z<m_iHistogramRes;z++)
		fprintf(a," %#.10G;  %#.10G\n",(z+0.5)*(m_fMaxEntry-m_fMinEntry)/m_iHistogramRes,m_pHistogram[z]);
	
	fclose(a);
	BTOUT;
}


void C2DF::Log()
{
	int z;

	for (z=0;z<m_iRes[0]*m_iRes[1];z++)
	{
		if (m_pBin[z] >= 1.0E-10)
			m_pBin[z] = log10(m_pBin[z])+10.0;
				else m_pBin[z] = 0;
	}
}
