#!/usr/bin/python

# Copyright (C) 2009, Hendrik van Antwerpen
# Copyright (C) 2010 Jelmer Vernooij <jelmer@samba.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
# This software may be used and distributed according to the terms
# of the GNU General Public License, incorporated herein by reference.

"""Bazaar plugin for integration with trac.

To have bzr notify trac of new commits, you need to configure
the location of your trac. To do this set the configuration option
``trac_project``. If ``trac_project`` is empty, nothing will be done.
"""

version_info = (1, 0, 0, 'dev', 0)
if version_info[3] == 'final':
  version_string = '%d.%d.%d' % version_info[:3]
else:
  version_string = '%d.%d.%d%s%d' % version_info
__version__ = version_string

from bzrlib import trace
from bzrlib.commands import plugin_cmds
from bzrlib.branch import Branch


def check_trac_version():
    import trac
    trac_version = (int(trac.__version__[0]), int(trac.__version__[2:4]))
    if trac_version < (0,12):
        trace.warning("bzr-trac hook needs Trac >= 0.12")


def branch_tip_hook(params):
    """This is the post_update_branch_tip hook that runs after branch updates."""
    branch    = params.branch
    old_revno = params.old_revno
    new_revno = params.new_revno

    config   = branch.get_config();
    project  = config.get_user_option("trac_project")

    if project is None:
        return

    # calculate actual new revisions, if none we stop to prevent double
    # comments at tickets
    revnos = map(str,range(old_revno+1,new_revno+1))
    if not revnos:
        return

    from bzrlib import errors, urlutils
    try:
        repository_path = branch.repository.user_transport.local_abspath(".")
    except errors.NotLocalUrl:
        return

    # calculate prefixed revnos in shared repository enviroment
    if branch.repository.is_shared():
        offset_path = urlutils.relative_url(
            branch.repository.user_transport.base,
            branch.user_transport.base).rstrip("/")
        prefix      = offset_path+','
        revnos      = map(lambda revno: prefix+revno, revnos)

    check_trac_version()
    # fire changeset_added event in Trac
    from trac.env import open_environment
    from trac.versioncontrol.api import RepositoryManager
    env = open_environment(project)
    rm = RepositoryManager(env)
    rm.notify('changeset_added', repository_path, revnos)


plugin_cmds.register_lazy("cmd_close_trac_bugs",
        [], "bzrlib.plugins.trac.commands")
Branch.hooks.install_named_hook('post_change_branch_tip',
    branch_tip_hook,
    'bzr-trac')
