# $Id: roster.tcl,v 1.129 2005/08/14 16:25:27 aleksey Exp $

namespace eval roster {
    variable undef_group_name [::msgcat::mc "Undefined"]
    variable chats_group_name [::msgcat::mc "Active Chats"]
    array set long_statusdesc [list \
	available   [::msgcat::mc "%s is available"] \
	chat        [::msgcat::mc "%s is free to chat"] \
	away        [::msgcat::mc "%s is away"] \
	xa          [::msgcat::mc "%s is extended away"] \
	dnd         [::msgcat::mc "%s doesn't want to be disturbed"] \
	invisible   [::msgcat::mc "%s is invisible"] \
	unavailable [::msgcat::mc "%s is unavailable"]]
}

proc roster::process_item {connid jid name groups subsc ask category subtype} {
    variable roster
    variable undef_group_name
    variable chats_group_name

    debugmsg roster "ROSTER_ITEM: $jid; $name; $groups; $subsc; $ask; $category; $subtype"

    set jid [tolower_node_and_domain $jid]

    if {$subsc != "remove"} {
	if {![lcontain $roster(jids,$connid) $jid]} {
	    lappend roster(jids,$connid) $jid
	}
	set ind [lsearch -exact $groups $undef_group_name]
	if {$ind >= 0} {
	    set groups [lreplace $groups $ind $ind]
	}
	set ind [lsearch -exact $groups $chats_group_name]
	if {$ind >= 0} {
	    set groups [lreplace $groups $ind $ind]
	}
	set roster(group,$connid,$jid)    $groups
	set roster(name,$connid,$jid)     $name
	set roster(subsc,$connid,$jid)    $subsc
	set roster(ask,$connid,$jid)      $ask
	set roster(category,$connid,$jid) $category
	set roster(subtype,$connid,$jid)  $subtype

	catch {unset roster(cached_category_and_subtype,$connid,$jid)}
	lassign [get_category_and_subtype $connid $jid] ccategory csubtype
	set roster(isuser,$connid,$jid) [cequal $ccategory user]

	if {[info exists roster(subscribed_name,$connid,$jid)]} {
	    if {$roster(name,$connid,$jid) == ""} {
		set roster(name,$connid,$jid) $roster(subscribed_name,$connid,$jid)
		send_item $connid $jid
	    }
	    unset roster(subscribed_name,$connid,$jid)
	}
    } else {
	lvarpop roster(jids,$connid) [lsearch $roster(jids,$connid) $jid]

	unset roster(group,$connid,$jid)
	unset roster(name,$connid,$jid)
	unset roster(subsc,$connid,$jid)
	unset roster(ask,$connid,$jid)
	unset roster(category,$connid,$jid)
	unset roster(subtype,$connid,$jid)
	unset roster(isuser,$connid,$jid)
	catch {unset roster(cached_category_and_subtype,$connid,$jid)}
    }
}

hook::add roster_item_hook [namespace current]::roster::process_item
hook::add roster_push_hook [namespace current]::roster::process_item

proc client:roster_item {connid jid name groups subsc ask category subtype} {
    hook::run roster_item_hook \
	$connid $jid $name $groups $subsc $ask $category $subtype
}

proc client:roster_push {connid jid name groups subsc ask category subtype} {
    hook::run roster_push_hook \
	$connid $jid $name $groups $subsc $ask $category $subtype
    ::redraw_roster
}

proc client:roster_cmd {connid status} {
    debugmsg roster "ROSTER_CMD: $status"
    
    if {[cequal $status END_ROSTER]} {
	::redraw_roster
    }
}

proc roster::request_roster {connid} {
    variable roster

    set roster(jids,$connid) {}
    jlib::roster_get -command client:roster_cmd -connection $connid
}

hook::add connected_hook [namespace current]::roster::request_roster 15

proc roster::get_group_jids {connid group args} {
    variable roster
    variable undef_group_name

    if {![info exists roster(jids,$connid)]} {
	return {}
    }

    set nested 0
    set delim "::"
    foreach {opt val} $args {
	switch -- $opt {
	    -nested { set nested $val }
	    -delimiter { set delim $val }
	}
    }

    set jids {}
    if {[cequal $group $undef_group_name]} {
	foreach jid $roster(jids,$connid) {
	    if {[lempty [::roster::itemconfig $connid $jid -group]]} {
		lappend jids $jid
	    }
	}
    } else {
	foreach jid $roster(jids,$connid) {
	    foreach jgroup [::roster::itemconfig $connid $jid -group] {
		if {($nested && \
			[string first "$group$delim" "$jgroup$delim"] == 0) || \
			[cequal $group $jgroup]} {
		    lappend jids $jid
		    break
		}
	    }
	}
    }
    return $jids
}

proc roster::get_jids {connid} {
    variable roster

    if {[info exists roster(jids,$connid)]} {
	return [lsort -dictionary $roster(jids,$connid)]
    } else {
	return {}
    }
}

proc roster::get_groups {connid args} {
    variable roster
    variable undef_group_name

    if {![info exists roster(jids,$connid)]} {
	return {}
    }

    set nested 0
    set delimiter "::"
    set undefined 0
    set groups {}

    foreach {opt val} $args {
	switch -- $opt {
	    -nested { set nested $val }
	    -delimiter { set delimiter $val }
	    -raw {
		if {$val} {
		    foreach jid $roster(jids,$connid) {
			set groups [concat $groups $roster(group,$connid,$jid)]
		    }		    
		    return [lrmdups $groups]
		}
	    }
	    -undefined { set undefined $val }
	}
    }

    set empty 0
    foreach jid $roster(jids,$connid) {
	set jid_groups [::roster::itemconfig $connid $jid -group]
	if {![lempty $jid_groups]} {
	    foreach group $jid_groups {
		if {$nested} {
		    set sgroup [::textutil::splitx $group $delimiter]
		} else {
		    set sgroup [list $group]
		}
		set deep [llength $sgroup]
		for {set i 0} {$i < $deep} {incr i} {
			set sgr [lrange $sgroup 0 $i]
			lappend groups [join $sgr "\u0000"]
		}
	    }
	} else {
	    set empty 1
	}
    }
    set res {}
    foreach sgroup [lsort -unique -dictionary $groups] {
	lappend res [join [split $sgroup "\u0000"] $delimiter]
    }
    if {$empty && $undefined} {
	lappend res $undef_group_name
    }

    return $res
}

proc roster::itemconfig {connid jid args} {
    variable roster

    if {[llength $args] == 1} {
	lassign $args attr
	switch -- $attr {
	    -group    {set param group}
	    -name     {set param name}
	    -subsc    {set param subsc}
	    -ask      {set param ask}
	    -category {set param category}
	    -subtype  {set param subtype}
	    -isuser   {set param isuser}
	    default {return}
	}
	if {[info exists roster($param,$connid,$jid)]} {
	    return $roster($param,$connid,$jid)
	} else {
	    return ""
	}
    } else {
	foreach {attr val} $args {
	    switch -- $attr {
		-group    {set param group}
		-name     {set param name}
		-subsc    {set param subsc}
		-ask      {set param ask}
		-category {set param category}
		-subtype  {set param subtype}
		-isuser   {set param isuser}
		default   {set param ""}
	    }
	    set roster($param,$connid,$jid) $val
	}
    }
}

proc roster::on_change_jid_presence {connid jid} {
    variable long_statusdesc
    variable roster
    
    set rjid [find_jid $connid $jid]
    debugmsg roster "$jid $rjid"

    if {$rjid != ""} {
	lassign [get_category_and_subtype $connid $rjid] category subtype
	
	if {$category == "user"} {
	    set status [get_user_status $connid $rjid]
	    set label [get_label $connid $rjid]
	    if {![catch {set desc $long_statusdesc($status)}]} {
		set_status [format $desc $label]
	    }
	    hook::run on_change_user_presence_hook $label $status
	}
    }
    ::redraw_roster
}

proc roster::find_jid {connid jid} {
    variable roster

    if {[info exists roster(category,$connid,$jid)]} {
	return $jid
    }

    lassign [get_category_and_subtype $connid $jid] category subtype
    if {$category == "user"} {
	set rjid [node_and_server_from_jid $jid]
	if {[info exists roster(category,$connid,$rjid)]} {
	    lassign [get_category_and_subtype $connid $rjid] rcategory rsubtype
	    if {$category == $rcategory} {
		return $rjid
	    }
	}
    }
    return ""
}

proc roster::get_label {connid jid} {
    set name [itemconfig $connid $jid -name]
    if {[string equal $name ""]} {
	return $jid
    } else {
	return $name
    }
}

proc roster::get_category_and_subtype {connid jid} {
    variable roster

    if {[info exists roster(cached_category_and_subtype,$connid,$jid)]} {
	return $roster(cached_category_and_subtype,$connid,$jid)
    }

    if {[info exists roster(category,$connid,$jid)] && \
	    $roster(category,$connid,$jid) != ""} {
	return [list $roster(category,$connid,$jid) $roster(subtype,$connid,$jid)]
    }
    
    return [heuristically_get_category_and_subtype $connid $jid]
}

proc roster::heuristically_get_category_and_subtype {connid jid} {
    variable roster

    set node [node_from_jid $jid]
    set updomain [lindex [split [server_from_jid $jid] .] 0]

    if {$node == ""} {
	set category service

	if {[lcontain {aim icq irc jabber jud msn pager rss serverlist \
			   sms smtp yahoo} $updomain]} {
	    set subtype $updomain
	} elseif {[cequal icqv7 $updomain]} {
	    set subtype icq
	} elseif {[cequal gg $updomain]} {
	    set subtype x-gadugadu
	} elseif {([cequal pogoda $updomain]) \
		      || ([cequal weather $updomain])} {
	    set subtype x-weather
	} else {
	    set subtype ""
	}

	set roster(cached_category_and_subtype,$connid,$jid) [list $category $subtype]
	return [list $category $subtype]
    }

    if {[resource_from_jid $jid] == ""} {
	switch -- $updomain {
	    tach -
	    conference {
		set category conference
		set subtype ""
	    }
	    irc {
		set category user
		set subtype ""
		if {[string first "%" $node] != -1} {
		    set category conference
		    set subtype irc
		}
	    }
	    default {
		set category user
		set subtype ""
	    }
	}
	set roster(cached_category_and_subtype,$connid,$jid) [list $category $subtype]
	return [list $category $subtype]
    }
    set roster(cached_category_and_subtype,$connid,$jid) {user client}
    return {user client}
}

proc roster::clean {} {
    variable roster

    array unset roster jids,*
    array unset roster group,*
    array unset roster name,*
    array unset roster subsc,*
    array unset roster ask,*
    array unset roster category,*
    array unset roster subtype,*
    array unset roster isuser,*
    array unset roster cached_category_and_subtype,*
    ::redraw_roster
}

proc roster::clean_connection {connid} {
    variable roster

    array unset roster jids,$connid
    array unset roster group,$connid,*
    array unset roster name,$connid,*
    array unset roster subsc,$connid,*
    array unset roster ask,$connid,*
    array unset roster category,$connid,*
    array unset roster subtype,$connid,*
    array unset roster isuser,$connid,*
    array unset roster cached_category_and_subtype,$connid,*

    ::redraw_roster
}

proc roster::is_user {connid jid} {
    return [cequal [lindex [get_category_and_subtype $connid $jid] 0] "user"]
}

proc roster::process_subscribed {connid from x args} {
    variable roster

    set jid [tolower_node_and_domain $from]

    set name ""
    foreach {opt val} $args {
	switch -- $opt {
	    -name { set name $val }
	}
    }

    if {([find_jid $connid $jid] == "") && ($name != "")} {
	set roster(subscribed_name,$connid,$jid) $name
    }
}

proc roster::item_to_xml {connid jid} {
    variable roster
    variable undef_group_name
    variable chats_group_name

    set grtags {}
    foreach group $roster(group,$connid,$jid) {
	if {![cequal $group $undef_group_name] && ![cequal $group $chats_group_name]} {
	    lappend grtags [jlib::wrapper:createtag group -chdata $group]
	}
    }

    set vars [list jid $jid]

    if {$roster(name,$connid,$jid) != ""} {
	lappend vars name $roster(name,$connid,$jid)
    }

#    if {$roster(category,$connid,$jid) != ""} {
#	lappend vars category $roster(category,$connid,$jid)
#	if {$roster(subtype,$connid,$jid) != ""} {
#	    lappend vars type $roster(subtype,$connid,$jid)
#	}
#    }

    return [jlib::wrapper:createtag item \
		-vars $vars \
		-subtags $grtags]
}

proc roster::send_item {connid jid} {
    jlib::send_iq set \
	[jlib::wrapper:createtag query \
	     -vars {xmlns jabber:iq:roster} \
	     -subtags [list [roster::item_to_xml $connid $jid]]] \
	-connection $connid
}

proc roster::remove_item {connid jid} {

    if {[itemconfig $connid $jid -subsc] == "bookmark"} {
	plugins::conferences::remove_bookmark $connid $jid
	return
    }

    jlib::send_iq set \
	[jlib::wrapper:createtag query \
	     -vars {xmlns jabber:iq:roster} \
	     -subtags [list [jlib::wrapper:createtag item \
				 -vars [list jid $jid \
					    subscription remove]]]] \
	-connection $connid
	
    jlib::send_presence -to $jid -type unsubscribe -connection $connid

    lassign [get_category_and_subtype $connid $jid] category subtype

    if {($category == "service") && ($jid != [jlib::connection_server $connid])} {
	jlib::send_iq set \
	    [jlib::wrapper:createtag query \
		-vars {xmlns jabber:iq:register} \
		-subtags [list [jlib::wrapper:createtag remove]]] \
	    -to $jid \
	    -connection $connid
    }
}

###############################################################################

proc roster::send_rename_group {connid name new_name} {
    variable roster
    variable undef_group_name

    if {[string equal $new_name $name]} return

    plugins::conferences::rename_group $connid $name $new_name

    set items {}

    foreach jid $roster(jids,$connid) {
	if {[itemconfig $connid $jid -subsc] == "bookmark"} {
	    continue
	}

	if {[lcontain $roster(group,$connid,$jid) $name] || \
		($name == $undef_group_name && \
		     $roster(group,$connid,$jid) == {})} {
	    set idx [lsearch -exact $roster(group,$connid,$jid) $name]
	    if {$new_name != ""} {
		set roster(group,$connid,$jid) \
		    [lreplace $roster(group,$connid,$jid) $idx $idx $new_name]
	    } else {
		set roster(group,$connid,$jid) \
		    [lreplace $roster(group,$connid,$jid) $idx $idx]
	    }
	    set roster(group,$connid,$jid) [lrmdups $roster(group,$connid,$jid)]
	    lappend items [item_to_xml $connid $jid]
	}
    }

    if {$items != {}} {
	jlib::send_iq set \
	    [jlib::wrapper:createtag query \
		 -vars {xmlns jabber:iq:roster} \
		 -subtags $items] \
	    -connection $connid
    }
}

proc roster::send_remove_users_group {connid name} {
    variable roster
    variable undef_group_name

    plugins::conferences::remove_bookmarks_group $connid $name

    set items {}

    foreach jid $roster(jids,$connid) {
	if {[itemconfig $connid $jid -subsc] == "bookmark"} {
	    continue
	}

	set groups $roster(group,$connid,$jid)
	if {(([llength $groups] == 1) && [lcontain $groups $name]) || \
		(($name == $undef_group_name) && ($groups == {}))} {
	    remove_item $connid $jid
	} elseif {[lcontain $groups $name]} {
	    set idx [lsearch -exact $groups $name]
	    set roster(group,$connid,$jid) [lreplace $groups $idx $idx]
	    lappend items [item_to_xml $connid $jid]
	}
    }

    if {$items != {}} {
	jlib::send_iq set \
	    [jlib::wrapper:createtag query \
		 -vars {xmlns jabber:iq:roster} \
		 -subtags $items] \
	    -connection $connid
    }
}

proc roster::resubscribe_group {connid name} {
    variable roster

    foreach jid $roster(jids,$connid) {
	if {[lcontain $roster(group,$connid,$jid) $name]} {
	    lassign [get_category_and_subtype $connid $jid] category type
	    if {$category == "user"} {
		jlib::send_presence \
		    -to $jid \
		    -connection $connid \
		    -type subscribe
	    }
	}
    }
}

proc roster::add_group_by_jid_regexp {name regexp} {
    variable roster

    # TODO: connid
    if {$name == ""} return

    foreach connid [jlib::connections] {
	set items {}

	foreach jid $roster(jids,$connid) {
	    if {[regexp $regexp $jid]} {
		set idx [lsearch -exact $roster(group,$connid,$jid) $name]
		lappend roster(group,$connid,$jid) $name
		set roster(group,$connid,$jid) \
		    [lrmdups $roster(group,$connid,$jid)]
		lappend items [item_to_xml $connid $jid]
	    }
	}

	if {$items != {}} {
	    jlib::send_iq set \
		[jlib::wrapper:createtag query \
		     -vars {xmlns jabber:iq:roster} \
		     -subtags $items] \
		-connection $connid
	}
    }
}



###############################################################################

proc roster::send_users {gw jid args} {
    variable roster
    global send_uc

    foreach {opt val} $args {
	switch -- $opt {
	    -connection { set connid $val }
	}
    }
    if {![info exists connid]} {
	set connid [jlib::route $jid]
    }

    set sf [$gw getframe].sw.sf
    set choices {}
    foreach uc [array names send_uc] {
        if {$send_uc($uc)} {
            lappend choices $uc
        }
    }

    destroy $gw

    set subtags {}
    set body [::msgcat::mc "Contact Information"]
    foreach choice $choices {
	lassign $choice con uc
	lappend subtags [item_to_xml $con $uc]
	set nick [roster::get_label $con $uc]
        append body "\n$nick - xmpp:$uc"
    }

    message::send_msg $jid -type normal -body $body \
	-xlist [list [jlib::wrapper:createtag x \
	                  -vars [list xmlns jabber:x:roster] \
	                  -subtags $subtags]] \
	-connection $connid
}

###############################################################################

proc roster::export_to_file {connid} {
    variable roster

    set filename [tk_getSaveFile \
		      -initialdir ~/.tkabber/ \
		      -initialfile [jlib::connection_user $connid].roster \
		      -filetypes [list \
				      [list [::msgcat::mc "Roster Files"] \
					   .roster] \
				      [list [::msgcat::mc "All Files"] *]]]
    if {$filename != ""} {
	set items {}

	foreach jid $roster(jids,$connid) {
	    lappend items [item_to_xml $connid $jid]
	}

	set fd [open $filename w]
	fconfigure $fd -encoding utf-8
	puts $fd $items
	close $fd
    }
}

proc roster::import_from_file {connid} {
    variable roster

    set filename [tk_getOpenFile \
		      -initialdir ~/.tkabber/ \
		      -initialfile [jlib::connection_user $connid].roster \
		      -filetypes [list \
				      [list [::msgcat::mc "Roster Files"] \
					   .roster] \
				      [list [::msgcat::mc "All Files"] *]]]
    if {$filename != ""} {
	set fd [open $filename r]
	fconfigure $fd -encoding utf-8
	set items [read $fd]
	close $fd

	if {$items != {}} {
	    jlib::send_iq set \
		[jlib::wrapper:createtag query \
		     -vars [list xmlns "jabber:iq:roster"] \
		     -subtags $items] \
		-connection $connid
	}
    }
}

