######################################################################
#
# $Header: /home/cvs/tkabber/tkabber/jabberlib-tclxml/jabberlib.tcl,v 1.88 2005/08/04 20:33:02 aleksey Exp $
#
# This is JabberLib (abbreviated jlib), the Tcl library for 
# use in making Jabber clients.
#
#
# Variables used in JabberLib :
#	roster(users)                : Users currently in roster
#
#	roster(group,$username)      : Groups $username is in.
#
#	roster(name,$username)       : Name of $username.
#
#	roster(subsc,$username)      : Subscription of $username 
#                                  ("to" | "from" | "both" | "")
#
#	roster(ask,$username)        : "Ask" of $username 
#                                  ("subscribe" | "unsubscribe" | "")
#
#	lib(wrap)                    : Wrap ID
#
#	lib(sck)                     : SocketName
#
#	lib(sckstats)                : Socket status, "on" or "off"
#
#	lib(disconnect)              : disconnect procedure
#
#	iq(num)                      : Next iq id-number. Sent in 
#                                  "id" attributes of <iq> packets.
#
#	iq($id)                      : Callback to run when result packet 
#                                  of $id is received.
#
#
######################################################################
#
# Procedures defined in this library
#
if {0} {
proc jlib::connect {sck server}
proc jlib::disconnect {}
proc jlib::got_stream {vars}
proc jlib::end_of_parse {}
proc jlib::outmsg {msg}
proc jlib::inmsg {}
proc jlib::clear_vars {}
proc jlib::clear_iqs {}
proc jlib::parse {xmldata}
proc jlib::parse_send_auth {cmd type data}
proc jlib::parse_send_create {cmd type data}
proc jlib::parse_roster_get {connid ispush cmd type data}
proc jlib::parse_roster_set {item cmd groups name type data}
proc jlib::parse_roster_del {item cmd type data}
proc jlib::send_iq {type xmldata args}
proc jlib::send_auth {user pass res cmd}
proc jlib::send_create {user pass name mail cmd}
proc jlib::send_msg {to args}
proc jlib::send_presence {args}
proc jlib::roster_get {args}
proc jlib::roster_set {item args}
proc jlib::roster_del {item args}
proc jlib::error {type condition args}
proc ::LOG text
proc jlib::noop args
}

lappend auto_path [file dirname [info script]]
if {![info exists use_external_tclxml] || $use_external_tclxml == 0} {
    package require -exact xml 2.0
} else {
    package require xml 2.0
}
package require sha1
package require msgcat


namespace eval jlib {

    package require streamerror 1.0

    # Load XML:Wrapper
    source [file join [file dirname [info script]] wrapper.tcl]

    #set lib(wrap) [wrapper:new \
    #    	       [namespace current]::got_stream \
    #    	       [namespace current]::end_of_parse \
    #    	       [namespace current]::parse]

    set lib(capabilities,auth) {non_sasl}

    # Load connection transports
    source [file join [file dirname [info script]] transports.tcl]

    set lib(connections) {}
    set lib(connid) 0
    set iq(num) 0
    set lib(handshake) ""

    # Export procedures.
    #
    namespace export \
	wrapper:splitxml wrapper:createtag \
	wrapper:createxml wrapper:xmlcrypt \
	wrapper:isattr wrapper:getattr

    variable NS
    array set NS \
	[list tls      "urn:ietf:params:xml:ns:xmpp-tls" \
	      sasl     "urn:ietf:params:xml:ns:xmpp-sasl" \
	      bind     "urn:ietf:params:xml:ns:xmpp-bind" \
	      session  "urn:ietf:params:xml:ns:xmpp-session" \
	      auth     "http://jabber.org/features/iq-auth" \
	      register "http://jabber.org/features/iq-register"]
}

if {![catch {package require jlibsasl 1.0}]} {
    lappend [namespace current]::jlib::lib(capabilities,auth) sasl
}

######################################################################
if {![info exists keep_alive]} {
    set keep_alive 0
}
if {![info exists keep_alive_interval]} {
    set keep_alive_interval 10
}

######################################################################
proc jlib::capabilities {type} {
    variable lib

    set res {}
    switch -- $type {
	proxy {
	    set res [transport::proxy::capabilities]
	}
	transport {
	    set res [transport::capabilities]
	}
	auth {
	    set res $lib(capabilities,$type)
	}
    }
    return $res
}

######################################################################
proc jlib::new {args} {
    variable lib
    variable connjid
    variable connhist

    foreach {attr val} $args {
	switch -- $attr {
	    -user          {set user $val}
	    -server        {set server $val}
	    -resource      {set resource $val}
	}
    }

    if {![info exists user] || ![info exists server] || \
	    ![info exists resource]} {
	return -code error "Usage: jlib::new -user username \
-server servername -resource resourcename"
    }

    set jid $user@$server/$resource
    if {[info exists connhist($jid)]} {
	set connid $connhist($jid)
	if {[lsearch -exact $lib(connections) $connid] >= 0} {
	    set connid [incr lib(connid)]
	}
    } else {
	set connid [incr lib(connid)]
	set connhist($jid) $connid
    }

    set connjid($connid,user) $user
    set connjid($connid,server) $server
    set connjid($connid,resource) $resource

    ::LOG "(jlib::new) JID:'$jid' ConnectionID:'$connid'"
    return $connid
}

######################################################################
proc jlib::connect {connid args} {
    variable lib
    variable connjid

    set user $connjid($connid,user)
    set server $connjid($connid,server)
    set resource $connjid($connid,resource)

    set transport tcp
    set host $server
    set port 5222
    set xmlns jabber:client
    set use_sasl 0
    set allow_sasl_plain 0
    set use_starttls 0
    set cafile ""
    set certfile ""
    set keyfile ""
    set send_to 0

    foreach {attr val} $args {
	switch -- $attr {
	    -transport      {set transport $val}
	    -host           {set host $val}
	    -port           {set port $val}
	    -xmlns          {set xmlns $val}
	    -usesasl        {set use_sasl $val}
	    -allowsaslplain {set allow_sasl_plain $val}
	    -usestarttls    {set use_starttls $val}
	    -cafile         {set cafile $val}
	    -certfile       {set certfile $val}
	    -keyfile        {set keyfile $val}
	    -sendto         {set send_to $val}
	}
    }

    ::LOG "(jlib::connect) Server:'$server' ConnectionID:'$connid'"

    if {[catch {
	    set host [jlib::idna_domain_toascii $host]
	    eval [list transport::${transport}::connect $connid $host $port] $args
	} sock]} {
	::LOG "error (jlib::connect) Can't connect to the server: $sock"
	return -code error $sock
    } else {
	set lib($connid,sck) $sock
    }

    lappend lib(connections) $connid

    set lib($connid,transport) $transport
    add_connection_route $connid $server
    set lib($connid,disconnect) "client:reconnect"
    set lib($connid,parse_end) 0
    set lib($connid,use_sasl) $use_sasl
    set lib($connid,allow_sasl_plain) $allow_sasl_plain
    set lib($connid,use_starttls) $use_starttls
    set lib($connid,cafile) $cafile
    set lib($connid,certfile) $certfile
    set lib($connid,keyfile) $keyfile
    set lib($connid,send_to) $send_to
    set lib($connid,disconnecting) 0

    catch { unset lib($connid,features) }
    set lib($connid,version) 0.0

    set lib($connid,wrap) \
	[wrapper:new [list [namespace current]::got_stream $connid] \
		     [list [namespace current]::end_of_parse $connid] \
		     [list [namespace current]::parse $connid]]

    if {[info commands sasl_new] != ""} {
	sasl_new $connid
    }

    set params [list -xmlns $xmlns -xml:lang [get_lang]]
    if {$use_sasl || $use_starttls} {
	lappend params -version "1.0"
    }	

    outmsg [eval [list wrapper:streamheader $server] $params] \
	-connection $connid

    return $connid
}

proc jlib::reset {connid} {
    variable lib

    wrapper:reset $lib($connid,wrap)
    catch { unset lib($connid,features) }
}

######################################################################
proc jlib::disconnect {{connections {}}} {
    variable lib

    ::LOG "(jlib::disconnect) $connections"

    if {$connections == {}} {
	set connections $lib(connections)
    }

    foreach connid $connections {
	cancel_keepalive $connid
	outmsg [wrapper:streamtrailer] -connection $connid
	set lib($connid,disconnecting) 1
	transport::$lib($connid,transport)::disconnect $connid
	transport::$lib($connid,transport)::close $connid

	clear_vars $connid
	set idx [lsearch -exact $lib(connections) $connid]
	set lib(connections) [lreplace $lib(connections) $idx $idx]
    }

    if {$lib(connections) == {}} {
	clear_iqs
    }
}

######################################################################
proc jlib::got_stream {connid vars} {
    variable lib

    set lib($connid,sessionid) [jlib::wrapper:getattr $vars id]
    set version [jlib::wrapper:getattr $vars version]
    if {($lib($connid,use_starttls) || $lib($connid,use_sasl)) && \
	    [string is double -strict $version] && ($version >= 1.0)} {
	set lib($connid,version) $version
    }

    ::LOG "(jlib::got_stream $connid)\
Session ID = $lib($connid,sessionid), Version = $lib($connid,version)"
}

######################################################################
proc jlib::end_of_parse {connid} {
    variable lib

    ::LOG "(jlib::end_of_parse $connid)"

    set lib($connid,parse_end) 1
    if {$lib(connections) == {}} {
	::LOG "error (jlib::end_of_parse) No connection"
	return -1
	# Already disconnected
    }

    cancel_keepalive $connid
    transport::$lib($connid,transport)::close $connid

    if {!$lib($connid,disconnecting)} {
	after idle [list [namespace current]::emergency_disconnect $connid]
    }
}

######################################################################
proc jlib::outmsg {msg args} {
    global keep_alive keep_alive_interval
    variable keep_alive_id
    variable lib

    foreach {attr val} $args {
	switch -- $attr {
	    -connection {set connid $val}
	}
    }

    if {$lib(connections) == {}} {
	::LOG "error (jlib::outmsg) No connections"
	return -1
    }

    if {![info exists connid]} {
	set connid [lindex $lib(connections) 0]
    }

    cancel_keepalive $connid

    if {[lsearch -exact $lib(connections) $connid] < 0} {
	::LOG "error (jlib::outmsg) Connection $connid doesn't exist"
	return -1
    }

    if {$lib($connid,disconnecting)} {
	::LOG "error (jlib::outmsg) Message while disconnecting..."
	return -1
    }

    ::LOG "(jlib::outmsg) ($connid) '$msg'"
    ::LOG_OUTPUT $connid $msg

    if {$keep_alive} {
	set keep_alive_id [after [expr $keep_alive_interval * 60 * 1000] \
			       [namespace current]::out_keepalive $connid]
    }

    return [transport::$lib($connid,transport)::outmsg $connid $msg]
}

######################################################################
proc jlib::out_keepalive {connid} {
    outmsg " " -connection $connid
}

# TODO: connid
proc jlib::cancel_keepalive {connid} {
    variable keep_alive_id

    if {[info exists keep_alive_id]} {
	after cancel $keep_alive_id
    }
}

######################################################################
proc jlib::inmsg {connid msg eof} {
    global keep_alive keep_alive_interval
    variable keep_alive_id
    variable lib

    # TODO
    if {$keep_alive} {
	cancel_keepalive $connid
	set keep_alive_id [after [expr $keep_alive_interval * 60 * 1000] \
			       [namespace current]::out_keepalive $connid]
    }

    ::LOG "(jlib::inmsg) ($connid) '$msg'"
    ::LOG_INPUT $connid $msg
    wrapper:parser $lib($connid,wrap) parse $msg

    if {!$lib($connid,parse_end) && $eof} {
	cancel_keepalive $connid
	transport::$lib($connid,transport)::close $connid

	if {$lib($connid,disconnecting)} {
	    ::LOG "(jlib::inmsg) Socket is closed by server. Disconnecting..."
	} else {
	    ::LOG "error (jlib::inmsg) Socket is closed by server. Disconnecting..."
	    after idle [list [namespace current]::emergency_disconnect $connid]
	}
    }
}

######################################################################
proc jlib::emergency_disconnect {connid} {
    variable lib

    set idx [lsearch -exact $lib(connections) $connid]

    if {$idx < 0} return

    set lib(connections) [lreplace $lib(connections) $idx $idx]

    uplevel #0 $lib($connid,disconnect) $connid

    clear_vars $connid
    if {$lib(connections) == {}} {
	clear_iqs
    }
}

######################################################################
proc jlib::clear_vars {connid} {
    #
    # unset all the variables
    #
    variable roster
    variable pres
    variable lib

    if {![info exists lib($connid,wrap)]} return

    wrapper:free $lib($connid,wrap)
    if {[info commands sasl_free] != ""} {
	sasl_free $connid
    }

    array unset lib $connid,*

    set lib($connid,disconnect) "client:reconnect"
    set lib(handshake) ""
}

######################################################################
proc jlib::clear_iqs {} {
    variable iq

    array unset iq presence,*

    foreach id [array names iq] {
	if {$id != "num"} {
	    uplevel #0 $iq($id) [list DISCONNECT {}]
	    unset iq($id)
	}
    }

    set iq(num) 0
}

######################################################################
proc jlib::connections {} {
    variable lib
    return $lib(connections)
}

proc jlib::connection_jid {connid} {
    variable connjid
    return $connjid($connid,user)@$connjid($connid,server)/$connjid($connid,resource)
}

proc jlib::connection_bare_jid {connid} {
    variable connjid
    return $connjid($connid,user)@$connjid($connid,server)
}

proc jlib::connection_user {connid} {
    variable connjid
    return $connjid($connid,user)
}

proc jlib::connection_server {connid} {
    variable connjid
    return $connjid($connid,server)
}

######################################################################
proc jlib::register_xmlns {connid xmlns callback} {
    variable lib

    set lib($connid,registered,$xmlns) $callback
}

######################################################################
proc jlib::unregister_xmlns {connid xmlns} {
    variable lib

    catch {unset lib($connid,registered,$xmlns)}
}

######################################################################
proc jlib::parse {connid xmldata} {
    variable NS
    variable global
    variable roster
    variable pres
    variable lib
    variable iq

    ::LOG "(jlib::parse) xmldata:'$xmldata'"
    ::LOG_INPUT_XML $connid $xmldata

    if {$lib(connections) == {}} {
        ::LOG "error (jlib::parse) No connection"
        return -1
    }

    wrapper:splitxml $xmldata tag vars isempty chdata children

    set usefrom 0
    set from ""
    if {[wrapper:isattr $vars from] == 1} {
	set usefrom 1
	set from [wrapper:getattr $vars from]
    }

    set xmlns [wrapper:getattr $vars xmlns]

    if {[info exists lib($connid,registered,$xmlns)]} {
	uplevel \#0 $lib($connid,registered,$xmlns) [list $xmldata]
	return
    }

    switch -- $tag {
	iq {
	    set useid 0
	    set id ""
	    set type [wrapper:getattr $vars type]

	    if {[wrapper:isattr $vars id] == 1} {
		set useid 1
		set id [wrapper:getattr $vars id]
	    }

	    if {$type != "result" && $type != "error" && $type != "get" && $type != "set"} {
		::LOG "(error) iq: unknown type:'$type' id ($useid):'$id'"
		return
	    }

	    if {$type == "result"} {
		if {$useid == 0} {
		    ::LOG "(error) iq:result: no id reference"
		    return
		}
		if {[info exists iq($id)] == 0} {
		    ::LOG "(error) iq:result: id doesn't exists in memory. Probably a re-replied iq"
		    return
		}

		set cmd $iq($id)
		unset iq($id)

		uplevel \#0 $cmd [list OK [lindex $children 0]]
	    } elseif {$type == "error"} {
		if {$useid == 0} {
		    ::LOG "(error) iq:result: no id reference"
		    return
		}
		if {[info exists iq($id)] == 0} {
		    ::LOG "(error) iq:result: id doesn't exists in memory. Probably a re-replied iq."
		    return
		}

		set cmd $iq($id)
		unset iq($id)

		set child ""
		foreach child $children {
		    if {[lindex $child 0] == "error"} { break }
		    set child ""
		}
		if {$child == ""} {
		    set errcode ""
		    set errtype ""
		    set errmsg ""
		} else {
		    set errcode [wrapper:getattr [lindex $child 1] code]
		    set errtype [wrapper:getattr [lindex $child 1] type]
		    set errmsg [lindex $child 3]
		}
		if {$errtype == ""} {
		    uplevel #0 $cmd [list ERR [list $errcode $errmsg]]
		} else {
		    uplevel #0 $cmd [list ERR [list $errtype $child]]
		}
	    } elseif {$type == "get" || $type == "set"} {
		set child [lindex $children 0]

		if {$child == ""} {
		    ::LOG "(error) iq:$type: Cannot find 'query' tag"
		    return
		}

		#
		# Before calling the 'client:iqreply' procedure, we should check
		# the 'xmlns' attribute, to understand if this is some 'iq' that
		# should be handled inside jlib, such as a roster-push.
		#
		if {$type == "set" && \
			[wrapper:getattr [lindex $child 1] xmlns] == "jabber:iq:roster"} {
		    if {$from != "" && \
			    !([string equal -nocase $from [connection_server $connid]] || \
			    [string equal -nocase $from [connection_bare_jid $connid]] || \
			    [string equal -nocase $from [connection_jid $connid]])} {
			send_iq error \
			    [error cancel not-allowed -xml $child] \
			    -id [wrapper:getattr $vars id] \
			    -to $from \
			    -connection $connid
			return
		    }

		    # Found a roster-push
		    ::LOG "(info) iq packet is roster-push. Handling internally"

		    # First, we reply to the server, saying that, we 
		    # got the data, and accepted it.
		    #
		    if [wrapper:isattr $vars id] {
			send_iq result \
			    [wrapper:createtag query \
				 -vars [list xmlns jabber:iq:roster]] \
			    -id [wrapper:getattr $vars id] \
			    -connection $connid
		    } else {
			send_iq result \
			    [wrapper:createtag query \
				 -vars [list xmlns jabber:iq:roster]] \
			    -connection $connid
		    }

		    # And then, we call the jlib::parse_roster_get, because this
		    # data is the same as the one we get from a roster-get.
		    parse_roster_get \
			$connid 1 [namespace current]::noop OK $child
		    return
		}

		if ($lib($connid,send_to)) {
		    set param [list -to [wrapper:getattr $vars to]]
		} else {
		    set param {}
		}
		uplevel \#0 \
		    [list client:iqreply $connid $from $useid $id $type $child] $param
	    }
	}
	message {
	    set type [wrapper:getattr $vars type]
	    set id [wrapper:getattr $vars id]

	    set body     ""
	    set err      [list "" ""]
	    set is_subject 0
	    set subject  ""
	    set priority ""
	    set thread   ""
	    set x        ""

	    foreach child $children {
		wrapper:splitxml $child ctag cvars cisempty cchdata cchildren

		switch -- $ctag {
		    body {set body $cchdata}
		    error {
			set errmsg $cchdata
			set errcode [wrapper:getattr $cvars code]
			set errtype [wrapper:getattr $cvars type]
			if {$errtype == ""} {
			    set err [list $errcode $errmsg]
			} else {
			    set err [list $errtype $child]
			}
		    }
		    subject {
			set is_subject 1
			set subject $cchdata
		    }
		    priority {set priority $cchdata}
		    thread {set thread $cchdata}
		    default {
			if {[wrapper:getattr $cvars xmlns] != ""} {
			    lappend x $child
			}
		    }
		}
	    }

	    if ($lib($connid,send_to)) {
		set param [list -to [wrapper:getattr $vars to]]
	    } else {
		set param {}
            }
	    uplevel \#0 \
		[list client:message $connid $from $id $type $is_subject $subject $body $err $thread $priority $x] $param
	}
	presence {
	    set type [wrapper:getattr $vars type]

	    set cmd      ""
	    set status   ""
	    set priority ""
	    set meta     ""
	    set icon     ""
	    set show     ""
	    set loc      ""
	    set x        ""

	    set param    ""
	    if ($lib($connid,send_to)) {
		lappend param -to [wrapper:getattr $vars to]
	    }

	    if {[wrapper:isattr $vars id]} {
		set id [wrapper:getattr $vars id]
		if {[info exists iq(presence,$id)]} {
		    set cmd $iq(presence,$id)
		    unset iq(presence,$id)
		}
		lappend param -id $id
	    }

	    if {[wrapper:isattr $vars name]} {
		lappend param -name [wrapper:getattr $vars name]
	    }

	    foreach child $children {
		wrapper:splitxml $child ctag cvars cisempty cchdata cchildren

		switch -- $ctag {
		    status   {lappend param -status   $cchdata}
		    priority {lappend param -priority $cchdata}
		    meta     {lappend param -meta     $cchdata}
		    icon     {lappend param -icon     $cchdata}
		    show     {lappend param -show     $cchdata}
		    loc      {lappend param -loc      $cchdata}
		    x        {lappend x $child}
		    error {
			set errcode [wrapper:getattr $cvars code]
			set errtype [wrapper:getattr $cvars type]
			if {$errtype == ""} {
			    set err [list $errcode $cchdata]
			} else {
			    set err [list $errtype $child]
			}
			lappend param -status [lindex [error_to_list $err] 2]
			lappend param -error [lrange [error_to_list $err] 0 1]
		    }
		}
	    }

	    set cont ""
	    if {$cmd != ""} {
		set cont \
		    [uplevel \#0 $cmd [list $connid $from $type $x] $param]
	    }

	    if {$cont != "break"} {
		uplevel #0 \
		    [list client:presence $connid $from $type $x] $param
	    }
	}
        handshake {
	    set cmd $lib(handshake)
	    set lib(handshake) ""

	    if {[string compare $cmd ""]} {
                uplevel #0 $cmd OK
	    }
	}
	stream:error {
	    if {[string compare [set cmd $lib(handshake)] ""]} {
		set lib(handshake) ""

		uplevel #0 $cmd [list ERR $chdata]
		return
	    }

	    switch -- [streamerror::condition $xmldata] {
		bad-format -
		bad-namespace-prefix -
		connection-timeout -
		invalid-from -
		invalid-id -
		invalid-namespace -
		invalid-xml -
		remote-connection-failed -
		restricted-xml -
		unsupported-encoding -
		unsupported-stanza-type -
		xml-not-well-formed {
		    set lib($connid,disconnect) client:reconnect
		}
		default {
		    set lib($connid,disconnect) client:disconnect
		}
	    }
	    client:errormsg [streamerror::message $xmldata]
	}
	stream:features {
	    parse_stream_features $connid $children
	}
	failure -
	success {
	    if {[wrapper:getattr $vars xmlns] == $NS(tls)} {
		starttls_result $connid $tag
	    }
	}
	proceed {
	    if {[wrapper:getattr $vars xmlns] == $NS(tls)} {
		starttls_result $connid $tag
	    }
	}
    }
}

######################################################################
proc jlib::parse_send_auth {cmd type data} {
    variable lib

    ::LOG "(jlib::parse_send_auth) type:'$type'"

    if {$type == "ERR"} {           ;# Got an error reply
	::LOG "error (jlib::parse_send_auth) errtype:'[lindex $data 0]'"
	::LOG "error (jlib::parse_send_auth) errdesc:'[lindex $data 1]'"
	client:status [::msgcat::mc "Authentication failed"]
	uplevel #0 $cmd [list ERR $data]
	return
    }
    client:status [::msgcat::mc "Authentication successful"]
    uplevel #0 $cmd [list OK {}]
}

######################################################################
proc jlib::parse_send_create {cmd type data} {
    variable lib

    ::LOG "(jlib::parse_send_create) type:'$type'"

    if {$type == "ERR"} {           ;# Got an error reply
	::LOG "error (jlib::parse_send_create) errtype:'[lindex $data 0]'"
	::LOG "error (jlib::parse_send_create) errdesc:'[lindex $data 1]'"
	uplevel #0 $cmd [list ERR [lindex $data 1]]
	return
    }
    uplevel #0 $cmd [list OK {}]
}

######################################################################
proc jlib::parse_roster_get {connid ispush cmd type data} {
    variable lib
    variable roster
    
    ::LOG "(jlib::parse_roster_get) ispush:'$ispush' type:'$type'"

    if {$type == "ERR"} {           ;# Got an error reply
	::LOG "error (jlib::parse_roster_get) errtype:'[lindex $data 0]'"
	::LOG "error (jlib::parse_roster_get) errdesc:'[lindex $data 1]'"
	uplevel #0 $cmd [list $connid ERR]
	return
    }
    if {!$ispush} {
	client:status [::msgcat::mc "Got roster"]
	uplevel #0 $cmd [list $connid BEGIN_ROSTER]
    }

    wrapper:splitxml $data tag vars isempty chdata children

    if {![cequal [wrapper:getattr $vars xmlns] jabber:iq:roster]} {::LOG "warning (jlib::parse_roster_get) 'xmlns' attribute of query tag doesn't match 'jabber:iq:roster': '[wrapper:getattr $vars xmlns]"}

    foreach child $children {
	wrapper:splitxml $child ctag cvars cisempty cchdata cchildren
	
	switch -- $ctag {
	    default {
		set groups ""
		set jid   [wrapper:getattr $cvars jid]
		set name  [wrapper:getattr $cvars name]
		set subsc [wrapper:getattr $cvars subscription]
		set ask   [wrapper:getattr $cvars ask]
		if {$ctag != {item}} {
		    set category $ctag
		} else {
		    set category [wrapper:getattr $cvars category]
		}
		set subtype [wrapper:getattr $cvars type]
		
		foreach subchild $cchildren {
		    wrapper:splitxml $subchild subtag tmp tmp subchdata tmp
		    
		    switch -- $subtag {
			group {lappend groups $subchdata}
		    }
		}
		
		# Ok, collected information about item.
		# Now we can set our variables...
		#
		if {[lsearch $roster(users) $jid] == -1} {
		    lappend roster(users) $jid
		}
		
		set roster(group,$jid) $groups
		set roster(name,$jid)  $name
		set roster(subsc,$jid) $subsc
		set roster(ask,$jid)   $ask

		add_connection_route $connid $jid

		# ...and call client procedures
		if ($lib($connid,send_to)) {
		    set param [list -to [wrapper:getattr $vars to]]
		} else {
		    set param {}
		}
		if $ispush {
		    uplevel \#0 [list client:roster_push $connid $jid \
				     $name $groups $subsc $ask \
				     $category $subtype] $param
		} else {
		    uplevel \#0 [list client:roster_item $connid $jid \
				     $name $groups $subsc $ask \
				     $category $subtype] $param
		}
	    }
	}
    }
    if {!$ispush} {
	uplevel #0 $cmd [list $connid END_ROSTER]
    }
}

######################################################################
proc jlib::parse_roster_set {item cmd groups name type data} {
    variable lib
    variable roster

    ::LOG "(jlib::parse_roster_set) item:'$item' type:'$type'"

    if {$type == "ERR"} {           ;# Got an error reply
	::LOG "error (jlib::parse_roster_set) errtype:'[lindex $data 0]'"
	::LOG "error (jlib::parse_roster_set) errdesc:'[lindex $data 1]'"
	uplevel #0 $cmd ERR
	return
    }

    if { [lsearch $roster(users) $item] == -1}   {
	lappend roster(users) $item
	set roster(subsc,$item) "none"
	set roster(ask,$item)   ""
    }

    set roster(group,$item) $groups
    set roster(name,$item)  $name

    uplevel #0 $cmd OK
}

######################################################################
proc jlib::parse_roster_del {item cmd type data} {
    variable lib
    variable roster

    ::LOG "(jlib::parse_roster_del) item:'$item' type:'$type'"

    if {$type == "ERR"} {           ;# Got an error reply
	::LOG "error (jlib::parse_roster_set) errtype:'[lindex $data 0]'"
	::LOG "error (jlib::parse_roster_set) errdesc:'[lindex $data 1]'"
	uplevel #0 $cmd ERR
	return
    }

    if {[set num [lsearch $roster(users) $item]] != -1} {
	set roster(users) [lreplace $roster(users) $num $num]

	catch {unset roster(group,$item) }
	catch {unset roster(name,$item)  }
	catch {unset roster(subsc,$item) }
	catch {unset roster(ask,$item)   }
    } else {
	::LOG "warning (jlib::parse_roster_del) Item '$item' doesn't exist in roster for deletion."
    }
    uplevel #0 $cmd OK
}

######################################################################
proc jlib::parse_stream_features {connid childrens} {
    variable lib

    set features {}
    foreach child $childrens {
	wrapper:splitxml $child tag vars isempty chdata children

	set xmlns [wrapper:getattr $vars xmlns]

	if {[info exists lib($connid,registered,$xmlns)]} {
	    lappend features $xmlns
	    uplevel \#0 $lib($connid,registered,$xmlns) [list $child]
	    continue
	}

	switch -- $tag {
	    starttls {
		lappend features starttls
	    }
	    register {
		lappend features register
	    }
	    auth {
		lappend features auth
	    }
	}
    }
    set lib($connid,features) $features
}

######################################################################
proc jlib::send_iq {type xmldata args} {
    variable lib
    variable iq

    ::LOG "(jlib::send_iq) type:'$type'"
    if {$lib(connections) == {}} {
          ::LOG "error (jlib::send_iq) No connection"
          return -1
    }

    set useto  0
    set useid  0
    set to     {}
    set id     {}
    set cmd    [namespace current]::noop
    set vars   {}

    foreach {attr val} $args {
	switch -- $attr {
	    -command    {set cmd $val}
	    -to         {set useto 1; set to $val}
	    -id         {set useid 1; set id $val}
	    -connection {set connid $val}
	    -from       {lappend vars from $val}
	}
    }
    if {![info exists connid]} {
	set connid [route $to]
    }

    if {$type != "set" && $type != "result" && $type != "error"} {
	set type "get"
    }

    ::LOG "(jlib::send_iq) type:'$type' to ($useto):'$to' cmd:'$cmd' xmldata:'$xmldata'"

    # Temporary hack that allows to insert more than 1 subtag in error iqs
    if {($type != "error") && ($xmldata != "")} {
	set xmldata [list $xmldata]
    }

    if {$type == "get" || $type == "set"} {
	lappend vars id $iq(num)
	set iq($iq(num)) $cmd
	incr iq(num)
    } elseif {$useid} {
	lappend vars id $id
    }

    if {$useto == 1} {
	lappend vars to $to
    }
    lappend vars type $type xml:lang [get_lang]

    if {$xmldata != ""} {
	set data [wrapper:createtag iq -vars $vars -subtags $xmldata]
    } else {
	set data [wrapper:createtag iq -vars $vars]
    }
    ::LOG_OUTPUT_XML $connid $data
    outmsg [wrapper:createxml $data] -connection $connid
}

######################################################################

proc jlib::parse_get_authtypes {connid type data} {
    variable lib

    ::LOG "(jlib::parse_get_authtypes) type:'$type'"

    if {$type == "DISCONNECT"} {
	return
    }
    if {$type == "ERR"} {
	::LOG "error (jlib::parse_get_authtypes) errtype:'[lindex $data 0]'"
	::LOG "error (jlib::parse_get_authtypes) errdesc:'[lindex $data 1]'"
	set lib($connid,authtypes) [list ERR $data]
	return
    }
    set authtypes {}
    wrapper:splitxml $data tag vars isempty chdata children
    foreach child $children {
	wrapper:splitxml $child tag1 vars1 isempty1 chdata1 children1
	switch -- $tag1 {
	    password {
		lappend authtypes plain
	    }
	    digest {
		lappend authtypes $tag1
	    }
	}
    }
    set lib($connid,authtypes) [list OK $authtypes]
}

proc jlib::get_authtypes {user {connid ""}} {
    variable lib
    variable NS

    if {$lib($connid,version) >= 1.0} {
	if {$lib($connid,use_sasl)} {
	    if {[lsearch -exact $lib($connid,features) $NS(sasl)] >= 0} {
		return SASL
	    } else {
		set err [error modify not-acceptable -text [::msgcat::mc \
			     "Server haven't provided SASL authentication feature"]]
		return [list ERR [list modify [lindex $err 0]]]
	    }
	}

	if {[lsearch -exact $lib($connid,features) auth] < 0} {
	    set err [error modify not-acceptable -text [::msgcat::mc \
			 "Server haven't provided non-SASL authentication feature"]]
	    return [list ERR [list modify [lindex $err 0]]]
	}
    }
    
    if {$connid == ""} {
	set connid [lindex $lib(connections) 0]
    }

    ::LOG "(jlib::get_authtypes) username:'$user'"
    if { $lib(connections) == {} } {
	::LOG "error (jlib::get_authtypes) No connection"
	return -1
    }

    set data [wrapper:createtag query \
		  -vars    [list xmlns "jabber:iq:auth"] \
		  -subtags [list [wrapper:createtag username \
				     -chdata $user]]]

    send_iq get $data \
	-command [list [namespace current]::parse_get_authtypes $connid] \
	-connection $connid
    
    client:status [::msgcat::mc "Waiting for authentication mechanisms"]
    
    vwait [namespace current]::lib($connid,authtypes)

    client:status [::msgcat::mc "Got authentication mechanisms"]
    return $lib($connid,authtypes)
}


proc jlib::send_auth {connid user pass res cmd {authtype plain}} {
    variable lib
    variable connjid

    ::LOG "(jlib::send_auth) username:'$user' password:'$pass' resource:'$res'"
    if {$lib(connections) == {}} {
	::LOG "error (jlib::send_auth) No connection"
	return -1
    }

    set lib($connid,pass) $pass
    set connjid($connid,user) $user
    set connjid($connid,resource) $res

    if {$lib($connid,use_starttls)} {
	if {[lsearch -exact $lib($connid,features) starttls] >= 0} {
	    starttls $connid
	} else {
	    set err [error modify not-acceptable -text [::msgcat::mc \
			 "Server haven't provided STARTTLS feature"]]
	    return [list ERR [list modify [lindex $err 0]]]
	}
    }

    if {($lib($connid,version) >= 1.0) && $lib($connid,use_sasl)} {
	sasl_auth $connid $user $res $pass $lib($connid,allow_sasl_plain) $cmd
	return
    }

    switch -- $authtype {
	plain {
	    set data [wrapper:createtag query \
			  -vars    [list xmlns "jabber:iq:auth"] \
			  -subtags [list \
					[wrapper:createtag username \
					     -chdata $user] \
					[wrapper:createtag password \
					     -chdata $pass] \
					[wrapper:createtag resource \
					     -chdata $res]]]
	}
	digest {
	    set digest [sha1::sha1 [cconcat $lib($connid,sessionid) \
					[encoding convertto utf-8 $pass]]]
	    set data [wrapper:createtag query \
			  -vars    [list xmlns "jabber:iq:auth"] \
			  -subtags [list \
					[wrapper:createtag username \
					     -chdata $user] \
					[wrapper:createtag digest \
					     -chdata $digest] \
					[wrapper:createtag resource \
					     -chdata $res]]]
	}

        handshake {
            set lib(handshake) $cmd
	    set digest [sha1::sha1 $lib($connid,sessionid)$pass]
	    outmsg [wrapper:createxml \
			[wrapper:createtag handshake -chdata $digest]] \
		-connection $connid
	    client:status [::msgcat::mc "Waiting for authentication results"]
	    return
	}
    }

    send_iq set $data \
	-command [list [namespace current]::parse_send_auth $cmd] \
	-connection $connid
    client:status [::msgcat::mc "Waiting for authentication results"]
}

######################################################################
proc jlib::starttls {connid} {
    variable lib
    variable NS

    set data [wrapper:createtag starttls \
		  -vars   [list xmlns $NS(tls)]]
    
    outmsg [wrapper:createxml $data] -connection $connid

    vwait [namespace current]::lib($connid,starttls_result)

    set lib($connid,use_starttls) 0
}

proc jlib::starttls_result {connid res} {
    variable lib
    variable connjid

    switch -- $res {
	failure {}
	proceed {
	    set args {}
	    foreach key {certfile cafile keyfile} {
		if {$lib($connid,$key) != ""} {
		    lappend args -$key $lib($connid,$key)
		}
	    }
	    # TODO: cleanup
	    set transport::tls::lib($connid,socket) \
		$transport::tcp::lib($connid,socket)
	    eval [list transport::tls::tls_import $connid] $args
	    set lib($connid,transport) tls
	}
    }

    reset $connid

    outmsg [wrapper:streamheader [connection_server $connid] \
		-xml:lang [get_lang] -version "1.0"] \
	-connection $connid
	
    vwait [namespace current]::lib($connid,features)

    set lib($connid,starttls_result) $res
}

######################################################################
proc jlib::route {jid} {
    variable lib

    if {[catch { set calling_routine [info level -1] }]} {
	set calling_routine none
    }

    if { $lib(connections) == {} } {
	::LOG "error (jlib::send_create) No connection"
	return -1
    }

    set user $jid
    regexp {([^/]*)/.*} $jid temp user
    set serv $user
    regexp {[^@]*@(.*)} $user temp serv

    set connid [lindex $lib(connections) 0]
    foreach dest [list $user $serv] {
	foreach c $lib(connections) {
	    if {[info exists lib($c,route,$dest)]} {
		debugmsg jlib::route "{$jid}: $c \[$calling_routine\]"
		return $c
	    }
	}
    }

    debugmsg jlib::route "{$jid}: $connid \[$calling_routine\]"
    return $connid
}


proc jlib::add_connection_route {connid jid} {
    variable lib

    set lib($connid,route,$jid) 1
}


######################################################################
# TODO
proc jlib::send_create {user pass name email cmd} {
    variable lib

    ::LOG "(jlib::send_create) username:'$user' password:'$pass' name:'$name' email:'$email'"
    if { $lib(connections) == {} } {
	::LOG "error (jlib::send_create) No connection"
	return -1
    }

    set data [wrapper:createtag query \
		  -vars    [list xmlns "jabber:iq:register"] \
		  -subtags [list \
				[wrapper:createtag name     -chdata $name] \
				[wrapper:createtag email    -chdata $email] \
				[wrapper:createtag username -chdata $user] \
				[wrapper:createtag password -chdata $pass]]]

    send_iq set $data \
	-command [list [namespace current]::parse_send_create $cmd]
}

######################################################################
proc jlib::send_msg {to args} {
    variable lib

    ::LOG "(jlib::send_msg) to:'$to'"
    if {$lib(connections) == {}} {
	::LOG "error (jlib::send_msg) No connection"
	return -1
    }

    if {[wrapper:isattr $args -connection]} {
	set connid [wrapper:getattr $args -connection]
    } else {
	set connid [route $to]
    }

    set children ""

    if {[wrapper:isattr $args -subject]} {
	lappend children [wrapper:createtag subject \
			      -chdata [wrapper:getattr $args -subject]]
    }
    if {[wrapper:isattr $args -thread]} {
	lappend children [wrapper:createtag thread \
			      -chdata [wrapper:getattr $args -thread]]
    }
    if {[wrapper:isattr $args -body]} {
	lappend children [wrapper:createtag body \
			      -chdata [wrapper:getattr $args -body]]
    }
    if {[wrapper:isattr $args -xlist]} {
	foreach a [wrapper:getattr $args -xlist] {
	    lappend children $a
	}
    }

    set vars [list to $to]
    if {[wrapper:isattr $args -from]} {
	lappend vars from [wrapper:getattr $args -from]
    }
    if {[wrapper:isattr $args -type]} {
	lappend vars type [wrapper:getattr $args -type]
    }
    lappend vars xml:lang [get_lang]

    set data [wrapper:createtag message -vars $vars -subtags $children]
    ::LOG_OUTPUT_XML $connid $data
    outmsg [wrapper:createxml $data] -connection $connid
}

######################################################################
proc jlib::send_presence {args} {
    variable lib
    variable iq

    ::LOG "(jlib::send_presence)"
    if {$lib(connections) == {}} {
	::LOG "error (jlib::send_presence) No connection"
	return -1
    }

    set children ""
    set vars     ""

    # TODO
    if {[wrapper:isattr $args -from]} {
	lappend vars from [wrapper:getattr $args -from]
    }
    if {[wrapper:isattr $args -to]} {
	set to [wrapper:getattr $args -to]
	lappend vars to $to
    }
    if {[wrapper:isattr $args -type]} {
	lappend vars type [wrapper:getattr $args -type]
    }
    if {[wrapper:isattr $args -command]} {
	lappend vars id $iq(num)
	set iq(presence,$iq(num)) [wrapper:getattr $args -command]
	incr iq(num)
    }

    if {[wrapper:isattr $args -connection]} {
	set connid [wrapper:getattr $args -connection]
    } else {
	if {[info exists to]} {
	    set connid [route $to]
	} else {
	    set connid [lindex $lib(connections) 0]
	}
    }

    if {[wrapper:isattr $args -stat]} {
	lappend children [wrapper:createtag status \
			      -chdata [wrapper:getattr $args -stat]]
    }
    if {[wrapper:isattr $args -pri]} {
	lappend children [wrapper:createtag priority \
			      -chdata [wrapper:getattr $args -pri]]
    }
    if {[wrapper:isattr $args -meta]} {
	lappend children [wrapper:createtag meta \
			      -chdata [wrapper:getattr $args -meta]]
    }
    if {[wrapper:isattr $args -icon]} {
	lappend children [wrapper:createtag icon \
			      -chdata [wrapper:getattr $args -icon]]
    }
    if {[wrapper:isattr $args -show]} {
	lappend children [wrapper:createtag show \
			      -chdata [wrapper:getattr $args -show]]
    }
    if {[wrapper:isattr $args -loc]} {
	lappend children [wrapper:createtag loc \
			      -chdata [wrapper:getattr $args -loc]]
    }

    if {[wrapper:isattr $args -xlist]} {
	foreach a [wrapper:getattr $args -xlist] {
	    lappend children $a
	}
    }
    lappend vars xml:lang [get_lang]

    set data [wrapper:createtag presence -vars $vars -subtags $children]
    ::LOG_OUTPUT_XML $connid $data
    outmsg [wrapper:createxml $data] -connection $connid
}

######################################################################
proc jlib::roster_get {args} {
    variable lib
    variable roster

    ::LOG "(jlib::roster_get)"
    if { $lib(connections) == {} } {
	::LOG "error (jlib::roster_get) No connection"
	return -1
    }

    set cmd "[namespace current]::noop"
    set connid [lindex $lib(connections) 0]
    foreach {attr val} $args {
	switch -- $attr {
	    -command    {set cmd $val}
	    -connection {set connid $val}
	}
    }

    foreach array [array names roster] {
	unset roster($array)
    }
    set roster(users) ""

    set vars [list xmlns "jabber:iq:roster"]
    set data [wrapper:createtag query -empty 1 -vars $vars]
    send_iq get $data \
	-command [list [namespace current]::parse_roster_get $connid 0 $cmd] \
	-connection $connid
    client:status [::msgcat::mc "Waiting for roster"]
}

######################################################################
proc jlib::roster_set {item args} {
    variable lib
    variable roster

    ::LOG "(jlib::roster_set) item:'$item'"
    if {$lib(connections) == {}} {
	::LOG "error (jlib::roster_set) No connection"
	return -1
    }

    set usename 0
    set name ""
    if { [lsearch $roster(users) $item] == -1 } {
	set groups ""
    } else {
	set groups $roster(group,$item)
    }

    if {[wrapper:isattr $args -name]} {
	set usename 1
	set name [wrapper:getattr $args -name]
    }
    if {[wrapper:isattr $args -groups]} {
	set groups [wrapper:getattr $args -groups]
    }
    if {[wrapper:isattr $args -command]} {
	set cmd [wrapper:getattr $args -command]
    } else {
	set cmd [namespace current]::noop
    }

    set vars [list jid $item]
    if {$usename} {
	lappend vars name $name
    }

    set subdata ""
    foreach group $groups {
	lappend subdata [wrapper:createtag group -chdata $group]
    }

     set xmldata [wrapper:createtag query \
		      -vars    [list xmlns jabber:iq:roster] \
		      -subtags [list [wrapper:createtag item \
					  -vars    $vars \
					  -subtags $subdata]]]

    send_iq set $xmldata \
	-command [list [namespace current]::parse_roster_set $item $cmd $groups $name]
}

######################################################################
proc jlib::roster_del {item args} {
    variable lib
    variable roster

    ::LOG "(jlib::roster_del) item:'$item'"
    if { $lib(connections) == {} } {
	::LOG "error (jlib::roster_del) No connection"
	return -1
    }

    # TODO

    if [wrapper:isattr $args -command] {
	set cmd [wrapper:getattr $args -command]
    } else {
	set cmd [namespace current]::noop
    }

    set xmldata [wrapper:createtag query \
		     -vars    [list xmlns jabber:iq:roster] \
		     -subtags [list [wrapper:createtag item \
					 -vars [list jid $item \
						     subscription remove]]]]

    send_iq set $xmldata \
	-command [list [namespace current]::parse_roster_del $item $cmd]
}

######################################################################
proc jlib::wait_for_stream {{connid ""}} {
    variable lib

    client:status [::msgcat::mc "Waiting for stream"]
    if {$connid == ""} {
	set connid [lindex $lib(connections) 0]
    }
    vwait [namespace current]::lib($connid,sessionid)
    client:status [::msgcat::mc "Got stream"]

    if {$lib($connid,version) >= 1.0} {
	client:status [::msgcat::mc "Waiting for stream features"]
	vwait [namespace current]::lib($connid,features)
	client:status [::msgcat::mc "Got stream features"]
    }
}

######################################################################

set ::NS(stanzas) "urn:ietf:params:xml:ns:xmpp-stanzas"

array set ::error_type [list \
	auth	    [::msgcat::mc "Authentication Error"] \
	cancel	    [::msgcat::mc "Unrecoverable Error"] \
	continue    [::msgcat::mc "Warning"] \
	modify	    [::msgcat::mc "Request Error"] \
	wait	    [::msgcat::mc "Temporary Error"]]

set ::defined_error_conditions {}
# Code is zero iff the condition isn't mentioned in JEP-0086
foreach {clist lcode type cond description} [list \
      {400}	    400 modify	bad-request		[::msgcat::mc "Bad Request"] \
      {409}	    409 cancel	conflict		[::msgcat::mc "Conflict"] \
      {501}	    501 cancel	feature-not-implemented [::msgcat::mc "Feature Not Implemented"] \
      {403}	    403 auth	forbidden		[::msgcat::mc "Forbidden"] \
      {302}	    302	modify	gone			[::msgcat::mc "Gone"] \
      {500}	    500 wait	internal-server-error   [::msgcat::mc "Internal Server Error"] \
      {404}	    404 cancel	item-not-found		[::msgcat::mc "Item Not Found"] \
      {}	    400 modify	jid-malformed		[::msgcat::mc "JID Malformed"] \
      {406}	    406	modify	not-acceptable		[::msgcat::mc "Not Acceptable"] \
      {405}	    405 cancel	not-allowed		[::msgcat::mc "Not Allowed"] \
      {401}	    401	auth	not-authorized		[::msgcat::mc "Not Authorized"] \
      {402}	    402 auth	payment-required	[::msgcat::mc "Payment Required"] \
      {}	    404 wait	recipient-unavailable   [::msgcat::mc "Recipient Unavailable"] \
      {}	    302 modify	redirect		[::msgcat::mc "Redirect"] \
      {407}	    407 auth	registration-required   [::msgcat::mc "Registration Required"] \
      {}	    404 cancel	remote-server-not-found [::msgcat::mc "Remote Server Not Found"] \
      {408 504}     504 wait	remote-server-timeout   [::msgcat::mc "Remote Server Timeout"] \
      {}	    500 wait	resource-constraint	[::msgcat::mc "Resource Constraint"] \
      {502 503 510} 503 cancel	service-unavailable	[::msgcat::mc "Service Unavailable"] \
      {}	    407 auth	subscription-required   [::msgcat::mc "Subscription Required"] \
      {}	    500 any	undefined-condition	[::msgcat::mc "Undefined Condition"] \
      {}	    400 wait	unexpected-request	[::msgcat::mc "Unexpected Request"]] \
    {
	lappend ::defined_error_conditions $cond
	set ::error_description($type,$::NS(stanzas),$cond) $description
	# JEP-0086
	foreach code $clist {
	    set ::error_type_descelem($code) [list $type $cond]
	}
	set ::legacy_error_codes($cond) $lcode
    }

proc jlib::error_to_list {errmsg} {
    global error_type defined_error_conditions error_description error_type_descelem

    lassign $errmsg code desc
    if {[string is integer $code]} {
	if {[info exists error_type_descelem($code)]} {
	    lassign $error_type_descelem($code) type descelem
	} else {
	    lassign {none none} type descelem
	}
	return [list $type $descelem "$code ([::msgcat::mc $desc])"]
    } else {
	set type $code
	set errelem $desc
	set condition "undefined-condition"
	set description ""
	set textdescription ""
	wrapper:splitxml $errelem tag vars isempty chdata children
	foreach child $children {
	    wrapper:splitxml $child tag1 vars1 isempty1 chdata1 children1
	    set xmlns [jlib::wrapper:getattr $vars1 xmlns]
	    set cond $tag1
	    switch -- $cond {
		text {
		    if {$xmlns == $::NS(stanzas)} {
			set textdescription ": $chdata1"
		    }
		}
		undefined-condition {
		    # TODO
		}
		default {
		    if {[lsearch -exact $defined_error_conditions $cond] >= 0} {
			set condition $cond
			if {[info exists error_description($type,$xmlns,$cond)] && \
				($description == "")} {
			    set description $error_description($type,$xmlns,$cond)
			}
		    } else {
			# TODO
		    }
		}
	    }
	}
	if {[info exists error_type($type)]} {
	    set typedesc $error_type($type)
	}
	set res ""
	if {$description != ""} {
	    set res $description
	}
	if {[info exists typedesc] && $typedesc != ""} {
	    if {$res == ""} {
		set res $typedesc
	    } else {
		set res "$typedesc ($res)"
	    }
	}
	return [list $type $condition "$res$textdescription"]
    }
}

proc jlib::error {type condition args} {
    return [eval xmpp_error $type $condition $args]
}

proc jlib::legacy_error {type condition args} {
    global legacy_error_codes error_description

    if {[info exists legacy_error_codes($condition)] && \
	    $legacy_error_codes($condition)} {
	set code $legacy_error_codes($condition)
    } else {
	set code 501
    }
    if {[info exists error_description($type,$::NS(stanzas),$condition)]} {
	set description $error_description($type,$::NS(stanzas),$condition)
    } else {
	set description ""
    }
    set xml ""
    foreach {opt val} $args {
	switch -- $opt {
	    -xml {
		set xml $val
	    }
	    -text {
		set description $val
	    }
	}
    }
    set err [wrapper:createtag error \
		-vars [list code $code] \
		-chdata $description]
    if {$xml == ""} {
	return [list $err]
    } else {
	return [list $xml $err]
    }
}

proc jlib::xmpp_error {type condition args} {
    global legacy_error_codes

    set subtags [list [wrapper:createtag $condition \
			   -vars [list xmlns $::NS(stanzas)]]]
    set xml ""
    foreach {opt val} $args {
	switch -- $opt {
	    -xml {
		set xml $val
	    }
	    -text {
		lappend subtags [wrapper:createtag text \
				     -vars [list xmlns $::NS(stanzas)] \
				     -chdata $val]
	    }
	    -application-specific {
		lappend subtags $val
	    }
	}
    }
    set vars [list type $type]
    if {[info exists legacy_error_codes($condition)] && \
	    $legacy_error_codes($condition)} {
	lappend vars code $legacy_error_codes($condition)
    }
    set err [wrapper:createtag error \
		-vars $vars \
		-subtags $subtags]
    if {$xml == ""} {
	return [list $err]
    } else {
	return [list $xml $err]
    }
}

######################################################################
#
proc ::LOG text {
#
# For debugging purposes.
#
    puts "LOG: $text\n"
}

proc ::LOG_OUTPUT     {connid t} {}
proc ::LOG_OUTPUT_XML {connid x} {}
proc ::LOG_INPUT      {connid t} {}
proc ::LOG_INPUT_XML  {connid x} {}

######################################################################
proc jlib::noop args {}

######################################################################
proc jlib::get_lang {} {
    set prefs [::msgcat::mcpreferences]
    set lang [lindex $prefs end]
    switch -- $lang {
	"" -
	c -
	posix {
	    return en
	}
    }
    set lang2 [lindex $prefs end-1]
    if {[regexp {^([A-Za-z]+)_([0-9A-Za-z]+)} $lang2 ignore l1 l2]} {
	return "[string tolower $l1]-[string toupper $l2]"
    } else {
	return $lang
    }
}


######################################################################
#
# IDNA (RFC3490)
#
######################################################################

proc jlib::idna_domain_toascii {domain} {
    set domain [string tolower $domain]
    set parts [split $domain "\u002E\u3002\uFF0E\uFF61"]
    set res {}
    foreach p $parts {
	set r [idna_toascii $p]
	lappend res $r
    }
    return [join $res .]
}

proc jlib::idna_toascii {name} {
    # TODO: Steps 2, 3 and 5 from RFC3490

    if {![string is ascii $name]} {
	set name [idna_punycode_encode $name]
	set name "xn--$name"
    }
    return $name
}

proc jlib::idna_punycode_encode {input} {
    set base 36
    set tmin 1
    set tmax 26
    set skew 38
    set damp 700
    set initial_bias 72
    set initial_n 0x80

    set n $initial_n
    set delta 0
    set out 0
    set bias $initial_bias
    set output ""
    set input_length [string length $input]
    set nonbasic {}

    for {set j 0} {$j < $input_length} {incr j} {
	set c [string index $input $j]
	if {[string is ascii $c]} {
	    append output $c
	} else {
	    lappend nonbasic $c
	}
    }

    set nonbasic [lsort -unique $nonbasic]

    set h [set b [string length $output]];

    if {$b > 0} {
	append output -
    }

    while {$h < $input_length} {
	set m [scan [string index $nonbasic 0] %c]
	set nonbasic [lrange $nonbasic 1 end]

	incr delta [expr {($m - $n) * ($h + 1)}]
	set n $m

	for {set j 0} {$j < $input_length} {incr j} {
	    set c [scan [string index $input $j] %c]

	    if {$c < $n} {
		incr delta
	    } elseif {$c == $n} {
		for {set q $delta; set k $base} {1} {incr k $base} {
		    set t [expr {$k <= $bias ? $tmin :
				 $k >= $bias + $tmax ? $tmax : $k - $bias}]
		    if {$q < $t} break;
		    append output \
			[idna_punycode_encode_digit \
			     [expr {$t + ($q - $t) % ($base - $t)}]]
		    set q [expr {($q - $t) / ($base - $t)}]
		}

		append output [idna_punycode_encode_digit $q]
		set bias [idna_punycode_adapt \
			      $delta [expr {$h + 1}] [expr {$h == $b}]]
		set delta 0
		incr h
	    }
	}
	
	incr delta
	incr n
    }

    return $output;
}

proc jlib::idna_punycode_adapt {delta numpoints firsttime} {
    set base 36
    set tmin 1
    set tmax 26
    set skew 38
    set damp 700

    set delta [expr {$firsttime ? $delta / $damp : $delta >> 1}]
    incr delta [expr {$delta / $numpoints}]

    for {set k 0} {$delta > (($base - $tmin) * $tmax) / 2}  {incr k $base} {
	set delta [expr {$delta / ($base - $tmin)}];
    }

    return [expr {$k + ($base - $tmin + 1) * $delta / ($delta + $skew)}]
}

proc jlib::idna_punycode_encode_digit {d} {
    return [format %c [expr {$d + 22 + 75 * ($d < 26)}]]
}

######################################################################
#
# Now that we're done...
#

package provide jabberlib 0.9.0

