
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>

#include <debug/memory.h>
#include <debug/log.h>

#include <abz/typedefs.h>
#include <abz/atop.h>
#include <abz/atoa.h>
#include <abz/error.h>

#include <ber/ber.h>

#include <tinysnmp/tinysnmp.h>
#include <tinysnmp/manager/snmp.h>

#include "show.h"

/* default to 5 seconds */
#define TIMEOUT 5000

/* default to no retries */
#define RETRIES 0

/* we determine our function by checking how we were invoked */
#define I_AM_SNMPGET		0
#define I_AM_SNMPGETNEXT	1
#define I_AM_SNMPWALK		2
#define I_AM_SNMPTABLE		3

typedef struct
{
   int whatami;					/* what are we supposed to do? */
   struct sockaddr_in addr;		/* address/port of agent */
   const char *community;		/* community string */
   uint32_t timeout;			/* timeout in milliseconds between operations */
   uint16_t retries;			/* number of times we try to do a transaction */
   uint32_t rows;				/* maximum number of rows */
   uint32_t **oid;				/* list of Object Identifiers to query on agent */
   uint32_t n;					/* number of OIDs in list */
} cmdline_t;

typedef struct
{
   const char *s;
   const char *l;
   const char **var;
} option_t;

typedef int (*applet_t) (snmp_agent_t *,uint32_t *const *,size_t);

#define PRINTABLE(x) (isprint(x) || isspace(x))

#define usage(p,w) help_stub(p,w,0)
#define help(p,w) help_stub(p,w,1)
static void help_stub (const char *progname,int whatami,int verbose)
{
#ifndef GETSERVBYNAME
   static const char sopt[] =
	 "    -p | --port=PORT         The port which the server binds to. By default\n"
	 "                             port 161 is used.\n";
#else	/* #ifndef GETSERVBYNAME */
   static const char sopt[] =
	 "    -p | --port=SERVICE      The port which the server binds to. You may\n"
	 "                             specify either a service as found in /etc/services\n"
	 "                             or a valid port number. By default, snmp/udp is\n"
	 "                             searched for in the /etc/services file and the\n"
	 "                             resulting port number is used. Failing that, port\n"
	 "                             161 is used.\n";
#endif	/* #ifndef GETSERVBYNAME */
   static const char *args[] =
	 {
		[I_AM_SNMPGET]     "<objectID> [[objectID] ... ]",
		[I_AM_SNMPGETNEXT] "[objectID [[objectID] ... ]]",
		[I_AM_SNMPWALK]    "[objectID]",
		[I_AM_SNMPTABLE]   "<objectID> [[objectID] ... ]"
	 };
#ifndef GETHOSTBYNAME
   static const char host[] = "ip-address";
   static const char hopt[] =
	 "    <ip-address>             IP-address of agent.\n";
#else	/* #ifndef GETHOSTBYNAME */
   static const char host[] = "hostname";
   static const char hopt[] =
	 "    <hostname>               Hostname or ip-address of agent.\n";
#endif	/* #ifndef GETHOSTBYNAME */

   log_printf (LOG_ERROR,
			   "usage: %s [OPTIONS] <%s> <community> %s\n"
			   "       %s -h | --help\n",
			   progname,host,args[whatami],
			   progname);

   if (verbose)
	 {
		log_printf (LOG_ERROR,
					"\n"
					"where OPTIONS can be any of the following:\n"
					"\n"
					"%s"
					"\n"
					"    -t | --timeout=SECONDS   Timeout in seconds between operations. This\n"
					"                             setting affects how long the client waits for the\n"
					"                             agent to respond to queries. The default is %d\n"
					"                             seconds.\n"
					"\n"
					"    -r | --retries=NUM       Number of times that the client will try to resend\n"
					"                             requests to the agent. The default is %d. Any\n"
					"                             number of retries between 0 and 65535 is allowed.\n",
					sopt,TIMEOUT / 1000,RETRIES);

		if (whatami == I_AM_SNMPTABLE)
		  log_printf (LOG_ERROR,
					  "\n"
					  "    -m | --max-rows=ROWS     Maximum number of rows to fetch. The default is\n"
					  "                             unlimited. Any number above 0 and unlimited is\n"
					  "                             allowed.\n");

		log_printf (LOG_ERROR,
					"\n"
					"    -h | --help              Show this help message.\n"
					"\n"
					"    <community>              The community string which clients must use to\n"
					"                             connect to the server.\n"
					"\n"
					"%s"
					"\n"
					"    <objectID>               A list of Object Identifiers (OIDs) that should be\n"
					"                             retrieved from the agent. OIDs should be absolute,\n"
					"                             i.e. no assumptions is made about them and does not\n"
					"                             start with a dot (e.g. 1.3.6.1 is a valid OID, but\n"
					"                             .1.3.6.1 is not)\n"
					"\n",
					hopt);
	 }

   exit (EXIT_FAILURE);
}

static int parse_option (option_t *option,int n,int *counter,int argc,char *const argv[])
{
   int i;
   size_t len;

   for (i = 0; i < n; i++)
	 {
		len = strlen (option[i].l);
		if (!strcmp (argv[*counter],option[i].s))
		  {
			 if ((*counter)++ >= argc || *option[i].var != NULL) return (-1);
			 *option[i].var = argv[*counter];
			 return (0);
		  }
		else if (!strncmp (argv[*counter],option[i].l,len))
		  {
			 if (*option[i].var != NULL) return (-1);
			 *option[i].var = argv[*counter] + len;
			 return (0);
		  }
	 }

   return (-1);
}

static void parse_cmdline (cmdline_t *cmd,int argc,char *argv[])
{
   const char *progname,*service = NULL,*timeout = NULL,*retries = NULL,*rows = NULL;
   option_t common[] =
	 {
		{ "-p", "--port=", &service },
		{ "-t", "--timeout=",&timeout },
		{ "-r", "--retries=",&retries }
	 };
   option_t table[] =
	 {
		{ "-m", "--max-rows=", &rows }
	 };
   option_t options[ARRAYSIZE (common) + ARRAYSIZE (table)];
   unsigned long int value;
   char *endptr;
   int i,j,n = 0;

   /* make sure that we're invoked correctly */
   (progname = strrchr (argv[0],'/')) ? progname++ : (progname = argv[0]);
   if (!strcmp (progname,"tinysnmpget"))
	 cmd->whatami = I_AM_SNMPGET;
   else if (!strcmp (progname,"tinysnmpgetnext"))
	 cmd->whatami = I_AM_SNMPGETNEXT;
   else if (!strcmp (progname,"tinysnmpwalk"))
	 cmd->whatami = I_AM_SNMPWALK;
   else if (!strcmp (progname,"tinysnmptable"))
	 cmd->whatami = I_AM_SNMPTABLE;
   else
	 {
		log_printf (LOG_ERROR,
					"This program should be invoked as one of the following commands:\n"
					"     tinysnmpget\n"
					"     tinysnmpgetnext\n"
					"     tinysnmpwalk\n"
					"     tinysnmptable\n");
		exit (EXIT_FAILURE);
	 }

   /* is the user requesting help? */
   if (argc == 2 && (!strcmp (argv[1],"-h") || !strcmp (argv[1],"--help")))
	 help (progname,cmd->whatami);

   if (argc < 3) usage (progname,cmd->whatami);

   /* what options should we parse? */
   for (i = 0; i < ARRAYSIZE (common); i++)
	 options[n++] = common[i];
   if (cmd->whatami == I_AM_SNMPTABLE)
	 for (i = 0; i < ARRAYSIZE (table); i++)
	   options[n++] = table[i];

   /* [OPTIONS] */
   for (i = 1; i < argc - 2 && argv[i][0] == '-'; i++)
	 if (parse_option (options,n,&i,argc - 2,argv) < 0)
	   usage (progname,cmd->whatami);

   /* [--timeout=SECONDS] */
   if (timeout != NULL)
	 {
		value = strtoul (timeout,&endptr,10);
		if (*timeout == '\0' || *endptr != '\0' || value >= ULONG_MAX / 1000 || value >= 0xffffffffUL / 1000)
		  usage (progname,cmd->whatami);
		cmd->timeout = value * 1000;
	 }
   else cmd->timeout = TIMEOUT;

   /* [--retries=NUM] */
   if (retries != NULL)
	 {
		value = strtoul (retries,&endptr,10);
		if (*retries == '\0' || *endptr != '\0' || value > 65535)
		  usage (progname,cmd->whatami);
		cmd->retries = value;
	 }
   else cmd->retries = RETRIES;

   /* [--max-rows=NUM] */
   if (rows != NULL)
	 {
		if (strcmp (rows,"unlimited"))
		  {
			 value = strtoul (rows,&endptr,10);
			 if (*rows == '\0' || *endptr != '\0' || !value)
			   usage (progname,cmd->whatami);
			 cmd->rows = value;
		  }
		else cmd->rows = 0;
	 }
   else cmd->rows = 0;

   memset (&cmd->addr,0L,sizeof (cmd->addr));
   cmd->addr.sin_family = PF_INET;

   /* <hostname> / <ip-address> */
   if (atoa (&cmd->addr.sin_addr.s_addr,argv[i]))
	 usage (progname,cmd->whatami);

   /* [--port=SERVICE] */
   if (service != NULL)
	 {
		if (atop (&cmd->addr.sin_port,service))
		  usage (progname,cmd->whatami);
	 }
   else
	 {
#ifdef GETSERVBYNAME
		if (atop (&cmd->addr.sin_port,"snmp"))
#endif	/* #ifdef GETSERVBYNAME */
		  cmd->addr.sin_port = htons (161);
	 }

   /* <community> */
   cmd->community = argv[++i];

   /* OIDs */
   cmd->n = argc - i - 1;

   if ((!cmd->n && cmd->whatami == I_AM_SNMPGET) ||
	   (cmd->n > 1 && cmd->whatami == I_AM_SNMPWALK) ||
	   (!cmd->n && cmd->whatami == I_AM_SNMPTABLE))
	 usage (progname,cmd->whatami);

   if (!cmd->n)
	 cmd->oid = NULL;
   else
	 {
		if ((cmd->oid = mem_alloc (cmd->n * sizeof (uint32_t *))) == NULL)
		  {
			 log_printf (LOG_ERROR,"failed to allocate memory: %m\n");
			 exit (EXIT_FAILURE);
		  }

		for (j = i + 1, i = 0; i < cmd->n; i++, j++)
		  if ((cmd->oid[i] = makeoid (argv[j])) == NULL)
			{
			   log_printf (LOG_ERROR,"invalid ObjectID %s: %s\n",argv[j],abz_get_error ());
			   for (j = 0; j < i; j++) mem_free (cmd->oid[j]);
			   mem_free (cmd->oid);
			   exit (EXIT_FAILURE);
			}
	 }
}

static int show_get_request (snmp_agent_t *agent,uint32_t *const *oid,size_t n)
{
   snmp_value_t *values;
   int i;

   if ((values = snmp_get (agent,oid,n)) == NULL)
	 return (-1);

   for (i = 0; i < n; i++) show (oid[i],values[i].type,&values[i].data);

   snmp_free_values (&values,n);

   return (0);
}

static int show_get_next_request (snmp_agent_t *agent,uint32_t *const *oid,size_t n)
{
   snmp_next_value_t *next;
   size_t i;

   if ((next = snmp_get_next (agent,oid,n)) == NULL)
	 return (-1);

   if (!n) n = 1;

   for (i = 0; i < n; i++) show (next[i].oid,next[i].value.type,&next[i].value.data);

   snmp_free_next_values (&next,n);

   return (0);
}

static int subtree (uint32_t *const base,uint32_t *const leaf)
{
   uint32_t i = 0;

   if (base[i] >= leaf[0])
	 return (0);

   for (i = 1; i <= base[0]; i++)
	 if (base[i] != leaf[i])
	   return (0);

   return (1);
}

static int show_walk (snmp_agent_t *agent,uint32_t *const *oid,size_t n)
{
   snmp_next_value_t *prev,*next;

   if ((next = snmp_get_next (agent,oid,n)) == NULL)
	 return (-1);

   if (!n) n = 1;

   while (next->value.type != BER_NULL && (oid == NULL || subtree (oid[0],next->oid)))
	 {
		show (next->oid,next->value.type,&next->value.data);
		prev = next;

		if ((next = snmp_get_next (agent,&prev->oid,n)) == NULL)
		  {
			 snmp_free_next_values (&prev,n);
			 return (-1);
		  }

		snmp_free_next_values (&prev,n);
	 }

   snmp_free_next_values (&next,n);

   return (0);
}

static int show_tables (snmp_agent_t *agent,uint32_t *const *oid,size_t n)
{
   snmp_table_t *table;
   size_t i;
   uint32_t j;

   if ((table = snmp_get_tables (agent,oid,n,0)) == NULL)
	 return (-1);

   for (i = 0; i < n; i++)
	 for (j = 0; j < table[i].n; j++)
	   show (table[i].oid[j],table[i].type,table[i].data + j);

   snmp_free_tables (&table,n);

   return (0);
}

int main (int argc,char *argv[])
{
   cmdline_t cmd;
   snmp_agent_t agent;
   int status;
   static const applet_t applet[] =
	 {
		[I_AM_SNMPGET]     show_get_request,
		[I_AM_SNMPGETNEXT] show_get_next_request,
		[I_AM_SNMPWALK]    show_walk,
		[I_AM_SNMPTABLE]   show_tables
	 };

   mem_open (NULL);
   log_open (NULL,LOG_NORMAL,LOG_HAVE_COLORS);
   atexit (log_close);
   atexit (mem_close);

   parse_cmdline (&cmd,argc,argv);

   if (snmp_open (&agent,&cmd.addr,cmd.community,cmd.timeout,cmd.retries) < 0)
	 exit (EXIT_FAILURE);

   status = applet[cmd.whatami] (&agent,cmd.oid,cmd.n) < 0 ? EXIT_FAILURE : EXIT_SUCCESS;

   if (status == EXIT_FAILURE)
	 log_printf (LOG_ERROR,"%s\n",abz_get_error ());

   snmp_close (&agent);

   if (cmd.n)
	 {
		uint32_t i;
		for (i = 0; i < cmd.n; i++) mem_free (cmd.oid[i]);
		mem_free (cmd.oid);
	 }

   exit (status);
}

