/* -*- mode: c; c-file-style: "gnu" -*-
 * nqueue_devpoll.c - /dev/poll-based event system
 * Copyright (C) 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "system.h"

#if defined(HAVE_SYS_DEVPOLL_H) || defined(HAVE_LINUX_DEVPOLL_H)
#include <sys/types.h>
#include <inttypes.h>

#include <errno.h>
#include <fcntl.h>
#include <string.h>
#if HAVE_SYS_DEVPOLL_H
#include <sys/devpoll.h>
#endif
#if HAVE_LINUX_DEVPOLL_H
#include <linux/devpoll.h>
#endif
#if HAVE_SYS_POLL_H
#include <sys/poll.h>
#endif
#include <sys/time.h>
#include <unistd.h>

#include "compat/compat.h"

#include "nqueue.h"
#include "nqueue_devpoll.h"
#include "types.h"
#include "thy.h"

static struct dvpoll events;
static int dpfd; /**< /dev/poll FD. */
static int *rfdidx; /** FD->returned event lookup table. */

int
thy_nq_init_devpoll (void)
{
  dpfd = open ("/dev/poll", O_RDWR);
  if (dpfd == -1)
    {
      bhc_error ("open(/dev/poll): %s", strerror (errno));
      return -1;
    }

  events.dp_fds = (struct pollfd *)bhc_calloc (_THY_MAXCONN + 1,
					       sizeof (struct pollfd));
  rfdidx = (int *)bhc_calloc (_THY_MAXCONN + 1, sizeof (int));

  return 0;
}

void
thy_nq_fd_control_devpoll (int fd, thy_nq_event_t event, int set)
{
  struct pollfd pfd;
  int rem = 0;

  if (fd < 0)
    return;

  if (fdstates[fd])
    rem = 1;

  if (rem && thy_nq_act_fds)
    thy_nq_act_fds--;

  if (trigger[fd])
    trigger_count--;

  if (fd > thy_nq_max_fd)
    thy_nq_max_fd = fd;

  fdstates[fd] = 0;
  trigger[fd] = 0;

  pfd.fd = fd;
  switch (event)
    {
    case THY_NQ_EVENT_INPUT:
      pfd.events = POLLIN | POLLHUP | POLLERR;
      break;
    case THY_NQ_EVENT_OUTPUT:
      pfd.events = POLLOUT | POLLHUP | POLLERR;
      break;
    case THY_NQ_EVENT_TRIGGER:
      trigger_count++;
      trigger[fd] = 1;
      return;
    default:
      pfd.events = POLLIN | POLLOUT | POLLHUP | POLLERR;
      break;
    }

  if (set)
    {
      fdstates[fd] = 1;
      thy_nq_act_fds++;
    }
  else
    if (rem)
      pfd.events = POLLREMOVE;

  write (dpfd, &pfd, sizeof (struct pollfd));
}

int
thy_nq_fd_check_devpoll (int fd, thy_nq_event_t event)
{
  int ridx, rval = 0;

  if (fd > thy_nq_max_fd)
    thy_nq_max_fd = fd;

  if (event == THY_NQ_EVENT_TRIGGER && trigger[fd])
    return 1;

  ridx = rfdidx[fd];

  if (events.dp_fds[ridx].fd != fd)
    return 0;

  switch (event)
    {
    case THY_NQ_EVENT_INPUT:
      if (events.dp_fds[ridx].revents & (POLLIN | POLLHUP | POLLERR))
	rval = 1;
      break;
    case THY_NQ_EVENT_OUTPUT:
      if (events.dp_fds[ridx].revents & (POLLOUT | POLLHUP | POLLERR))
	rval = 1;
      break;
    default:
      return 0;
    }

  return rval;
}

int
thy_nq_wait_devpoll (long int timeout)
{
  int i, r;

  errno = 0;
  events.dp_nfds = _THY_MAXCONN;
  events.dp_timeout = timeout * 1000;
  r = ioctl (dpfd, DP_POLL, &events);

  if (r == -1)
    return -1;

  for (i = 0; i < r; i++)
    rfdidx[events.dp_fds[i].fd] = i;

  return r;
}
#endif

/* arch-tag: 15ffeda8-b921-4cbc-8118-db78b952b37d */
