/* -*- mode: c; c-file-style: "gnu" -*-
 * tls.c -- TLS related functions.
 * Copyright (C) 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file tls.c
 * TLS abstraction layer.
 *
 * This module contains the functions to turn a simple #session_t into
 * a TLS enabled session. And most of the TLS handling code is also
 * here (except the networking parts, which are in network.c): prime
 * generation and TLS session resume database handling stuff (these
 * routines are called by GnuTLS internally).
 */

#include "options.h"
#include "tls.h"

#if THY_OPTION_TLS

#include <fcntl.h>
#include <gnutls/extra.h>
#include <gnutls/gnutls.h>
#include <gnutls/x509.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>

#include "compat/compat.h"
#include "bh-libs/list.h"

#include "cgi.h"
#include "config.h"
#include "misc.h"
#include "primes.h"
#include "thy.h"
#include "types.h"

/** @internal Convert a value to hex. */
#define HEX(x) (((x)>9)?(('a'-10)+(x)):('0'+(x)))

/** X509 Credentials. */
static gnutls_certificate_server_credentials cred;
static gnutls_srp_server_credentials srp_cred; /**< SRP credentials. */
static gnutls_dh_params dh_params; /**< Diffie Hellman parameters. */
static gnutls_rsa_params rsa_params; /**< RSA parameters. */

static int dh_bits = 1024; /**< Number of DH key bits. */
/** Caching variable to hold config->ssl.cert_type. */
static thy_tls_cert_type_t cert_type = THY_TLS_CERT_TYPE_X509;
/** Caching variable to hold config->ssl.verify_level. */
static int verify_level = 0;

static bhl_list_t *thy_tls_cache; /**< TLS Session cache. */

static void thy_tls_db_init (void);
static int thy_tls_db_store (void *dbf, gnutls_datum key,
			     gnutls_datum data);
static gnutls_datum thy_tls_db_fetch(void *dbf, gnutls_datum key);
static int thy_tls_db_delete (void *dbf, gnutls_datum key);

/** @internal Type of an entry in the TLS session cache.
 */
typedef struct
{
  char *session_id; /**< ID of a TLS session. */
  size_t session_id_size; /**< Size of the ID. */

  char *session_data; /**< TLS session data. */
  size_t session_data_size; /**< Size of the TLS session data. */
} thy_tls_cache_t;

/** @internal Generic gnutls_x509_get_*_dn prototype.
 */
typedef int (*_thy_tls_gnutls_dn_wrap_f) (gnutls_x509_crt cert,
					  char *buf, size_t *size);

/** @internal Get a DN of a certificate.
 * Given a certificate and a function to get a DN out of it, allocates
 * memory for the DN, then copies the stuff there.
 *
 * @param cert is the certificate to work on.
 * @param wrapper is the appropriate GnuTLS function.
 *
 * @returns A newly allocated string on success, NULL otherwise.
 *
 * @note The returned value must be freed by the caller!
 */
static char *
_thy_tls_gnutls_dn_wrap (gnutls_x509_crt cert,
			 _thy_tls_gnutls_dn_wrap_f wrapper)
{
  char *buf;
  size_t size = 0;

  (*wrapper) (cert, NULL, &size);
  size += 2;
  buf = (char *)bhc_malloc (size);
  if ((*wrapper) (cert, buf, &size) < 0)
    {
      free (buf);
      return NULL;
    }
  return buf;
}

/** @internal Extract a certificate serial number.
 * Given a certificate, extracts the serial number, and returns it as
 * a HEX string.
 *
 * @param cert is the certificate.
 *
 * @returns The serial as a HEX string, or NULL on error.
 *
 * @note The returned string MUST NOT be freed by the caller.
 */
static const char *
_thy_tls_gnutls_crt_get_serial (gnutls_x509_crt cert)
{
  char serial[64];
  static char str_serial[129];
  size_t size, i;

  size = sizeof (serial);
  if (gnutls_x509_crt_get_serial (cert, serial, &size) >= 0)
    {
      char *p = str_serial;

      for (i = 0; i < size; i++)
	{
	  *p++ = HEX((serial[i] >> 4) & 0x0f);
	  *p++ = HEX((serial[i]) & 0x0f);
	}
      *p = 0;

      return str_serial;
    }
  return NULL;
}

/** @internal Check & Log the alert we got.
 * @param session is the TLS session.
 * @param ret is the value returned by the last TLS-related call.
 */
static void
_thy_tls_alert_check (gnutls_session session, int ret)
{
  gnutls_alert_description last_alert;

  if (ret == GNUTLS_E_WARNING_ALERT_RECEIVED
      || ret == GNUTLS_E_FATAL_ALERT_RECEIVED)
    {
      last_alert = gnutls_alert_get (session);
      if (last_alert == GNUTLS_A_NO_RENEGOTIATION &&
	  ret == GNUTLS_E_WARNING_ALERT_RECEIVED)
	{
	  bhc_log ("%s", "TLS: Client does not support renegotiation.");
	}
      else
	{
	  bhc_log ("TLS: Received alert '%d': %s.", (int)last_alert,
		   gnutls_alert_get_name (last_alert));
	}
    }
}

/** @internal Callback function.
 * This function gets called by GnuTLS to determine which certificate
 * to use.
 *
 * It checks if the client indicated which host it is connecting to,
 * and if yes, tries to find an appropriate cert.
 *
 * @returns The most appropriate certificate's index on success, zero
 * (the first cert) if no other certificate is appropriate.
 */
static int
_thy_tls_session_cert_select (gnutls_session session,
			      gnutls_datum *server_certs, int ncerts)
{
  size_t data_length = 0;
  unsigned int name_type;
  int idx, i;
  char *name, *aname = (char *)gnutls_session_get_ptr (session);

  /* If we do not have any certs, report an error. */
  if (ncerts <= 0)
    return -1;

  /* If we only have one cert, return that. */
  if (ncerts < 2)
    return 0;

  /* Get the length of the data. Note that we only care about the
     first hostname sent. At least, for now. */
  data_length = 0;
  i = gnutls_server_name_get (session, NULL, &data_length,
			      &name_type, 0);

  /* And get the data itself */
  if (i == GNUTLS_E_SHORT_MEMORY_BUFFER)
    {
      data_length += 2;
      name = (char *)bhc_malloc (data_length);
      gnutls_server_name_get (session, name, &data_length, &name_type, 0);
      if (name_type != GNUTLS_NAME_DNS)
	{
	  free (name);
	  return 0;
	}
    }
  else
    {
      if (aname)
	name = bhc_strdup (aname);
      else
	return 0;
    }

  /* Iterate through the certs and select the appropriate one. */
  for (idx = 0; idx < ncerts; idx++)
    {
      gnutls_x509_crt crt;

      gnutls_x509_crt_init (&crt);
      gnutls_x509_crt_import (crt, &server_certs[idx], GNUTLS_X509_FMT_DER);

      if (gnutls_x509_crt_check_hostname (crt, name) != 0)
	{
	  gnutls_x509_crt_deinit (crt);
	  free (name);
	  return idx;
	}

      gnutls_x509_crt_deinit (crt);
    }

  free (name);
  return 0;
}

/** @internal Verify the client's certificate if any.
 * Depending on the value of #verify_level, perform a verification on
 * the client's certificate if any.
 *
 * @param session is the session to verify.
 *
 * @returns Zero on success, 1 otherwise.
 */
static int
_thy_tls_session_verify (session_t *session)
{
  session->cert_verified = "NONE";

  if (verify_level >= 1)
    {
      int verify, ret, valid;
      char *name = NULL;
      const gnutls_datum *cert_list;
      int cert_list_size;
      gnutls_x509_crt crt = NULL;

      ret = gnutls_x509_crt_init( &crt);
      if (ret < 0)
	{
	  bhc_error ("Error in gnutls_x509_crt_init(): %s",
		     gnutls_strerror (ret));
	  return 1;
	}

      cert_list = gnutls_certificate_get_peers (session->tls_session,
						&cert_list_size);

      if (cert_list)
	{
	  ret = gnutls_x509_crt_import (crt, &cert_list[0],
					GNUTLS_X509_FMT_DER);
	  if (ret < 0)
	    {
	      bhc_error (" Could not import X.509 certificate: %s",
			 gnutls_strerror (ret));
	      return 1;
	    }

	  name = _thy_tls_gnutls_dn_wrap (crt, gnutls_x509_crt_get_dn);
	  if (!name)
	    name = strdup ("Unknown");
	}

      verify = gnutls_certificate_verify_peers (session->tls_session);

      if (cert_list == NULL)
	{
	  bhc_error ("%s", "Peer did not send a certificate.");
	  if (verify_level >= 2)
	    {
	      free (name);
	      return 1;
	    }
	}
      else
	{
	  char *err, *tmp;

	  valid = 0;

	  if (gnutls_x509_crt_get_expiration_time (crt) < time (NULL))
	    {
	      asprintf (&err, "X.509 Certificate by '%s' is %s",
			name, "Expired");
	      valid = 1;
	    }

	  if (gnutls_x509_crt_get_activation_time( crt) > time (NULL))
	    {
	      if (!valid)
		asprintf (&err, "X.509 Certificate by '%s' is %s",
			  name, "Not yet activated");
	      valid = 1;
	  }

	  if (!valid)
	    asprintf (&err, "X.509 Certificate by '%s' is ", name);

	  if (valid || verify & GNUTLS_CERT_INVALID ||
	      verify & GNUTLS_CERT_REVOKED)
	    {
	      int revoked = 0, signer_not_known = 0, signer_not_ca = 0;
	      char *e;

	      session->cert_verified = "FAILED";
	      if (verify & GNUTLS_CERT_REVOKED)
		revoked = 1;
	      if (verify & GNUTLS_CERT_SIGNER_NOT_FOUND)
		signer_not_known = 1;
	      if (verify & GNUTLS_CERT_SIGNER_NOT_CA)
		signer_not_ca = 1;

	      asprintf (&e, "%s%s%s%s%s.", (valid) ? "" : ", ",
			"NOT trusted", (revoked) ? ", Revoked" : "",
			(signer_not_known) ? ", Issuer not known" : "",
			(signer_not_ca) ? ", Issuer is not a CA" : "");
	      asprintf (&tmp, "%s%s", err, e);
	      free (err);
	      free (e);
	      err = tmp;

	      gnutls_x509_crt_deinit (crt);
	      free (name);

	      bhc_log ("%s", err);

	      return 1;
	    }
	  else
	    {
	      session->cert_verified = "SUCCESS";
	      asprintf (&tmp, "%s%s", err, "trusted.");
	      free (err);
	      err = tmp;
	    }
	  bhc_log ("%s", err);
	  free (name);
	}

      gnutls_x509_crt_deinit (crt);
    }

  return 0;
}

/** @internal Create a GnuTLS session.
 * Initialises the cyphers and the session database for a new TLS
 * session.
 *
 * @returns The newly created TLS session.
 */
static gnutls_session
_thy_tls_session_create (void)
{
  gnutls_session session;
  const int cert_type_prio[] = { GNUTLS_CRT_OPENPGP, 0 };
  const int comp_prio[] = { GNUTLS_COMP_ZLIB, GNUTLS_COMP_LZO,
				GNUTLS_COMP_NULL, 0 };
  const int mac_prio[] = { GNUTLS_MAC_SHA, GNUTLS_MAC_MD5, 0 };
  const int kx_prio[] = { GNUTLS_KX_DHE_DSS, GNUTLS_KX_RSA,
			  GNUTLS_KX_DHE_RSA, GNUTLS_KX_SRP,
			  GNUTLS_KX_SRP_DSS, GNUTLS_KX_SRP_RSA,
			  GNUTLS_KX_RSA_EXPORT, 0 };
  const int cipher_prio[] = { GNUTLS_CIPHER_RIJNDAEL_128_CBC,
			      GNUTLS_CIPHER_3DES_CBC,
			      GNUTLS_CIPHER_ARCFOUR_128,
			      GNUTLS_CIPHER_ARCFOUR_40, 0 };
  const int protocol_prio[] = { GNUTLS_TLS1, GNUTLS_SSL3, 0 };

  gnutls_init (&session, GNUTLS_SERVER);

  gnutls_cipher_set_priority (session, cipher_prio);
  gnutls_compression_set_priority (session, comp_prio);
  gnutls_kx_set_priority (session, kx_prio);
  gnutls_protocol_set_priority (session, protocol_prio);
  gnutls_mac_set_priority (session, mac_prio);

  if (cert_type == THY_TLS_CERT_TYPE_OPENPGP)
    gnutls_certificate_type_set_priority (session, cert_type_prio);

  gnutls_credentials_set (session, GNUTLS_CRD_SRP, srp_cred);
  gnutls_credentials_set (session, GNUTLS_CRD_CERTIFICATE, cred);

  if (cert_type == THY_TLS_CERT_TYPE_OPENPGP)
    gnutls_certificate_server_set_request (session, GNUTLS_CERT_REQUEST);
  else
    {
      if (verify_level == 1)
	{
	  gnutls_certificate_server_set_request (session,
						 GNUTLS_CERT_REQUEST);
	}
      else if (verify_level >= 2)
	{
	  gnutls_certificate_server_set_request (session,
						 GNUTLS_CERT_REQUIRE);
	}
      else
	{
	  gnutls_certificate_server_set_request (session,
						 GNUTLS_CERT_IGNORE);
	}
    }

  gnutls_dh_set_prime_bits (session, dh_bits);

  gnutls_db_set_retrieve_function (session, thy_tls_db_fetch);
  gnutls_db_set_remove_function (session, thy_tls_db_delete);
  gnutls_db_set_store_function (session, thy_tls_db_store);
  gnutls_db_set_ptr (session, NULL);

  gnutls_handshake_set_private_extensions (session, 1);

  if (cert_type == THY_TLS_CERT_TYPE_X509)
    gnutls_certificate_server_set_select_function
      (session, (gnutls_certificate_server_select_function *)
       _thy_tls_session_cert_select);

  return session;
}

/** @internal (Re)Initialise Diffie Hellman parameters.
 * @returns Zero.
 */
static int
thy_dh_params_generate (void)
{
  if (gnutls_dh_params_init (&dh_params) < 0)
    {
      bhc_error ("%s", "Error in dh parameter initialisation.");
      bhc_exit (3);
    }

  if (gnutls_dh_params_generate2 (dh_params, dh_bits) < 0)
    {
      bhc_error ("%s", "Error in prime generation.");
      bhc_exit (3);
    }

  gnutls_certificate_set_dh_params (cred, dh_params);

  return 0;
}

/** @internal (Re)Initialise RSA parameters.
 * @returns Zero.
 */
static int
thy_rsa_params_generate (void)
{
  if (gnutls_rsa_params_init (&rsa_params) < 0)
    {
      bhc_error ("%s", "Error in RSA parameter initialisation.");
      bhc_exit (3);
    }

  if (gnutls_rsa_params_generate2 (rsa_params, 512) < 0)
    {
      bhc_error ("%s", "Error in RSA parameter generation.");
      bhc_exit (3);
    }

  gnutls_certificate_set_rsa_params (cred, rsa_params);

  return 0;
}

/** Initialise the TLS engine.
 * This one initialises the TLS engine and sets up everything needed
 * to process TLS requests.
 *
 * @note Should be called only once, upon startup.
 */
void
thy_tls_init (void)
{
  const config_t *config = config_get ();

  cert_type = config->ssl.cert_type;
  verify_level = config->ssl.verify_level;

  if (thy_tls_enabled == THY_BOOL_FALSE)
    return;

  if (gnutls_check_version (LIBGNUTLS_VERSION) == NULL ||
      gnutls_extra_check_version (LIBGNUTLS_EXTRA_VERSION) == NULL)
    {
      bhc_error ("%s", "GnuTLS version check failed.");
      bhc_exit (3);
    }

  if (gnutls_global_init () < 0)
    {
      bhc_error ("%s", "Global TLS state initialisation failed.");
      bhc_exit (3);
    }

  if (gnutls_global_init_extra () < 0)
    {
      bhc_error ("%s", "Global TLS state initialisation failed.");
      bhc_exit (3);
    }

  if (gnutls_certificate_allocate_credentials (&cred) < 0)
    {
      bhc_error ("%s", "Couldn't allocate credentials.");
      bhc_exit (3);
    }

  if (gnutls_srp_allocate_server_credentials (&srp_cred) < 0)
    {
      bhc_error ("%s", "Couldn't allocate credentials.");
      bhc_exit (3);
    }

  if (config->ssl.srp.passwd != NULL &&
      config->ssl.srp.conf != NULL)
    if (gnutls_srp_set_server_credentials_file
	(srp_cred, config->ssl.srp.passwd, config->ssl.srp.conf) < 0)
      {
	bhc_error ("%s", "Couldn't initialise SRP credentials.");
	bhc_exit (3);
      }

  if (cert_type == THY_TLS_CERT_TYPE_X509)
    {
      size_t i;

      for (i = 0; i < bhl_list_size (config->ssl.x509.cafiles); i++)
	{
	  char *ca;

	  bhl_list_get (config->ssl.x509.cafiles, i, (void **)&ca);
	  if (gnutls_certificate_set_x509_trust_file
	      (cred, ca, GNUTLS_X509_FMT_PEM) < 0)
	    {
	      if (thy_tls_enabled == THY_BOOL_UNSET)
		{
		  thy_tls_enabled = THY_BOOL_FALSE;
		  return;
		}
	      bhc_error ("%s", "Error reading X.509 CA Certificate.");
	      bhc_exit (3);
	    }
	  free (ca);
	}

      for (i = 0; i < bhl_list_size (config->ssl.x509.certfiles); i++)
	{
	  char *cert, *key;

	  bhl_list_get (config->ssl.x509.certfiles, i, (void **)&cert);
	  bhl_list_get (config->ssl.x509.keyfiles, i, (void **)&key);

	  if (gnutls_certificate_set_x509_key_file
	      (cred, cert, key, GNUTLS_X509_FMT_PEM) < 0)
	    {
	      if (thy_tls_enabled == THY_BOOL_UNSET)
		{
		  thy_tls_enabled = THY_BOOL_FALSE;
		  return;
		}
	      bhc_error ("%s",
			 "Error reading X.509 key or certificate file.");
	      bhc_exit (3);
	    }

	  free (cert);
	  free (key);
	}
    }

  if (cert_type == THY_TLS_CERT_TYPE_OPENPGP)
    {
      if (gnutls_certificate_set_openpgp_keyring_file
	  (cred, config->ssl.openpgp.keyring) < 0)
	{
	  if (thy_tls_enabled == THY_BOOL_UNSET)
	    {
	      thy_tls_enabled = THY_BOOL_FALSE;
	      return;
	    }
	  bhc_error ("%s", "Error reading OpenPGP keyring file.");
	  bhc_exit (3);
	}

      if (gnutls_certificate_set_openpgp_key_file
	  (cred, config->ssl.openpgp.certfile,
	   config->ssl.openpgp.keyfile) < 0)
	{
	  if (thy_tls_enabled == THY_BOOL_UNSET)
	    {
	      thy_tls_enabled = THY_BOOL_FALSE;
	      return;
	    }
	  bhc_error ("%s",
		     "Error reading OpenPGP key or certificate file.");
	  bhc_exit (3);
	}
    }

  thy_tls_reinit (0);
  thy_tls_db_init ();
}

#if THY_OPTION_DEBUG
/** @internal Report some information about the TLS session.
 * Log the used protocol, key exchange method, cipher, compression,
 * etc..
 *
 * @param session is the session to report about.
 */
static void
_thy_tls_session_report (const session_t *session)
{
  gnutls_kx_algorithm kx;
  const char *kx_s, *proto_s, *comp_s, *cipher_s, *mac_s;

  kx = gnutls_kx_get (session->tls_session);
  kx_s = gnutls_kx_get_name (kx);

  proto_s = gnutls_protocol_get_name
    (gnutls_protocol_get_version (session->tls_session));

  comp_s = gnutls_compression_get_name
    (gnutls_compression_get (session->tls_session));

  cipher_s = gnutls_cipher_get_name
    (gnutls_cipher_get (session->tls_session));

  mac_s = gnutls_mac_get_name (gnutls_mac_get (session->tls_session));

  bhc_log ("Connection from %s using %s with %s compression, "
	   "%s cipher, %s MAC, and %s KX%s", session->origin, proto_s,
	   comp_s, cipher_s, mac_s, kx_s,
	   (gnutls_session_is_resumed
	    (session->tls_session) == 0) ? "" : " (Resumed)");
}
#endif

/** Initialise the TLS parts of a session.
 * If TLS is enabled, do the handshake, and set up the TLS parts of
 * the session appropriately.
 *
 * @returns One on success, zero on error.
 *
 * @note Modifies session in-place!
 */
int
thy_tls_session_init (session_t *session)
{
  int ret;
  int handshake = 0;

  if (!thy_tls_istls (session))
    {
      session_state_change (session, SESSION_STATE_INPUT_REQUEST);
      return 1;
    }
  if (session->state != SESSION_STATE_HANDSHAKE)
    return 1;

  if (!session->tls_session)
    {
      session->tls_session = _thy_tls_session_create ();
      gnutls_transport_set_ptr2
	(session->tls_session,
	 (gnutls_transport_ptr)((long)session->io.in),
	 (gnutls_transport_ptr)((long)session->io.out));
      if (session->request)
	{
	  if (session->request->upgrade == THY_UPGRADE_TLS10)
	    gnutls_session_set_ptr (session->tls_session,
				    session->request->host);
	}
    }

  ret = gnutls_handshake (session->tls_session);
  if (ret < 0 && gnutls_error_is_fatal (ret) == 0)
    _thy_tls_alert_check (session->tls_session, ret);
  else
    if (ret < 0 && gnutls_error_is_fatal (ret) == 1)
      {
	int rret = 0;
	do
	  {
	    rret = gnutls_alert_send_appropriate
	      (session->tls_session, rret);
	  } while (rret == GNUTLS_E_AGAIN);
	handshake = -1;
      }
    else
      if (ret == 0)
	handshake = 1;

  switch (handshake)
    {
    case -1:
      bhc_log ("Handshake has failed (%s) with %s", gnutls_strerror (ret),
	       session->origin);
      return 0;
    case 1:
      if (_thy_tls_session_verify (session) != 0)
	return 0;
#if THY_OPTION_DEBUG
      _thy_tls_session_report (session);
#endif
      if (session->request)
	{
	  if (session->request->upgrade != THY_UPGRADE_TLS10_POST)
	    session_state_change (session, SESSION_STATE_INPUT_REQUEST);
	  else
	    session_state_change (session, SESSION_STATE_PROCESS);
	}
      else
	session_state_change (session, SESSION_STATE_INPUT_REQUEST);
      break;
    }

  return 1;
}

int
thy_tls_istls (const session_t *session)
{
  if (session)
    return session->ssl;
  return 0;
}

/** Regenerate DH and RSA params.
 * This should be called once in a while...
 */
void
thy_tls_reinit (int dh_gen)
{
  if (dh_gen)
    thy_dh_params_generate ();
  else
    {
      gnutls_datum dh_prime_bits, dh_generator;

      dh_prime_bits.data = thy_prime_dh_1024;
      dh_prime_bits.size = sizeof (thy_prime_dh_1024);

      dh_generator.data = thy_generator_dh;
      dh_generator.size = sizeof (thy_generator_dh);

      gnutls_dh_params_init (&dh_params);
      gnutls_dh_params_import_raw (dh_params, &dh_prime_bits, &dh_generator);
    }
  thy_rsa_params_generate ();

  gnutls_certificate_set_dh_params (cred, dh_params);
  gnutls_certificate_set_rsa_params (cred, rsa_params);
}

/** @internal Wrapper around gnutls_cipher_suite_get_name().
 * gnutls_cipher_suite_get_name() returns stuff separated with _, but
 * for mod_ssl compatibility, we want -. This function does just that.
 *
 * @param kx is the key exchange algorithm.
 * @param cipher is the cipher used.
 * @param mac is the MAC used.
 *
 * @returns A newly allocated string. Must be freed by the caller!
 */
static char *
_thy_tls_cipher_suite_get_name (gnutls_kx_algorithm kx,
				gnutls_cipher_algorithm cipher,
				gnutls_mac_algorithm mac)
{
  char *suite;
  size_t i = 0;

  suite = strdup (gnutls_cipher_suite_get_name (kx, cipher, mac));

  while (suite[i])
    {
      if (suite[i] == '_')
	suite[i] = '-';
      i++;
    }

  return suite;
}

/** Setup CGI environment for TLS connections.
 * Adds some extra (mod_ssl compatible) environment variables to the
 * CGI environment, if necessary.
 *
 * @param session is the session to work on.
 *
 * @note Most of this only works with GnuTLS 0.9 or later. With GnuTLS
 * 0.8, only HTTPS is set to "on" when in TLS mode.
 */
void
thy_tls_cgi_setup_env (session_t *session)
{
   char session_id[GNUTLS_MAX_SESSION_ID];
   size_t session_id_length = sizeof (session_id);
   int cert_list_size, ret;
   char str_session_id[(GNUTLS_MAX_SESSION_ID * 2) + 1];
   size_t i;
   char *tmp;
   const gnutls_datum *cert_list;
   gnutls_x509_crt crt;

   if (thy_tls_istls (session))
     cgi_addenv (session->request, "HTTPS", "on");
   else
     return;

   switch (gnutls_protocol_get_version (session->tls_session))
     {
     case GNUTLS_SSL3:
       cgi_addenv (session->request, "SSL_PROTOCOL", "SSLv3");
       break;
     case GNUTLS_TLS1:
       cgi_addenv (session->request, "SSL_PROTOCOL", "TLSv1");
       break;
     default:
       cgi_addenv (session->request, "SSL_PROTOCOL",
		   gnutls_protocol_get_name
		   (gnutls_protocol_get_version (session->tls_session)));
     }

   cgi_addenv (session->request, "SSL_CLIENT_VERIFY",
	       session->cert_verified);

   asprintf (&tmp, "GnuTLS/%s", gnutls_check_version (NULL));
   cgi_addenv (session->request, "SSL_VERSION_LIBRARY", tmp);
   free (tmp);

   cgi_addenv (session->request, "SSL_VERSION_INTERFACE",
	       thy_servername (NULL));

   tmp = _thy_tls_cipher_suite_get_name
     (gnutls_kx_get (session->tls_session),
      gnutls_cipher_get (session->tls_session),
      gnutls_mac_get (session->tls_session));
   cgi_addenv (session->request, "SSL_CIPHER", tmp);
   free (tmp);

   i = gnutls_cipher_get_key_size
     (gnutls_cipher_get (session->tls_session)) * 8;
   asprintf (&tmp, SIZET_FORMAT, i);
   cgi_addenv (session->request, "SSL_CIPHER_ALGKEYSIZE", tmp);
   cgi_addenv (session->request, "SSL_CIPHER_USEKEYSIZE", tmp);
   cgi_addenv (session->request, "SSL_CIPHER_EXPORT",
	       (i <= 40) ? "true" : "false");

   /* generate a printable (HEX) session ID */
   if (gnutls_session_get_id (session->tls_session, session_id,
			      &session_id_length) >= 0)
     {
       char *p = str_session_id;

       for (i = 0; i < session_id_length; i++)
	 {
	   *p++ = HEX((session_id[i] >> 4) & 0x0f);
	   *p++ = HEX((session_id[i]) & 0x0f);
	 }
       *p = 0;

       cgi_addenv (session->request, "SSL_SESSION_ID", str_session_id);
     }

   cert_list = gnutls_certificate_get_ours (session->tls_session);

   /* Generate the server's DN */
   if (cert_list)
     {
       time_t vtime;

       ret = gnutls_x509_crt_init (&crt);
       if (ret < 0)
	 return;

       ret = gnutls_x509_crt_import (crt, &cert_list[0],
				     GNUTLS_X509_FMT_DER);
       if (ret < 0)
	 {
	   gnutls_x509_crt_deinit (crt);
	   return;
	 }

       tmp = _thy_tls_gnutls_dn_wrap (crt, gnutls_x509_crt_get_dn);
       if (!tmp)
	 tmp = strdup ("Unknown");
       cgi_addenv (session->request, "SSL_SERVER_S_DN", tmp);
       free (tmp);

       tmp = _thy_tls_gnutls_dn_wrap (crt, gnutls_x509_crt_get_issuer_dn);
       if (!tmp)
	 tmp = strdup ("Unknown");
       cgi_addenv (session->request, "SSL_SERVER_I_DN", tmp);
       free (tmp);

       cgi_addenv (session->request, "SSL_SERVER_M_SERIAL",
		   _thy_tls_gnutls_crt_get_serial (crt));

       vtime = gnutls_x509_crt_get_expiration_time (crt);
       cgi_addenv (session->request, "SSL_SERVER_V_END",
		   rfc822_date (vtime));

       vtime = gnutls_x509_crt_get_activation_time (crt);
       cgi_addenv (session->request, "SSL_SERVER_V_START",
		   rfc822_date (vtime));

       gnutls_x509_crt_deinit (crt);
     }

   if (verify_level >= 1)
     {
       /* Read peer's certificate - if any */
       cert_list = gnutls_certificate_get_peers (session->tls_session,
						 &cert_list_size);

       if (cert_list != NULL)
	 {
	   time_t vtime;

	   ret = gnutls_x509_crt_init (&crt);
	   if (ret < 0)
	     return;

	   ret = gnutls_x509_crt_import (crt, &cert_list[0],
					 GNUTLS_X509_FMT_DER);
	   if (ret < 0)
	     {
	       gnutls_x509_crt_deinit (crt);
	       return;
	     }

	   tmp = _thy_tls_gnutls_dn_wrap (crt, gnutls_x509_crt_get_dn);
	   if (!tmp)
	     tmp = strdup ("Unknown");
	   cgi_addenv (session->request, "SSL_CLIENT_S_DN", tmp);
	   free (tmp);

	   tmp = _thy_tls_gnutls_dn_wrap
	     (crt, gnutls_x509_crt_get_issuer_dn);
	   if (!tmp)
	     tmp = strdup ("Unknown");
	   cgi_addenv (session->request, "SSL_CLIENT_I_DN", tmp);
	   free (tmp);

	   cgi_addenv (session->request, "SSL_CLIENT_M_SERIAL",
		       _thy_tls_gnutls_crt_get_serial (crt));

	   vtime = gnutls_x509_crt_get_expiration_time (crt);
	   cgi_addenv (session->request, "SSL_CLIENT_V_END",
		       rfc822_date (vtime));

	   vtime = gnutls_x509_crt_get_activation_time (crt);
	   cgi_addenv (session->request, "SSL_CLIENT_V_START",
		       rfc822_date (vtime));

	   gnutls_x509_crt_deinit( crt);
	 }
     }
}

/** @internal Free one cache element.
 * Used by bhl_list_t.
 */
static void
_thy_tls_db_free (void *ptr)
{
  thy_tls_cache_t *data = (thy_tls_cache_t *)ptr;

  if (!data)
    return;

  free (data->session_id);
  free (data->session_data);
}

/** @internal Initialise the TLS session database.
 */
static void
thy_tls_db_init (void)
{
  thy_tls_cache = bhl_list_init (50, _thy_tls_db_free);
}

/** @internal Store a value in the TLS session db.
 * @note Used by GnuTLS.
 */
static int
thy_tls_db_store (void *dbf, gnutls_datum key, gnutls_datum data)
{
  thy_tls_cache_t *cache;

  cache = (thy_tls_cache_t *)bhc_malloc (sizeof (thy_tls_cache_t));
  cache->session_id = (char *)bhc_malloc (key.size);
  memcpy (cache->session_id, key.data, key.size);
  cache->session_id_size = key.size;

  cache->session_data = (char *)bhc_malloc (data.size);
  memcpy (cache->session_data, data.data, data.size);
  cache->session_data_size = data.size;

  bhl_list_insert (thy_tls_cache, cache, sizeof (thy_tls_cache_t));
  free (cache);

  return 0;
}

/** @internal Fetch an entry from the TLS session db.
 * @note Used by GnuTLS.
 */
static gnutls_datum
thy_tls_db_fetch (void *dbf, gnutls_datum key)
{
  gnutls_datum res = { NULL, 0 };
  size_t i;

  for (i = 0; i < bhl_list_size (thy_tls_cache); i++)
    {
      thy_tls_cache_t *cache = NULL;

      if (!bhl_list_get (thy_tls_cache, i, (void **)&cache))
	continue;
      if (key.size == cache->session_id_size &&
	  memcmp (key.data, cache->session_id, key.size) == 0)
	{
	  res.size = cache->session_data_size;

	  res.data = gnutls_malloc (res.size);
	  if (res.data == NULL)
	    {
	      free (cache);
	      return res;
	    }
	  memcpy (res.data, cache->session_data, res.size);
	  return res;
	}
      free (cache);
    }
  return res;
}

/** @internal Delete an entry from the TLS session db.
 * @note Used by GnuTLS.
 */
static int
thy_tls_db_delete (void *dbf, gnutls_datum key)
{
  size_t i;

  for (i = 0; i < bhl_list_size (thy_tls_cache); i++)
    {
      thy_tls_cache_t *cache = NULL;

      if (!bhl_list_get (thy_tls_cache, i, (void **)&cache))
	continue;
      if (key.size == cache->session_id_size &&
	  memcmp (key.data, cache->session_id, key.size) == 0)
	bhl_list_delete (thy_tls_cache, i);
      free (cache);
    }
  return 0;
}
#else
int
thy_tls_istls (const session_t *session)
{
  return 0;
}
#endif /* THY_OPTION_TLS */
