/** @file session.c
 * HTTP session handling rutines *//* -*- mode: c; c-file-style: "gnu" -*-
 * session.c -- HTTP session handling routines
 * Copyright (C) 2002, 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <unistd.h>

#include "compat/compat.h"
#include "bh-libs/list.h"

#include "auth.h"
#include "cgi.h"
#include "config.h"
#include "dirindex.h"
#include "fabs.h"
#include "gzip.h"
#include "http.h"
#include "mime.h"
#include "misc.h"
#include "network.h"
#include "nqueue.h"
#include "options.h"
#include "queue.h"
#include "session.h"
#include "stats.h"
#include "thy.h"
#include "tls.h"
#include "types.h"

#if THY_OPTION_IDENT
#include <ident.h>
#endif

#if THY_OPTION_TLS
#include <gnutls/gnutls.h>
#endif

#if THY_OPTION_ZLIB
#include <zlib.h>
#endif

static int _session_throttle_headers (void *data);
static int _session_throttle_buffer (void *data);

session_code_map_entry_t session_code_map[] = {
  C99_INIT([HTTP_STATUS_100]) { 100, "Continue" },
  C99_INIT([HTTP_STATUS_101]) { 101, "Switching Protocols" },
  C99_INIT([HTTP_STATUS_200]) { 200, "Ok" },
  C99_INIT([HTTP_STATUS_206]) { 206, "Partial Content" },
  C99_INIT([HTTP_STATUS_301]) { 301, "Moved Permanently" },
  C99_INIT([HTTP_STATUS_302]) { 302, "Found" },
  C99_INIT([HTTP_STATUS_304]) { 304, "Not Modified" },
  C99_INIT([HTTP_STATUS_400]) { 400, "Bad Request" },
  C99_INIT([HTTP_STATUS_401]) { 401, "Unauthorized" },
  C99_INIT([HTTP_STATUS_403]) { 403, "Forbidden" },
  C99_INIT([HTTP_STATUS_404]) { 404, "Not Found" },
  C99_INIT([HTTP_STATUS_405]) { 405, "Method Not Allowed" },
  C99_INIT([HTTP_STATUS_408]) { 408, "Request Timeout" },
  C99_INIT([HTTP_STATUS_411]) { 411, "Length Required" },
  C99_INIT([HTTP_STATUS_412]) { 412, "Precondition Failed" },
  C99_INIT([HTTP_STATUS_413]) { 413, "Request Entity Too Large" },
  C99_INIT([HTTP_STATUS_416]) { 416, "Requested Range Not Satisfiable" },
  C99_INIT([HTTP_STATUS_417]) { 417, "Expectation Failed" },
  C99_INIT([HTTP_STATUS_500]) { 500, "Internal Server Error" },
  C99_INIT([HTTP_STATUS_501]) { 501, "Not Implemented" },
  C99_INIT([HTTP_STATUS_503]) { 503, "Service Unavailable" },
  C99_INIT([HTTP_STATUS_505]) { 505, "HTTP Version Not Supported" }
};

/** @internal Parse an absolute URL and split it up.
 * Parse SESSION->request->url into ->host and ->url, if
 * ->request->url is of the form http://HOST/URL.
 * If it is not in that form, this function does not do anything.
 *
 * @note Modifies session in-place.
 */
static void
_session_parse_absoluteuri (session_t *session)
{
  char *tmp1, *tmp2;

  if (!session->request->url)
    return;

  if (strncmp ("http://", session->request->url, 7))
    return;

  tmp1 = strstr (session->request->url, "http://");
  tmp2 = bhc_strdup (strchr (&tmp1[7], '/'));
  if (tmp2)
    {
      size_t l = strlen (tmp1);
      tmp1[l - strlen (tmp2)] = 0;
      free (session->request->host);
      session->request->host = bhc_strdup (&tmp1[7]);
      free (session->request->url);
      session->request->url = tmp2;
    }
  else
    {
      free (session->request->host);
      session->request->host = bhc_strdup (&tmp1[7]);
      free (session->request->url);
      session->request->url = bhc_strdup ("/");
    }
}

/** Determine if a session has a directory index.
 * Checks if SESSION has a directory index.
 *
 * @returns The name of the file (allocated dynamically) if there is
 * one, NULL if there is not.
 */
char *
session_isindex (const session_t *session, const char *fn)
{
  const char *fname = (fn) ? fn : session->request->resolved;
  thy_mappable_config_t *config =
    config_get_mapped (session->absuri, fname);
  char *tmp;
  size_t i;

  for (i = 0; i < bhl_list_size (config->indexes); i++)
    {
      char *t;

      bhl_list_get (config->indexes, i, (void **)&t);
      asprintf (&tmp, "%s/%s", fname, t);
      free (t);

      if (!fabs_access (tmp, R_OK))
	{
	  free (config);
	  return tmp;
	}
      free (tmp);
    }
  free (config);
  return NULL;
}

/** @internal Wrapper function.
 * Wrapper function around session_header, session_error and
 * session_end.
 */
static int
_session_error_end (session_t *session, http_status_t code)
{
  session_header (session, code);
  session_error (session);
  return 0;
}

/** @internal Final setup for sending out SESSION from a buffer.
 * @param session is the session in question.
 * @param body is the session body.
 * @param len is its size.
 */
static void
_session_out (session_t *session, const char *body, size_t len)
{
  char *tmp;
  thy_mappable_config_t *config;

#if THY_OPTION_ZLIB
  if (session->compress.encoding != ENCODING_TYPE_DYNAMIC)
#endif
    {
      asprintf (&tmp, SIZET_FORMAT, len);
      session_header_add (session, "Content-Length", tmp);
      free (tmp);
    }
#if THY_OPTION_ZLIB
  else
    session->request->keepalive = 0;
  if (session->request->encoding == CONTENT_ENCODING_GZIP &&
      session->compress.encoding == ENCODING_TYPE_DYNAMIC)
    session_header_add (session, "Content-Encoding", "gzip");
#endif

  if (session->request->upgrade == THY_UPGRADE_NONE ||
      session->request->upgrade == THY_UPGRADE_TLS10_POST)
    session_header_add (session, "Connection",
			(session->request->keepalive &&
			 (session->request->http_minor == 1)) ?
			"Keep-Alive" : "close");
  else
    session_header_add (session, "Connection", "Upgrade");

  config = config_get_mapped (session->absuri, NULL);
  if (config->cache.no_cache == THY_BOOL_TRUE)
    session_header_add (session, "Cache-Control", "no-cache");
  else if (config->cache.no_store == THY_BOOL_TRUE)
    session_header_add (session, "Cache-Control", "no-store");
  else if (config->cache.no_transform == THY_BOOL_TRUE)
    session_header_add (session, "Cache-Control", "no-transform");
  else if (config->cache.must_revalidate == THY_BOOL_TRUE)
    session_header_add (session, "Cache-Control", "must-revalidate");
  else if (config->cache.do_max_age == THY_BOOL_TRUE)
    {
      asprintf (&tmp, "max-age=%ld", config->cache.max_age);
      session_header_add (session, "Cache-Control", tmp);
      free (tmp);

      if (config->cache.expiry_base == THY_EXPIRY_BASE_NOW)
	session_header_add (session, "Expires",
			    rfc822_date (time (NULL) + config->cache.max_age));
    }
  free (config);

  session_headers_flush (session);

  free (session->body.fn);
  session->body.fn = NULL;
  free (session->body.buffer);
  session->body.buffer = bhc_strdup (body);
}

/** @internal Create an ETag header.
 * Crates an ETag header from the supplied arguments. If the
 * modification time is within two seconds of the current time, this
 * function will return a weak ETag.
 *
 * @param inode is the inode of the file.
 * @param size is the size of the file.
 * @param mtime is the modification time of the file.
 *
 * @returns A newly allocated string containing the ETag header value.
 */
static char *
_session_etag_create (unsigned int inode, unsigned int size,
		      unsigned int mtime)
{
  char *tmp;

  asprintf (&tmp, "%s\"%x-%x-%x\"",
	    (time (NULL) - mtime > 2) ? "" : "W/", inode, size, mtime);
  return tmp;
}

/** @internal Final setup for sending out SESSION from a file.
 * @param session is the session in question.
 * @param fn is the file to send out.
 */
static void
_session_putfile (session_t *session, const char *fn)
{
  char *tmp;
  thy_mappable_config_t *config;
  struct stat st;

  session->filefd = fabs_open (fn, O_NONBLOCK);
  if (session->filefd < 0)
    {
      switch (errno)
	{
	case EACCES:
	  _session_error_end (session, HTTP_STATUS_403);
	  return;
	case ENOENT:
	  _session_error_end (session, HTTP_STATUS_404);
	  return;
	default:
	  /* FIXME: We should do something sensible here, like
	     postponing this stuff */
	  return;
	}
    }

  config = config_get_mapped (session->absuri, fn);

  fabs_stat (fn, &st);

  if (session->body.content_size == session->body.content_length &&
      session->request->range_end == 0)
    {
      session->body.content_length = st.st_size;
      session->body.content_size = session->body.content_length;
    }

#if THY_OPTION_ZLIB
  if (session->compress.encoding != ENCODING_TYPE_DYNAMIC)
#endif
    {
      asprintf (&tmp, SIZET_FORMAT, session->body.content_length);
      session_header_add (session, "Content-Length", tmp);
      free (tmp);
    }
#if THY_OPTION_ZLIB
  else
    session->request->keepalive = 0;
  if (session->request->encoding == CONTENT_ENCODING_GZIP)
    session_header_add (session, "Content-Encoding", "gzip");
#endif

  if (session->request->upgrade == THY_UPGRADE_NONE ||
      session->request->upgrade == THY_UPGRADE_TLS10_POST)
    session_header_add (session, "Connection",
			(session->request->keepalive &&
			 (session->request->http_minor == 1)) ?
			"Keep-Alive" : "close");
  else
    session_header_add (session, "Connection", "Upgrade");

  if (config->cache.no_cache == THY_BOOL_TRUE)
    session_header_add (session, "Cache-Control", "no-cache");
  else if (config->cache.no_store == THY_BOOL_TRUE)
    session_header_add (session, "Cache-Control", "no-store");
  else if (config->cache.no_transform == THY_BOOL_TRUE)
    session_header_add (session, "Cache-Control", "no-transform");
  else if (config->cache.must_revalidate == THY_BOOL_TRUE)
    session_header_add (session, "Cache-Control", "must-revalidate");
  else if (config->cache.do_max_age == THY_BOOL_TRUE)
    {
      time_t base;

      asprintf (&tmp, "max-age=%ld", config->cache.max_age);
      session_header_add (session, "Cache-Control", tmp);
      free (tmp);

      switch (config->cache.expiry_base)
	{
	case THY_EXPIRY_BASE_NOW:
	  base = time (NULL);
	  break;
	case THY_EXPIRY_BASE_MODIFICATION:
	default:
	  base = st.st_mtime;
	  break;
	}

      session_header_add (session, "Expires",
			  rfc822_date (base + config->cache.max_age));
    }

  session_headers_flush (session);

  free (session->body.fn);
  session->body.fn = NULL;

  if (fn)
    {
      if (!session->request->resolved)
	session->body.fn = bhc_strdup (fn);
      else if (strcmp (fn, session->request->resolved))
	session->body.fn = bhc_strdup (fn);
    }
  if (!session->body.fn)
    session->body.fn = bhc_strdup (session->request->resolved);

  free (config);
}

/** @internal Internal safety function.
 * If the FN is found, send that. If not, send a hard-coded HTTP 500
 * response, and log this fact.
 *
 * @note Modifies SESSION in-place.
 */
static void
_session_putfile_safe (session_t *session, const char *fn)
{
  if (fabs_access (fn, R_OK))
    {
      char *message;

      asprintf
	(&message,
	 "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 "
	 "Transitional//EN\">\n"
	 "<html>\n<head>\n <title>500 Internal Server Error</title>\n"
	 " <meta name=\"generator\" content=\"%s\">\n"
	 " <meta http-equiv=\"Content-Type\" content=\"text/html; "
	 "charset=\"iso-8859-1\">\n"
	 "</head>\n\n"
	 "<body>\n<h1>500 Internal Server Error</h1>\n\n"
	 "An internal server error occurred while serving your request. "
	 "Please contact the site administrator and try again later.\n"
	 "</body>\n</html>\n", thy_servername (NULL));

      session->body.offset = 0;
      session->body.content_size = strlen (message);
      session->body.content_length = session->body.content_size;
      session_header (session, HTTP_STATUS_500);
      session_header_add (session, "Content-Type", "text/html");
      _session_out (session, message, session->body.content_size);
      free (message);

      bhc_log ("ErrorDocument not found: %s", fn);
    }
  else
    if (session->cgi.running == 0)
      _session_putfile (session, fn);
}

/** @internal Look up a cached directory index.
 * Looks up if PATH is already indexed. If it is, and is not the same
 * as FD, and it was last checked within TIMEOUT seconds relative to
 * CONNTIME, then consider it valid.
 *
 * @returns The session with the cached index, if any. NULL otherwise.
 */
static session_t *
_session_dirindex_find (const char *path, int fd, time_t conntime,
			time_t timeout)
{
  long int idx;
  session_t *session;

  for (idx = 0; idx < _THY_MAXCONN; idx++)
    {
      session = queue_get (idx);
      if (!session || (session->state != SESSION_STATE_OUTPUT_HEAD ||
		       session->state != SESSION_STATE_OUTPUT_BODY) ||
	  !session->request->resolved)
	continue;
      if (fd != session->io.in && session->body.buffer &&
	  abs (session->conntime - conntime) < timeout &&
	  !strcmp (path, session->request->resolved))
	return session;
    }
  return NULL;
}

/** @internal Handle directory index requests.
 * If finds one cached, compares the mtime of the dir, and regenerates
 * the index if it isn't fresh enough. If nothing is found in the
 * cache, then generates the index for the first time.
 *
 * @returns Zero.
 */
static int
_session_handle_dirindex (session_t *session)
{
  thy_mappable_config_t *map_config =
    config_get_mapped (session->absuri, session->request->resolved);
  const config_t *config = config_get ();
  char *di;
  session_t *csession;
  struct stat buf;

  if (map_config->options.dirindex != THY_BOOL_TRUE)
    {
      free (map_config);

      session_header (session, HTTP_STATUS_403);
      session_error (session);
      return 0;
    }

  if (session->request->method != HTTP_METHOD_GET &&
      session->request->method != HTTP_METHOD_HEAD)
    {
      session_header (session, HTTP_STATUS_405);
#if THY_OPTION_TRACE
      session_header_add (session, "Allow", "GET, HEAD, OPTIONS, TRACE");
#else
      session_header_add (session, "Allow", "GET, HEAD, OPTIONS");
#endif
      session_error (session);
      free (map_config);
      return 0;
    }

  fabs_stat (session->request->resolved, &buf);
  if ((csession = _session_dirindex_find (session->request->resolved,
					  session->io.in,
					  session->conntime,
					  config->timeout)) == NULL)
    di = directory_index (session->request->resolved,
			  session->request->url);
  else
    {
      if (buf.st_mtime > csession->conntime)
	di = directory_index (session->request->resolved,
			      session->request->url);
      else
	di = bhc_strdup (csession->body.buffer);
    }

  session_header (session, HTTP_STATUS_200);
  session_header_add (session, "Content-Type", "text/html");

  if (map_config->etag.etag == THY_BOOL_TRUE &&
      map_config->etag.dirtag == THY_BOOL_TRUE)
    session->etag = _session_etag_create ((unsigned int)buf.st_ino,
					  (unsigned int)buf.st_size,
					  (unsigned int)buf.st_mtime);
  free (map_config);

  if (!di)
    {
      session_header (session, HTTP_STATUS_403);
      session_error (session);
      return 0;
    }

  session->body.offset = 0;
  session->body.content_size = strlen (di);
  session->body.content_length = session->body.content_size;
  _session_out (session, di, session->body.content_size);
  free (di);

  return 0;
}

/** Handle keep-alive.
 * @returns Zero if the session should be kept alive, -1 otherwise.
 */
int
session_handle_keepalive (session_t *session)
{
  if (session->request->keepalive &&
      session->request->http_minor == 1)
    return queue_keep (session->io.in);
  return -1;
}

/** Create a new session object.
 * @param infd is the FD we accept input on.
 * @param outfd is the FD we send output to.
 * @param sport is the servers listening port.
 * @param ssl signals TLS mode.
 * @param origin is the originating IP of the client.
 * @param tls_data is the (optional) TLS data associated with the
 * client.
 *
 * @returns A newly allocated session object, which must be freed
 * later.
 */
session_t *
session_init (int infd, int outfd, int sport, int ssl, const char *origin,
	      void *tls_data)
{
  session_t *session;

  bhc_debug ("Session %d initialising for INPUT_REQUEST state.", infd);

  session = (session_t *)bhc_malloc (sizeof (session_t));
  session->io.in = infd;
  session->io.out = outfd;
  session->port = sport;
  session->filefd = -1;
  session->ssl = ssl;
  session->conntime = time (NULL);
  session->acttime = time (NULL);
  session->state = SESSION_STATE_INPUT_REQUEST;
  session->origin = bhc_strdup (origin);
  session->request = NULL;
  session->root = NULL;
  session->absuri = NULL;
  session->body.buffer = NULL;
  session->body.size = 0;
  session->body.len = 0;
  session->body.content_length = 0;
  session->body.content_size = 0;
  session->body.offset = 0;
  session->body.fn = NULL;
  session->body.method = SESSION_OUTPUT_METHOD_NONE;
  session->body.header.buff = NULL;
  session->body.header.size = 0;
  session->body.header.offset = 0;
  session->body.footer.buff = NULL;
  session->body.footer.size = 0;
  session->body.footer.offset = 0;
  session->header.fields = NULL;
  session->header.code = HTTP_STATUS_UNKNOWN;
  session->cgi.child = -1;
  session->cgi.argv = NULL;
  session->cgi.handler = NULL;
  session->cgi.status = -1;
  session->cgi.pipes.in[0] = -1;
  session->cgi.pipes.in[1] = -1;
  session->cgi.pipes.out[0] = -1;
  session->cgi.pipes.out[1] = -1;
  session->cgi.running = 0;
  session->cgi.eof = 0;
  session->cgi.content_length = 0;
  session->cgi.post.size = 0;
  session->cgi.post.offset = 0;
  session->cgi.post.length = 0;
  session->cgi.post.len = 0;
  session->cgi.post.buffer = NULL;
  session->chunked.enabled = 0;
  session->chunked.prechunk.size = 0;
  session->chunked.prechunk.offset = 0;
  session->chunked.prechunk.buff = NULL;
  session->chunked.postchunk.size = 0;
  session->chunked.postchunk.offset = 0;
  session->chunked.postchunk.buff = NULL;
  session->vary.accept_encoding = 0;
  session->mime = NULL;
  session->request_count = 0;
  session->etag = NULL;

#if THY_OPTION_ZLIB
  session->compress.gz_session = NULL;
#endif

#if THY_OPTION_TLS
  session->cert_verified = "NONE";

  if (!tls_data)
    {
      session_state_change (session, SESSION_STATE_HANDSHAKE);
      session->tls_session = NULL;
      if (!thy_tls_session_init (session))
	{
	  session_kill (session, 1);
	  return NULL;
	}
    }
  else
    session->tls_session = (gnutls_session)tls_data;
#endif

  return session;
}

/** Finish up the response headers.
 * Put the headers of SESSION in a ready-to-go state, and put a final
 * CRLF after them.
 */
void
session_headers_flush (session_t *session)
{
  size_t i;
  pair_t *pair;
  thy_mappable_config_t *config =
    config_get_mapped (session->absuri, session->request->resolved);

  for (i = 0; i < bhl_list_size (config->headers); i++)
    {
      bhl_list_get (config->headers, i, (void **)&pair);
      session_header_add (session, pair->field, pair->value);
      free (pair);
    }

  if (config->options.vary == THY_BOOL_TRUE)
    {
      int v_enc = session->vary.accept_encoding;

      if (v_enc)
	{
	  if (session->request->http_minor == 0)
	    {
	      session_header_add (session, "Expires",
				  "Thu, 01 Jan 1970 00:00:00 GMT");
	      session_header_add (session, "Cache-Control",
				  "max-age=604800");
	    }
	  session_header_add (session, "Vary", "accept-encoding");
	}
    }

  if (session->etag && config->etag.etag == THY_BOOL_TRUE)
    session_header_add (session, "ETag", session->etag);

  session->header.len = strlen (session->header.fields) + 2;
  session->header.fields = (char *)bhc_realloc (session->header.fields,
						session->header.len + 1);
  memcpy (&session->header.fields[session->header.len] - 2,
	  "\r\n", 2);
  session->header.offset = 0;

  free (config);
}

/** @internal Determine if we can serve a gzip precompressed file.
 * If there is a compressed file of the same name, but with a .gz
 * suffix, alter SESSION so we'll use that.
 *
 * @note Alters session in-place.
 */
static void
_session_canonicalize_gzip (session_t *session)
{
  char *tmp, *fn;
  size_t l;
  thy_mappable_config_t *config;

  fn = (session->request->resolved) ?
    session->request->resolved : session->request->file;
  if (!fn || session->request->encoding == CONTENT_ENCODING_NONE)
    {
#if THY_OPTION_ZLIB
      session->compress.encoding = ENCODING_TYPE_NONE;
#endif
      return;
    }

  config = config_get_mapped (session->absuri, fn);

  if (config->encoding.type == ENCODING_TYPE_NONE)
    {
      session->request->encoding = CONTENT_ENCODING_NONE;
      free (config);
      return;
    }
#if !THY_OPTION_ZLIB
  if (config->encoding.type != ENCODING_TYPE_STATIC)
    {
      session->request->encoding = CONTENT_ENCODING_NONE;
      free (config);
      return;
    }
#endif

  session->request->encoding = CONTENT_ENCODING_GZIP;
  l = strlen (fn);
  tmp = (char *)bhc_malloc (l + 4);
  mempcpy (mempcpy (tmp, fn, l), ".gz\0", 4);
  if (fabs_access (tmp, R_OK))
    {
      free (tmp);
#if THY_OPTION_ZLIB
      if (config->encoding.type == ENCODING_TYPE_DYNAMIC)
	{
	  session->vary.accept_encoding = 1;
	  session->compress.encoding = ENCODING_TYPE_DYNAMIC;
	}
      else
	{
	  struct stat st;

	  fabs_stat (fn, &st);
	  if (!S_ISDIR (st.st_mode))
#endif
	    session->request->encoding = CONTENT_ENCODING_NONE;
#if THY_OPTION_ZLIB
	}
#endif
    }
  else
    {
      struct stat mf, of;

      fabs_stat (tmp, &mf);
      fabs_stat (fn, &of);

      if (mf.st_mtime >= of.st_mtime)
	{
	  session->vary.accept_encoding = 1;
#if THY_OPTION_ZLIB
	  session->compress.encoding = ENCODING_TYPE_STATIC;
#endif
	  free (session->request->resolved);
	  session->request->resolved = fabs_realpath (tmp);
	}
      else
	session->request->encoding = CONTENT_ENCODING_NONE;
      free (tmp);
    }
  free (config);
}

/** @internal Canonicalizes the file name.
 * This handles the static encoding types (ie, returns a .gz
 * postfixed file if one exists) and translated documents too.
 */
static void
_session_canonicalize_file_name (session_t *session)
{
  _session_canonicalize_gzip (session);

  if (!session->request->resolved)
    session->request->resolved = fabs_realpath (session->request->file);
}

/** Update an initialised, empty session.
 * Update SESSION, by parsing the request, resolving the URL, and so
 * on.
 *
 * @note Modifies SESSION in-place.
 */
void
session_update (session_t *session)
{
  session->request = (request_t *)bhc_malloc (sizeof (request_t));

  session->result = http_request_parse (session->request,
					session->body.buffer);
  if (session->request->url)
    {
      asprintf (&session->absuri, "http%s://%s%s",
		(session->ssl) ? "s" : "",
		session->request->host, session->request->url);
      _session_parse_absoluteuri (session);
      urldecode (session->request->url);
      cgi_setup (session);
      session->request->file = fabs_urlmap (session->request->url,
					    session->request->host,
					    session->absuri);
      _session_canonicalize_file_name (session);
    }
  else
    {
      session->request->file = NULL;
      session->request->resolved = NULL;
    }

  session->body.offset = session->request->range_start;
}

/** Set up the basic HTTP response header.
 * Set the HTTP response header of SESSION to RVAL.
 *
 * @note Modifies SESSION in-place.
 * @note All the former headers will be forgotten.
 */
void
session_header (session_t *session, http_status_t code)
{
  char *tmp;
  const char *servername = thy_servername (NULL);

  session->header.code = code;
  free (session->header.fields);

  if (code < HTTP_STATUS_UNKNOWN)
    tmp = session_code_map[code].message;
  else
    {
      tmp = session_code_map[HTTP_STATUS_500].message;
      session->header.code = HTTP_STATUS_500;
      bhc_error ("Unknown code in %s(%d): %d", __FILE__, __LINE__,
		 (int)code);
    }

  asprintf (&session->header.fields,
	    "HTTP/1.%d %d %s\r\n"
	    "Server: %s\r\n"
	    "Accept-Ranges: bytes\r\n"
	    "Date: %s\r\n", session->request->http_minor,
	    session_code_map[session->header.code].code, tmp,
	    servername, rfc822_date (time (NULL)));
}

/** @internal Prepare and do the fork()ing for a CGI.
 * Check if FN is a CGI in SESSION.
 *
 * @returns One, if it is, two, if there are too many children, zero
 * otherwise.
 *
 * @note As a side-effect, this spawns off a child process to handle
 * the CGI.
 */
static int
_session_docgi (session_t *session, const char *fn)
{
  const config_t *config = config_get ();
  int rval = 0;

  if (cgi_iscgi (session, fn) && session->cgi.child == -1)
    {
      if (session->cgi.handler)
	session->cgi.argv[1] = bhc_strdup (fn);
      if (thy_active_cgis >= config->limits.cgis &&
	  config->limits.cgis != 0)
	return 2;

      rval = !cgi_launch ((session->cgi.handler) ?
			 session->cgi.handler : fn, session);
    }
  return rval;
}

/** Indicate than an error occurred.
 * @note Before calling this function, one must set
 * SESSION->response_code with session_header.
 * @note Modifies session in-place.
 */
void
session_error (session_t *session)
{
  thy_mappable_config_t *config =
    config_get_mapped (session->absuri, NULL);
  char *fn = NULL;
  char *handler;

  session_header_add (session, "Content-Type", "text/html");
  session->request->range_start = 0;
  session->request->range_end = 0;
  session->body.offset = 0;
  session->request->encoding = CONTENT_ENCODING_NONE;
#if THY_OPTION_ZLIB
  session->compress.encoding = ENCODING_TYPE_NONE;
#endif
  session->vary.accept_encoding = 0;
  session->body.content_length = session->body.content_size = 0;

  if (session->header.code < HTTP_STATUS_UNKNOWN)
    fn = config->http_status[session->header.code];
  else
    {
      fn = config->http_status[HTTP_STATUS_500];
      bhc_error ("Unknown code in %s(%d): %d", __FILE__, __LINE__,
		 (int)session->header.code);
    }

  if ((handler = session_handler_check (fn, session->absuri)) != NULL)
    {
      session->cgi.argv = (char **)bhc_calloc (3, sizeof (char *));
      session->cgi.argv[0] = bhc_strdup (handler);
      session->cgi.argv[1] = NULL;
      session->cgi.argv[2] = NULL;
      session->cgi.handler = bhc_strdup (handler);
      free (handler);
    }

  free (config);

  switch (_session_docgi (session, fn))
    {
    case 1:
      _session_putfile_safe (session, fn);
      break;
    case 2:
      if (session->header.code != HTTP_STATUS_503)
	{
	  session_header (session, HTTP_STATUS_503);
	  session_error (session);
	}
      else
	_session_putfile_safe (session, NULL);
      break;
    default:
      _session_putfile_safe (session, fn);
      break;
    }
}

/** Add a new HTTP response header.
 * Add a new header (HDR), which has the value VALUE to the set of
 * headers of SESSION.
 *
 * @note Modifies session in-place.
 */
void
session_header_add (session_t *session, const char *hdr,
		    const char *value)
{
  size_t len_rh, len_hdr, len_value;

  len_rh = strlen (session->header.fields);
  len_hdr = strlen (hdr);
  len_value = strlen (value);

  session->header.fields = (char *)bhc_realloc (session->header.fields,
						len_rh + len_hdr +
						len_value + 5);

  memcpy (mempcpy (mempcpy (mempcpy (session->header.fields + len_rh, hdr,
				     len_hdr), ": ", 2), value, len_value),
	  "\r\n", 2);
  session->header.fields [len_rh + len_hdr + len_value + 4] = 0;
}

/** Log selected properties of the session to syslog.
 */
void
session_log (session_t *session)
{
#if (THY_OPTION_LOGGING) && (BHC_OPTION_ALL_LOGGING)
  char *id, *qs, *pi;
  size_t range = 0;

  if (!session)
    return;

  if (session->header.code == HTTP_STATUS_UNKNOWN || !session->request ||
      !session->request->method_str)
    return;

  id = bhc_strdup ("-");
  if (session->body.offset >= session->request->range_start)
    {
      if (session->request->range_start < 0)
	range = -session->request->range_start;
      else
	range = (size_t) (session->body.offset -
			  session->request->range_start);
    }
  else
    range = session->body.content_length;
  if (session->cgi.content_length > 0)
    range = session->cgi.content_length;
  if (session->request->method == HTTP_METHOD_HEAD ||
      session->request->method == HTTP_METHOD_OPTIONS)
    range = 0;

  thy_stats_transfer_inc (range);

#if THY_OPTION_IDENT
  free (id);
  id = ident_id (session->fd, 1);
  if (!id)
    id = bhc_strdup ("-");
#endif /* THY_OPTION_IDENT */

  qs = session->request->query_string;
  pi = session->request->path_info;

  syslog (LOG_INFO, "%s %s - %s [%s] \"%s %s%s%s%s HTTP/%d.%d\" %d "
	  SIZET_FORMAT " \"%s\" \"%s\"",
	  (session->request->host) ? session->request->host : "-",
	  session->origin, id, log_date (time (NULL)),
	  (session->request->method_str[0]) ?
	  session->request->method_str : "UNKNOWN",
	  (session->request->url) ? session->request->url : "UNKNOWN",
	  (pi) ? pi : "", (qs) ? "?" : "", (qs) ? qs : "",
	  session->request->http_major, session->request->http_minor,
	  (session->cgi.status != -1) ? session->cgi.status :
	  session_code_map[session->header.code].code, range,
	  (session->request->referer) ? session->request->referer : "-",
	  (session->request->ua) ? session->request->ua : "-");
  free (id);
#endif /* THY_OPTION_LOGGING && BHC_OPTION_ALL_LOGGING */
}

/** Free up SESSION, and all its components.
 * Go and free up everything associated with the session. If CLSE is
 * non-zero, close the connection too.
 */
void
session_kill (session_t *session, int clse)
{
  unsigned int i = 0;

#if (THY_OPTION_LOGGING) && (BHC_OPTION_ALL_LOGGING)
  session_log (session);
#endif

  if (clse)
    {
#if THY_OPTION_TLS
      if (session->tls_session)
	gnutls_bye (session->tls_session, GNUTLS_SHUT_RDWR);
#endif
      shutdown (session->io.in, SHUT_RDWR);
      close (session->io.in);
      shutdown (session->io.out, SHUT_RDWR);
      close (session->io.out);
    }
  shutdown (session->cgi.pipes.in[0], SHUT_RDWR);
  close (session->cgi.pipes.in[0]);
  shutdown (session->cgi.pipes.out[0], SHUT_RDWR);
  close (session->cgi.pipes.out[0]);
  shutdown (session->cgi.pipes.in[1], SHUT_RDWR);
  close (session->cgi.pipes.in[1]);
  shutdown (session->cgi.pipes.out[1], SHUT_RDWR);
  close (session->cgi.pipes.out[1]);
  free (session->cgi.post.buffer);
  if (session->request)
    {
      free (session->request->method_str);
      free (session->request->host);
      free (session->request->ua);
      free (session->request->url);
      free (session->request->referer);
      free (session->request->resolved);
      free (session->request->file);
      free (session->request->content_type);
      free (session->request->query_string);
      free (session->request->path_info);
      free (session->request->raw);
      free (session->request->auth_token);
      free (session->request->auth_realm);
      free (session->request->auth_file);
      while (session->request->cgienvlen > i)
	free (session->request->cgienv[i++]);
      free (session->request->cgienv);
      bhl_list_free (session->request->if_match);
      bhl_list_free (session->request->if_none_match);
      free (session->request->if_range);
      free (session->request->expect);
      free (session->request);
    }
  i = 0;
  if (session->cgi.argv)
    while (session->cgi.argv[i])
      {
	free (session->cgi.argv[i]);
	i++;
      }
  free (session->cgi.argv);
  free (session->cgi.handler);
  free (session->header.fields);
  free (session->body.buffer);
  free (session->body.fn);
  free (session->body.header.buff);
  free (session->body.footer.buff);
  free (session->chunked.prechunk.buff);
  free (session->chunked.postchunk.buff);
  free (session->origin);
  free (session->root);
  free (session->mime);
  free (session->absuri);
  free (session->etag);
  if (session->filefd > 0)
    fabs_close (session->filefd);

#if THY_OPTION_TLS
  if (clse && session->tls_session)
    gnutls_deinit (session->tls_session);
#endif
#if THY_OPTION_ZLIB
  thy_zlib_free (session->compress.gz_session);
#endif
  if (session->cgi.child > 0 && session->cgi.running > 0)
    kill (session->cgi.child, SIGKILL);
  free (session);
}

/** Determine if there is a handler for a file.
 * @returns The handler if there is one, NULL otherwise.
 */
char *
session_handler_check (const char *fn, const char *absuri)
{
  thy_mappable_config_t *config;
  size_t i, fnlen;
  char *tmp;

  tmp = fabs_realpath (fn);
  if (!tmp)
    return NULL;

  fnlen = strlen (tmp);
  config = config_get_mapped (absuri, fn);

  for (i = 0; i < bhl_list_size (config->handlers); i++)
    {
      pair_t *t;
      bhl_list_get (config->handlers, i, (void **)&t);

      if (!strcmp (&tmp[fnlen - strlen (t->field)], t->field))
	{
	  char *handler = bhc_strdup (t->value);
	  free (t);
	  free (tmp);
	  free (config);
	  return handler;
	}
      free (t);
    }

  free (config);
  free (tmp);
  return NULL;
}

/** Determine if there is a handler for a method.
 * @returns The method if there is one, NULL otherwise.
 */
static char *
_session_method_check (const char *fn, const char *absuri)
{
  thy_mappable_config_t *config;
  size_t i;

  if (!fn)
    return NULL;

  config = config_get_mapped (absuri, fn);

  for (i = 0; i < bhl_list_size (config->methods); i++)
    {
      pair_t *t;

      bhl_list_get (config->methods, i, (void **)&t);

      if (!strcmp (fn, t->field))
	{
	  char *handler = bhc_strdup (t->value);
	  free (t);
	  free (config);
	  return handler;
	}
      free (t);
    }

  free (config);
  return NULL;
}

/** @internal Compare two dates.
 * @returns "0" if the dates are equal, "1" if date is newer, "2" if
 * date2 is.
 */
static int
_session_date_compare (const struct tm *date1,
		       const struct tm *date2)
{
  if (date1->tm_year != date2->tm_year)
    return 1 + (date1->tm_year < date2->tm_year);

  if (date1->tm_mon != date2->tm_mon)
    return 1 + (date1->tm_mon < date2->tm_mon);

  if (date1->tm_mday != date2->tm_mday)
    return 1 + (date1->tm_mday < date2->tm_mday);

  if (date1->tm_hour != date2->tm_hour)
    return 1 + (date1->tm_hour < date2->tm_hour);

  if (date1->tm_min != date2->tm_min)
    return 1 + (date1->tm_min < date2->tm_min);

  if (date1->tm_sec != date2->tm_sec)
    return 1 + (date1->tm_sec < date2->tm_sec);

  return 0;
}

/** @internal Find an ETag in a list of ETags.
 * @param list is the list to search in.
 * @param tag is the ETag to search for.
 *
 * @returns Zero if the ETag was not found, 1 otherwise.
 */
static int
_session_etag_find (bhl_list_t *list, const char *tag)
{
  size_t i;
  char *tmp;

  for (i = 0; i < bhl_list_size (list); i++)
    {
      bhl_list_get (list, i, (void **)&tmp);
      if (tmp[0] == '*')
	{
	  free (tmp);
	  return 1;
	}
      if (!strcmp (tmp, tag))
	{
	  free (tmp);
	  return 1;
	}
      free (tmp);
    }
  return 0;
}

/** Handle a session.
 * Given a SESSION, determine how to handle it. This is pretty long
 * and complex, but well commented.
 *
 * @returns What session_end() returns, which is zero.
 */
int
session_handle (session_t *session)
{
  const config_t *config = config_get ();
  thy_mappable_config_t *map_config;
  struct stat st;
  char *tmp, *default_type;
  int handle_range = 1;

  /* First, handle the keep-alive stuff. */
  if (session->request_count >= config->keep_max &&
      config->keep_max != 0)
    session->request->keepalive = 0;

  /* Do we have a supported method? */
  if (session->request->method == HTTP_METHOD_UNKNOWN)
    {
      char *mhandler =
	_session_method_check (session->request->method_str,
			       session->absuri);

      if (!mhandler)
	return _session_error_end (session, HTTP_STATUS_501);

      free (session->request->resolved);
      free (session->request->file);
      session->request->resolved = NULL;
      session->request->file = bhc_strdup (mhandler);
      _session_canonicalize_file_name (session);

      session->cgi.argv = (char **)bhc_calloc (3, sizeof (char *));
      session->cgi.argv[0] = bhc_strdup (mhandler);
      session->cgi.argv[1] = NULL;
      session->cgi.argv[2] = NULL;
      session->cgi.handler = bhc_strdup (mhandler);

      free (mhandler);
    }

  /* Do we have an url? */
  if ((!session->request->url ||
       session->result == HTTP_PARSE_PREMATURE ||
       session->request->url[0] != '/') &&
      session->request->method != HTTP_METHOD_OPTIONS)
    return _session_error_end (session, HTTP_STATUS_400);

  /* Do we have /../ in the URL? */
  if (strstr (session->request->url, "/../"))
    return _session_error_end (session, HTTP_STATUS_400);

  /* Could we parse the request? */
  if (session->result != HTTP_PARSE_OK)
    return _session_error_end (session, HTTP_STATUS_500);

  /* Can we handle this HTTP version? */
  if (session->request->http_major != 1 ||
      session->request->http_minor > 1)
    return _session_error_end (session, HTTP_STATUS_505);

  /* Do we have a Content-Length for POST? */
  if (session->request->method == HTTP_METHOD_POST)
    {
      if (session->request->content_length == 0)
	return _session_error_end (session, HTTP_STATUS_411);
      if (session->request->content_length > config->limits.post_buffer &&
	  config->limits.post_buffer > 0)
	{
	  /* According to the RFC, we MAY close the connection. We do
	     not do that (except when configured so), but discard all
	     of the POST data. */
	  if (session->header.code != HTTP_STATUS_413 &&
	      config->options.hardlimit == THY_BOOL_FALSE)
	    {
	      session_state_change (session, SESSION_STATE_INPUT_DISCARD);
	      session_header (session, HTTP_STATUS_413);
	      return 0;
	    }
	  else
	    {
	      if (config->options.hardlimit == THY_BOOL_TRUE)
		session->request->keepalive = 0;
	      return _session_error_end (session, HTTP_STATUS_413);
	    }
	}
    }

#if THY_OPTION_TRACE
  /* Is this a TRACE request? */
  if (session->request->method == HTTP_METHOD_TRACE)
    {
      session_header (session, HTTP_STATUS_200);
      session_header_add (session, "Content-Type", "message/http");
      session->body.content_length = strlen (session->request->raw);
      session->body.size = session->body.content_length;
      session->body.content_size = session->body.size;
      _session_out (session, session->request->raw, session->body.size);
      return 0;
    }
#endif

  /* Is this Expect: 100-continue? */
  if (session->request->expect &&
      config->options.expect == THY_BOOL_TRUE &&
      session->request->http_minor >= 1 &&
      session->request->method != HTTP_METHOD_GET &&
      session->request->method != HTTP_METHOD_HEAD &&
      session->request->method != HTTP_METHOD_OPTIONS)
    {
      if (!strcasecmp (session->request->expect, "100-continue"))
	{
	  session_header (session, HTTP_STATUS_100);
	  session->body.content_length = 0;
	  session->body.size = session->body.content_length;
	  session_headers_flush (session);
	  session_state_change (session, SESSION_STATE_OUTPUT_100);
	  return 0;
	}
      else
	return _session_error_end (session, HTTP_STATUS_417);
    }

  /* Is this a candidate for Upgrade? */
  if (session->request->upgrade != THY_UPGRADE_NONE &&
      session->request->upgrade != THY_UPGRADE_TLS10_POST &&
      thy_tls_enabled != THY_BOOL_FALSE)
    {
      session->request->http_minor = 1;
      session_header (session, HTTP_STATUS_101);
      switch (session->request->upgrade)
	{
	case THY_UPGRADE_HTTP11:
	  session_header_add (session, "Upgrade", "HTTP/1.1");
	  break;
	case THY_UPGRADE_TLS10:
	  session_header_add (session, "Upgrade", "TLS/1.0, HTTP/1.1");
	  break;
	default:
	  break;
	}
      session_header_add (session, "Connection", "Upgrade");
      session_headers_flush (session);
      session_state_change (session, SESSION_STATE_OUTPUT_101);
      return 0;
    }

  map_config = config_get_mapped (session->absuri, NULL);
  default_type = map_config->default_type;

  /* Could we canonicialize the filename? */
  if (map_config->options.vhosting == THY_BOOL_TRUE)
    {
      if (!session->request->host)
	{
	  free (session->request->resolved);
	  tmp = fabs_urlmap (session->request->url, "default",
			     session->absuri);
	  if (!(session->request->resolved = fabs_realpath (tmp)))
	    {
	      free (tmp);
	      tmp = fabs_urlmap (session->request->url, NULL,
				 session->absuri);
	      session->request->resolved = fabs_realpath (tmp);
	      session->root = bhc_strdup (map_config->webroot);
	    }
	  else
	    asprintf (&session->root, "%s/%s", map_config->webroot,
		      "default");
	  free (tmp);
	}
      else
	{
	  asprintf (&tmp, "%s/%s", map_config->webroot,
		    session->request->host);
	  if (!(session->root = fabs_realpath (tmp)))
	    {
	      free (tmp);
	      asprintf (&tmp, "%s/%s", map_config->webroot, "default");
	      session->root = tmp;
	    }
	  else
	    free (tmp);

	  if (!session->request->resolved)
	    {
	      tmp = http_url_resolve (session->request->url, "default",
				      session->absuri);
	      session->request->resolved = fabs_realpath (tmp);
	      free (tmp);
	    }
	}
    }
  else
    session->root = bhc_strdup (map_config->webroot);

  tmp = fabs_realpath (session->root);
  free (session->root);
  session->root = tmp;

  free (map_config);
  map_config = config_get_mapped (session->absuri,
				  session->request->resolved);

  /** Handle OPTIONS requests. */
  if (session->request->method == HTTP_METHOD_OPTIONS)
    {
      int post = cgi_iscgi (session, NULL);
      char *methods;
      pair_t *method;
      size_t i;

      session_header (session, HTTP_STATUS_200);

      asprintf (&methods, "GET, HEAD, OPTIONS%s",
		(post) ? ", POST" : "");
      for (i = 0; i < bhl_list_size (map_config->methods); i++)
	{
	  bhl_list_get (map_config->methods, i, (void **)&method);
	  asprintf (&tmp, "%s, %s", methods, method->field);
	  free (method);
	  free (methods);
	  methods = tmp;
	}
      session_header_add (session, "Allow", methods);
      free (methods);

      session_header_add (session, "Content-Length", "0");
      if (strcmp ("*", session->request->url))
	session_header_add (session, "Content-Type", "text/plain");
      if (session->request->upgrade == THY_UPGRADE_NONE ||
	  session->request->upgrade == THY_UPGRADE_TLS10_POST)
	session_header_add (session, "Connection",
			    (session->request->keepalive &&
			     (session->request->http_minor == 1)) ?
			    "Keep-Alive" : "close");
      else
	session_header_add (session, "Connection", "Upgrade");

      session_headers_flush (session);
      free (map_config);
      return 0;
    }

  /* Is session->root inside the webroot? */
  if (map_config->options.vhosting == THY_BOOL_TRUE)
    {
      if ((session->request->host &&
	   (session->request->host[0] == '.' ||
	    session->request->host[0] == '/')))
	{
	  free (map_config);
	  return _session_error_end (session, HTTP_STATUS_400);
	}
    }

  /* Do we have a webroot? */
  if (fabs_access (session->root, F_OK))
    {
      bhc_error ("%s", "I have no root and I want to scream!");
      free (map_config);
      return _session_error_end (session, HTTP_STATUS_500);
    }

  if (!session->request->resolved)
    {
      free (map_config);
      return _session_error_end (session, HTTP_STATUS_404);
    }

  fabs_stat (session->request->resolved, &st);

  /* Is it inside our webroot? */
  if (map_config->options.userdir == THY_BOOL_TRUE &&
      session->request->url[1] == '~')
    {
      char *ud = userdir (session->request->url, map_config->userdir);
      if (!strstr (session->request->resolved, ud))
	{
	  if (!follow_ifowner (session))
	    {
	      free (ud);
	      free (map_config);
	      return _session_error_end (session, HTTP_STATUS_404);
	    }
	}
      free (ud);
    }
  else
    {
      if (!strstr (session->request->resolved, session->root))
	{
	  if (!follow_ifowner (session))
	    {
	      free (map_config);
	      return _session_error_end (session, HTTP_STATUS_404);
	    }
	}
    }
  free (map_config);

  /* Do we need authentication? */
  if (auth_need (session))
    {
      if (!session->request->auth_token && session->request->auth_realm)
	{
	  asprintf (&tmp, "Basic realm=\"%s\"",
		    session->request->auth_realm);
	  session_header (session, HTTP_STATUS_401);
	  session_header_add (session, "WWW-Authenticate", tmp);
	  free (tmp);
	  session_error (session);
	  return 0;
	}
      else
	if (auth_authenticate (session))
	  return _session_error_end (session, HTTP_STATUS_403);
    }

  /* Is the file reachable? */
  if (fabs_access (session->request->resolved, F_OK))
    return _session_error_end (session, HTTP_STATUS_404);

  /* Is it a directory? */
  if (S_ISDIR (st.st_mode))
    {
      /* Does the URL end with a /? */
      if (session->request->url[strlen (session->request->url) - 1] == '/')
	{
	  tmp = session_isindex (session, NULL);
	  /* Do we have an index.html ? */
	  if (!tmp)
	    return _session_handle_dirindex (session);
	  else
	    {
	      char *handler = session_handler_check (tmp, session->absuri);

	      fabs_stat (tmp, &st);
	      free (session->request->resolved);
	      free (session->request->file);
	      session->request->resolved = NULL;
	      session->request->file = bhc_strdup (tmp);
	      _session_canonicalize_file_name (session);
	      if (handler)
		{
		  session->cgi.argv =
		    (char **)bhc_calloc (3, sizeof (char *));
		  session->cgi.argv[0] = bhc_strdup (handler);
		  session->cgi.argv[1] = NULL;
		  session->cgi.argv[2] = NULL;
		  session->cgi.handler = bhc_strdup (handler);
		}
	      free (handler);
	    }
	  free (tmp);
	}
      else
	{
	  session_header (session, HTTP_STATUS_302);
	  asprintf (&tmp, "%s/", session->request->url);
	  session_header_add (session, "Location", tmp);
	  free (tmp);
	  session_error (session);
	  return 0;
	}
    }

  /* Is it readable? */
  if (fabs_access (session->request->resolved, R_OK))
    return _session_error_end (session, HTTP_STATUS_403);

  /* So it is not a directory. Is it a file? */
  if (!S_ISREG (st.st_mode))
    return _session_error_end (session, HTTP_STATUS_403);

  /* Do we have a handler ? */
  if (!session->cgi.handler)
    {
      char *handler = session_handler_check (session->request->resolved,
					     session->absuri);

      if (handler)
	{
	  session->cgi.argv = (char **)bhc_calloc (3, sizeof (char *));
	  session->cgi.argv[0] = bhc_strdup (handler);
	  session->cgi.argv[1] = NULL;
	  session->cgi.argv[2] = NULL;
	  session->cgi.handler = bhc_strdup (handler);
	}
      free (handler);
    }

  /* Is it a CGI ? */
  switch (_session_docgi (session, session->request->resolved))
    {
    case 1:
      return 0;
    case 2:
      return _session_error_end (session, HTTP_STATUS_503);
    default:
      break;
    }

  /* We are not CGI.. Is the request method OK still? */
  if (session->request->method != HTTP_METHOD_GET &&
      session->request->method != HTTP_METHOD_HEAD)
    {
      session_header (session, HTTP_STATUS_405);
#if THY_OPTION_TRACE
      session_header_add (session, "Allow", "GET, HEAD, OPTIONS, TRACE");
#else
      session_header_add (session, "Allow", "GET, HEAD, OPTIONS");
#endif
      session_error (session);
      return 0;
    }

  /* It is a file */
  session_header (session, HTTP_STATUS_200);
  session->body.content_length = st.st_size;
  session->body.content_size = st.st_size;
  session->etag = _session_etag_create ((unsigned int)st.st_ino,
					(unsigned int)st.st_size,
					(unsigned int)st.st_mtime);

  /* Handle If-(None-)Match */

  /* If-Match */
  if ((bhl_list_size (session->request->if_match) > 0) &&
      (_session_etag_find (session->request->if_match, session->etag) == 0) &&
      (session->etag[0] == '"'))
    return _session_error_end (session, HTTP_STATUS_412);

  /* If-None-Match */
  if ((bhl_list_size (session->request->if_none_match) > 0) &&
      (session->etag[0] == '"'))
    {
      if (_session_etag_find (session->request->if_none_match,
			      session->etag) == 0)
	{
	  /* No matches, disable If-Modified-Since. */
	  session->request->modified_since.tm_mday = 0;
	}
      else
	{
	  /* One of the Entities matched... */
	  if (session->request->method == HTTP_METHOD_GET ||
	      session->request->method == HTTP_METHOD_HEAD)
	    {
	      free (session->body.buffer);
	      session->body.buffer = NULL;
	      session->body.content_size = 0;

	      session_header (session, HTTP_STATUS_304);
	      session_headers_flush (session);
	      return 0;
	    }
	  else
	    return _session_error_end (session, HTTP_STATUS_412);
	}
    }

  if (session->body.offset != 0 || session->request->range_end != 0)
    {
      /* Handle If-Range */
      if (session->request->if_range)
	{
	  int is_etag = 0, is_weak = 0;

	  /* First, check if the supplied range is an ETag or
	     not... */
	  if (session->request->if_range[0] == '"')
	    is_etag = 1;

	  if (!is_etag && strlen (session->request->if_range) > 2)
	    {
	      if (session->request->if_range[0] == 'W' &&
		  session->request->if_range[0] == '/')
		{
		  is_etag = 1;
		  is_weak = 1;
		}
	    }

	  if (is_etag)
	    {
	      /* If it is an ETag, compare it with ours */

	      if (!is_weak)
		{
		  if (session->etag[0] == 'W')
		    handle_range = 0;
		  if (strcmp (session->request->if_range, session->etag))
		    handle_range = 0;
		}
	      else
		handle_range = 0;
	    }
	  else
	    {
	      /* If it is a date, return the whole document if it was
		 modified since then. Otherwise handle the range
		 request. */
	      struct tm date;

	      strptime (session->request->if_range,
			"%a, %d %b %Y %H:%M:%S", &date);
	      if (_session_date_compare (&date,
					 gmtime (&(st.st_mtime))) != 0)
		handle_range = 0;
	    }
	}
    }
  else
    handle_range = 0;

  if (handle_range)
    {
      /* Check if the range is satisfiable */
      if ((session->request->range_end > 0 &&
	   session->body.offset > session->request->range_end) ||
	  ((size_t) session->body.offset > session->body.content_size &&
	   session->body.offset > 0) ||
	  (session->body.offset < 0 &&
	   (size_t) -session->body.offset > session->body.content_size) ||
	  ((size_t) session->request->range_end >
	   session->body.content_size))
	{
	  session->body.offset = 0;
	  session->body.content_length = 0;
	  return _session_error_end (session, HTTP_STATUS_416);
	}

      session_header (session, HTTP_STATUS_206);
      if (session->request->range_end > 0)
	{
	  asprintf (&tmp, "bytes " SIZET_FORMAT "-" SIZET_FORMAT
		    "/" SIZET_FORMAT, (size_t)session->body.offset,
		    (size_t)session->request->range_end,
		    (size_t)st.st_size);
	  session->body.content_length =
	    session->request->range_end - session->body.offset;
	  session->body.content_size = session->request->range_end;
	}
      else
	{
	  if (session->body.offset >= 0)
	    {
	      asprintf (&tmp, "bytes " SIZET_FORMAT "-" SIZET_FORMAT
			"/" SIZET_FORMAT, (size_t)session->body.offset,
			(size_t)st.st_size, (size_t)st.st_size);
	      session->body.content_length =
		st.st_size - session->body.offset;
	    }
	  else
	    {
	      asprintf (&tmp, "bytes " SIZET_FORMAT "-" SIZET_FORMAT
			"/" SIZET_FORMAT, (size_t)(st.st_size +
						   session->body.offset),
			(size_t)st.st_size, (size_t)st.st_size);
	      session->body.content_length = -session->body.offset;
	      session->body.offset = st.st_size + session->body.offset;
	    }
	}
      session_header_add (session, "Content-Range", tmp);
      free (tmp);
    }

  if (!session->mime)
    session->mime = bhc_strdup (mime_type (session->request->file,
					   default_type,
					   session->absuri));
  session_header_add (session, "Content-Type", session->mime);
  session_header_add (session, "Last-Modified", rfc822_date (st.st_mtime));

  /* Is it modified since ... ? */
  if (session->request->modified_since.tm_mday &&
      _session_date_compare (&(session->request->modified_since),
			     gmtime (&(st.st_mtime))) != 2)
    {
      free (session->body.buffer);
      session->body.buffer = NULL;
      session->body.content_size = 0;

      session_header (session, HTTP_STATUS_304);
      session_headers_flush (session);
      return 0;
    }

  /* Is it unmodified since ... ? */
  if (session->request->unmodified_since.tm_mday &&
      _session_date_compare (&(session->request->unmodified_since),
			     gmtime (&(st.st_mtime))) != 1)
    return _session_error_end (session, HTTP_STATUS_412);

  _session_putfile (session, session->request->resolved);
  return 0;
}

/** Finalise a session.
 * Finalise the session, by determining how it should be sent to the
 * client.
 */
void
session_finalise (session_t *session)
{
  if (session->body.fn && session->cgi.child < 0)
    {
      if (thy_tls_istls (session))
	session->body.method = SESSION_OUTPUT_METHOD_SIMPLEFILE;
      else
	{
#if THY_OPTION_ZLIB
	  if (session->compress.encoding == ENCODING_TYPE_DYNAMIC)
	    session->body.method = SESSION_OUTPUT_METHOD_SIMPLEFILE;
	  else
#endif
	    session->body.method = SESSION_OUTPUT_METHOD_SENDFILE;
	}
    }
  else
    session->body.method = SESSION_OUTPUT_METHOD_BUFFER;

  if (session->state == SESSION_STATE_OUTPUT_HEAD ||
      session->state == SESSION_STATE_OUTPUT_BODY ||
      session->state == SESSION_STATE_OUTPUT_100 ||
      session->state == SESSION_STATE_OUTPUT_101)
    session->throttle = _session_throttle_headers;
  else if (session->state == SESSION_STATE_CGI_HEADER_OUTPUT ||
	   session->state == SESSION_STATE_CGI_BODY_OUTPUT)
    session->throttle = _session_throttle_buffer;
}

/*
 * These are all internal session_throttle_f functions. They send out
 * part of the response from various sources in different ways.
 */
/** @internal Send out a file using put_file().
 */
static int
_session_throttle_sendfile (void *data)
{
  session_t *session = (session_t *)data;

  if ((size_t) session->body.offset < session->body.content_size &&
      session->body.content_length > 0)
    return put_file (session->filefd, session,
		     session->body.content_size,
		     &session->body.offset);

  return -1;
}

/** @internal A small helper function for put_buffer().
 * Pushes a buffer out to the network, using put_buffer(), if it is
 * not already there.
 */
static int
_session_throttle_smallbuff (session_t *session, char *buff,
			     size_t size, off_t *offset)
{
  if ((size_t)(*offset) < size)
    return put_buffer (session, buff, size, offset);
  else
    return -1;
}

/** @internal Send out a buffer using put_buffer().
 */
static int
_session_throttle_buffer (void *data)
{
  session_t *session = (session_t *)data;
  int i = -1;

  if (session->chunked.enabled)
    i = _session_throttle_smallbuff (session,
				     session->chunked.prechunk.buff,
				     session->chunked.prechunk.size,
				     &session->chunked.prechunk.offset);

  if (i != -1)
    return i;

  if ((size_t) session->body.offset < session->body.content_size &&
      session->body.content_length > 0)
    i = put_buffer (session, session->body.buffer,
		    session->body.content_size,
		    &session->body.offset);

  if (i != -1)
    return i;

  if (session->chunked.enabled)
    return _session_throttle_smallbuff
      (session, session->chunked.postchunk.buff,
       session->chunked.postchunk.size,
       &session->chunked.postchunk.offset);
  else
    return -1;
}

/** @internal Send out a file using put_file_simple().
 */
static int
_session_throttle_simplefile (void *data)
{
  session_t *session = (session_t *)data;

  if ((size_t) session->body.offset < session->body.content_size &&
      session->body.content_length > 0)
    return put_file_simple (session->filefd, session,
			    session->body.content_size,
			    &session->body.offset);

  return -1;
}

/** @internal Send out session headers.
 * Send out session headers, and change the throttling function if all
 * headers are out.
 */
static int
_session_throttle_headers (void *data)
{
  session_t *session = (session_t *)data;

  if ((size_t) session->header.offset < session->header.len)
    {
      return put_buffer (session, session->header.fields,
			 session->header.len - session->header.offset,
			 &session->header.offset);
    }

  if (session->state != SESSION_STATE_OUTPUT_100 &&
      session->state != SESSION_STATE_OUTPUT_101)
    session_state_change (session, SESSION_STATE_OUTPUT_BODY);

  if (session->request->method != HTTP_METHOD_HEAD &&
      session->request->method != HTTP_METHOD_OPTIONS)
    {
      switch (session->body.method)
	{
	case SESSION_OUTPUT_METHOD_SENDFILE:
	  session->throttle = _session_throttle_sendfile;
	  break;
	case SESSION_OUTPUT_METHOD_BUFFER:
	  session->throttle = _session_throttle_buffer;
	  break;
	case SESSION_OUTPUT_METHOD_SIMPLEFILE:
	  session->throttle = _session_throttle_simplefile;
	  break;
	default:
	  return -1;
	}
      return 0;
    }
  else
    return -1;
}

/** @internal Convert a session state name to a string.
 * Converts the internal representation of a state to a normal,
 * outputable string.
 *
 * @param nstate is the state to convert to a string.
 *
 * @returns The string representation of the state.
 */
#if THY_OPTION_DEBUG
static const char *_session_state_map[] = {
  "HANDSHAKE",
  "INPUT_REQUEST",
  "OUTPUT_HEAD",
  "OUTPUT_BODY",
  "PROCESS",
  "CGI_HEADER_INPUT",
  "CGI_HEADER_OUTPUT",
  "CGI_BODY_INPUT",
  "CGI_BODY_OUTPUT",
  "POST_INPUT"
  "POST_OUTPUT",
  "OUTPUT_100",
  "OUTPUT_101",
  "INPUT_DISCARD",
  NULL
};

static const char *
_session_state_name (session_state_t nstate)
{
  if (nstate <= SESSION_STATE_INPUT_DISCARD)
    return _session_state_map[nstate];
  else
    return NULL;
}
#endif

/** Change the state of a session.
 * This function is the only allowed way to fiddle with
 * session->state. For now, it only logs what changed and changes the
 * state for real. In the future it may do more, though...
 *
 * @param session is the session we are working on.
 * @param nstate is the state we would like to change to.
 */
void
session_state_change (session_t *session, session_state_t nstate)
{
#if THY_OPTION_DEBUG
  const char *from, *to;

  from = _session_state_name (session->state);
  to = _session_state_name (nstate);

  bhc_debug ("Session %d changing state from %s to %s.",
	     session->io.in, from, to);
#endif

  switch (nstate)
    {
    case SESSION_STATE_POST_INPUT:
      thy_nq_fd_control (session->cgi.pipes.out[1], THY_NQ_EVENT_NONE, 0);
      thy_nq_fd_control (session->io.in, THY_NQ_EVENT_INPUT, 1);
      break;

    case SESSION_STATE_POST_OUTPUT:
      thy_nq_fd_control (session->cgi.pipes.in[0], THY_NQ_EVENT_NONE, 0);
      thy_nq_fd_control (session->cgi.pipes.out[1],
			 THY_NQ_EVENT_OUTPUT, 1);
      break;

    case SESSION_STATE_INPUT_REQUEST:
    case SESSION_STATE_INPUT_DISCARD:
    case SESSION_STATE_HANDSHAKE:
      thy_nq_fd_control (session->io.in, THY_NQ_EVENT_INPUT, 1);
      break;

    case SESSION_STATE_OUTPUT_HEAD:
    case SESSION_STATE_OUTPUT_BODY:
    case SESSION_STATE_OUTPUT_100:
    case SESSION_STATE_OUTPUT_101:
    case SESSION_STATE_CGI_HEADER_OUTPUT:
    case SESSION_STATE_CGI_BODY_OUTPUT:
      thy_nq_fd_control (session->io.out, THY_NQ_EVENT_OUTPUT, 1);
      break;

    case SESSION_STATE_CGI_HEADER_INPUT:
      thy_nq_fd_control (session->cgi.pipes.out[1], THY_NQ_EVENT_NONE, 0);
    case SESSION_STATE_CGI_BODY_INPUT:
      thy_nq_fd_control (session->io.in, THY_NQ_EVENT_NONE, 0);
      thy_nq_fd_control (session->io.out, THY_NQ_EVENT_NONE, 0);
      thy_nq_fd_control (session->cgi.pipes.in[0], THY_NQ_EVENT_INPUT, 1);
      break;

    case SESSION_STATE_PROCESS:
      thy_nq_fd_control (session->io.in, THY_NQ_EVENT_TRIGGER, 1);
      break;
    }

  session->state = nstate;
}
