/* -*- mode: c; c-file-style: "gnu" -*-
 * conffile.y -- Configuration file grammar for Thy.
 * Copyright (C) 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
/* arch-tag: 9891e4ba-36c3-41f0-aeff-0ad891faf6c9 */
%{
#define YYSTYPE char *
#define add_arg(x,y) { bhl_list_append_string (conffile_list, (x)); \
		       bhl_list_append_string (conffile_list, (y)); }
#define add_arg2(x,y,z) { char *a_tmp; asprintf (&a_tmp, (y), (z)); \
			  add_arg ((x), a_tmp); free (a_tmp); }

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include "system.h"

#include "compat/compat.h"
#include "bh-libs/list.h"

extern const char *current_file;
#if YYTEXT_POINTER
extern char *yytext;
#else
extern char yytext[];
#endif
extern bhl_list_t *conffile_list;
extern int yylineno;
char *tmp;
char *vhost = NULL;
int map = 0;

int yylex (void);
int yyerror (char const *msg);
%}

%token COMMENT ERROR

%token T_BOOL_OFF T_BOOL_ON
%token T_DOCUMENTROOT T_LISTEN T_DEFAULT_TYPE T_TIMEOUT
%token T_USERDIR T_UID T_PIDFILE T_IPV4 T_IPV6 T_MAP T_INDEXES
%token T_CGIEXTS T_ERRORDOC T_CGIDIRS T_ADDENV T_ADDHANDLER
%token T_ADDMETHOD T_ADDHEADER T_ADDTYPE T_GZIP T_TYPE
%token T_LEVEL T_GZIP_TYPE T_GZIP_LEVEL T_USERID T_UNMAP

%token T_LIMIT T_LIMIT_CGIS T_LIMIT_HEADER T_LIMIT_POST_BUFFER
%token T_CGIS T_HEADER T_BUFFER T_MAXCLIENTS T_LIMIT_MAXCLIENTS
%token T_MMAP T_LIMIT_MMAP

%token T_AUTH T_AUTH_PATH T_AUTH_UID T_AUTH_ARGS
%token T_PATH T_ARGS T_FILE T_AUTH_FILE

%token T_ETAG T_DIRETAG

%token T_WORKER T_WORKER_PATH T_WORKER_UID T_WORKER_ARGS

%token T_SSL T_CA T_CERT T_KEY T_KEYRING T_TRUSTDB T_VERIFY
%token T_SSL_CA T_SSL_CERT T_SSL_KEY T_SSL_KEYRING T_SSL_TRUSTDB
%token T_SSL_VERIFY T_SSL_TYPE
%token T_SRPPASSWD T_SRPCONF T_SSL_SRPPASSWD T_SSL_SRPCONF

%token T_CHEAP_VHOST T_CGI T_USERCGI T_CASEMIME
%token T_FOLLOWALL T_SERVER T_VARY T_CHROOT
%token T_DIRINDEX T_EXPECT T_HARDLIMIT T_STATS T_LAZYCGI

%token T_VIRTUALHOST T_UNVIRTUALHOST

%token T_ALIAS T_SCRIPTALIAS

%token T_CACHE T_CACHE_NO_TRANSFORM T_CACHE_NO_CACHE T_CACHE_NO_STORE
%token T_CACHE_MAX_AGE T_CACHE_MUST_REVALIDATE T_CACHE_EXPIRY_BASE

%token T_KEEPALIVETIMEOUT T_MAXKEEPALIVEREQUESTS

%token T_INT T_STRING
%%

rcfile		: /* empty */
		| statement_list
		;

statement_list	: statement
		| statement_list statement
		;

statement	: COMMENT { /* do nothing */ }

		| ERROR { yyerror ("syntax error"); YYERROR; }

		| T_DOCUMENTROOT T_STRING { add_arg ("-w", $2); }

		| T_LISTEN T_STRING { add_arg ("-l", $2); }
		| T_IPV4 { bhl_list_append_string (conffile_list, "-4"); }
		| T_IPV6 { bhl_list_append_string (conffile_list, "-6"); }

		| T_DEFAULT_TYPE T_STRING { add_arg ("-d", $2); }
		| T_TIMEOUT T_INT { add_arg ("-t", $2); }
		| T_KEEPALIVETIMEOUT T_INT { add_arg2 ("-t", "keepalive=%s", $2); }
		| T_USERDIR T_STRING { add_arg ("-u", $2); }
		| T_USERID T_INT { add_arg ("-U", $2); }

		| T_INDEXES list { add_arg ("-i", $2); }
		| T_CGIEXTS list { add_arg ("-c", $2); }
		| T_CGIDIRS list { add_arg ("-C", $2); }
		| T_ERRORDOC pair { add_arg ("-e", $2); }

		| T_ADDENV pair { add_arg ("-E", $2); }
		| T_ADDHANDLER pair { add_arg ("-H", $2); }
		| T_ADDMETHOD pair { add_arg ("-M", $2); }
		| T_ADDHEADER pair { add_arg ("-D", $2); }
		| T_ADDTYPE pair { add_arg ("-m", $2); }

		| T_ALIAS pair { add_arg ("-A", $2); }

		| T_VIRTUALHOST T_STRING '>'
			{
			  if (vhost)
			    {
			      yyerror ("syntax error: nested virtual hosts not supported");
			      YYERROR;
			    }
			  add_arg2 ("-R", "^https*://%s[:/]", $2);
			  vhost = strdup ($2);
			}
		| T_UNVIRTUALHOST '>'
			{
			  if (vhost == NULL)
			    {
			      yyerror ("syntax error: no starting tag for </virtualhost>");
			      YYERROR;
			    }
			  bhl_list_append_string (conffile_list, "-F");
			  free (vhost);
			  vhost = NULL;
			}

		| T_MAP T_STRING '>'
			{
			  if (map == 1)
			    {
			      yyerror ("syntax error: nested maps are not supported");
			      YYERROR;
			    }
			  map = 1;
			  if (vhost == NULL)
			    {
			      add_arg ("-R", $2);
			    }
			  else
			    {
			      char *t = ($2[0] == '/') ? "" : ".*";
			      asprintf (&tmp, "^https*://%s[^/]%s%s", vhost, t, $2);
			      add_arg ("-R", tmp);
			      free (tmp);
			    }
			}
		| T_UNMAP '>'
			{
			  if (map == 0)
			    {
			      yyerror ("syntax error: no starting tag for </map>");
			      YYERROR;
			    }
			  if (vhost == NULL)
			    bhl_list_append_string (conffile_list, "-F");
			  else
			    add_arg2 ("-R", "^https*://%s[:/]", vhost);
			  map = 0;
			}

		| T_GZIP { add_arg ("-g", "type=static"); }
		| T_GZIP T_BOOL_OFF { add_arg ("-g", "type=none"); }
		| T_GZIP T_BOOL_ON { add_arg ("-g", "type=static"); }
		| T_GZIP T_TYPE '=' T_STRING { add_arg2 ("-g", "type=%s", $4); }
		| T_GZIP T_LEVEL '=' T_INT { add_arg2 ("-g", "level=%s", $4); }
		| T_GZIP_TYPE T_STRING { add_arg2 ("-g", "type=%s", $2); }
		| T_GZIP_LEVEL T_INT { add_arg2 ("-g", "level=%s", $2); }

		| T_LIMIT T_CGIS '=' T_INT { add_arg2 ("-L", "cgis=%s", $4); }
		| T_LIMIT_CGIS T_INT { add_arg2 ("-L", "cgis=%s", $2); }
		| T_LIMIT T_HEADER '=' T_INT { add_arg2 ("-L", "header=%s", $4); }
		| T_LIMIT_HEADER T_INT { add_arg2 ("-L", "header=%s", $2); }
		| T_LIMIT T_BUFFER '=' T_INT { add_arg2 ("-L", "post_buffer=%s", $4); }
		| T_LIMIT_POST_BUFFER T_INT { add_arg2 ("-L", "post_buffer=%s", $2); }
		| T_MAXKEEPALIVEREQUESTS T_INT { add_arg2 ("-L", "keepalive=%s", $2); }
		| T_LIMIT T_MAXCLIENTS '=' T_INT { add_arg2 ("-L", "maxclients=%s", $4); }
		| T_LIMIT_MAXCLIENTS T_INT { add_arg2 ("-L", "maxclients=%s", $2); }
		| T_LIMIT T_MMAP '=' T_INT { add_arg2 ("-L", "mmap=%s", $4); }
		| T_LIMIT_MMAP T_INT { add_arg2 ("-L", "mmap=%s", $2); }

		| T_AUTH T_BOOL_OFF { add_arg ("-o", "noauth"); }
		| T_AUTH T_BOOL_ON { add_arg ("-o", "auth"); }
		| T_AUTH T_PATH '=' T_STRING { add_arg2 ("-a", "path=%s", $4); }
		| T_AUTH_PATH T_STRING { add_arg2 ("-a", "path=%s", $2); }
		| T_AUTH T_UID '=' T_INT { add_arg2 ("-a", "uid=%s", $4); }
		| T_AUTH_UID T_INT { add_arg2 ("-a", "uid=%s", $2); }
		| T_AUTH T_ARGS '=' arglist { add_arg ("-a", $4); }
		| T_AUTH_ARGS arglist { add_arg ("-a", $2); }
		| T_AUTH T_FILE '=' T_STRING { add_arg2 ("-a", "file=%s", $4); }
		| T_AUTH_FILE T_STRING { add_arg2 ("-a", "file=%s", $2); }
		| T_AUTH { add_arg ("-o", "auth"); }

		| T_ETAG T_BOOL_OFF { add_arg ("-T", "noetag"); }
		| T_ETAG T_BOOL_ON { add_arg ("-T", "etag"); }
		| T_ETAG { add_arg ("-T", "etag"); }
		| T_DIRETAG T_BOOL_OFF { add_arg ("-T", "nodirtag"); }
		| T_DIRETAG T_BOOL_ON { add_arg ("-T", "dirtag"); }
		| T_DIRETAG { add_arg ("-T", "dirtag"); }

		| T_WORKER T_BOOL_OFF { add_arg ("-o", "noworker"); }
		| T_WORKER T_BOOL_ON { add_arg ("-o", "worker"); }
		| T_WORKER T_PATH '=' T_STRING { add_arg2 ("-W", "path=%s", $4); }
		| T_WORKER_PATH T_STRING { add_arg2 ("-W", "path=%s", $2); }
		| T_WORKER T_UID '=' T_INT { add_arg2 ("-W", "uid=%s", $4); }
		| T_WORKER_UID T_INT { add_arg2 ("-W", "uid=%s", $2); }
		| T_WORKER T_ARGS '=' arglist { add_arg ("-W", $4); }
		| T_WORKER_ARGS arglist { add_arg ("-W", $2); }
		| T_WORKER { add_arg ("-o", "worker"); }

		| T_SSL { add_arg ("-o", "ssl"); }
		| T_SSL T_BOOL_OFF { add_arg ("-o", "nossl"); }
		| T_SSL T_BOOL_ON { add_arg ("-o", "ssl"); }
		| T_SSL T_CA '=' T_STRING { add_arg2 ("-s", "ca=%s", $4); }
		| T_SSL_CA T_STRING { add_arg2 ("-s", "ca=%s", $2); }
		| T_SSL T_CERT '=' T_STRING { add_arg2 ("-s", "cert=%s", $4); }
		| T_SSL_CERT T_STRING { add_arg2 ("-s", "cert=%s", $2); }
		| T_SSL T_KEY '=' T_STRING { add_arg2 ("-s", "key=%s", $4); }
		| T_SSL_KEY T_STRING { add_arg2 ("-s", "key=%s", $2); }
		| T_SSL T_KEYRING '=' T_STRING { add_arg2 ("-s", "keyring=%s", $4); }
		| T_SSL_KEYRING T_STRING { add_arg2 ("-s", "keyring=%s", $2); }
		| T_SSL T_TRUSTDB '=' T_STRING { add_arg2 ("-s", "trustdb=%s", $4); }
		| T_SSL_TRUSTDB T_STRING { add_arg2 ("-s", "trustdb=%s", $2); }
		| T_SSL T_TYPE '=' T_STRING { add_arg2 ("-s", "type=%s", $4); }
		| T_SSL_TYPE T_STRING { add_arg2 ("-s", "type=%s", $2); }
		| T_SSL T_VERIFY '=' T_INT { add_arg2 ("-s", "verify=%s", $4); }
		| T_SSL_VERIFY T_INT { add_arg2 ("-s", "verify=%s", $2); }
		| T_SSL T_SRPPASSWD '=' T_STRING { add_arg2 ("-s", "srppasswd=%s", $4); }
		| T_SSL_SRPPASSWD T_STRING { add_arg2 ("-s", "srppasswd=%s", $2); }
		| T_SSL T_SRPCONF '=' T_STRING { add_arg2 ("-s", "srpconf=%s", $4); }
		| T_SSL_SRPCONF T_STRING { add_arg2 ("-s", "srpconf=%s", $2); }

		| T_CHEAP_VHOST { add_arg ("-o", "vhost"); }
		| T_CHEAP_VHOST T_BOOL_ON { add_arg ("-o", "vhost"); }
		| T_CHEAP_VHOST T_BOOL_OFF { add_arg ("-o", "novhost"); }

		| T_USERDIR { add_arg ("-o", "userdir"); }
		| T_USERDIR T_BOOL_ON { add_arg ("-o", "userdir"); }
		| T_USERDIR T_BOOL_OFF { add_arg ("-o", "nouserdir"); }

		| T_CGI { add_arg ("-o", "cgi"); }
		| T_CGI T_BOOL_ON { add_arg ("-o", "cgi"); }
		| T_CGI T_BOOL_OFF { add_arg ("-o", "nocgi"); }

		| T_USERCGI { add_arg ("-o", "usercgi"); }
		| T_USERCGI T_BOOL_ON { add_arg ("-o", "usercgi"); }
		| T_USERCGI T_BOOL_OFF { add_arg ("-o", "nousercgi"); }

		| T_CASEMIME { add_arg ("-o", "casemime"); }
		| T_CASEMIME T_BOOL_ON { add_arg ("-o", "casemime"); }
		| T_CASEMIME T_BOOL_OFF { add_arg ("-o", "nocasemime"); }

		| T_BUFFER { add_arg ("-o", "buffer"); }
		| T_BUFFER T_BOOL_ON { add_arg ("-o", "buffer"); }
		| T_BUFFER T_BOOL_OFF { add_arg ("-o", "nobuffer"); }
		| T_BUFFER T_INT { add_arg2 ("-o", "buffer=%s", $2); }

		| T_FOLLOWALL { add_arg ("-o", "followall"); }
		| T_FOLLOWALL T_BOOL_ON { add_arg ("-o", "followall"); }
		| T_FOLLOWALL T_BOOL_OFF { add_arg ("-o", "nofollowall"); }

		| T_SERVER T_STRING { add_arg2 ("-o", "server=%s", $2); }
		| T_SERVER T_BOOL_OFF { add_arg ("-o", "noserver"); }
		| T_SERVER T_BOOL_ON { /* ignore */ }

		| T_VARY { add_arg ("-o", "vary"); }
		| T_VARY T_BOOL_ON { add_arg ("-o", "vary"); }
		| T_VARY T_BOOL_OFF { add_arg ("-o", "novary"); }

		| T_CHROOT { add_arg ("-o", "chroot"); }
		| T_CHROOT T_BOOL_ON { add_arg ("-o", "chroot"); }
		| T_CHROOT T_BOOL_OFF { add_arg ("-o", "nochroot"); }

		| T_DIRINDEX { add_arg ("-o", "dirindex"); }
		| T_DIRINDEX T_BOOL_ON { add_arg ("-o", "dirindex"); }
		| T_DIRINDEX T_BOOL_OFF { add_arg ("-o", "nodirindex"); }

		| T_EXPECT { add_arg ("-o", "expect"); }
		| T_EXPECT T_BOOL_ON { add_arg ("-o", "expect"); }
		| T_EXPECT T_BOOL_OFF { add_arg ("-o", "noexpect"); }

		| T_HARDLIMIT { add_arg ("-o", "hardlimit"); }
		| T_HARDLIMIT T_BOOL_ON { add_arg ("-o", "hardlimit"); }
		| T_HARDLIMIT T_BOOL_OFF { add_arg ("-o", "nohardlimit"); }

		| T_STATS { /* ignore */ }
		| T_STATS T_BOOL_ON { /* ignore */ }
		| T_STATS T_BOOL_OFF { add_arg ("-o", "nostats"); }
		| T_STATS T_INT { add_arg2 ("-o", "stats=%s", $2); }

		| T_LAZYCGI { add_arg ("-o", "lazycgi"); }
		| T_LAZYCGI T_BOOL_ON { add_arg ("-o", "lazycgi"); }
		| T_LAZYCGI T_BOOL_OFF { add_arg ("-o", "nolazycgi"); }

		| T_PIDFILE T_STRING { add_arg2 ("-o", "pidfile=%s", $2); }
		| T_PIDFILE T_BOOL_ON { /* ignore */ }
		| T_PIDFILE T_BOOL_OFF { add_arg ("-o", "nopidfile"); }

		| T_CACHE T_BOOL_ON { add_arg ("-o", "cache"); }
		| T_CACHE T_BOOL_OFF { add_arg ("-o", "nocache"); }
		| T_CACHE T_CACHE_MAX_AGE '=' T_INT { add_arg2 ("-X", "max-age=%s", $4); }
		| T_CACHE T_CACHE_MAX_AGE T_BOOL_OFF { add_arg ("-X", "nomax-age"); }
		| T_CACHE T_CACHE_NO_CACHE { add_arg ("-X", "no-cache"); }
		| T_CACHE T_CACHE_NO_CACHE T_BOOL_ON { add_arg ("-X", "no-cache"); }
		| T_CACHE T_CACHE_NO_CACHE T_BOOL_OFF { add_arg ("-X", "cache"); }
		| T_CACHE T_CACHE_NO_STORE { add_arg ("-X", "no-store"); }
		| T_CACHE T_CACHE_NO_STORE T_BOOL_ON { add_arg ("-X", "no-store"); }
		| T_CACHE T_CACHE_NO_STORE T_BOOL_OFF { add_arg ("-X", "store"); }
		| T_CACHE T_CACHE_NO_TRANSFORM { add_arg ("-X", "no-transform"); }
		| T_CACHE T_CACHE_NO_TRANSFORM T_BOOL_ON { add_arg ("-X", "no-transform"); }
		| T_CACHE T_CACHE_NO_TRANSFORM T_BOOL_OFF { add_arg ("-X", "transform"); }
		| T_CACHE T_CACHE_MUST_REVALIDATE  { add_arg ("-X", "must-revalidate"); }
		| T_CACHE T_CACHE_EXPIRY_BASE '=' T_STRING { add_arg2 ("-X", "expiry-base=%s", $4); }

		| T_SCRIPTALIAS T_STRING {tmp = strdup ($2);}
				T_STRING
			{
			  char *tmp2;

			  asprintf (&tmp2, "%s=%s", tmp, $4);
			  free (tmp);
			  add_arg ("-A", tmp2);
			  free (tmp2);
			  add_arg ("-C", $4);
			}
		;

pair		: T_STRING {tmp = strdup ($1);} T_STRING
			{
			  asprintf (&$$, "%s=%s", tmp, $3);
			  free (tmp);
			}
		;

list		: T_STRING { $$ = strdup ($1); }
		| list T_STRING { asprintf (&$$, "%s,%s", $1, $2); }
		;

arg		: T_STRING { asprintf (&$$, "arg=%s", $1); }
		;
arglist		: arg
		| arglist arg { asprintf (&$$, "%s,%s", $1, $2); }
		;
%%
int
yyerror (char const *msg)
{
  fprintf (stderr, "thy: %s in config file `%s' on line %d!\n", msg,
	   current_file, yylineno);
  return 1;
}
