/*
 *  Copyright (C) 2000 by Marco G"otze.
 *
 *  This code is part of the ThoughtTracker source package, which is
 *  distributed under the terms of the GNU GPL2.
 */

#ifndef DATABASE_H_INCLUDED
#define DATABASE_H_INCLUDED

#include <string>
#include <list>

#include <glib.h>  // for system-dependent int type definitions etc.

#include <gdbm.h>

#include "thoughttracker.h"

/*
    the GDBM databases used by the class following below look like this:

      - record identifiers are of type dbid_t (4-byte unsigned int),
        little-endian
      - normal record IDs are >= 0, some negative IDs identify special records

      - file "data", the DB file storing entries' data:
         - keys of type dbid_t, data is the concatenation of two 1-byte char,
           zero-terminated strings (the first holding the record's summary,
           the second its details), prefixed an unsigned char for storing flags
         - special records with negative IDs record the maximum ID used so
           far (referred to as MAX_ID below), and the DB layout version (in
           case of future changes)

      - file "links", the DB file storing the records' cross links:
         - keys of type dbid_t, data is an array of dbid_t, listing the entries
           the entry specified by the key is linked to -> clearly, this implies
           redundancies, as each link has to occur in both of the involved
           entries' link records

      - for determining a new record's ID, special MAX_ID record is maintained
        that's increased with each added record and used to assign its ID;
        that way, a new record will always have an ID not hitherto used for
        any record
*/

typedef gint32 dbid_t;  // record ID type
typedef guchar dbf_t;   // flags type, should be sufficient

#define DB_SUCCESS  0
#define DB_FAILURE -1

typedef enum {
  DB_FLAG_FIXED = 1<<0,  // use fixed font
  DB_FLAG_HUB   = 1<<1   // entry is a hub
} dbflags_t;  // per-entry flags

typedef enum {  // contents to search
  DB_SEARCH_SUMMARIES = 1<<0,  // search only summaries (otherwise, everything)
  DB_SEARCH_OR        = 1<<1,  // OR logic (otherwise, AND)
  DB_SEARCH_HUB       = 1<<2,  // search hubs
  DB_SEARCH_ENT       = 1<<3,  // search regular entries
  DB_SEARCH_LNK       = 1<<4,  // search linked entries
  DB_SEARCH_ISO       = 1<<5   // search isolated entries
} dbsearch_t;

class TTDatabase {
  public:
    /* opens/creates database files in <path>, calls <callback> after a
       critical error has occurred */
    TTDatabase(const string &path, void (*callback)(const string&) = 0);

    virtual ~TTDatabase();

    /* stores a data record in the DB, possibly overwriting an existing record
       of the same ID; if <req_id> is negative, the new ID is determined
       automatically; returns the saved record's ID (DB_FAILURE on error) */
    virtual dbid_t store(dbid_t req_id, const dbf_t flags,
	  const string &summary, const string &text);

    /* returns pointers to the summary and text parts of a data record (whose
       memory has to be delete()d by the caller; return value is a DB_*
       constant if the return value is DB_FAILURE, no memory has been allocated
       for <summary> and <text> */
    virtual int read(dbid_t id, dbf_t &flags, string &summary, string &text);

    /* returns true if the specified record exists in the database */
    virtual bool exists(dbid_t);

    /* deletes data record #<ID> AND all links involving it; returns a DB_*
       success code */
    virtual int del(dbid_t id);

    /* links two data records, returns a DB_* success code */
    virtual int link(dbid_t, dbid_t);

    /* returns a truth value specifying whether the two records are linked
       (true is also returned if id1 == id2); DB_FAILURE indicates an error
       condition; as an implementation detail that _normally_ is of no
       interest, the function looks in the first entry's list for the
       second entry's ID */
    virtual bool linked(dbid_t, dbid_t);

    /* returns the number of records #<ID> is linked to, 0 even if an
       error occurred querying the database */
    virtual int linked(dbid_t);

    /* creates a list of all links for record #<id>, returns a DB_* success
       code */
    virtual int links(dbid_t, list<dbid_t>&);

    /* removes a link between two IDs; returns a DB_* success code */
    virtual int unlink(dbid_t, dbid_t);

    /* deletes all links involving ID <id>; returns a DB_* success code */
    virtual int unlink_all(dbid_t id);

    /* searches for records matching the search string and forms a list of
       their IDs, up to <max> of them in <list>, returning the number of
       elements that were actually saved; returns a DB_* success code */
    virtual int search(const string&, unsigned, list<dbid_t> &list,
	  unsigned int max);

    /* consistency check functions (check & repair) returning the number of
       errors found or DB_FAILURE if there was some other error, DB_SUCCESS
       indicates that there were no errors; these functions are recommended
       to be run in the order they're declared here */
    virtual int check_administrative_information(bool repair);
    virtual int check_data_structures(bool repair);
    virtual int check_stray_linklists(bool repair);
    virtual int check_linklist_structures(bool repair);
    virtual int check_links_consistency(bool repair);

    /* will try to optimize the database and return true on success, false
       otherwise */
    virtual bool optimize();

  private:
    static const dbid_t max_id_key;
    static const dbid_t ver_key;
    static const datum max_id_key_datum;

    GDBM_FILE dbf_data, dbf_links;
    void (*critical_callback)(const string&);
    string critical_msg;
    bool critical;

    /* set critical_msg to <msg> and critical to true, thus preventing most
       other functions from doing anything */
    virtual void set_critical(string msg);

    /* adds <link> to the link list of id <list_of>, return DB_* */
    virtual int add_link_to_list(dbid_t list_of, dbid_t link);

    /* removes <link> from the link list of id <list_of>, returns DB_* */
    virtual int del_link_from_list(dbid_t list_of, dbid_t link);

    /* returns a boolean value depending on whether <search_string> matches
       <string>; AND is assumed if <mode> is true, OR otherwise */
    virtual bool match(const string &string, const string &search_string,
      bool mode);

    /* upgrade the DB from version <from> to the current one (whose ID is
       given with <to>; sets critical on errors */
    virtual void upgrade_db(const string &filename, dbid_t from, dbid_t to);

    /* adapts the byte order of <id> for portable storage */
    static dbid_t encode(dbid_t id) {
#if G_BYTE_ORDER == G_BIG_ENDIAN
      return (dbid_t) GUINT32_SWAP_LE_BE((guint32) id);
#elif G_BYTE_ORDER == G_PDP_ENDIAN
      return (dbid_t) GUINT32_SWAP_LE_PDP((guint32) id);
#else
      return id;
#endif
    }
    static dbid_t decode(dbid_t id) { return encode(id); }

    /* allocates an entry record via malloc() and initializes it with the
       data given; the memory has to be free()d by the caller */
    static datum construct_record(const dbf_t flags, const string summary,
      const string text);

    /* initializes an entry's data elements from a raw record provided in
       <data>; returns true if the record's integrity was OK, false if
       adaptations due to integrity faults had to be made (i.e., wrong
       record size due to a damaged record etc.) */
    static bool evaluate_record(const datum data, dbf_t &flags,
      string &summary, string &text);
};

#endif  /* DATABASE_H_INCLUDED */

