/***************************************************************************
                         th-job-cropping-dialog.c
                         ------------------------
    begin                : Tue Nov 23 2004
    copyright            : (C) 2004 by Tim-Philipp Mller
    email                : t.i.m@orange.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "th-utils.h"
#include "th-job-cropping-dialog.h"

#include "gst-plugins/th-pixbufsink.h"

#include <gtk/gtk.h>
#include <gst/gst.h>
#include <glib/gi18n.h>
#include <string.h>

enum
{
	TH_RESPONSE_RETRIEVE_ERROR = 9312
};

struct _ThJobCroppingDialogPrivate
{
	ThJob            *job;
	guint             job_len_secs;

	GstElement       *pipeline;

	GError           *error;

	GdkPixbuf        *cur_frame;
	gint              par_n;         /* pixel aspect ratio numerator   */
	gint              par_d;         /* pixel aspect ratio denominator */

	gint              crop_left;
	gint              crop_right;
	gint              crop_bottom;
	gint              crop_top;

	GtkWidget        *image;
	GtkWidget        *spin_top;
	GtkWidget        *spin_left;
	GtkWidget        *spin_right;
	GtkWidget        *spin_bottom;
	GtkWidget        *slider;
	
	GtkWidget        *cancel_button;
	GtkWidget        *ok_button;
};

static void             job_cropping_dialog_class_init    (ThJobCroppingDialogClass *klass);

static void             job_cropping_dialog_instance_init (ThJobCroppingDialog *cp);

static void             job_cropping_dialog_finalize      (GObject *object);

static gboolean         jcd_setup_pipeline           (ThJobCroppingDialog *jcd, GError **err);

static void             jcd_update_frame             (ThJobCroppingDialog *jcd);

static void             jcd_cropping_changed         (ThJobCroppingDialog *jcd,
                                                      GtkWidget           *spin_button);

static void             jcd_setup_with_dimensions    (ThJobCroppingDialog *jcd);

static gchar           *jcd_format_slider_value      (ThJobCroppingDialog *jcd, gdouble val);

static void             jcd_slider_value_changed     (ThJobCroppingDialog *jcd, GtkRange *slider);

/* variables */

static GObjectClass    *jcd_parent_class; /* NULL */

/***************************************************************************
 *
 *   job_cropping_dialog_class_init
 *
 ***************************************************************************/

static void
job_cropping_dialog_class_init (ThJobCroppingDialogClass *klass)
{
	GObjectClass  *object_class; 

	object_class = G_OBJECT_CLASS (klass);
	
	jcd_parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = job_cropping_dialog_finalize;
}

/***************************************************************************
 *
 *   job_cropping_dialog_instance_init
 *
 ***************************************************************************/

static void
job_cropping_dialog_instance_init (ThJobCroppingDialog *jcd)
{
	jcd->priv = g_new0 (ThJobCroppingDialogPrivate, 1);

	g_signal_connect (jcd, "delete-event",
	                  G_CALLBACK (gtk_widget_hide_on_delete),
	                  NULL);
	
	gtk_window_set_icon_from_file (GTK_WINDOW (jcd),
	                               DATADIR "/ui/icons/thoggen.png",
	                               NULL);
}

/***************************************************************************
 *
 *   job_cropping_dialog_finalize
 *
 ***************************************************************************/

static void
job_cropping_dialog_finalize (GObject *object)
{
	ThJobCroppingDialog *jcd;

	jcd = (ThJobCroppingDialog*) object;

	g_print ("ThJobCroppingDialog: finalize\n");

	if (jcd->priv->error)
		g_error_free (jcd->priv->error);

	memset (jcd->priv, 0xab, sizeof (ThJobCroppingDialogPrivate));
	g_free (jcd->priv);
	jcd->priv = NULL;

	/* chain up */
	jcd_parent_class->finalize (object);
}


/***************************************************************************
 *
 *   th_job_cropping_dialog_get_type
 *
 ***************************************************************************/

GType
th_job_cropping_dialog_get_type (void)
{
	static GType type; /* 0 */

	if (type == 0)
	{
		static GTypeInfo info =
		{
			sizeof (ThJobCroppingDialogClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) job_cropping_dialog_class_init,
			NULL, NULL,
			sizeof (ThJobCroppingDialog),
			0,
			(GInstanceInitFunc) job_cropping_dialog_instance_init
		};

		type = g_type_register_static (GTK_TYPE_DIALOG, "ThJobCroppingDialog", &info, 0);
	}

	return type;
}


/***************************************************************************
 *
 *   th_job_cropping_dialog_new
 *
 ***************************************************************************/

GtkWidget *
th_job_cropping_dialog_new (void)
{
	ThJobCroppingDialog *jcd;
	GtkWidget           *glade_window = NULL;
	GtkWidget           *toplevel_vbox;

	jcd = (ThJobCroppingDialog *) g_object_new (TH_TYPE_JOB_CROPPING_DIALOG, NULL);
	
	if (!th_utils_ui_load_interface ("th-job-cropping-dialog.glade",  FALSE,
	                                 "th-job-cropping-dialog", &glade_window,
	                                 "th-toplevel-vbox",       &toplevel_vbox,
	                                 "th-crop-image",          &jcd->priv->image,
	                                 "th-top-spin",            &jcd->priv->spin_top,
	                                 "th-left-spin",           &jcd->priv->spin_left,
	                                 "th-right-spin",          &jcd->priv->spin_right,
	                                 "th-bottom-spin",         &jcd->priv->spin_bottom,
	                                 "th-slider",              &jcd->priv->slider,
	                                 NULL))
	{
		g_printerr ("th_utils_ui_load_interface (\"%s\") failed.\n",
		            "th-job-cropping-dialog.glade");
		if (glade_window)
			gtk_widget_destroy (glade_window);
		gtk_widget_destroy (GTK_WIDGET (jcd));
		return NULL;
	}
	
	g_object_ref (toplevel_vbox);
	gtk_container_remove (GTK_CONTAINER (glade_window), toplevel_vbox);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (jcd)->vbox), toplevel_vbox);
	g_object_unref (toplevel_vbox);

	jcd->priv->cancel_button = gtk_dialog_add_button (GTK_DIALOG (jcd),
	                                                  GTK_STOCK_CANCEL,
	                                                  GTK_RESPONSE_CANCEL);

	/* makes cancel button stick to the left; let's see
	 *  how long it takes until some HIGian complains ;) */
	gtk_button_box_set_child_secondary (GTK_BUTTON_BOX (GTK_DIALOG(jcd)->action_area),
	                                    jcd->priv->cancel_button,
	                                    TRUE);

	jcd->priv->ok_button = gtk_dialog_add_button (GTK_DIALOG (jcd),
	                                              GTK_STOCK_OK,
	                                              GTK_RESPONSE_ACCEPT);
	
	g_signal_connect_swapped (jcd->priv->spin_top, "value-changed",
	                          G_CALLBACK (jcd_cropping_changed), jcd);

	g_signal_connect_swapped (jcd->priv->spin_left, "value-changed",
	                          G_CALLBACK (jcd_cropping_changed), jcd);

	g_signal_connect_swapped (jcd->priv->spin_right, "value-changed",
	                          G_CALLBACK (jcd_cropping_changed), jcd);

	g_signal_connect_swapped (jcd->priv->spin_bottom, "value-changed",
	                          G_CALLBACK (jcd_cropping_changed), jcd);

	g_signal_connect_swapped (jcd->priv->slider, "format-value",
	                          G_CALLBACK (jcd_format_slider_value), jcd);

	g_signal_connect_swapped (jcd->priv->slider, "value-changed",
	                          G_CALLBACK (jcd_slider_value_changed), jcd);

	gtk_spin_button_set_increments (GTK_SPIN_BUTTON (jcd->priv->spin_top), 
	                                2.0, 16.0);
	
	gtk_spin_button_set_increments (GTK_SPIN_BUTTON (jcd->priv->spin_left), 
	                                2.0, 16.0);
	
	gtk_spin_button_set_increments (GTK_SPIN_BUTTON (jcd->priv->spin_right), 
	                                2.0, 16.0);
	
	gtk_spin_button_set_increments (GTK_SPIN_BUTTON (jcd->priv->spin_bottom), 
	                                2.0, 16.0);

	gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (jcd->priv->spin_top), TRUE);
	gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (jcd->priv->spin_left), TRUE);
	gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (jcd->priv->spin_right), TRUE);
	gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (jcd->priv->spin_bottom), TRUE);
	
	/* spin button ranges are set in jcd_setup_with_dimensions()
	 *  once we know the size of the input and of the snapshots */

	return GTK_WIDGET (jcd);
}

/***************************************************************************
 *
 *   jcd_format_slider_value
 *
 ***************************************************************************/

static gchar *
jcd_format_slider_value (ThJobCroppingDialog *jcd, gdouble val)
{
	gint hh, mm, ss, secs;

	secs = (gint) val;
	hh = secs / (60*60);
	mm = (secs % (60*60)) / 60;
	ss = secs % 60;

	return g_strdup_printf ("%u:%02u:%02u", hh, mm, ss);
}

/***************************************************************************
 *
 *   jcd_slider_value_changed
 *
 ***************************************************************************/

static void
jcd_slider_value_changed (ThJobCroppingDialog *jcd, GtkRange *slider)
{
	gboolean ret;
	gdouble dval;
	gint64 seek_pos;
	
	dval = gtk_range_get_value (GTK_RANGE (slider)) * (gdouble) GST_SECOND;

	seek_pos = (gint64) dval;

	/* g_print ("Seek to %" GST_TIME_FORMAT, GST_TIME_ARGS (seek_pos)); */

	ret = gst_element_seek_simple (jcd->priv->pipeline, GST_FORMAT_TIME,
	                        GST_SEEK_FLAG_FLUSH | GST_SEEK_FLAG_ACCURATE,
	                        seek_pos);

	if (!ret) {
		g_message ("Cropping dialog seek failed");
	} else {
		/* wait either until the seek is done or 1/10th of a second has passed;
		 * slows seeking down technically, but gives better visual impression */
		gst_element_get_state (jcd->priv->pipeline, NULL, NULL, GST_SECOND / 10);
	}
}

/***************************************************************************
 *
 *   jcd_cropping_changed
 *
 ***************************************************************************/

static void
jcd_cropping_changed (ThJobCroppingDialog *jcd, GtkWidget *spin_button)
{
	gint v = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin_button));

	if (spin_button == jcd->priv->spin_top)
		jcd->priv->crop_top = v;

	if (spin_button == jcd->priv->spin_right)
		jcd->priv->crop_right = v;

	if (spin_button == jcd->priv->spin_bottom)
		jcd->priv->crop_bottom = v;

	if (spin_button == jcd->priv->spin_left)
		jcd->priv->crop_left = v;

	jcd_update_frame (jcd);
}

/***************************************************************************
 *
 *   jcd_error_msg_cb
 *
 ***************************************************************************/

static void
jcd_error_msg_cb (ThJobCroppingDialog *jcd, GstMessage *msg, GstBus *bus)
{
	GError *err = NULL;
	gchar *dbg = NULL;

	gst_message_parse_error (msg, &err, &dbg);

	g_print ("===== Cropping Dialog ERROR: %s [%s]\n", err->message,
	         (dbg) ? dbg : "no details");

	if (jcd->priv->error == NULL) {
		jcd->priv->error = err;
		gtk_dialog_response (GTK_DIALOG (jcd), TH_RESPONSE_RETRIEVE_ERROR);
	} else {
		g_error_free (err);
	}

	g_free (dbg);
}

/***************************************************************************
 *
 *   jcd_setup_with_dimensions
 *
 *   called when we get the first frame (and thus know the picture size)
 *
 ***************************************************************************/

static void
jcd_setup_with_dimensions (ThJobCroppingDialog *jcd)
{
	gint width, height;

	width = gdk_pixbuf_get_width (jcd->priv->cur_frame);
	height = gdk_pixbuf_get_height (jcd->priv->cur_frame);

	/* The remaining picture should be at least 16x16 in size */
	gtk_spin_button_set_range (GTK_SPIN_BUTTON (jcd->priv->spin_right),
	                           0.0, (width / 2) - 8.0);
	
	gtk_spin_button_set_range (GTK_SPIN_BUTTON (jcd->priv->spin_left),
	                           0.0, (width / 2) - 8.0);
	
	gtk_spin_button_set_range (GTK_SPIN_BUTTON (jcd->priv->spin_top),
	                           0.0, (height / 2) - 8.0);
	
	gtk_spin_button_set_range (GTK_SPIN_BUTTON (jcd->priv->spin_bottom),
	                           0.0, (height / 2) - 8.0);
}

/***************************************************************************
 *
 *   jcd_update_frame
 *
 ***************************************************************************/

static void
jcd_update_frame (ThJobCroppingDialog *jcd)
{
	GdkPixbuf *sub, *scaled;
	gdouble new_width;
	gint width, height;

	if (jcd->priv->cur_frame == NULL)
		return;

	width = gdk_pixbuf_get_width (jcd->priv->cur_frame);
	height = gdk_pixbuf_get_height (jcd->priv->cur_frame);

	/* do cropping */
	width -=  jcd->priv->crop_left + jcd->priv->crop_right;
	height -=  jcd->priv->crop_top + jcd->priv->crop_bottom;
	sub = gdk_pixbuf_new_subpixbuf (jcd->priv->cur_frame, jcd->priv->crop_left,
	                                jcd->priv->crop_top, width, height);

	width = gdk_pixbuf_get_width (sub);
	height = gdk_pixbuf_get_height (sub);

	/* scale to something smaller, and adjust for PAR */
	new_width = (gdouble) width * jcd->priv->par_n / (gdouble) jcd->priv->par_d;

	scaled = gdk_pixbuf_scale_simple (sub, (gint) new_width / 2, height / 2,
	                                  GDK_INTERP_BILINEAR);

	/* this should also trigger a redraw for us */
	gtk_image_set_from_pixbuf (GTK_IMAGE (jcd->priv->image), scaled);

	g_object_unref (scaled);
	g_object_unref (sub);
}

/***************************************************************************
 *
 *   jcd_element_msg_cb
 *
 ***************************************************************************/

static void
jcd_element_msg_cb (ThJobCroppingDialog *jcd, GstMessage *msg, GstBus *bus)
{
	gboolean first_frame;
	const GValue *val;

	if (msg->src == NULL || !TH_IS_PIXBUF_SINK (msg->src))
		return;

	val = gst_structure_get_value (msg->structure, "pixbuf");
	g_return_if_fail (val != NULL);

	if (jcd->priv->cur_frame) {
		first_frame = FALSE;
		g_object_unref (jcd->priv->cur_frame);
		jcd->priv->cur_frame = NULL;
	} else {
		first_frame = TRUE;
	}

	jcd->priv->cur_frame = GDK_PIXBUF (g_value_dup_object (val));

	if (!gst_structure_get_fraction (msg->structure, "pixel-aspect-ratio",
	    &jcd->priv->par_d, &jcd->priv->par_n)) {
		jcd->priv->par_d = 1;
		jcd->priv->par_n = 1;
	}

	if (first_frame) {
		jcd_setup_with_dimensions (jcd);
	}

	jcd_update_frame (jcd);
}

/***************************************************************************
 *
 *   jcd_setup_pipeline
 *
 ***************************************************************************/

static gboolean
jcd_setup_pipeline (ThJobCroppingDialog *jcd, GError **err)
{
	GstElement  *pipeline;
	GstBus      *bus;
	gchar       *desc, *device;
	guint        title_num;

	g_return_val_if_fail (TH_IS_JOB_CROPPING_DIALOG (jcd), FALSE);
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);
	g_return_val_if_fail (jcd->priv->pipeline == NULL, FALSE);

	g_object_get (jcd->priv->job, 
	              "device", &device, 
	              "title-num", &title_num, 
	              "title-length", &jcd->priv->job_len_secs,
	              NULL);

	desc = g_strdup_printf ("dvdreadsrc name=dvdreadsrc"
	                        " ! dvddemux"
	                        " ! mpeg2dec"
	                        " ! ffmpegcolorspace"
	                        " ! thpixbufsink");

	pipeline = gst_parse_launch (desc, err);
	
	g_free (desc);
	desc = NULL;
	
	if (err && *err)
	{
		if (pipeline)
			gst_object_unref (pipeline);
		g_free (device);
		return FALSE;
	} 
	
	if (pipeline == NULL)
	{
		g_free (device);
		return FALSE;
	}
	
	/* set location via property to avoid escaping issues when
	 * using a local directory path instead of a device */
	th_bin_set_child_properties (GST_BIN (pipeline), "dvdreadsrc",
	                             "device", device,
	                             "title", title_num + 1,
	                             NULL);

	bus = gst_pipeline_get_bus (GST_PIPELINE (pipeline));

	gst_bus_add_signal_watch (bus);

	g_signal_connect_swapped (bus, "message::error", 
	                          G_CALLBACK (jcd_error_msg_cb),
	                          jcd);

	/* we receive frames from the sink via element messages */
	g_signal_connect_swapped (bus, "message::element", 
	                          G_CALLBACK (jcd_element_msg_cb),
	                          jcd);
	
	jcd->priv->pipeline = pipeline;
	
	gst_object_unref (bus);
	g_free (device);

	return TRUE;
}

/***************************************************************************
 *
 *   th_job_cropping_dialog_run_for_job
 *
 *   Returns TRUE if 'OK' was pressed, FALSE otherwise.
 *
 ***************************************************************************/

gboolean
th_job_cropping_dialog_run_for_job (ThJob *job, GtkWindow *parent_window)
{
	ThJobCroppingDialog *jcd;
	GError              *err = NULL;
	gint                 ret;
	gint                 t, l, r, b;

	g_return_val_if_fail (TH_IS_JOB (job), FALSE);

	jcd = (ThJobCroppingDialog*) th_job_cropping_dialog_new ();

	jcd->priv->job = job;

	if (!jcd_setup_pipeline (jcd, &err))
		goto pipeline_setup_failed;

	if (parent_window)
	{
		gtk_window_set_transient_for (GTK_WINDOW (jcd), parent_window);
	}

	gtk_window_set_type_hint (GTK_WINDOW (jcd), GDK_WINDOW_TYPE_HINT_NORMAL);
	gtk_window_set_position (GTK_WINDOW (jcd), GTK_WIN_POS_CENTER);
	
	gtk_widget_show (jcd->priv->ok_button);

	g_object_get (jcd->priv->job, 
	              "crop-top", &t, "crop-left", &l, 
	              "crop-right", &r, "crop-bottom", &b, 
	              NULL);
	
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (jcd->priv->spin_top), t);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (jcd->priv->spin_left), l);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (jcd->priv->spin_right), r);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (jcd->priv->spin_bottom), b);

	gtk_range_set_range (GTK_RANGE (jcd->priv->slider), 0.0,
	                     0.99 * jcd->priv->job_len_secs);

	gtk_widget_set_sensitive (jcd->priv->slider, TRUE);
	
	/* make sure dialog is visible */
	gtk_window_present (GTK_WINDOW (jcd));

	/* start up pipeline (will be run in a background thread) */
	gst_element_set_state (jcd->priv->pipeline, GST_STATE_PAUSED);
	
	ret = gtk_dialog_run (GTK_DIALOG (jcd));
	
	gst_element_set_state (jcd->priv->pipeline, GST_STATE_NULL);
	gst_object_unref (jcd->priv->pipeline);
	jcd->priv->pipeline = NULL;

	if (ret == TH_RESPONSE_RETRIEVE_ERROR)
		goto error_running;

	if (ret == GTK_RESPONSE_ACCEPT)
	{
		gint t = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (jcd->priv->spin_top));
		gint l = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (jcd->priv->spin_left));
		gint r = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (jcd->priv->spin_right));
		gint b = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (jcd->priv->spin_bottom));

		g_object_set (jcd->priv->job, 
		              "crop-top", t, 
		              "crop-left", l, 
		              "crop-right", r, 
		              "crop-bottom", b, 
		              NULL);
	}

	gtk_widget_destroy (GTK_WIDGET (jcd));
	
	return (ret == GTK_RESPONSE_ACCEPT);

/* ERRORS */
pipeline_setup_failed:
	{
		GtkWidget *dlg;
		
		dlg = gtk_message_dialog_new (parent_window,
		                              GTK_DIALOG_MODAL,
		                              GTK_MESSAGE_ERROR,
		                              GTK_BUTTONS_OK,
		                              _("Failed to create pipeline.\n%s\n"),
		                              (err) ? err->message : _("unknown error"));
		
		(void) gtk_dialog_run (GTK_DIALOG (dlg));
		gtk_widget_destroy (dlg);
		gtk_widget_destroy (GTK_WIDGET (jcd));
		if (err)
			g_error_free (err);
		return FALSE;
	}

error_running:
	{
		GtkWidget *dlg;
		
		dlg = gtk_message_dialog_new (parent_window,
		                              GTK_DIALOG_MODAL,
		                              GTK_MESSAGE_ERROR,
		                              GTK_BUTTONS_OK,
		                              _("Error:\n%s\n"),
		                              (jcd->priv->error) ?
		                                jcd->priv->error->message
		                               : _("unknown error"));
		
		(void) gtk_dialog_run (GTK_DIALOG (dlg));
		gtk_widget_destroy (dlg);
		gtk_widget_destroy (GTK_WIDGET (jcd));
		return FALSE;
	}
}
