/***************************************************************************
                             th-power-manager-agent.c
                             ------------------------
    begin                : Thu Jul 13 2006
    copyright            : (C) 2006 by Tim-Philipp Müller
    email                : tim centricular net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/* FIXME:
 *  - there is no way to query the current status of the inhibition
 *    or the validity of the cookie, is there? How do we make this
 *    stuff work through a DBUS daemon or gnome-power-manager restart?
 *
 *  - does it make sense to query for 'LowPowerMode' (g-p-m >= 2.15.x)
 *    and pause ourselves when that is the on? (does this actually indicate
 *    that we're almost running out of battery, or something else?)
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "th-power-manager-agent.h"
#include "th-utils.h"

#include <dbus/dbus-glib.h>
#include <glib/gi18n.h>

enum
{
  PROP_INHIBIT_POWER_MANAGER = 1
};

#define	GNOME_POWER_MANAGER_DBUS_SERVICE   "org.gnome.PowerManager"
#define	GNOME_POWER_MANAGER_DBUS_OBJ_PATH  "/org/gnome/PowerManager"
#define	GNOME_POWER_MANAGER_DBUS_INTERFACE "org.gnome.PowerManager"

/* every 60 seconds */
#define TH_PMA_TIMEOUT (60 * 1000)

struct _ThPowerManagerAgentPrivate {
  gboolean         inhibited;
  guint            cookie;

  gboolean         low_power_mode;

  gulong           timeout_id;
};

G_DEFINE_TYPE (ThPowerManagerAgent, th_power_manager_agent, G_TYPE_OBJECT);

static gboolean th_power_manager_agent_update (gpointer user_data);
static gboolean th_power_manager_agent_ensure_dbus (ThPowerManagerAgent *agent,
    DBusGProxy ** p_proxy);
static void th_power_manager_agent_get_property (GObject *obj, guint param_id,
    GValue *val, GParamSpec *pspec);
static void th_power_manager_agent_set_property (GObject *obj, guint param_id,
    const GValue *val, GParamSpec *pspec);
static void th_power_manager_agent_set_inhibited (ThPowerManagerAgent * agent,
    gboolean do_inhibit);

#define TH_POWER_MANAGER_GET_PRIVATE(a) \
  G_TYPE_INSTANCE_GET_PRIVATE (a, TH_TYPE_POWER_MANAGER_AGENT, ThPowerManagerAgentPrivate);

static void
th_power_manager_agent_init (ThPowerManagerAgent * agent)
{
  agent->priv = TH_POWER_MANAGER_GET_PRIVATE (agent);

  agent->priv->timeout_id = g_timeout_add (TH_PMA_TIMEOUT,
                                           th_power_manager_agent_update,
                                           agent);
}

static void
th_power_manager_agent_finalize (GObject *object)
{
  ThPowerManagerAgent *agent = TH_POWER_MANAGER_AGENT (object);

  if (agent->priv->timeout_id > 0)
    g_source_remove (agent->priv->timeout_id);

  if (agent->priv->inhibited && agent->priv->cookie != 0)
    th_power_manager_agent_set_inhibited (agent, FALSE);

  G_OBJECT_CLASS (th_power_manager_agent_parent_class)->finalize (object);
}

static void
th_power_manager_agent_class_init (ThPowerManagerAgentClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (klass, sizeof (ThPowerManagerAgentPrivate));

  G_OBJECT_CLASS (klass)->finalize = th_power_manager_agent_finalize;
  G_OBJECT_CLASS (klass)->get_property = th_power_manager_agent_get_property;
  G_OBJECT_CLASS (klass)->set_property = th_power_manager_agent_set_property;

#if 0
  g_object_class_install_property (gobject_class, PROP_LOW_POWER_MODE,
      g_param_spec_boolean ("low-power-mode", "Low Power Mode",
          "Whether the computer is in low power mode (ie. we should pause)",
          FALSE, G_PARAM_READABLE));
#endif

  g_object_class_install_property (gobject_class, PROP_INHIBIT_POWER_MANAGER,
      g_param_spec_boolean ("inhibit-power-manager", "Inhibit Power Manager",
          "Prevent power manager from automatically hibernating/suspending",
          FALSE, G_PARAM_READWRITE));
}

ThPowerManagerAgent *
th_power_manager_agent_new (void)
{
  GObject *obj;

  obj = g_object_new (TH_TYPE_POWER_MANAGER_AGENT, NULL);

  return TH_POWER_MANAGER_AGENT (obj);
}

static gboolean
th_power_manager_agent_ensure_dbus (ThPowerManagerAgent  * agent,
                                    DBusGProxy          ** p_proxy)
{
  DBusGConnection *conn;
  GError *err = NULL;

  conn = dbus_g_bus_get (DBUS_BUS_SESSION, &err);
  if (err != NULL) {
    th_log ("Cannot connect to DBUS daemon: %s", err->message); /* FIXME */
    g_error_free (err);
    return FALSE;
  }

  if (conn == NULL) {
    th_log ("Cannot connect to DBUS daemon: %s", "unknown error"); /* FIXME */
    return FALSE;
  }

  /* get the gnome-power-manager proxy */
  *p_proxy = dbus_g_proxy_new_for_name (conn, GNOME_POWER_MANAGER_DBUS_SERVICE,
      GNOME_POWER_MANAGER_DBUS_OBJ_PATH, GNOME_POWER_MANAGER_DBUS_INTERFACE);

  return (*p_proxy != NULL);
}

#if 0
static void
th_power_manager_agent_check_low_power_mode (ThPowerManagerAgent *agent)
{
  DBusGProxy *proxy = NULL;
  gboolean ret, low_power_mode;
  GError *err = NULL;

  if (!th_power_manager_agent_ensure_dbus (agent, &proxy))
    return;

  ret = dbus_g_proxy_call (proxy, "GetLowPowerMode", &err, G_TYPE_INVALID,
      G_TYPE_BOOLEAN, &low_power_mode, G_TYPE_INVALID);

  g_object_unref (proxy);

  /* fail silently, this method was only introduced
   * recently in gnome-power-manager >= 2.15.x */
  if (!ret) {
    g_error_free (err);
    return;
  }

  /* has it changed? */
  if (low_power_mode != agent->priv->low_power_mode) {
    agent->priv->low_power_mode = low_power_mode;
    g_object_notify (G_OBJECT (agent), "low-power-mode");
  }
}
#endif

static gboolean
th_power_manager_agent_update (gpointer user_data)
{
  ThPowerManagerAgent *agent;

  agent = TH_POWER_MANAGER_AGENT (user_data);

  /* th_power_manager_agent_check_low_power_mode (agent); */

  return TRUE; /* call again */
}

static void
th_power_manager_agent_set_inhibited (ThPowerManagerAgent * agent,
                                      gboolean              do_inhibit)
{
  DBusGProxy *proxy = NULL;
  gboolean ret;
  GError *err = NULL;

  if (!th_power_manager_agent_ensure_dbus (agent, &proxy))
    return;

  if (do_inhibit == agent->priv->inhibited) {
    g_return_if_fail (!do_inhibit || agent->priv->inhibited > 0);
    return;
  }

  if (do_inhibit) {
    ret = dbus_g_proxy_call (proxy, "Inhibit", &err,
        G_TYPE_STRING, "Thoggen",
        G_TYPE_STRING, _("DVD ripping in progress"),
        G_TYPE_INVALID, G_TYPE_UINT, &agent->priv->cookie, G_TYPE_INVALID);

    if (err && err->code == DBUS_GERROR_UNKNOWN_METHOD) {
      g_error_free (err);
      err = NULL;

      /* try old gnome-power-manager 2.14.x interface */
      ret = dbus_g_proxy_call (proxy, "InhibitInactiveSleep", &err,
          G_TYPE_STRING, "Thoggen",
          G_TYPE_STRING, _("DVD ripping in progress"),
          G_TYPE_INVALID, G_TYPE_UINT, &agent->priv->cookie, G_TYPE_INVALID);
    }

    if (ret) {
      agent->priv->inhibited = TRUE;
    }
  } else {
    ret = dbus_g_proxy_call (proxy, "UnInhibit", &err,
        G_TYPE_UINT, agent->priv->cookie, G_TYPE_INVALID, G_TYPE_INVALID);

    if (err && err->code == DBUS_GERROR_UNKNOWN_METHOD) {
      g_error_free (err);
      err = NULL;

      /* try old gnome-power-manager 2.14.x interface */
      ret = dbus_g_proxy_call (proxy, "AllowInactiveSleep", &err,
        G_TYPE_UINT, agent->priv->cookie, G_TYPE_INVALID, G_TYPE_INVALID);
    }

    if (ret) {
      agent->priv->cookie = 0;
      agent->priv->inhibited = FALSE;
    }
  }

  g_object_unref (proxy);

  if (!ret) {
    th_log ("GnomePowerManager::%s failed: %s (cookie now %u)",
        (do_inhibit) ? "Inhibit" : "UnInhibit",
        (err) ? err->message : "???", agent->priv->cookie);
    g_error_free (err);
  } else {
    th_log ("GnomePowerManager::%s succeeded (cookie now %u)",
        (do_inhibit) ? "Inhibit" :  "UnInhibit", agent->priv->cookie);
  }
}

static void
th_power_manager_agent_get_property (GObject *obj, guint param_id,
    GValue *val, GParamSpec *pspec)
{
  ThPowerManagerAgent *agent = TH_POWER_MANAGER_AGENT (obj);

  switch (param_id) {
    case PROP_INHIBIT_POWER_MANAGER:
      g_value_set_boolean (val, agent->priv->inhibited);
      break;
    default: {
      G_OBJECT_WARN_INVALID_PROPERTY_ID (obj, param_id, pspec);
      break;
    }
  }
}

static void
th_power_manager_agent_set_property (GObject *obj, guint param_id,
    const GValue *val, GParamSpec *pspec)
{
  ThPowerManagerAgent *agent = TH_POWER_MANAGER_AGENT (obj);

  switch (param_id) {
    case PROP_INHIBIT_POWER_MANAGER: {
      gboolean do_inhibit;

      do_inhibit = g_value_get_boolean (val);
      th_log ("%sinhibiting power manager", (do_inhibit) ? "" : "un");
      th_power_manager_agent_set_inhibited (agent, do_inhibit);      
      break;
    }
    default: {
      G_OBJECT_WARN_INVALID_PROPERTY_ID (obj, param_id, pspec);
      break;
    }
  }
}
