/********************************************************************
 * config.c: Anything that deals with reading the config file
 *
 * This work is licensed under a Creative Commons Attribution-Share Alike 3.0
 * United States License. See http://creativecommons.org/licenses/by-sa/3.0/us/
 * for details.
 *
 * This file is part of thinkfan. See thinkfan.c for further info.
 * ******************************************************************/
#include "config.h"
#include <syslog.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include "message.h"
#include "system.h"

/***********************************************************************
 * readconfig(char *fname) reads the config file and
 * returns a pointer to a struct tf_config. Returns NULL if there's any
 * problem with the config.
 * Non-matching lines are skipped.
 **********************************************************************/
tf_config *readconfig(char* fname) {
	FILE *cfg_file;
	int i=0, line_count=0, sensor_count=0, max_lvl=0;
	ssize_t line_len;
	size_t ll;
	tf_config *cfg_local;
	char *line=NULL;
	char buf[1024];

	prefix = "\n";

	cfg_local = (tf_config *) malloc(sizeof(tf_config));
	cfg_local->items = (thm_tuple *) malloc(sizeof(thm_tuple));
	cfg_local->sensors = NULL;
	cfg_local->fan = NULL;
	cfg_local->sensor_count = 0;
	if ((cfg_file = fopen(fname, "r")) == NULL) {
		showerr(fname);
		goto end;
	}
	while ((line_len = getline (&line, &ll, cfg_file)) >= 0) {
		line_count++;
		if (sscanf(line, " sensor %1023c", buf) == 1) {
			cfg_local->sensors = (char **) realloc(
					cfg_local->sensors, (sensor_count+2) * sizeof(char *));
			buf[strcspn(buf, "\n")] = 0;
			cfg_local->sensors[sensor_count] = (char *) malloc(
					(strlen(buf)+1) * sizeof(char));
			strcpy(cfg_local->sensors[sensor_count], buf);
			cfg_local->sensor_count = ++sensor_count;
			cfg_local->sensors[sensor_count] = NULL;
			if (cfg_local->sensor_count > 1 && !strcmp(buf, IBM_TEMP)) {
				message(LOG_ERR, MSG_ERR_CONF_MIX(fname, line_count, line));
				goto end;
			}
		}
		else if (sscanf(line, " fan %1023c", buf) == 1) {
			buf[strcspn(buf, "\n")] = 0;
			if (cfg_local->fan == NULL) {
				cfg_local->fan = (char *) malloc((strlen(buf)+1) * sizeof(char));
				strcpy(cfg_local->fan, buf);
			}
			else {
				message(LOG_WARNING, MSG_ERR_CONF_FAN(fname, line_count, line));
				goto end;
			}
		}
		else if (sscanf(line, " ( %d , %d , %d ) ",
		 &cfg_local->items[i].level, &cfg_local->items[i].low,
		 &cfg_local->items[i].high) == 3) {
			if (cfg_local->items[i].level > max_lvl)
				max_lvl = cfg_local->items[i].level;
			if ((cfg_local->items = realloc(cfg_local->items,
			 (i+2) * sizeof(thm_tuple))) == NULL) {
				showerr("Allocating memory for config");
				goto end;
			}
			// check for correct ordering of fan levels in config
			if ((i >= 2) && (cfg_local->items[i].level <=
			 cfg_local->items[i-1].level)) {
				if (nodaemon && chk_sanity && !quiet)
					fprintf(stderr, MSG_ERR_CONF_ORDER(fname, line_count, line));
				if (!nodaemon && !quiet)
					syslog(LOG_ERR, MSG_ERR_CONF_ORDER(fname, line_count, line));
				if (chk_sanity) goto end;
			}
			if (cfg_local->items[i].high <= cfg_local->items[i].low) {
				if (nodaemon && chk_sanity && !quiet)
					fprintf(stderr, MSG_ERR_CONF_LOWHIGH(fname, line_count, line));
				if (!nodaemon && !quiet)
					syslog(LOG_ERR, MSG_ERR_CONF_LOWHIGH(fname, line_count, line));
				if (chk_sanity)
					goto end;
			}
			cfg_local->max_idx = i;
			i++;
		}
		free(line);
		line = NULL;
	}
	free(line);
	line = NULL;
	if (i <= 0) {
		message(LOG_ERR, MSG_ERR_CONF_PARSE);
		goto end;
	}
	fclose(cfg_file);

	// a sysfs sensor was specified in the config file
	if (cfg_local->fan != NULL && strcmp(cfg_local->fan, IBM_FAN)) {
		if (resume_is_safe) {
			cfg_local->setfan = setfan_sysfs;
		}
		else {
			cfg_local->setfan = setfan_sysfs_safe;
			message(LOG_WARNING, MSG_WRN_SYSFS_SAFE);
		}
		cfg_local->init_fan = init_fan_sysfs_once;
		cfg_local->uninit_fan = uninit_fan_sysfs;
	}
	else {
		if (cfg_local->fan == NULL) {
			cfg_local->fan = (char *) calloc(strlen(IBM_FAN)+1, sizeof(char));
			strcpy(cfg_local->fan, IBM_FAN);
		}
		cfg_local->setfan = setfan_ibm;
		cfg_local->init_fan = init_fan_ibm;
		cfg_local->uninit_fan = uninit_fan_ibm;
	}

	if (cfg_local->sensor_count > 0 &&
	 strcmp(cfg_local->sensors[cfg_local->sensor_count - 1], IBM_TEMP)) {
		if (depulse) cfg_local->get_temp = depulse_and_get_temp_sysfs;
		else cfg_local->get_temp = get_temp_sysfs;
	}
	else {
		if (depulse) cfg_local->get_temp = depulse_and_get_temp_ibm;
		else cfg_local->get_temp = get_temp_ibm;
	}

	if (cfg_local->fan != NULL && strcmp(cfg_local->fan, IBM_FAN)
	 && max_lvl <= 7) {
		message(LOG_WARNING, MSG_WRN_FANLVL(max_lvl));
		if (chk_sanity) {
			message(LOG_INFO, MSG_INF_SANITY);
			goto end;
		}
		else message(LOG_INFO, MSG_INF_INSANITY);
	}

	return cfg_local;

end:
	if (cfg_file) fclose(cfg_file);
	free(line);
	free_config(cfg_local);
	return NULL;
}

void free_config(tf_config *cfg) {
	free(cfg->fan);
	if (cfg->sensor_count > 0) {
		int j;
		for (j=0; j < cfg->sensor_count; j++)
			free(cfg->sensors[j]);
		free(cfg->sensors);
	}
	free(cfg->items);
	free(cfg);
}

