/*

    File: photorec.c

    Copyright (C) 1998-2005 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#include <stdarg.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>	/* geteuid */
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_TIME_H
#include <time.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#include <ctype.h>      /* toupper, tolower */
#ifdef HAVE_LOCALE_H
#include <locale.h>	/* setlocale */
#endif
#ifdef HAVE_SIGNAL
#include <signal.h>
#endif
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif
#include <errno.h>
#include "types.h"
#include "common.h"
#include "lang.h"
#include "intrf.h"
#include "godmode.h"
#include "fnctdsk.h"
#include "photorec.h"

extern const t_arch_fnct arch_i386;
extern const t_arch_fnct arch_sun;
extern const t_arch_fnct arch_none;
#define INTER_SELECT_X	0
#define INTER_SELECT_Y	23
#define INTER_SELECT	15
#define MAX_FILES_PER_DIR	500
typedef struct struct_alloc_data t_alloc_data;

struct struct_alloc_data
{
  uint64_t start;
  uint64_t end;
  t_alloc_data *prev;
  t_alloc_data *next;
  t_file_stat *file_stat;
};

extern const t_file_hint file_hint_avi;
extern const t_file_hint file_hint_au;
extern const t_file_hint file_hint_bmp;
extern const t_file_hint file_hint_bz2;
extern const t_file_hint file_hint_c;
extern const t_file_hint file_hint_crw;
extern const t_file_hint file_hint_ctg;
extern const t_file_hint file_hint_dir;
extern const t_file_hint file_hint_doc;
extern const t_file_hint file_hint_dsc;
extern const t_file_hint file_hint_html;
extern const t_file_hint file_hint_jpg;
extern const t_file_hint file_hint_mov;
extern const t_file_hint file_hint_mp3;
extern const t_file_hint file_hint_mpg;
extern const t_file_hint file_hint_mrw;
extern const t_file_hint file_hint_orf;
extern const t_file_hint file_hint_pdf;
extern const t_file_hint file_hint_php;
extern const t_file_hint file_hint_pl;
extern const t_file_hint file_hint_png;
extern const t_file_hint file_hint_raf;
extern const t_file_hint file_hint_raw;
extern const t_file_hint file_hint_rdc;
extern const t_file_hint file_hint_rtf;
extern const t_file_hint file_hint_sh;
extern const t_file_hint file_hint_tar;
extern const t_file_hint file_hint_tiff;
extern const t_file_hint file_hint_wma;
extern const t_file_hint file_hint_x3f;
extern const t_file_hint file_hint_zip;


static void reset_file_stats(t_file_stat *file_stats);
static void reset_file_recovery(t_file_recovery *file_recovery);
static void copy_file_recovery(t_file_recovery *file_recovery, const t_file_recovery *file_recovery_org);
static void write_stats_log(const t_file_stat *file_stats);
static void write_stats_stdout(const t_file_stat *file_stats);
static void update_stats(t_file_stat *file_stats, t_alloc_data *list_free_space);
static void interface_select(t_param_disk *disk_car, const int debug,const int paranoid, const char *recup_dir, const int keep_corrupted_file, t_file_stat *file_stats);
static void interface_file_select(t_file_stat *file_stats);
static void interface_options_photorec(int *paranoid, int *allow_partial_last_cylinder,int *halt_on_errors, const t_arch_fnct **arch, int *keep_corrupted_file);
static int do_curses_photorec(int debug, const char *recup_dir, const t_list_disk *list_disk, const t_arch_fnct **arch, t_file_stat *file_stats);
static int photorec(t_param_disk *disk_car, t_partition *partition, const int debug, const int paranoid, const char *recup_dir, const int keep_corrupted_file, const int interface, t_file_stat *file_stats);
static t_partition *new_whole_disk(const t_param_disk *disk_car);

static FILE* init_log(const char*filename,int argc, char**argv);

static FILE *f_rapport=NULL;
static int f_status=0;
static unsigned int find_blocksize(t_alloc_data *list_file, const unsigned int default_blocksize);
static t_alloc_data * update_blocksize(unsigned int blocksize, t_alloc_data *list_free_space);
static t_alloc_data *update_free_space(const char *filename_out,const t_file_recovery *file_recovery, t_alloc_data *list_free_space, t_alloc_data **new_current_free_space, const unsigned int blocksize, const unsigned int sector_size);
static unsigned int photorec_mkdir(const char *recup_dir, const unsigned int initial_dir_num);

void ecrit_rapport_string(const char *string,const int max_length)
{
  int i;
  for(i=0;(string[i]!='\0')&&(i<max_length);i++)
    ecrit_rapport("%c",string[i]);
}

int ecrit_rapport(const char *_format, ...)
{
  int res=0;
  if(f_rapport!=NULL)
  {
    va_list ap;
    fflush(stderr);
    va_start(ap,_format);
    res=vfprintf(f_rapport,_format,ap);
    va_end(ap);
    if(res<0)
    {
      f_status=1;
    }
    if(fflush(f_rapport))
    {
      f_status=1;
    }
  }
  return res;
}

static FILE* init_log(const char*filename,int argc, char**argv)
{
  FILE*f_file=fopen(filename,"a");
  if(f_file==NULL)
  {
    printf("\nCan not create photorec.log: %s\n",strerror(errno));
    exit(1);
  }
  else
  {
	int i;
	time_t my_time;
	my_time=time(NULL);
	fprintf(f_file,"\n\n%s",ctime(&my_time));
	fprintf(f_file,"PhotoRec command line :");
	for(i=1;i<argc;i++)
	  fprintf(f_file," %s", argv[i]);
	fprintf(f_file,"\n");
	fflush(f_file);
#ifdef HAVE_DUP2
	dup2(fileno(f_file),2);
#endif
  }
  return f_file;
}

#ifdef HAVE_SIGACTION
void sighup_hdlr(int shup)
{
  ecrit_rapport("SIGHUP detected! PhotoRec has been killed.\n");
  if(f_rapport!=NULL)
  {
    fclose(f_rapport);
  }
  exit(1);
}
#endif
static t_alloc_data *update_free_space(const char *filename_out,const t_file_recovery *file_recovery, t_alloc_data *list_free_space, t_alloc_data **new_current_free_space, const unsigned int blocksize, const unsigned int sector_size)
{
  t_alloc_data *current_free_space;
  uint64_t file_size;
  file_size=(file_recovery->file_size+blocksize-1)/blocksize*blocksize;
  for(current_free_space=list_free_space;current_free_space!=NULL && (current_free_space->next!=NULL && current_free_space->next->start<=file_recovery->start);current_free_space=current_free_space->next);
  if(current_free_space->start<file_recovery->start)
  {
    t_alloc_data *new_free_space;
    new_free_space=(t_alloc_data*)MALLOC(sizeof(*new_free_space));
    new_free_space->start=file_recovery->start;
    new_free_space->end=current_free_space->end;
    new_free_space->prev=current_free_space;
    new_free_space->next=current_free_space->next;
    new_free_space->file_stat=NULL;
    current_free_space->end=file_recovery->start-1;
    current_free_space->next=new_free_space;
    current_free_space=new_free_space;
  }
  if(current_free_space!=NULL && current_free_space->start==file_recovery->start)
  {
    current_free_space->file_stat=file_recovery->file_stat;
  }
  if(filename_out!=NULL && file_recovery->file_size>0)
  {
    ecrit_rapport("%s",filename_out);
  }
  while(current_free_space!=NULL && current_free_space->end-current_free_space->start+1<=file_size)
  {
    t_alloc_data *next;
    next=current_free_space->next;
    file_size-=current_free_space->end-current_free_space->start+1;
    ecrit_rapport(" %lu-%lu",(long unsigned)(current_free_space->start/sector_size),
	(long unsigned)(current_free_space->end/sector_size));
    if(file_recovery->file_size>0)
    {
      if(current_free_space->prev!=NULL)
	current_free_space->prev->next=current_free_space->next;
      if(current_free_space->next!=NULL)
	current_free_space->next->prev=current_free_space->prev;
      if(list_free_space==current_free_space)
	list_free_space=next;
      FREE(current_free_space);
    }
    current_free_space=next;
  }
  if(current_free_space!=NULL && file_size>0)
  {
    ecrit_rapport(" %lu-%lu",(long unsigned)(current_free_space->start/sector_size),
	(long unsigned)((current_free_space->start+file_size-1)/sector_size));
    if(file_recovery->file_size>0)
    {
      current_free_space->start+=file_size;
      current_free_space->file_stat=NULL;
    }
  }
  if(file_recovery->file_size>0)
  {
    *new_current_free_space=current_free_space;
    ecrit_rapport("\n");
  }
  return list_free_space;
}

static unsigned int menu_choose_blocksize(unsigned int blocksize, const unsigned int sector_size)
{
  /* TODO: let the user choose the blocksize */
  /* return sector_size; */
  return blocksize;
}

static t_alloc_data *init_list_free_space(const t_partition *partition)
{
  t_alloc_data *list_free_space;
  list_free_space=(t_alloc_data*)MALLOC(sizeof(*list_free_space));
  list_free_space->start=partition->part_offset;
  list_free_space->end=partition->part_offset+partition->part_size-1;
  list_free_space->file_stat=NULL;
  list_free_space->prev=NULL;
  list_free_space->next=NULL;
  return list_free_space;
}

static void reset_file_stats(t_file_stat *file_stats)
{
  unsigned int i;
  for(i=0;file_stats[i].file_hint!=NULL;i++)
  {
    file_stats[i].not_recovered=0;
    file_stats[i].recovered=0;
  }
}

static void reset_file_recovery(t_file_recovery *file_recovery)
{
  file_recovery->filename[0]='\0';
  file_recovery->file_stat=NULL;
  file_recovery->handle=NULL;
  file_recovery->header_distance=0;
  file_recovery->file_size=0;
  file_recovery->start=0;
}

static void copy_file_recovery(t_file_recovery *file_recovery_dst, const t_file_recovery *file_recovery_src)
{
  memcpy(file_recovery_dst->filename,file_recovery_src->filename,sizeof(file_recovery_dst->filename));
  file_recovery_dst->file_stat=file_recovery_src->file_stat;
  file_recovery_dst->handle=file_recovery_src->handle;
  file_recovery_dst->header_distance=file_recovery_src->header_distance;
  file_recovery_dst->file_size=file_recovery_src->file_size;
  file_recovery_dst->start=file_recovery_src->start;
}

static unsigned int photorec_mkdir(const char *recup_dir, const unsigned int initial_dir_num)
{
  char working_recup_dir[2048];
  int dir_ok=0;
  int dir_num=initial_dir_num;
  do
  {
    snprintf(working_recup_dir,sizeof(working_recup_dir)-1,"%s.%d",recup_dir,dir_num);
#ifdef HAVE_MKDIR
#ifdef __MINGW32__
    if(mkdir(working_recup_dir)!=0 && errno==EEXIST)
#else
      if(mkdir(working_recup_dir, 0700)!=0 && errno==EEXIST)
#endif
#else
#error You need a mkdir function!
#endif
      {
	dir_num++;
      }
      else
      {
	dir_ok=1;
      }
  } while(dir_ok==0);
  return dir_num;
}

static int photorec(t_param_disk *disk_car, t_partition *partition, const int debug, const int paranoid, const char *recup_dir, const int keep_corrupted_file, const int interface, t_file_stat *file_stats)
{
  time_t real_start_time;
  t_alloc_data *list_free_space;
  t_alloc_data *current_free_space;
  t_alloc_data *list_file=NULL;
  unsigned int file_nbr=0;
  unsigned int dir_num=1;
  int ind_stop=0;
  /* 0: run
     1: stop by user request
     2: run and keep_broken_file
     3: end
   */
  unsigned int pass;
  unsigned int blocksize=disk_car->sector_size;
  aff_buffer(BUFFER_RESET,"Q");
  reset_file_stats(file_stats);
  if(interface)
  {
    aff_copy(stdscr);
    wmove(stdscr,4,0);
    wdoprintf(stdscr,"%s",disk_car->description(disk_car));
    mvwaddstr(stdscr,5,0,msg_PART_HEADER_LONG);
    wmove(stdscr,6,0);
    aff_part(stdscr,AFF_PART_ORDER,disk_car,partition);
    wrefresh(stdscr);
    wmove(stdscr,22,0);
    wattrset(stdscr,A_STANDOUT);
    waddstr(stdscr,"  Stop  ");
    wattroff(stdscr,A_STANDOUT);
  }
  ecrit_rapport("\nAnalyse ");
  aff_part_rapport(disk_car,partition);
  if(debug>0)
  {
    ecrit_rapport("debug=%u, paranoid=%u\n",debug,paranoid);
  }
  list_free_space=init_list_free_space(partition);
  real_start_time=time(NULL);
  dir_num=photorec_mkdir(recup_dir,dir_num);
  for(pass=1;ind_stop==0 || ind_stop==2;pass++)
  {
    uint64_t offset;
    unsigned int old_file_nbr=file_nbr;
    unsigned char double_buffer[2*blocksize];
    unsigned char *buffer=&double_buffer[blocksize];
    time_t start_time;
    time_t previous_time;
    t_file_recovery file_recovery;
    reset_file_recovery(&file_recovery);

    start_time=time(NULL);
    previous_time=start_time;
    memset(double_buffer,0,2*blocksize);
    ecrit_rapport("Pass %u (blocksize=%u)\n",pass,blocksize);
    if(debug>1)
    {
      t_alloc_data *cur_free_space;
      ecrit_rapport("Explore ");
      for(cur_free_space=list_free_space;cur_free_space!=NULL;cur_free_space=cur_free_space->next)
	ecrit_rapport(" %lu-%lu",(long unsigned)(cur_free_space->start/disk_car->sector_size),
	    (long unsigned)(cur_free_space->end/disk_car->sector_size));
      ecrit_rapport("\n");
    }
    {
      for(current_free_space=list_free_space;current_free_space!=NULL && (ind_stop==0 || ind_stop==2);)
      {
	for(offset=current_free_space->start;current_free_space!=NULL && (offset<current_free_space->end) && (ind_stop==0 || ind_stop==2) &&
	    ((pass==1 && file_nbr<10)||(pass==2)||(pass>2 && (file_recovery.handle!=NULL || current_free_space->file_stat!=NULL)));
	    offset+=blocksize)
	{
	  t_file_recovery file_recovery_old;
	  reset_file_recovery(&file_recovery_old);
	  if(debug>1)
	  {
	    ecrit_rapport("Reading sector %10lu/%lu\n",(unsigned long)((offset-partition->part_offset)/disk_car->sector_size),(unsigned long)(partition->part_size/disk_car->sector_size));
	  }
	  if(disk_car->read(disk_car,blocksize, buffer, offset)!=0)
	  { /* Read error */
	    memset(buffer,0,blocksize);
	  }
	  {
	    if(interface!=0 && offset%(123*0x200)==0 && offset!=partition->part_offset)
	    {
	      time_t current_time;
	      current_time=time(NULL);
	      if(current_time>previous_time)
	      {
		previous_time=current_time;
		wmove(stdscr,9,0);
		wclrtoeol(stdscr);
		wdoprintf(stdscr,"Reading sector %10lu/%lu, %u pictures or videos found\n",(unsigned long)((offset-partition->part_offset)/disk_car->sector_size),(unsigned long)(partition->part_size/disk_car->sector_size),file_nbr);
		wmove(stdscr,10,0);
		wclrtoeol(stdscr);
		wdoprintf(stdscr,"Elapsed time %uh%02um%02us - Estimated time for achievement %uh%02um%02u\n",
		    (unsigned)((current_time-real_start_time)/60/60),
		    (unsigned)((current_time-real_start_time)/60%60),
		    (unsigned)((current_time-real_start_time)%60),
		    (unsigned)((partition->part_offset+partition->part_size-1-offset)*(current_time-real_start_time)/(offset-partition->part_offset)/3600),
		    (unsigned)(((partition->part_offset+partition->part_size-1-offset)*(current_time-real_start_time)/(offset-partition->part_offset)/60)%60),
		    (unsigned)((partition->part_offset+partition->part_size-1-offset)*(current_time-real_start_time)/(offset-partition->part_offset))%60);
		wrefresh(stdscr);
		switch(wgetch_nodelay(stdscr))
		{
		  case KEY_ENTER:
#ifdef PADENTER
		  case PADENTER:
#endif
		  case '\n':
		  case '\r':
		  case 's':
		  case 'S':
		    ind_stop=1;
		    break;
		}
	      }
	    }
	    if(file_recovery.file_stat!=NULL &&
	      file_recovery.file_stat->file_hint->min_header_distance > 0 &&
	     file_recovery.header_distance<=file_recovery.file_stat->file_hint->min_header_distance)
	    {
	      /* if stream, only check against my own header, otherwise no check */
	      if(file_recovery.file_stat->file_hint->stream>0 &&
		file_recovery.file_stat->file_hint->header_check(buffer,blocksize)!=0)
	      {
		file_recovery.header_distance=0;
	      }
	      else
	      {
		if(debug>1)
		{
		  ecrit_rapport("Don't check header, too near from the beginning of a previous header, sector %lu.\n",
			(unsigned long)((offset-partition->part_offset)/disk_car->sector_size));
		}
	      }
	    }
	    else if(file_recovery.file_stat!=NULL && file_recovery.file_stat->file_hint==&file_hint_tar &&
	      file_recovery.file_stat->file_hint->header_check(double_buffer+blocksize-0x200,blocksize)!=0)
	    {
		if(debug>1)
		{
		  ecrit_rapport("Currently saving a tar file, sector %lu.\n",
			(unsigned long)((offset-partition->part_offset)/disk_car->sector_size));
		}
	    }
	    else
	    {
	      unsigned int i;
	      for(i=0;file_stats[i].file_hint!=NULL && ( file_stats[i].enable==0 ||
		file_stats[i].file_hint->header_check(buffer,blocksize)==0);i++);
	      if(file_stats[i].file_hint!=NULL)
	      {
		/* ecrit_rapport("%s Ok\n",file_stats[i].file_hint->description); */
		if(file_recovery.file_stat!=NULL && file_recovery.file_stat->file_hint==file_stats[i].file_hint && file_recovery.file_stat->file_hint->stream>0)
		{
		  file_recovery.header_distance=0;
		  /*
		     if(debug>1)
		     ecrit_rapport("same header, streaming file: already saving this file\n");
		   */
		}
		else
		{
		  copy_file_recovery(&file_recovery_old,&file_recovery);

		  reset_file_recovery(&file_recovery);
		  file_recovery.file_stat=&file_stats[i];
		  file_recovery.start=offset;
		  if(debug>0)
		  {
		    ecrit_rapport("%s header found at sector %lu\n",file_recovery.file_stat->file_hint->extension,
			(unsigned long)((offset-partition->part_offset)/disk_car->sector_size));
		  }
		  if(pass==1)
		  {
		    /* Backup file offset */
		    t_alloc_data *new_file_alloc;
		    new_file_alloc=(t_alloc_data*)MALLOC(sizeof(*new_file_alloc));
		    new_file_alloc->start=file_recovery.start;
		    new_file_alloc->end=0;
		    new_file_alloc->prev=NULL;
		    new_file_alloc->next=list_file;
		    if(list_file!=NULL)
		      list_file->prev=new_file_alloc;
		    list_file=new_file_alloc;
		  }
		}
	      }
	    }
	    if(file_recovery.file_stat!=NULL && file_recovery.file_stat->file_hint->max_filesize>0 && file_recovery.file_size>=file_recovery.file_stat->file_hint->max_filesize)
	    {
	      ecrit_rapport("File should not be bigger than %llu, stop adding data\n",
		  (long long unsigned)file_recovery.file_stat->file_hint->max_filesize);
	      copy_file_recovery(&file_recovery_old,&file_recovery);
	      reset_file_recovery(&file_recovery);
	      file_recovery.start=offset;
	    }
	    if(file_recovery_old.handle)
	    {
	      if(ind_stop!=2)
	      {
		if(file_recovery_old.file_stat!=NULL && file_recovery_old.file_stat->file_hint->file_check!=NULL && paranoid>0)
		{
		  file_recovery_old.file_stat->file_hint->file_check(&file_recovery_old);
		}
		if(file_recovery_old.file_stat!=NULL && file_recovery_old.file_size < file_recovery_old.file_stat->file_hint->min_filesize)
		{ 
		  ecrit_rapport("File too small, reject it\n");
		  file_recovery_old.file_size=0;
		}
#ifdef HAVE_FTRUNCATE
		fflush(file_recovery_old.handle);
		ftruncate(fileno(file_recovery_old.handle), file_recovery_old.file_size);
#endif
	      }
	      fclose(file_recovery_old.handle);
	      file_recovery_old.handle=NULL;
	      if(file_recovery_old.file_size>0 || ind_stop==2)
	      {
		if((++file_nbr)%MAX_FILES_PER_DIR==0)
		{
		  dir_num=photorec_mkdir(recup_dir,dir_num+1);
		}
		file_recovery_old.file_stat->recovered++;
	      }
	      else
	      {
		unlink(file_recovery_old.filename);
	      }
	    }
	    if(file_recovery_old.file_stat!=NULL)
	    {
	      list_free_space=update_free_space(file_recovery_old.filename,&file_recovery_old,list_free_space,&current_free_space,blocksize,disk_car->sector_size);
	      file_recovery_old.file_stat=NULL;
	    }

	    if(file_recovery.file_stat!=NULL && file_recovery.handle==NULL)
	    {
	      if(file_recovery.file_stat->file_hint->extension==NULL)
	      {
		snprintf(file_recovery.filename,sizeof(file_recovery.filename)-1,"%s.%u/%u",recup_dir,dir_num,file_nbr+1);
	      }
	      else
	      {
		snprintf(file_recovery.filename,sizeof(file_recovery.filename)-1,"%s.%u/%u.%s",recup_dir,dir_num,file_nbr+1,file_recovery.file_stat->file_hint->extension);
	      }
	      if(file_recovery.file_stat->file_hint->recover==1)
	      {
		if(!(file_recovery.handle=fopen(file_recovery.filename,"w+b")))
		{
		  perror("out");
		}
	      }
	    }
	    if(file_recovery.handle!=NULL)
	    {
	      fwrite(buffer,blocksize,1,file_recovery.handle);
	    }
	    if(file_recovery.file_stat!=NULL)
	    {
	      if(file_recovery.file_stat->file_hint->data_check!=NULL && file_recovery.file_stat->file_hint->data_check(double_buffer,2*blocksize,&file_recovery)==2)
	      {
		/* EOF found */
		copy_file_recovery(&file_recovery_old,&file_recovery);
		reset_file_recovery(&file_recovery);

		if(file_recovery_old.handle!=NULL)
		{
		  if(ind_stop!=2)
		  {
		    if(file_recovery_old.file_stat!=NULL && file_recovery_old.file_stat->file_hint->file_check!=NULL && paranoid>0)
		    {
		      file_recovery_old.file_stat->file_hint->file_check(&file_recovery_old);
		    }
		    if(file_recovery_old.file_stat!=NULL && file_recovery_old.file_size < file_recovery_old.file_stat->file_hint->min_filesize)
		    { 
		      ecrit_rapport("File too small, reject it\n");
		      file_recovery_old.file_size=0;
		    }
#ifdef HAVE_FTRUNCATE
		    fflush(file_recovery_old.handle);
		    ftruncate(fileno(file_recovery_old.handle), file_recovery_old.file_size);
#endif
		  }
		  fclose(file_recovery_old.handle);
		  file_recovery_old.handle=NULL;
		  if(file_recovery_old.file_size>0 || ind_stop==2)
		  {
		    if((++file_nbr)%MAX_FILES_PER_DIR==0)
		      dir_num=photorec_mkdir(recup_dir,dir_num+1);
		    file_recovery_old.file_stat->recovered++;
		  }
		  else
		  {
		    unlink(file_recovery_old.filename);
		  }
		}
		if(file_recovery_old.file_stat!=NULL)
		{
		  list_free_space=update_free_space(file_recovery_old.filename,&file_recovery_old,list_free_space,&current_free_space,blocksize,disk_car->sector_size);
		  file_recovery_old.file_stat=NULL;
		}
	      }
	      else
	      {
		file_recovery.header_distance+=blocksize;
		file_recovery.file_size+=blocksize;
	      }
	    }
	    memcpy(&double_buffer,buffer,blocksize);
	  }
	}
	if(current_free_space!=NULL)
	  current_free_space=current_free_space->next;
      }
    } /* Pass is finished */

    if(file_recovery.handle)
    {
      if(ind_stop!=2)
      {
	if(pass==1 && file_nbr>0 && (file_recovery.file_stat!=NULL && file_recovery.file_stat->file_hint->stream>0))
	{
	  ecrit_rapport("stream, don't save the new file yet.\n");
	  file_recovery.file_size=0;
	}
	else
	{
	  if(file_recovery.file_stat!=NULL && file_recovery.file_stat->file_hint->file_check!=NULL && paranoid>0)
	  {
	    file_recovery.file_stat->file_hint->file_check(&file_recovery);
	  }
	  if(file_recovery.file_stat!=NULL && file_recovery.file_size<file_recovery.file_stat->file_hint->min_filesize)
	  { 
	    ecrit_rapport("File too small, reject it\n");
	    file_recovery.file_size=0;
	  }
	}
#ifdef HAVE_FTRUNCATE
	fflush(file_recovery.handle);
	ftruncate(fileno(file_recovery.handle), file_recovery.file_size);
#endif
      }
      fclose(file_recovery.handle);
      file_recovery.handle=NULL;
      if(file_recovery.file_size>0 || ind_stop==2)
      {
	  if((++file_nbr)%MAX_FILES_PER_DIR==0)
	  dir_num=photorec_mkdir(recup_dir,dir_num+1);
	file_recovery.file_stat->recovered++;
      }
      else
      {
	unlink(file_recovery.filename);
      }
    }
    if(file_recovery.file_stat!=NULL)
    {
      list_free_space=update_free_space(file_recovery.filename,&file_recovery,list_free_space,&current_free_space,blocksize,disk_car->sector_size);
      file_recovery.file_stat=NULL;
    }
    if(ind_stop==2)
    {
      ind_stop=3;
    }
    else if(old_file_nbr==file_nbr || (paranoid==0 && pass==2))
    {
      if(keep_corrupted_file>0)
	ind_stop=2;
      else
	ind_stop=3;
    }
    else
    {
      if(pass==1)
      {
	blocksize=find_blocksize(list_file,disk_car->sector_size);
	if(blocksize>disk_car->sector_size)
	{
	  blocksize=menu_choose_blocksize(blocksize, disk_car->sector_size);
	}
	list_free_space=update_blocksize(blocksize,list_free_space);
      }
    }
    {
      time_t current_time;
      current_time=time(NULL);
      ecrit_rapport("Elapsed time %uh%02um%02us\n",
	  (unsigned)((current_time-real_start_time)/60/60),
	  (unsigned)((current_time-real_start_time)/60%60),
	  (unsigned)((current_time-real_start_time)%60));
    }
    update_stats(file_stats,list_free_space);
    ecrit_rapport("Pass %u +%u file%s\n",pass,file_nbr-old_file_nbr,(file_nbr-old_file_nbr<=1?"":"s"));
    write_stats_log(file_stats);
    if(interface==0)
    {
      printf("Pass %u +%u file%s\n",pass,file_nbr-old_file_nbr,(file_nbr-old_file_nbr<=1?"":"s"));
      write_stats_stdout(file_stats);
      fflush(stdout);
    }
  }
  {
    unsigned long int nbr_headers=0;
    uint64_t sectors_with_unknown_data=0;
    t_alloc_data *next;
    /* Free memory */
    for(current_free_space=list_file;current_free_space!=NULL;current_free_space=next)
    {
      next=current_free_space->next;
      FREE(current_free_space);
    }
    for(current_free_space=list_free_space;current_free_space!=NULL;current_free_space=next)
    {
      if(current_free_space->file_stat!=NULL)
      {
	nbr_headers++;
	current_free_space->file_stat->not_recovered++;
      }
      sectors_with_unknown_data+=(current_free_space->end-current_free_space->start+disk_car->sector_size-1)/disk_car->sector_size;
      if(debug>0)
      {
	ecrit_rapport("%lu-%lu: %s\n",(long unsigned)(current_free_space->start/disk_car->sector_size),
	    (long unsigned)(current_free_space->end/disk_car->sector_size), (current_free_space->file_stat!=NULL && current_free_space->file_stat->file_hint!=NULL?current_free_space->file_stat->file_hint->extension:NULL));
      }
      next=current_free_space->next;
      FREE(current_free_space);
    }
    ecrit_rapport("%llu sectors contains unknown data, %lu invalid files found and rejected.\n",
	(long long unsigned)sectors_with_unknown_data, (long unsigned)nbr_headers);
  }
  if(ind_stop==1)
    ecrit_rapport("PhotoRec has been stopped, ");
  if(interface)
  {
    wmove(stdscr,9,0);
    wclrtoeol(stdscr);
    wdoprintf(stdscr,"%u pictures or videos found\n",file_nbr);
    wmove(stdscr,10,0);
    wclrtoeol(stdscr);
    aff_buffer(BUFFER_DISPLAY,"Q",stdscr);
  }
  return 0;
}

static void update_stats(t_file_stat *file_stats, t_alloc_data *list_free_space)
{
  int i;
  t_alloc_data *current_free_space;
  /* Reset */
  for(i=0;file_stats[i].file_hint!=NULL;i++)
    file_stats[i].not_recovered=0;
  /* Update */
  for(current_free_space=list_free_space;current_free_space!=NULL;current_free_space=current_free_space->next)
  {
    if(current_free_space->file_stat!=NULL)
    {
      current_free_space->file_stat->not_recovered++;
    }
  }
}

static void write_stats_log(const t_file_stat *file_stats)
{
  int i;
  unsigned int file_nbr=0;
  for(i=0;file_stats[i].file_hint!=NULL;i++)
  {
    if(file_stats[i].recovered+file_stats[i].not_recovered>0)
    {
      file_nbr+=file_stats[i].recovered;
      ecrit_rapport("%s: %u/%u recovered\n",file_stats[i].file_hint->extension,
	  file_stats[i].recovered, file_stats[i].recovered+file_stats[i].not_recovered);
    }
  }
  if(file_nbr>1)
  {
    ecrit_rapport("Total: %u files found\n\n",file_nbr);
  }
  else
  {
    ecrit_rapport("Total: %u file found\n\n",file_nbr);
  }
}

static void write_stats_stdout(const t_file_stat *file_stats)
{
  int i;
  unsigned int file_nbr=0;
  for(i=0;file_stats[i].file_hint!=NULL;i++)
  {
    if(file_stats[i].recovered+file_stats[i].not_recovered>0)
    {
      file_nbr+=file_stats[i].recovered;
      printf("%s: %u/%u recovered\n",file_stats[i].file_hint->extension,
	  file_stats[i].recovered, file_stats[i].recovered+file_stats[i].not_recovered);
    }
  }
  if(file_nbr>1)
  {
    printf("Total: %u pictures or videos found\n\n",file_nbr);
  }
  else
  {
    printf("Total: %u picture or video found\n\n",file_nbr);
  }
}

static int do_curses_photorec(int debug, const char *recup_dir, const t_list_disk *list_disk, const t_arch_fnct **arch, t_file_stat *file_stats)
{
  int command;
  int done=0;
  int allow_partial_last_cylinder=0;
  int halt_on_errors=1;
  int paranoid=1;
  int menu=0;
  int keep_corrupted_file=0;
  const t_list_disk *element_disk;
  const t_list_disk *current_disk=list_disk;
  static struct MenuItem menuMain[]=
  {
	{'S',"Search","Search pictures"},
	{'O',"Options","Modify options"},
	{'F',"File Opt","Modify file options"},
	{'Q',"Quit","Quit program"},
	{0,NULL,NULL}
  };
  if(list_disk==NULL)
    return 0;
/* use_env(TRUE); */
#ifdef HAVE_SETENV
#ifdef DJGPP
#elif defined(BSD)
    setenv("TERM","cons25",0);
#elif defined(LINUX)
    setenv("TERM","linux",0);
#elif defined(__CYGWIN__)
    setenv("TERM","cygwin",0);
#endif
#endif
#ifdef DJGPP
  initscr();
#else
  if(newterm(NULL,stdout,stdin)==NULL)
  {
#ifdef HAVE_SETENV
    setenv("TERMINFO",".",1);
#endif
    if(newterm(NULL,stdout,stdin)==NULL)
    {
      ecrit_rapport("Terminfo file is missing.\n");
      printf("Terminfo file is missing.\nCheck if you have not forgot to extract the subdirectories.\n");
      return 1;
    }
  }
#endif
  if(LINES<25)
  {
    ecrit_rapport("Terminal has only %u lines\n",LINES);
    endwin();
    return 1;
  }
  noecho();
#ifndef DJGPP
  nonl(); /*don't use for Dos version but enter will work with it... dilema */
#endif
  /*  intrflush(stdscr, FALSE); */
  cbreak();
  /* Should solve a problem with users who redefined the colors */
  if(has_colors())
  {
    start_color();
  }
  curs_set(0);
  /* ncurses interface */
  while(done==0)
  {
    aff_copy(stdscr);
    wmove(stdscr,5,0);
    for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
    {
      if(element_disk!=current_disk)
	wdoprintf(stdscr,"%s\n",element_disk->disk->description(element_disk->disk));
      else
      {
	wattrset(stdscr,A_STANDOUT);
	wdoprintf(stdscr,"%s\n",element_disk->disk->description(element_disk->disk));
	wattroff(stdscr,A_STANDOUT);
      }
    }
    current_disk->disk->halt_on_errors=halt_on_errors;
    command = wmenuSelect(stdscr,INTER_MAIN_Y, INTER_MAIN_X, menuMain, 8,
	  "SOFQ", MENU_HORIZ | MENU_BUTTON | MENU_ACCEPT_OTHERS, menu);
    switch(command)
    {
      case 's':
      case 'S':
	interface_select(current_disk->disk, debug, paranoid,recup_dir,keep_corrupted_file,file_stats);
	menu=3;
	break;
      case 'o':
      case 'O':
	{
	  int old_allow_partial_last_cylinder=allow_partial_last_cylinder;
	  interface_options_photorec(&paranoid,&allow_partial_last_cylinder,&halt_on_errors,arch,&keep_corrupted_file);
	  for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
	  {
	    element_disk->disk->arch=*arch;
	  }
	  if(old_allow_partial_last_cylinder!=allow_partial_last_cylinder)
	    hd_parse_bis(list_disk,allow_partial_last_cylinder,*arch);
	  menu=1;
	}
	break;
      case 'f':
      case 'F':
	interface_file_select(file_stats);
	menu=2;
	break;
      case 'q':
      case 'Q':
	done = 1;
	break;
      case MENU_UP:
	if(current_disk->prev!=NULL)
	  current_disk=current_disk->prev;
	break;
      case MENU_DOWN:
	if(current_disk->next!=NULL)
	  current_disk=current_disk->next;
	break;
    }
  }
  wclear(stdscr);
  wrefresh(stdscr);
  nl();
  endwin();
  ecrit_rapport("\n");
  return 0;
}

static void interface_options_photorec(int *paranoid, int *allow_partial_last_cylinder,int *halt_on_errors, const t_arch_fnct **arch, int *keep_corrupted_file)
{
  int done = FALSE;
  int menu = 5;
  while (done==FALSE)
  {
    int car;
    struct MenuItem menuOptions[]=
    {
      { 'P', NULL, "Check JPG files" },
      { 'A',NULL,"" },
      { 'H',NULL,"Halt on Disk I/O errors" },
      { 'T',NULL,"Partition table type"},
      { 'K',NULL,"Keep corrupted files"},
      { 'Q',"Quit","Return to main menu"},
      { 0, NULL, NULL }
    };
    menuOptions[0].name=*paranoid?"Paranoid : Yes":"Paranoid : No";
    menuOptions[1].name=*allow_partial_last_cylinder?"Allow partial last cylinder : Yes":"Allow partial last cylinder : No";
    menuOptions[2].name=*halt_on_errors?"Halt on errors : Yes":"Halt on errors : No";
    menuOptions[3].name=(*arch)->part_name;
    menuOptions[4].name=*keep_corrupted_file?"Keep corrupted files : Yes":"Keep corrupted files : No";
    /* Jpg
       Mov
       Mpg
       Minolta MRW
       Canon CRW
       Signa/Foveon X3F
       Fuji RAF
       Rollei RDC
       MP3

    */
    car=toupper( wmenuSelect(stdscr,INTER_OPTION_Y, INTER_OPTION_X, menuOptions, 0, "PAHTKQ", MENU_VERT, menu));
    switch(car)
    {
      case 'P':
	*paranoid=!*paranoid;
	menu=0;
	break;
      case 'A':
	*allow_partial_last_cylinder=!*allow_partial_last_cylinder;
	menu=1;
	break;
      case 'H':
	*halt_on_errors=!*halt_on_errors;
	menu=2;
	break;
      case 'T':
       if(*arch==&arch_i386)
	 *arch=&arch_sun;
       else if(*arch==&arch_sun)
	 *arch=&arch_none;
       else
	 *arch=&arch_i386;
	menu=3;
	break;
      case 'K':
	*keep_corrupted_file=!*keep_corrupted_file;
	menu=4;
	break;
      case key_ESC:
      case 'Q':
	done = TRUE;
	break;
    }
  }
  /* write new options to log file */
  ecrit_rapport("New options :\n Paranoid : %s\n",
      *paranoid?"Yes":"No");
  ecrit_rapport("\n Allow partial last cylinder : %s\n Halt on errors : %s\n Partition table type : %s\nKeep corrupted files : %s\n",
      *allow_partial_last_cylinder?"Yes":"No",
      *halt_on_errors?"Yes":"No",
      (*arch==&arch_i386?"Intel":(*arch==&arch_sun?"Sun":"None")),
      *keep_corrupted_file?"Yes":"No");
}

static t_partition *new_whole_disk(const t_param_disk *disk_car)
{
  t_partition *fake_partition;
  fake_partition=partition_new();
  fake_partition->part_offset=0;
  fake_partition->part_size=disk_car->disk_size;
  fake_partition->part_type=0;
  strncpy(fake_partition->name,"Whole disk",sizeof(fake_partition->name)-1);
  return fake_partition;
}

static void interface_select(t_param_disk *disk_car, const int debug,const int paranoid, const char *recup_dir, const int keep_corrupted_file, t_file_stat *file_stats)
{
  t_list_part *list_part;
  t_list_part *element;
  t_list_part *current_element;
  int quit;
  int offset=0;
  int current_element_num=0;
  int rewrite=1;
  int menu=0;
  ecrit_rapport("\nInterface Select\n");
  list_part=disk_car->arch->read_part(disk_car,debug,0);
  list_part=insert_new_partition(list_part,new_whole_disk(disk_car));
  current_element=list_part;
  for(element=list_part;element!=NULL;element=element->next)
  {
    aff_part_rapport(disk_car,element->part);
  }
  do
  {
    static struct MenuItem menuAdv[]=
    {
      {'s',"Search","Start photo recovery"},
      {'q',"Quit","Return to main menu"},
      {0,NULL,NULL}
    };
    int i;
    int command;
    if(rewrite!=0)
    {
      aff_copy(stdscr);
      wmove(stdscr,4,0);
      wdoprintf(stdscr,"%s",disk_car->description(disk_car));
      mvwaddstr(stdscr,6,0,msg_PART_HEADER_LONG);
      rewrite=0;
    }
    for(i=0,element=list_part;(element!=NULL) && (i<offset);element=element->next,i++);
    for(i=offset;(element!=NULL) && ((i-offset)<INTER_SELECT);i++,element=element->next)
    {
      wmove(stdscr,5+2+i-offset,0);
      wclrtoeol(stdscr);	/* before addstr for BSD compatibility */
      if(element==current_element)
      {
	wattrset(stdscr,A_STANDOUT);
	aff_part(stdscr,AFF_PART_ORDER,disk_car,element->part);
	wattroff(stdscr,A_STANDOUT);
      } else
      {
	aff_part(stdscr,AFF_PART_ORDER,disk_car,element->part);
      }
    }
    quit=0;
    command = wmenuSelect(stdscr,INTER_SELECT_Y, INTER_SELECT_X, menuAdv, 8,
	"sq", MENU_HORIZ | MENU_BUTTON | MENU_ACCEPT_OTHERS, menu);
    switch(command)
    {
      case MENU_UP:
	if(current_element!=NULL)
	{
	  if(current_element->prev!=NULL)
	  {
	    current_element=current_element->prev;
	    current_element_num--;
	  }
	  if(current_element_num<offset)
	    offset--;
	}
	break;
      case MENU_DOWN:
	if(current_element!=NULL)
	{
	  if(current_element->next!=NULL)
	  {
	    current_element=current_element->next;
	    current_element_num++;
	  }
	  if(current_element_num>=offset+INTER_SELECT)
	    offset++;
	}
	break;
      case 'q':
      case 'Q':
	quit=1;
	break;
      case 's':
      case 'S':
	if(current_element!=NULL)
	{
	  photorec(disk_car, current_element->part, debug, paranoid, recup_dir,keep_corrupted_file,1,file_stats);
	  rewrite=1;
	  menu=1;
	}
	break;
    }
  } while(quit==0);
  delete_list_part(list_part);
}

static void interface_file_select(t_file_stat *file_stats)
{
  int quit;
  int offset=0;
  int current_element_num=0;
  int rewrite=1;
  int menu=0;
  ecrit_rapport("\nInterface File Select\n");
  do
  {
    static struct MenuItem menuAdv[]=
    {
      {'q',"Quit","Return to main menu"},
      {0,NULL,NULL}
    };
    int i;
    int command;
    if(rewrite!=0)
    {
      aff_copy(stdscr);
      wmove(stdscr,5,0);
      wdoprintf(stdscr,"PhotoRec will try to locate the following files");
      rewrite=0;
    }
    for(i=offset;file_stats[i].file_hint!=NULL && ((i-offset)<INTER_SELECT);i++)
    {
      wmove(stdscr,5+2+i-offset,0);
      wclrtoeol(stdscr);	/* before addstr for BSD compatibility */
      if(i==current_element_num)
      {
	wattrset(stdscr,A_STANDOUT);
	wdoprintf(stdscr,"[%c] %-4s %s", (file_stats[i].enable==0?' ':'X'),
	    file_stats[i].file_hint->extension, file_stats[i].file_hint->description);
	wattroff(stdscr,A_STANDOUT);
      } else
      {
	wdoprintf(stdscr,"[%c] %-4s %s", (file_stats[i].enable==0?' ':'X'),
	    file_stats[i].file_hint->extension, file_stats[i].file_hint->description);
      }
    }
    quit=0;
    command = wmenuSelect(stdscr,INTER_SELECT_Y, INTER_SELECT_X, menuAdv, 8,
	"q", MENU_BUTTON | MENU_ACCEPT_OTHERS, menu);
    switch(command)
    {
      case MENU_UP:
	if(current_element_num>0)
	{
	  current_element_num--;
	  if(current_element_num<offset)
	    offset--;
	}
	break;
      case KEY_PPAGE:
	for(i=0;(i<INTER_SELECT) && (current_element_num>0);i++)
	{
	  current_element_num--;
	  if(current_element_num<offset)
	    offset--;
	}
	break;
      case MENU_DOWN:
	if(file_stats[current_element_num+1].file_hint!=NULL)
	{
	  current_element_num++;
	  if(current_element_num>=offset+INTER_SELECT)
	    offset++;
	}
	break;
      case KEY_NPAGE:
	for(i=0;(i<INTER_SELECT) && (file_stats[current_element_num+1].file_hint!=NULL);i++)
	{
	  current_element_num++;
	  if(current_element_num>=offset+INTER_SELECT)
	    offset++;
	}
	break;
      case MENU_RIGHT:
      case KEY_RIGHT:
      case '+':
      case ' ':
      case MENU_LEFT:
      case KEY_LEFT:
      case '-':
      case 'x':
      case 'X':
	file_stats[current_element_num].enable=1-file_stats[current_element_num].enable;
	break;
      case 'q':
      case 'Q':
	quit=1;
	break;
    }
  } while(quit==0);
}

void aff_copy(WINDOW *window)
{
  wclear(window);
  keypad(window, TRUE); /* Need it to get arrow key */
  wmove(window,0,0);
  wdoprintf(window, "PhotoRec %s, Data Recovery Utility, %s\n",VERSION,TESTDISKDATE);
  wmove(window,1,0);
  wdoprintf(window, "Christophe GRENIER <grenier@cgsecurity.org>");
  wmove(window,2,0);
  wdoprintf(window, "http://www.cgsecurity.org");
}

static unsigned int find_blocksize(t_alloc_data *list_file, const unsigned int default_blocksize)
{
  t_alloc_data *current_file;
  int blocksize_ok=0;
  unsigned int blocksize;
  uint64_t offset;
  if(list_file==NULL)
    return default_blocksize;
  for(blocksize=64*512;blocksize>=default_blocksize && blocksize_ok==0;blocksize=blocksize>>1)
  {
    blocksize_ok=1;
    offset=list_file->start%blocksize;
    for(current_file=list_file;current_file!=NULL && blocksize_ok!=0;current_file=current_file->next)
    {
      if(current_file->start%blocksize!=offset)
	blocksize_ok=0;
    }
  }
  blocksize=blocksize<<1;
  return blocksize;
}

static t_alloc_data * update_blocksize(unsigned int blocksize, t_alloc_data *list_free_space)
{
  t_alloc_data *current_free_space;
  uint64_t offset;
  if(list_free_space==NULL)
    return list_free_space;
  offset=list_free_space->end%blocksize;
  for(current_free_space=list_free_space;current_free_space!=NULL;)
  {
    t_alloc_data *next=current_free_space->next;
    current_free_space->start=(current_free_space->start-(offset+1)%blocksize+blocksize-1)/blocksize*blocksize+(offset+1)%blocksize;
    if(current_free_space->start>current_free_space->end)
    {
      if(current_free_space->prev!=NULL)
	current_free_space->prev->next=current_free_space->next;
      if(current_free_space->next!=NULL)
	current_free_space->next->prev=current_free_space->prev;
      if(list_free_space==current_free_space)
	list_free_space=next;
      FREE(current_free_space);
    }
    current_free_space=next;
  }
  return list_free_space;
}


int main( int argc, char **argv )
{
  int i;
  int help=0, create_log=0, debug=0, interface=1;
  const char *recup_dir="recup_dir";
  t_list_disk *list_disk=NULL;
  t_list_disk *element_disk;
#ifdef SOLARIS
  const t_arch_fnct *arch=&arch_sun;
#else
  const t_arch_fnct *arch=&arch_i386;
#endif
#ifdef HAVE_SIGACTION
  struct sigaction action, old_action;
#endif
  t_file_stat file_stats[]=
  {
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_au   },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_avi  },
    { .enable=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_bmp  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_bz2  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_c    },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_crw  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_ctg  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_dir  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_doc  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_dsc  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_html },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_jpg  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_mov  },
    { .enable=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_mp3  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_mpg  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_mrw  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_orf  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_pdf  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_php  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_pl   },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_png  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_raf  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_raw  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_rdc  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_rtf  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_sh   },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_tar  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_tiff },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_wma  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_x3f  },
    { .enable=1, .not_recovered=0, .recovered=0, .file_hint=&file_hint_zip  },
    { .enable=0, .not_recovered=0, .recovered=0, .file_hint=NULL }
  };

#ifdef TESTING
  srand(1);
#endif
#ifdef HAVE_SIGACTION
  /* set up the signal handler for SIGHUP */
  action.sa_handler  = sighup_hdlr;
  action.sa_flags = 0;
  if(sigaction(SIGHUP, &action, &old_action)==-1)
  {
    printf("Error on SIGACTION call\n");
    return -1;
  }
#endif
#ifdef __CYGWIN__
  /* Average Windows user doesn't know how to specify a parameter */
  if(argc==1)
  {
    create_log=1;
    debug=1;
  }
#endif
  printf("PhotoRec %s, Data Recovery Utility, %s\nChristophe GRENIER <grenier@cgsecurity.org>\nhttp://www.cgsecurity.org\n",VERSION,TESTDISKDATE);
  for(i=1;i<argc;i++)
  {
    if((strcmp(argv[i],"/log")==0) ||(strcmp(argv[i],"-log")==0))
      create_log=1;
    else if((strcmp(argv[i],"/debug")==0) || (strcmp(argv[i],"-debug")==0))
    {
      debug++;
      create_log=1;
    }
    else if(((strcmp(argv[i],"/d")==0)||(strcmp(argv[i],"-d")==0)) &&(i+1<argc))
    {
      recup_dir=argv[i+1];
      i++;
    }
    else if((strcmp(argv[i],"/noui")==0) ||(strcmp(argv[i],"-noui")==0))
      interface=0;
    else if((strcmp(argv[i],"/help")==0) || (strcmp(argv[i],"-help")==0) || (strcmp(argv[i],"--help")==0) ||
      (strcmp(argv[i],"/h")==0) || (strcmp(argv[i],"-h")==0))
      help=1;
    else
    {
      list_disk=insert_new_disk(list_disk,file_test_availability(argv[i],debug,arch));
      if(list_disk==NULL)
      {
	printf("\nUnable to open file or device %s\n",argv[i]);
	help=1;
      }
    }
  }
  if(help!=0)
  {
    printf("\nUsage: photorec [/log] [/debug] [/d recup_dir] [file or device]"\
	    "\n" \
	    "\n/log          : create a photorec.log file" \
	    "\n/debug        : add debug information" \
	    "\n" \
	    "\nPhotoRec searches JPEG and MOV files, it stores them in recup_dir directory." \
	    "\n" \
	    "\nIf you have problems with PhotoRec or bug reports, please contacte me.\n");
    return 0;
  }
  if(create_log!=0)
  {
/*    const char *ext2fs_version=NULL; */
    f_rapport=init_log("photorec.log",argc,argv);
  ecrit_rapport("PhotoRec %s, Data Recovery Utility, %s\nChristophe GRENIER <grenier@cgsecurity.org>\nhttp://www.cgsecurity.org\n",VERSION,TESTDISKDATE);
#ifdef DJGPP
    ecrit_rapport("Dos version");
#elif defined(BSD)
    ecrit_rapport("BSD version");
#elif defined(LINUX)
    ecrit_rapport("Linux version");
#elif defined(__CYGWIN__)
    ecrit_rapport("Windows version");
#else
    ecrit_rapport("Undefined OS");
#endif
    ecrit_rapport("\n");
  }
  printf("Please wait...\n");
  {
    const char *locale;
    locale = setlocale (LC_ALL, "");
    if (locale==NULL) {
      locale = setlocale (LC_ALL, NULL);
      ecrit_rapport("Failed to set locale, using default '%s'.\n", locale);
    } else {
      ecrit_rapport("Using locale '%s'.\n", locale);
    }
  }
  aff_buffer(BUFFER_RESET,"Q");
  list_disk=hd_parse(list_disk,debug,arch);
  if(list_disk==NULL)
  {
    printf("No harddisk found\n");
#ifdef __CYGWIN__
    printf("You need to be administrator to use PhotoRec\n");
    printf("Under Win9x, use the DOS version instead\n");
    ecrit_rapport("You need to be administrator to use PhotoRec\n");
#else
#ifndef DJGPP
#ifdef HAVE_GETEUID
    if(geteuid()!=0)
    {
      printf("You need to be root to use PhotoRec\n");
      ecrit_rapport("You need to be root to use PhotoRec\n");
    }
#endif
#endif
#endif
  }
#ifdef DJGPP
  for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
  {
    printf("%s\n",element_disk->disk->description(element_disk->disk));
  }
#endif
  hd_parse_bis(list_disk,0,arch);
  /* save disk parameters to rapport */
  ecrit_rapport("Hard disk list\n");
  for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
  {
    printf("%s, sector size=%u\n",element_disk->disk->description(element_disk->disk),element_disk->disk->sector_size);
    ecrit_rapport("%s, sector size=%u\n",element_disk->disk->description(element_disk->disk),element_disk->disk->sector_size);
  }
  printf("\n");
  ecrit_rapport("\n");
  if(interface)
  {
    if(do_curses_photorec(debug,recup_dir,list_disk,&arch,file_stats))
    {
      printf("PhotoRec need 25 lines to work.\nPlease enlarge the terminal and restart PhotoRec.\n");
    }
  }
  else
  {
    int done=0;
    t_param_disk *disk_car=NULL;
    for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
    {
      disk_car=element_disk->disk;
    }
    /* last disk, whole disk, debug, paranoid=1, recup_dir, keep_corrupted_file=0, interface=0 */
    if(done==0)
    {
      photorec(disk_car, new_whole_disk(disk_car), debug, 1, recup_dir,0,0,file_stats);
    }
  }
  for(element_disk=list_disk;element_disk!=NULL;)
  {
    t_list_disk *element_disk_next=element_disk->next;
    if(element_disk->disk->clean!=NULL)
      element_disk->disk->clean(element_disk->disk);
    FREE(element_disk->disk);
    FREE(element_disk);
    element_disk=element_disk_next;
  }
  if(f_rapport!=NULL)
  {
    ecrit_rapport("PhotoRec exited normally.\n");
    if(fclose(f_rapport))
    {
      f_status=1;
    }
  }
  if(f_status!=0)
  {
    printf("PhotoRec: Log file corrupted!\n");
  }
  else
  {
    printf("PhotoRec exited normally.\n");
  }
  return 0;
}
