/*

    File: partnone.c

    Copyright (C) 1998-2004 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <ctype.h>      /* toupper, tolower */
#include "types.h"
#include "common.h"
#include "testdisk.h"
#include "fnctdsk.h"
#include "analyse.h"
#include "lang.h"
#include "intrf.h"
#include "chgtype.h"
#include "savehdr.h"
#include "bfs.h"
#include "bsd.h"
#include "cramfs.h"
#include "ext2.h"
#include "fat.h"
#include "hfs.h"
#include "hfsp.h"
#include "jfs_superblock.h"
#include "jfs.h"
#include "lvm.h"
#include "md.h"
#include "ntfs.h"
#include "rfs.h"
#include "swap.h"
#include "ufs.h"
#include "xfs.h"

static int check_part_none(t_param_disk *disk_car, const int debug,t_partition *partition,const int saveheader);
static int get_geometry_from_nonembr(const unsigned char *buffer, const int debug, t_CHS *geometry);
static t_list_part *read_part_none(t_param_disk *disk_car, const int debug, const int saveheader);
static int write_part_none(t_param_disk *disk_car, const t_list_part *list_part, const int ro , const int debug, const int align);
static t_list_part *init_part_order_none(const t_param_disk *disk_car, t_list_part *list_part);
static int write_MBR_code_none(unsigned char *buffer);
static t_list_part *add_partition_none(t_param_disk *disk_car,t_list_part *list_part, const int debug);
static void set_next_status_none(const t_param_disk *disk_car, t_partition *partition);
static int test_structure_none(t_list_part *list_part);
static int is_part_type_ok_none(unsigned int part_type);
static void init_structure_none(const t_param_disk *disk_car,t_list_part *list_part, const int debug);
static void erase_list_part_none(unsigned char *buffer);

static const struct systypes none_sys_types[] = {
  {"\x00" "empty"},
  {"\x01" "FAT12"},
  {"\x02" "XENIX root"},
  {"\x03" "XENIX /usr"},
  {"\x04" "FAT16 <32M"},
  {"\x05" "extended"},
  {"\x06" "FAT16 >32M"},
  {"\x07" "HPFS - NTFS"},
  {"\x09" "AIX data"},
  {"\x0a" "OS/2 Boot Manager"},
  {"\x0b" "FAT32"},
  {"\x0c" "FAT32 LBA"},
  {"\x0E" "FAT16 LBA"},
  {"\x0F" "extended LBA"},
  {"\x10" "OPUS"},
  {"\x11" "hid. FAT12"},
  {"\x12" "Compaq Diagnostics"},
  {"\x14" "hid. FAT16 <32M"},
  {"\x16" "hid. FAT16 >32M"},
  {"\x17" "hid. HPFS/NTFS"},
  {"\x18" "AST swap"},
  {"\x19" "Willowtech Photon"},
  {"\x1b" "hid. FAT32"},
  {"\x1c" "hid. FAT32 LBA"},
  {"\x1e" "hid. FAT16 LBA"},
  {"\x20" "Willowsoft OFS1"},
  {"\x24" "NEC MS-DOS 3.x"},
  {"\x38" "Theos"},
  {"\x3c" "PMagic recovery"},
  {"\x40" "VENIX 80286"},
  {"\x41" "PPC PReP Boot"},
  {"\x42" "W2K Dynamic/SFS"},
  {"\x50" "OnTrack DM RO"},
  {"\x51" "OnTrack DM RW-NOVEL"},
  {"\x52" "CP/M-Microport V/386"},
  {"\x53" "OnTrack DM WO ???"},
  {"\x54" "OnTrack DM DDO"},
  {"\x55" "EZ-Drive"},
  {"\x56" "GoldenBow VFeature"},
  {"\x61" "SpeedStor"},
  {"\x63" "Unixware, HURD, SCO"},
  {"\x64" "NetWare 286"},
  {"\x65" "NetWare 3.11+"},
  {"\x67" "Novell"},
  {"\x68" "Novell"},
  {"\x69" "Novell"},
  {"\x70" "DiskSecure MB"},
  {"\x75" "PC/IX"},
  {"\x80" "Minix v1.1-1.4a"},
  {"\x81" "Minix / old Linux"},
  {"\x82" "Linux Swap"},
  {"\x83" "Linux"},
  {"\x85" "Linux extended"},
  {"\x86" "NT FAT16 V/S set"},
  {"\x87" "HPFS FT mirror-V/S set"},
  {"\x8e" "Linux LVM"},
  {"\x93" "Amoeba"},
  {"\x94" "Amoeba bad block"},
  {"\xa0" "NoteBIOS save2disk"},
  {"\xa5" "FreeBSD"},
  {"\xa6" "OpenBSD"},
  {"\xa8" "Darwin UFS"},
  {"\xa9" "NetBSD"},
  {"\xab" "Darwin boot"},
  {"\xaf" "HFS"},
  {"\xb7" "BSDI"},
  {"\xb8" "BSDI swap"},
  {"\xbe" "Solaris boot"},
  {"\xc1" "secured FAT12"},
  {"\xc4" "secured FAT16"},
  {"\xc6" "sec. Huge-bad FAT16"},
  {"\xc7" "Syrinx Boot-bad NTFS"},
  {"\xd8" "CP/M-86"},
  {"\xdb" "CP/M"},
  {"\xde" "Dell Utility"},
  {"\xe1" "SpeedStor FAT12 ext"},
  {"\xe3" "DOS RO"},
  {"\xe4" "SpeedStor FAT16 ext"},
  {"\xeb" "BeFS"},
  {"\xee" "EFI GPT"},          /* Intel EFI GUID Partition Table */
  {"\xef" "EFI (FAT-12/16/32)"},/* Intel EFI System Partition */
  {"\xf0" "Linux/PA-RISC boot"},/* Linux/PA-RISC boot loader */
  {"\xf1" "Storage Dimensions"},
  {"\xf2" "DOS secondary"},
  {"\xf4" "SpeedStor"},
  {"\xfd" "Linux RAID"},
  {"\xfe" "LANstep"},
  {"\xff" "Xenix bad block"},
  { NULL }
};

t_arch_fnct arch_none=
{
  .part_name="Partition table type : None",
  .parttype_name_table=none_sys_types,
  .read_part=read_part_none,
  .write_part=write_part_none,
  .init_part_order=init_part_order_none,
  .get_geometry_from_mbr=get_geometry_from_nonembr,
  .check_part=check_part_none,
  .write_MBR_code=write_MBR_code_none,
  .add_partition=add_partition_none,
  .set_prev_status=set_next_status_none,
  .set_next_status=set_next_status_none,
  .test_structure=test_structure_none,
  .is_part_type_ok=is_part_type_ok_none,
  .init_structure=init_structure_none,
  .erase_list_part=erase_list_part_none
};

int get_geometry_from_nonembr(const unsigned char *buffer, const int debug, t_CHS *geometry)
{
  geometry->cylinder=0;
  geometry->head=0;
  geometry->sector=0;
  return 0;
}

t_list_part *read_part_none(t_param_disk *disk_car, const int debug, const int saveheader)
{
  t_list_part *list_part;
  t_partition *partition;
  t_partition *new_partition;
  int res;
  new_partition=partition_new();
  partition_reset(new_partition);
  new_partition->order=NO_ORDER;
  new_partition->part_offset=0;
  new_partition->part_size=disk_car->disk_size;
  new_partition->part_type=0;
  new_partition->status=STATUS_PRIM;

  partition=partition_new();
  res=search_type_128(disk_car,partition,debug,0);
  if(res<=0)
    res=search_type_64(disk_car,partition,debug,0);
  if(res<=0)
    res=search_type_16(disk_car,partition,debug,0);
  if(res<=0)
    res=search_type_2(disk_car,partition,debug,0);
  if(res<=0)
    res=search_type_1(disk_car,partition,debug,0);
  if(res<=0)
    res=search_type_0(disk_car,partition,debug,0);
  if(res>0)
  { /* limited dup_t_partition */
    new_partition->boot_sector=partition->boot_sector;
    new_partition->part_type=partition->part_type;
    new_partition->upart_type=partition->upart_type;
    strncpy(new_partition->info,partition->info,sizeof(new_partition->info));
    strncpy(new_partition->name,partition->name,sizeof(new_partition->name));
    new_partition->arch=partition->arch;
  }
  aff_buffer(BUFFER_RESET,"Q");
  disk_car->arch->check_part(disk_car,debug,new_partition,saveheader);
  FREE(partition);
  list_part=insert_new_partition(NULL,new_partition);
  aff_part_buffer(AFF_PART_ORDER,disk_car,new_partition);
  return list_part;
}

static int write_part_none(t_param_disk *disk_car, const t_list_part *list_part, const int ro, const int debug, const int align)
{
  return 0;
}

static t_list_part *init_part_order_none(const t_param_disk *disk_car, t_list_part *list_part)
{
  return list_part;
}

static int write_MBR_code_none(unsigned char *buffer)
{
  return 0;
}

static t_list_part *add_partition_none(t_param_disk *disk_car,t_list_part *list_part, const int debug)
{
  return list_part;
}

static void set_next_status_none(const t_param_disk *disk_car, t_partition *partition)
{
}

static int test_structure_none(t_list_part *list_part)
{
  return 0;
}

static int is_part_type_ok_none(unsigned int part_type)
{
  return 1;
}

static void init_structure_none(const t_param_disk *disk_car,t_list_part *list_part, const int debug)
{
  t_list_part *element;
  for(element=list_part;element!=NULL;element=element->next)
  {
    element->part->status=STATUS_PRIM;
  }
}

static void erase_list_part_none(unsigned char *buffer)
{
}

static int check_part_none(t_param_disk *disk_car,const int debug,t_partition *partition, const int saveheader)
{
  int ret=0;
  switch(partition->part_type)
  {
    case P_12FAT:
    case P_16FAT:
    case P_16FATBD:
    case P_32FAT:
    case P_32FAT_LBA:
    case P_16FATBD_LBA:
    case P_12FATH:
    case P_16FATH:
    case P_16FATBDH:
    case P_32FATH:
    case P_32FAT_LBAH:
    case P_16FATBD_LBAH:
      ret=check_FAT(disk_car,partition,debug);
      break;
    case P_LINUX:
      ret=check_JFS(disk_car,partition,debug);
      if(ret!=0)
      {
	ret=check_rfs(disk_car,partition,debug);
      }
      if(ret!=0)
      {
	ret=check_EXT2(disk_car,partition,debug);
      }
      if(ret!=0)
      {
	ret=check_cramfs(disk_car,partition,debug);
      }
      if(ret!=0)
      {
	ret=check_xfs(disk_car,partition,debug);
      }
      if(ret!=0)
      { aff_buffer(BUFFER_ADD,"No EXT2, JFS, Reiser, cramfs or XFS marker\n"); }
      break;
    case P_NTFS:
    case P_NTFSH:
      ret=check_NTFS(disk_car,partition,debug,0);
      if(ret!=0)
      { aff_buffer(BUFFER_ADD,"Invalid NTFS boot\n"); }
      break;
    case P_LVM:
      ret=check_LVM(disk_car,partition,debug);
      if(ret!=0)
      {
	ret=check_LVM2(disk_car,partition,debug);
      }
      break;
    case P_BEOS:
      ret=check_BeFS(disk_car,partition,debug);
      break;
    case P_FREEBSD:
      ret=check_BSD(disk_car,partition,debug,BSD_MAXPARTITIONS);
      break;
    case P_OPENBSD:
      ret=check_BSD(disk_car,partition,debug,OPENBSD_MAXPARTITIONS);
      break;
    case P_NETBSD:
      ret=check_BSD(disk_car,partition,debug,BSD_MAXPARTITIONS);
      break;
    case P_LINSWAP:
      ret=check_Linux_SWAP(disk_car,partition,debug);
      break;
    case P_RAID:
      ret=check_MD(disk_car,partition,debug);
      if(ret!=0)
      { aff_buffer(BUFFER_ADD,"Invalid RAID superblock\n"); }
      break;
    case P_HFS:
      ret=check_HFS(disk_car,partition,debug);
      if(ret!=0)
      {
	ret=check_HFSP(disk_car,partition,debug);
      }
      break;
    case P_EXTENDED:
    case P_EXTENDX:
    case P_LINUXEXTENDX:
    case P_NETWARE:
      break;
    default:
      if(debug>0)
      {
	ecrit_rapport("check_part_none %u type %02X: no test\n",partition->order,partition->part_type);
      }
      break;
  }
  return ret;
}

