/*

    File: fnctdsk.c

    Copyright (C) 1998-2004 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include "types.h"
#include "common.h"
#include "fnctdsk.h"
#include "lang.h"
#include "testdisk.h"
#include "intrf.h"
#include "analyse.h"
static unsigned int get_geometry_from_list_part_aux(const t_param_disk *disk_car, const t_list_part *list_part, const int debug);

unsigned long int CHS2LBA(const t_param_disk *disk_car,const t_CHS*CHS)
{ return ((unsigned long int)CHS->cylinder*(disk_car->CHS.head+1)+CHS->head)*disk_car->CHS.sector+CHS->sector-1;
}

unsigned long int C_H_S2LBA(const t_param_disk *disk_car,const unsigned int C, const unsigned int H, const unsigned int S)
{ return ((unsigned long int)C*(disk_car->CHS.head+1)+H)*disk_car->CHS.sector+S-1;
}

uint64_t CHS2offset(const t_param_disk *disk_car,const t_CHS*CHS)
{ return (((uint64_t)CHS->cylinder*(disk_car->CHS.head+1)+CHS->head)*disk_car->CHS.sector+CHS->sector-1)*disk_car->sector_size;
}

uint64_t C_H_S2offset(const t_param_disk *disk_car,const unsigned int C, const unsigned int H, const unsigned int S)
{ return (((uint64_t)C*(disk_car->CHS.head+1)+H)*disk_car->CHS.sector+S-1)*disk_car->sector_size;
}

unsigned int offset2sector(const t_param_disk *disk_car, const uint64_t offset)
{ return ((offset/disk_car->sector_size)%disk_car->CHS.sector)+1; }

unsigned int offset2head(const t_param_disk *disk_car, const uint64_t offset)
{ return ((offset/disk_car->sector_size)/disk_car->CHS.sector)%(disk_car->CHS.head+1); }

unsigned int offset2cylinder(const t_param_disk *disk_car, const uint64_t offset)
{ return ((offset/disk_car->sector_size)/disk_car->CHS.sector)/(disk_car->CHS.head+1); }

void offset2CHS(const t_param_disk *disk_car,const uint64_t offset, t_CHS*CHS)
{
  uint64_t pos=offset/disk_car->sector_size;
  CHS->sector=(pos%disk_car->CHS.sector)+1;
  pos/=disk_car->CHS.sector;
  CHS->head=pos%(disk_car->CHS.head+1);
  CHS->cylinder=pos/(disk_car->CHS.head+1);
}

void dup_t_CHS(t_CHS * CHS_dest, const t_CHS * CHS_source)
{
  CHS_dest->cylinder=CHS_source->cylinder;
  CHS_dest->head=CHS_source->head;
  CHS_dest->sector=CHS_source->sector;
}

void dup_t_partition(t_partition *dest, const t_partition *src)
{
  dest->part_offset=src->part_offset;
  dest->part_size=src->part_size;
  dest->boot_sector=src->boot_sector;
  dest->blocksize=src->blocksize;
  dest->part_type=src->part_type;
  dest->upart_type=src->upart_type;
  dest->status=src->status;
  dest->order=src->order;
  dest->errcode=src->errcode;
  strncpy(dest->info,src->info,sizeof(dest->info));
  strncpy(dest->name,src->name,sizeof(dest->name));
  dest->arch=src->arch;
}

int read_MBR(t_param_disk *disk_car,void *buffer)
{
  if(disk_car->read(disk_car,DEFAULT_SECTOR_SIZE, buffer, (uint64_t)0))
  {
    wdoprintf(stdscr,msg_PART_RD_ERR);
    return 1;
  }
  return 0;
}

int write_MBR(t_param_disk *disk_car,void *buffer)
{
  if(disk_car->write(disk_car,DEFAULT_SECTOR_SIZE, buffer, (uint64_t)0))
  {
    wdoprintf(stdscr,msg_PART_WR_ERR);
    return 1;
  }
  return 0;
}

t_list_disk *insert_new_disk(t_list_disk *list_disk, t_param_disk *disk_car)
{
  if(disk_car==NULL)
    return list_disk;
  {
    t_list_disk *prev;
    t_list_disk *new_disk=(t_list_disk *)MALLOC(sizeof(*new_disk));
    new_disk->disk=disk_car;
    /* Add it at the end */
    for(prev=list_disk;prev!=NULL && prev->next!=NULL;prev=prev->next);
    if(prev!=NULL)
    {
      prev->next=new_disk;
    }
    new_disk->prev=prev;
    new_disk->next=NULL;
    return (list_disk!=NULL?list_disk:new_disk);
  }
}

t_list_part *insert_new_partition(t_list_part *list_part, t_partition *part)
{
  t_list_part *new_list_part;
#ifdef DEBUG
  check_list_part(list_part);
#endif
  new_list_part=insert_new_partition_aux(list_part, element_new(part),0);
#ifdef DEBUG
  check_list_part(new_list_part);
#endif
  return new_list_part;
}

t_list_part *insert_new_partition_aux(t_list_part *list_part, t_list_part *new_element, const unsigned int dont_free_part)
{
/*dont_free_part
  0 if new_element->part is already known, don't insert it and free it
  1 if new_element->part is already known, don't insert it but free it
  2 even if new_element->part is already known, insert it
*/
 /* new partition musn't be used after insert !*/  t_list_part *prev=NULL;
  t_list_part *next;
  for(next=list_part;;next=next->next)
  { /* prev new next */
    if((next==NULL)||
      (new_element->part->part_offset<next->part->part_offset) ||
      (new_element->part->part_offset==next->part->part_offset &&
       ((new_element->part->part_size<next->part->part_size) ||
	(new_element->part->part_size==next->part->part_size && (dont_free_part!=2 || new_element->part->boot_sector<next->part->boot_sector)))))
    {
      if(dont_free_part!=2 &&
	(next!=NULL)&&(next->part->part_offset==new_element->part->part_offset) &&
	(next->part->part_size==new_element->part->part_size) &&
	(next->part->part_type==new_element->part->part_type) &&
	(next->part->upart_type==new_element->part->upart_type || new_element->part->upart_type==UP_UNK))
      { /*CGR 2004/05/31*/
	if(next->part->status==STATUS_DELETED)
	{
	  next->part->status=new_element->part->status;
	}
	if(dont_free_part==0)
	  FREE(new_element->part);
	FREE(new_element);
	return list_part;
      }
      else
      { /* prev new_element next */
	new_element->next=next;
	new_element->prev=prev;
	if(next!=NULL)
	  next->prev=new_element;
	if(prev!=NULL)
	{
	  prev->next=new_element;
	  return list_part;
	}
	return new_element;
      }
    }
    prev=next;
  }
}

int check_list_part(t_list_part *list_part)
{
  t_list_part *prev=NULL;
  t_list_part *parts;
  if((list_part!=NULL) && (list_part->prev!=NULL))
  {
    ecrit_rapport("\ncheck_list_part error: list_part->prev!=NULL\n");
    exit(EXIT_FAILURE);
  }
  ecrit_rapport("check_list_part\n");
  for(parts=list_part;parts!=NULL;parts=parts->next)
  {
    ecrit_rapport("%p %p %p\n",parts->prev, parts, parts->next);
	if(prev!=parts->prev)
	{
	  ecrit_rapport("\ncheck_list_part error: prev!=parts->prev\n");
	  exit(EXIT_FAILURE);
	}
	prev=parts;
  }
  if((prev!=NULL) && (prev->next!=NULL))
  {
    ecrit_rapport("\ncheck_list_part error: prev->next!=NULL\n");
    exit(EXIT_FAILURE);
  }
  return 0;
}

t_list_part *sort_list_part(t_list_part *list_part)
{
  t_list_part *new_list_part=NULL;
  t_list_part *element;
  t_list_part *next;
#ifdef DEBUG
  check_list_part(list_part);
#endif
  for(element=list_part;element!=NULL;element=next)
  {
    next=element->next;
    new_list_part=insert_new_partition_aux(new_list_part,element,0);
  }
#ifdef DEBUG
  check_list_part(new_list_part);
#endif
  return new_list_part;
}

void delete_list_part(t_list_part *list_part)
{
  t_list_part *element;
#ifdef DEBUG
  check_list_part(list_part);
#endif
  /* Libere la memoire */
  element=list_part;
  while(element!=NULL)
  {
    t_list_part *next=element->next;
    FREE(element->part);
    FREE(element);
    element=next;
  }
}

void  partition_reset(t_partition *partition)
{
/* partition->lba=0; Don't reset lba, used by search_part */
  partition->part_size=(uint64_t)0;
  partition->boot_sector=0;
  partition->blocksize=0;
  partition->part_type=P_NO_OS;
  partition->upart_type=UP_UNK;
  partition->status=STATUS_DELETED;
  partition->order=NO_ORDER;
  partition->errcode=BAD_NOERR;
  partition->name[0]='\0';
  partition->info[0]='\0';
  partition->arch=NULL;
}

t_partition *partition_new()
{
  t_partition *partition=(t_partition *)MALLOC(sizeof(*partition));
  partition_reset(partition);
  return partition;
}

t_list_part *element_new(t_partition *part)
{
  t_list_part *new_element=(t_list_part*)MALLOC(sizeof(*new_element));
  new_element->part=part;
  new_element->prev=new_element->next=NULL;
  new_element->to_be_removed=0;
  return new_element;
}

static unsigned int get_geometry_from_list_part_aux(const t_param_disk *disk_car, const t_list_part *list_part, const int debug)
{
  const t_list_part *element;
  unsigned int nbr=0;
  for(element=list_part;element!=NULL;element=element->next)
  {
    t_CHS start;
    t_CHS end;
    offset2CHS(disk_car,element->part->part_offset,&start);
    offset2CHS(disk_car,element->part->part_offset+element->part->part_size-1,&end);
    if(start.sector==1 && start.head<=1 && end.head==disk_car->CHS.head)
    {
      /* Doesn't check if end.sector==disk_car->CHS.sector */
      nbr++;
    }
  }
  if(nbr>0)
  {
    ecrit_rapport("get_geometry_from_list_part_aux head=%u nbr=%u\n",disk_car->CHS.head+1,nbr);
    if(debug>1)
    {
      for(element=list_part;element!=NULL;element=element->next)
      {
	t_CHS start;
	t_CHS end;
	offset2CHS(disk_car,element->part->part_offset,&start);
	offset2CHS(disk_car,element->part->part_offset+element->part->part_size-1,&end);
	if(start.sector==1 && start.head<=1 && end.head==disk_car->CHS.head)
	{
	  aff_part_rapport(disk_car,element->part);
	}
      }
    }
  }
  return nbr;
}

unsigned int get_geometry_from_list_part(const t_param_disk *disk_car, const t_list_part *list_part, const int debug)
{
  const unsigned int head_list[]={8,16,32,64,128,240,255,0};
  unsigned int nbr_max;
  unsigned int nbr;
  unsigned int h_index=0;
  unsigned int head_max=disk_car->CHS.head;
  t_param_disk *new_disk_car=MALLOC(sizeof(*new_disk_car));
  memcpy(new_disk_car,disk_car,sizeof(*new_disk_car));
  nbr_max=get_geometry_from_list_part_aux(new_disk_car, list_part, debug);
  for(h_index=0;head_list[h_index]!=0;h_index++)
  {
    new_disk_car->CHS.head=head_list[h_index]-1;
    nbr=get_geometry_from_list_part_aux(new_disk_car, list_part, debug);
    if(nbr>=nbr_max)
    {
      nbr_max=nbr;
      head_max=new_disk_car->CHS.head;
    }
  }
  FREE(new_disk_car);
  return head_max;
}
