/*

    File: bsd.c

    Copyright (C) 1998-2004 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#include "types.h"
#include "common.h"
#include "bsd.h"
#include "intrf.h"
#include "fnctdsk.h"
int check_BSD(t_param_disk *disk_car,t_partition *partition,const int debug, const unsigned int max_partitions)
{
  unsigned char buffer[8*DEFAULT_SECTOR_SIZE];
  if(disk_car->read(disk_car,sizeof(buffer), &buffer, partition->part_offset))
  { return 1; }
  if(test_BSD(disk_car,(const struct disklabel*)&buffer[0x200],partition,debug,0,max_partitions))
    return 1;
  set_part_name(partition,((const struct disklabel*)(&buffer[0x200]))->d_packname,16);
  if(check_volume_name(partition->name,16))
    partition->name[0]='\0';
  return 0;
}

int test_BSD(t_param_disk *disk_car, const struct disklabel*bsd_header,t_partition *partition,const int debug, const int dump_ind, const unsigned int max_partitions)
{
  if((le32(bsd_header->d_magic) == DISKMAGIC)&&
      (le32(bsd_header->d_magic2)==DISKMAGIC))
  {
    unsigned int i;
    const uint16_t* cp;
    uint16_t crc;
    if(debug)
      ecrit_rapport("\nBSD offset %lu, nbr_part %u, CHS=(%u,%u,%u) ",(long unsigned)(partition->part_offset/disk_car->sector_size),
	  (unsigned int)bsd_header->d_npartitions,(unsigned int)bsd_header->d_ncylinders, (unsigned int)bsd_header->d_ntracks, (unsigned int)bsd_header->d_nsectors);
    if(bsd_header->d_npartitions>max_partitions)
      return 1;
    crc=0;
    for(cp=(const uint16_t*)bsd_header;
	cp<(const uint16_t*)&bsd_header->d_partitions[le16(bsd_header->d_npartitions)];cp++)
      crc^=*cp;
    if(crc==0)
    {
      if(debug)
      {
	ecrit_rapport("CRC Ok\n");
      }
    }
    else
      ecrit_rapport("Bad CRC! CRC must be xor'd by %04X\n",crc);
    for(i=0;i<bsd_header->d_npartitions;i++)
    {
      if(bsd_header->d_partitions[i].p_fstype>0)
      {
	if(debug)
	{
	  /* UFS UFS2 SWAP */
	  ecrit_rapport("BSD %c: ", 'a'+i);
	  switch(bsd_header->d_partitions[i].p_fstype)
	  {
	    case TST_FS_SWAP:
	      ecrit_rapport("swap");
	      break;
	    case TST_FS_BSDFFS:
	      ecrit_rapport("4.2BSD fast filesystem");
	      break;
	    case TST_FS_BSDLFS:
	      ecrit_rapport("4.4BSD log-structured filesystem");
	      break;
	    default:
	      ecrit_rapport("type %02X", bsd_header->d_partitions[i].p_fstype);
	      break;
	  }
	  ecrit_rapport(", offset %9u, size %9u ",
	      (unsigned int)le32(bsd_header->d_partitions[i].p_offset),
	      (unsigned int)le32(bsd_header->d_partitions[i].p_size));
	  aff_LBA2CHS_rapport(disk_car,le32(bsd_header->d_partitions[i].p_offset));
	  ecrit_rapport(" -> ");
	  aff_LBA2CHS_rapport(disk_car,le32(bsd_header->d_partitions[i].p_offset)+le32(bsd_header->d_partitions[i].p_size)-1);
	  ecrit_rapport("\n");
	}
      }
    }
    if(crc)
      return 1;
    if(max_partitions==BSD_MAXPARTITIONS)
      partition->upart_type=UP_FREEBSD;
    else
      partition->upart_type=UP_OPENBSD;
    if(dump_ind!=0)
    {
      dump(stdscr,bsd_header,DEFAULT_SECTOR_SIZE);
    }
    return 0;
  }
  return 1;
}

int recover_BSD(t_param_disk *disk_car, const struct disklabel*bsd_header,t_partition *partition,const int debug, const int dump_ind)
{
  int i;
  int i_max_p_offset=-1;
  if(test_BSD(disk_car,bsd_header,partition,debug,dump_ind,BSD_MAXPARTITIONS)==0)
  {
    for(i=0;i<BSD_MAXPARTITIONS;i++)
    {
      if(bsd_header->d_partitions[i].p_fstype>0)
      {
	if(i_max_p_offset==-1 || le32(bsd_header->d_partitions[i].p_offset)>le32(bsd_header->d_partitions[i_max_p_offset].p_offset))
	  i_max_p_offset=i;
      }
    }
    if(i_max_p_offset>=0)
      partition->part_size=(uint64_t)(le32(bsd_header->d_partitions[i_max_p_offset].p_size) +
	  le32(bsd_header->d_partitions[i_max_p_offset].p_offset) - 1) * disk_car->sector_size - partition->part_offset;
    else
      partition->part_size=0;
    partition->part_type=P_FREEBSD;
    set_part_name(partition,bsd_header->d_packname,16);
    if(check_volume_name(partition->name,16))
      partition->name[0]='\0';
    return 0;
  }
  if(test_BSD(disk_car,bsd_header,partition,debug,dump_ind,OPENBSD_MAXPARTITIONS)==0)
  {
    for(i=0;i<OPENBSD_MAXPARTITIONS;i++)
    {
      if(bsd_header->d_partitions[i].p_fstype>0)
      {
	if(i_max_p_offset==-1 || le32(bsd_header->d_partitions[i].p_offset)>le32(bsd_header->d_partitions[i_max_p_offset].p_offset))
	  i_max_p_offset=i;
      }
    }
    if(i_max_p_offset>=0)
      partition->part_size=(uint64_t)(le32(bsd_header->d_partitions[i_max_p_offset].p_size) +
	  le32(bsd_header->d_partitions[i_max_p_offset].p_offset) - 1) * disk_car->sector_size - partition->part_offset;
    else
      partition->part_size=0;
    partition->part_type=P_OPENBSD;
    set_part_name(partition,bsd_header->d_packname,16);
    if(check_volume_name(partition->name,16))
      partition->name[0]='\0';
    return 0;
  }
  return 1;
}
