[comment {-*- tcl -*-}]
[manpage_begin struct::graph n 2.1]
[copyright {2002 Andreas Kupries <andreas_kupries@users.sourceforge.net>}]
[moddesc   {Tcl Data Structures}]
[titledesc {Create and manipulate directed graph objects}]
[require Tcl 8.2]
[require struct::graph [opt 2.1]]
[require struct::list  [opt 1.5]]
[description]
[para]


A directed graph is a structure containing two collections of
elements, called [emph nodes] and [emph arcs] respectively, together
with a relation ("connectivity") that places a general structure upon
the nodes and arcs.

[para]

Each arc is connected to two nodes, one of which is called the

[emph source] and the other the [emph target]. This imposes a
direction upon the arc, which is said to go from the source to the
target. It is allowed that source and target of an arc are the same
node. Such an arc is called a [emph loop]. Whenever a node is source
or target of an arc both are said to be [emph adjacent]. This extends
into a relation between nodes, i.e. if two nodes are connected through
at least one arc they are said to be [emph adjacent] too.

[para]

Each node can be the source and target for any number of arcs. The
former are called the [emph {outgoing arcs}] of the node, the latter
the [emph {incoming arcs}] of the node. The number of edges in either
set is called the [emph in-] resp. the [emph out-degree] of the node.

[para]

In addition to maintaining the node and arc relationships, this graph
implementation allows any number of keyed values to be associated with
each node and arc.

[para]

[emph Note:] The major version of the package [package struct] has
been changed to version 2.0, due to backward incompatible changes in
the API of this module. Please read the section

[sectref {Changes for 2.0}] for a full list of all changes,
incompatible and otherwise.

[para]

[emph Note:] A C-implementation of the command can be had from the
location [uri http://www.purl.org/NET/schlenker/tcl/cgraph]. See also
[uri http://wiki.tcl.tk/cgraph].  This implementation uses a bit less
memory than the tcl version provided here directly, and is faster.

[para]

The main command of the package is:

[list_begin definitions]

[call [cmd ::struct::graph] [opt [arg graphName]] [opt "[const =]|[const :=]|[const as]|[const deserialize] [arg source]"]]

The command creates a new graph object with an associated global Tcl
command whose name is [arg graphName].  This command may be used to
invoke various operations on the graph.  It has the following general
form:

[list_begin definitions]
[call [cmd graphName] [arg option] [opt [arg "arg arg ..."]]]

[arg Option] and the [arg arg]s determine the exact behavior of the
command.

[list_end]
[nl]

If [arg graphName] is not specified a unique name will be generated by
the package itself. If a [arg source] is specified the new graph will
be initialized to it. For the operators [const =], [const :=], and
[const as] [arg source] is interpreted as the name of another graph
object, and the assignment operator [method =] will be executed. For
[const deserialize] the [arg source] is a serialized graph object and
[method deserialize] will be executed.

[nl]

In other words
[nl]
[example {
    ::struct::graph mygraph = b
}]
[nl]
is equivalent to
[nl]
[example {
    ::struct::graph mygraph
    mygraph = b
}]
[nl]
and 
[nl]
[example {
    ::struct::graph mygraph deserialize $b
}]
[nl]
is equivalent to
[nl]
[example {
    ::struct::graph mygraph
    mygraph deserialize $b
}]


[list_end]

[para]

The following commands are possible for graph objects:

[list_begin definitions]

[call [arg graphName] [method =] [arg sourcegraph]]

This is the assignment operator for graph objects. It copies the graph
contained in the graph object [arg sourcegraph] over the graph data in
[arg graphName]. The old contents of [arg graphName] are deleted by
this operation.

[nl]

This operation is in effect equivalent to
[nl]
[example_begin]
    [arg graphName] [method deserialize] [lb][arg sourcegraph] [method serialize][rb]
[example_end]


[call [arg graphName] [method -->] [arg destgraph]]

This is the reverse assignment operator for graph objects. It copies
the graph contained in the graph object [arg graphName] over the graph
data in the object [arg destgraph].

The old contents of [arg destgraph] are deleted by this operation.

[nl]

This operation is in effect equivalent to
[nl]
[example_begin]
    [arg destgraph] [method deserialize] [lb][arg graphName] [method serialize][rb]
[example_end]


[call [arg graphName] [method append] [arg key] [arg value]]

Appends a [arg value] to one of the keyed values associated with the graph.
Returns the new value given to the attribute [arg key].


[call [arg graphName] [method deserialize] [arg serialization]]

This is the complement to [method serialize]. It replaces graph data
in [arg graphName] with the graph described by the [arg serialization]
value. The old contents of [arg graphName] are deleted by this
operation.


[call [arg graphName] [method destroy]]

Destroy the graph, including its storage space and associated command.


[call [arg graphName] [method {arc append}] [arg arc] [arg key] [arg value]]

Appends a [arg value] to one of the keyed values associated with an
[arg arc]. Returns the new value given to the attribute [arg key].


[call [arg graphName] [method {arc attr}] [arg key]]
[call [arg graphName] [method {arc attr}] [arg key] [option -arcs] [arg list]]
[call [arg graphName] [method {arc attr}] [arg key] [option -glob] [arg globpattern]]
[call [arg graphName] [method {arc attr}] [arg key] [option -regexp] [arg repattern]]

This method retrieves the value of the attribute named [arg key], for
all arcs in the graph (matching the restriction specified via one of
the possible options) and having the specified attribute.

[nl]

The result is a dictionary mapping from arc names to the value of
attribute [arg key] at that arc.

Arcs not having the attribute [arg key], or not passing a
specified restriction, are not listed in the result.

[nl]

The possible restrictions are:

[list_begin opt]
[opt_def -arcs]

The value is a list of arcs. Only the arcs mentioned in this list
are searched for the attribute.

[opt_def -glob]

The value is a glob pattern. Only the arcs in the graph whose names
match this pattern are searched for the attribute.

[opt_def -regexp]

The value is a regular expression. Only the arcs in the graph whose
names match this pattern are searched for the attribute.

[list_end]
[nl]


[call [arg graphName] [method {arc delete}] [arg arc] [opt "[arg arc] ..."]]

Remove the specified arcs from the graph.


[call [arg graphName] [method {arc exists}] [arg arc]]

Return true if the specified [arg arc] exists in the graph.


[call [arg graphName] [method {arc get}] [arg arc] [arg key]]

Return the value associated with the key [arg key] for the [arg arc].


[call [arg graphName] [method {arc getall}] [arg arc] [opt [arg pattern]]]

Returns a dictionary (suitable for use with [lb][cmd {array set}][rb])
for the [arg arc].

If the [arg pattern] is specified only the attributes whose names
match the pattern will be part of the returned dictionary. The pattern
is a [cmd glob] pattern.


[call [arg graphName] [method {arc keys}] [arg arc] [opt [arg pattern]]]

Returns a list of keys for the [arg arc].

If the [arg pattern] is specified only the attributes whose names
match the pattern will be part of the returned list. The pattern is a
[cmd glob] pattern.


[call [arg graphName] [method {arc keyexists}] [arg arc] [arg key]]

Return true if the specified [arg key] exists for the [arg arc].


[call [arg graphName] [method {arc insert}] [arg start] [arg end] [opt [arg child]]]

Insert an arc named [arg child] into the graph beginning at the node
[arg start] and ending at the node [arg end]. If the name of the new
arc is not specified the system will generate a unique name of the
form [emph arc][arg x].


[call [arg graphName] [method {arc lappend}] [arg arc] [arg key] [arg value]]

Appends a [arg value] (as a list) to one of the keyed values
associated with an [arg arc]. Returns the new value given to the
attribute [arg key].


[call [arg graphName] [method {arc rename}] [arg arc] [arg newname]]

Renames the arc [arg arc] to [arg newname]. An error is thrown if
either the arc does not exist, or a arc with name [arg newname] does
exist. The result of the command is the new name of the arc.


[call [arg graphName] [method {arc set}] [arg arc] [arg key] [opt [arg value]]]

Set or get one of the keyed values associated with an arc. 

An arc may have any number of keyed values associated with it.  
If [arg value] is not specified, this command returns the current value assigned to the key; 
if [arg value] is specified, this command assigns that value to the key, and returns
that value.


[call [arg graphName] [method {arc source}] [arg arc]]

Return the node the given [arg arc] begins at.


[call [arg graphName] [method {arc target}] [arg arc]]

Return the node the given [arg arc] ends at.


[call [arg graphName] [method {arc move-source}] [arg arc] [arg newsource]]

Changes the source node of the arc to [arg newsource].


[call [arg graphName] [method {arc move-target}] [arg arc] [arg newtarget]]

Changes the target node of the arc to [arg newtarget].


[call [arg graphName] [method {arc move}] [arg arc] [arg newsource] [arg newtarget]]

Changes both source and target nodes of the arc to [arg newsource],
and [arg newtarget] resp.


[call [arg graphName] [method {arc unset}] [arg arc] [arg key]]

Remove a keyed value from the arc [arg arc]. The method will do
nothing if the [arg key] does not exist.


[call [arg graphName] [method arcs] [opt "-key [arg key]"] [opt "-value [arg value]"] [opt "-filter [arg cmdprefix]"] [opt "-in|-out|-adj|-inner|-embedding [arg nodelist]"]]

Return a list of arcs in the graph. If no restriction is specified a
list containing all arcs is returned. Restrictions can limit the list
of returned arcs based on the nodes that are connected by the arc, on
the keyed values associated with the arc, or both. A general filter
command can be used as well. The restrictions that involve connected
nodes have a list of nodes as argument, specified after the name of
the restriction itself.

[list_begin definitions]
[lst_item [option -in]]

Return a list of all arcs whose target is one of the nodes in the
[arg nodelist].

[lst_item [option -out]]

Return a list of all arcs whose source is one of the nodes in the
[arg nodelist].

[lst_item [option -adj]]

Return a list of all arcs adjacent to at least one of the nodes in the
[arg nodelist]. This is the union of the nodes returned by

[option -in] and [option -out].

[lst_item [option -inner]]

Return a list of all arcs adjacent to two of the nodes in the

[arg nodelist]. This is the set of arcs in the subgraph spawned by the
specified nodes.

[lst_item [option -embedding]]

Return a list of all arcs adjacent to exactly one of the nodes in the
[arg nodelist]. This is the set of arcs connecting the subgraph
spawned by the specified nodes to the rest of the graph.

[lst_item "[option -key] [arg key]"]

Limit the list of arcs that are returned to those arcs that have an
associated key [arg key].

[lst_item "[option -value] [arg value]"]

This restriction can only be used in combination with

[option -key]. It limits the list of arcs that are returned to those
arcs whose associated key [arg key] has the value [arg value].


[lst_item "[option -filter] [arg cmdrefix]"]

Limit the list of arcs that are returned to those arcs that pass the
test. The command in [arg cmdprefix] is called with two arguments, the
name of the graph object, and the name of the arc in question. It is
executed in the context of the caller and has to return a boolean
value. Arcs for which the command returns [const false] are removed
from the result list before it is returned to the caller.

[list_end]
[nl]

The restrictions imposed by either [option -in], [option -out],
[option -adj], [option -inner], or [option -embedded] are applied
first. Specifying more than one of them is illegal.

After that the restrictions set via [option -key]
(and [option -value]) are applied.
Specifying more than one [option -key] (and [option -value]) is
illegal.

Any restriction set through [option -filter] is applied
last. Specifying more than one [option -filter] is illegal.


[call [arg graphName] [method lappend] [arg key] [arg value]]

Appends a [arg value] (as a list) to one of the keyed values
associated with the graph. Returns the new value given to the
attribute [arg key].


[call [arg graphName] [method {node append}] [arg node] [arg key] [arg value]]

Appends a [arg value] to one of the keyed values associated with an
[arg node]. Returns the new value given to the attribute [arg key].


[call [arg graphName] [method {node attr}] [arg key]]
[call [arg graphName] [method {node attr}] [arg key] [option -nodes] [arg list]]
[call [arg graphName] [method {node attr}] [arg key] [option -glob] [arg globpattern]]
[call [arg graphName] [method {node attr}] [arg key] [option -regexp] [arg repattern]]

This method retrieves the value of the attribute named [arg key], for
all nodes in the graph (matching the restriction specified via one of
the possible options) and having the specified attribute.

[nl]

The result is a dictionary mapping from node names to the value of
attribute [arg key] at that node.

Nodes not having the attribute [arg key], or not passing a
specified restriction, are not listed in the result.

[nl]

The possible restrictions are:

[list_begin opt]
[opt_def -nodes]

The value is a list of nodes. Only the nodes mentioned in this list
are searched for the attribute.

[opt_def -glob]

The value is a glob pattern. Only the nodes in the graph whose names
match this pattern are searched for the attribute.

[opt_def -regexp]

The value is a regular expression. Only the nodes in the graph whose
names match this pattern are searched for the attribute.

[list_end]
[nl]


[call [arg graphName] [method {node degree}] [opt -in|-out] [arg node]]

Return the number of arcs adjacent to the specified [arg node]. If one
of the restrictions [option -in] or [option -out] is given only the
incoming resp. outgoing arcs are counted.


[call [arg graphName] [method {node delete}] [arg node] [opt "[arg node] ..."]]

Remove the specified nodes from the graph.  All of the nodes' arcs
will be removed as well to prevent unconnected arcs.


[call [arg graphName] [method {node exists}] [arg node]]

Return true if the specified [arg node] exists in the graph.


[call [arg graphName] [method {node get}] [arg node] [arg key]]

Return the value associated with the key [arg key] for the [arg node].


[call [arg graphName] [method {node getall}] [arg node] [opt [arg pattern]]]

Returns a dictionary (suitable for use with [lb][cmd {array set}][rb])
for the [arg node].

If the [arg pattern] is specified only the attributes whose names
match the pattern will be part of the returned dictionary. The pattern
is a [cmd glob] pattern.


[call [arg graphName] [method {node keys}] [arg node] [opt [arg pattern]]]

Returns a list of keys for the [arg node].

If the [arg pattern] is specified only the attributes whose names
match the pattern will be part of the returned list. The pattern is a
[cmd glob] pattern.


[call [arg graphName] [method {node keyexists}] [arg node] [arg key]]

Return true if the specified [arg key] exists for the [arg node].


[call [arg graphName] [method {node insert}] [opt [arg child]]]

Insert a node named [arg child] into the graph. The nodes has no arcs
connected to it. If the name of the new child is not specified the
system will generate a unique name of the form [emph node][arg x].


[call [arg graphName] [method {node lappend}] [arg node] [arg key] [arg value]]

Appends a [arg value] (as a list) to one of the keyed values
associated with an [arg node]. Returns the new value given to the
attribute [arg key].


[call [arg graphName] [method {node opposite}] [arg node] [arg arc]]

Return the node at the other end of the specified [arg arc], which has
to be adjacent to the given [arg node].


[call [arg graphName] [method {node rename}] [arg node] [arg newname]]

Renames the node [arg node] to [arg newname]. An error is thrown if
either the node does not exist, or a node with name [arg newname] does
exist. The result of the command is the new name of the node.


[call [arg graphName] [method {node set}] [arg node] [arg key] [opt [arg value]]]

Set or get one of the keyed values associated with a node. A node may have any
number of keyed values associated with it.  If [arg value] is not
specified, this command returns the current value assigned to the key;
if [arg value] is specified, this command assigns that value to the
key.


[call [arg graphName] [method {node unset}] [arg node] [arg key]]

Remove a keyed value from the node [arg node]. The method will do
nothing if the [arg key] does not exist.


[call [arg graphName] [method nodes] [opt "-key [arg key]"] [opt "-value [arg value]"] [opt "-filter [arg cmdprefix]"] [opt "-in|-out|-adj|-inner|-embedding [arg nodelist]"]]

Return a list of nodes in the graph. Restrictions can limit the list
of returned nodes based on neighboring nodes, or based on the keyed
values associated with the node. The restrictions that involve
neighboring nodes have a list of nodes as argument, specified after
the name of the restriction itself.

[nl]

The possible restrictions are the same as for method

[method arcs]. The set of nodes to return is computed as the union of
all source and target nodes for all the arcs satisfying the
restrictions as defined for [method arcs].

[nl]

[emph Note] that here the [option -filter] command is applied to the
list of nodes, not arcs.


[call [arg graphName] [method get] [arg key]]

Return the value associated with the key [arg key] for the graph.


[call [arg graphName] [method getall] [opt [arg pattern]]]

Returns a dictionary (suitable for use with [lb][cmd {array set}][rb])
for the whole graph.

If the [arg pattern] is specified only the attributes whose names
match the pattern will be part of the returned dictionary. The pattern
is a [cmd glob] pattern.


[call [arg graphName] [method keys] [opt [arg pattern]]]

Returns a list of keys for the whole graph.

If the [arg pattern] is specified only the attributes whose names
match the pattern will be part of the returned list. The pattern is a
[cmd glob] pattern.


[call [arg graphName] [method keyexists] [arg key]]

Return true if the specified [arg key] exists for the whole graph.


[call [arg graphName] [method serialize] [opt [arg node]...]]

This method serializes the sub-graph spanned up by the [arg node]s. In
other words it returns a tcl [emph value] completely describing that
graph. If no nodes are specified the whole graph will be serialized.

This allows, for example, the transfer of graph objects (or parts
thereof) over arbitrary channels, persistence, etc.

This method is also the basis for both the copy constructor and
the assignment operator.

[nl]

The result of this method has to be semantically identical over all
implementations of the graph interface. This is what will enable us to
copy graph data between different implementations of the same
interface.

[nl]

The result is a list containing a multiple of three items, plus one!
In other words, '[lb]llength $serial[rb] % 3 == 1'. Valid values
include 1, 4, 7, ...

[nl]

The last element of the list is a dictionary containing the attributes
associated with the whole graph.

Regarding the other elements; each triple consists of

[list_begin enum]
[enum]
The name of the node to be described,

[enum]
A dictionary containing the attributes associated with the node,

[enum]
And a list describing all the arcs starting at that node.
[list_end]
[nl]

The elements of the arc list are lists containing three elements each, i.e.

[list_begin enum]
[enum]
The name of the arc described by the element,

[enum]

A reference to the destination node of the arc. This reference is an
integer number given the index of that node in the main serialization
list. As that it is greater than or equal to zero, less than the
length of the serialization, and a multiple of three.

[emph Note:] For internal consistency no arc name may be used twice,
whether in the same node, or at some other node. This is a global
consistency requirement for the serialization.

[enum]
And a dictionary containing the attributes associated with the arc.
[list_end]
[nl]

For all attribute dictionaries they keys are the names of the
attributes, and the values are the values for each name.

[nl]

[emph Note:] The order of the nodes in the serialization has no
relevance, nor has the order of the arcs per node.

[example {
    # A possible serialization for the graph structure
    #
    #        d -----> %2
    #       /         ^ \\
    #      /         /   \\
    #     /         b     \\
    #    /         /       \\
    #  %1 <- a - %0         e
    #    ^         \\      /
    #     \\        c     /
    #      \\        \\  /
    #       \\        v v
    #        f ------ %3
    # is
    #
    # %3 {} {{f 6 {}}} %0 {} {{a 6 {}} {b 9 {}} {c 0 {}}} %1 {} {{d 9 {}}} %2 {} {{e 0 {}}} {}
    #
    # This assumes that the graph has no attribute data.
}]
[nl]


[call [arg graphName] [method set] [arg key] [opt [arg value]]]

Set or get one of the keyed values associated with a graph. A graph
may have any number of keyed values associated with it. If [arg value]
is not specified, this command returns the current value assigned to
the key; if [arg value] is specified, this command assigns that value
to the key.


[call [arg graphName] [method swap] [arg node1] [arg node2]]

Swap the position of [arg node1] and [arg node2] in the graph.


[call [arg graphName] [method unset] [arg key]]

Remove a keyed value from the graph. The method will do nothing if the
[arg key] does not exist.


[call [arg graphName] [method walk] [arg node] [opt "-order [arg order]"] [opt "-type [arg type]"] [opt "-dir [arg direction]"] -command [arg cmd]]

Perform a breadth-first or depth-first walk of the graph starting at
the node [arg node] going in either the direction of outgoing or
opposite to the incoming arcs.

[nl]

The type of walk, breadth-first or depth-first, is determined by the
value of [arg type]; [const bfs] indicates breadth-first,

[const dfs] indicates depth-first.  Depth-first is the default.

[nl]

The order of the walk, pre-order, post-order or both-order is
determined by the value of [arg order]; [const pre] indicates
pre-order, [const post] indicates post-order, [const both] indicates
both-order. Pre-order is the default. Pre-order walking means that a
node is visited before any of its neighbors (as defined by the

[arg direction], see below). Post-order walking means that a parent is
visited after any of its neighbors. Both-order walking means that a
node is visited before [emph and] after any of its neighbors. The
combination of a bread-first walk with post- or both-order is illegal.

[nl]

The direction of the walk is determined by the value of [arg dir];
[const backward] indicates the direction opposite to the incoming
arcs, [const forward] indicates the direction of the outgoing arcs.

[nl]

As the walk progresses, the command [arg cmd] will be evaluated at
each node, with the mode of the call ([const enter] or
[const leave]) and values [arg graphName] and the name of the current
node appended. For a pre-order walk, all nodes are [const enter]ed, for a
post-order all nodes are left. In a both-order walk the first visit of
a node [const enter]s it, the second visit [const leave]s it.

[list_end]


[section {Changes for 2.0}]
The following noteworthy changes have occurred:

[list_begin enum]
[enum]

The API for accessing attributes and their values has been
simplified.

[nl]

All functionality regarding the default attribute "data" has been
removed. This default attribute does not exist anymore. All accesses
to attributes have to specify the name of the attribute in
question. This backward [emph incompatible] change allowed us to
simplify the signature of all methods handling attributes.

[nl]

Especially the flag [option -key] is not required anymore, even more,
its use is now forbidden. Please read the documentation for the arc
and node methods [method set], [method get], [method getall],

[method unset], [method append], [method lappend], [method keyexists]
and [method keys] for a description of the new API's.

[enum]

The methods [method keys] and [method getall] now take an optional
pattern argument and will return only attribute data for keys matching
this pattern.

[enum]

Arcs and nodes can now be renamed. See the documentation for the
methods [method {arc rename}] and [method {node rename}].

[enum]

The structure has been extended with API's for the serialization and
deserialization of graph objects, and a number of operations based on
them (graph assignment, copy construction).

[nl]

Please read the documentation for the methods [method serialize],
[method deserialize], [method =], and [method -->], and the
documentation on the construction of graph objects.

[nl]

Beyond the copying of whole graph objects these new API's also enable
the transfer of graph objects over arbitrary channels and for easy
persistence.


[enum]

A new method, [method attr], was added to both [method arc] and
[method node] allowing the query and retrieval of attribute data
without regard to arc and node relationships.

[enum]

Both methods [method arcs] and [method nodes] have been extended with
the ability to select arcs and nodes based on an arbitrary filtering
criterium.


[list_end]


[keywords graph cgraph serialization]
[manpage_end]

