/*
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "config.h"

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>

#include "tpa-capabilities.h"

#include "tpa-capabilities-private.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CAPABILITIES

#include <tapioca/base/tpa-signals-marshal.h>
#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-errors.h>

/* signal enum */
enum
{
    CAPABILITIES_CHANGED,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

struct _TpaCapabilitiesIfacePrivate {
    GObject *object;
};

/* we need to define the get_type function */
GType
tpa_capabilities_get_type()
{
    static GType object_type = 0;

    if (!object_type) {
        static const GTypeInfo object_info = {
            sizeof(TpaCapabilitiesIface),
            NULL,	/* base init */
            NULL,	/* base finalize */
        };
        object_type =
            g_type_register_static(G_TYPE_INTERFACE,
                "TpaCapabilities",
                &object_info, 0);
    }
    return object_type;
}

void
tpa_capabilities_init (TpaCapabilitiesIface *iface,
                       gpointer data)
{
    VERBOSE ("(%p, %p)", iface, data);

    iface->advertise_capabilities = NULL;
    iface->get_capabilities = NULL;

    /* Interface signals */
    signals[CAPABILITIES_CHANGED] =
        g_signal_new ("capabilities-changed",
                      G_OBJECT_CLASS_TYPE (iface),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      g_cclosure_marshal_VOID__BOXED,
                      G_TYPE_NONE, 1, (dbus_g_type_get_collection ("GPtrArray", (dbus_g_type_get_struct ("GValueArray", G_TYPE_UINT, G_TYPE_STRING, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_INVALID)))));

    iface->priv = g_new0 (TpaCapabilitiesIfacePrivate, 1);
}

void
tpa_capabilities_finalize (GObject *obj)
{
    TpaCapabilitiesIface *iface = TPA_ICAPABILITIES (obj);
    VERBOSE ("(%p)", obj);

    if (iface->priv)
        g_free (iface->priv);
}

/**
 * tpa_capabilities_advertise_capabilities
 *
 * Implements DBus method AdvertiseCapabilities
 * on interface org.freedesktop.Telepathy.Connection.Interface.Capabilities
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_capabilities_advertise_capabilities (GObject *obj,
                                         const gchar **add,
                                         const gchar **remove,
                                         GError **error)
{
    TpaCapabilitiesIface *iface = TPA_ICAPABILITIES (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p)", obj);

    g_return_error_val_if_fail (iface->advertise_capabilities != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->advertise_capabilities (obj, add, remove);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_capabilities_get_capabilities
 *
 * Implements DBus method GetCapabilities
 * on interface org.freedesktop.Telepathy.Connection.Interface.Capabilities
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_capabilities_get_capabilities (GObject *obj,
                                   guint  handle,
                                   GPtrArray **ret,
                                   GError **error)
{
    TpaCapabilitiesIface *iface = TPA_ICAPABILITIES (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p)", obj);

    g_return_error_val_if_fail (iface->get_capabilities != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->get_capabilities (obj, handle, ret);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_capabilities_signal_capabilities_changed
 *
 * Implements DBus signal CapabilitiesChanged
 * on interface org.freedesktop.Telepathy.Connection.Interface.Capabilities
 */
void
tpa_capabilities_signal_capabilities_changed (GObject *obj,
                                              GPtrArray *capabilities)
{
    g_assert (TPA_IS_ICAPABILITIES (obj));

    VERBOSE ("(%p, %p)", obj, capabilities);

    g_signal_emit (obj, signals[CAPABILITIES_CHANGED], 0, capabilities);
}
