/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <stdlib.h>

#include "tpa-profile-priv.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CONNECTION_MANAGER

#include <tapioca/base/tpa-debug.h>

struct _TpaProfilePrivate {
    gchar *protocol;
    gchar *description;
    gchar *name;
    gchar *icon;
    GHashTable *parameters;
    gboolean disposed;
};

enum
{
    ARG_0,
    ARG_NAME,
    ARG_DESCRIPTION,
    ARG_PROTOCOL,
    ARG_ICON
};

G_DEFINE_TYPE(TpaProfile, tpa_profile, G_TYPE_OBJECT)

static void
_get_all_parameters  (gpointer key,
                      gpointer value,
                      gpointer user_data)
{
    g_ptr_array_add (user_data, g_object_ref (value));
}

static GType
tpa_profile_gsignature (const gchar *sig)
{
    /* TODO: Add All types */
    if (g_str_has_prefix (sig, "s"))
        return G_TYPE_STRING;
    else if (g_str_has_prefix (sig, "b"))
        return G_TYPE_BOOLEAN;
    else if (g_str_has_prefix (sig, "q") || g_str_has_prefix (sig, "u"))
        return G_TYPE_UINT;
    else if (g_str_has_prefix (sig, "i") || g_str_has_prefix (sig, "n"))
        return G_TYPE_UINT;
    else
        return G_TYPE_NONE;
}

static void
tpa_profile_dispose (GObject *object)
{
    TpaProfile *self = TPA_PROFILE (object);

    if (self->priv->disposed)
       /* If dispose did already run, return. */
       return;

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    /* Free private data */
    if (self->priv->name)
        g_free (self->priv->name);

    if (self->priv->description)
        g_free (self->priv->description);

    if (self->priv->protocol)
        g_free (self->priv->protocol);

    if (self->priv->icon)
        g_free (self->priv->icon);

    if (self->priv->parameters)
        g_hash_table_destroy (self->priv->parameters);

    G_OBJECT_CLASS (tpa_profile_parent_class)->dispose (object);
}

static void
tpa_profile_set_property (GObject *object,
                          guint prop_id,
                          const GValue *value,
                          GParamSpec *pspec)
{
    TpaProfile *self = TPA_PROFILE (object);

    switch (prop_id) {
        case ARG_NAME:
            self->priv->name = g_value_dup_string (value);
            break;
        case ARG_DESCRIPTION:
            self->priv->description = g_value_dup_string (value);
            break;
        case ARG_PROTOCOL:
            self->priv->protocol = g_value_dup_string (value);
            break;
        case ARG_ICON:
            self->priv->icon = g_value_dup_string (value);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void
tpa_profile_get_property (GObject *object,
                          guint prop_id,
                          GValue *value,
                          GParamSpec *pspec)
{
    TpaProfile *self = TPA_PROFILE (object);

    switch (prop_id) {
        case ARG_NAME:
            g_value_set_string (value, self->priv->name);
            break;
        case ARG_DESCRIPTION:
            g_value_set_string (value, self->priv->description);
            break;
        case ARG_PROTOCOL:
            g_value_set_string (value, self->priv->protocol);
            break;
        case ARG_ICON:
            g_value_set_string (value, self->priv->icon);
            break;
        default:
          G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
          break;
    }
}

static void
tpa_profile_class_init (TpaProfileClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;
    tpa_profile_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaProfilePrivate));

    gobject_class->dispose = tpa_profile_dispose;
    gobject_class->set_property = tpa_profile_set_property;
    gobject_class->get_property = tpa_profile_get_property;

    g_object_class_install_property (gobject_class,
                                     ARG_NAME,
                                     g_param_spec_string ("name",
                                     "name",
                                     "name",
                                     NULL,
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_DESCRIPTION,
                                     g_param_spec_string ("description",
                                     "description",
                                     "description",
                                     NULL,
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_PROTOCOL,
                                     g_param_spec_string ("protocol",
                                     "protocol",
                                     "protocol",
                                     NULL,
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_ICON,
                                     g_param_spec_string ("icon",
                                     "icon",
                                     "icon",
                                     NULL,
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));
}

static void
tpa_profile_init (TpaProfile *self)
{
    self->priv = TPA_PROFILE_GET_PRIVATE (self);
    self->priv->disposed = FALSE;
    self->priv->parameters = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_object_unref);
}

/**
 * tpa_profile_new:
 * @returns: #TpaProfile instance.
 *
 * Create new instance of #TpaProfile.
 */
TpaProfile *
tpa_profile_new (const gchar *name,
                 const gchar *description,
                 const gchar *protocol,
                 const gchar *icon)
{
    TpaProfile *self = NULL;

    self = TPA_PROFILE (g_object_new (TPA_TYPE_PROFILE,
                        "name", name,
                        "description", description,
                        "protocol", protocol,
                        "icon", icon,
                        NULL));

    return self;
}

/**
 * tpa_profile_get_name:
 * @self: #TpaProfile instance.
 * @returns: Profile name.
 *
 * Get profile's name.
 *
 */
const gchar *
tpa_profile_get_name (TpaProfile *self)
{
    g_assert (self);

    return self->priv->name;
}

/**
 * tpa_profile_get_description:
 * @self: #TpaProfile instance.
 * @returns: Profile description.
 *
 * Get profile's description.
 *
 */
const gchar *
tpa_profile_get_description (TpaProfile *self)
{
    g_assert (self);

    return self->priv->description;
}

/**
 * tpa_profile_get_protocol:
 * @self: #TpaProfile instance.
 * @returns: Profile protocol.
 *
 * Get profile's protocol.
 *
 */
const gchar *
tpa_profile_get_protocol (TpaProfile *self)
{
    g_assert (self);

    return self->priv->protocol;
}

/**
 * tpa_profile_get_icon:
 * @self: #TpaProfile instance.
 * @returns: Profile protocol.
 *
 * Get profile's protocol.
 *
 */
const gchar *
tpa_profile_get_icon (TpaProfile *self)
{
    g_assert (self);

    return self->priv->icon;
}

/**
 * tpa_profile_get_parameter:
 * @self: #TpaProfile instance.
 * @name: New Parameter name.
 * @returns: FALSE if fail.
 *
 * Get parameter by name.
 *
 */
TpaParameter *
tpa_profile_get_parameter (TpaProfile *self,
                           const gchar *name)
{
    TpaParameter *parameter = NULL;

    g_assert (self);
    g_return_val_if_fail (name != NULL, NULL);

    parameter = g_hash_table_lookup (self->priv->parameters, name);

    return g_object_ref (parameter);
}

/**
 * tpa_profile_add_parameter_signature:
 * @self: #TpaProfile instance.
 * @name: New Parameter name.
 * @signature: Parameter type in form of dbus signature.
 * @returns: new #TpaParameter instance or NULL if fail.
 *
 * Add parameter name, if a parameter with same
 * name was already added it will fail.
 *
 */
TpaParameter *
tpa_profile_add_parameter_signature (TpaProfile *self,
                                     const gchar *name,
                                     const gchar *signature)
{
    GType type = G_TYPE_NONE;

    VERBOSE ("(%p, %s, %s)", self, name, signature);

    g_return_val_if_fail (signature != NULL, FALSE);

    type = tpa_profile_gsignature (signature);

    return tpa_profile_add_parameter (self, name, type);
}

/**
 * tpa_profile_add_parameter:
 * @self: #TpaProfile instance.
 * @name: Parameter name.
 * @type: Parameter #GType.
 * @returns: new #TpaParameter instance or NULL if fail.
 *
 * Add parameter name, if a parameter with same
 * name was already added it will fail.
 *
 */
TpaParameter *
tpa_profile_add_parameter (TpaProfile *self,
                           const gchar *name,
                           GType type)
{
    TpaParameter *parameter = NULL;

    VERBOSE ("(%p, %s, %s)", self, name, g_type_name (type));

    parameter = tpa_parameter_new (name, type);

    g_return_val_if_fail (parameter != NULL, NULL);

    g_hash_table_insert (self->priv->parameters, g_strdup (name), parameter);

    return parameter;
}

/**
 * tpa_profile_remove_parameter:
 * @self: #TpaProfile instance.
 * @name: New Parameter name.
 * @returns: FALSE if fail.
 *
 * Remove parameter by name.
 *
 */
gboolean
tpa_profile_remove_parameter (TpaProfile *self,
                              const gchar *name)
{
    VERBOSE ("(%p, %s)", self, name);

    return g_hash_table_remove (self->priv->parameters, name);
}

/**
 * tpa_profile_get_all_parameters:
 * @self: #TpaProfile instance.
 * @returns: #GPtrArray with profile names.
 *
 * Get all profile names available.
 *
 */
GPtrArray *
tpa_profile_get_all_parameters (TpaProfile *self)
{
    GPtrArray *parameters;

    VERBOSE ("(%p)", self);

    parameters = g_ptr_array_new ();
    g_hash_table_foreach (self->priv->parameters, _get_all_parameters, parameters);

    return parameters;
}

/**
 * tpa_profile_get_mandatory_parameters:
 * @self: #TpaProfile instance.
 * @returns: #GPtrArray with profile names.
 *
 * Get all profile names available that are mandatory.
 *
 */
GPtrArray *
tpa_profile_get_mandatory_parameters (TpaProfile *self)
{
    GPtrArray *parameters;
    guint i;

    VERBOSE ("(%p)", self);

    parameters = tpa_profile_get_all_parameters (self);

    for (i = 0; i < parameters->len; i++) {
        TpaParameter *parameter = g_ptr_array_index (parameters, i);
        if (!tpa_parameter_has_required_flag (parameter) ||
             !tpa_parameter_get_default_value_as_string (parameter)) {
            g_ptr_array_remove_index(parameters, i);
            g_object_unref (parameter);
            --i;
        }
    }

    return parameters;
}

/**
 * tpa_profile_get_registration_parameters:
 * @self: #TpaProfile instance.
 * @returns: #GPtrArray with profile names.
 *
 * Get all profile names available that has register flag.
 *
 */
GPtrArray *
tpa_profile_get_registration_parameters (TpaProfile *self)
{
    GPtrArray *parameters;
    guint i;

    VERBOSE ("(%p)", self);
    parameters = tpa_profile_get_all_parameters (self);

    for (i = 0; i < parameters->len; i++) {
        TpaParameter *parameter = g_ptr_array_index (parameters, i);
        if (!tpa_parameter_has_register_flag (parameter)) {
            g_ptr_array_remove_index(parameters, i);
            g_object_unref (parameter);
            --i;
        }
    }

    return parameters;
}

/**
 * tpa_profile_get_formatted_table:
 * @self: #TpaProfile instance
 * @returns: formatted profile into a GHashTable or NULL if
 * fail.
 *
 * Function to format profile into a new GHashTable to be
 * sent to connection managers.
 *
 */
GHashTable *
tpa_profile_get_formatted_table (TpaProfile *self)
{
    GPtrArray *parameters = NULL;
    GHashTable *table = NULL;
    guint i = 0;

    VERBOSE ("(%p)", self);
    parameters = tpa_profile_get_all_parameters (self);
    if (parameters) {
        table = g_hash_table_new_full (g_str_hash, g_str_equal, NULL, NULL);
        for (i = 0; i < parameters->len; i++) {
            TpaParameter *parameter = g_ptr_array_index (parameters, i);
            const gchar *name = tpa_parameter_get_name (parameter);
            gchar *value = tpa_parameter_get_value_as_string (parameter);
            gchar *default_value = tpa_parameter_get_default_value_as_string (parameter);
            /* Add profile that has value and does not match with default value*/
            if (value && !g_str_equal (value, "") && !g_str_equal (value, "0") &&
                (!default_value || !g_str_equal (value, default_value))) {
                GValue *gvalue = tpa_parameter_get_value (parameter);
                g_hash_table_insert (table, (gpointer) name, gvalue);
                if ((g_str_equal ("password", name)) || (g_str_equal ("passwd", name))) {
                    INFO ("parameter %s value %s added", name, "*******");
                }
                else {
                    INFO ("parameter %s value %s added", name, value);
                }
            }
        }
    }

    return table;
}
