/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#include "tpa-channel-target-priv.h"
#include "tpa-channel-priv.h"
#include "tpa-media-stream.h"
#include "tpa-thread.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CHANNEL

#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-channel-bindings.h>
#include <tapioca/base/tpa-signals-marshal.h>

struct _TpaMediaStreamPrivate {
    TpaChannel *channel;
    DBusGProxy *stream_engine;
    DBusGProxy *channel_handler;
    gboolean disposed;
};

G_DEFINE_TYPE(TpaMediaStream, tpa_media_stream, TPA_TYPE_OBJECT)

/* Signals */
enum {
    RECEIVING,
    LAST_SIGNAL
};

/*static guint tpa_media_stream_signals[LAST_SIGNAL] = { 0 };*/

static GObject*
tpa_media_stream_constructor (GType type,
                              guint n_construct_params,
                              GObjectConstructParam *construct_params)
{
    GObject *object;
    TpaMediaStream *self;
    DBusGConnection *bus;

    object = G_OBJECT_CLASS (tpa_media_stream_parent_class)->constructor
                            (type, n_construct_params, construct_params);
    self = TPA_MEDIA_STREAM (object);

    bus = tpa_thread_get_bus ();

    if (!bus) {
        ERROR ("failed to open connection to dbus");
        return NULL;
    }

    self->priv->channel_handler = dbus_g_proxy_new_for_name (bus,
                                  "org.freedesktop.Telepathy.StreamEngine",
                                  "/org/freedesktop/Telepathy/StreamEngine",
                                  "org.freedesktop.Telepathy.ChannelHandler");
    tpa_object_add_proxy (TPA_OBJECT (self), self->priv->channel_handler);

    self->priv->stream_engine = dbus_g_proxy_new_for_name (bus,
                                "org.freedesktop.Telepathy.StreamEngine",
                                "/org/freedesktop/Telepathy/StreamEngine",
                                "org.freedesktop.Telepathy.StreamEngine");
    tpa_object_add_proxy (TPA_OBJECT (self), self->priv->stream_engine);

    return object;
}

static void
tpa_media_stream_dispose (GObject *object)
{
    TpaMediaStream *self = TPA_MEDIA_STREAM (object);

    if (self->priv->disposed)
       /* If dispose did already run, return. */
       return;

    if (self->priv->channel_handler)
        g_object_unref (self->priv->channel_handler);
    if (self->priv->stream_engine)
        g_object_unref (self->priv->stream_engine);

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    G_OBJECT_CLASS (tpa_media_stream_parent_class)->dispose (object);
}

static void
tpa_media_stream_class_init (TpaMediaStreamClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;
    tpa_media_stream_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaMediaStreamPrivate));

    gobject_class->dispose = tpa_media_stream_dispose;
    gobject_class->constructor = tpa_media_stream_constructor;
}

static void
tpa_media_stream_init (TpaMediaStream *self)
{
    self->priv = TPA_MEDIA_STREAM_GET_PRIVATE (self);
    self->priv->channel_handler = NULL;
    self->priv->stream_engine = NULL;
    self->priv->disposed = FALSE;
}

TpaMediaStream *
tpa_media_stream_new (TpaObject *object)
{
    TpaMediaStream *self = NULL;

    g_return_val_if_fail (object != NULL, NULL);

    /* Pass givin arguments to constructor */
    self = TPA_MEDIA_STREAM (g_object_new (TPA_TYPE_MEDIA_STREAM,
                                           "object", object,
                                           NULL));

    return self;
}

gboolean
tpa_media_stream_handle_channel (TpaMediaStream *self,
                                 TpaChannel *channel)
{
    TpaUserContact *user;
    DBusGProxy *proxy;
    DBusGProxy *conn_proxy;
    const gchar *service_name;
    const gchar *conn_path;
    const gchar *channel_path;
    GError *error = NULL;

    VERBOSE ("(%p, %p)", self, channel);
    g_assert (self);
    g_return_val_if_fail (channel, FALSE);

    g_object_get (channel, "user", &user, NULL);
    proxy = tpa_object_get_proxy (TPA_OBJECT (self), TPA_INTERFACE_CHANNEL);
    conn_proxy = tpa_object_get_proxy (TPA_OBJECT (user), TPA_INTERFACE_CONNECTION);

    service_name = dbus_g_proxy_get_bus_name (conn_proxy);
    conn_path = dbus_g_proxy_get_path (conn_proxy);
    channel_path = dbus_g_proxy_get_path (proxy);
    dbus_g_proxy_call (self->priv->channel_handler, "HandleChannel", &error,
                       G_TYPE_STRING, service_name,
                       DBUS_TYPE_G_OBJECT_PATH, conn_path,
                       G_TYPE_STRING, TPA_INTERFACE_STREAMED_MEDIA,
                       DBUS_TYPE_G_OBJECT_PATH, channel_path,
                       G_TYPE_UINT, 0,
                       G_TYPE_UINT, 0,
                       G_TYPE_INVALID, G_TYPE_INVALID);

    if (error) {
        ERROR ("%s", error->message);
        g_error_free (error);
        VERBOSE ("return false");
        return FALSE;
    }

    INFO ("media stream %p handling channel %p", self, channel);
    self->priv->channel = channel;

    g_object_unref (proxy);
    g_object_unref (conn_proxy);
    VERBOSE ("return true");
    return TRUE;
}

void
tpa_media_stream_set_audio_volume (TpaMediaStream *self,
                                   guint stream,
                                   guint volume)
{
    const gchar *path;
    GError *error = NULL;

    VERBOSE ("(%p, %d, %d)", self, stream, volume);
    g_assert (self);

    path = tpa_channel_get_object_path (TPA_CHANNEL (self->priv->channel));
    dbus_g_proxy_call (self->priv->stream_engine, "SetOutputVolume", &error,
                       DBUS_TYPE_G_OBJECT_PATH, path,
                       G_TYPE_UINT, stream,
                       G_TYPE_UINT, volume,
                       G_TYPE_INVALID, G_TYPE_INVALID);
    if (error) {
        ERROR ("%s", error->message);
        g_error_free (error);
    }

    VERBOSE ("return");
}

void
tpa_media_stream_mute_input (TpaMediaStream *self,
                             guint stream,
                             gboolean mute)
{
    const gchar *path;
    GError *error = NULL;

    VERBOSE ("(%p, %d, %d)", self, stream, mute);
    g_assert (self);

    path = tpa_channel_get_object_path (TPA_CHANNEL (self->priv->channel));
    dbus_g_proxy_call (self->priv->stream_engine, "MuteInput", &error,
                       DBUS_TYPE_G_OBJECT_PATH, path,
                       G_TYPE_UINT, stream,
                       G_TYPE_BOOLEAN, mute,
                       G_TYPE_INVALID, G_TYPE_INVALID);
    if (error) {
        ERROR ("%s", error->message);
        g_error_free (error);
    }

    VERBOSE ("return");
}

void
tpa_media_stream_mute_output (TpaMediaStream *self,
                              guint stream,
                              gboolean mute)
{
    const gchar *path;
    GError *error = NULL;

    VERBOSE ("(%p, %d, %d)", self, stream, mute);
    g_assert (self);

    path = tpa_channel_get_object_path (TPA_CHANNEL (self->priv->channel));
    dbus_g_proxy_call (self->priv->stream_engine, "MuteOutput", &error,
                       DBUS_TYPE_G_OBJECT_PATH, path,
                       G_TYPE_UINT, stream,
                       G_TYPE_BOOLEAN, mute,
                       G_TYPE_INVALID, G_TYPE_INVALID);
    if (error) {
        ERROR ("%s", error->message);
        g_error_free (error);
    }

    VERBOSE ("return");
}

void
tpa_media_stream_add_video_output (TpaMediaStream *self,
                                   guint stream,
                                   guint window)
{
    const gchar *path;
    GError *error = NULL;

    VERBOSE ("(%p, %d, %d)", self, stream, window);
    g_assert (self);

    INFO ("adding video output of %d to stream %d", window, stream);
    path = tpa_channel_get_object_path (TPA_CHANNEL (self->priv->channel));
    dbus_g_proxy_call (self->priv->stream_engine, "SetOutputWindow", &error,
                       DBUS_TYPE_G_OBJECT_PATH, path,
                       G_TYPE_UINT, stream,
                       G_TYPE_UINT, window,
                       G_TYPE_INVALID, G_TYPE_INVALID);
    if (error) {
        ERROR ("%s", error->message);
        g_error_free (error);
    }

    VERBOSE ("return");
}

void
tpa_media_stream_remove_video_output (TpaMediaStream *self,
                                      guint window)
{
    VERBOSE ("(%p, %d)", self, window);
    g_assert (self);
    VERBOSE ("return");
}

void
tpa_media_stream_add_video_input (TpaMediaStream *self,
                                  guint window)
{
    GError *error = NULL;

    VERBOSE ("(%p, %d)", self, window);
    g_assert (self);

    INFO ("adding video input %d", window);
    dbus_g_proxy_call (self->priv->stream_engine, "AddPreviewWindow", &error,
                       G_TYPE_UINT, window,
                       G_TYPE_INVALID, G_TYPE_INVALID);
    if (error) {
        ERROR ("%s", error->message);
        g_error_free (error);
    }

   VERBOSE ("return");
}

void
tpa_media_stream_remove_video_input (TpaMediaStream *self,
                                     guint window)
{
    GError *error = NULL;

    VERBOSE ("(%p, %d)", self, window);
    g_assert (self);

    INFO ("removing video input %d", window);
    dbus_g_proxy_call (self->priv->stream_engine, "RemovePreviewWindow", &error,
                       G_TYPE_UINT, window,
                       G_TYPE_INVALID, G_TYPE_INVALID);
    if (error) {
        ERROR ("%s", error->message);
        g_error_free (error);
    }

    VERBOSE ("return");
}

