/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#include "tpa-contact-priv.h"
#include "tpa-channel-target-priv.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CHANNEL

#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-channel-bindings.h>
#include <tapioca/base/tpa-signals-marshal.h>

/* Signals */
enum {
    SUBSCRIPTION_STATUS_CHANGED,
    AUTHORIZATION_STATUS_CHANGED,
    HIDE_STATUS_CHANGED,
    BLOCK_STATUS_CHANGED,
    LAST_SIGNAL
};

struct _TpaContactPrivate {
    DBusGProxy *proxy_subscribe;
    DBusGProxy *proxy_publish;
    DBusGProxy *proxy_hide;
    DBusGProxy *proxy_deny;
    TpaSubscriptionStatus subscription;
    TpaAuthorizationStatus authorization;
    gboolean hidden;
    gboolean blocked;
    gboolean disposed;
};

static guint tpa_contact_signals[LAST_SIGNAL] = { 0 };
static const gchar *subscription_status[TPA_SUBSCRIPTION_STATUS_CURRENT + 1] =
    { "none", "remote-pending", "local-pending" , "current" };
static const gchar *authorization_status[TPA_AUTHORIZATION_STATUS_AUTHORIZED + 1] =
    { "non-existent", "local-pending", "authorized" };

static void         tpa_contact_change_group            (TpaContact *self,
                                                         DBusGProxy *proxy,
                                                         gboolean value);

G_DEFINE_TYPE(TpaContact, tpa_contact, TPA_TYPE_CONTACT_BASE)

static GObject*
tpa_contact_constructor (GType type,
                         guint n_construct_params,
                         GObjectConstructParam *construct_params)
{
    GObject *object;
    TpaContact *self;

    object = G_OBJECT_CLASS (tpa_contact_parent_class)->constructor
                            (type, n_construct_params, construct_params);
    self = TPA_CONTACT (object);
    self->priv->proxy_subscribe = tpa_object_get_proxy (TPA_OBJECT (self), TPA_INTERFACE_LIST_SUBSCRIBE);
    self->priv->proxy_publish = tpa_object_get_proxy (TPA_OBJECT (self), TPA_INTERFACE_LIST_PUBLISH);
    self->priv->proxy_hide = tpa_object_get_proxy (TPA_OBJECT (self), TPA_INTERFACE_LIST_HIDE);
    self->priv->proxy_deny = tpa_object_get_proxy (TPA_OBJECT (self), TPA_INTERFACE_LIST_DENY);

    return object;
}

static void
tpa_contact_dispose (GObject *object)
{
    TpaContact *self = TPA_CONTACT (object);

    if (self->priv->disposed)
       /* If dispose did already run, return. */
       return;

    if (self->priv->proxy_subscribe)
        g_object_unref (self->priv->proxy_subscribe);
    if (self->priv->proxy_publish)
        g_object_unref (self->priv->proxy_publish);
     if (self->priv->proxy_hide)
        g_object_unref (self->priv->proxy_hide);
     if (self->priv->proxy_deny)
        g_object_unref (self->priv->proxy_deny);

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    G_OBJECT_CLASS (tpa_contact_parent_class)->dispose (object);
}

static void
tpa_contact_class_init (TpaContactClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;
    tpa_contact_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaContactPrivate));

    gobject_class->constructor = tpa_contact_constructor;
    gobject_class->dispose = tpa_contact_dispose;

    /* Let's create our signals */
    tpa_contact_signals[SUBSCRIPTION_STATUS_CHANGED] = g_signal_new ("subscription-status-changed",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL,
        NULL,
        g_cclosure_marshal_VOID__UINT,
        G_TYPE_NONE,
        1,
        G_TYPE_UINT);

    tpa_contact_signals[AUTHORIZATION_STATUS_CHANGED] = g_signal_new ("authorization-status-changed",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL,
        NULL,
        g_cclosure_marshal_VOID__UINT,
        G_TYPE_NONE,
        1,
        G_TYPE_UINT);

    tpa_contact_signals[HIDE_STATUS_CHANGED] = g_signal_new ("hide-status-changed",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL,
        NULL,
        g_cclosure_marshal_VOID__BOOLEAN,
        G_TYPE_NONE,
        1,
        G_TYPE_BOOLEAN);

    tpa_contact_signals[BLOCK_STATUS_CHANGED] = g_signal_new ("block-status-changed",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL,
        NULL,
        g_cclosure_marshal_VOID__BOOLEAN,
        G_TYPE_NONE,
        1,
        G_TYPE_BOOLEAN);
}

static void
tpa_contact_init (TpaContact *self)
{
    self->priv = TPA_CONTACT_GET_PRIVATE (self);
    self->priv->disposed = FALSE;
    self->priv->hidden = FALSE;
    self->priv->blocked = FALSE;
    self->priv->subscription = TPA_SUBSCRIPTION_STATUS_NONE;
    self->priv->authorization = TPA_AUTHORIZATION_STATUS_NON_EXISTENT;
    self->priv->proxy_subscribe = NULL;
    self->priv->proxy_publish = NULL;
    self->priv->proxy_hide = NULL;
    self->priv->proxy_deny = NULL;
}

static void
tpa_contact_change_group (TpaContact *self,
                          DBusGProxy *proxy,
                          gboolean value)
{
    GError *error = NULL;
    GArray *array = NULL;
    TpaHandle *handle = NULL;
    guint id = 0;
    const gchar *message = "";

    VERBOSE ("(%p, %p, %s)", self, proxy, value ? "true" : "false");
    g_return_if_fail (proxy != NULL);

    array = g_array_new (TRUE, TRUE, sizeof (guint));
    handle = tpa_channel_target_get_handle (TPA_CHANNEL_TARGET (self));
    id = tpa_handle_get_id (handle);
    g_array_append_val (array, id);

    if (value) {
        if (!org_freedesktop_Telepathy_Channel_Interface_Group_add_members (proxy,
            array, message, &error)
            || error) {
            ERROR ("%s", error->message);
            g_error_free (error);
        }
    }
    else {
        if (!org_freedesktop_Telepathy_Channel_Interface_Group_remove_members (proxy,
            array, message, &error)
            || error) {
            ERROR ("%s", error->message);
            g_error_free (error);
        }
    }
    VERBOSE ("return");
}

/**
 * tpa_contact_new:
 * @handle: #TpaHandle instance.
 * @object: #TpaObject instance.
 * @returns: new #TpaContact instance.
 *
 * Create a new instance of #TpaContact.
 */
TpaContact *
tpa_contact_new (TpaHandle *handle,
                 TpaObject *object)
{
    TpaContact *self;

    VERBOSE ("(%p, %p)", handle, object);
    g_return_val_if_fail (handle != NULL, NULL);
    g_return_val_if_fail (object != NULL, NULL);

    self = TPA_CONTACT (g_object_new (TPA_TYPE_CONTACT, "handle", handle, "object", object, NULL));

    VERBOSE ("return %p", self);
    return self;
}

/**
 * tpa_contact_get_subscription_status:
 * @self: #TpaContact instance
 * @returns: #TpaSubscriptionStatus status
 *
 * Get subscription status of the contact.
 */
TpaSubscriptionStatus
tpa_contact_get_subscription_status (TpaContact *self)
{
    VERBOSE ("(%p)", self);
    g_assert (self);

    VERBOSE ("return %d", self->priv->subscription);
    return self->priv->subscription;
}

/**
 * tpa_contact_set_authorization_status:
 * @self: #TpaContact instance
 * @status: #TpaSubscriptionStatus status
 *
 * Private function for ContactList set subscription status.
 */
void
tpa_contact_set_subscription_status (TpaContact *self,
                                     TpaSubscriptionStatus status)
{
    const gchar *uri;

    VERBOSE ("(%p, %d)", self, status);
    g_assert (self);

    if (status != self->priv->subscription) {
        self->priv->subscription = status;
        if (DEBUGGING) {
            uri = tpa_channel_target_get_uri (TPA_CHANNEL_TARGET (self));
            INFO ("[subscription-status-changed %s] %s", uri, subscription_status[status]);
        }
        g_signal_emit (self, tpa_contact_signals[SUBSCRIPTION_STATUS_CHANGED], 0, status);
    }
    VERBOSE ("return");
}

/**
 * tpa_contact_subscribe:
 * @self: #TpaContact instance
 * @value: subsrcibe status
 *
 * Subscribe or unsubscribe contact.
 */
void
tpa_contact_subscribe (TpaContact *self,
                       gboolean value)
{
    VERBOSE ("(%p, %s)", self, value ? "TRUE" : "FALSE");
    g_assert (self);

    if (self->priv->proxy_subscribe) {
        tpa_contact_change_group (self, self->priv->proxy_subscribe, value);
    }
    else {
        TpaSubscriptionStatus status = value ? TPA_SUBSCRIPTION_STATUS_CURRENT :
            TPA_SUBSCRIPTION_STATUS_NONE;
        tpa_contact_set_subscription_status (self, status);
    }
    VERBOSE ("return");
}

/**
 * tpa_contact_get_authorization_status:
 * @self: #TpaContact instance
 * @returns: #TpaAuthorizationStatus status
 *
 * Get authorization status of the contact.
 */
TpaAuthorizationStatus
tpa_contact_get_authorization_status (TpaContact *self)
{
    VERBOSE ("(%p)", self);
    g_assert (self);

    VERBOSE ("return %d", self->priv->authorization);
    return self->priv->authorization;
}

/**
 * tpa_contact_set_authorization_status:
 * @self: #TpaContact instance
 * @status: #TpaAuthorizationStatus status
 *
 * Private function for ContactList set authorization status.
 */
void
tpa_contact_set_authorization_status (TpaContact *self,
                                      TpaAuthorizationStatus status)
{
    const gchar *uri;

    VERBOSE ("(%p, %d)", self, status);
    g_assert (self);

    if (status != self->priv->authorization) {
        self->priv->authorization = status;
        if (DEBUGGING) {
            uri = tpa_channel_target_get_uri (TPA_CHANNEL_TARGET (self));
            INFO ("[authorization-status-changed %s] %s", uri, authorization_status[status]);
        }
        g_signal_emit (self, tpa_contact_signals[AUTHORIZATION_STATUS_CHANGED], 0, status);
    }
    VERBOSE ("return");
}

/**
 * tpa_contact_authorize:
 * @self: #TpaContact instance
 * @value: authorize status.
 *
 * Authorize or deny contact.
 */
void
tpa_contact_authorize (TpaContact *self,
                       gboolean value)
{
    VERBOSE ("(%p, %s)", self, value ? "TRUE" : "FALSE");
    g_assert (self);

    tpa_contact_change_group (self, self->priv->proxy_publish, value);
    if (self->priv->proxy_publish)
       tpa_contact_change_group (self, self->priv->proxy_publish, value);

    else {
        TpaAuthorizationStatus status = value ? TPA_AUTHORIZATION_STATUS_AUTHORIZED :
            TPA_AUTHORIZATION_STATUS_NON_EXISTENT;
        tpa_contact_set_authorization_status (self, status);
    }
    VERBOSE ("return");
}

/**
 * tpa_contact_hide:
 * @self: #TpaContact instance
 * @value: hide status
 *
 * Set contact hide status.
 */
void
tpa_contact_hide (TpaContact *self,
                  gboolean value)
{
    VERBOSE ("(%p, %s)", self, value ? "TRUE" : "FALSE");
    g_assert (self);

    if (self->priv->proxy_hide)
        tpa_contact_change_group (self, self->priv->proxy_hide, value);
    else
        tpa_contact_set_hide (self, value);
    VERBOSE ("return");
}

/**
 * tpa_contact_set_hide:
 * @self: #TpaContact instance
 * @value: hide value
 *
 * Private function for ContactList set hide status.
 */
void
tpa_contact_set_hide (TpaContact *self,
                      gboolean value)
{
    const gchar *uri;

    VERBOSE ("(%p, %s)", self, value ? "true" : "false");
    g_assert (self);

    if (value != self->priv->hidden) {
        self->priv->hidden = value;
        if (DEBUGGING) {
            uri = tpa_channel_target_get_uri (TPA_CHANNEL_TARGET (self));
            INFO ("[hide-status-changed %s] %s", uri, value ? "true" : "false");
        }
        g_signal_emit (self, tpa_contact_signals[HIDE_STATUS_CHANGED], 0, value);
    }
    VERBOSE ("return");
}

/**
 * tpa_contact_is_hidden:
 * @self: #TpaContact instance
 * @returns: hide status
 *
 * Check whether contact is hidden or not.
 */
gboolean
tpa_contact_is_hidden (TpaContact *self)
{
    VERBOSE ("(%p)", self);
    g_assert (self);

    VERBOSE ("return %s", self->priv->hidden ? "true" : "false");
    return self->priv->hidden;
}

/**
 * tpa_contact_block:
 * @self: #TpaContact instance.
 * @value: boolean value.
 *
 * Block or unblock contact.
 */
void
tpa_contact_block (TpaContact *self,
                   gboolean value)
{
    VERBOSE ("(%p, %s)", self, value ? "TRUE" : "FALSE");
    g_assert (self);

    if (self->priv->proxy_deny)
        tpa_contact_change_group (self, self->priv->proxy_deny, value);
    else
        tpa_contact_set_block (self, value);
    VERBOSE ("return");
}

/**
 * tpa_contact_set_hide:
 * @self: #TpaContact instance
 * @value: boolean value
 *
 * Private function for ContactList set block status.
 */
void
tpa_contact_set_block (TpaContact *self,
                       gboolean value)
{
    const gchar *uri;

    VERBOSE ("(%p, %s)", self, value ? "true" : "false");
    g_assert (self);

    if (value != self->priv->blocked) {
        self->priv->blocked = value;
        if (DEBUGGING) {
            uri = tpa_channel_target_get_uri (TPA_CHANNEL_TARGET (self));
            INFO ("[block-status-changed %s] %s", uri, value ? "true" : "false");
        }
        g_signal_emit (self, tpa_contact_signals[BLOCK_STATUS_CHANGED], 0, value);
    }
    VERBOSE ("return");
}

/**
 * tpa_contact_is_blocked:
 * @self: #TpaContact instance
 * @returns: TRUE if contact could be blocked
 *
 * Checks whether contact is blocked or not.
 */
gboolean
tpa_contact_is_blocked (TpaContact *self)
{
    VERBOSE ("(%p)", self);
    g_assert (self);

    VERBOSE ("return %s", self->priv->blocked ? "true" : "false");
    return self->priv->blocked;
}
