/*
 * Tagged Collection document for a Document-View editing architecture
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#pragma implementation

#include "TagcollDocument.h"

#include <FilterChain.h>
#include <Implications.h>
#include <TagcollFilter.h>
#include <DerivedTags.h> 
#include <InputMerger.h>
#include <ItemGrouper.h>
#include <TagcollSerializer.h>
#include <TagcollBuilder.h>
#include <TagcollParser.h>
#include <ExpressionFilter.h>
#include <StdioParserInput.h>
#include <StringParserInput.h>

#include "Environment.h"

#include <errno.h>

using namespace std;

template class TagcollDocument<std::string>;


const char* fn_tagdb = "/var/lib/debtags/package-tags";
const char* fn_impls = "/var/lib/debtags/implications";
const char* fn_deriv = "/var/lib/debtags/derived-tags";

// TagcollConsumer that builds a tagged collection 
class TagcollRebuilder : public TagcollConsumer<int>
{
protected:
	TagCollection<int> coll;

public:
	TagcollRebuilder() throw () {}
	virtual ~TagcollRebuilder() throw () {}

	virtual void consume(const int& item) throw ()
	{
		coll.add(item);
	}
	
	virtual void consume(const int& item, const OpSet<string>& tags) throw ()
	{
		coll.add(tags, item);
	}

	virtual void consume(const OpSet<int>& items) throw ()
	{
		coll.add(items);
	}
	
	virtual void consume(const OpSet<int>& items, const OpSet<string>& tags) throw ()
	{
		coll.add(tags, items);
	}

	// Retrieve the resulting collection
	TagCollection<int> collection() throw () { return coll; }
	const TagCollection<int> collection() const throw () { return coll; }
};


static void readCollection(const string& file, TagcollConsumer<string>& consumer)
	throw (FileException, ParserException)
{
	StdioParserInput in(file);
	TagcollParser::parseTagcoll(in, consumer);
}

static void readDerivedTags(const string& file, DerivedTagList& derivedTags)
	throw (FileException)
{
	if (file == "-")
	{
		StdioParserInput input(stdin, "<stdin>");
		derivedTags.parse(input);
	}
	else
	{
		StdioParserInput input(file);
		derivedTags.parse(input);
	}
}

template<class T>
void TagcollDocument<T>::do_changed()
{
	_signal_changed.emit();
}

template<class T>
void TagcollDocument<T>::do_filename_changed()
{
	_signal_filename_changed.emit();
}

template<class T>
bool TagcollDocument<T>::canUndo() const
{
	return _undoTail > 0;
}

template<class T>
bool TagcollDocument<T>::canRedo() const
{
	return _undoTail < _undoBuffer.size();
}

template<class T>
void TagcollDocument<T>::undo()
{
	if (!canUndo())
		return;

	debug("Pre undo: tail: %d, size: %d\n", _undoTail, _undoBuffer.size());

	_undoTail--;
	_undoBuffer[_undoTail] = _collection.applyChange(_undoBuffer[_undoTail]);

	debug("Post undo: tail: %d, size: %d\n", _undoTail, _undoBuffer.size());

	do_changed();
}

template<class T>
void TagcollDocument<T>::redo()
{
	if (!canRedo())
		return;

	debug("Pre redo: tail: %d, size: %d\n", _undoTail, _undoBuffer.size());

	_undoBuffer[_undoTail] = _collection.applyChange(_undoBuffer[_undoTail]);
	_undoTail++;

	debug("Post redo: tail: %d, size: %d\n", _undoTail, _undoBuffer.size());

	do_changed();
}

template<class T>
void TagcollDocument<T>::applyChange(const TagCollection<int>::Change& change)
{
	TagCollection<int>::Change reverse = _collection.applyChange(change);
	
	if (reverse != change)
	{
		debug("Pre change: tail: %d, size: %d\n", _undoTail, _undoBuffer.size());

		// Push the reverse change in the undo buffer
		if (_undoTail == _undoBuffer.size())
		{
			_undoBuffer.push_back(reverse);
			_undoTail++;
		}
		else
		{
			_undoBuffer[_undoTail++] = reverse;
			_undoBuffer.resize(_undoTail);
		}

		debug("Post change: tail: %d, size: %d\n", _undoTail, _undoBuffer.size());
			
		do_changed();
	} else
		debug("Change had no effect\n");
}

template<class T>
void TagcollDocument<T>::load(const string& file)
	throw (FileException, ParserException)
{
	TagcollBuilder builder(_handleMaker);
	if (file == "-")
	{
		StdioParserInput in(stdin, "<stdin>");
		TagcollParser::parseTagcoll(in, builder);
	}
	else
	{
		StdioParserInput in(file);
		TagcollParser::parseTagcoll(in, builder);
	}
	_collection = builder.collection();
	_undoBuffer.clear();
	_undoTail = 0;
	_fileName = file;

	do_filename_changed();
	do_changed();
}

template<class T>
void TagcollDocument<T>::loadDebtags()
	throw (FileException, ParserException)
{
	// Prepare the input filter chain
	FilterChain<T> filters;

	// Expand implications
	ImplicationList implications;
	readCollection(fn_impls, implications);
	// Pack the structure for faster expansions
	implications.pack();
	TagcollFilter<T>* ximpl = new ApplyImplications(implications);
	filters.appendFilter(ximpl);

	// Add derived tags
	DerivedTagList derivedTags;
	readDerivedTags(fn_deriv, derivedTags);
	TagcollFilter<T>* xderv = new ApplyDerivedTags(derivedTags);
	filters.appendFilter(xderv);

	// Add further tags implicated by the derived tags
	filters.appendFilter(ximpl);

	// Read and expand the database
	InputMerger<T> merger;
	filters.setConsumer(&merger);
	readCollection(fn_tagdb, filters);

	delete ximpl;
	delete xderv;

	// Output the merged collection to consumer
	TagcollBuilder builder(_handleMaker);
	merger.output(builder);
	_collection = builder.collection();
	_undoBuffer.clear();
	_undoTail = 0;
	_fileName = "";

	do_filename_changed();
	do_changed();
}

template<class T>
void TagcollDocument<T>::save(const std::string& file) throw (FileException)
{
	FILE* out = fopen(file.c_str(), "wt");
	if (!out)
		throw FileException(errno, "opening file " + file);
	
	// Output the collection, grouping the items
	TagcollSerializer serializer(out);
	ItemGrouper<string> grouper;
	output(grouper);
	grouper.output(serializer);

	fclose(out);

	if (file != _fileName)
	{
		_fileName = file;
		do_filename_changed();
	}
}

template<class T>
OpSet<T> TagcollDocument<T>::handlesToItems(const OpSet<int>& handles) const throw ()
{
	OpSet<T> res;
	for (set<int>::const_iterator i = handles.begin(); i != handles.end(); i++)
		res += _handleMaker.getItem(*i);
	return res;
}


template<class T>
void TagcollDocument<T>::output(TagcollConsumer<T>& consumer) throw ()
{
	for (typename TagCollection<int>::const_iterator i = _collection.begin();
			i != _collection.end(); i++)
		consumer.consume(handlesToItems(i->second), i->first);
}

template<class T>
void TagcollDocument<T>::output(TagcollConsumer<int>& consumer) throw ()
{
	for (typename TagCollection<int>::const_iterator i = _collection.begin();
			i != _collection.end(); i++)
		consumer.consume(i->second, i->first);
}
// vim:set ts=4 sw=4:
