/*
 * synaptiks -- a touchpad control tool
 *
 *
 * Copyright (C) 2009, 2010 Sebastian Wiesner <basti.wiesner@gmx.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


#include "synaptikswidgets.h"
#include "ui_touchpadinformationpage.h"
#include "ui_autotouchpadcontrolpage.h"
#include "ui_generalpage.h"
#include "ui_motionpage.h"
#include "ui_scrollingpage.h"
#include "ui_tappingpage.h"
#include "ui_notificationspage.h"
#include "ui_errorpage.h"
#include <KIconLoader>
#include <KDebug>
#include <QtDBus/QDBusInterface>
#include <QtDBus/QDBusReply>
#include <QtCore/QList>


using namespace synaptiks;

TouchpadInformationPage::TouchpadInformationPage(QWidget *parent):
    QWidget(parent) {
    Ui::TouchpadInformationPage ui;
    ui.setupUi(this);
    ui.errorLabel->setVisible(false);
    this->showTouchpad();
}

void TouchpadInformationPage::showTouchpad() {
    QDBusInterface touchpad("org.kde.synaptiks", "/Touchpad",
                            "org.kde.Touchpad");
    if (!touchpad.isValid()) {
        this->setEnabled(false);
        this->setToolTip(i18nc("@info touchpad information error",
                               "<warning>No touchpad found.</warning>"));
        return;
    }

    QPixmap ok = KIconLoader::global()->loadIcon(
        "dialog-ok", KIconLoader::Small);
    QPixmap notOk = KIconLoader::global()->loadIcon(
        "dialog-cancel", KIconLoader::Small);
    try {
        QDBusReply<QString> name = touchpad.call("name");
        if (name.isValid()) {
            QLabel *nameLabel = this->findChild<QLabel*>("nameLabel");
            Q_ASSERT(nameLabel);
            nameLabel->setText(
                i18nc("@info touchpad name",
                      "<title><resource>%1</resource></title>",
                      name.value()));
        } else {
            throw name.error();
        }

        QRegExp pattern("button_(.*)");
        QList<QLabel *> buttonLabels = this->findChildren<QLabel*>(pattern);
        QString methodName = "has%1Button";
        foreach(QLabel *label, buttonLabels) {
            pattern.indexIn(label->objectName());
            QString buttonName = pattern.cap(1);
            QDBusReply<bool> hasButton = touchpad.call(
                methodName.arg(buttonName));

            if (hasButton.isValid()) {
                label->setPixmap(hasButton.value()? ok: notOk);
            } else {
                throw hasButton.error();
            }
        }
        QDBusReply<int> fingers = touchpad.call("fingerDetection");
        if (fingers.isValid()) {
            QSpinBox *fingerDetection = this->findChild<QSpinBox*>(
                "fingerDetection");
            Q_ASSERT(fingerDetection);
            fingerDetection->setValue(fingers.value());
        } else {
            throw fingers.error();
        }
    } catch (const QDBusError &error) {
        QLabel *errorLabel = this->findChild<QLabel*>("errorLabel");
        Q_ASSERT(errorLabel);
        errorLabel->setVisible(true);
        this->setEnabled(false);
        this->setToolTip(error.message());
        return;
    }
}

AutoTouchpadControlPage::AutoTouchpadControlPage(QWidget *parent):
    QWidget(parent) {
    Ui::AutoTouchpadControlPage ui;
    ui.setupUi(this);
    ui.kcfg_IgnoredMouseDevices->setTouchpadsIgnored(true);
}

GeneralPage::GeneralPage(QWidget *parent): QWidget(parent) {
    Ui::GeneralPage ui;
    ui.setupUi(this);
}

MotionPage::MotionPage(QWidget *parent): QWidget(parent) {
    Ui::MotionPage ui;
    ui.setupUi(this);
    this->connect(ui.kcfg_MinimumSpeed, SIGNAL(valueChanged(double)),
                  SLOT(disableAccelerationFactor()));
    this->connect(ui.kcfg_MaximumSpeed, SIGNAL(valueChanged(double)),
                  SLOT(disableAccelerationFactor()));
}

void MotionPage::disableAccelerationFactor() {
    QRegExp pattern("kcfg_.+Speed");
    QList<KDoubleNumInput*> speedInputs =
        this->findChildren<KDoubleNumInput*>(pattern);
    Q_ASSERT(speedInputs.size() == 2);
    KDoubleNumInput *accelFactor = this->findChild<KDoubleNumInput*>(
            "kcfg_AccelerationFactor");
    Q_ASSERT(accelFactor);
    accelFactor->setDisabled(
        speedInputs[0]->value() == speedInputs[1]->value());
}

ScrollingPage::ScrollingPage(QWidget *parent): QWidget(parent) {
    Ui::ScrollingPage ui;
    ui.setupUi(this);
    QDBusInterface touchpad("org.kde.synaptiks",  "/Touchpad",
                            "org.kde.Touchpad");
    QDBusReply<int> fingers = touchpad.call("fingerDetection");
    if (fingers.isValid()) {
        if (fingers.value() < 2) {
            ui.kcfg_HorizontalTwoFingerScrolling->setEnabled(false);
            ui.kcfg_VerticalTwoFingerScrolling->setEnabled(false);
        }
    }
    this->connect(ui.kcfg_HorizontalEdgeScrolling, SIGNAL(toggled(bool)),
                  SLOT(disableCoastingConfiguration()));
    this->connect(ui.kcfg_VerticalEdgeScrolling, SIGNAL(toggled(bool)),
                  SLOT(disableCoastingConfiguration()));
}

void ScrollingPage::disableCoastingConfiguration() {
    QRegExp pattern("kcfg_.+EdgeScrolling");
    QList<QCheckBox*> edgeScrollingOptions =
        this->findChildren<QCheckBox*>(pattern);
    Q_ASSERT(edgeScrollingOptions.size() == 2);
    bool edgeScrollingEnabled = false;
    foreach (const QCheckBox *option, edgeScrollingOptions) {
        edgeScrollingEnabled |= option->isChecked();
        if (edgeScrollingEnabled)
            break;
    }
    QGroupBox *kcfg_Coasting = this->findChild<QGroupBox*>("kcfg_Coasting");
    Q_ASSERT(kcfg_Coasting);
    kcfg_Coasting->setEnabled(edgeScrollingEnabled);
}

TappingPage::TappingPage(QWidget *parent):
    QWidget(parent) {
    Ui::TappingPage ui;
    ui.setupUi(this);
    QDBusInterface touchpad("org.kde.synaptiks", "/Touchpad",
                            "org.kde.Touchpad");
    if (!touchpad.isValid()) {
        return;
    }
    QDBusReply<int> fingers = touchpad.call("fingerDetection");
    if (fingers.isValid()) {
        for (int i=fingers; i < 3; i++) {
            QWidget *widget = ui.fingerButtonsLayout->itemAt(
                i, QFormLayout::FieldRole)->widget();
            widget->setEnabled(false);
            ui.fingerButtonsLayout->labelForField(
                widget)->setEnabled(false);
        }
    }
}

NotificationsPage::NotificationsPage(const QString &appname,
                                         QWidget *parent): QWidget(parent) {
    Ui::NotificationsPage ui;
    ui.setupUi(this);
    QList<KNotifyConfigWidget *> notifications =
        this->findChildren<KNotifyConfigWidget *>();
    QRegExp pattern("notifications_(.*)");
    foreach (KNotifyConfigWidget *widget, notifications) {
        QString reason;
        QString context;
        if (pattern.indexIn(widget->objectName()) != -1) {
            reason = pattern.cap(1);
            context = "reason";
        }
        widget->setApplication(appname, context, reason);
        this->connect(widget, SIGNAL(changed(bool)),
                      SIGNAL(changed(bool)));
    }
}

void NotificationsPage::saveNotifications() {
    QList<KNotifyConfigWidget *> notifications =
        this->findChildren<KNotifyConfigWidget *>();
    foreach (KNotifyConfigWidget *widget, notifications) {
        widget->save();
    }
}

ErrorPage::ErrorPage(const QString &message, QWidget *parent):
    QWidget(parent) {
    Ui::ErrorPage ui;
    ui.setupUi(this);
    ui.iconLabel->setPixmap(KIcon("dialog-warning").pixmap(128, 128));
    ui.detailsLabel->setText(message);
}

#include "moc_synaptikswidgets.cpp"
