/***************************************************************************
 *   Copyright (C) 2004 by Ivan Forcada Atienza                            *
 *   ivan@forcada.info                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "gpserial.h"
#include "myevents.h"

GPSerial::GPSerial()
{
}

GPSerial::~GPSerial()
{
	closePort(des);
	qDebug("Deleting gpserial...");
}

void GPSerial::run()
{
	char buf[255];
	while (getData(buf))
	{
		if (QString(buf).left(6) == "$GPRMC") printf ("%s", buf);
	}
	printf("Connecion to GPS is lost\n");
	
}
void GPSerial::closePort(int d)
{
	close(d);
}


int GPSerial::openPort(char * dev)
{
	//int fd; /* File descriptor for the port */

	des = open(dev, O_RDONLY | O_NOCTTY | O_NONBLOCK);	// Open only for reading, avoid ^C, 
	if (des == -1)  // Could not open the por
		perror("openPort: Unable to open device");
	else
		fcntl(des, F_SETFL, 0);  // Set default behabiour of the port (read blocking)
	
	return (des);
}


void GPSerial::configure(int fd)
{
	struct termios tio;
	
	int speed[] = {B600, B1200, B2400, B4800, B9600, B19200};
	int databit[] = {CS5, CS6, CS7, CS8};
	
	/* May be not the best way, but the easiest for me is to use the frmconfigini combobox indexes, and the two 
	array above to configure the serial settings. So I will instantiate a frmconfigini, but not show it, and use 
	it's combobox indexes. */
	
	frmConfigIni * frmconf = new frmConfigIni;
	
	// Get the current options of the port
	if (tcgetattr(fd, &tio) == -1) perror("tcgetattr");	
	
	bzero((void *)&tio, sizeof(tio));	// Reset the termios struct

/*************** CONTROL OPTIONS **************/
/* PARITY */
	switch (frmconf->cmbParity->currentItem())
	{
	/*even*/case 0: 
			tio.c_cflag |= PARENB;	// Set the parity bits
			tio.c_cflag &= ~PARODD;	// Unset the Odd bit (not odd = even)
			break;
	/*odd*/	case 1:	
			tio.c_cflag |= PARENB;	// Set the parity bits
			tio.c_cflag |= PARODD;	// Set the Odd bit
			break;
	/*none*/case 2: tio.c_cflag &= ~PARENB;	// Unset the parity bits
			break;
	}
	
/* STOP BITS */
	if (frmconf->cmbStop->currentItem() == 0)	// 1 stop bit
		tio.c_cflag &= ~CSTOPB;	// Unset the "2 stop bits" (if not 2 stop bits the 1)
	else	// 2 stop bits
		tio.c_cflag |= CSTOPB;	// Set the "2 stop bits"
		
/* DATA BITS */
	tio.c_cflag &= ~CSIZE;	// Umask every possible previous databit configuration whith the generic mask
	tio.c_cflag |= databit[frmconf->cmbBits->currentItem()];	// Set the apropriate databit mask
	
/* SPEED */
	// Modern way of setting the i/o speed (baud):
	if (cfsetispeed(&tio, speed[frmconf->cmbBaud->currentItem()]) == -1) perror("cfsetispeed");
	if (cfsetospeed(&tio, speed[frmconf->cmbBaud->currentItem()]) == -1) perror("cfsetospeed");			

/* OTHER SETTINGS */	
	// Set hardwere flow control
	tio.c_cflag |= CRTSCTS;
	
	tio.c_cflag |= (CLOCAL | CREAD);

/*************** LOCAL OPTIONS **************/
	tio.c_lflag |= (ICANON | ECHO | ECHOE);	// Canonical input
		
/*************** INPUT OPTIONS **************/	
	tio.c_iflag |= (INPCK | ISTRIP);	// Parity checking
	//tio.c_iflag |= (IXON | IXOFF | IXANY);	// Enable software flow control
	tio.c_iflag &= ~(IXON | IXOFF | IXANY); // Disable software flow control
	
/*************** OUTPUT OPTIONS **************/
	tio.c_oflag |= (OPOST | ONLCR  ) ;	// Not used since I only want to read data

/*************** CONTROL CHARATERS **************/	
	tio.c_cc[VTIME]    = 0;     // Timeouts are ignored in canonical input
	tio.c_cc[VMIN]     = 1;     
	

	//tcflush(fd, TCIFLUSH);	//Don't know if it's necesary, but...	
	// Set the new options, but NOW!
	if (tcsetattr(fd, TCSANOW, &tio) == -1) perror("tcsetattr");	
	
	delete frmconf;
}

int GPSerial::getData(char * buf)
{
	int fd, n = 0;
	geodata gpsdata;
	
	fd = getDescriptor();
	n = read(fd, buf,255);
	if (n == -1)
		fprintf(stderr, "Reading port(%d): %s", fd, strerror(errno));
	else
	{
		buf[n]=0;
		if (QString(buf).left(6)=="$GPRMC")	// only for '$GPRMC' sentences
		{
			printf("$GPRMC sentence found!!\n");
			if (parseData(buf, &gpsdata) == 0);	// fill the struct if valid data
			{
				printf("Data parsed successfully!!\n");
				newPosEvent* pe = new newPosEvent( gpsdata );
				QApplication::postEvent( parentWin, pe );	// Qt will delete it when done
				printf("Sending data to main thread...\n");
			}
		}
	}
	
	return n;
}

int GPSerial::parseData(char * line, geodata* data)
{
	QStringList fields = fields.split(",",QString(line),TRUE);
	int i=0;
		
	for ( QStringList::Iterator it = fields.begin(); it != fields.end(); ++it) 
	{
		//printf("%s\n",QString(*it).ascii());
		switch(i)
		{
			case status: 
				if (QString(*it) == QString("V")) return -1; // Invalid data
				break;
			case lat: 
				data->lat_dmin = toDMin(*it);
				data->lat_dms = toDMS(data->lat_dmin);
				data->lat_ddeg = toDDeg(data->lat_dmin);
				break;
			case lat_hemisf: 
				if (QString(*it) == QString("S")) 
				{
					data->lat_dms*=(-1);
					data->lat_ddeg*=(-1);
					data->lat_dmin*=(-1);
				}
				printf("LAT) %f = %f = %f\n", data->lat_dmin, data->lat_dms, data->lat_ddeg);
				break;
			case lng: 
				data->long_dmin = toDMin(*it);
				data->long_dms = toDMS(data->long_dmin);
				data->long_ddeg = toDDeg(data->long_dmin);
				break;
			case lng_hemisf:
				 //printf("NS\n");
				if (QString(*it) == QString("W"))
				{
					data->long_dms*=(-1);
					data->long_ddeg*=(-1);
					data->long_dmin*=(-1);
				}
				printf("LON) %f = %f = %f\n", data->long_dmin, data->long_dms, data->long_ddeg);
				break;
			case speed: 
				data->speed_knots = QVariant(*it).toDouble();
				data->speed_mph = data->speed_knots*1.1515;
				data->speed_kph = data->speed_knots*1.852;
				printf("SP) %f = %f = %f\n", data->speed_knots, data->speed_mph, data->speed_kph);
				break;
			default:;
		}
		i++;
	}
	return 0;
}
double GPSerial::toDMS(double dmin)
{ // ie: 4038,3493 ~ 40/38,3493' ==> 403820,598 ~ 40/38'/20,598''
	double seg;
	int min;
	min=(int)dmin;	// truncate the decimals (min=4038)
	seg = dmin - (double)min;	// extract the decimals into seg (seg=0,3493')
	seg*=60;	//from minutes to secs (seg=20,598'')
	return min*100 + seg;	//DDMMSS,ss format (403800+20,598=403820,598) => 40/38'/20,598''
}
double GPSerial::toDMin(QString s)
{
	return QVariant(s).toDouble();
}
double GPSerial::toDDeg(double dmin)
{// ie: 4038,3493 ~ 40/38,3493' ==> 40,639155
	int deg;
	
	dmin/=100;	//dmin=40,383493 ~ 40/38,3493'
	deg=(int)dmin;	//deg = 40
	dmin=dmin-deg;  //dmin=0,383493
	dmin*=100;	//dmin=38.3493
	dmin /=60;	//from min to deg => dmin=0,639155
	return deg+dmin;//DD,dddddd format => 40,639155
}
