'''
 ====================================================================
 Copyright (c) 2009 Barry A Scott.  All rights reserved.

 This software is licensed as described in the file LICENSE.txt,
 which you should have received as part of this distribution.

 ====================================================================

    wb_subversion_report_revision_changes.py

'''
import types
import wx

import wb_config
import wb_ids
import wb_images
import wb_exceptions
import wb_list_panel_common
import wb_subversion_diff
import wb_subversion_utils
import wb_subversion_list_handler_common
import wb_platform_specific

class ReportRevisionChangesFrame(wx.Frame):
    def __init__( self, app, project_info, all_files, info1, info2 ):
        wx.Frame.__init__( self, None, -1,
                           "Revision changes - r%d vs. r%d" % (info1.revision.number, info2.revision.number),
                           size=(700,500) )

        self.app = app

        self.info1 = info1
        self.info2 = info2
        self.menu_text = 'Diff r%d vs. r%d...' % (self.info1.revision.number, self.info2.revision.number)

        self.menu_actions = wx.Menu()
        self.menu_actions.Append( wb_ids.id_SP_DiffRevisionRevision, self.menu_text, self.menu_text )

        self.menu_actions.AppendSeparator()
        self.menu_actions.Append( wb_ids.id_SP_Annotate, 'Annotate...', 'Annotate...' )
        self.menu_actions.Append( wb_ids.id_SP_History, 'Log history...', 'Log history...' )
        self.menu_actions.Append( wb_ids.id_SP_Info, 'Information...', 'Information...' )

        self.menu_bar = wx.MenuBar()
        self.menu_bar.Append( self.menu_actions, "&Actions" )

        self.SetMenuBar( self.menu_bar )

        # Add tool bar
        t = self.CreateToolBar( name="main",
                                style=wx.TB_HORIZONTAL )

        bitmap_size = (32,32)
        t.SetToolBitmapSize( bitmap_size )
        t.AddSimpleTool( wb_ids.id_SP_DiffRevisionRevision,
            wb_images.getBitmap( 'toolbar_images/diff.png', bitmap_size ),
            self.menu_text, self.menu_text )
        t.AddSimpleTool( wb_ids.id_SP_History,
            wb_images.getBitmap( 'toolbar_images/history.png', bitmap_size ),
            'Show History log', 'Show History log' )
        t.AddSimpleTool( wb_ids.id_SP_Info,
            wb_images.getBitmap( 'toolbar_images/info.png', bitmap_size ),
            'File Information', 'File Information' )

        try_wrapper = wb_exceptions.TryWrapperFactory( self.app.log )

        # Set the application icon
        self.SetIcon( wb_images.getIcon( 'wb.png' ) )

        # create the individule panels
        self.panel_list = ReportRevisionChangesListPanel( app, self, self )

        wx.EVT_CLOSE( self, self.OnCloseWindow )

        wx.EVT_BUTTON( self.panel_list, wx.ID_OK, self.app.eventWrapper( self.OnOk ) )

        self.project_info = ReportRevisionChangesProjectInfo( project_info, all_files )
        self.list_handler = ReportRevisionChangesListHandler( self.app, self.panel_list, self.project_info, self.menu_text )

        # draw the list - it updates the status info.
        self.panel_list.setHandler( self.list_handler )

        wx.EVT_MENU( self, wb_ids.id_SP_Annotate, self.app.eventWrapper( self.OnSpAnnotate ) )
        wx.EVT_MENU( self, wb_ids.id_SP_DiffRevisionRevision, self.app.eventWrapper( self.OnSpDiffRevisionRevision ) )
        wx.EVT_MENU( self, wb_ids.id_SP_History, self.app.eventWrapper( self.OnSpHistory ) )
        wx.EVT_MENU( self, wb_ids.id_SP_Info, self.app.eventWrapper( self.OnSpInfo ) )

    def clearUpdateUiState( self ):
        pass

    def getUpdateUiState( self ):
        pass

    def setEventHandler( self, handler ):
        self.handler = handler

    def OnCloseWindow( self, event ):
        self.Destroy()

    def OnCancel( self, event ):
        self.Destroy()

    def OnOk( self, event ):
        self.Destroy()

    # command events
    def OnSpAnnotate( self, event ):
        return self.panel_list.OnSpAnnotate()

    def OnSpDiffRevisionRevision( self, event ):
        for filename, url in [(self.list_handler.getFilename( row ), self.list_handler.getUrl( row ))
                                for row in self.panel_list.getSelectedRows()]:

            info1 = self.info1.copy()

            info1.peg_path = url
            info1.title = filename + '@r' + str(info1.revision.number)

            info2 = self.info2.copy()

            info2.peg_path = url
            info2.title = filename + '@r' + str(info2.revision.number)

            self.app.setAction( 'Diff -r%d:%d %s...' % (info1.revision.number, info2.revision.number, url) )

            generator = wb_subversion_diff.subversionDiffFiles(
                                self.app,
                                self.project_info,
                                info1,
                                info2 )

            if type(generator) == types.GeneratorType:
                while True:
                    try:
                        where_to_go_next = generator.next()
                    except StopIteration:
                        # no problem all done
                        break

                    yield where_to_go_next

        self.app.setAction( 'Ready' )

    def OnSpHistory( self, event ):
        return self.panel_list.OnSpHistory()

    def OnSpInfo( self, event ):
        return self.panel_list.OnSpInfo()

class ReportRevisionChangesListHandler(wb_subversion_list_handler_common.SubversionListHandlerCommon):
    def __init__( self, app, parent, project_info, menu_text ):
        wb_subversion_list_handler_common.SubversionListHandlerCommon.__init__( self, app, parent, project_info )

        self.all_excluded_files = {}
        self.menu_text = menu_text

    def _getNameColPrefix( self ):
        return 0

    # use the repos status in the report
    def getTextStatus( self, row_or_status ):
        status = self.getStatusFromRowOrStatus( row_or_status )
        return status.repos_text_status

    # use the repos status in the report
    def getPropStatus( self, row_or_status ):
        status = self.getStatusFromRowOrStatus( row_or_status )
        return status.repos_prop_status

    def statusFormatString( self, file ):
        text_code = wb_subversion_utils.wc_status_kind_map[ file.text_status ]
        prop_code = wb_subversion_utils.wc_status_kind_map[ file.prop_status ]
        if text_code == ' ' and prop_code != ' ':
            text_code = '_'

        return file.get('branch_text_states', '') + text_code + prop_code

    def setupColumnInfo( self ):
        self.column_info.setFrom( ['State','Name'], [5, 100] )

    def statusColour( self, file ):
        # show that a file is on the exclude list
        if file.path in self.getAllGreyFilenames():
            return wb_config.colour_status_disabled
        else:
            return wb_config.colour_status_normal

    def getContextMenu( self ):
        menu_template = \
            [('', wb_ids.id_SP_DiffRevisionRevision, self.menu_text )
            ,('-', 0, 0 )
            ,('', wb_ids.id_SP_Annotate, 'Annotate...' )
            ,('', wb_ids.id_SP_History, 'Log history...' )
            ,('', wb_ids.id_SP_Info, 'Information...' )
            ]
        return wb_subversion_utils.populateMenu( wx.Menu(), menu_template )

    def getAllGreyFilenames( self ):
        # show all excluded files in grey
        return self.all_excluded_files

class ReportRevisionChangesProjectInfo:
    def __init__( self, project_info, all_files ):
        self.all_files = all_files
        self.need_properties = False
        self.project_name = project_info.project_name
        self.url = project_info.url
        self.wc_path = project_info.wc_path
        self.need_checkout = False

        self.client_fg = project_info.client_fg
        self.client_bg = project_info.client_bg

    def getTagsUrl( self, rel_url ):
        return None

    def setNeedProperties( self, need_properties ):
        self.need_properties = need_properties

    def updateStatus( self ):
        pass

    def getFilesStatus( self ):
        return self.all_files

    def getProperty( self, filename, prop_name ):
        return ''

    def getWorkingDir( self ):
        return self.wc_path

class ReportRevisionChangesListPanel(wb_list_panel_common.WbListPanelCommon):
    def __init__( self, app, frame, parent ):
        wb_list_panel_common.WbListPanelCommon.__init__( self, app, frame, parent )

    def addToSizer( self, v_sizer ):
        pass

    def getAcceleratorTableInit( self ):
        if wx.Platform == '__WXMAC__':
            acc_init =[
                (wx.ACCEL_ALT, ord('D'), wb_ids.id_SP_DiffRevisionRevision),
                (wx.ACCEL_ALT, ord('L'), wb_ids.id_SP_History),
                ]
        elif wx.Platform == '__WXMSW__':
            acc_init =[
                (wx.ACCEL_CTRL, ord('D'), wb_ids.id_SP_DiffRevisionRevision),
                (wx.ACCEL_CTRL, ord('L'), wb_ids.id_SP_History),
                ]
        else:
            # Unix
            acc_init =[
                (wx.ACCEL_CTRL, ord('D'), wb_ids.id_SP_DiffWorkBase),
                (wx.ACCEL_CTRL, ord('L'), wb_ids.id_SP_History),
                ]

        return acc_init
