#ifdef ENABLE_ASPELL

/*
 *	subtitle editor
 *
 *	http://kitone.free.fr/subtitleeditor/
 *
 *	Copyright @ 2005-2006, kitone
 *
 *	Contact: kitone at free dot fr
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 *	General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA
 *
 *	See gpl.txt for more information regarding the GNU General Public License.
 *
 *
 *	\file
 *	\brief 
 *	\author kitone (kitone at free dot fr)
 */

#include "SpellDialog.h"
#include "utility.h"


bool is_end_char(gchar c)
{
	switch(c)
	{
	case ' ':
	case '\0':
	case '.':
	case ',':
	case ':':
	case ';':
	case '*':
	case '+':
	case '=':
	case '\t':
	case '(':
	case ')':
	case '!':
	case '|':
	case '?':
	case '-':
	case '"':
	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
		return true;
	default:
	 return false;
	}
	return false;
}

/*
 *
 */
void CurrentWord::reset()
{
	position_start = 0;
	position_next_word = 0;
	text = "";
	word = "";
}

/*
 *
 */
void CurrentWord::replace(const Glib::ustring &new_word)
{
	text.replace(position_start, word.size(), new_word);

	SubtitleColumnRecorder column;
	(*iter)[column.text]=text;

	position_next_word += (new_word.size() - word.size());
	word = new_word;
}

/*
 *
 */
bool CurrentWord::next_word()
{
	for(unsigned int i=position_next_word; i<=text.size(); ++i)
	{
		if(is_end_char(text[i]))
		{
			position_start = position_next_word;//+1 ?

			Glib::ustring tmp = text.substr(position_next_word, i-position_next_word);
			position_next_word = i+1;
			if(tmp.size()>0)
			{
	word = tmp;
	return true;
			}
		}
	}
	return false;
}

/*
 *
 */
bool CurrentWord::next_line()
{
	reset();

	if(!iter)
		return false;

	++iter;
	if(!iter)
		return false;
	
	SubtitleColumnRecorder column;
	text = (*iter)[column.text];

	return true;
}

/*
 *
 */
SpellDialog::SpellDialog()
:Gtk::Dialog(_("Spell Check"))
{
	set_border_width(6);
	//set_resizable (false);
	set_has_separator (true);
	set_skip_taskbar_hint (true);
	get_vbox ()->set_spacing (8);
	
	set_default_size(600,400);

	createLine();
	createTreeViewAndButton();

	show_all();

	add_button(Gtk::Stock::CLOSE, Gtk::RESPONSE_CLOSE);
}

/*
 *
 */
void SpellDialog::createLine()
{
	Gtk::HBox *hbox = manage(new Gtk::HBox(false, 3));
	Gtk::Label *label = manage(new Gtk::Label(_("Text:"), 1.0,0.5));
	m_entryText = manage(new Gtk::Entry);
	m_entryText->set_editable(false);

	hbox->pack_start(*label, false,false);
	hbox->pack_start(*m_entryText, true,true);

	get_vbox()->pack_start(*hbox, false,false);
}

/*
 *
 */
void SpellDialog::createTreeViewAndButton()
{
	Gtk::HBox *hbox = manage(new Gtk::HBox(false, 6));
	get_vbox()->pack_start(*hbox, true,true);

	//
	Gtk::VBox *vbox1 = manage(new Gtk::VBox(false, 0));
	hbox->pack_start(*vbox1, true,true);

	Gtk::ScrolledWindow *scroll = manage(new Gtk::ScrolledWindow);
	scroll->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	vbox1->pack_start(*scroll, true,true);
	
	m_spellView = manage(new SpellView);
	scroll->add(*m_spellView);

	Gtk::HBox *hbox2=manage(new Gtk::HBox(true,2));
	vbox1->pack_start(*hbox2,false,false, 6);

	Gtk::Label* labelReplaceWith=manage(new Gtk::Label(_("Replace with:"),1.0,0.5));
	hbox2->pack_start(*labelReplaceWith,true,true);

	m_entryReplaceWith = manage(new Gtk::Entry);
	hbox2->pack_start(*m_entryReplaceWith,true,true);
	m_entryReplaceWith->signal_changed().connect(
			sigc::mem_fun(*this, &SpellDialog::on_entry_replace_with_changed));

	//
	Gtk::VBox *vbox2 = manage(new Gtk::VBox(false, 0));
	hbox->pack_start(*vbox2, false,false);

	
	m_comboboxDicts = manage(new Gtk::ComboBoxText);
	vbox2->pack_start(*m_comboboxDicts, false,false);

	{
		std::list<Glib::ustring>	dicts;
		if(m_spell.get_list_dict(dicts))
		{
			std::list<Glib::ustring>::const_iterator it;
			for(it=dicts.begin(); it!=dicts.end(); ++it)
			{
				m_comboboxDicts->append_text(*it);
			}
		}
	}
	// marche pas ?...
	m_comboboxDicts->set_active_text(m_spell.get_dict());

	m_comboboxDicts->signal_changed().connect(
			sigc::mem_fun(*this, &SpellDialog::on_combobox_dics_changed));
	
	vbox2->pack_start(*manage(new Gtk::HSeparator), false,false, 6);

	
	m_buttonReplace = manage(new Gtk::Button(_("_Replace"), true));
	m_buttonReplace->set_image(
			*manage(new Gtk::Image(Gtk::Stock::CONVERT, Gtk::ICON_SIZE_BUTTON)));
	vbox2->pack_start(*m_buttonReplace,false,false);
	m_buttonReplace->signal_clicked().connect(
		sigc::bind<ACTION>(sigc::mem_fun(*this, &SpellDialog::callback), 
			REPLACE));
	m_buttonReplace->set_sensitive(false);

	
	m_buttonIgnore = manage(new Gtk::Button(_("_Ignore"), true));
	vbox2->pack_start(*m_buttonIgnore,false,false);
	m_buttonIgnore->signal_clicked().connect(
		sigc::bind<ACTION>(sigc::mem_fun(*this, &SpellDialog::callback), 
			IGNORE));

	
	m_buttonIgnoreAll = manage(new Gtk::Button(_("Ig_nore all"), true));
	vbox2->pack_start(*m_buttonIgnoreAll,false,false);
	m_buttonIgnoreAll->signal_clicked().connect(
		sigc::bind<ACTION>(sigc::mem_fun(*this, &SpellDialog::callback), 
			IGNORE_ALL));

	
	m_buttonAddWord = manage(new Gtk::Button(_("_Add word"),true));
	m_buttonAddWord->set_image(
			*manage(new Gtk::Image(Gtk::Stock::ADD, Gtk::ICON_SIZE_BUTTON)));
	vbox2->pack_start(*m_buttonAddWord,false,false, 12);
	m_buttonAddWord->signal_clicked().connect(
		sigc::bind<ACTION>(sigc::mem_fun(*this, &SpellDialog::callback), 
			ADD_WORD));
	m_buttonAddWord->set_sensitive(false);



	m_spellView->get_selection()->signal_changed().connect(
			sigc::mem_fun(*this, &SpellDialog::on_selection_changed));
}

/*
 *
 */
void SpellDialog::on_selection_changed()
{
	m_entryReplaceWith->set_text("");
	
	Gtk::TreeIter iter = m_spellView->get_selection()->get_selected();
	if(iter)
	{
		SpellColumn column;
		m_entryReplaceWith->set_text((*iter)[column.string]);
	}
}

/*
 *
 */
void SpellDialog::on_entry_replace_with_changed()
{
	if(m_entryReplaceWith->get_text().size()==0)
	{
		m_buttonReplace->set_sensitive(false);
		m_buttonAddWord->set_sensitive(false);
	}else
	{
		m_buttonReplace->set_sensitive(true);
		m_buttonAddWord->set_sensitive(true);
	}
}

void SpellDialog::on_combobox_dics_changed()
{
	Glib::ustring txt = m_comboboxDicts->get_active_text();
	m_spell.set_dict(txt);
	action(CHECK_WORD);
}

/*
 *
 */
void SpellDialog::execute(const Glib::RefPtr<SubtitleModel> &model)
{
	m_finish=false;

	current.position_start = 0;
	current.position_next_word = 0;
	current.text.clear();
	current.word.clear();

	show();

	SubtitleColumnRecorder m_column;

	current_model = model;

	Gtk::TreeIter iter = model->getFirst();
	set(iter);

	run();
}

/*
 *
 */
bool SpellDialog::set(Gtk::TreeIter &iter)
{
	g_return_val_if_fail(iter, false);
	//if(!iter) { action(CLOSE); return false; }

	current.iter = iter;
	current.position_start = 0;
	current.position_next_word = 0;
	current.text.clear();
	current.word.clear();


	SubtitleColumnRecorder m_column;

	current.text = (*iter)[m_column.text];

	return action(NEXT_WORD);
}

/*
 *
 */
void SpellDialog::callback(SpellDialog::ACTION act)
{
	action(act);
}

/*
 *
 */
bool SpellDialog::action(SpellDialog::ACTION act)
{
	switch(act)
	{
	case START:
		break;
	case CLOSE:
		{
			Gtk::Dialog::response(Gtk::RESPONSE_CLOSE);
		}break;
	case SET_LINE:
		{
			m_entryText->set_text(current.text);
			m_entryText->select_region(
			current.position_start, current.position_next_word -1);
		}break;
	case NEXT_WORD:
		{
			if(current.next_word())
				action(CHECK_WORD);
			else
				action(NEXT_LINE);
		} break;
	case CHECK_WORD:
		{
			m_spellView->m_model->clear();
			//
			std::list<Glib::ustring>	suggestions;
			ASpell::SPELL_RESPONSE rep = m_spell.check(current.word, suggestions);

			if(rep == ASpell::SPELL_CORRECT)
			{
				action(NEXT_WORD);
			}
			else if(rep == ASpell::SPELL_SUGGESTION)
			{
				action(SET_LINE);
				
				m_spellView->m_model->clear();

				std::list<Glib::ustring>::const_iterator it;
				for(it=suggestions.begin(); it!=suggestions.end(); ++it)
					m_spellView->m_model->add(*it);
			}
			else if(rep == ASpell::SPELL_NO_SUGGESTION)
			{
				action(SET_LINE);
				m_spellView->m_model->clear();
				m_spellView->m_model->add(_("no suggestion"));
			}

		} break;
	case NEXT_LINE:
		{
			if(current.next_line())
				action(NEXT_WORD);
			else
				action(CLOSE);
		}break;
	case REPLACE:
		{
			Glib::ustring word = current.word;
			Glib::ustring new_word = m_entryReplaceWith->get_text();
			current.replace(new_word);
			
			action(NEXT_WORD);
		}break;
	case IGNORE:
		{
			action(NEXT_WORD);
		}break;
	case IGNORE_ALL:
		{
			m_spell.ignore_word(current.word);
			action(NEXT_WORD);
		}break;
	case ADD_WORD:
		{
			m_spell.add_word(current.word);
			action(NEXT_WORD);
		}break;
	default:
		break;
	}

	return true;
}




/*
 *
 */
DialogSpellCheck::DialogSpellCheck(BaseObjectType *cobject, const Glib::RefPtr<Gnome::Glade::Xml>& refGlade)
:Gtk::Dialog(cobject)
{
	Gtk::HBox* m_boxComboDicts = NULL;
	
	refGlade->get_widget("entryText", m_entryText);
	refGlade->get_widget("treeview", m_treeview);
	refGlade->get_widget("entryReplaceWith", m_entryReplaceWith);
	// HACK! :(
	refGlade->get_widget("boxComboDicts", m_boxComboDicts);
	m_comboDicts = manage(new Gtk::ComboBoxText);
	m_boxComboDicts->pack_start(*m_comboDicts,true,true);
	m_comboDicts->show();

	refGlade->get_widget("buttonReplace", m_buttonReplace);
	refGlade->get_widget("buttonIgnore", m_buttonIgnore);
	refGlade->get_widget("buttonIgnoreAll", m_buttonIgnoreAll);
	refGlade->get_widget("buttonNextLine", m_buttonNextLine);
	refGlade->get_widget("buttonAddWord", m_buttonAddWord);


	m_buttonReplace->signal_clicked().connect(
		sigc::bind<ACTION>(sigc::mem_fun(*this, &DialogSpellCheck::action), 
			REPLACE));
	m_buttonIgnore->signal_clicked().connect(
		sigc::bind<ACTION>(sigc::mem_fun(*this, &DialogSpellCheck::action), 
			IGNORE));
	m_buttonIgnoreAll->signal_clicked().connect(
		sigc::bind<ACTION>(sigc::mem_fun(*this, &DialogSpellCheck::action), 
			IGNORE_ALL));
	m_buttonAddWord->signal_clicked().connect(
		sigc::bind<ACTION>(sigc::mem_fun(*this, &DialogSpellCheck::action), 
			ADD_WORD));
	m_buttonNextLine->signal_clicked().connect(
		sigc::bind<ACTION>(sigc::mem_fun(*this, &DialogSpellCheck::action), 
			NEXT_LINE));	
	createTreeView();

	
	m_entryReplaceWith->signal_changed().connect(
			sigc::mem_fun(*this, &DialogSpellCheck::on_entry_replace_with_changed));
	
	// desactive par defaut
	m_buttonReplace->set_sensitive(false);
	m_buttonAddWord->set_sensitive(false);


	//
	{
		std::list<Glib::ustring>	dicts;
		if(m_spell.get_list_dict(dicts))
		{
			std::list<Glib::ustring>::const_iterator it;
			for(it=dicts.begin(); it!=dicts.end(); ++it)
			{
				m_comboDicts->append_text(*it);
			}
		}
		m_comboDicts->set_active_text(m_spell.get_dict());
	
		m_comboDicts->signal_changed().connect(
			sigc::mem_fun(*this, &DialogSpellCheck::on_combo_dicts_changed));
	}


}

void DialogSpellCheck::init(Document *doc)
{
	current_model = doc->m_subtitleModel;

	current.position_start = 0;
	current.position_next_word = 0;
	current.text.clear();
	current.word.clear();

	Gtk::TreeIter iter = current_model->getFirst();
	set(iter);

	show();

	run();

	hide();
}

void DialogSpellCheck::createTreeView()
{
	m_word_model = Glib::RefPtr<SpellModel>(new SpellModel);

	m_treeview->set_model(m_word_model);

	SpellColumn m_column;
	Gtk::TreeViewColumn* column = NULL;
	Gtk::CellRendererText* renderer = NULL;

	column = manage(new Gtk::TreeViewColumn(_("Suggestions")));
	renderer = manage(new Gtk::CellRendererText);

	column->pack_start(*renderer, false);
	column->add_attribute(renderer->property_text(), m_column.string);

	m_treeview->append_column(*column);
	
	m_treeview->get_selection()->signal_changed().connect(
			sigc::mem_fun(*this, &DialogSpellCheck::on_selection_changed));
}


/*
 *
 */
bool DialogSpellCheck::set(Gtk::TreeIter &iter)
{
	g_return_val_if_fail(iter, false);
	//if(!iter) { action(CLOSE); return false; }

	current.iter = iter;
	current.position_start = 0;
	current.position_next_word = 0;
	current.text.clear();
	current.word.clear();


	SubtitleColumnRecorder m_column;

	current.text = (*iter)[m_column.text];

	action(NEXT_WORD);

	return true;
}


/*
 *
 */
void DialogSpellCheck::action(DialogSpellCheck::ACTION act)
{
	switch(act)
	{
	case START:
		break;
	case CLOSE:
		{
			Gtk::Dialog::response(Gtk::RESPONSE_CLOSE);
		}break;
	case SET_LINE:
		{
			m_entryText->set_text(current.text);
			m_entryText->select_region(
			current.position_start, current.position_next_word -1);
		}break;
	case NEXT_WORD:
		{
			if(current.next_word())
				action(CHECK_WORD);
			else
				action(NEXT_LINE);
		} break;
	case CHECK_WORD:
		{
			m_word_model->clear();
			//
			std::list<Glib::ustring>	suggestions;
			ASpell::SPELL_RESPONSE rep = m_spell.check(current.word, suggestions);

			if(rep == ASpell::SPELL_CORRECT)
			{
				action(NEXT_WORD);
			}
			else if(rep == ASpell::SPELL_SUGGESTION)
			{
				action(SET_LINE);
				
				m_word_model->clear();

				std::list<Glib::ustring>::const_iterator it;
				for(it=suggestions.begin(); it!=suggestions.end(); ++it)
					m_word_model->add(*it);
			}
			else if(rep == ASpell::SPELL_NO_SUGGESTION)
			{
				action(SET_LINE);
				m_word_model->clear();
				m_word_model->add(_("no suggestion"));
			}

		} break;
	case NEXT_LINE:
		{
			if(current.next_line())
				action(NEXT_WORD);
			else
				action(CLOSE);
		}break;
	case REPLACE:
		{
			Glib::ustring word = current.word;
			Glib::ustring new_word = m_entryReplaceWith->get_text();
			current.replace(new_word);
			
			action(NEXT_WORD);
		}break;
	case IGNORE:
		{
			action(NEXT_WORD);
		}break;
	case IGNORE_ALL:
		{
			m_spell.ignore_word(current.word);
			action(NEXT_WORD);
		}break;
	case ADD_WORD:
		{
			m_spell.add_word(current.word);
			action(NEXT_WORD);
		}break;
	default:
		break;
	}
}

/*
 *
 */
void DialogSpellCheck::on_selection_changed()
{
	m_entryReplaceWith->set_text("");
	
	Gtk::TreeIter iter = m_treeview->get_selection()->get_selected();
	if(iter)
	{
		SpellColumn column;
		m_entryReplaceWith->set_text((*iter)[column.string]);
	}
}

/*
 *
 */
void DialogSpellCheck::on_entry_replace_with_changed()
{
	if(m_entryReplaceWith->get_text().size()==0)
	{
		m_buttonReplace->set_sensitive(false);
		m_buttonAddWord->set_sensitive(false);
	}else
	{
		m_buttonReplace->set_sensitive(true);
		m_buttonAddWord->set_sensitive(true);
	}
}

void DialogSpellCheck::on_combo_dicts_changed()
{
	Glib::ustring txt = m_comboDicts->get_active_text();
	m_spell.set_dict(txt);
	action(CHECK_WORD);
}

#endif//ENABLE_ASPELL
