/*
 *	subtitle editor
 *
 *	http://kitone.free.fr/subtitleeditor/
 *
 *	Copyright @ 2005-2006, kitone
 *
 *	Contact: kitone at free dot fr
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 *	General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA
 *
 *	See gpl.txt for more information regarding the GNU General Public License.
 *
 *
 *	\file
 *	\brief 
 *	\author kitone (kitone at free dot fr)
 */


#include "Document.h"
#include "Config.h"
#include <iostream>
#include "utility.h"

#include "SubtitleSystem.h"


/*
 *	constructeur
 *	s'il y a une option dans le fichier config [encodings] default=xxxx
 *	on l'utilise pour choisir le charset, sinon on utilise UTF-8
 */
Document::Document()
{
	Config &cfg = Config::getInstance();
	Glib::ustring encoding;
	
	if(cfg.get_value_string("encodings", "default", encoding))
		m_charset = encoding;
	else
		m_charset = "UTF-8";
#warning "Fixme > Use Config Option"
	m_format = "SubRip";

	m_subtitleModel = Glib::RefPtr<SubtitleModel>(new SubtitleModel);

	m_styleModel = Glib::RefPtr<StyleModel>(new StyleModel);

	//m_nameModel = Glib::RefPtr<NameModel>(new NameModel);
}

/*
 *	constructeur par copy
 */
Document::Document(Document &src)
{
	m_subtitleModel = Glib::RefPtr<SubtitleModel>(new SubtitleModel);
	m_styleModel = Glib::RefPtr<StyleModel>(new StyleModel);

	g_return_if_fail(m_subtitleModel);
	g_return_if_fail(m_styleModel);

	m_format = src.m_format;
	m_charset = src.m_charset;
	m_scriptInfo = src.m_scriptInfo;

	filenameDocument = src.filenameDocument;
	m_uriMovie = src.m_uriMovie;

	m_subtitleModel->copy(src.get_subtitle_model());

	m_styleModel->copy(src.get_style_model());
}

/*
 *	destructeur
 */
Document::~Document()
{
}

/*
 *
 */
Glib::ustring Document::getFormat()
{
	return m_format;
}

/*
 *	retourne le charset (encodage) du document
 *	a permet de sauvegarder dans le meme encodage qu'a la lecture
 *	...
 */
Glib::ustring Document::getCharset()
{
	return m_charset;
}
	
/*
 *	efface les informations :
 *	 - script info
 *	 - subtitle model
 *	 - style model
 *	
 *	reinitialize le charset
 */
void Document::clear()
{
	Config &cfg = Config::getInstance();
	Glib::ustring encoding;

	if(cfg.get_value_string("encodings", "default", encoding))
		m_charset = encoding;
	else
		m_charset = "UTF-8";

	//m_format = ASS;
	
	m_scriptInfo = ScriptInfo();
	
	filenameDocument.clear();
	m_uriMovie.clear();
	
	m_subtitleModel->clear();
	m_styleModel->clear();
	//m_nameModel->clear();
}

/*
 *	encodings ""=auto, "UTF-8", ...
 *	clear permet d'effacer l'ancien model, sinon on ajoute a la suite du document	
 */
bool Document::open(const Glib::ustring &filename, const Glib::ustring &encoding, bool clear_doc)
{
	return _open(filename, encoding, clear_doc);
}


/*
 *
 */
bool Document::_open(const Glib::ustring &filename, const Glib::ustring &encoding, bool clear_doc)
{
	if(clear_doc)
		clear();
	
	SubtitleFormat* sub = NULL;
	
	try
	{
		Glib::ustring format = SubtitleSystem::getInstance().find_subtitle_format(filename);

		if(!format.empty())
		{
			SubtitleFormat* sub = SubtitleSystem::getInstance().create_subtitle_format(format, this);

			if(sub)
			{
				bool res = sub->open(filename, encoding);

				if(res && clear_doc)
				{
					filenameDocument = filename;
					m_charset = sub->get_charset();
					m_format = format;
				}

				delete sub;

				return res;
			}
		}
		else
			;//throw
	}
	catch(const SubtitleException &ex)
	{
		delete sub;
		dialog_error(_("Open Document Failed."), ex.what());
	}
	catch(const Glib::ConvertError &ex)
	{
		delete sub;
		dialog_error(_("Open Document Failed."), ex.what());
	}

	return false;

}

/*
 *	encodings ""=auto, "UTF-8", ...
 *	format "SubRip", "MicroDVD", "MPL2", ...
 *	rename, change le nom du document (filenameDocument)
 */
bool Document::save(const Glib::ustring &filename, const Glib::ustring &format, const Glib::ustring &encoding, bool clear_doc)
{
	return _save(filename, format, encoding, clear_doc);
}

/*
 *
 */
bool Document::_save(const Glib::ustring &filename, const Glib::ustring &format, const Glib::ustring &encoding, bool rename)
{
	Glib::ustring charset=encoding;

	//
	if(!charset.empty() && rename)
		m_charset = charset;
	//
	if(encoding.empty())
		charset = getCharset();

	if(rename)
	{
		filenameDocument = filename;
		
		m_format = format;
	}

	SubtitleFormat *subtitle = NULL;
	
	try
	{
		subtitle = SubtitleSystem::getInstance().create_subtitle_format(format, this);
		if(subtitle != NULL)
		{
			bool state = subtitle->save(filename, charset);

			delete subtitle;

			return state;
		}
	}
	catch(const SubtitleException &ex)
	{
		delete subtitle;
		dialog_error(_("Save Document Failed."), ex.what());
	}

	return false;
}

/*
 *	retourne le model pour la gestion des sous-titres
 *	on passe par un model (gtk) pour eviter d'avoir des doublons 
 *	dans les donnes et donc une grosse consomation memoire (inutile)
 */
Glib::RefPtr<SubtitleModel> Document::get_subtitle_model()
{
	return m_subtitleModel;
}

/*
 *	retourne le model pour la gestion des styles
 */
Glib::RefPtr<StyleModel> Document::get_style_model()
{
	return m_styleModel;
}

/*
 *	retourne les informations sur le script 
 *	principalement pour SSA/ASS
 */
ScriptInfo& Document::get_script_info()
{
	return m_scriptInfo;
}

