use core:lang;
use core:asm;
use lang:bs:macro;

on Compiler:

/**
 * Class representing an expression in C++.
 */
class Expr extends Stmt {
	init(SrcPos pos) {
		init(pos) {}
	}

	// Adapter from the statement.
	Bool code(CodeGen gen) : final {
		var r = result();
		code(gen, CodeResult());
		!r.nothing();
	}

	// Compute the type of the result.
	ExprResult result() : abstract;

	// Suggest that the expression return a particular type. Assumes that 'type' differs from
	// whatever is returned from 'result'.
	Bool suggestResult(Type type) { false; }

	// Does this expression represent a temporary value? I.e. is this an r-value?
	Bool temporary() {
		true;
	}

	// Does this expression represent a constant value? Close to 'temporary', but relevant when
	// dereferencing pointers.
	Bool constant() {
		temporary();
	}

	// Get a static value for this expression, if possible. Asks the result to have the type
	// supplied in the call. If not, a void value is returned.
	Variant staticValue(Type type) {
		Variant();
	}

	// Get the reported result value.
	Variant staticValue() : final {
		if (t = result.type.type)
			staticValue(t);
		else
			Variant();
	}

	// Generate code. 'desired' is the desired return type.
	void code(CodeGen gen, CodeResult res) : abstract;

	// Compute a pointer/reference to this value. This might involve creating a temporary object.
	// Stores a base ptr in 'ptrB' and an offset in 'ecx'.
	void codePtr(CodeGen gen, Type type) : abstract;
}

/**
 * Convert a numeric value to a Nat.
 */
Int? variantToInt(Variant v) {
	if (v as Byte)
		return v.int;
	if (v as Int)
		return v;
	if (v as Nat)
		return v.int;
	if (v as Long)
		return v.int;
	if (v as Word)
		return v.int;
	null;
}


/**
 * An expression that supports creating a value inside a pre-allocated variable.
 */
class ExprRef extends Expr {
	init(SrcPos pos) {
		init(pos) {}
	}

	// Additional member.
	void codeRef(CodeGen gen, CppVar var) : abstract;

	// Generate a pointer to a temporary.
	void codePtr(CodeGen gen, Type type) : override {
		CppVar v = allocTemporary(gen, type);
		codeRef(gen, v);
		gen.l << location(SrcPos());
		v.ptr(gen);
	}
}

/**
 * Integer literals.
 */
class IntLiteral extends Expr {
	init(SrcPos pos, Long val) {
		init(pos) { val = val; }
	}

	init(SStr str) {
		init(str.pos) { val = str.v.toLong(); }
	}

	private Long val;

	ExprResult result() : override {
		Value(named{Int}, true);
	}

	Bool suggestResult(Type t) : override {
		return t is named{Byte} | t is named{Int} | t is named{Long} | t is named{Nat} | t is named{Word};
	}

	Variant staticValue(Type type) : override {
		if (type is named{Byte})
			return Variant(val.byte);
		if (type is named{Int})
			return Variant(val.int);
		if (type is named{Nat})
			return Variant(val.nat);
		if (type is named{Long})
			return Variant(val);
		if (type is named{Word})
			return Variant(val.word);
		Variant();
	}

	void code(CodeGen gen, CodeResult res) : override {
		if (!res.needed())
			return;

		var to = res.location(gen);
		if (t = res.type.type) {
			if (res.type.ref) {
				CppVar var = allocType(gen, t);
				var.adjust(gen, ptrA);
				if (t is named{Byte}) {
					gen.l << mov(byteRel(ptrA), byteConst(val.byte));
				} else if (t is named{Int} | t is named{Nat}) {
					gen.l << mov(intRel(ptrA), intConst(val.int));
				} else if (t is named{Long} | t is named{Word}) {
					gen.l << mov(longRel(ptrA), longConst(val));
				} else {
					throw SyntaxError(pos, "Unknown type!");
				}
				gen.l << mov(to, ptrA);
				var.created(gen);
			} else {
				if (t is named{Byte}) {
					gen.l << mov(to, byteConst(val.byte));
				} else if (t is named{Int} | t is named{Nat}) {
					gen.l << mov(to, intConst(val.int));
				} else if (t is named{Long} | t is named{Word}) {
					gen.l << mov(to, longConst(val));
				} else {
					throw SyntaxError(pos, "Unknown type!");
				}
			}
		}
		res.created(gen);
	}

	void codePtr(CodeGen gen, Type type) : override {
		CppVar var = allocTemporary(gen, type);
		var.adjust(gen, ptrA);
		if (type is named{Byte}) {
			gen.l << mov(byteRel(ptrA), byteConst(val.byte));
		} else if (type is named{Int} | type is named{Nat}) {
			gen.l << mov(intRel(ptrA), intConst(val.int));
		} else if (type is named{Long} | type is named{Word}) {
			gen.l << mov(longRel(ptrA), longConst(val));
		} else {
			throw SyntaxError(pos, "Unknown type!");
		}
		var.created(gen);
		// Make it known to the visualization.
		gen.l << location(SrcPos());
		var.ptr(gen);
	}
}

/**
 * Sizeof expression.
 */
Expr sizeofExpr(SrcPos pos, SizeofParam param) {
	if (type = param.type) {
		return IntLiteral(pos, type.size.aligned.current.long);
	} else if (local = param.arraySrc as LocalVar) {
		// We know that it is an array due to the ctor in SizeofParam.
		if (count = variantToInt(local.initializer))
			return IntLiteral(pos, count.long * local.type.size.aligned.current.long);
		else
			return DynamicSizeof(pos, local);
	} else if (global = param.arraySrc as CppGlobalArray) {
		var ptr = unsafe:RawPtr:fromGlobal(global);
		var alloc = ptr.readPtr(0);
		Long total = alloc.readCount.long * alloc.readSize.long;
		return IntLiteral(pos, total);
	}

	throw SyntaxError(pos, "Invalid parameter to 'sizeof'.");
}

/**
 * What to return from a sizeof expression?
 */
class SizeofParam on Compiler {
	// Create from known type:
	init(Value t) {
		// Note: We don't need a position in this case!
		init { type = t.type; }
	}

	// Create from expression:
	init(Expr expr) {
		init { pos = expr.pos; type = expr.result.type.type; }
	}

	// Create from possibly ambiguous name:
	init(Block block, SStr name) {
		init { pos = name.pos; }

		SimpleName simple(name.v);
		var found = block.scope.find(simple);
		if (found as Type) {
			this.type = found;
		} else if (found as LocalVar) {
			if (found.array) {
				this.arraySrc = found;
			} else {
				this.type = found.type;
			}
		} else if (found as CppGlobalArray) {
			this.arraySrc = found;
		} else {
			// Fall back.
			var expr = NameExpr(name.pos, block.scope, simple).resolve();
			this.type = expr.result.type.type;
		}
	}

	// Location.
	SrcPos pos;

	// Type described, if any.
	Type? type;

	// Special case if we describe an array.
	Named? arraySrc;

	// Extract the type from here. Used for 'malloc'.
	Type extract() {
		unless (type)
			throw SyntaxError(pos, "Invalid sizeof expression for use with 'malloc'.\nThis implementation needs to know the type of memory allocations to function properly. Try 'malloc(n*sizeof(type))' instead.");
		type;
	}
}


/**
 * Dynamic sizeof for dynamic sized arrays.
 */
class DynamicSizeof extends Expr {
	init(SrcPos pos, LocalVar var) {
		init(pos) { var = var; }
	}

	LocalVar var;

	ExprResult result() {
		return ExprResult(named{Int});
	}

	void code(CodeGen gen, CodeResult res) {
		if (!res.needed())
			return;

		var to = res.location(gen);
		if (res.type.ref) {
			CppVar var = allocType(gen, named{Int});

			gen.l << fnParam(ptrDesc, this.var.var.v);
			gen.l << fnCall(named{readArraySize<unsafe:RawPtr>}.ref, false, intDesc, ecx);

			var.adjust(gen, ptrA);
			gen.l << mov(intRel(ptrA), ecx);
			var.created(gen);
		} else {
			gen.l << fnParam(ptrDesc, var.var.v);
			gen.l << fnCall(named{readArraySize<unsafe:RawPtr>}.ref, false, intDesc, to);
		}
		res.created(gen);
	}

	void codePtr(CodeGen gen, Type type) {
		CppVar var = allocTemporary(gen, type);

		gen.l << fnParam(ptrDesc, this.var.var.v);
		gen.l << fnCall(named{readArraySize<unsafe:RawPtr>}.ref, false, intDesc, ecx);

		var.adjust(gen, ptrA);
		gen.l << mov(intRel(ptrA), ecx);
		var.created(gen);
		gen.l << location(SrcPos());
		var.ptr(gen);
	}

	private Nat readArraySize(unsafe:RawPtr alloc) : static {
		alloc.readSize * alloc.readCount;
	}
}


/**
 * Char literal.
 */
IntLiteral charLiteral(SStr content) {
	Str c = content.v.unescape();
	var iter = c.begin();
	if (iter == c.end())
		throw SyntaxError(content.pos, "Can not have an empty character literal.");

	Char ch = iter.v;
	++iter;
	if (iter != c.end())
		throw SyntaxError(content.pos, "This character literal contains more than one character!");

	IntLiteral(content.pos, ch.codepoint.int);
}

/**
 * String literal.
 */
class StrLiteral extends ExprRef {
	init(SrcPos pos, Str val) {
		init(pos) {
			val = val.unescape();
		}
	}

	private Str val;

	ExprResult result() : override {
		Value(named{ConstPtr<Byte>}, true);
	}

	void code(CodeGen gen, CodeResult res) : override {
		if (!res.needed())
			return;

		if (res.type.ref) {
			CppVar var = allocType(gen, named{ConstPtr<Byte>});
			codeRef(gen, var);
			var.adjust(gen, res.location(gen));
			res.created(gen);
		} else {
			gen.l << lea(ptrA, res.location(gen));
			loadStr(gen, ptrB);
			gen.l << mov(ptrRel(ptrA, Offset()), ptrB);
			gen.l << mov(intRel(ptrA, Offset(sPtr)), natConst(sPtr*2));
			res.created(gen);
		}
	}

	void codeRef(CodeGen gen, CppVar var) : override {
		var.adjust(gen, ptrA);
		loadStr(gen, ptrB);
		gen.l << mov(ptrRel(ptrA, Offset()), ptrB);
		gen.l << mov(intRel(ptrA, Offset(sPtr)), natConst(sPtr*2));
		var.created(gen);
	}

	private void loadStr(CodeGen gen, Reg to) {
		var member = named{ByteStr:ptr<ByteStr>};

		gen.l << mov(to, objPtr(ByteStr(pos, val)));
		gen.l << mov(to, ptrRel(to, member.offset));
	}

	// Wrapper so that we may store a raw pointer to the array somewhere!
	class ByteStr {
		unsafe:RawPtr ptr;

		init(SrcPos pos, Str str) {
			Nat count = 1;
			for (c in str) {
				count++;
				if (c.codepoint > 127)
					throw SyntaxError(pos, "This implementation only supports strings containing ASCII characters.");
			}

			unsafe:RawPtr array = unsafe:RawPtr:allocArray(named{Byte}, count);
			count = 0;
			for (c in str) {
				array.writeByte(count, c.codepoint.byte);
				count++;
			}
			array.writeByte(count, 0);
			array.writeFilled((count + 1) | AllocFlags:arrayAlloc.v | AllocFlags:staticAlloc.v);

			init {
				ptr = array;
			}
		}
	}
}

/**
 * Constant used to represent an offset parameter.
 *
 * Does not generate any code, just used during lookup.
 */
class OffsetExpr extends Expr {
	init() { init(SrcPos()) {} }
	ExprResult result() : override {
		Value(named{Nat}, false);
	}
	void code(CodeGen gen, CodeResult res) : override {
		throw InternalError("Should not use OffsetExpr to generate code!");
	}
	void codePtr(CodeGen gen, Type res) : override {
		throw InternalError("Should not use OffsetExpr to generate code!");
	}
}

/**
 * Constant used to represent a 'this' parameter for something.
 */
class TypeThisExpr extends Expr {
	init(Type type) { init(SrcPos()) { type = type; } }
	Type type;
	ExprResult result() : override {
		thisPtr(type);
	}
	void code(CodeGen gen, CodeResult res) : override {
		throw InternalError("Should not use TypeThisExpr to generate code!");
	}
	void codePtr(CodeGen gen, Type type) : override {
		throw InternalError("Should not use TypeThisExpr to generate code!");
	}
}

/**
 * Boolean literals.
 */
class BoolLiteral extends Expr {
	init(SrcPos pos, Bool val) {
		init(pos) { val = val; }
	}

	private Bool val;

	ExprResult result() : override {
		Value(named{Bool}, true);
	}

	void code(CodeGen gen, CodeResult res) : override {
		if (!res.needed())
			return;

		Byte val = if (this.val) { 1b; } else { 0b; };
		var to = res.location(gen);
		if (res.type.ref) {
			CppVar var = allocType(gen, named{Bool});
			var.adjust(gen, ptrA);
			gen.l << mov(byteRel(ptrA), byteConst(val));
			gen.l << mov(to, ptrA);
		} else {
			gen.l << mov(to, byteConst(val));
		}
		res.created(gen);
	}

	void codePtr(CodeGen gen, Type type) : override {
		Byte val = if (this.val) { 1b; } else { 0b; };

		CppVar var = allocTemporary(gen, named{Bool});
		var.adjust(gen, ptrA);
		gen.l << mov(byteRel(ptrA), byteConst(val));
		var.created(gen);
		gen.l << location(SrcPos());
		var.ptr(gen);
	}
}

/**
 * Nullptr.
 */
class Nullptr extends Expr {
	init(SrcPos pos) {
		init(pos) {}
	}

	ExprResult result() : override {
		// We don't know yet. Perhaps we should say "void *"?
		Value();
	}

	Bool suggestResult(Type type) : override {
		if (type as PtrType)
			return !type.isRef;
		false;
	}

	void code(CodeGen gen, CodeResult res) : override {
		if (!res.needed())
			return;

		var to = res.location(gen);
		if (res.type.ref) {
			unless (type = res.type.type)
				throw InternalError("Unable to create a nullptr for 'void'.");

			// We don't need to initialize it. Zero is exactly what we want!
			CppVar var = allocType(gen, type);
			var.created(gen);

			var.adjust(gen, to);
		} else {
			gen.l << mov(ptrRel(to), ptrConst(0));
			gen.l << mov(intRel(to, Offset(sPtr)), natConst(0));
		}
		res.created(gen);
	}

	void codePtr(CodeGen gen, Type type) : override {
		// No initialization needed. Allocations are zero-initialized by default.
		CppVar var = allocTemporary(gen, type);
		var.created(gen);
		gen.l << location(SrcPos());
		var.ptr(gen);
	}
}


/**
 * Access the this ptr.
 */
class ThisPtr extends ExprRef {
	FnRoot fn;

	init(SrcPos pos, Scope scope) {
		init(pos) { fn = expectFnRoot(scope, pos); }
	}

	ExprResult result() : override {
		wrapPtr(fn.owner.params[0].asRef(false)).asRef();
	}

	void code(CodeGen gen, CodeResult res) : override {
		if (!res.needed())
			return;

		if (res.type.ref) {
			CppVar var = allocType(gen, safeType(res.type, pos));
			codeRef(gen, var);

			var.adjust(gen, res.location(gen));
			res.created(gen);
		} else {
			codePtr(gen, safeType(res.type, pos));
			var location = res.location(gen);
			gen.l << mov(ptrRel(location, Offset()), ptrB);
			gen.l << mov(intRel(location, Offset(sPtr)), ecx);
			res.created(gen);
		}
	}

	void codePtr(CodeGen gen, Type type) : override {
		gen.l << mov(ptrB, fn.thisPtr);
		gen.l << mov(ecx, fn.thisOffset);

		// Compute the base ptr.
		gen.l << ucast(ptrA, ecx);
		gen.l << sub(ptrB, ptrA);
	}

	void codeRef(CodeGen gen, CppVar var) : override {
		codePtr(gen, safeType(result.type, pos));

		var.adjust(gen, ptrA);
		gen.l << mov(ptrRel(ptrA, Offset()), ptrB);
		gen.l << mov(intRel(ptrA, Offset(sPtr)), ecx);
		var.created(gen);
	}

	// Get the 'this' ptr variable.
	Var ptr() {
		fn.thisPtr;
	}

}

/**
 * Node representing a yet unresolved identifier.
 *
 * If used on its own, it will try to act as a variable access.
 */
class NameExpr extends Expr {
	Scope scope;
	SimpleName name;
	Expr? context;
	private Expr? resolved;

	// Regular name without any context.
	init(SrcPos pos, Scope scope, SimpleName name) {
		init(pos) { scope = scope; name = name; }
	}

	// Name give some context using the . operator.
	init(SrcPos pos, Scope scope, SimpleName name, Expr context) {
		init(pos) { scope = scope; name = name; context = deref(context); }
	}

	void code(CodeGen gen, CodeResult result) : override {
		resolve.code(gen, result);
	}

	ExprResult result() : override {
		resolve.result();
	}

	Bool temporary() : override {
		resolve.temporary();
	}

	Bool constant() : override {
		resolve.constant();
	}

	void codePtr(CodeGen gen, Type type) : override {
		resolve.codePtr(gen, type);
	}

	Variant staticValue(Type t) : override {
		resolve.staticValue(t);
	}

	Expr resolve() {
		if (resolved)
			return resolved;

		Expr r = doResolve();
		resolved = r;
		r;
	}

	// Result from 'findNamed'.
	class Found on Compiler {
		// What we found.
		Named named;

		// Any context used.
		Expr? context;

		// Did we use an offset pointer for the context?
		Bool contextOffset;

		// Create.
		init(Named named) { init() { named = named; } }
		init(Named named, Expr? context, Bool offset) { init() { named = named; context = context; contextOffset = offset; } }
	}

	// Find a name with a given context.
	private Found? findCtx(Expr context, Expr[] actuals) {
		Expr[] params = actuals.clone();
		params.insert(0, context);

		if (t = context.result.type.type) {
			if (found = t.find(CppPart(name.last.name, params), scope)) {
				return Found(found, context, false);
			}

			// Try with an offset:
			params.insert(1, OffsetExpr());
			if (found = t.find(CppPart(name.last.name, params), scope)) {
				return Found(found, context, true);
			}
		}
		return null;
	}

	// Will try to add '*this' as a context if suitable.
	Found? findNamed(Expr[] actuals) {
		if (context) {
			return findCtx(deref(context), actuals);
		}

		// If empty, see if we find a local variable, they have priority over member variables.
		if (actuals.empty) {
			if (found = scope.find(name) as LocalVar)
				return Found(found);
		}

		// Try to use 'this' if suitable.
		if (fnRoot = findFnRoot(scope)) {
			if (member = fnRoot.memberOf) {
				if (found = findCtx(Deref:ptr(ThisPtr(pos, scope)), actuals))
					return found;
			}
		}

		// Try the global scope.
		SimpleName n = name.clone;
		n.last = CppPart(name.last.name, actuals);
		if (found = scope.find(n))
			return Found(found);

		return null;
	}

	Found? findNamed() {
		findNamed([]);
	}

	private Expr doResolve() {
		Found? found = findNamed();
		unless (found) {
			if (context) {
				if (context.result.type.isCppPtr)
					throw SyntaxError(pos, "Unable to resolve '${name}' in this context. Did you mean to use '->'?");
			}

			throw SyntaxError(pos, "Unable to resolve the name '${name}' in this context!");
		}

		if (n = found.named as LocalVar) {
			LocalVarAccess access(pos, n);
			if (n.array)
				return AddressOf(access);
			else
				return access;
		} else if (n = found.named as MemberVar) {
			if (c = found.context) {
				return MemberVarAccess(pos, c, n);
			}
		} else if (n = found.named as GlobalVar) {
			return GlobalVarAccess(pos, n);
		}

		throw SyntaxError(pos, "'${name}' refers to ${typeOf(found.named).name}, which I don't know how to handle here.");
	}
}

// Try to "convert" a name expression into a function call.
Expr fnCall(SrcPos pos, Expr lhs, Expr[] actuals) {
	unless (lhs as NameExpr) {
		// TODO: We should try to call the function call operator on the object.
		throw SyntaxError(lhs.pos, "This is not something we can call.");
	}

	NameExpr:Found? found = lhs.findNamed(actuals);
	unless (found) {
		try {
			return ctorCall(pos, lhs, actuals);
		} catch (SyntaxError e) {
			throw SyntaxError(lhs.pos, "Could not find anything named ${lhs.name} with the supplied paramers to call.");
		}
	}

	if (fn = found.named as Function) {
		// Add the 'this' ptr if required.
		Bool expandFirst = false;
		if (c = found.context) {
			actuals.insert(0, c);
			expandFirst = found.contextOffset;
		}
		return FnCall(pos, fn, actuals, expandFirst);
	}

	throw SyntaxError(lhs.pos, "Found ${found}, which is not a function.");
}

// Try to "convert" a name expression into a constructor call.
Expr ctorCall(SrcPos pos, Expr lhs, Expr[] actuals) {
	unless (lhs as NameExpr) {
		throw SyntaxError(pos, "This is not the name of a type.");
	}

	unless (type = lhs.scope.find(lhs.name) as Type) {
		throw SyntaxError(pos, "${lhs.name} does not the name of a type.");
	}

	if (ctor = cppCtor(type, actuals, lhs.scope))
		return CtorCall(pos, ctor, actuals, true);

	if (ctor = stormCtor(type, actuals, lhs.scope))
		return CtorCall(pos, ctor, actuals, false);

	throw SyntaxError(pos, "Unable to find a suitable constructor to call.");
}


/**
 * Function calls.
 */
class FnCall extends ExprRef {
	private Function fn;
	private Value fnResult;
	private Expr[] actuals;

	// Expand the first parameter into a pointer and an offset. Used for 'this' pointers.
	private Bool addOffset;

	init(SrcPos pos, Function fn, Expr[] actuals, Bool addOffset) {
		Value result = fn.result;
		if (fn as CppFunction) {
			result = fn.findResult();
		}

		init(pos) {
			fn = fn;
			fnResult = result;
			actuals = actuals;
			addOffset = addOffset;
		}
	}

	ExprResult result() : override {
		// We make sure to return references.
		fnResult.asRef();
	}

	Variant staticValue(Type type) : override {
		// Only try to "constant fold" pure functions.
		if (!fn.pure)
			return Variant();

		if (!Value(type).mayReferTo(fnResult))
			return Variant();

		Variant[] params;
		for (i, x in actuals) {
			unless (type = fn.params[i].type)
				return Variant();

			Variant a = x.staticValue(type);
			if (a.empty)
				return Variant();

			params << a;
		}

		try {
			return dynamicCall(fn, params);
		} catch (UsageError error) {
			// This is mostly a last resort. If something fails, we just state that we could not
			// compute the value statically.
			print(error.toS);
			return Variant();
		}
	}

	void code(CodeGen gen, CodeResult res) : override {
		Operand[] params = createParams(gen);

		if (fn.hasResultParam) {
			resParamCode(gen, res, params);
			return;
		}

		// The regular case.
		plainCode(gen, res, params);
	}

	// Generate code for evaluating parameters.
	private Operand[] createParams(CodeGen gen) {
		Operand[] params;
		Nat skip = actuals.count;

		// Skip adding the result param.
		if (fn.hasResultParam) {
			if (fn.isMember)
				skip = 1;
			else
				skip = 0;
		}

		Nat formalId = 0;
		for (i, x in actuals) {
			if (i == skip)
				formalId++;

			if (i == 0 & addOffset) {
				paramOffsetCode(params, gen, x, fn.params[0], fn.params[1]);
				formalId += 2;
			} else {
				params << paramCode(gen, x, fn.params[formalId]);
				formalId += 1;
			}
		}

		params;
	}

	// Call a regular, Storm function, or a C++ function without param.
	private void plainCode(CodeGen gen, CodeResult res, Operand[] params) {
		addBarrier(gen);

		if (t = fnResult.type) {
			if (res.type.ref & !fn.result.ref) {
				// Create a variable to store the result in.
				CppVar var = allocType(gen, t);
				var ref = res.safeLocation(gen, res.type);
				var.adjust(gen, ref);

				fn.autoCallRef(gen, params, ref);
				var.created(gen);
				res.created(gen);
			} else {
				fn.autoCall(gen, params, res);
			}
		} else {
			// Void result.
			fn.autoCall(gen, params, res);
		}
	}

	// Call a function that returns the value through a pointer.
	private void resParamCode(CodeGen gen, CodeResult res, Operand[] params) {
		unless (t = fnResult.type)
			throw InternalError("Void values should never be returned by pointer.");

		addBarrier(gen);

		// Create a variable to store the result in.
		CppVar var = allocTemporary(gen, t);
		var.created(gen);

		// Create a pointer and initialize it. It is fine to create it on the stack.
		Var ptr = gen.l.createVar(gen.block, sPtr*2);
		gen.l << mov(ptrRel(ptr, Offset()), var.v);
		gen.l << mov(intRel(ptr, Offset(sPtr)), natConst(sPtr*2));
		gen.l << lea(ptrA, ptr);

		// Then we can just call the function!
		if (!fn.isMember)
			gen.l << fnParam(ptrDesc, ptrA);
		for (i, param in params) {
			gen.l << fnParam(fn.params[i].desc(), param);

			if (fn.isMember & addOffset & i == 1)
				gen.l << fnParam(ptrDesc, ptrA);
			else if (fn.isMember & !addOffset & i == 0)
				gen.l << fnParam(ptrDesc, ptrA);
		}

		gen.l << fnCall(fn.ref, fn.isMember);

		// Store the pointer or the value somewhere.
		if (res.type.ref) {
			var.adjust(gen, res.safeLocation(gen, result.type));
		} else {
			// TODO: We should look at this.
			throw InternalError("Returning values of complex types is not yet supported.");
		}

		res.created(gen);
	}

	// Add a barrier if needed.
	private void addBarrier(CodeGen gen) {
		var type = fnBarrier(fn);
		if (type != progvis:program:Barrier:none)
			gen.l << meta(progvis:program:SrcBarrier(type, pos));
	}

	// Generate code for when someone else has already created the storage for us.
	void codeRef(CodeGen gen, CppVar var) : override {
		Operand[] params = createParams(gen);

		if (fn.hasResultParam()) {
			resParamCodeRef(gen, var, params);
			return;
		}

		plainCodeRef(gen, var, params);
	}

	// Call a regular, Storm function, or a C++ function without param.
	private void plainCodeRef(CodeGen gen, CppVar var, Operand[] params) {
		if (t = fn.result.type) {
			if (!fn.result.ref) {
				Var v = gen.l.createVar(gen.block, sPtr);
				var.adjust(gen, v);

				fn.autoCallRef(gen, params, v);
				var.created(gen);
			} else if (copyCtor = t.copyCtor) {
				// Need to copy!
				CodeResult r(fn.result, gen.block);
				fn.autoCall(gen, params, r);

				var.adjust(gen, ptrB);

				gen.l << lea(ptrA, r.location(gen));
				gen.l << fnParam(ptrDesc, ptrB);
				gen.l << fnParam(ptrDesc, ptrA);
				gen.l << fnCall(copyCtor.ref, true);
				var.created(gen);
			} else {
				throw SyntaxError(pos, "Unable to copy the result.");
			}
		} else {
			// Void result.
			fn.autoCall(gen, params, CodeResult());
		}
	}

	// Call a function that returns the value through a pointer.
	private void resParamCodeRef(CodeGen gen, CppVar var, Operand[] params) {
		unless (t = fnResult.type)
			throw InternalError("Void values should never be returned by pointer.");

		// So that it shows up properly in the visualization.
		var.created(gen);
		gen.l << location(SrcPos());

		// Create a pointer and initialize it. It is fine to create it on the stack.
		Var ptr = gen.l.createVar(gen.block, sPtr*2);
		gen.l << mov(ptrRel(ptr, Offset()), var.v);
		gen.l << mov(intRel(ptr, Offset(sPtr)), natConst(sPtr*2));
		gen.l << lea(ptrA, ptr);

		// Then we can just call the function!
		if (!fn.isMember)
			gen.l << fnParam(ptrDesc, ptrA);

		for (i, param in params) {
			gen.l << fnParam(fn.params[i].desc(), param);

			if (fn.isMember & addOffset & i == 1)
				gen.l << fnParam(ptrDesc, ptrA);
			else if (fn.isMember & !addOffset & i == 0)
				gen.l << fnParam(ptrDesc, ptrA);
		}

		gen.l << fnCall(fn.ref, fn.isMember);
	}
}

/**
 * Constructor calls.
 */
class CtorCall extends ExprRef {
	private Function fn;
	private Expr[] actuals;
	// Expand the first parameter into a pointer and an offset. Used for 'this' pointers.
	private Bool addOffset;


	init(SrcPos pos, Function fn, Expr[] actuals, Bool addOffset) {
		init(pos) {
			fn = fn;
			actuals = actuals;
			addOffset = addOffset;
		}
	}

	ExprResult result() : override {
		// We make sure to return references.
		fn.params[0].asRef();
	}

	void code(CodeGen gen, CodeResult res) : override {
		unless (type = fn.params[0].type) {
			throw SyntaxError(pos, "This copy-constructor seems to create 'void' types...");
		}

		Operand[] params;
		Nat offset = 1;
		if (addOffset)
			offset++;
		for (i, x in actuals) {
			params << paramCode(gen, x, fn.params[i + offset]);
		}

		if (res.type.ref) {
			CppVar var = allocType(gen, type);
			Var ref = res.safeLocation(gen, res.type);
			var.created(gen); // A bit early, but for the visualization.
			gen.l << location(SrcPos()); // To make the visualization update its display, but not
										 // stop. Otherwise this object might be incorrectly shown
										 // on the heap.

			var.adjust(gen, ref);

			params.insert(0, ref);
			if (addOffset) {
				Var offset = gen.l.createVar(gen.block, sNat);
				gen.l << mov(offset, natConst(sPtr * 2));
				params.insert(1, offset);
			}
			fn.autoCall(gen, params, CodeResult());
		} else {
			if (addOffset)
				throw InternalError("Can not use constructors to generate value types");
			Var val = res.safeLocation(gen, result.type);
			gen.l << lea(ptrA, val);
			params.insert(0, ptrA);
			fn.autoCall(gen, params, CodeResult());
		}
		res.created(gen);
	}

	// Generate code for when someone else has already created the storage for us.
	void codeRef(CodeGen gen, CppVar var) : override {
		unless (type = fn.params[0].type) {
			throw SyntaxError(pos, "This copy-constructor seems to create 'void' types...");
		}

		Operand[] params;
		Nat offset = 1;
		if (addOffset)
			offset++;
		for (i, x in actuals) {
			params << paramCode(gen, x, fn.params[i + offset]);
		}

		var.created(gen); // A bit early, for the visualization to work.
		gen.l << location(SrcPos()); // To make the visualization update its display, but not
									 // stop. Otherwise this object might incorrectly be shown on the heap.

		var.adjust(gen, ptrA);
		params.insert(0, ptrA);

		if (addOffset) {
			Var offset = gen.l.createVar(gen.block, sNat);
			gen.l << mov(offset, natConst(sPtr * 2));
			params.insert(1, offset);
		}

		fn.autoCall(gen, params, CodeResult());
	}

	// Initialize a variable at an offset inside a variable. 'offset' is an index into the array
	// (ptr sized), i.e. not in bytes.
	void codeRef(CodeGen gen, CppVar var, Operand offsetVar) {
		unless (type = fn.params[0].type) {
			throw SyntaxError(pos, "This copy-constructor seems to create 'void' types...");
		}

		Operand[] params;
		Nat offset = 1;
		if (addOffset)
			offset++;
		for (i, x in actuals) {
			params << paramCode(gen, x, fn.params[i + offset]);
		}

		var.adjust(gen, ptrA);
		gen.l << mov(ptrB, offsetVar);
		gen.l << mul(ptrB, ptrConst(Value(type).size.aligned));
		gen.l << add(ptrA, ptrB);

		params.insert(0, ptrA);

		if (addOffset) {
			Var offset = gen.l.createVar(gen.block, sNat);
			gen.l << ucast(offset, ptrB);
			gen.l << add(offset, natConst(sPtr * 2));
			params.insert(1, offset);
		}

		fn.autoCall(gen, params, CodeResult());
	}
}

// Generate code for acquiring one parameter.
package Operand paramCode(CodeGen gen, Expr actual, Value formal) on Compiler {
	actual = expectCast(actual, safeType(formal, actual.pos));

	Value result = formal.asRef(actual.result.type.ref);
	if (formal.ref & !result.ref) {
		// Create a temporary variable and make a reference to it.
		result = formal.asRef(false);
		VarInfo tmpV = gen.createVar(result);
		CodeResult gr(result, tmpV);
		actual.code(gen, gr);

		VarInfo tmpRef = gen.createVar(formal);
		gen.l << lea(tmpRef.v, ptrRel(tmpV.v));
		tmpRef.created(gen);
		tmpRef.v;
	} else if (!formal.ref & result.ref) {
		// Tell the implementation to create a copy for us.
		CodeResult r(formal.asRef(false), gen.block);
		actual.code(gen, r);
		return r.location(gen);
	} else {
		CodeResult gr(formal, gen.block);
		actual.code(gen, gr);
		gr.location(gen);
	}
}

// Generate code for the first parameter, and a separate parameter for the offset.
// We assume that 'formal1' is a reference and 'formal2' is an integer.
private void paramOffsetCode(Operand[] out, CodeGen gen, Expr actual, Value formal1, Value formal2) on Compiler {
	actual = expectCast(actual, safeType(formal1, actual.pos));
	if (!formal1.ref | formal2.ref)
		throw InternalError("Cannot use 'paramOffsetCode' in this context! It is intended for 'this' ptrs.");

	var ptr = gen.createVar(formal1);
	var offset = gen.createVar(formal2);

	actual.codePtr(gen, safeType(actual.result.type, actual.pos));
	// Do the calculation now, so we get a real reference.
	gen.l << ucast(ptrA, ecx);
	gen.l << add(ptrB, ptrA);
	gen.l << mov(ptr.v, ptrB);

	// But store the offset as well.
	gen.l << mov(offset.v, ecx);

	out << ptr.v;
	out << offset.v;
	ptr.created(gen);
	offset.created(gen);
}

// Create an operator.
Expr operator(Block block, Expr lhs, SStr op, Expr rhs) on Compiler {
	lhs = deref(lhs);
	rhs = deref(rhs);

	var lType = lhs.result.type;
	var pos = lhs.pos.extend(rhs.pos);

	CppPart opName(op.v, [lhs, rhs]);
	if (t = lType.type) {
		if (fn = t.find(opName, block.scope) as Function)
			return FnCall(pos, fn, [lhs, rhs], false);

		if (fn = t.find(CppPart(op.v, [lhs, OffsetExpr(), rhs]), block.scope) as Function)
			return FnCall(pos, fn, [lhs, rhs], true);
	}

	if (fn = block.scope.find(SimpleName(opName)) as Function) {
		return FnCall(pos, fn, [lhs, rhs], false);
	}

	throw SyntaxError(op.pos, "Failed to find operator ${op.v} for ${lType} and ${rhs.result.type}.");
}

// Operator, that don't accept LHS to be constant.
Expr operatorNoConst(Block block, Expr lhs, SStr op, Expr rhs) on Compiler {
	lhs = deref(lhs);
	rhs = deref(rhs);

	// This is mostly to be able to re-use Storm operators. This does disallow some possible (but
	// nonsensical) implementations of assignment operators in C++.
	if (lhs.constant)
		throw SyntaxError(op.pos, "Can not call operator ${op.v} with a left hand side marked with const.");

	var lType = lhs.result.type;
	var pos = lhs.pos.extend(rhs.pos);

	CppPart opName(op.v, [lhs, rhs]);
	if (t = lType.type) {
		if (fn = t.find(opName, block.scope) as Function)
			return FnCall(pos, fn, [lhs, rhs], false);

		if (fn = t.find(CppPart(op.v, [lhs, OffsetExpr(), rhs]), block.scope) as Function)
			return FnCall(pos, fn, [lhs, rhs], true);
	}

	if (fn = block.scope.find(SimpleName(opName)) as Function) {
		return FnCall(pos, fn, [lhs, rhs], false);
	}

	throw SyntaxError(op.pos, "Failed to find operator ${op.v} for ${lType} and ${rhs.result.type}.");
}

// Doesn't add a star to the name.
Expr unaryOperator(Block block, SStr op, Expr rhs) on Compiler {
	rhs = deref(rhs);

	var rType = rhs.result.type;
	var pos = op.pos.extend(rhs.pos);

	CppPart opName(op.v, [rhs]);
	if (t = rType.type) {
		if (fn = t.find(opName, block.scope) as Function)
			return FnCall(pos, fn, [rhs], false);

		if (fn = t.find(CppPart(op.v, [rhs, OffsetExpr()]), block.scope) as Function)
			return FnCall(pos, fn, [rhs], true);
	}

	if (fn = block.scope.find(SimpleName(opName)) as Function) {
		return FnCall(pos, fn, [rhs], false);
	}

	throw SyntaxError(op.pos, "Failed to find prefix operator ${op.v} for ${rType}.");
}

Expr operatorNoConst(Block block, SStr op, Expr rhs) on Compiler {
	rhs = deref(rhs);

	// This is mostly to be able to re-use Storm operators. This does disallow some possible (but
	// nonsensical) implementations of pre- and post increment operators in C++.
	if (rhs.constant)
		throw SyntaxError(op.pos, "Can not call a prefix operator ${op.v} on a constant value.");

	var rType = rhs.result.type;
	var pos = op.pos.extend(rhs.pos);

	CppPart opName(op.v + "*", [rhs]);
	if (t = rType.type) {
		if (fn = t.find(opName, block.scope) as Function)
			return FnCall(pos, fn, [rhs], false);

		if (fn = t.find(CppPart(op.v, [rhs, OffsetExpr()]), block.scope) as Function)
			return FnCall(pos, fn, [rhs], true);
	}

	if (fn = block.scope.find(SimpleName(opName)) as Function) {
		return FnCall(pos, fn, [rhs], false);
	}

	throw SyntaxError(op.pos, "Failed to find prefix operator ${op.v} for ${rType}.");
}

Expr operatorNoConst(Block block, Expr lhs, SStr op) on Compiler {
	lhs = deref(lhs);

	// This is mostly to be able to re-use Storm operators. This does disallow some possible (but
	// nonsensical) implementations of pre- and post increment operators in C++.
	if (lhs.constant)
		throw SyntaxError(op.pos, "Can not call a postfix operator ${op.v} with a constant value.");

	var lType = lhs.result.type;
	var pos = lhs.pos.extend(op.pos);

	CppPart opName("*" + op.v, [lhs]);
	if (t = lType.type) {
		if (fn = t.find(opName, block.scope) as Function)
			return FnCall(pos, fn, [lhs], false);

		if (fn = t.find(CppPart(op.v, [lhs, OffsetExpr()]), block.scope) as Function)
			return FnCall(pos, fn, [lhs], true);
	}

	if (fn = block.scope.find(SimpleName(opName)) as Function) {
		return FnCall(pos, fn, [lhs], false);
	}

	throw SyntaxError(op.pos, "Failed to find postfix operator ${op.v} for ${lType}.");
}

Expr dotOperator(Block block, Expr lhs, SStr rhs, Bool ptrLhs) {
	// If lhs is a pointer, just dereference it first. That's what the dot operator means anyway.
	if (ptrLhs)
		lhs = Deref:ptr(lhs);
	NameExpr(lhs.pos.extend(rhs.pos), block.scope, SimpleName(rhs.v), lhs);
}

Expr arrayOperator(SrcPos pos, Block block, Expr base, Expr offset) {
	// Basically: *(<base> + <offset>)
	var result = Deref:ptr(operator(block, base, SStr("+", pos), offset));
	result.pos = pos;
	result;
}

// Called from the syntax, performs basic sanity checking.
// We don't want to do that in the AddressOf ctor, since we actually do take the
// address of, or references to, temporary values in normal operations.
AddressOf addressOf(SrcPos pos, Expr of) on Compiler {
	if (of.temporary) {
		throw SyntaxError(pos, "Can not create a pointer to a temporary value.");
	}

	AddressOf(pos, of);
}


/**
 * Make a pointer from an l-value. The & operator.
 */
class AddressOf extends ExprRef {
	init(Expr of) {
		init(of.pos) { of = of; }
	}

	init(SrcPos pos, Expr of) {
		init(pos) { of = of; }
	}

	Expr of;

	ExprResult result() : override {
		if (of.constant)
			wrapConstPtr(of.result.type).asRef();
		else
			wrapPtr(of.result.type).asRef();
	}

	void code(CodeGen gen, CodeResult res) : override {
		Value resType = result.type;
		unless (type = resType.type)
			throw InternalError("Failed to find an appropriate pointer type!");

		if (!res.needed())
			return;

		if (res.type.ref) {
			CppVar var = allocType(gen, type);
			codeRef(gen, var);

			var.adjust(gen, res.location(gen));
		} else {
			ptrNoCheck(of, gen, type);
			gen.l << lea(ptrA, res.location(gen));
			gen.l << mov(ptrRel(ptrA, Offset()), ptrB);
			gen.l << mov(intRel(ptrA, Offset(sPtr)), ecx);
		}
		res.created(gen);
	}

	void codeRef(CodeGen gen, CppVar var) : override {
		ptrNoCheck(of, gen, safeType(of.result.type, pos));

		var.adjust(gen, ptrA);
		gen.l << mov(ptrRel(ptrA, Offset()), ptrB);
		gen.l << mov(intRel(ptrA, Offset(sPtr)), ecx);
		var.created(gen);
	}

	void codePtr(CodeGen gen, Type type) : override {
		CppVar var = allocTemporary(gen, type);
		codeRef(gen, var);
		gen.l << location(SrcPos());
		var.ptr(gen);
	}

	// Call 'codePtr' without checking for buffer overruns.
	private void ptrNoCheck(Expr of, CodeGen gen, Type type) {
		if (of as Deref) {
			of.codePtrNoCheck(gen, type);
		} else {
			of.codePtr(gen, type);
		}
	}
}

/**
 * Make a reference from an r-value or a l-value.
 */
class MakeRef extends AddressOf {
	init(Expr of) {
		init(of) {}
	}

	// We just need to tell 'AddressOf' to produce a reference instead.
	ExprResult result() : override {
		wrapRef(of.result.type).asRef();
	}
}

/**
 * Make an r-value reference.
 */
class MakeRRef extends AddressOf {
	init(Expr of) {
		init(of) {}
	}

	ExprResult result() : override {
		wrapRRef(of.result.type).asRef();
	}
}

/**
 * De-reference a pointer or reference.
 */
class Deref extends Expr {
	private init(Expr of) {
		init(of.pos) { of = of; }
	}

	Deref ptr(Expr original) : static {
		unless (original.result.type.isCppPtr)
			throw SyntaxError(original.pos, "Unable to dereference a value of type ${original.result.type}.");

		Deref(original);
	}

	Deref ptr(SrcPos pos, Expr original) : static {
		Deref r = ptr(original);
		r.pos = pos;
		r;
	}

	Deref ref(Expr original) : static {
		unless (original.result.type.isCppRef)
			throw SyntaxError(original.pos, "Unable to dereference a value of type ${original.result.type}. Not a reference.");

		Deref(original);
	}

	Expr of;

	ExprResult result() : override {
		unwrapPtrOrRef(of.result.type).asRef();
	}

	void code(CodeGen gen, CodeResult res) : override {
		if (!res.needed())
			return;

		if (of as ThisPtr) {
			// If it is the this-ptr, we can just read it.
			gen.l << mov(ptrB, of.ptr);
		} else {
			CodeResult p(of.result.type.asRef(), gen.block);
			of.code(gen, p);

			check(gen, p.location(gen), result.type.asRef(false).size());

			gen.l << mov(ptrA, p.location(gen));

			// Load the base.
			gen.l << mov(ptrB, ptrRel(ptrA));

			// Add the offset.
			gen.l << ucast(ptrC, intRel(ptrA, Offset(sPtr)));
			gen.l << add(ptrB, ptrC);
		}

		if (res.type.ref) {
			// Store it!
			gen.l << mov(res.location(gen), ptrB);
		} else if (res.type.isAsmType()) {
			// Copy the value.
			Var dst = res.location(gen);
			gen.l << mov(dst, xRel(dst.size, ptrB));
		} else {
			// Copy ctor.
			gen.l << fnParam(ptrDesc, ptrA);
			gen.l << fnParam(ptrDesc, ptrB);
			gen.l << fnCall(res.type.copyCtor, true);
		}
		res.created(gen);
	}

	Bool temporary() {
		false;
	}

	Bool constant() {
		if (t = of.result.type.type as PtrType) {
			t.isConst;
		} else {
			false;
		}
	}

	void codePtr(CodeGen gen, Type type) : override {
		CodeResult p(of.result.type.asRef(), gen.block);
		of.code(gen, p);

		check(gen, p.location(gen), result.type.asRef(false).size());

		gen.l << mov(ptrA, p.location(gen));

		// Load the base and offset.
		gen.l << mov(ptrB, ptrRel(ptrA, Offset()));
		gen.l << mov(ecx, intRel(ptrA, Offset(sPtr)));
	}

	// Like 'codePtr' but does not check for buffer overruns.
	void codePtrNoCheck(CodeGen gen, Type type) {
		CodeResult p(of.result.type.asRef(), gen.block);
		of.code(gen, p);

		gen.l << mov(ptrA, p.location(gen));

		// Load the base and offset.
		gen.l << mov(ptrB, ptrRel(ptrA, Offset()));
		gen.l << mov(ecx, intRel(ptrA, Offset(sPtr)));
	}

	// Generate a check for pointer validity.
	void check(CodeGen gen, Var src, Size size) {
		var checkFn = named{checkPtr<unsafe:RawPtr, Nat, Nat>};

		gen.l << mov(ptrA, src);
		gen.l << fnParam(ptrDesc, ptrRel(ptrA, Offset()));
		gen.l << fnParam(intDesc, intRel(ptrA, Offset(sPtr)));
		gen.l << fnParam(intDesc, natConst(size));
		gen.l << fnCall(checkFn.ref, false);
	}
}

/**
 * New operator.
 */
class NewExpr extends ExprRef {
	// Allocate uninitialized elements.
	init(SrcPos pos, Block parent, Value val) {
		var type = safeType(val, pos);
		init(pos) {
			type = type;
			initExpr = findInit(pos, parent.scope, type, []);
		}
	}

	// Allocate with initializer.
	init(SrcPos pos, Block parent, Value val, Expr[] actuals) {
		var type = safeType(val, pos);
		init(pos) {
			type = type;
			initExpr = findInit(pos, parent.scope, type, actuals);
		}
	}

	// Allocate an array.
	init(SrcPos pos, Block parent, Value val, Expr count) {
		var type = safeType(val, pos);
		init(pos) {
			type = type;
			initExpr = findInit(pos, parent.scope, type, []);
			countExpr = expectCast(count, named{Int});
		}
	}

	// Type to create.
	Type type;

	// Init expression.
	CtorCall initExpr;

	// Count expression, if we're allocating an array.
	Expr? countExpr;

	// Result type.
	ExprResult result() : override {
		wrapPtr(Value(type)).asRef();
	}

	// Compute an init expression.
	private CtorCall findInit(SrcPos pos, Scope scope, Type type, Expr[] actuals) : static {
		if (ctor = type.cppCtor(actuals, scope))
			return CtorCall(pos, ctor, actuals, true);

		if (ctor = type.stormCtor(actuals, scope))
			return CtorCall(pos, ctor, actuals, false);

		throw SyntaxError(pos, "No suitable constructor for ${type.name} found.");
	}

	// Generate code.
	void code(CodeGen gen, CodeResult res) : override {
		Type resultType = safeType(result.type, pos);

		if (res.type.ref) {
			CppVar var = allocType(gen, safeType(result.type, pos));
			codeRef(gen, var);

			var.adjust(gen, ptrA);
			gen.l << mov(res.safeLocation(gen, Value(resultType, true)), ptrA);
		} else {
			var mem = doAlloc(gen);

			gen.l << lea(ptrB, res.safeLocation(gen, Value(resultType)));
			gen.l << mov(ptrRel(ptrB, Offset()), mem);
			gen.l << mov(intRel(ptrB, Offset(sPtr)), natConst(sPtr * 2));
		}
	}

	// Initialize another variable.
	void codeRef(CodeGen gen, CppVar var) : override {
		var mem = doAlloc(gen);
		var.adjust(gen, ptrB);
		gen.l << mov(ptrRel(ptrB, Offset()), mem);
		gen.l << mov(intRel(ptrB, Offset(sPtr)), natConst(sPtr * 2));
		var.created(gen);
	}

	// Create the allocation and initialize it.
	private Operand doAlloc(CodeGen gen) {
		if (countExpr) {
			doAllocArray(gen, countExpr);
		} else {
			doAllocSingle(gen);
		}
	}

	private Operand doAllocSingle(CodeGen gen) {
		Var tmp = gen.l.createVar(gen.block, sPtr);

		// Allocate memory.
		gen.l << fnParam(ptrDesc, type.typeRef);
		gen.l << fnParam(ptrDesc, ptrConst(1));
		gen.l << fnCall(ref(BuiltIn:allocArray), false, ptrDesc, ptrA);
		gen.l << mov(tmp, ptrA);

		// Set 'filled' to "count | (1 << 32)"
		Nat mask = AllocFlags:heapAlloc.v | 1;
		gen.l << mov(ptrRel(ptrA, Offset(sPtr)), ptrConst(mask));

		// Call the constructor.
		initExpr.codeRef(gen, CppVar(tmp));

		tmp;
	}

	private Operand doAllocArray(CodeGen gen, Expr countExpr) {
		Var tmp = gen.l.createVar(gen.block, sPtr);
		Var count = gen.l.createVar(gen.block, sPtr);

		CodeResult r(Value(named{Int}), gen.block);
		countExpr.code(gen, r);

		// Check if the value is way too big.
		gen.l << fnParam(intDesc, r.location(gen));
		gen.l << fnCall(named{rangeCheck<Int>}.ref, false);

		// Proceed.
		gen.l << ucast(count, r.location(gen));

		// Allocate memory.
		gen.l << fnParam(ptrDesc, type.typeRef);
		gen.l << fnParam(ptrDesc, count);
		gen.l << fnCall(ref(BuiltIn:allocArray), false, ptrDesc, ptrA);
		gen.l << mov(tmp, ptrA);

		// Set 'filled' to "count | (1 << 32) | (1 << 31)"
		Nat mask = AllocFlags:heapAlloc.v | AllocFlags:arrayAlloc.v;
		gen.l << mov(ptrB, count);
		gen.l << or(ptrB, ptrConst(mask));
		gen.l << mov(ptrRel(ptrA, Offset(sPtr)), ptrB);

		// Call the constructors.
		Var current = gen.l.createVar(gen.block, sPtr);
		Label start = gen.l.label();
		Label again = gen.l.label();

		gen.l << mov(current, ptrConst(0));
		gen.l << jmp(start);

		gen.l << again;

		initExpr.codeRef(gen, CppVar(tmp), current);

		gen.l << add(current, ptrConst(1));

		gen.l << start;
		gen.l << cmp(current, count);
		gen.l << jmp(again, CondFlag:ifBelow);

		tmp;
	}

	// Preform a range-check on the allocation.
	private void rangeCheck(Int val) : static {
		// Some more or less sensible values. We don't allow more than ~65536 entries in an array to
		// be at least somewhat safe.
		if (val < 0 | val > 65535)
			throw InternalError("Improper size of an array allocation: ${val}");
	}
}

// Descriptive error for malloc().
Expr mallocError(SrcPos pos) {
	throw SyntaxError(pos, "Calls to malloc in this implementation must have the form: malloc(sizeof(type)), malloc(sizeof(type)*n) or malloc(n*sizeof(type)).");
}

/**
 * The comma operator.
 */
class CommaExpr extends Expr {
	init(SrcPos pos, Expr lhs, Expr rhs) {
		init(pos) {
			lhs = lhs;
			rhs = rhs;
		}
	}

	private Expr lhs;
	private Expr rhs;

	// Result.
	ExprResult result() : override {
		if (rhs.result.nothing)
			return noReturn();
		rhs.result();
	}

	// Generate code.
	void code(CodeGen gen, CodeResult res) : override {
		if (lhs as CommaExpr) {
		} else {
			gen.l << location(lhs.pos);
		}

		CodeResult first;
		lhs.code(gen, first);

		gen.l << location(rhs.pos);
		rhs.code(gen, res);
	}

	// Ptr code.
	void codePtr(CodeGen gen, Type type) : override {
		if (lhs as CommaExpr) {
		} else {
			gen.l << location(lhs.pos);
		}

		CodeResult first;
		lhs.code(gen, first);

		gen.l << location(rhs.pos);
		rhs.codePtr(gen, type);
	}

}

// Descriptive error for c-style casts.
Expr castError(SrcPos pos) {
	throw SyntaxError(pos, "C-style casts are not supported. You don't need to cast the return value of malloc().");
}
