use core:lang;
use lang:bs;
use lang:bs:macro;

/**
 * Handler definition.
 *
 * Contains a list of handler clauses, each handling an effect or a return value.
 */
class Handler extends Class {
	// The type returned from handler blocks.
	Value result;

	// Parameter accepted to the return clause. I.e. what we expect the body of the handler to
	// evaluate to.
	Value param;

	// Effect handler clauses.
	private EffectHandlerClause[] handles;
	EffectHandlerClause[] effectClauses() {
		forceLoad();
		handles;
	}

	// Return clauses.
	private HandlerClause? returnHandler;
	HandlerClause? returnClause() {
		forceLoad();
		returnHandler;
	}

	// Create.
	init(SrcPos pos, Scope scope, Str name, SHandlerBody body, Value result, Value param) {
		init(TypeFlags:typeClass, pos, scope, name, body) {
			result = result;
			param = param;
		}
	}

	class Body extends ClassBody {
		// Effect (same as 'owner' in the parent class).
		Handler handler;

		// Collected clauses.
		private SHandlerClause[] clauses;

		// Create.
		init(Class handler) {
			Handler h = if (handler as Handler) {
				handler;
			} else {
				throw InternalError("Must supply a handler to 'HandlerBody'!");
			};

			init(h) {
				handler = h;
			}
		}

		// Capture clauses.
		void add(TObject toAdd) : override {
			if (toAdd as SHandlerClause) {
				clauses << toAdd;
			} else {
				super:add(toAdd);
			}
		}

		// Load clauses when we are done with other members (functions are eagerly resolved):
		void finished() : override {
			Set<Effect> used;

			for (c in clauses) {
				var toAdd = c.transform(handler);

				if (toAdd as EffectHandlerClause) {
					if (used.has(toAdd.effect))
						throw SyntaxError(toAdd.pos, "This effect has already been handled in this handler.");

					used.put(toAdd.effect);
					handler.handles << toAdd;

				} else {
					if (handler.returnHandler.any)
						throw SyntaxError(toAdd.pos, "The 'return' handler may only be specified once.");
					handler.returnHandler = toAdd;
				}
			}

			if (handler.returnHandler.empty) {
				if (!handler.result.mayStore(handler.param)) {
					throw SyntaxError(handler.pos, "A return handler is required when the parameter and result types differ.");
				}
			}
		}
	}
}

/**
 * Base case for effect handler clauses. Used for the return value.
 */
class HandlerClause on Compiler {
	// Position.
	SrcPos pos;

	// Function that implements the handler.
	Function handler;

	// Create.
	init(SrcPos pos, Function handler) {
		init {
			pos = pos;
			handler = handler;
		}
	}
}

/**
 * Handler for a particular effect.
 */
class EffectHandlerClause extends HandlerClause {
	// Effect being handled.
	Effect effect;

	// Create.
	init(SrcPos pos, Function handler, Effect effect) {
		init(pos, handler) {
			effect = effect;
		}
	}
}


// Create a return clause for an effect.
HandlerClause returnClause(SrcPos pos, Class handler, SStr? param, SBlock bodyBlock) on Compiler {
	unless (handler as Handler)
		throw InternalError("Must provide a handler to 'returnClause'!");

	ValParam[] fnParams;
	fnParams << thisParam(handler);

	if (param) {
		if (handler.param.empty)
			throw SyntaxError(pos, "You can not specify a parameter for a handler that accepts 'void'.");
		fnParams << ValParam(handler.param, param);
	} else {
		if (handler.param.any)
			throw SyntaxError(pos, "You need to specify a parameter for a handler that accepts a value.");
	}

	BSTreeFn f(handler.result, SStr("handle-return", bodyBlock.pos), fnParams, null);
	f.parentLookup = handler;

	FnBody body(f, handler.scope);
	f.body = body;

	body.add(bodyBlock.transform(body));

	HandlerClause(pos, f);
}

// Create a clause for a named effect.
EffectHandlerClause effectClause(SrcPos pos, Class handler, SrcName effectName, NameParam[] params, SStr contParam, SBlock bodyBlock) on Compiler{
	unless (handler as Handler)
		throw InternalError("Must provide a handler to 'effectClause'!");

	Scope scope = handler.scope;

	var resolvedParams = params.resolve(scope);

	SrcName toFind = effectName.clone();
	toFind.last = SimplePart(toFind.last.name, resolvedParams.values());

	unless (found = scope.find(toFind) as Effect) {
		throw SyntaxError(effectName.pos, "Unable to find an effect named ${effectName}.");
	}

	// Add a HandlerFrame as the first parameter. That is where we store our result.
	resolvedParams.insert(0, ValParam(named{HandlerFrame}, "@result"));

	// We accept the continuation as Continuation, since the effect will not know the actual
	// type of the result. We could parameterize it on the result, but since we need a wrapper
	// anyway, we just do everything there.
	resolvedParams << ValParam(named{Continuation}, "@continuation");

	BSTreeFn f(Value(), SStr("handle-" + found.name, bodyBlock.pos), resolvedParams, null);
	f.parentLookup = handler;

	FnBody body(f, scope);
	f.body = body;

	var resultVar = body.parameters[0];
	var contVar = body.parameters.last;

	unless (typedFrame = named{}.find("HandlerFrame", handler, Scope()) as Type) {
		throw InternalError("Failed to find a handler frame.");
	}

	Function contWrapper = wrapContinuation(pos, scope, handler, found.result);
	contWrapper.parentLookup = handler;

	WeakDowncast cast(body, LocalVarAccess(pos, resultVar), typedFrame);
	If check(body, cast);

	if (var = cast.result) {
		LocalVarAccess varAccess(pos, var);

		ExprBlock inSuccess(bodyBlock.pos, check.successBlock);
		check.success(inSuccess);

		// Variable for 'this' (from 'object' in the block). Note that scoping is a little bit off,
		// but that is fine. It should also be marked as a "this"-variable.
		inSuccess.add(Var(body, handler, SStr("this"), namedExpr(body, pos, "object", varAccess)));

		// Variable for the continuation:
		inSuccess.add(Var(body, contParam, FnPtr(LocalVarAccess(pos, contVar), contWrapper, pos)));

		// The body, wrapped in a return point to allow return statements to work properly.
		ReturnPoint bodyWrap(bodyBlock.pos, inSuccess, handler.result);
		bodyWrap.body = bodyBlock.transform(bodyWrap);

		if (handler.result.any) {
			inSuccess.add(pattern(inSuccess) { ${varAccess}.result = ${bodyWrap}; });
		} else {
			inSuccess.add(bodyWrap);
		}
	}

	check.fail(pattern(body) {
						throw InternalError("Invalid types!");
					});

	body.add(check);

	EffectHandlerClause(pos, f, found);
}

// Helper to create a function that wraps a continuation.
private Function wrapContinuation(SrcPos pos, Scope scope, Handler handler, Value effectResult) on Compiler {
	ValParam[] params;
	params << ValParam(named{Continuation}, "c");
	if (effectResult.any)
		params << ValParam(effectResult, "param");
	BSTreeFn f(Value(handler.result), SStr("wrap-continuation", pos), params, null);

	FnBody body(f, scope);
	f.body = body;

	// Get the frame and copy the parameter:
	if (effectResult.any) {
		unless (typedEffect = named{}.find("EffectFrame", effectResult, Scope()) as Type)
			throw InternalError("Failed to find an effect frame.");

		var call = namedExpr(body, pos, "frame", namedExpr(body, pos, "c", Actuals()));

		WeakDowncast cast(body, call, typedEffect);
		cast.name(SStr("frame"));

		If check(body, cast);

		check.success(pattern(check.successBlock) {
						frame.result = param;
					});

		check.fail(pattern(body) { throw InternalError("Invalid parameter type!"); });

		body.add(check);
	}

	// Call resume and get the result:
	{
		unless (typedHandler = named{}.find("HandlerFrame", handler, Scope()) as Type)
			throw InternalError("Failed to find a handler frame.");

		var call = namedExpr(body, pos, "resume", namedExpr(body, pos, "c", Actuals()));

		WeakDowncast cast(body, call, typedHandler);
		cast.name(SStr("result"));

		If check(body, cast);

		if (handler.result.any) {
			check.success(pattern(check.successBlock) {
							if (r = result.result) {
								r;
							} else {
								throw InternalError("No result set!");
							}
						});
		}

		check.fail(pattern(body) { throw InternalError("Invalid parameter type!"); });

		body.add(check);
	}

	f;
}
