use core:lang;
use core:sync;
use lang:bs;
use lang:bs:unsafe;
use lang:bs:macro;

/**
 * List of handlers that are active for each thread.
 *
 * The structure is as follows:
 *
 * Each handle-expr launches the body of the function on a new UThread. The UThread can then be
 * copied in its entirety and stored as a continuation. This file keeps track of what effects
 * are handled by the handler "just above" the start of each UThread.
 *
 * Subclasses of this object also includes a "value" member that stores the result from the handler.
 */
package class HandlerFrame {
	// Effects that are handled at this location.
	Map<Effect, FnBase> effects;

	// UThread associated with us.
	Word threadId;

	// Previous set of handlers in the chain.
	HandlerFrame? prev;

	// Exception that ocurred, if any.
	Exception? error;

	// Ctor.
	init() {
		init {
			sema(0);
			waitAgain = false;
		}
	}

	// Semaphore to wait for execution to finish.
	private Sema sema;

	// Wait additional times when we have been woken more than once.
	private Bool waitAgain;

	// Wait for the result to be present.
	void wait() {
		do {
			sema.down();

			// If we were re-scheduled we might need to wait once more.
			if (waitAgain)
				waitAgain = false;
			else
				break;
		}

		// Report error:
		if (error)
			throw error;

		// Resume?
		if (r = resume) {
			resume = null;
			r.call(this);
		}
	}

	// Indicate that a result is available.
	void signal() {
		sema.up();
	}

	// Should we resume by calling a handler?
	private HandlerResume? resume;

	// Called to resume from an HandlerResume object.
	void resume(HandlerResume r) {
		resume = r;
		signal();
	}

	// Called when the thread has been re-scheduled and have to wait again.
	void resurrected() {
		waitAgain = true;
	}
}

/**
 * Class that keeps track of active handlers.
 *
 * The class is declared as belonging to the compiler thread. There are, however, accessors that
 * break thread safety, which is why we need a manual lock.
 */
package class ActiveHandlers on Compiler {
	// Lookup of frames for active UThreads.
	private Map<Word, HandlerFrame> frames;

	// Lock for 'frames'.
	private Lock framesLock;

	// Create.
	init() {}

	// Get the current frame for a thread.
	HandlerFrame? current() {
		Lock:Guard z(framesLock);
		frames.at(currentUThread());
	}

	// Set the current frame for a thread.
	void set(HandlerFrame f) {
		Lock:Guard z(framesLock);
		frames.put(currentUThread(), f);
	}

	// Set the current frame for a thread.
	void set(Word threadId, HandlerFrame f) {
		Lock:Guard z(framesLock);
		frames.put(threadId, f);
	}

	// Remove the current frame for a thread.
	void remove() {
		Lock:Guard z(framesLock);
		frames.remove(currentUThread());
	}
}

private ActiveHandlers activeHandlers on Compiler;

package HandlerFrame? currentHandlerFrame() {
	as thread Compiler {
		activeHandlers.current();
	}
}

package void setHandlerFrame(HandlerFrame to) {
	as thread Compiler {
		activeHandlers.set(to);
	}
}

package void setHandlerFrame(Word threadId, HandlerFrame to) {
	as thread Compiler {
		activeHandlers.set(threadId, to);
	}
}

package void removeHandlerFrame() {
	as thread Compiler {
		activeHandlers.remove();
	}
}

HandlerFrame : generate(params) {
	if (params.count != 1)
		return null;

	unless (handler = params[0].type as Handler)
		return null;

	var retVal = handler.result;

	Type t("HandlerFrame", [handler], TypeFlags:typeClass);
	t.flags += NamedFlags:namedMatchNoInheritance;

	t.setSuper(named{HandlerFrame});
	t.add(MemberVar("object", handler, t));
	if (retVal.any)
		t.add(MemberVar("result", wrapMaybe(retVal), t));

	// Create a constructor with a single parameter:
	{
		SStr memberName("object");

		BSTreeCtor ctor([thisParam(t), ValParam(handler, memberName)], SrcPos());
		CtorBody body(ctor, Scope());
		ctor.body = body;

		InitBlock init(SrcPos(), body, null);
		init.init(Initializer(memberName, LocalVarAccess(SrcPos(), body.parameters[1])));
		body.add(init);

		t.add(ctor);
	}

	t.add(TypeCopyCtor(t));
	t.add(TypeAssign(t));
	if (needsDestructor(t))
		t.add(TypeDefaultDtor(t));

	{
		var fnParam = fnType([retVal, handler]);
		ValParam[] paramList = [
			thisParam(t),
			ValParam(fnParam, "body")
			];

		BSTreeFn fn(Value(), SStr("main"), paramList, null);
		FnBody body(fn, Scope(named{}, BSLookup()));
		fn.body = body;
		t.add(fn);

		var assign = if (retVal.any) {
			pattern(body) { this.result = body.call(this.object); };
		} else {
			pattern(body) { body.call(this.object); };
		};

		body.add(pattern(body) {
					this.threadId = currentUThread();
					setHandlerFrame(this);

					try {
						${assign};
					} catch (EffectUnwind u) {
						if (u.forMe(this)) {
							return;
						} else {
							this.error = u;
						}
					} catch (Exception e) {
						this.error = e;
					}

					removeHandlerFrame();
					this.signal();
				});
	}

	t;
}
