use ui;
use core:io;
use core:geometry;
use graphics;

/**
 * An elements that simply fills its assigned rectangle with a brush.
 */
class SolidFill extends Element {
	// The brush we're using to fill the screen.
	Brush fill;

	// Brush used to border the solid box, if any.
	Brush? border;

	// Create.
	init(Brush fill) {
		init() { fill = fill; }
	}

	// Create, select a solid color.
	init(Color color) {
		init() { fill = SolidBrush(color); }
	}

	// Set the border.
	void border(Color c) {
		border = SolidBrush(c);
	}

	// Minimum size.
	Size minSize() : override { Size(); }

	// Draw.
	void draw(Graphics g) : override {
		if (border) {
			g.draw(pos, border);
		}
		g.fill(pos, fill);
	}
}


/**
 * An element that displays text. Useful for headings etc.
 *
 * Text is not re-flowed to fit the container, and the text will be centered horizontally in the
 * given space.
 */
class Heading extends Element {
	// Pre-formatted text.
	Text text;

	// The brush used to fill the text.
	Brush brush;

	// Create.
	init(Str text, Font f, Brush brush) {
		init() {
			text(text, f);
			brush = brush;
		}
	}

	// Create, specifying a solid color.
	init(Str text, Font f, Color color) {
		init() {
			text(text, f);
			brush = SolidBrush(color);
		}
	}

	// Create from the supplied text style.
	init(Str text, TextStyle style) {
		init() {
			text(text, style.font);
			brush = style.fill;
		}
	}

	// Minimum size.
	Size minSize() { text.size; }

	// Draw.
	void draw(Graphics g) {
		var p = pos;
		Point center((p.size.w - text.size.w) / 2, 0);
		g.draw(text, brush, p.p0 + center);
	}
}

Heading heading(Presentation p, Str text) on Render { Heading(text, p.headingStyle); }
Heading subHeading(Presentation p, Str text) on Render { Heading(text, p.contentStyle); }


/**
 * A paragraph of text. Fits the text to the given rectangle.
 *
 * Use 'par' for easy creation of paragraphs.
 */
class Paragraph extends Element {
	// Pre-formatted text.
	Text text;

	// Brush used to fill the text.
	Brush brush;

	// Set color.
	void fill(Color color) {
		brush = SolidBrush(color);
	}

	// Create.
	init(Str text, Font f, Brush brush) {
		init() {
			text(text, f);
			brush = brush;
		}
	}

	// Minimum size (TODO: Can we do better somehow?)
	Size minSize() { text.size; }

	// Set size.
	assign pos(Rect r) : override {
		super:pos(r);
		text.layoutBorder = r.size;
	}

	// Draw.
	void draw(Graphics g) {
		g.draw(text, brush, pos.p0);
	}
}

// Create a paragraph, specifying a solid color.
Paragraph par(Str t, Font f, Color c) on Render { Paragraph(t, f, SolidBrush(c)); }

// Create a paragraph with a pre-defined style.
Paragraph par(Str t, TextStyle style) on Render { Paragraph(t, style.font, style.fill); }

// Create a paragraph with the default style.
Paragraph par(Presentation p, Str t) on Render { par(t, p.contentStyle); }


/**
 * A list of strings, each with a bullet point to the left.
 */
class List extends Element {
	// Text elements.
	private Text[] elements;

	// Height of the font.
	private Float fontHeight;

	// Brush used to fill the text.
	Brush brush;

	// Size of the bullets.
	Float bulletSize;

	// Space from the left margin of this element to the start of the text.
	Float margin;

	// Space between the text and the bullet or the number.
	Float padding;

	// Space between lines.
	Float lineSpace;

	// Pre-formatted text snippnets used if the list is numbered.
	private Text[] numbers;

	// Create.
	init(Str[] text, Font f, Float lineSpace, Brush brush) {
		init() {
			brush = brush;
			fontHeight = f.pxHeight;
			margin = f.pxHeight / 2;
			padding = f.pxHeight / 3;
			bulletSize = f.pxHeight / 4;
			lineSpace = lineSpace;
		}

		for (x in text) {
			elements << Text(x, f);
		}
	}

	// Create numbers.
	void ordered() {
		numbers.clear();

		if (elements.empty)
			return;

		Font f = elements.first.font;
		for (Nat i = 0; i < elements.count; i++) {
			Text t("${i + 1}.", f);
			numbers << t;

			margin = max(margin, t.size.w + padding);
		}
	}

	// Create alphabetical letters.
	void letters() {
		numbers.clear();

		if (elements.empty)
			return;

		Font f = elements.first.font;
		for (Nat i = 0; i < elements.count; i++) {
			Text t(Char(65 + i).toS + ":", f);
			numbers << t;

			margin = max(margin, t.size.w + padding);
		}
	}

	// Minimum size
	Size minSize() {
		Size sz;
		for (e in elements) {
			Size s = e.size;
			sz.w = max(sz.w, s.w);
			sz.h += s.h + lineSpace;
		}
		sz.h -= lineSpace;
		sz.w += margin;
		sz;
	}

	// Set size.
	assign pos(Rect r) : override {
		super:pos(r);

		Size s = r.size;
		s.w -= margin;
		for (e in elements) {
			e.layoutBorder = s;
		}
	}

	// Draw.
	void draw(Graphics g) {
		Point at = pos.p0;

		for (i, e in elements) {
			if (i >= numbers.count) {
				Point delta(margin - padding - bulletSize, (fontHeight - bulletSize) * 2 / 3);
				Rect ball(at + delta, Size(bulletSize));
				g.fillOval(ball, brush);
			} else {
				Size text = numbers[i].size;
				g.draw(numbers[i], brush, at + Point(margin - padding - text.w, 0));
			}

			g.draw(e, brush, at + Point(margin, 0));

			at.y += e.size.h + lineSpace;
		}
	}
}

// Create a list, specifying a solid color.
List list(Str[] t, Font f, Color c) on Render { List(t, f, 0, SolidBrush(c)); }

// Create a list with a pre-defined style.
List list(Str[] t, TextStyle style) on Render { List(t, style.font, style.space, style.fill); }

// Create a list with the default style.
List list(Presentation p, Str[] t) on Render { list(t, p.contentStyle); }


/**
 * An image.
 */
class Image extends Element {
	// The bitmap we're painting.
	private Bitmap image;

	// Fill mode: 0 = scale according to scale factor, 1 = fit to container, 2 = fill container (no borders)
	private Nat mode;

	// Scale. Only relevant in mode 0.
	private Float fixScale;

	// Create.
	init(Bitmap image) {
		init() {
			image = image;
			mode = 1;
			fixScale = 1.0;
		}
	}

	// Set an explicit scale.
	void scale(Float scale) {
		fixScale = scale;
		mode = 0;
	}

	// Fit to the container without clipping (default).
	void fit() {
		mode = 1;
	}

	// Fill the container (maybe clips).
	void fill() {
		mode = 2;
	}

	// Minimum size.
	Size minSize() {
		image.size * fixScale;
	}

	// Draw.
	void draw(Graphics g) {
		Size imageSz = image.size;
		Size size = pos.size;
		Point center = pos.center;

		Float scale = fixScale;
		if (mode == 1) {
			scale = min(size.w / imageSz.w, size.h / imageSz.h);
		} else if (mode == 2) {
			scale = max(size.w / imageSz.w, size.h / imageSz.h);
		}

		g.push();
		g.transform = core:geometry:scale(scale) * translate(center);
		g.draw(image, -Point(imageSz/2));
		g.pop();
	}
}

// Create an image.
Image image(Bitmap image) on Render { Image(image); }

// Create from an Image instance.
Image image(graphics:Image image) on Render { Image(Bitmap(image)); }

// Create from an Url. TODO: Perhaps we want to de-duplicate these!
Image image(Url url) on Render { Image(Bitmap(url.loadImage)); }


/**
 * Page numbering.
 */
class PageNumber extends Element {
	// Font.
	private Font font;

	// Fill.
	private Brush fill;

	// Margin, to make this element easy to position inside an anchor element.
	Size border;
	void border(Float w, Float h) {
		border = Size(w, h);
	}

	// Offset.
	Nat offset;

	// Maximum size.
	private Size maxSize;

	// Cache of text objects to draw.
	private Nat->Text cache;

	// Create.
	init(TextStyle style) {
		init() {
			font = style.font;
			fill = style.fill;
			offset = 1;
			border = Size(20, 15);
		}

		maxSize = Text("99", font).size;
	}

	Size minSize() : override {
		maxSize + border*2;
	}

	// We need access to 'step'.
	void draw(Graphics g, Nat step, Duration time) : override {
		if (before(g, step, time)) {
			Text t = find(step);
			Point at = pos.p1 - t.size - border;
			g.draw(t, fill, at);
		}
		after(g, step, time);
	}

	void draw(Graphics g) : override {}

	private Text find(Nat id) {
		id += offset;

		if (!cache.has(id)) {
			cache.put(id, Text(id.toS, font));
		}

		cache.get(id);
	}
}

// Create page numbering conveniently.
PageNumber pageNumber(Presentation p) {
	PageNumber(p.contentStyle);
}


/**
 * Big arrow.
 */
class BigArrow extends Element {
	// Fill color.
	Brush brush;
	void fill(Color color) { brush = SolidBrush(color); }

	// Angle.
	Angle angle;

	// Size.
	Float size;

	// Arrow path.
	private Path arrow;

	// Initialize, specify direction.
	init(Angle angle) {
		init() {
			angle = angle;
			size = 40;
			brush = SolidBrush(black);
		}

		arrow.start(Point(0, -0.8));
		arrow.line(Point(1, 0.1));
		arrow.line(Point(0.4, 0.0));
		arrow.line(Point(0.4, 0.7));
		arrow.line(Point(0, 0.5));
		arrow.line(Point(-0.4, 0.7));
		arrow.line(Point(-0.4, 0.0));
		arrow.line(Point(-1, 0.1));
		arrow.close();
	}

	// Minimum size.
	Size minSize() : override {
		Size(size);
	}

	// Draw.
	void draw(Graphics g) : override {
		g.push();
		g.transform = scale(size/2) * rotate(angle) * translate(pos.center);
		g.fill(arrow, brush);
		g.pop();
	}
}

/**
 * Line, good for using together with Grid to make tables.
 */
class HLine extends Element {
	init(Brush draw) {
		init() { draw = draw; width = 1.0; }
	}

	init(Color draw) {
		init() { draw = SolidBrush(draw); width = 1.0; }
	}

	// Color.
	Brush draw;

	// Line width.
	Float width;

	// Minimum length of the line.
	Float minLength;

	// Minimum size.
	Size minSize() : override {
		Size(minLength, width);
	}

	// Draw.
	void draw(Graphics g) : override {
		Rect p = pos;
		g.push();
		g.lineWidth = width;
		g.line(p.p0, p.p0 + Size(p.size.w, 0), draw);
		g.pop();
	}
}

/**
 * Line, good for using together with Grid to make tables.
 */
class VLine extends Element {
	init(Brush draw) {
		init() { draw = draw; width = 1.0; }
	}

	init(Color draw) {
		init() { draw = SolidBrush(draw); width = 1.0; }
	}

	// Color.
	Brush draw;

	// Line width.
	Float width;

	// Minimum length of the line.
	Float minLength;

	// Minimum size.
	Size minSize() : override {
		Size(width, minLength);
	}

	// Draw.
	void draw(Graphics g) : override {
		Rect p = pos;
		g.push();
		g.lineWidth = width;
		g.line(p.p0, p.p0 + Size(0, p.size.h), draw);
		g.pop();
	}
}
