#! /usr/bin/env perl

#
#   Copyright (C) Heinz-Josef Claes (2002,2004)
#                 hjclaes@web.de
#   
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#   
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#   
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#


push @VERSION, '$Id: storeBackup_du.pl 328 2004-04-04 07:56:52Z hjc $ ';

use strict;



sub libPath
{
    my $file = shift;

    my $dir;

    # Falls Datei selbst ein symlink ist, solange folgen, bis aufgelst
    if (-f $file)
    {
	while (-l $file)
	{
	    my $link = readlink($file);

	    if (substr($link, 0, 1) ne "/")
	    {
		$file =~ s/[^\/]+$/$link/;
	    }
	    else
	    {
		$file = $link;
	    }
	}

	($dir, $file) = &splitFileDir($file);
	$file = "/$file";
    }
    else
    {
	print STDERR "<$file> does not exist!\n";
	exit 1;
    }

    $dir .= "/../lib";           # Pfad zu den Bibliotheken
    my $oldDir = `/bin/pwd`;
    chomp $oldDir;
    if (chdir $dir)
    {
	my $absDir = `/bin/pwd`;
	chop $absDir;
	chdir $oldDir;

	return (&splitFileDir("$absDir$file"));
    }
    else
    {
	print STDERR "<$dir> does not exist, exiting\n";
    }
}
sub splitFileDir
{
    my $name = shift;

    return ('.', $name) unless ($name =~/\//);    # nur einfacher Dateiname

    my ($dir, $file) = $name =~ /^(.*)\/(.*)$/s;
    $dir = '/' if ($dir eq '');                   # gilt, falls z.B. /filename
    return ($dir, $file);
}
my ($req, $prog) = &libPath($0);
push @INC, "$req";

require 'checkParam.pl';
require 'version.pl';
require 'humanRead.pl';



my $Help = <<EOH;
This program evaluates the disk usage in one or more backup directories.

usage:
	$0 [-v] [-l] backupdirs ...

--verbose  -v  Print accumulated values for multiple versions (days)
	       of backuped files. Shows the steps when calculating the
	       space used by the specified backups
--links	   -l  Also print statistic about how many links the files have
	       and how much space this saves.

Copyright (c) 2002,2004 by Heinz-Josef Claes
Published under the GNU General Public License
EOH
    ;

die "$Help" unless @ARGV;

&printVersions(\@ARGV, '-V');


my $CheckPar =
    CheckParam->new('-allowLists' => 'yes',
		    '-list' => [Option->new('-option' => '-v',
					    '-alias' => '--verbose'),
				Option->new('-option' => '-l',
					    '-alias' => '--links')
				]
		    );

$CheckPar->check('-argv' => \@ARGV,
                 '-help' => $Help
                 );

my $verbose = $CheckPar->getOptWithoutPar('-v');
my $links = $CheckPar->getOptWithoutPar('-l');

die "directories missing in argument list"
    if ($CheckPar->getListPar() == 0);

# Prfen, ob Directories berhaupt existieren
my $d;
foreach $d ($CheckPar->getListPar())
{
    die "<$d> does not exist" unless (-e $d);
    die "<$d> is not a directories" unless (-d $d);
}



%main::noLinks;    # key = inode,
                   #              'size'    => Dateigre (einzelne Datei)
                   #              'restNoLinks' => bisher gezhlte Anzahl Links

$main::sumLocal = 0;
@main::noFilesWithLinks = ();
@main::sizeFilesWithLinks = ();


my $s = '';
foreach $d ($CheckPar->getListPar())
{
    print "________checking $d ...\n" if ($verbose);
    chdir "$d";
    &traverseTrees();
    chdir "..";
    if ($verbose)
    {
	print " sumLocal = ", &humanReadable($main::sumLocal), "$s\n";
	my $sumShared;
	foreach $d (keys %main::noLinks)
	{
	    $sumShared += $main::noLinks{$d}{'size'};
	}
	print "sumShared = ", &humanReadable($sumShared), "$s\n";
	$s = " (accumulated)";
    }
}

# Zusammenrechnen der Ergebnisse
unless ($verbose)
{
    print " sumLocal = ", &humanReadable($main::sumLocal), "\n";
    my $sumShared;
    foreach $d (keys %main::noLinks)
    {
	$sumShared += $main::noLinks{$d}{'size'};
    }
    print "sumShared = ", &humanReadable($sumShared), "\n";
}

if ($links)
{
    print "\nlinks |  files | size | l * s | less\n";
    print "------+--------+------+------ +-----\n";
    for ($d = 1 ; $d < @main::noFilesWithLinks ; $d++)
    {
	next unless $main::noFilesWithLinks[$d];
	my $ls = $d * $main::sizeFilesWithLinks[$d];
	printf(" %4d | %6d | %s |  %s | %s\n", $d, $main::noFilesWithLinks[$d],
	       &humanReadable($main::sizeFilesWithLinks[$d]),
	       &humanReadable($ls),
	       &humanReadable($ls - $main::sizeFilesWithLinks[$d]));
    }
}

exit 0;


######################################################################
sub traverseTrees
{
    local *DIR;
    opendir(DIR, '.');

    my $f;
    while ($f = readdir DIR)
    {
	if (-l $f)
	{
	    my ($inode, $nlink, $size) = (lstat($f))[1,3,7];
	    if (defined($main::noLinks{$inode}))
	    {
		if (--$main::noLinks{$inode}{'restNoLinks'} == 0)
		{
		    $main::sumLocal += $main::noLinks{$inode}{'size'};
		    delete $main::noLinks{$inode};
		    next;
		}
	    }
	    else
	    {
		if ($nlink == 1)
		{
		    $main::sumLocal += $size;
		    next;
		}
		$main::noLinks{$inode}{'size'} = $size;
		$main::noLinks{$inode}{'restNoLinks'} = $nlink - 1;
	    }
	}
	elsif (-f $f)
	{
	    my ($inode, $nlink, $size) = (stat($f))[1,3,7];
	    if (defined($main::noLinks{$inode}))
	    {
		if (--$main::noLinks{$inode}{'restNoLinks'} == 0)
		{
		    $main::sumLocal += $main::noLinks{$inode}{'size'};
		    delete $main::noLinks{$inode};
		    next;
		}
	    }
	    else
	    {
		++$main::noFilesWithLinks[$nlink];
		$main::sizeFilesWithLinks[$nlink] += $size;
		if ($nlink == 1)
		{
		    $main::sumLocal += $size;
		    next;
		}
		$main::noLinks{$inode}{'size'} = $size;
		$main::noLinks{$inode}{'restNoLinks'} = $nlink - 1;
	    }
	}
	elsif (-d $f)
	{
	    next if ($f eq '.' or $f eq '..');
	    chdir "$f";
	    &traverseTrees();
	    chdir "..";
	}
    }
    close(DIR);
}
