# -*- coding: utf-8 -*-
# Copyright (c) 2007 Jimmy Rönnholm
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

import os
import time
import locale
import gettext

import pygtk
pygtk.require('2.0')
import gtk
import gtk.glade
import gnome.ui
import gobject

from bootconfig import StartupThread, ShutdownThread, FloppyThread

APP_NAME = 'startupmanager'
APP_VERSION = '1.9.10'
PREFIX_DIR = '/usr/share'
LOCALE_DIR = '/usr/share/locale'
GLADE_FILE = '/usr/share/startupmanager/startupmanager.glade'

try:
    locale.setlocale(locale.LC_ALL, '')
except:
    print 'Could not set locale, falling back to standard locale'
gettext.bindtextdomain(APP_NAME, LOCALE_DIR)
gettext.textdomain(APP_NAME)
_ = gettext.gettext

def convert_resolution(number):
    resolution = number[0]
    if resolution == '0':
        res = '640x480'
    elif resolution == '1':
        res = '800x600'
    elif resolution == '2':
        res = '1024x768'
    elif resolution == '3':
        res = '1280x1024'
    else:
        res = '1600x1200'
    return res

def convert_color(color, wanted_type, revert):
    """Convert color name to combo box index."""
    if wanted_type == 0:
        color_dict = {
        'black' : 0 ,
        'blue' : 1 ,
        'green' : 2 ,
        'cyan' : 3 ,
        'red' : 4 ,
        'magenta' : 5 ,
        'brown' : 6 ,
        'light-gray' : 7
        }
    if wanted_type == 1:
        color_dict = {
        'black' : 0 ,
        'blue' : 1 ,
        'green' : 2 ,
        'cyan' : 3 ,
        'red' : 4 ,
        'magenta' : 5 ,
        'brown' : 6 ,
        'light-gray' : 7 ,
        'dark-gray' : 8 ,
        'light-blue' : 9 ,
        'light-green' : 10 ,
        'light-cyan' : 11 ,
        'light-red' : 12 ,
        'light-magenta' : 13 ,
        'yellow' : 14 ,
        'white' : 15
        }
    if revert:
        reverse_color_dict = dict((v,k) for (k,v) in color_dict.iteritems())
        if reverse_color_dict.has_key(color):
            return reverse_color_dict[color]
        else:
            if wanted_type == 0:
                return 'blue'
            if wanted_type == 1:
                return 'cyan'
    else:
        if color_dict.has_key(color):
            return color_dict[color]
        else:
            if wanted_type == 0:
                return 1
            if wanted_type == 1:
                return 3

def convert_vga(vga, wanted, revert=False):
    """Convert vga number to combo box index."""
    dic = {
    769 : '00' ,
    771 : '10' ,
    773 : '20' ,
    775 : '30' ,
    796 : '40' ,
    758 : '01' ,
    788 : '11' ,
    791 : '21' ,
    794 : '31' ,
    798 : '41' ,
    786 : '02' ,
    789 : '12' ,
    792 : '22' ,
    795 : '32' ,
    799 : '42'
    }
    if revert:
        reverse_dic = dict((v,k) for (k,v) in dic.iteritems())
        return reverse_dic[vga]
    if dic.has_key(vga):
        settings = dic[vga]
    else:
        settings = dic[769]
    return int(settings[wanted])

def convert_gfx(gfxcode):
    dic = {
    '1600x1200' : 4,
    '1280x1024' : 3,
    '1024x768' : 2,
    '800x600' : 1,
    '640x480' : 0
    }
    if dic.has_key(gfxcode):
        return int(dic[gfxcode])
    else:
        return 0


class SumGui:

    def __init__(self):
        self.glade_xml = gtk.glade.XML(GLADE_FILE, None ,APP_NAME)
        self.get_widgets()
        self.splashy_set = False
        self.usplash_set = False
        #Display a Progress dialog during config initialization
        dialog = gtk.Dialog(title=_('StartUp-Manager'), parent=self.main_window)
        progress = gtk.ProgressBar()
        progress.set_text(_('Performing pre-configuration tasks'))
        dialog.vbox.add(progress)
        progress.show()
        dialog.show()
        thread = StartupThread()
        thread.start()
        while(1):
            while(gtk.events_pending()):
                gtk.main_iteration()
            if not thread.isAlive():
                break
            else:
                progress.pulse()
                time.sleep(0.1)
        dialog.destroy()

        #Set the config classes
        self.grub = thread.grub
        self.grub_legacy_mode = thread.grub_legacy_mode
        self.splashy = thread.splashy
        self.usplash = thread.usplash

        #Setup the ui
        self.setup_widgets()
        props = {gnome.PARAM_APP_DATADIR : PREFIX_DIR}
        self.gnome_program = gnome.program_init(APP_NAME, APP_VERSION, properties=props)
        gobject.threads_init()
        self.main_window.show()
        gtk.main()

    def setup_widgets(self):
        if self.grub:
            self.set_shared_grub_widgets()
            if self.grub_legacy_mode:
                self.set_legacy_grub_widgets()
                self.refresh_legacy_grub_themes()
            else:
                self.set_grub_widgets()
        if self.splashy:
            self.set_splashy_widgets()
        if self.usplash:
            self.set_usplash_widgets()
        self.glade_xml.signal_autoconnect(self)

    def set_shared_grub_widgets(self):
        self.boot_options_vbox.show()
        self.timeout_frame.show()
        self.defaultboot_frame.show()
        self.misc_frame.show()
        self.advanced_vbox.show()
        self.advanced_misc_frame.show()
        self.rescuefloppy_button.show()

        self.timeout_spinner.set_value(self.grub.get_timeout())
        self.default_boot_combo.get_model().clear()
        for i, title in enumerate(self.grub.get_titles()):
            self.default_boot_combo.append_text(title)
            if title == self.grub.get_default_boot():
                self.default_boot_combo.set_active(i)

    def set_grub_widgets(self):
        self.display_frame.show()
        self.color_depth_label.show()
        self.color_depth_combo.show()
        self.boot_splash_check.show()
        self.boot_text_check.show()
        self.grub_resolution_frame.show()
        
        self.boot_text_check.set_active(self.grub.get_boot_text_visible())
        self.boot_splash_check.set_active(self.grub.get_splash_active())
        self.resolution_combo.set_active(convert_vga(self.grub.get_vga_code(), 0))
        self.color_depth_combo.set_active(convert_vga(self.grub.get_vga_code(), 1))
        self.grub_resolution_combo.set_active(convert_gfx(self.grub.get_gfxmode()))

    def set_legacy_grub_widgets(self):
        self.timeout_check.show()
        self.grub_menu_check.show()
        self.appearance_vbox.show()
        self.grub_color_frame.show()
        self.grub_theme_frame.show()
        self.security_vbox.show()
        self.restore_settings_button.show()
        if self.grub.has_update_grub():
            self.display_frame.show()
            self.color_depth_label.show()
            self.color_depth_combo.show()
            self.boot_splash_check.show()
            self.boot_text_check.show()
            self.protect_old_check.show()
            self.protect_rescuemode_check.show()
            self.limit_kernel_frame.show()
            self.memtest_check.show()
            self.recovery_check.show()
            self.updatedefault_check.show()

        self.grub_store = gtk.ListStore(str)
        self.grub_treeview.append_column(gtk.TreeViewColumn(_('Image'), gtk.CellRendererText(), text=0))
        self.grub_treeview.set_model(self.grub_store)

        if self.grub.get_user_edited_file():
            dialog = gtk.MessageDialog(parent=self.main_window, 
                flags=gtk.DIALOG_MODAL, type=gtk.MESSAGE_QUESTION,
                buttons=gtk.BUTTONS_YES_NO,
                message_format=_('Your grub configuration file lack the line '
                    '"### END DEBIAN AUTOMAGIC KERNELS LIST".\nYou have '
                    'probably hand-edited the configuration file.\n\n'
                    'Choose "yes" to have the file auto-generated. '
                    'This means that any custom boot options will be lost.\n'
                    'If you only have one operating system installed and you '
                    'do not know what I am talking about, it should be safe '
                    'to answer "yes".\n\nOtherwise, choose "no". '
                    'That will stop this program.\nIt is however recommended '
                    'to look into this, since system upgrades could '
                    'also rewrite the configuration file.'))
            dialog.set_default_response(gtk.RESPONSE_NO)
            response = dialog.run()
            dialog.destroy()
            if response == gtk.RESPONSE_NO:
                self.grub.restore_config()
                raise SystemExit('User requested shutdown.')
            elif response == gtk.RESPONSE_YES:
                pass
            else:
                self.grub.restore_config()
                raise SystemExit('User requested shutdown.')

        self.default_boot_combo.append_text(_('Last used'))
        default_boot = self.grub.get_default_boot()
        if default_boot == -1:
            number = self.default_boot_combo.get_model().iter_n_children(None)
            self.default_boot_combo.set_active(number-1)
        self.timeout_check.set_active(self.grub.get_timeout_used())
        self.grub_menu_check.set_active(self.grub.get_menu_visible())
        self.grub_color_check.set_active(self.grub.get_use_colors())
        self.normal_background_combo.set_active(convert_color(self.grub.get_color_normal_bg(), 0, False))
        self.highlighted_background_combo.set_active(convert_color(self.grub.get_color_highlighted_bg(), 0, False))
        self.normal_foreground_combo.set_active(convert_color(self.grub.get_color_normal_fg(), 1, False))
        self.highlighted_foreground_combo.set_active(convert_color(self.grub.get_color_highlighted_fg(), 1, False))
        self.normal_blink_check.set_active(self.grub.get_color_blink_normal())
        self.highlighted_blink_check.set_active(self.grub.get_color_blink_highlight())
        if self.grub.has_update_grub():
            self.boot_text_check.set_active(self.grub.get_boot_text_visible())
            self.boot_splash_check.set_active(self.grub.get_splash_active())
            self.resolution_combo.set_active(convert_vga(self.grub.get_vga_code(), 0))
            self.color_depth_combo.set_active(convert_vga(self.grub.get_vga_code(), 1))
            self.protect_rescuemode_check.set_active(self.grub.get_protect_rescuemode())
            self.protect_old_check.set_active(self.grub.get_protect_oldmode())
        self.grub_splash_check.set_active(self.grub.get_splash_visible())

        self.refresh_legacy_grub_themes()

        self.password_protect_check.set_active(self.grub.get_password_protection())
        

        self.kernel_check.set_active(self.grub.get_limit_kernel())
        self.kernel_spinner.set_value(1)
        if type(self.grub.get_kernel_limit()) == type(1):
            self.kernel_spinner.set_value(self.grub.get_kernel_limit())

        self.memtest_check.set_active(self.grub.get_memtest())
        self.recovery_check.set_active(self.grub.get_create_alternative())
        self.updatedefault_check.set_active(self.grub.get_update_default())

    def refresh_legacy_grub_themes(self):
        self.grub_splash_combo.get_model().clear()
        self.grub_store.clear()
        grub_images = self.grub.get_images()
        for image in grub_images:
            self.grub_splash_combo.append_text(image)
            self.grub_store.append([image])

        default_image = self.grub.get_splash()
        if default_image != '':
            place = 0
            for grub_file in grub_images:
                if grub_file == default_image:
                    self.grub_splash_combo.set_active(place)
                place += 1

    def set_splashy_widgets(self):
        if not self.splashy_set:
            self.splashy_set = True
            self.appearance_vbox.show()
            self.splashy_frame.show()
            self.advanced_vbox.show()
            self.advanced_splashy_frame.show()
            self.splashy_store = gtk.ListStore(str)
            self.splashy_treeview.append_column(gtk.TreeViewColumn(_('Theme'), gtk.CellRendererText(), text=0))
            self.splashy_treeview.set_model(self.splashy_store)
        self.splashy_initramfs_check.set_active(self.splashy.get_use_initramfs())
        self.splashy_combo.get_model().clear()
        self.splashy_store.clear()
        splashy_themes = self.splashy.get_all_themes()
        for theme in splashy_themes:
            self.splashy_combo.append_text(theme)
            self.splashy_store.append([theme])
        default_splashy = self.splashy.get_active_theme()
        if default_splashy != '':
            place = 0
            for splashy_theme in splashy_themes:
                if splashy_theme == default_splashy:
                    self.splashy_combo.set_active(place)
                place += 1

    def set_usplash_widgets(self):
        if not self.usplash_set:
            self.usplash_set = True
            self.boot_options_vbox.show()
            self.display_frame.show()
            self.appearance_vbox.show()
            self.usplash_frame.show()
            self.advanced_vbox.show()
            self.usplash_store = gtk.ListStore(str)
            self.usplash_treeview.append_column(gtk.TreeViewColumn(_('Theme'), gtk.CellRendererText(), text=0))
            self.usplash_treeview.set_model(self.usplash_store)
        self.usplash_combo.get_model().clear()
        self.usplash_store.clear()
        usplash_themes = self.usplash.get_all_themes()
        for theme in usplash_themes:
            self.usplash_combo.append_text(theme)
            self.usplash_store.append([theme])
        default_usplash = self.usplash.get_active_theme()
        if default_usplash != '':
            place = 0
            for usplash_theme in usplash_themes:
                if usplash_theme == default_usplash:
                    self.usplash_combo.set_active(place)
                place += 1

    def on_default_boot_combo_changed(self, widget):
        text = self.default_boot_combo.get_active_text()
        number = self.default_boot_combo.get_model().iter_n_children(None)
        if self.grub_legacy_mode and self.default_boot_combo.get_active() == number - 1:
            self.grub.set_default_boot(-1)
        else:
            self.grub.set_default_boot(text)

    def on_timeout_check_toggled(self, widget):
        active = self.timeout_check.get_active()
        self.grub.set_timeout_used(active)

    def on_timeout_spinner_value_changed(self, widget):
        timeout = self.timeout_spinner.get_value_as_int()
        self.grub.set_timeout(timeout)

    def on_resolution_combo_changed(self, widget):
        number = str(self.resolution_combo.get_active()) + \
                 str(self.color_depth_combo.get_active())
        self.grub.set_vga_code(convert_vga(number, 1, True))
        if self.usplash:
            self.usplash.set_resolution(convert_resolution(number))

    def on_color_depth_combo_changed(self, widget):
        number = str(self.resolution_combo.get_active()) + \
                 str(self.color_depth_combo.get_active())
        self.grub.set_vga_code(convert_vga(number, 1, True))

    def on_boot_text_check_toggled(self, widget):
        active = self.boot_text_check.get_active()
        self.grub.set_boot_text_visible(active)

    def on_grub_menu_check_toggled(self, widget):
        active = self.grub_menu_check.get_active()
        self.grub.set_menu_visible(active)

    def on_grub_color_check_toggled(self, widget):
        active = self.grub_color_check.get_active()
        self.grub.set_use_colors(active)

    def on_normal_background_combo_changed(self, widget):
        color = convert_color(self.normal_background_combo.get_active(), 
                              0, True)
        self.grub.set_color_normal_bg(color)

    def on_highlighted_background_combo_changed(self, widget):
        color = convert_color(self.highlighted_background_combo.get_active(), 
                              0, True)
        self.grub.set_color_highlighted_bg(color)

    def on_normal_foreground_combo_changed(self, widget):
        color = convert_color(self.normal_foreground_combo.get_active(), 
                              1, True)
        self.grub.set_color_normal_fg(color)

    def on_highlighted_foreground_combo_changed(self, widget):
        color = convert_color(self.highlighted_foreground_combo.get_active(), 
                              1, True)
        self.grub.set_color_highlighted_fg(color)

    def on_normal_blink_check_toggled(self, widget):
        active = self.normal_blink_check.get_active()
        self.grub.set_color_blink_normal(active)

    def on_highlighted_blink_check_toggled(self, widget):
        active = self.highlighted_blink_check.get_active()
        self.grub.set_color_blink_highlight(active)

    def on_grub_splash_check_toggled(self, widget):
        if self.grub_splash_combo.get_active() != -1:
            image = self.grub_splash_combo.get_active_text()
            active = self.grub_splash_check.get_active()
            self.grub.set_splash(active, image)
        else:
            self.grub_splash_check.set_active(False)

    def on_grub_resolution_combo_changed(self, widget):
        res = str(self.grub_resolution_combo.get_active_text())
        self.grub.set_gfxmode(res)

    def on_grub_splash_combo_changed(self, widget):
        active = self.grub_splash_check.get_active()
        image = self.grub_splash_combo.get_active_text()
        if image:
            self.grub.set_splash(active, image)

    def on_grub_themes_button_clicked(self, widget):
        self.grub_dialog.show()

    def on_add_grub_button_clicked(self, widget):
        chooser = gtk.FileChooserDialog(title=None, parent=self.grub_dialog, 
                                        action=gtk.FILE_CHOOSER_ACTION_OPEN,
                                        buttons=(gtk.STOCK_CANCEL,
                                                 gtk.RESPONSE_CANCEL,
                                                 gtk.STOCK_OPEN,
                                                 gtk.RESPONSE_OK), 
                                        backend=None)
        chooser.set_default_response(gtk.RESPONSE_OK)
        chooser.set_current_folder(os.environ.get('HOME'))
        if os.environ.get('HOME') != '/root':
            chooser.add_shortcut_folder(os.environ.get('HOME'))

        filter = gtk.FileFilter()
        filter.set_name(_('Common formats for Grub background images'))
        filter.add_mime_type('image/png')
        filter.add_mime_type('image/jpeg')
        filter.add_mime_type('image/gif')
        filter.add_mime_type('bitmap/bmp')
        for format in self.grub.get_image_formats():
            filter.add_pattern('*.' + format)
        chooser.add_filter(filter)
        filter = gtk.FileFilter()
        filter.set_name(_('All files'))
        filter.add_pattern('*')
        chooser.add_filter(filter)
        response = chooser.run()
        if response == gtk.RESPONSE_OK:
            filename = chooser.get_filename()
            if not self.grub.add_image(filename):
                dialog = gtk.MessageDialog(parent=self.main_window, 
                                           flags=gtk.DIALOG_MODAL,
                                           type=gtk.MESSAGE_ERROR, 
                                           buttons=gtk.BUTTONS_OK,
                                           message_format=_('Could not '
                                                            'add image'))
                dialog.run()
                dialog.destroy()
            self.refresh_legacy_grub_themes()
        chooser.destroy()

    def on_remove_grub_button_clicked(self, widget):
        active = self.grub_splash_combo.get_active_text()
        selected = self.grub_treeview.get_selection().get_selected()
        model = selected[0]
        iter = selected[1]
        name = model.get_value(iter, 0)
        if active == name:
            self.grub_splash_check.set_active(False)
        self.grub.remove_image(name)
        self.refresh_legacy_grub_themes()

    def on_grub_file_chooser_update_preview(self, widget):
        filename = self.grub_file_chooser.get_filename()
        self.grub.add_image(filename)

    def on_boot_splash_check_toggled(self, widget):
        active = self.boot_splash_check.get_active()
        self.grub.set_splash_active(active)

    def on_usplash_combo_changed(self, widget):
        active = self.usplash_combo.get_active_text()
        if active:
            self.usplash.set_active_theme(active)

    def on_usplash_themes_button_clicked(self, widget):
        self.usplash_dialog.show()

    def on_add_usplash_button_clicked(self, widget):
        chooser = gtk.FileChooserDialog(title=None, 
                                        parent=self.usplash_dialog, 
                                        action=gtk.FILE_CHOOSER_ACTION_OPEN,
                                        buttons=(gtk.STOCK_CANCEL,
                                                 gtk.RESPONSE_CANCEL,
                                                 gtk.STOCK_OPEN,
                                                 gtk.RESPONSE_OK), 
                                        backend=None)
        chooser.set_default_response(gtk.RESPONSE_OK)
        chooser.set_current_folder(os.environ.get('HOME'))
        if os.environ.get('HOME') != '/root':
            chooser.add_shortcut_folder(os.environ.get('HOME'))

        filter = gtk.FileFilter()
        filter.set_name(_('Usplash theme files(*.so)'))
        filter.add_pattern('*.so')
        chooser.add_filter(filter)
        filter = gtk.FileFilter()
        filter.set_name(_('All files'))
        filter.add_pattern('*')
        chooser.add_filter(filter)
        response = chooser.run()
        if response == gtk.RESPONSE_OK:
            filename = chooser.get_filename()
            self.usplash.add_theme(filename)
            self.set_usplash_widgets()
        chooser.destroy()

    def on_remove_usplash_button_clicked(self, widget):
        active = self.usplash_combo.get_active_text()
        selected = self.usplash_treeview.get_selection().get_selected()
        model = selected[0]
        iter = selected[1]
        name = model.get_value(iter, 0)
        if active == name:
            dialog = gtk.MessageDialog(parent=self.main_window, 
                                       flags=gtk.DIALOG_MODAL, 
                                       type=gtk.MESSAGE_QUESTION,
                                       buttons=gtk.BUTTONS_YES_NO,
                                       message_format=_('You are removing '
                                                        'the currently '
                                                        'active usplash '
                                                        'theme. Do you '
                                                        'really want to '
                                                        'do that?'))
            dialog.set_default_response(gtk.RESPONSE_NO)
            response = dialog.run()
            dialog.destroy()
            if response == gtk.RESPONSE_NO:
                return
            elif response == gtk.RESPONSE_YES:
                dialog2 = gtk.MessageDialog(parent=self.main_window, 
                                            flags=gtk.DIALOG_MODAL, 
                                            type=gtk.MESSAGE_INFO, 
                                            buttons=gtk.BUTTONS_OK,
                                            message_format=_('Make sure '
                                                             'to select a '
                                                             'new theme '
                                                             'if you want '
                                                             'usplash to '
                                                             'function '
                                                             'properly.'))
                dialog2.run()
                dialog2.destroy()
            else:
                return
        self.usplash.remove_theme(name)
        self.set_usplash_widgets()

    def on_splashy_combo_changed(self, widget):
        active = self.splashy_combo.get_active_text()
        if active:
            self.splashy.set_active_theme(active)

    def on_splashy_initramfs_check_toggled(self, widget):
        active = self.splashy_initramfs_check.get_active()
        self.splashy.set_use_initramfs(active)

    def on_splashy_themes_button_clicked(self, widget):
        self.splashy_dialog.show()

    def on_splashy_preview_button_clicked(self, widget):
        dialog = gtk.MessageDialog(parent=self.main_window, 
                                   flags=gtk.DIALOG_MODAL,
                                   type=gtk.MESSAGE_INFO, 
                                   buttons=gtk.BUTTONS_OK,
                                   message_format=_('When you wish to leave '
                                                    'the preview, press Esc.'))
        dialog.run()
        dialog.destroy()
        self.splashy.show_preview()

    def on_add_splashy_button_clicked(self, widget):
        chooser = gtk.FileChooserDialog(title=None, 
                                        parent=self.splashy_dialog, 
                                        action=gtk.FILE_CHOOSER_ACTION_OPEN,
                                        buttons=(gtk.STOCK_CANCEL,
                                                 gtk.RESPONSE_CANCEL,
                                                 gtk.STOCK_OPEN,
                                                 gtk.RESPONSE_OK), 
                                        backend=None)
        chooser.set_default_response(gtk.RESPONSE_OK)
        chooser.set_current_folder(os.environ.get('HOME'))
        if os.environ.get('HOME') != '/root':
            chooser.add_shortcut_folder(os.environ.get('HOME'))

        filter = gtk.FileFilter()
        filter.set_name(_('Splashy theme files(*.tar.gz, *.tar.bz2)'))
        filter.add_pattern('*.tar.gz')
        filter.add_pattern('*.tar.bz2')
        chooser.add_filter(filter)
        filter = gtk.FileFilter()
        filter.set_name(_('All files'))
        filter.add_pattern('*')
        chooser.add_filter(filter)
        response = chooser.run()
        if response == gtk.RESPONSE_OK:
            filename = chooser.get_filename()
            self.splashy.add_theme(filename)
            self.set_splashy_widgets()
        chooser.destroy()

    def on_remove_splashy_button_clicked(self, widget):
        active = self.splashy_combo.get_active_text()
        selected = self.splashy_treeview.get_selection().get_selected()
        model = selected[0]
        iter = selected[1]
        name = model.get_value(iter, 0)
        if active == name:
            dialog = gtk.MessageDialog(parent=self.main_window, 
                                       flags=gtk.DIALOG_MODAL, 
                                       type=gtk.MESSAGE_INFO,
                                       buttons=gtk.BUTTONS_OK,
                                       message_format=_('You can not remove '
                                                        'the currently '
                                                        'active splashy '
                                                        'theme.'))
            dialog.run()
            dialog.destroy()
        else:
            self.splashy.remove_theme(name)
            self.set_splashy_widgets()

    def on_password_protect_check_toggled(self, widget):
        active = self.password_protect_check.get_active()
        self.grub.set_password_protection(active)

    def on_protect_rescuemode_check_toggled(self, widget):
        active = self.protect_rescuemode_check.get_active()
        self.grub.set_protect_rescuemode(active)

    def on_protect_old_check_toggled(self, widget):
        active = self.protect_old_check.get_active()
        self.grub.set_protect_oldmode(active)

    def on_update_password_button_clicked(self, widget):
        password = self.password_entry.get_text()
        active = self.password_protect_check.get_active()
        if password == self.confirm_password_entry.get_text() and \
           len(password) > 3:
            try:
                self.grub.update_password(password, active)
                message_text = _('Password changed')
                message_type = gtk.MESSAGE_INFO
            except:
                message_text = _('Error while changing passwords.')
                message_type = gtk.MESSAGE_ERROR

            dialog = gtk.MessageDialog(parent=self.main_window, 
                                       flags=gtk.DIALOG_MODAL, 
                                       type=message_type, 
                                       buttons=gtk.BUTTONS_OK,
                                       message_format= message_text)
            dialog.run()
            dialog.destroy()

        else:
            if len(password) < 4:
                message_text = _('Password must be at least 4 characters')
            else:
                message_text = _('Passwords do not match')

            dialog = gtk.MessageDialog(parent=self.main_window, 
                                       flags=gtk.DIALOG_MODAL, 
                                       type=gtk.MESSAGE_ERROR, 
                                       buttons=gtk.BUTTONS_OK,
                                       message_format= message_text)
            dialog.run()
            dialog.destroy()

    def on_kernel_check_toggled(self, widget):
        active = self.kernel_check.get_active()
        number = self.kernel_spinner.get_value_as_int()
        self.grub.set_kernel_limit(active, number)

    def on_kernel_spinner_value_changed(self, widget):
        active = self.kernel_check.get_active()
        number = self.kernel_spinner.get_value_as_int()
        self.grub.set_kernel_limit(active, number)

    def on_memtest_check_toggled(self, widget):
        active = self.memtest_check.get_active()
        self.grub.set_memtest(active)

    def on_recovery_check_toggled(self, widget):
        active = self.recovery_check.get_active()
        self.grub.set_create_alternative(active)

    def on_updatedefault_check_toggled(self, widget):
        active = self.updatedefault_check.get_active()
        self.grub.set_update_default(active)

    def on_rescuefloppy_button_clicked(self, widget):
        ret = 1
        dialog = gtk.MessageDialog(parent=self.main_window, 
                                   flags=gtk.DIALOG_MODAL, 
                                   type=gtk.MESSAGE_INFO,\
                                   buttons=gtk.BUTTONS_OK_CANCEL,\
                                   message_format=_('Insert a floppy. '
                                                    'Note that everything '
                                                    'on the floppy will '
                                                    'be deleted.'))
        dialog.set_default_response(gtk.RESPONSE_OK)
        response = dialog.run()
        dialog.destroy()
        if not response == gtk.RESPONSE_OK:
            return
        else:
            while not ret == 0:
                dialog = gtk.Dialog(title='', parent=self.main_window, 
                                    flags=gtk.DIALOG_MODAL)
                progress=gtk.ProgressBar()
                progress.set_text(_('Formatting floppy'))
                dialog.vbox.add(progress)
                progress.show()
                dialog.show()
                thread = FloppyThread(self.grub, 'format')
                thread.start()
                while(1):
                    while(gtk.events_pending()):
                        gtk.main_iteration()
                    if not thread.isAlive():
                        break
                    else:
                        progress.pulse()
                        time.sleep(0.1)
                dialog.destroy()
                ret = thread.ret
                if ret == 1:
                    dialog = gtk.MessageDialog(parent=self.main_window,
                                               flags=gtk.DIALOG_MODAL, 
                                               type=gtk.MESSAGE_QUESTION,
                                               buttons=gtk.BUTTONS_YES_NO,
                                               message_format=_('Floppy not '
                                                                'found. '
                                                                'Try again?'))
                    dialog.set_default_response(gtk.RESPONSE_YES)
                    response = dialog.run()
                    dialog.destroy()
                    if not response == gtk.RESPONSE_YES:
                        ret = 2
                if ret == 2:
                    return
                elif ret >= 3 or ret < 0:
                    print ret
                    dialog = gtk.MessageDialog(parent=self.main_window, 
                                               flags=gtk.DIALOG_MODAL, 
                                               type=gtk.MESSAGE_INFO,
                                               buttons=gtk.BUTTONS_OK,
                                               message_format=_('Error.'))
                    dialog.run()
                    dialog.destroy()
                    return
        dialog = gtk.Dialog(title='', parent=self.main_window, 
                            flags=gtk.DIALOG_MODAL)
        progress = gtk.ProgressBar()
        progress.set_text(_('Creating rescue floppy'))
        dialog.vbox.add(progress)
        progress.show()
        dialog.show()
        thread = FloppyThread(self.grub, 'write')
        thread.start()
        while(1):
            while(gtk.events_pending()):
                gtk.main_iteration()
            if not thread.isAlive():
                break
            else:
                progress.pulse()
                time.sleep(0.1)
        dialog.destroy()
        ret = thread.ret
        if ret == 0:
            dialog = gtk.MessageDialog(parent=self.main_window, 
                                       flags=gtk.DIALOG_MODAL, 
                                       type=gtk.MESSAGE_INFO,
                                       buttons=gtk.BUTTONS_OK,
                                       message_format=_('Rescue floppy created'
                                                        ' successfully.'))
            dialog.run()
            dialog.destroy()
        else:
            dialog = gtk.MessageDialog(parent=self.main_window, 
                                       flags=gtk.DIALOG_MODAL, 
                                       type=gtk.MESSAGE_INFO,
                                       buttons=gtk.BUTTONS_OK,
                                       message_format=_('Error.'))
            dialog.run()
            dialog.destroy()

    def on_restore_settings_button_clicked(self, widget):
        if self.grub.restore_config():
            dialog = gtk.MessageDialog(parent=self.main_window, 
                                       flags=gtk.DIALOG_MODAL, 
                                       type=gtk.MESSAGE_INFO, 
                                       buttons=gtk.BUTTONS_OK,
                                       message_format=_('Original settings '
                                                        'restored.\n'
                                                        'Shutting down'))
            dialog.run()
            dialog.destroy()
            self.grub.update_grub = False
            self.on_main_window_destroy(widget)
        else:
            dialog = gtk.MessageDialog(parent=self.main_window, 
                                       flags=gtk.DIALOG_MODAL, 
                                       type=gtk.MESSAGE_ERROR, 
                                       buttons=gtk.BUTTONS_OK,
                                       message_format=_('Failed to '
                                                        'restore config'))
            dialog.run()
            dialog.destroy()

    def on_help_button_clicked(self, widget):
        gnome.help_display(APP_NAME, '')

    def on_grub_help_button_clicked(self, widget):
        gnome.help_display(APP_NAME, 'grub')

    def on_usplash_help_button_clicked(self, widget):
        gnome.help_display(APP_NAME, 'usplash')

    def on_splashy_help_button_clicked(self, widget):
        gnome.help_display(APP_NAME, 'splashy')

    def on_grub_close_button_clicked(self, widget):
        self.grub_dialog.hide()

    def on_usplash_close_button_clicked(self, widget):
        self.usplash_dialog.hide()

    def on_splashy_close_button_clicked(self, widget):
        self.splashy_dialog.hide()

    def on_close_button_clicked(self, widget):
        self.on_main_window_destroy(widget)

    def on_main_window_destroy(self, widget):
        self.main_window.hide()
        dialog = gtk.Dialog(title=_('StartUp-Manager'), parent=self.main_window)
        progress = gtk.ProgressBar()
        progress.set_text(_('Performing post-configuration tasks'))
        dialog.vbox.add(progress)
        progress.show()
        dialog.show()
        if not self.grub:
            self.grub = None
        if not self.usplash:
            self.usplash = None
        if not self.splashy:
            self.splashy = None

        ramfs = ShutdownThread(self.grub, self.usplash, self.splashy)
        ramfs.start()

        while(1):
            while(gtk.events_pending()):
                gtk.main_iteration()
            if not ramfs.isAlive():
                break
            else:
                progress.pulse()
                time.sleep(0.1)
        dialog.destroy()
        gtk.main_quit()

    def get_widgets(self):
        get_widget = self.glade_xml.get_widget
        self.main_window = get_widget('main_window')
        self.boot_options_vbox = get_widget('boot_options_vbox')
        self.timeout_frame = get_widget('timeout_frame')
        self.defaultboot_frame = get_widget('defaultboot_frame')
        self.display_frame = get_widget('display_frame')
        self.misc_frame = get_widget('misc_frame')
        self.timeout_check = get_widget('timeout_check')
        self.timeout_spinner = get_widget('timeout_spinner')
        self.default_boot_combo = get_widget('default_boot_combo')
        self.resolution_combo = get_widget('resolution_combo')
        self.color_depth_label = get_widget('color_depth_label')
        self.color_depth_combo = get_widget('color_depth_combo')
        self.grub_menu_check = get_widget('grub_menu_check')
        self.boot_splash_check = get_widget('boot_splash_check')
        self.boot_text_check = get_widget('boot_text_check')
        self.appearance_vbox = get_widget('appearance_vbox')
        self.grub_color_frame = get_widget('grub_color_frame')
        self.grub_theme_frame = get_widget('grub_theme_frame')
        self.splashy_frame = get_widget('splashy_frame')
        self.usplash_frame = get_widget('usplash_frame')
        self.grub_color_check = get_widget('grub_color_check')
        self.normal_background_combo = get_widget('normal_background_combo')
        self.normal_foreground_combo = get_widget('normal_foreground_combo')
        self.normal_blink_check = get_widget('normal_blink_check')
        self.highlighted_background_combo = get_widget('highlighted_background_combo')
        self.highlighted_foreground_combo = get_widget('highlighted_foreground_combo')
        self.highlighted_blink_check = get_widget('highlighted_blink_check')
        self.grub_splash_check = get_widget('grub_splash_check')
        self.grub_splash_combo = get_widget('grub_splash_combo')
        self.splashy_combo = get_widget('splashy_combo')
        self.usplash_combo = get_widget('usplash_combo')
        self.security_vbox = get_widget('security_vbox')
        self.password_protect_check = get_widget('password_protect_check')
        self.protect_rescuemode_check = get_widget('protect_rescuemode_check')
        self.protect_old_check = get_widget('protect_old_check')
        self.password_entry = get_widget('password_entry')
        self.confirm_password_entry = get_widget('confirm_password_entry')
        self.advanced_vbox = get_widget('advanced_vbox')
        self.limit_kernel_frame = get_widget('limit_kernel_frame')
        self.advanced_splashy_frame = get_widget('advanced_splashy_frame')
        self.advanced_misc_frame = get_widget('advanced_misc_frame')
        self.kernel_check = get_widget('kernel_check')
        self.kernel_spinner = get_widget('kernel_spinner')
        self.grub_resolution_frame = get_widget('grub_resolution_frame')
        self.grub_resolution_combo = get_widget('grub_resolution_combo')
        self.splashy_initramfs_check = get_widget('splashy_initramfs_check')
        self.memtest_check = get_widget('memtest_check')
        self.recovery_check = get_widget('recovery_check')
        self.updatedefault_check = get_widget('updatedefault_check')
        self.rescuefloppy_button = get_widget('rescuefloppy_button')
        self.restore_settings_button = get_widget('restore_settings_button')
        self.grub_dialog = get_widget('grub_dialog')
        self.grub_treeview = get_widget('grub_treeview')
        self.splashy_dialog = get_widget('splashy_dialog')
        self.splashy_treeview = get_widget('splashy_treeview')
        self.usplash_dialog = get_widget('usplash_dialog')
        self.usplash_treeview = get_widget('usplash_treeview')

