#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include "mainwin.h"
#include "stardict.h"


#ifndef _WIN32
#  include <libgnome/libgnome.h>
#  include <libgnomeui/libgnomeui.h>
#endif

#ifdef _WIN32
#define VERSION "2.4.3"
#  include "win32/intl.h"
#  include "win32/gnome-about.h"
#  include <gdk/gdkwin32.h>
#endif


#define STARDICT_RESPONSE_FIND 100

/* GTK+ < 2.2.2 hack, see ui.h for details. */
#ifndef GTK_WRAP_WORD_CHAR
#define GTK_WRAP_WORD_CHAR GTK_WRAP_WORD
#endif

/**************************************************/
TopWin::TopWin()
{
	WordCombo = NULL; //need by save_yourself_cb().
	HisList = NULL;
	BackList = NULL;
	enable_change_cb = true;
	MainMenu = NULL;
	HistoryMenu = NULL;
	BackMenu = NULL;
}

TopWin::~TopWin()
{
	g_list_foreach (HisList, (GFunc)g_free, NULL);
	g_list_free(HisList);

	GSList *list = BackList;
	while (list) {
		g_free(((BackListData *)(list->data))->word);
		g_free(list->data);
		list = list->next;
	}
	g_slist_free(BackList);
}

void TopWin::Create(GtkWidget *vbox)
{
	GtkWidget *hbox = gtk_hbox_new(false,0);
	gtk_widget_show(hbox);
	gtk_box_pack_start(GTK_BOX(vbox),hbox,false,false,3);
	
	GtkWidget *button;
	button = gtk_button_new_from_stock(GTK_STOCK_CLEAR);
	gtk_widget_show(button);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	g_signal_connect(G_OBJECT(button),"clicked", G_CALLBACK(ClearCallback),this);
	g_signal_connect(G_OBJECT(button),"enter_notify_event", G_CALLBACK(stardict_on_enter_notify), NULL);
	gtk_box_pack_start(GTK_BOX(hbox),button,false,false,3);
	gtk_tooltips_set_tip(gpAppFrame->oAppCore.tooltips,button,_("Clear the search box"),NULL);
	
	WordCombo = gtk_combo_new();
	gtk_widget_set_size_request(WordCombo,50,-1);
	gtk_widget_show(WordCombo);
	gtk_combo_set_case_sensitive(GTK_COMBO(WordCombo),true);
	gtk_combo_disable_activate(GTK_COMBO(WordCombo));
	gtk_combo_set_use_arrows(GTK_COMBO(WordCombo), false);
	gtk_entry_set_max_length(GTK_ENTRY(GTK_COMBO(WordCombo)->entry), 256);
	g_signal_connect (G_OBJECT (GTK_COMBO(WordCombo)->entry), "changed",
			  G_CALLBACK (on_entry_changed), this);
	g_signal_connect (G_OBJECT (GTK_COMBO(WordCombo)->entry), "activate",
			  G_CALLBACK (on_entry_activate), this);
	gtk_box_pack_start(GTK_BOX(hbox),WordCombo,true,true,3);
	
	button=gtk_button_new();
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock(GTK_STOCK_FIND,GTK_ICON_SIZE_BUTTON));	
	gtk_widget_show_all(button);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	g_signal_connect(G_OBJECT(button),"clicked", G_CALLBACK(GoCallback),this);
	g_signal_connect(G_OBJECT(button),"enter_notify_event", G_CALLBACK(stardict_on_enter_notify), NULL);
	gtk_box_pack_start(GTK_BOX(hbox),button,false,false,0);
	gtk_tooltips_set_tip(gpAppFrame->oAppCore.tooltips,button,_("Fuzzy Query"),NULL);

	button=gtk_button_new();
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock(GTK_STOCK_UNDO,GTK_ICON_SIZE_BUTTON));	
	gtk_widget_show_all(button);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	g_signal_connect(G_OBJECT(button),"clicked", G_CALLBACK(BackCallback),this);
	g_signal_connect(G_OBJECT(button),"button_press_event", G_CALLBACK(on_back_button_press),this);
	g_signal_connect(G_OBJECT(button),"enter_notify_event", G_CALLBACK(stardict_on_enter_notify), NULL);
	gtk_box_pack_start(GTK_BOX(hbox),button,false,false,0);
	gtk_tooltips_set_tip(gpAppFrame->oAppCore.tooltips,button,_("Go Back - Right button: history (Alt+Left)"),NULL);

	button=gtk_button_new();	
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock(GTK_STOCK_GO_BACK,GTK_ICON_SIZE_BUTTON));
	gtk_widget_show_all(button);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	g_signal_connect(G_OBJECT(button),"clicked", G_CALLBACK(PreviousCallback),this);
	g_signal_connect(G_OBJECT(button),"enter_notify_event", G_CALLBACK(stardict_on_enter_notify), NULL);
	gtk_box_pack_start(GTK_BOX(hbox),button,false,false,0);
	gtk_tooltips_set_tip(gpAppFrame->oAppCore.tooltips,button,_("Previous word (Alt+Up)"),NULL);

	button=gtk_button_new();
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock(GTK_STOCK_GO_FORWARD,GTK_ICON_SIZE_BUTTON));
	gtk_widget_show_all(button);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	g_signal_connect(G_OBJECT(button),"clicked", G_CALLBACK(NextCallback),this);
	g_signal_connect(G_OBJECT(button),"enter_notify_event", G_CALLBACK(stardict_on_enter_notify), NULL);
	gtk_box_pack_start(GTK_BOX(hbox),button,false,false,0);
	gtk_tooltips_set_tip(gpAppFrame->oAppCore.tooltips,button,_("Next word (Alt+Down)"),NULL);

	button=gtk_button_new();
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock(GTK_STOCK_HOME,GTK_ICON_SIZE_BUTTON));
	gtk_widget_show_all(button);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	g_signal_connect(G_OBJECT(button),"clicked", G_CALLBACK(MenuCallback),this);
	g_signal_connect(G_OBJECT(button),"enter_notify_event", G_CALLBACK(stardict_on_enter_notify), NULL);
	gtk_box_pack_start(GTK_BOX(hbox),button,false,false,0);	
	gtk_tooltips_set_tip(gpAppFrame->oAppCore.tooltips,button,_("Show the main menu (Alt+M)"),NULL);
}

void TopWin::on_entry_changed(GtkEntry *entry, TopWin *oTopWin)
{
	if (!(oTopWin->enable_change_cb))
		return;
	const gchar *sWord = gtk_entry_get_text(entry);
    if(sWord[0]!='\0')
    {
        gpAppFrame->oAppCore.TopWinWordChange(sWord);
    }
	else
	{
		gpAppFrame->oAppCore.oMidWin.oTextWin.ShowTips();
	}
}

void TopWin::on_entry_activate(GtkEntry *entry, TopWin *oTopWin)
{
	gpAppFrame->oAppCore.TopWinEnterWord(gtk_entry_get_text(entry));
}

gboolean TopWin::on_back_button_press(GtkWidget * widget, GdkEventButton * event , TopWin *oTopWin)
{
	if (event->button == 3) {
		if (!(oTopWin->BackList))
			return true;
		gint index=0;
		GSList *list;
		list = oTopWin->BackList;
		if (!strcmp(((BackListData *)(list->data))->word, gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(oTopWin->WordCombo)->entry)))) {
			list = g_slist_next(list);
			index++;
			if (!list)
				return true;
		}
		if (oTopWin->BackMenu)
			gtk_widget_destroy(oTopWin->BackMenu);
			
		oTopWin->BackMenu = gtk_menu_new();
		
		GtkWidget *menuitem;
		while (list) {			
			menuitem = gtk_image_menu_item_new_with_label(((BackListData *)(list->data))->word);
			g_signal_connect(G_OBJECT(menuitem), "activate", G_CALLBACK(on_back_menu_item_activate), GINT_TO_POINTER(index));
			gtk_menu_shell_append(GTK_MENU_SHELL(oTopWin->BackMenu), menuitem);
			list = g_slist_next(list);
			index++;
		}
		gtk_widget_show_all(oTopWin->BackMenu);
		gtk_menu_popup(GTK_MENU(oTopWin->BackMenu), NULL, NULL, NULL, NULL, event->button, event->time);		
		return true;
	}
	return false;
}

void TopWin::on_back_menu_item_activate(GtkMenuItem *menuitem, gint index)
{
	for (int i=0;i<index;i++) {
		g_free(((BackListData *)((gpAppFrame->oAppCore.oTopWin.BackList)->data))->word);
		g_free((gpAppFrame->oAppCore.oTopWin.BackList)->data);
		gpAppFrame->oAppCore.oTopWin.BackList = g_slist_delete_link(gpAppFrame->oAppCore.oTopWin.BackList, gpAppFrame->oAppCore.oTopWin.BackList);	
	}
	gpAppFrame->oAppCore.oTopWin.InsertHisList(gpAppFrame->oAppCore.oTopWin.GetText());
	gpAppFrame->oAppCore.oTopWin.SetText(((BackListData *)((gpAppFrame->oAppCore.oTopWin.BackList)->data))->word);
	if (((BackListData *)(gpAppFrame->oAppCore.oTopWin.BackList->data))->adjustment_value != -1) {
		gpAppFrame->ProcessGtkEvent();
		gtk_adjustment_set_value(gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(gpAppFrame->oAppCore.oMidWin.oTextWin.scrolled_window)), ((BackListData *)(gpAppFrame->oAppCore.oTopWin.BackList->data))->adjustment_value);
	}
	g_free(((BackListData *)((gpAppFrame->oAppCore.oTopWin.BackList)->data))->word);
	g_free((gpAppFrame->oAppCore.oTopWin.BackList)->data);
	gpAppFrame->oAppCore.oTopWin.BackList = g_slist_delete_link(gpAppFrame->oAppCore.oTopWin.BackList, gpAppFrame->oAppCore.oTopWin.BackList);	
}

void TopWin::ClearCallback(GtkWidget *widget, TopWin *oTopWin)
{
	if (gpAppFrame->enable_sound_event) {
#ifdef _WIN32
		gchar *filename = g_build_filename(stardict_data_dir, "sounds", "buttonactive.wav", NULL);
		PlaySound(filename, 0, SND_ASYNC | SND_FILENAME);
		g_free(filename);
#else
		gnome_sound_play(STARDICT_DATA_DIR "/sounds/buttonactive.wav");
#endif
	}
	oTopWin->InsertHisList(oTopWin->GetText());
	oTopWin->InsertBackList();
	oTopWin->SetText("");
	oTopWin->GrabFocus();
}

void TopWin::GoCallback(GtkWidget *widget, TopWin *oTopWin)
{
	if (gpAppFrame->enable_sound_event) {
#ifdef _WIN32
		gchar *filename = g_build_filename(stardict_data_dir, "sounds", "buttonactive.wav", NULL);
		PlaySound(filename, 0, SND_ASYNC | SND_FILENAME);
		g_free(filename);
#else
		gnome_sound_play(STARDICT_DATA_DIR "/sounds/buttonactive.wav");
#endif
	}
	const gchar *text = oTopWin->GetText();
	if (text[0]=='\0')
		return;
	
	if (text[0]=='/')
		gpAppFrame->oAppCore.LookupWithFuzzyToMainWin(text+1);
    else if( bContainRule(text) )
        gpAppFrame->oAppCore.LookupWithRuleToMainWin(text);
	else
		gpAppFrame->oAppCore.LookupWithFuzzyToMainWin(text);

	oTopWin->TextSelectAll();
	oTopWin->GrabFocus();
	oTopWin->InsertHisList(text);
	oTopWin->InsertBackList(text);
}

void TopWin::do_back()
{
	if (!BackList)
		return;
	GSList *list;
	list = BackList;
	if (!strcmp(((BackListData *)(list->data))->word, gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(WordCombo)->entry)))) {
		list = g_slist_next(list);
		if (!list)
			return;
	}
	InsertHisList(GetText());
	SetText(((BackListData *)(list->data))->word);
	if (GTK_WIDGET_HAS_FOCUS(GTK_WIDGET(GTK_COMBO(WordCombo)->entry)))
		gtk_editable_select_region(GTK_EDITABLE(GTK_COMBO(WordCombo)->entry),0,-1);
	if (((BackListData *)(list->data))->adjustment_value != -1) {
		gpAppFrame->ProcessGtkEvent(); // so all the definition text have been inserted.
		gtk_adjustment_set_value(gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(gpAppFrame->oAppCore.oMidWin.oTextWin.scrolled_window)), ((BackListData *)(list->data))->adjustment_value);
	}
	g_free(((BackListData *)(list->data))->word);
	g_free(list->data);
	BackList = g_slist_delete_link(BackList,list);	
}

void TopWin::BackCallback(GtkWidget *widget, TopWin *oTopWin)
{
	if (gpAppFrame->enable_sound_event) {
#ifdef _WIN32
		gchar *filename = g_build_filename(stardict_data_dir, "sounds", "buttonactive.wav", NULL);
		PlaySound(filename, 0, SND_ASYNC | SND_FILENAME);
		g_free(filename);
#else
		gnome_sound_play(STARDICT_DATA_DIR "/sounds/buttonactive.wav");
#endif
	}
	oTopWin->do_back();
}

void TopWin::do_previous()
{
	if (gpAppFrame->oAppCore.oMidWin.oIndexWin.oListWin.list_word_type == LIST_WIN_EMPTY)
		return;
	GtkTreeSelection *selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(gpAppFrame->oAppCore.oMidWin.oIndexWin.oListWin.treeview));
	GtkTreeModel *model;
	GtkTreeIter iter;

	gboolean selected = gtk_tree_selection_get_selected(selection,&model,&iter);
	if (gpAppFrame->oAppCore.oMidWin.oIndexWin.oListWin.list_word_type == LIST_WIN_NORMAL_LIST) {
		if (!(selected && GTK_WIDGET_HAS_FOCUS(gpAppFrame->oAppCore.oMidWin.oIndexWin.oListWin.treeview))) {
			if (!gtk_tree_model_get_iter_first(model,&iter))
				return; //this should never happen.
		}
		GtkTreePath* path = gtk_tree_model_get_path(model,&iter);	
		if (gtk_tree_path_prev(path)) {
			gtk_tree_model_get_iter(model,&iter,path);
			gtk_tree_selection_select_iter(selection,&iter);
			gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(gpAppFrame->oAppCore.oMidWin.oIndexWin.oListWin.treeview),path,NULL,false,0,0);
		}
		else {
			// user have selected the first row.
			gchar *preword = gpAppFrame->oAppCore.oLibs.poGetPreWord(gpAppFrame->oAppCore.iCurrentIndex);
			if (preword) {
				SetText_without_notify(preword);
				if (GTK_WIDGET_HAS_FOCUS(GTK_WIDGET(GTK_COMBO(WordCombo)->entry)))
					gtk_editable_select_region(GTK_EDITABLE(GTK_COMBO(WordCombo)->entry),0,-1);
				gpAppFrame->oAppCore.SimpleLookupToTextWin(preword,gpAppFrame->oAppCore.iCurrentIndex,true);
				
				gpAppFrame->oAppCore.ListWords(gpAppFrame->oAppCore.iCurrentIndex);
				// the next lines should be faster than ListWords(),but as the TreeView can't be freezed,it make it interface changing not so smoothly.
				/*gpAppFrame->oAppCore.oMidWin.oListWin.Prepend(preword);
				gpAppFrame->oAppCore.oMidWin.oListWin.RemoveLast();
				gpAppFrame->oAppCore.oMidWin.oListWin.UnSelectAll();
        		gpAppFrame->oAppCore.oMidWin.oListWin.ReScroll();*/
			}
		}		
		gtk_tree_path_free(path);
	}
	else if (gpAppFrame->oAppCore.oMidWin.oIndexWin.oListWin.list_word_type == LIST_WIN_FUZZY_LIST ||
			gpAppFrame->oAppCore.oMidWin.oIndexWin.oListWin.list_word_type == LIST_WIN_PATTERN_LIST) {
		if (!selected) {
			if (!gtk_tree_model_get_iter_first(model,&iter))
				return; //this should never happen.
		}
		GtkTreePath* path = gtk_tree_model_get_path(model,&iter);	
		if (gtk_tree_path_prev(path)) {
			gtk_tree_model_get_iter(model,&iter,path);
			gtk_tree_selection_select_iter(selection,&iter);
			gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(gpAppFrame->oAppCore.oMidWin.oIndexWin.oListWin.treeview),path,NULL,false,0,0);
		}
		else {
			// user have selected the first row,no action is need.
		}
		gtk_tree_path_free(path);
	}	
}

void TopWin::PreviousCallback(GtkWidget *widget, TopWin *oTopWin)
{	
	if (gpAppFrame->enable_sound_event) {
#ifdef _WIN32
		gchar *filename = g_build_filename(stardict_data_dir, "sounds", "buttonactive.wav", NULL);
		PlaySound(filename, 0, SND_ASYNC | SND_FILENAME);
		g_free(filename);
#else
		gnome_sound_play(STARDICT_DATA_DIR "/sounds/buttonactive.wav");
#endif
	}
	oTopWin->do_previous();
}

void TopWin::do_next()
{
	if (gpAppFrame->oAppCore.oMidWin.oIndexWin.oListWin.list_word_type == LIST_WIN_EMPTY)
		return;
	GtkTreeSelection *selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(gpAppFrame->oAppCore.oMidWin.oIndexWin.oListWin.treeview));
	GtkTreeModel *model;
	GtkTreeIter iter;

	gboolean selected = gtk_tree_selection_get_selected(selection,&model,&iter); //make sure this will run,so model is set.
	if (gpAppFrame->oAppCore.oMidWin.oIndexWin.oListWin.list_word_type == LIST_WIN_NORMAL_LIST) {
		gboolean real_selected = (selected && GTK_WIDGET_HAS_FOCUS(gpAppFrame->oAppCore.oMidWin.oIndexWin.oListWin.treeview));
		if (!real_selected) {
			if (!gtk_tree_model_get_iter_first(model,&iter))
				return; //this should never happen.
		}
		GtkTreeIter new_iter = iter; //if gtk_tree_model_iter_next fail,iter will be invalid,so save it.
		gchar *word;
		if (gtk_tree_model_iter_next(model,&iter)) {
			if (!real_selected) {
				gtk_tree_model_get (model, &iter, 0, &word, -1);
				SetText(word);
				if (GTK_WIDGET_HAS_FOCUS(GTK_WIDGET(GTK_COMBO(WordCombo)->entry)))
					gtk_editable_select_region(GTK_EDITABLE(GTK_COMBO(WordCombo)->entry),0,-1);
				g_free(word);
			}
			else {
				gtk_tree_selection_select_iter(selection,&iter);
				GtkTreePath* path = gtk_tree_model_get_path(model,&iter);
				gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(gpAppFrame->oAppCore.oMidWin.oIndexWin.oListWin.treeview),path,NULL,false,0,0);
				gtk_tree_path_free(path);				
			}
		}
		else {
			// user have selected the last row.
			gtk_tree_model_get (model, &new_iter, 0, &word, -1);
			glong *iNextIndex = (glong *)g_malloc(sizeof(glong) * gpAppFrame->oAppCore.oLibs.total_libs());
			gchar *nextword = gpAppFrame->oAppCore.oLibs.poGetNextWord(word, iNextIndex);
			if (nextword) {
				SetText_without_notify(nextword);
				if (GTK_WIDGET_HAS_FOCUS(GTK_WIDGET(GTK_COMBO(WordCombo)->entry)))
					gtk_editable_select_region(GTK_EDITABLE(GTK_COMBO(WordCombo)->entry),0,-1);
				gpAppFrame->oAppCore.SimpleLookupToTextWin(nextword,iNextIndex,true);
				gpAppFrame->oAppCore.ListWords(iNextIndex);
			}
			g_free(iNextIndex);
			g_free(word);
		}		
	}
	else if (gpAppFrame->oAppCore.oMidWin.oIndexWin.oListWin.list_word_type == LIST_WIN_FUZZY_LIST ||
			gpAppFrame->oAppCore.oMidWin.oIndexWin.oListWin.list_word_type == LIST_WIN_PATTERN_LIST) {
		if (!selected) {
			if (!gtk_tree_model_get_iter_first(model,&iter))
				return; //this should never happen.
		}
		if (gtk_tree_model_iter_next(model,&iter)) {
			gtk_tree_selection_select_iter(selection,&iter);
			GtkTreePath* path = gtk_tree_model_get_path(model,&iter);
			gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(gpAppFrame->oAppCore.oMidWin.oIndexWin.oListWin.treeview),path,NULL,false,0,0);
			gtk_tree_path_free(path);
		}
		else {
			// user have selected the last row,no action is need.
		}
	}
}

void TopWin::NextCallback(GtkWidget *widget, TopWin *oTopWin)
{
	if (gpAppFrame->enable_sound_event) {
#ifdef _WIN32
		gchar *filename = g_build_filename(stardict_data_dir, "sounds", "buttonactive.wav", NULL);
		PlaySound(filename, 0, SND_ASYNC | SND_FILENAME);
		g_free(filename);
#else
		gnome_sound_play(STARDICT_DATA_DIR "/sounds/buttonactive.wav");
#endif
	}
	oTopWin->do_next();
}

void TopWin::on_main_menu_preferences_activate(GtkMenuItem *menuitem, TopWin *oTopWin)
{	
	gpAppFrame->oAppCore.PopupPrefsDlg();
}

void TopWin::on_main_menu_dictmanage_activate(GtkMenuItem *menuitem, TopWin *oTopWin)
{
	gpAppFrame->oAppCore.PopupDictManageDlg();
}

void TopWin::on_main_menu_newversion_activate(GtkMenuItem *menuitem, TopWin *oTopWin)
{
#ifdef _WIN32
	ShellExecute((HWND)(GDK_WINDOW_HWND(gpAppFrame->oAppCore.window->window)), "OPEN", "http://stardict.sourceforge.net", NULL, NULL, SW_SHOWNORMAL);
#else
	gnome_url_show("http://stardict.sourceforge.net", NULL);
#endif
}

void TopWin::on_main_menu_help_activate(GtkMenuItem *menuitem, TopWin *oTopWin)
{
#ifdef _WIN32
	gchar *filename = g_strdup_printf(_("file:///%s/help/C/stardict.html"), stardict_data_dir);
	ShellExecute((HWND)(GDK_WINDOW_HWND(gpAppFrame->oAppCore.window->window)), "OPEN", filename, NULL, NULL, SW_SHOWNORMAL);
	g_free(filename);
#else
	gnome_help_display ("stardict.xml", NULL, NULL);
#endif
}

void TopWin::on_main_menu_about_activate(GtkMenuItem *menuitem, TopWin *oTopWin)
{
	const gchar *authors[] = {
    "Hu Zheng <huzheng_001@163.com>",
    "Opera Wang <wangvisual@sohu.com>",
    "Ma Su'an <msa@wri.com.cn>",
        NULL
    };
    gchar *documenters[] = {
		"Hu Zheng <huzheng_001@163.com>",
		"Will Robinson <wsr23@stanford.edu>",
		"Anthony Fok <foka@debian.org>",
	    NULL
    };
    gchar *translator_credits = _("translator_credits");
    GtkWidget *about;

    about = gnome_about_new (_("StarDict"), VERSION,
							"Copyright \xc2\xa9 1999 by Ma Su'an\n"
							"Copyright \xc2\xa9 2002 by Opera Wang\n"
                            "Copyright \xc2\xa9 2003 by Hu Zheng",
                            _("StarDict is an international dictionary for GNOME.\n"),
			     (const char **)authors,
			     (const char **)documenters,
			     strcmp (translator_credits, "translator_credits") != 0 ? translator_credits : NULL,
                             gpAppFrame->oAppSkin.stardict.icon.p[0]);
    gtk_window_set_icon(GTK_WINDOW (about), gpAppFrame->oAppSkin.stardict.icon.p[0]);
	
	gtk_window_set_transient_for (GTK_WINDOW (about), GTK_WINDOW (gpAppFrame->oAppCore.window));
	gtk_window_set_destroy_with_parent(GTK_WINDOW (about), true);
	gtk_widget_show(about);
}

void TopWin::on_main_menu_quit_activate(GtkMenuItem *menuitem, TopWin *oTopWin)
{
	gpAppFrame->Quit();
}

void TopWin::do_menu()
{
	if (gpAppFrame->enable_sound_event) {
#ifdef _WIN32
		gchar *filename = g_build_filename(stardict_data_dir, "sounds", "menushow.wav", NULL);
		PlaySound(filename, 0, SND_ASYNC | SND_FILENAME);
		g_free(filename);
#else
		gnome_sound_play(STARDICT_DATA_DIR "/sounds/menushow.wav");
#endif
	}
	if (!MainMenu) {
		MainMenu = gtk_menu_new();

		GtkWidget *menuitem;
		menuitem = gtk_image_menu_item_new_with_mnemonic(_("Pr_eferences"));
		GtkWidget *image;
		image = gtk_image_new_from_stock(GTK_STOCK_PREFERENCES, GTK_ICON_SIZE_MENU);
		gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), image);
		g_signal_connect(G_OBJECT(menuitem), "activate", G_CALLBACK(on_main_menu_preferences_activate), NULL);
		gtk_menu_shell_append(GTK_MENU_SHELL(MainMenu), menuitem);

		menuitem = gtk_image_menu_item_new_with_mnemonic(_("_Dict Manage"));
		image = gtk_image_new_from_stock(GTK_STOCK_PROPERTIES, GTK_ICON_SIZE_MENU);
		gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), image);
		g_signal_connect(G_OBJECT(menuitem), "activate", G_CALLBACK(on_main_menu_dictmanage_activate), NULL);
		gtk_menu_shell_append(GTK_MENU_SHELL(MainMenu), menuitem);
		menuitem = gtk_separator_menu_item_new();
		gtk_menu_shell_append(GTK_MENU_SHELL(MainMenu), menuitem);

		menuitem = gtk_image_menu_item_new_with_mnemonic(_("_New Version"));
		image = gtk_image_new_from_stock(GTK_STOCK_REFRESH, GTK_ICON_SIZE_MENU);
		gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), image);
		g_signal_connect(G_OBJECT(menuitem), "activate", G_CALLBACK(on_main_menu_newversion_activate), NULL);
		gtk_menu_shell_append(GTK_MENU_SHELL(MainMenu), menuitem);
		menuitem = gtk_separator_menu_item_new();
		gtk_menu_shell_append(GTK_MENU_SHELL(MainMenu), menuitem);

		menuitem = gtk_image_menu_item_new_with_mnemonic(_("_Help"));
		image = gtk_image_new_from_stock(GTK_STOCK_HELP, GTK_ICON_SIZE_MENU);
		gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), image);
		g_signal_connect(G_OBJECT(menuitem), "activate", G_CALLBACK(on_main_menu_help_activate), NULL);
		gtk_menu_shell_append(GTK_MENU_SHELL(MainMenu), menuitem);

		menuitem = gtk_image_menu_item_new_with_mnemonic(_("_About"));
#ifdef _WIN32
		image = gtk_image_new_from_pixbuf(gpAppFrame->oAppSkin.stardict.about_menu.p[0]);
#else
		image = gtk_image_new_from_stock (GNOME_STOCK_ABOUT, GTK_ICON_SIZE_BUTTON);
#endif
		gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), image);
		g_signal_connect(G_OBJECT(menuitem), "activate", G_CALLBACK(on_main_menu_about_activate), NULL);
		gtk_menu_shell_append(GTK_MENU_SHELL(MainMenu), menuitem);
		menuitem = gtk_separator_menu_item_new();
		gtk_menu_shell_append(GTK_MENU_SHELL(MainMenu), menuitem);

		menuitem = gtk_image_menu_item_new_with_mnemonic(_("_Quit"));
		image = gtk_image_new_from_stock(GTK_STOCK_QUIT, GTK_ICON_SIZE_MENU);
		gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), image);
		g_signal_connect(G_OBJECT(menuitem), "activate", G_CALLBACK(on_main_menu_quit_activate), NULL);
		gtk_menu_shell_append(GTK_MENU_SHELL(MainMenu), menuitem);

		gtk_widget_show_all(MainMenu);

	}
	gtk_menu_popup(GTK_MENU(MainMenu), NULL, NULL, NULL, NULL, 1, gtk_get_current_event_time());
}

void TopWin::MenuCallback(GtkWidget *widget, TopWin *oTopWin)
{
	oTopWin->do_menu();
}

void TopWin::SetText(const gchar *word)
{
	if (strcmp(word, gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(WordCombo)->entry))) == 0)
		return;
	enable_change_cb = false;
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(WordCombo)->entry),word); // this will emit "changed" signal twice, one for "", one for word. so disable it.
	enable_change_cb = true;
	on_entry_changed(GTK_ENTRY(GTK_COMBO(WordCombo)->entry), this);
}

void TopWin::SetText_without_notify(const gchar *word)
{
	enable_change_cb = false;
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(WordCombo)->entry),word);
	enable_change_cb = true;
}

const gchar* TopWin::GetText()
{
	return gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(WordCombo)->entry));
}

void TopWin::GrabFocus()
{
	gtk_widget_grab_focus(GTK_COMBO(WordCombo)->entry);
}

void TopWin::TextSelectAll()
{
	gtk_editable_select_region(GTK_EDITABLE(GTK_COMBO(WordCombo)->entry),0,-1);
}

gboolean TopWin::TextSelected()
{
    return (gtk_editable_get_selection_bounds(GTK_EDITABLE(GTK_COMBO(WordCombo)->entry),NULL,NULL));
}

gint TopWin::HisCompareFunc(gconstpointer a,gconstpointer b)
{
	return strcmp((const gchar *)a, (const gchar *)b);
/*	const gchar *c,*d;
	c= (const gchar *)a;
	d= (const gchar *)b;
	if (bContainRule(c) || bContainRule(d))
		return strcmp(c,d);
	else
		return g_ascii_strcasecmp(c,d); //fuzzy search also use casecmp.
*/
}

gint TopWin::BackListDataCompareFunc(gconstpointer a,gconstpointer b)
{
	return strcmp(((const BackListData *)a)->word, ((const BackListData *)b)->word);
}

void TopWin::InsertHisList(const gchar *word)
{
	if ((!word) || (word[0]=='\0'))
		return;
	GList* list;
	gchar *str = g_strdup(word);
	list = g_list_find_custom(HisList,str,HisCompareFunc);
	if (list)
	{
		g_free(list->data);
		HisList = g_list_delete_link(HisList,list);
		HisList = g_list_prepend(HisList,str);
	}
	else
	{
		HisList = g_list_prepend(HisList,str);
	}
	if (g_list_length(HisList)> MAX_HISTORY_WORD_ITEM_NUM)
	{
		list = g_list_last(HisList);
		g_free(list->data);
		HisList = g_list_delete_link(HisList,list);
	}
	const gchar *text = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(WordCombo)->entry));
	// if the list's first item don't equal to entry's text,gtk_combo_set_popdown_strings will change the entry's text to it.
	// so here make it always equal.
	if (strcmp(text,word))
	{
		str = g_strdup(text);
		list = g_list_append(NULL,str);
		list = g_list_concat(list,HisList);
		gtk_combo_set_popdown_strings(GTK_COMBO(WordCombo),list);
		g_free(str);
		HisList = g_list_delete_link(list,list);
	}
	else
		gtk_combo_set_popdown_strings(GTK_COMBO(WordCombo),HisList);
}

void TopWin::InsertBackList(const gchar *word)
{
	BackListData *backlistdata;
	if (word) {
		if (word[0] == '\0')
			return;
		backlistdata = (BackListData *)g_malloc(sizeof(BackListData));
		backlistdata->word = g_strdup(word);
		backlistdata->adjustment_value = -1;
	}
	else {
		word = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(WordCombo)->entry));
		if (word[0] == '\0')
			return;
		backlistdata = (BackListData *)g_malloc(sizeof(BackListData));
		backlistdata->word = g_strdup(word);
		backlistdata->adjustment_value = gtk_adjustment_get_value(gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(gpAppFrame->oAppCore.oMidWin.oTextWin.scrolled_window)));
	}	

	GSList* list;
	list = g_slist_find_custom(BackList, backlistdata, BackListDataCompareFunc);
	if (list)
	{
		g_free(((BackListData *)(list->data))->word);
		g_free(list->data);
		BackList = g_slist_delete_link(BackList,list);
		BackList = g_slist_prepend(BackList,backlistdata);
	}
	else
	{
		BackList = g_slist_prepend(BackList,backlistdata);
	}
	if (g_slist_length(BackList)> MAX_BACK_WORD_ITEM_NUM)
	{
		list = g_slist_last(BackList);
		g_free(((BackListData *)(list->data))->word);
		g_free(list->data);
		BackList = g_slist_delete_link(BackList,list);
	}
}

/**************************************************/
ListWin::ListWin()
{
}

void ListWin::Create(GtkWidget *notebook)
{
	list_word_type = LIST_WIN_EMPTY;
	
	GtkListStore *model = gtk_list_store_new (1,G_TYPE_STRING);
	treeview = gtk_tree_view_new_with_model (GTK_TREE_MODEL(model));
	gtk_widget_show(treeview);
	g_object_unref (model);
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (treeview), FALSE);
  
	GtkCellRenderer *renderer = gtk_cell_renderer_text_new ();
	GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes ("word", renderer,
						     "text", 0, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	GtkTreeSelection *selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
	g_signal_connect (G_OBJECT (selection), "changed", G_CALLBACK (on_selection_changed), this);
	
	g_signal_connect (G_OBJECT (treeview), "button_press_event", G_CALLBACK (on_button_press), this);

	GtkWidget *scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_show(scrolledwindow);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolledwindow),
				       GTK_SHADOW_ETCHED_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
		
	gtk_container_add(GTK_CONTAINER(scrolledwindow),treeview);

	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), scrolledwindow, NULL);
}
    
void ListWin::Clear()
{
	gtk_list_store_clear(GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(treeview))));
}

void ListWin::RemoveLast()
{
	//note,this function is not to remove the list row,but the LIST_WIN_ROW_NUM row.	
	gchar lastrow[5];
	sprintf(lastrow,"%d", LIST_WIN_ROW_NUM); // not LIST_WIN_ROW_NUM -1,as Prepend is done first.
	GtkTreePath *path = gtk_tree_path_new_from_string (lastrow);
	GtkTreeIter iter;
	if (gtk_tree_model_get_iter (gtk_tree_view_get_model(GTK_TREE_VIEW(treeview)), &iter, path)) {
		gtk_list_store_remove(GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(treeview))), &iter);
	}
	gtk_tree_path_free(path);
}

void ListWin::UnSelectAll()
{
	gtk_tree_selection_unselect_all(gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview)));
}

void ListWin::ReScroll()
{
	GtkTreePath *path = gtk_tree_path_new_from_string ("0");
	gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(treeview),path,NULL,false,0,0);
	gtk_tree_path_free(path);
	/*if (GTK_WIDGET_REALIZED(treeview))
		gtk_tree_view_scroll_to_point(GTK_TREE_VIEW(treeview),0,0);*/
}

void ListWin::InsertLast(const gchar *word)
{
	GtkListStore *model;
	GtkTreeIter iter;
	model = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(treeview)));
	gtk_list_store_append (model, &iter);
	gtk_list_store_set (model, &iter, 0, word, -1);
}

void ListWin::Prepend(const gchar *word)
{
	GtkListStore *model;
	GtkTreeIter iter;
	model = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(treeview)));
	gtk_list_store_prepend (model, &iter);
	gtk_list_store_set (model, &iter, 0, word, -1);
}

gboolean ListWin::on_button_press(GtkWidget * widget, GdkEventButton * event, ListWin *oListWin)
{
	if (event->type==GDK_2BUTTON_PRESS)
	{
		GtkTreeModel *model;
		GtkTreeIter iter;
		
		GtkTreeSelection *selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (oListWin->treeview));
		if (gtk_tree_selection_get_selected (selection, &model, &iter))
		{
			gchar *word;
			gtk_tree_model_get (model, &iter, 0, &word, -1);
			gpAppFrame->oAppCore.ListClick(word);
			g_free(word);
		}		
		return true;
	}
	else
	{
		return false;
	}
}

void ListWin::on_selection_changed(GtkTreeSelection *selection, ListWin *oListWin)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	
	if (gtk_tree_selection_get_selected (selection, &model, &iter))
	{
		gchar *word;
		gtk_tree_model_get (model, &iter, 0, &word, -1);
		gchar *path_str = gtk_tree_model_get_string_from_iter(model,&iter);
		// the "0" is the first iter's string.when the user select some row while the list have no selection,GTK will select
		// the first row,then select the line that user clicked.so here cache the first line is not bad :_)
		if (!strcmp(path_str,"0"))
			gpAppFrame->oAppCore.SimpleLookupToTextWin(word,gpAppFrame->oAppCore.iCurrentIndex,true); //the first word's index is already cached.
		else
			gpAppFrame->oAppCore.SimpleLookupToTextWin(word,NULL);
		g_free(path_str);
		g_free(word);
	}
}

/**************************************************/
TreeWin::TreeWin()
{
}

void TreeWin::Create(GtkWidget *notebook)
{
	GtkTreeStore *model = gpAppFrame->oAppCore.oTreeDicts.Load();
	treeview = gtk_tree_view_new_with_model (GTK_TREE_MODEL(model));
	gtk_widget_show(treeview);
	g_object_unref (model);
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (treeview), FALSE);
	GtkCellRenderer *renderer = gtk_cell_renderer_text_new ();
	GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes ("word", renderer,
						     "text", 0, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	GtkTreeSelection *selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
	g_signal_connect (G_OBJECT (selection), "changed", G_CALLBACK (on_selection_changed), this);
	
	g_signal_connect (G_OBJECT (treeview), "button_press_event", G_CALLBACK (on_button_press), this);
	GtkWidget *scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_show(scrolledwindow);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolledwindow),
				       GTK_SHADOW_ETCHED_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
		
	gtk_container_add(GTK_CONTAINER(scrolledwindow),treeview);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), scrolledwindow, NULL);
}

gboolean TreeWin::on_button_press(GtkWidget * widget, GdkEventButton * event, TreeWin *oTreeWin)
{
	if (event->type==GDK_2BUTTON_PRESS)
	{
		GtkTreeModel *model;
		GtkTreeIter iter;
		
		GtkTreeSelection *selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (oTreeWin->treeview));
		if (gtk_tree_selection_get_selected (selection, &model, &iter))
		{
			if (gtk_tree_model_iter_has_child(model, &iter)) {
				GtkTreePath *path = gtk_tree_model_get_path(model, &iter);
				if (gtk_tree_view_row_expanded(GTK_TREE_VIEW (oTreeWin->treeview), path))
					gtk_tree_view_collapse_row(GTK_TREE_VIEW (oTreeWin->treeview), path);
				else
					gtk_tree_view_expand_row(GTK_TREE_VIEW (oTreeWin->treeview), path, false);
				gtk_tree_path_free(path);
			}
			else {
				glong offset, size;
				gtk_tree_model_get (model, &iter, 1, &offset, 2, &size, -1);
				if (size != 0) {
					gchar *path_str = gtk_tree_model_get_string_from_iter(model,&iter);
					gint iTreeDict;
					gchar *p = strchr(path_str, ':');
					if (p)
						*p = '\0';
					iTreeDict = atoi(path_str);
					gpAppFrame->oAppCore.ShowTreeDictDataToTextWin(offset, size, iTreeDict);
					g_free(path_str);
				}
			}
		}		
		return true;
	}
	else
	{
		return false;
	}
}

void TreeWin::on_selection_changed(GtkTreeSelection *selection, TreeWin *oTreeWin)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	
	if (gtk_tree_selection_get_selected (selection, &model, &iter))
	{
		glong offset, size;
		gtk_tree_model_get (model, &iter, 1, &offset, 2, &size, -1);
		if (size != 0) {
			gchar *path_str = gtk_tree_model_get_string_from_iter(model,&iter);
			gint iTreeDict;
			gchar *p = strchr(path_str, ':');
			if (p)
				*p = '\0';
			iTreeDict = atoi(path_str);
			gpAppFrame->oAppCore.ShowTreeDictDataToTextWin(offset, size, iTreeDict);
			g_free(path_str);
		}
	}
}


/**************************************************/
IndexWin::IndexWin()
{
}

void IndexWin::Create(GtkWidget *hpaned)
{
	vbox = gtk_vbox_new(false, 3);
	gboolean hide;
#ifdef _WIN32
	rw_cfg_read_boolean (usercfgfile, "preferences/main_window", "hide_list", &hide);
#else
	gpAppFrame->oAppConf.read_bool("/apps/stardict/preferences/main_window/hide_list",&hide,false);
#endif	
	if (!hide)
		gtk_widget_show(vbox);
	
	gtk_paned_pack1(GTK_PANED(hpaned),vbox,false,true);
	notebook = gtk_notebook_new();
	gtk_widget_show(notebook);
	gtk_box_pack_start(GTK_BOX(vbox),notebook, true, true, 0);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(notebook), false);
	oListWin.Create(notebook);
	oTreeWin.Create(notebook);
	GtkWidget *hbox = gtk_hbox_new(true, 3);
	gtk_box_pack_start(GTK_BOX(vbox),hbox, false, false, 0);
	
	GtkWidget *wazard_button = gtk_radio_button_new(NULL);
	GTK_WIDGET_UNSET_FLAGS (wazard_button, GTK_CAN_FOCUS);
	gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(wazard_button), false);
	gtk_box_pack_start (GTK_BOX (hbox), wazard_button, true, true, 0);	
	GtkWidget *hbox1 = gtk_hbox_new(false, 2);
	gtk_container_add (GTK_CONTAINER (wazard_button), hbox1);
	GtkWidget *image = gtk_image_new_from_pixbuf(gpAppFrame->oAppSkin.stardict.index_wazard.p[0]);
	gtk_box_pack_start (GTK_BOX (hbox1), image, FALSE, FALSE, 0);
	GtkWidget *label = gtk_label_new_with_mnemonic(_("_List"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_box_pack_start (GTK_BOX (hbox1), label, FALSE, FALSE, 0);
	gtk_label_set_mnemonic_widget(GTK_LABEL(label), wazard_button);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(wazard_button), true);
	g_signal_connect(G_OBJECT(wazard_button),"toggled", G_CALLBACK(on_wazard_button_toggled), this);

	GtkWidget *appendix_button = gtk_radio_button_new_from_widget(GTK_RADIO_BUTTON(wazard_button));
	GTK_WIDGET_UNSET_FLAGS (appendix_button, GTK_CAN_FOCUS);
	gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(appendix_button), false);
	gtk_box_pack_start (GTK_BOX (hbox), appendix_button, true, true, 0);	
	hbox1 = gtk_hbox_new(false, 2);
	gtk_container_add (GTK_CONTAINER (appendix_button), hbox1);
	image = gtk_image_new_from_pixbuf(gpAppFrame->oAppSkin.stardict.index_appendix.p[0]);
	gtk_box_pack_start (GTK_BOX (hbox1), image, FALSE, FALSE, 0);
	label = gtk_label_new_with_mnemonic(_("_Tree"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_box_pack_start (GTK_BOX (hbox1), label, FALSE, FALSE, 0);
	gtk_label_set_mnemonic_widget(GTK_LABEL(label), appendix_button);		
	g_signal_connect(G_OBJECT(appendix_button),"toggled", G_CALLBACK(on_appendix_button_toggled), this);
	
	gtk_widget_show_all(hbox);
}

void IndexWin::on_wazard_button_toggled(GtkToggleButton *button, IndexWin *oIndexWin)
{
	if (gtk_toggle_button_get_active(button))
		gtk_notebook_set_current_page(GTK_NOTEBOOK(oIndexWin->notebook), 0);
}

void IndexWin::on_appendix_button_toggled(GtkToggleButton *button, IndexWin *oIndexWin)
{
	if (gtk_toggle_button_get_active(button))
		gtk_notebook_set_current_page(GTK_NOTEBOOK(oIndexWin->notebook), 1);
}

/************************************************/
ToolWin::ToolWin()
{
	find_text = NULL;
	search_from_beginning = true;
	search_dialog = NULL;
	search_dialog_entry = NULL;
}

ToolWin::~ToolWin()
{
	g_free(find_text);
}

void ToolWin::Create(GtkWidget *vbox)
{
	GtkWidget *hbox;
	hbox = gtk_hbox_new(false,0);
	gtk_widget_show(hbox);
	gtk_box_pack_start(GTK_BOX(vbox),hbox,false,false,3);
	
	GtkWidget *image;
	ShowListButton=gtk_button_new();
	image = gtk_image_new_from_stock(GTK_STOCK_GOTO_LAST,GTK_ICON_SIZE_SMALL_TOOLBAR);
	gtk_widget_show(image);
	gtk_container_add(GTK_CONTAINER(ShowListButton),image);
	gtk_button_set_relief (GTK_BUTTON (ShowListButton), GTK_RELIEF_NONE);
	GTK_WIDGET_UNSET_FLAGS (ShowListButton, GTK_CAN_FOCUS);
	g_signal_connect(G_OBJECT(ShowListButton),"clicked", G_CALLBACK(ShowListCallback),NULL);
	g_signal_connect(G_OBJECT(ShowListButton),"enter_notify_event", G_CALLBACK(stardict_on_enter_notify), NULL);
	gtk_box_pack_start(GTK_BOX(hbox),ShowListButton,false,false,5);
	gtk_tooltips_set_tip(gpAppFrame->oAppCore.tooltips,ShowListButton,_("Show the word list"),NULL);
	
	HideListButton=gtk_button_new();	
	image = gtk_image_new_from_stock(GTK_STOCK_GOTO_FIRST,GTK_ICON_SIZE_SMALL_TOOLBAR);
	gtk_widget_show(image);
	gtk_container_add(GTK_CONTAINER(HideListButton),image);
	gtk_button_set_relief (GTK_BUTTON (HideListButton), GTK_RELIEF_NONE);
	GTK_WIDGET_UNSET_FLAGS (HideListButton, GTK_CAN_FOCUS);
	g_signal_connect(G_OBJECT(HideListButton),"clicked", G_CALLBACK(HideListCallback),NULL);
	g_signal_connect(G_OBJECT(HideListButton),"enter_notify_event", G_CALLBACK(stardict_on_enter_notify), NULL);
	gtk_box_pack_start(GTK_BOX(hbox),HideListButton,false,false,5);
	gtk_tooltips_set_tip(gpAppFrame->oAppCore.tooltips, HideListButton,_("Hide the word list"),NULL);

	gboolean hide;
#ifdef _WIN32
	rw_cfg_read_boolean (usercfgfile, "preferences/main_window", "hide_list", &hide);
#else
	gpAppFrame->oAppConf.read_bool("/apps/stardict/preferences/main_window/hide_list", &hide, false);
#endif	
	if (hide) {
		gtk_widget_show(ShowListButton);
	}
	else {
		gtk_widget_show(HideListButton);
	}
	
	GtkWidget *button;
	button=gtk_button_new();
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock(GTK_STOCK_COPY,GTK_ICON_SIZE_SMALL_TOOLBAR));
	gtk_widget_show_all(button);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	g_signal_connect(G_OBJECT(button),"clicked", G_CALLBACK(CopyCallback),this);
	gtk_box_pack_start(GTK_BOX(hbox),button,false,false,5);	
	gtk_tooltips_set_tip(gpAppFrame->oAppCore.tooltips,button,_("Copy"),NULL);

	PronounceWordButton=gtk_button_new();
	gtk_container_add(GTK_CONTAINER(PronounceWordButton),gtk_image_new_from_stock(GTK_STOCK_EXECUTE,GTK_ICON_SIZE_SMALL_TOOLBAR));
	gtk_widget_show_all(PronounceWordButton);
	gtk_button_set_relief (GTK_BUTTON (PronounceWordButton), GTK_RELIEF_NONE);
	GTK_WIDGET_UNSET_FLAGS (PronounceWordButton, GTK_CAN_FOCUS);
	g_signal_connect(G_OBJECT(PronounceWordButton),"clicked", G_CALLBACK(PlayCallback),this);
	gtk_box_pack_start(GTK_BOX(hbox),PronounceWordButton,false,false,5);	
	gtk_tooltips_set_tip(gpAppFrame->oAppCore.tooltips,PronounceWordButton,_("Pronounce the word"),NULL);
	
	gtk_widget_set_sensitive(PronounceWordButton, false);
	
	button=gtk_button_new();	
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock(GTK_STOCK_SAVE,GTK_ICON_SIZE_SMALL_TOOLBAR));
	gtk_widget_show_all(button);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	g_signal_connect(G_OBJECT(button),"clicked", G_CALLBACK(SaveCallback),this);
	gtk_box_pack_start(GTK_BOX(hbox),button,false,false,5);	
	gtk_tooltips_set_tip(gpAppFrame->oAppCore.tooltips,button,_("Save to file"),NULL);

	gtk_widget_set_sensitive(button, false);
	
	button=gtk_button_new();	
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock(GTK_STOCK_PRINT,GTK_ICON_SIZE_SMALL_TOOLBAR));
	gtk_widget_show_all(button);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	g_signal_connect(G_OBJECT(button),"clicked", G_CALLBACK(PrintCallback),this);
	gtk_box_pack_start(GTK_BOX(hbox),button,false,false,5);	
	gtk_tooltips_set_tip(gpAppFrame->oAppCore.tooltips,button,_("Print"),NULL);
	
	gtk_widget_set_sensitive(button, false);

	button=gtk_button_new();	
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock(GTK_STOCK_FIND,GTK_ICON_SIZE_SMALL_TOOLBAR));
	gtk_widget_show_all(button);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	g_signal_connect(G_OBJECT(button),"clicked", G_CALLBACK(SearchCallback),this);
	gtk_box_pack_start(GTK_BOX(hbox),button,false,false,5);	
	gtk_tooltips_set_tip(gpAppFrame->oAppCore.tooltips,button,_("Search in the definition (Ctrl+F)"),NULL);
}

void ToolWin::ShowListCallback(GtkWidget *widget, gpointer data)
{
#ifdef _WIN32
	rw_cfg_write_boolean (usercfgfile, "preferences/main_window", "hide_list", false);
	on_conf_main_window_hide_list_changed(false);
#else
	gpAppFrame->oAppConf.write_bool("/apps/stardict/preferences/main_window/hide_list",false);
#endif
}

void ToolWin::HideListCallback(GtkWidget *widget, gpointer data)
{
#ifdef _WIN32
	rw_cfg_write_boolean (usercfgfile, "preferences/main_window", "hide_list", true);
	on_conf_main_window_hide_list_changed(true);
#else
	gpAppFrame->oAppConf.write_bool("/apps/stardict/preferences/main_window/hide_list",true);
#endif
}

void ToolWin::CopyCallback(GtkWidget *widget, ToolWin *oToolWin)
{
	GtkTextIter start, end;
	gtk_text_buffer_get_bounds(gtk_text_view_get_buffer(GTK_TEXT_VIEW(gpAppFrame->oAppCore.oMidWin.oTextWin.textview)),&start,&end);
	gchar *text = gtk_text_buffer_get_text(gtk_text_view_get_buffer(GTK_TEXT_VIEW(gpAppFrame->oAppCore.oMidWin.oTextWin.textview)),&start,&end,FALSE);

	GtkClipboard* clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
	gtk_clipboard_set_text(clipboard,text,-1);
	g_free(text);
}

void ToolWin::PlayCallback(GtkWidget *widget, ToolWin *oToolWin)
{
	gpAppFrame->oReadWord.read(gpAppFrame->oAppCore.oMidWin.oTextWin.pronounceWord.c_str());
}

void ToolWin::SaveCallback(GtkWidget *widget, ToolWin *oToolWin)
{
}

void ToolWin::PrintCallback(GtkWidget *widget, ToolWin *oToolWin)
{
}

void ToolWin::find_word_dialog_cb (GtkDialog *dialog, gint id, ToolWin *oToolWin)
{
     GtkEntry *entry;
     const gchar *str;

     entry = GTK_ENTRY (oToolWin->search_dialog_entry);

     switch (id) {
        case GTK_RESPONSE_DELETE_EVENT:
        case GTK_RESPONSE_CLOSE:
		gtk_widget_destroy (GTK_WIDGET (dialog));
		break;
	case STARDICT_RESPONSE_FIND:
		str = gtk_entry_get_text (entry);

		if (!str[0])
			return;
			
		if (gpAppFrame->oAppCore.oMidWin.oTextWin.Find(str, oToolWin->search_from_beginning))
		{
			oToolWin->search_from_beginning = FALSE;
			
			if (oToolWin->find_text != NULL)
				g_free (oToolWin->find_text);
			
			oToolWin->find_text = g_strdup (str);
		}
		else
			oToolWin->search_from_beginning = TRUE;
		
		break;
     }
}

void ToolWin::do_search()
{
    if (search_dialog == NULL)
    {
    	search_dialog = gtk_dialog_new_with_buttons (_("Find"),
					  GTK_WINDOW (gpAppFrame->oAppCore.window),
					  GTK_DIALOG_DESTROY_WITH_PARENT,
					  GTK_STOCK_CLOSE,
					  GTK_RESPONSE_CLOSE,
					  NULL);
    	GtkWidget *hbox = gtk_hbox_new (FALSE, 8);
	
    	GtkWidget *label = gtk_label_new_with_mnemonic (_("_Search for:"));
    
    	search_dialog_entry = gtk_entry_new ();
   		gtk_entry_set_activates_default (GTK_ENTRY (search_dialog_entry), TRUE);
    
		gtk_label_set_mnemonic_widget (GTK_LABEL (label), search_dialog_entry);

    	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    	gtk_box_pack_start (GTK_BOX (hbox), search_dialog_entry, TRUE, TRUE, 0);

    	gtk_container_set_border_width (GTK_CONTAINER (hbox), 6);
    
    	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (search_dialog)->vbox), hbox);
		
		
		GtkWidget *button = gtk_button_new_from_stock(GTK_STOCK_FIND);
		GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
		gtk_dialog_add_action_widget (GTK_DIALOG (search_dialog), button, STARDICT_RESPONSE_FIND);
    
    	gtk_dialog_set_default_response (GTK_DIALOG (search_dialog), STARDICT_RESPONSE_FIND);
		    
		g_signal_connect (G_OBJECT (search_dialog), "response", G_CALLBACK (find_word_dialog_cb), this);

		g_signal_connect(G_OBJECT (search_dialog), "destroy", G_CALLBACK (gtk_widget_destroyed), &search_dialog);

		gtk_window_set_resizable (GTK_WINDOW (search_dialog), FALSE);

		gtk_widget_show_all (GTK_WIDGET (search_dialog));

		search_from_beginning = TRUE;
		
		GtkTextIter start,end;
		GtkTextBuffer *buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(gpAppFrame->oAppCore.oMidWin.oTextWin.textview));
		if (gtk_text_buffer_get_selection_bounds(buffer, &start, &end)) {
			if (gtk_text_iter_get_offset(&end) - gtk_text_iter_get_offset(&start) < 80) {
				g_free(find_text);
				find_text = gtk_text_buffer_get_text(buffer, &start, &end, false);
			}
		}
    }
    else
		gtk_window_present (GTK_WINDOW (search_dialog));

    if (find_text != NULL)
	    gtk_entry_set_text (GTK_ENTRY (search_dialog_entry), find_text);

	gtk_editable_select_region(GTK_EDITABLE(search_dialog_entry),0,-1);
    gtk_widget_grab_focus (search_dialog_entry);
}

void ToolWin::SearchCallback(GtkWidget *widget, ToolWin *oToolWin)
{    
	oToolWin->do_search();
}


/**********************************************/
TextWin::TextWin()
{
}

TextWin::~TextWin()
{
}

void TextWin::Create(GtkWidget *vbox)
{		
	textview = gtk_text_view_new();
	gtk_widget_show(textview);
	gtk_text_view_set_editable(GTK_TEXT_VIEW(textview),false);
	gtk_text_view_set_cursor_visible(GTK_TEXT_VIEW(textview), false);
	gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(textview), GTK_WRAP_WORD_CHAR);
	gtk_text_view_set_left_margin(GTK_TEXT_VIEW(textview),5);
	gtk_text_view_set_right_margin(GTK_TEXT_VIEW(textview),5);

	g_signal_connect (G_OBJECT (textview), "button_press_event", G_CALLBACK (on_button_press), this);
	g_signal_connect (G_OBJECT (textview), "selection_received", G_CALLBACK (SelectionCallback), this);


	GtkTextBuffer *buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(textview));
		
	gtk_text_buffer_create_tag (buffer, "DictNameTag", "foreground", "blue", NULL);
	gtk_text_buffer_create_tag (buffer, "YinBiaoTag", "foreground", "red", NULL);
	gtk_text_buffer_create_tag (buffer, "WordTag", "weight", PANGO_WEIGHT_BOLD, "scale", PANGO_SCALE_X_LARGE, NULL);
	gtk_text_buffer_create_tag (buffer, "PhoneticTag", "scale", PANGO_SCALE_X_LARGE, NULL);
	//gtk_text_buffer_create_tag(buffer, "WordLink", "foreground", "blue", "underline", PANGO_UNDERLINE_SINGLE, NULL);
	//g_signal_connect(G_OBJECT(texttag), "event", G_CALLBACK(tag_event), strdup(url));


	scrolled_window = gtk_scrolled_window_new(NULL,NULL);
	gtk_widget_show(scrolled_window);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
	//altought textview's set_wrap_mode will cause this can be GTK_POLICY_NEVER,but...there are widgets that may make this broken.
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add(GTK_CONTAINER(scrolled_window),textview);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled_window), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(vbox),scrolled_window,true,true,0);
}

void TextWin::ShowInitFailed()
{
	Show(_("Warning! No dictionary is loaded.\nPlease go to StarDict's website and download some dictionaries:\nhttp://stardict.sourceforge.net"));
}

void TextWin::ShowTips()
{
	query_result = TEXT_WIN_TIPS;
	GtkTextBuffer *buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(textview));
	gtk_text_buffer_set_text(buffer,_(
		"        Welcome to StarDict!\n\n"
		"   Press Ctrl+Q to quit. Press Alt+Z to iconify the window or Alt+X to hide the window.\n"
		"   Press Alt+C or ESC to clear the input entry's text.\n"
		"   Press Space key to move focus to the input entry.\n"
		"   If the query word was not found, you can press Tab key to select the first word in the word list.\n"
		"   After selected some text, clicking the middle mouse button on the main window's Definition area or on the notification area icon will look up that word.\n"
		"   StarDict can match strings against patterns containing '*' (wildcard) and '?' (joker).\n"
		"   Input a word beginning with \'/\' to do a Fuzzy query.\n"
		"   When the floating window reports that a word was not found, double clicking will perform a fuzzy query.\n"
		),-1);
	gtk_adjustment_set_value(gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW(scrolled_window)), 0);
}

void TextWin::ShowInfo()
{
	query_result = TEXT_WIN_INFO;
	GtkTextBuffer *buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(textview));
	gtk_text_buffer_set_text(buffer,_(
		"       Welcome to StarDict\n"
		"StarDict is a Cross-Platform and international dictionary written in Gtk2. "
		"It has powerful features such as \"Glob-style pattern matching,\" \"Scan selected word,\" \"Fuzzy query,\" etc.\n\n"
		"       Here is an introduction to using StarDict:\n\n"
		"       1. Glob-style pattern matching\n"
		"       You can input strings containing \'*\' (wildcard) and \'?\' (joker) as the pattern. "
		"\'*\' matches an arbitrary, possibly empty, string, and \'?\' matches an arbitrary character. "
		"After pressing Enter, the words that match this pattern will be shown in the list.\n"
		"       2. Fuzzy query\n"
		"       When you can't remember how to spell a word exactly, you can try StarDict's Fuzzy query. "
		"It uses \"Levenshtein Edit Distance\" to compute the similarity between two words, and gives the match results which are most "
		"similar to the word that you input. "
		"To create a fuzzy query, just input the word with a beginning \"/\", and then press Enter.\n"
		"       3. Scan selected word\n"
		"       Turn on the check button at the bottom-left corner of the StarDict window to activate this feature.  "
		"When this feature is on, StarDict will automatically look up words, phrases, and Chinese characters in other applications.  "
        "Just highlight a word or phrase with your mouse, and a floating window will pop up showing the definition of the "
        "selected word.\n"
		"       4. Dictionary management\n"
		"       Click the \"Manage dictionaries\" button at the bottom-right corner of the window to access the dictionary management "
		"dialog.  From here, you can disable some dictionaries that you don't need, and set the dictionaries\' quering order.\n"
		"\n\n"
		"       Statement: This program is distributed in the hope that it will be useful, "
 		"but WITHOUT ANY WARRANTY; without the warranty "
 		"that the word spelling, definition, and phonetic information are correct.\n"
		"\n"
		"StarDict 2.4.3 released on 2004.2.19\n"
		"http://stardict.sourceforge.net\n"		
	),-1);

	gtk_adjustment_set_value(gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW(scrolled_window)), 0);
}

void TextWin::Show(const gchar *str)
{
	gtk_text_buffer_set_text (gtk_text_view_get_buffer(GTK_TEXT_VIEW(textview)), str, -1);
	gtk_adjustment_set_value(gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW(scrolled_window)), 0);
}

void TextWin::Show(gchar **Word, gchar **WordData, const gchar * sOriginWord)
{
	gchar *p;
	glong data_size,sec_size;
	GtkTextBuffer *buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(textview));
	gtk_text_buffer_begin_user_action(buffer); // will this speed up the showing?
	
	GtkTextIter start, end;	
	gtk_text_buffer_get_bounds(buffer,&start,&end);
	gtk_text_buffer_delete(buffer,&start,&end);
	gtk_adjustment_set_value(gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW(scrolled_window)), 0);
	
	GtkTextIter iter;
	gtk_text_buffer_get_iter_at_offset (buffer, &iter, 0);
	for (int i=0;i< gpAppFrame->oAppCore.oLibs.total_libs();i++)
	{
		if (Word[i]) {
			gtk_text_buffer_insert_with_tags_by_name(buffer, &iter, "<--- ",-1, "DictNameTag", NULL);
			gtk_text_buffer_insert_with_tags_by_name(buffer, &iter, gpAppFrame->oAppCore.oLibs.GetBookname(i),-1, "DictNameTag", NULL);
			gtk_text_buffer_insert_with_tags_by_name(buffer, &iter, " --->\n",-1, "DictNameTag", NULL);
			gtk_text_buffer_insert_with_tags_by_name(buffer, &iter, Word[i],-1, "WordTag", NULL);	
			gtk_text_buffer_insert (buffer, &iter,"\n", 1);
		}
		if (WordData[i]) {
			memcpy(&data_size,WordData[i],sizeof(glong));
			p=WordData[i]+sizeof(glong);
			while (p - (WordData[i] + sizeof(glong))< data_size)
			{	
				switch (*p)
				{
					case 'm':
					case 'o': //need more work...
						sec_size = strlen(p+sizeof(gchar));
						if (sec_size) {
							gtk_text_buffer_insert (buffer, &iter,p+sizeof(gchar),sec_size);
						}
						sec_size++;
						break;
					case 't':
						sec_size = strlen(p+sizeof(gchar));
						if (sec_size) {
							gtk_text_buffer_insert_with_tags_by_name(buffer, &iter,"[",1,"PhoneticTag",NULL);
							gtk_text_buffer_insert_with_tags_by_name(buffer, &iter,p+sizeof(gchar),sec_size,"PhoneticTag",NULL);
							gtk_text_buffer_insert_with_tags_by_name(buffer, &iter,"]",1,"PhoneticTag",NULL);
						}
						sec_size++;						
						break;
					case 'y':
						sec_size = strlen(p+sizeof(gchar));
						if (sec_size) {
							gtk_text_buffer_insert_with_tags_by_name(buffer, &iter,"[",1,"YinBiaoTag",NULL);
							gtk_text_buffer_insert_with_tags_by_name(buffer, &iter,p+sizeof(gchar),sec_size,"YinBiaoTag",NULL);
							gtk_text_buffer_insert_with_tags_by_name(buffer, &iter,"]",1,"YinBiaoTag",NULL);
						}
						sec_size++;
						break;
					case 'W':
						memcpy(&sec_size,p+sizeof(gchar),sizeof(glong));
						//enbale sound button.
						sec_size+= sizeof(glong);
						break;
					case 'P':
						memcpy(&sec_size,p+sizeof(gchar),sizeof(glong));
						//show this picture.
						sec_size+= sizeof(glong);
						break;
				}
				gtk_text_buffer_insert (buffer, &iter,"\n", 1);
				p = p+sizeof(gchar)+sec_size;
			}
		}		
	}
	gtk_text_buffer_end_user_action(buffer);
}

void TextWin::ShowTreeDictData(const gchar *data)
{
	GtkTextBuffer *buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(textview));
	gtk_text_buffer_begin_user_action(buffer);
	
	GtkTextIter start, end;	
	gtk_text_buffer_get_bounds(buffer,&start,&end);
	gtk_text_buffer_delete(buffer,&start,&end);
	gtk_adjustment_set_value(gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW(scrolled_window)), 0);
	
	GtkTextIter iter;
	gtk_text_buffer_get_iter_at_offset (buffer, &iter, 0);

	const gchar *p;
	glong data_size,sec_size;

	memcpy(&data_size,data,sizeof(glong));
	p=data+sizeof(glong);
	while (p - (data + sizeof(glong))< data_size) {						
		switch (*p) {
			case 'm':
			case 'o': //need more work...
				sec_size = strlen(p+sizeof(gchar));
				if (sec_size)
					gtk_text_buffer_insert (buffer, &iter,p+sizeof(gchar),sec_size);
				sec_size++;
				break;
			case 't':
				sec_size = strlen(p+sizeof(gchar));
				if (sec_size) {
					gtk_text_buffer_insert_with_tags_by_name(buffer, &iter,"[",1,"PhoneticTag",NULL);
					gtk_text_buffer_insert_with_tags_by_name(buffer, &iter,p+sizeof(gchar),sec_size,"PhoneticTag",NULL);
					gtk_text_buffer_insert_with_tags_by_name(buffer, &iter,"]",1,"PhoneticTag",NULL);
				}
				sec_size++;						
				break;
			case 'y':
				sec_size = strlen(p+sizeof(gchar));
				if (sec_size) {
					gtk_text_buffer_insert_with_tags_by_name(buffer, &iter,"[",1,"YinBiaoTag",NULL);
					gtk_text_buffer_insert_with_tags_by_name(buffer, &iter,p+sizeof(gchar),sec_size,"YinBiaoTag",NULL);
					gtk_text_buffer_insert_with_tags_by_name(buffer, &iter,"]",1,"YinBiaoTag",NULL);
				}
				sec_size++;
				break;
			case 'W':
				memcpy(&sec_size,p+sizeof(gchar),sizeof(glong));
				//enbale sound button.
				sec_size+= sizeof(glong);
				break;
			case 'P':
				memcpy(&sec_size,p+sizeof(gchar),sizeof(glong));
				//show this picture.
				sec_size+= sizeof(glong);
				break;
		}
		gtk_text_buffer_insert (buffer, &iter,"\n", 1);
		p = p+sizeof(gchar)+sec_size;
	}
	gtk_text_buffer_end_user_action(buffer);
}

gboolean TextWin::Find (const gchar *text, gboolean start)
{    
    GtkTextBuffer *buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview));

    GtkTextIter iter;    	
    if (start) 
		gtk_text_buffer_get_start_iter (buffer, &iter);
    else {
		GtkTextMark *mark = gtk_text_buffer_get_mark (buffer, "last_search");
    	
    	if (mark)
    	    gtk_text_buffer_get_iter_at_mark (buffer, &iter, mark);
    	else
    	    gtk_text_buffer_get_start_iter (buffer, &iter);
    }
    
	GtkTextIter match_start, match_end;
    if (gtk_text_iter_forward_search (&iter, text,
                                      (GtkTextSearchFlags) (GTK_TEXT_SEARCH_VISIBLE_ONLY | GTK_TEXT_SEARCH_TEXT_ONLY),
                                      &match_start, &match_end,
                                      NULL))
    {
        gtk_text_view_scroll_to_iter (GTK_TEXT_VIEW (textview), &match_start,
        			      0.0, TRUE, 0.5, 0.5);
        gtk_text_buffer_place_cursor (buffer, &match_end);
        gtk_text_buffer_move_mark (buffer,
                               gtk_text_buffer_get_mark (buffer, "selection_bound"),
                               &match_start);
        gtk_text_buffer_create_mark (buffer, "last_search", &match_end, FALSE);

		return TRUE;
    }
    else
    {
		GtkWidget *message_dlg;

		message_dlg = gtk_message_dialog_new (
				GTK_WINDOW (gpAppFrame->oAppCore.oMidWin.oToolWin.search_dialog),
				(GtkDialogFlags) (GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT),
				GTK_MESSAGE_INFO,
				GTK_BUTTONS_OK,
				_("The text \"%s\" was not found."), text);

		gtk_dialog_set_default_response (GTK_DIALOG (message_dlg), GTK_RESPONSE_OK);

		gtk_window_set_resizable (GTK_WINDOW (message_dlg), FALSE);

		gtk_dialog_run (GTK_DIALOG (message_dlg));
  		gtk_widget_destroy (message_dlg);
		return FALSE;
    }
}

gboolean TextWin::on_button_press(GtkWidget * widget, GdkEventButton * event, TextWin *oTextWin)
{
	if (event->button==2)
	{
		gtk_selection_convert (widget, GDK_SELECTION_PRIMARY, gpAppFrame->oAppCore.oSelection.UTF8_STRING_Atom, GDK_CURRENT_TIME);
		return true;
	}
	else
	{
		return false;
	}
}

void TextWin::SelectionCallback(GtkWidget* widget,GtkSelectionData *selection_data, guint time, TextWin *oTextWin)
{
	gchar *result;
	result = (gchar *)gtk_selection_data_get_text (selection_data);
	if (!result) {
		/* If we asked for UTF8 and didn't get it, try compound_text;
		* if we asked for compound_text and didn't get it, try string;
		* If we asked for anything else and didn't get it, give up.
		*/
		if (selection_data->target == gpAppFrame->oAppCore.oSelection.UTF8_STRING_Atom) {
			gtk_selection_convert (widget, GDK_SELECTION_PRIMARY, gpAppFrame->oAppCore.oSelection.COMPOUND_TEXT_Atom, GDK_CURRENT_TIME);
		}
		else if (selection_data->target == gpAppFrame->oAppCore.oSelection.COMPOUND_TEXT_Atom)
		{
			gtk_selection_convert (widget, GDK_SELECTION_PRIMARY, GDK_TARGET_STRING, GDK_CURRENT_TIME);
		}
		return;
    }
	gpAppFrame->oAppCore.Query(result);
	g_free (result);
}

/*********************************************/
MidWin::MidWin()
{
}

void MidWin::Create(GtkWidget *vbox)
{
	hpaned = gtk_hpaned_new();	
	gtk_widget_show(hpaned);
	gtk_box_pack_start(GTK_BOX(vbox),hpaned,true,true,0);

	oIndexWin.Create(hpaned);

	GtkWidget *vbox1 = gtk_vbox_new(false,0);
	gtk_widget_show(vbox1);
	gtk_paned_pack2(GTK_PANED(hpaned),vbox1,true,true);	
	oToolWin.Create(vbox1);
	oTextWin.Create(vbox1);	
	
	gint pos;
#ifdef _WIN32
	rw_cfg_read_int (usercfgfile, "preferences/main_window", "hpaned_pos", &pos);
#else
	gpAppFrame->oAppConf.read_int("/apps/stardict/preferences/main_window/hpaned_pos", &pos, DEFAULT_HPANED_POS);
#endif
	gtk_paned_set_position(GTK_PANED(hpaned),pos);
}


/*********************************************/
BottomWin::BottomWin()
{
	searchwebsite_list = NULL;
	SearchWebsiteMenu = NULL;
}

BottomWin::~BottomWin()
{
	g_slist_foreach (searchwebsite_list, (GFunc)g_free, NULL);
	g_slist_free (searchwebsite_list);
}

void BottomWin::Create(GtkWidget *vbox)
{
#ifdef _WIN32
	rw_cfg_read_strlist (usercfgfile, "preferences/main_window", "search_website_list", &searchwebsite_list);
#else
	gpAppFrame->oAppConf.read_list("/apps/stardict/preferences/main_window/search_website_list", GCONF_VALUE_STRING, &searchwebsite_list);
#endif	
	if (!searchwebsite_list) {
		gchar *default_website = _(""
		"dict.leo.org	http://dict.leo.org	http://dict.leo.org/?search=%s&lang=en\n"
		"H2G2	http://www.h2g2.com	http://www.h2g2.com/Search?searchstring=%s&searchtype=ARTICLE&skip=0&show=20&showapproved=1&shownormal=1&showsubmitted=1\n"
		"WhatIs	http://www.whatis.com	http://www.whatis.com/WhatIs_Search_Results_Exact/1,290214,,00.html?query=%s\n"
		"Altavista	http://www.altavista.com	http://www.altavista.com/cgi-bin/query?q=%s\n"
		"WEB.DE	http://suche.web.de	http://suche.web.de/search/?su=%s\n"
		"WebCrawler	http://www.webcrawler.com	http://www.webcrawler.com/cgi-bin/WebQuery?searchText=%s\n"
		"Google	http://www.google.com	http://www.google.com/search?q=%s\n"
		"Magellan	http://www.mckinley.com	http://www.mckinley.com/extsearch.cgi?query=%s\n"
		"Yahoo	http://search.yahoo.com	http://search.yahoo.com/bin/search?p=%s\n"
		"CMU	http://www.speech.cs.cmu.edu	http://www.speech.cs.cmu.edu/cgi-bin/cmudict?in=%s\n"	
		"");
		gchar *p = default_website;
		gchar *p1;
		GSList *list = NULL;
		while ((p1 = strchr(p, '\n'))!= NULL) {
			list = g_slist_append(list, g_strndup(p, p1-p));
			p= p1+1;
		}
#ifdef _WIN32
		rw_cfg_write_strlist (usercfgfile, "preferences/main_window", "search_website_list", list);
#else
		gpAppFrame->oAppConf.write_list("/apps/stardict/preferences/main_window/search_website_list", GCONF_VALUE_STRING, list);
#endif		
		//as oAppConf.EnableNotify() didn't call now, we change searchwebsite_list manually;
		searchwebsite_list = list;
		//g_slist_foreach (list, (GFunc)g_free, NULL);
		//g_slist_free(list);
	}
	
	GtkWidget *hbox = gtk_hbox_new(false,0);
	gtk_widget_show(hbox);
	gtk_box_pack_start(GTK_BOX(vbox),hbox,false,false,2);
	
	ScanSelectionCheckButton = gtk_check_button_new_with_mnemonic(_("_Scan"));
	gtk_widget_show(ScanSelectionCheckButton);
	GTK_WIDGET_UNSET_FLAGS (ScanSelectionCheckButton, GTK_CAN_FOCUS);
	gboolean scan;
#ifdef _WIN32
	rw_cfg_read_boolean (usercfgfile, "preferences/dictionary", "scan_selection", &scan);
#else
	gpAppFrame->oAppConf.read_bool("/apps/stardict/preferences/dictionary/scan_selection", &scan, true);
#endif
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ScanSelectionCheckButton),scan);
	g_signal_connect(G_OBJECT(ScanSelectionCheckButton),"toggled", G_CALLBACK(ScanCallback),NULL);
	gtk_box_pack_start(GTK_BOX(hbox),ScanSelectionCheckButton,false,false,0);
	gtk_tooltips_set_tip(gpAppFrame->oAppCore.tooltips,ScanSelectionCheckButton,_("Scan the selection"),NULL);

	GtkWidget *button;
	button=gtk_button_new();	
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock(GTK_STOCK_DIALOG_INFO,GTK_ICON_SIZE_SMALL_TOOLBAR));
	gtk_widget_show_all(button);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	g_signal_connect(G_OBJECT(button),"clicked", G_CALLBACK(AboutCallback), NULL);
	g_signal_connect(G_OBJECT(button),"enter_notify_event", G_CALLBACK(stardict_on_enter_notify), NULL);
	gtk_box_pack_start(GTK_BOX(hbox),button,false,false,8);
	gtk_tooltips_set_tip(gpAppFrame->oAppCore.tooltips,button,_("Show info"),NULL);

	button=gtk_button_new();	
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock(GTK_STOCK_QUIT,GTK_ICON_SIZE_SMALL_TOOLBAR));
	gtk_widget_show_all(button);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	g_signal_connect(G_OBJECT(button),"clicked", G_CALLBACK(QuitCallback), NULL);
	g_signal_connect(G_OBJECT(button),"enter_notify_event", G_CALLBACK(stardict_on_enter_notify), NULL);
	gtk_box_pack_start(GTK_BOX(hbox),button,false,false,0);
	gtk_tooltips_set_tip(gpAppFrame->oAppCore.tooltips,button,_("Quit"), NULL);


	// the next buttons will be pack from right to left.
	
	button=gtk_button_new();	
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock(GTK_STOCK_PREFERENCES,GTK_ICON_SIZE_SMALL_TOOLBAR));
	gtk_widget_show_all(button);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	g_signal_connect(G_OBJECT(button),"clicked", G_CALLBACK(PreferenceCallback),this);
	g_signal_connect(G_OBJECT(button),"enter_notify_event", G_CALLBACK(stardict_on_enter_notify), NULL);
	gtk_box_pack_end(GTK_BOX(hbox),button,false,false,0);
	gtk_tooltips_set_tip(gpAppFrame->oAppCore.tooltips,button,_("Preferences"),NULL);

	button=gtk_button_new();	
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock(GTK_STOCK_PROPERTIES,GTK_ICON_SIZE_SMALL_TOOLBAR));
	gtk_widget_show_all(button);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	g_signal_connect(G_OBJECT(button),"clicked", G_CALLBACK(DictManageCallback),this);
	g_signal_connect(G_OBJECT(button),"enter_notify_event", G_CALLBACK(stardict_on_enter_notify), NULL);
	gtk_box_pack_end(GTK_BOX(hbox),button,false,false,0);
	gtk_tooltips_set_tip(gpAppFrame->oAppCore.tooltips,button,_("Manage dictionaries"),NULL);

	button=gtk_button_new();	
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock(GTK_STOCK_INDEX,GTK_ICON_SIZE_SMALL_TOOLBAR));
	gtk_widget_show_all(button);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	g_signal_connect(G_OBJECT(button),"clicked", G_CALLBACK(NewVersionCallback),this);
	g_signal_connect(G_OBJECT(button),"enter_notify_event", G_CALLBACK(stardict_on_enter_notify), NULL);
	gtk_box_pack_end(GTK_BOX(hbox),button,false,false,0);
	gtk_tooltips_set_tip(gpAppFrame->oAppCore.tooltips,button,_("Go to the StarDict website"),NULL);

	button=gtk_button_new();	
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock(GTK_STOCK_JUMP_TO,GTK_ICON_SIZE_SMALL_TOOLBAR));
	gtk_widget_show_all(button);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	g_signal_connect(G_OBJECT(button),"clicked", G_CALLBACK(InternetSearchCallback),this);
	g_signal_connect(G_OBJECT(button),"button_press_event", G_CALLBACK(on_internetsearch_button_press),this);
	g_signal_connect(G_OBJECT(button),"enter_notify_event", G_CALLBACK(stardict_on_enter_notify), NULL);
	gtk_box_pack_end(GTK_BOX(hbox),button,false,false,0);
	gtk_tooltips_set_tip(gpAppFrame->oAppCore.tooltips,button,_("Search an Internet dictionary - Right button: website list"),NULL);
}

void BottomWin::ScanCallback(GtkToggleButton *button, gpointer data)
{
#ifdef _WIN32
	gboolean scan = gtk_toggle_button_get_active(button);
	rw_cfg_write_boolean (usercfgfile, "preferences/dictionary", "scan_selection", scan);
	on_conf_dictionary_scan_selection_changed(scan);	
#else
	gpAppFrame->oAppConf.write_bool("/apps/stardict/preferences/dictionary/scan_selection",gtk_toggle_button_get_active(button));
#endif
}

void BottomWin::AboutCallback(GtkButton *button, gpointer data)
{
	if (gpAppFrame->enable_sound_event) {
#ifdef _WIN32
		gchar *filename = g_build_filename(stardict_data_dir, "sounds", "buttonactive.wav", NULL);
		PlaySound(filename, 0, SND_ASYNC | SND_FILENAME);
		g_free(filename);
#else
		gnome_sound_play(STARDICT_DATA_DIR "/sounds/buttonactive.wav");
#endif
	}
	gpAppFrame->oAppCore.oMidWin.oTextWin.ShowInfo();
}

void BottomWin::QuitCallback(GtkButton *button, gpointer data)
{
	if (gpAppFrame->enable_sound_event) {
#ifdef _WIN32
		gchar *filename = g_build_filename(stardict_data_dir, "sounds", "buttonactive.wav", NULL);
		PlaySound(filename, 0, SND_ASYNC | SND_FILENAME);
		g_free(filename);
#else
		gnome_sound_play(STARDICT_DATA_DIR "/sounds/buttonactive.wav");
#endif
	}
	gpAppFrame->Quit();	
}

gboolean BottomWin::on_internetsearch_button_press(GtkWidget * widget, GdkEventButton * event , BottomWin *oBottomWin)
{
	if (event->button == 3) {
	
		if (!(oBottomWin->searchwebsite_list))
			return true;
	
		if (oBottomWin->SearchWebsiteMenu)
			gtk_widget_destroy(oBottomWin->SearchWebsiteMenu);
			
		oBottomWin->SearchWebsiteMenu = gtk_menu_new();
		
		GtkWidget *menuitem;
		gchar *p;
		GSList *list = oBottomWin->searchwebsite_list;
		while (list) {			
			p = strchr((gchar *)(list->data), '\t');
			if (p) {
				gchar *p1 = strchr(p+1, '\t');
				if (p1 && (strstr(p1+1, "%s"))) {
					*p = '\0';
					menuitem = gtk_image_menu_item_new_with_label((gchar *)(list->data));
					*p = '\t';
					g_signal_connect(G_OBJECT(menuitem), "activate", G_CALLBACK(on_internetsearch_menu_item_activate), list->data);
					gtk_menu_shell_append(GTK_MENU_SHELL(oBottomWin->SearchWebsiteMenu), menuitem);				
				}
			}
			list = g_slist_next(list);
		}

		gtk_widget_show_all(oBottomWin->SearchWebsiteMenu);
		gtk_menu_popup(GTK_MENU(oBottomWin->SearchWebsiteMenu), NULL, NULL, NULL, NULL, event->button, event->time);		
		return true;
	}
	return false;
}

void BottomWin::on_internetsearch_menu_item_activate(GtkMenuItem *menuitem, gchar *website)
{
	gchar *website_link = strchr(website, '\t');
	gchar *website_searchlink = strchr(website_link+1, '\t');
	*website_link = '\0';
	*website_searchlink = '\0';
	const gchar *text;
	text = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(gpAppFrame->oAppCore.oTopWin.WordCombo)->entry));
	if (text[0]) {		
		gchar *url;	
		url = g_strdup_printf(website_searchlink+1,text);
#ifdef _WIN32
		ShellExecute((HWND)(GDK_WINDOW_HWND(gpAppFrame->oAppCore.window->window)), "OPEN", url, NULL, NULL, SW_SHOWNORMAL);
#else
		gnome_url_show(url, NULL);
#endif
		g_free(url);
	}
	else {
#ifdef _WIN32
		ShellExecute((HWND)(GDK_WINDOW_HWND(gpAppFrame->oAppCore.window->window)), "OPEN", website_link+1, NULL, NULL, SW_SHOWNORMAL);
#else
		gnome_url_show(website_link+1, NULL);
#endif
	}
	*website_link = '\t';
	*website_searchlink = '\t';
}

void BottomWin::InternetSearchCallback(GtkButton *button, BottomWin *oBottomWin)
{	
	if (!oBottomWin->searchwebsite_list)
		return;
	gchar *website = (gchar *)(oBottomWin->searchwebsite_list->data);
	gchar *website_link = strchr(website, '\t');
	if (website_link)
		*website_link = '\0';
	else
		return;
	gchar *website_searchlink = strchr(website_link+1, '\t');
	if (website_searchlink && (strstr(website_searchlink+1, "%s")))
		*website_searchlink = '\0';
	else
		return;
	const gchar *text;
	text = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(gpAppFrame->oAppCore.oTopWin.WordCombo)->entry));
	if (text[0]) {		
		gchar *url;	
		url = g_strdup_printf(website_searchlink+1,text);
#ifdef _WIN32
		ShellExecute((HWND)(GDK_WINDOW_HWND(gpAppFrame->oAppCore.window->window)), "OPEN", url, NULL, NULL, SW_SHOWNORMAL);
#else
		gnome_url_show(url, NULL);
#endif
		g_free(url);
	}
	else {		
#ifdef _WIN32
		ShellExecute((HWND)(GDK_WINDOW_HWND(gpAppFrame->oAppCore.window->window)), "OPEN", website_link+1, NULL, NULL, SW_SHOWNORMAL);
#else
		gnome_url_show(website_link+1, NULL);
#endif
	}
	*website_link = '\t';
	*website_searchlink = '\t';
}

void BottomWin::NewVersionCallback(GtkButton *button, BottomWin *oBottomWin)
{
#ifdef _WIN32
	ShellExecute((HWND)(GDK_WINDOW_HWND(gpAppFrame->oAppCore.window->window)), "OPEN", "http://stardict.sourceforge.net", NULL, NULL, SW_SHOWNORMAL);
#else
	gnome_url_show("http://stardict.sourceforge.net", NULL);
#endif
}

void BottomWin::DictManageCallback(GtkButton *button, BottomWin *oBottomWin)
{
	gpAppFrame->oAppCore.PopupDictManageDlg();
}

void BottomWin::PreferenceCallback(GtkButton *button, BottomWin *oBottomWin)
{
	gpAppFrame->oAppCore.PopupPrefsDlg();
}
