//
//      srecord - manipulate eprom load files
//      Copyright (C) 2000, 2002, 2003, 2005-2008 Peter Miller
//
//      This program is free software; you can redistribute it and/or modify
//      it under the terms of the GNU General Public License as published by
//      the Free Software Foundation; either version 3 of the License, or
//      (at your option) any later version.
//
//      This program is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//      GNU General Public License for more details.
//
//      You should have received a copy of the GNU General Public License
//      along with this program. If not, see
//      <http://www.gnu.org/licenses/>.
//

#include <lib/srec/input/file/mos_tech.h>
#include <lib/srec/record.h>


srec_input_file_mos_tech::~srec_input_file_mos_tech()
{
}


srec_input_file_mos_tech::srec_input_file_mos_tech(
        const std::string &a_file_name) :
    srec_input_file(a_file_name),
    garbage_warning(false),
    seen_some_input(false),
    data_record_count(0)
{
}


srec_input::pointer
srec_input_file_mos_tech::create(const std::string &a_file_name)
{
    return pointer(new srec_input_file_mos_tech(a_file_name));
}


bool
srec_input_file_mos_tech::read_inner(srec_record &record)
{
    for (;;)
    {
        int c = get_char();
        if (c < 0)
            return false;
        if (c == 0x11)
        {
            // XOFF in the file also indicates end-of-file
            while (get_char() >= 0)
                ;
            return false;
        }
        if (c == '\0')
            continue;
        if (c == ';')
            break;
        if (c == '\n')
            continue;
        if (!garbage_warning)
        {
            warning("ignoring garbage lines");
            garbage_warning = true;
        }
        for (;;)
        {
            c = get_char();
            if (c < 0)
                return false;
            if (c == '\n')
                break;
        }
    }
    checksum_reset();
    int length = get_byte();
    if (length == 0)
    {
        //
        // This is a data count record,
        // and also the end-of-file record.
        //
        if (peek_char() != '\n' && peek_char() != '\r')
        {
            //
            // Only check the data count if it is present.
            //
            int nrecs = get_word();
            int csumX = checksum_get16();
            int csum = get_word();
            // In the only file example I have, the count is repeated
            // in the checksum, which would you make you think that the
            // address field is added as a 16-bit value, except that
            // only the data count line is wrong.  Sheesh.
            if (use_checksums() && csumX != csum && csum != nrecs)
                fatal_error("checksum mismatch (%04X != %04X)", csumX, csum);
            if (nrecs != data_record_count)
            {
                fatal_error
                (
                    "data record count mismatch (%d != %d)",
                    nrecs,
                    data_record_count
                );
            }
        }
        if (get_char() != '\n')
            fatal_error("end-of-line expected");
        while (get_char() >= 0)
            ;
        return false;
    }

    unsigned long address = get_word();
    unsigned char buffer[256];
    for (int j = 0; j < length; ++j)
        buffer[j] = get_byte();
    int csumX = checksum_get16();
    int csum = get_word();
    if (use_checksums() && csumX != csum)
        fatal_error("checksum mismatch (%04X != %04X)", csumX, csum);
    if (get_char() != '\n')
        fatal_error("end-of-line expected");

    srec_record::type_t type = srec_record::type_data;
    record = srec_record(type, address, buffer, length);
    ++data_record_count;
    return true;
}


bool
srec_input_file_mos_tech::read(srec_record &record)
{
    if (!read_inner(record))
    {
        if (!seen_some_input && garbage_warning)
            fatal_error("file contains no data");
        return false;
    }
    seen_some_input = true;
    return true;
}


const char *
srec_input_file_mos_tech::get_file_format_name()
    const
{
    return "MOS Technology";
}
