/// \file uiutils.cpp

/* Copyright (C) 2007 The SpringLobby Team. All rights reserved. */

#include <wx/colour.h>
#include <wx/image.h>
#include <wx/mstream.h>
#include <wx/bitmap.h>
#include <wx/log.h>
#include <wx/image.h>
#include <wx/clipbrd.h>
#include <wx/cmndata.h>
#include <wx/colordlg.h>
#include <wx/dataobj.h>

#include <cmath>

#include "uiutils.h"
#include "utils/math.h"
#include "utils/conversion.h"
#include "utils/debug.h"
#include "settings++/custom_dialogs.h"
#include "settings.h"

wxString RefineMapname( const wxString& mapname )
{
    wxString ret = mapname;
    ret = ret.BeforeLast( '.' );
    ret.Replace(_T("_"), _T(" ") );
    ret.Replace(_T("-"), _T(" ") );
    return ret;
}


wxString RefineModname( const wxString& modname )
{
    wxString ret = modname;
    ret.Replace(_T("Absolute Annihilation"), _T("AA") );
    ret.Replace(_T("Complete Annihilation"), _T("CA") );
    ret.Replace(_T("Balanced Annihilation"), _T("BA") );
    ret.Replace(_T("Expand and Exterminate"), _T("EE") );
    ret.Replace(_T("War Evolution"), _T("WarEv") );
    ret.Replace(_T("TinyComm"), _T("TC") );
    ret.Replace(_T("BETA"), _T("b") );
    ret.Replace(_T("Public Alpha"), _T("pa") );
    ret.Replace(_T("Public Beta"), _T("pb") );
    ret.Replace(_T("Public"), _T("p") );
    ret.Replace(_T("Alpha"), _T("a") );
    ret.Replace(_T("Beta"), _T("b") );
    return ret;
}


wxString RTFtoText( const wxString& rtfinput )
{
    wxString ret = rtfinput;
    ret = ret.AfterFirst( '{' ).BeforeLast( '}' );

    ret.Replace( _T("\\pard"), _T("") ); // remove a ambiguus char

    ret.Replace( _T("\\par"), _T(" \n") ); // convert the end of lines

    wxString BeforeBrack = ret.BeforeFirst( '{' );
    wxString AfterBrack = ret.AfterLast( '}' );
    ret = BeforeBrack + AfterBrack; // remove everyhting that matches { text }

    wxString out;
    while ( ret.Find('\\') >= 0 ) //remove anything in the form \please\ignore\this
    {
        out += ret.BeforeFirst( '\\' );
        ret = ret.AfterFirst ( '\\' );
        ret = ret.AfterFirst ( ' ' );
    } ;

    return out;
}

bool AreColoursSimilar( const wxColour& col1, const wxColour& col2, int mindiff )
{
    int r,g,b;
    r = col1.Red() - col2.Red();
    g = col1.Green() - col2.Green();
    b = col1.Blue() - col2.Blue();
    r = r>0?r:-r;
    g = g>0?g:-g;
    b = b>0?b:-b;
    int difference = std::min( r, g );
    difference = std::min( difference, b );
    return difference < mindiff;
}


void ColourDelta( int& r, int& g, int& b, const int& delta )
{
    int tmpdelta;
    if ( delta > 0 )
    {
        r += delta;
        tmpdelta = delta;
        if ( r > 255 )
        {
            tmpdelta += r - 255;
            r = 255;
        }
        g += tmpdelta;
        tmpdelta = delta;
        if ( g > 255 )
        {
            tmpdelta += g - 255;
            g = 255;
        }
        b += tmpdelta;
        if ( b > 255 ) b = 255;
    }
    else
    {
        r += delta;
        tmpdelta = -delta;
        if ( r < 0 )
        {
            tmpdelta -= r;
            r = 0;
        }
        g -= tmpdelta;
        tmpdelta = -delta;
        if ( g < 0 )
        {
            tmpdelta -= g;
            g = 0;
        }
        b -= tmpdelta;
        if ( b < 0 ) b = 0;
    }
}

wxColour ColourDelta( const wxColour& colour, const int& delta )
{
    int r = colour.Red();
    int g = colour.Green();
    int b = colour.Blue();
    ColourDelta( r, g, b, delta );
    return wxColour( r, g, b );
}



wxColour GetColorFromFloatStrng( const wxString& color )
{
    wxString c = color;
    float r = 0, g = 0, b = 0;
    r = FromwxString<double>(c.BeforeFirst( ' ' ));
    c = c.AfterFirst( ' ' );
    g = FromwxString<double>( c.BeforeFirst( ' ' ));
    c = c.AfterFirst( ' ' );
    b = FromwxString<double>(c.BeforeFirst( ' ' ));
    r = clamp( r, 0.f, 1.f  );
    g = clamp( g, 0.f, 1.f  );
    b = clamp( b, 0.f, 1.f  );
    return wxColour( (unsigned char)(r*256), (unsigned char)(g*256), (unsigned char)(b*256) );
}

/**
 @brief Blends two images based on alpha channel present in foreground image.
 @param foreground Foreground image, must have an alpha channel
 @param background Background image, may have an alpha channel
 @param blend_alpha Whether the returned image will have an alpha channel.
 @return A copy of the background image with the foreground image blended on
 top of it. The returned image will have an alpha channel iff the background
 image has an alpha channel. In that case the alpha channel is blended
 identical to the red/green/blue channels.
*/
wxImage BlendImage( const wxImage& foreground, const wxImage& background, bool blend_alpha )
{
    if ( ( foreground.GetWidth()  != background.GetWidth() ) || ( background.GetHeight() != foreground.GetHeight() ) )
    {
        wxLogDebugFunc(_T("size mismatch while blending"));
        return background;
    }

    bool zhu = blend_alpha && background.HasAlpha();
    if ( foreground.HasAlpha() )
    {
        wxImage ret( background.GetWidth(), foreground.GetHeight() );
        const unsigned char* background_data = background.GetData();
        const unsigned char* foreground_data = foreground.GetData();
        const unsigned char* background_alpha = NULL;
        const unsigned char* foreground_alpha = foreground.GetAlpha();
        unsigned char* result_data = ret.GetData();
        unsigned char* result_alpha = NULL;
        unsigned int pixel_count = background.GetWidth() * background.GetHeight();

        if ( zhu )
        {
          background_alpha = background.GetAlpha();
          ret.InitAlpha();
          result_alpha = ret.GetAlpha();
        }

        for ( unsigned int i = 0, i_a = 0; i < pixel_count * 3; i+=3,  i_a++ )
        {
            unsigned char fore_alpha = foreground_alpha[i_a] ;
            float back_blend_fac = ( 255 - fore_alpha)/255.0;
            float fore_blend_fac = fore_alpha/255.0 ;

            result_data[i]    = foreground_data[i]   * fore_blend_fac + background_data[i]   * back_blend_fac ;
            result_data[i+1]  = foreground_data[i+1] * fore_blend_fac + background_data[i+1] * back_blend_fac ;
            result_data[i+2]  = foreground_data[i+2] * fore_blend_fac + background_data[i+2] * back_blend_fac ;

            if ( zhu )
            {
              unsigned char back_alpha = background_alpha[i_a] ;
              result_alpha[i_a] = fore_alpha           * fore_blend_fac + back_alpha           * back_blend_fac ;
            }
        }
        return ret;
    }
    wxLogDebugFunc(_T("cannot blend without alpha"));
    return background;
}

wxBitmap charArr2wxBitmap(const unsigned char * arg, int size)
{
    return wxBitmap( charArr2wxImage( arg, size) );
}

//wxBitmap charArr2wxBitmap(const unsigned char * arg, int size)
//{
//    return wxBitmap( charArr2wxImage( arg, size) );
//}

wxImage charArr2wxImage(const unsigned char * arg, int size)
{
    wxMemoryInputStream istream( arg, size );
    return wxImage( istream, wxBITMAP_TYPE_PNG );
}

wxBitmap charArr2wxBitmapWithBlending(const unsigned char * dest, int dest_size, const unsigned char * text, int text_size )
{
    wxImage dest_img( charArr2wxImage( dest, dest_size ) );
    wxImage text_img( charArr2wxImage( text, text_size ) );
    wxImage ret = BlendImage(text_img, dest_img );

    return wxBitmap( ret );
}

wxBitmap BlendBitmaps( const wxBitmap& background, const wxBitmap& overlay, const int /*dim*/ )
{
    wxImage back = background.ConvertToImage();
    wxImage front = overlay.ConvertToImage();
    wxImage ret = BlendImage( front, back );
    return wxBitmap( ret );
}


namespace {
struct Resizer
{
	// Helper class for BorderInvariantResizeImage
	// Author: Tobi Vollebregt

	Resizer( wxImage& result, const wxImage& image, bool alpha )
		: width( result.GetWidth() )
		, height( result.GetHeight() )
		, imwidth( image.GetWidth() )
		, imheight( image.GetHeight() )
		, half_min_width( (std::min(width, imwidth) + 1) / 2 )    // round up to cover middle pixel
		, half_min_height( (std::min(height, imheight) + 1) / 2 ) // if new width/height is uneven.
		, bytes_per_pixel( alpha ? 1 : 3 )
		, image_data( alpha ? image.GetAlpha() : image.GetData() )
		, result_data( alpha ? result.GetAlpha() : result.GetData() )
	{
	}

	void CopyRow( int result_y, int image_y )
	{
		unsigned char* result_row = result_data + bytes_per_pixel * result_y * width;
		const unsigned char* image_row = image_data + bytes_per_pixel * image_y * imwidth;
		const int bytes = bytes_per_pixel * half_min_width;

		memcpy( result_row, image_row, bytes );
		memcpy( result_row + bytes_per_pixel * width - bytes, image_row + bytes_per_pixel * imwidth - bytes, bytes );

		if ( width > imwidth )
		{
			unsigned char* result_pixel = result_row + bytes;
			const unsigned char* image_pixel = image_row + bytes;

			for (int x = half_min_width; x < width - half_min_width; ++x, result_pixel += bytes_per_pixel)
			{
				memcpy( result_pixel, image_pixel, bytes_per_pixel );
			}
		}
	}

	void CopyTopAndBottomRows()
	{
		for (int y = 0; y < half_min_height; ++y)
		{
			CopyRow( y, y );
			CopyRow( height - 1 - y, imheight - 1 - y );
		}
	}

	void CopyCenterRows()
	{
		for (int y = half_min_height; y < height - half_min_height; ++y)
		{
			CopyRow( y, half_min_height - 1 );
		}
	}

	void operator () ()
	{
		CopyTopAndBottomRows();
		CopyCenterRows();
	}

	const int width, height;
	const int imwidth, imheight;
	const int half_min_width, half_min_height;
	const int bytes_per_pixel;
	const unsigned char* const image_data;
	unsigned char* const result_data;
};
}

typedef std::vector<double> huevec;


void hue(huevec& out, int amount, int level)
{
	if (level <= 1) {
		if (out.size() < amount) out.push_back(0.0);
		if (out.size() < amount) out.push_back(0.5);
	}
	else {
		hue(out, amount, level - 1);
		const int lower = out.size();
		hue(out, amount, level - 1);
		const int upper = out.size();
		for (int i = lower; i < upper; ++i)
			out.at(i) += 1.0 / (1 << level);
	}
}

void hue(huevec& out, int amount)
{
	int level = 0;
	while ((1 << level) < amount) ++level;

	out.reserve(amount);
	hue(out, amount, level);
}

std::vector<wxColour>& GetBigFixColoursPalette( int numteams )
{
    static std::vector<wxColour> result;
    wxLogDebugFunc( TowxString(numteams) );
		huevec huevector;
    static int satvalbifurcatepos;
    static std::vector<double> satvalsplittings;
    if ( satvalsplittings.empty() ) // insert ranges to bifurcate
    {
    	satvalsplittings.push_back( 1 );
    	satvalsplittings.push_back( 0 );
    	satvalbifurcatepos = 0;
    }
		hue( huevector, numteams );
    int bisectionlimit = 20;
    for ( int i = result.size(); i < numteams; i++ )
    {
    	double hue = huevector[i];
    	double saturation = 1;
    	double value = 1;
			int switccolors = i % 3; // why only 3 and not all combinations? because it's easy, plus the bisection limit cannot be divided integer by it

			if ( ( i % bisectionlimit ) == 0 )
			{
				satvalbifurcatepos = satvalbifurcatepos % ( satvalsplittings.size() -1 );
				std::vector<double>::iterator toinsert = satvalsplittings.begin() + satvalbifurcatepos + 1;
				satvalsplittings.insert( toinsert, ( satvalsplittings[satvalbifurcatepos] - satvalsplittings[satvalbifurcatepos + 1] ) / 2 + satvalsplittings[satvalbifurcatepos + 1] );
				satvalbifurcatepos += 2;
			}

			if ( switccolors == 1 )
			{
				saturation = satvalsplittings[satvalbifurcatepos -1];
			}
			else if ( switccolors == 2 )
			{
				value = satvalsplittings[satvalbifurcatepos -1];
			}
			hue += 0.17; // use as starting point a zone where color band is narrow so that small variations means high change in visual effect
			if ( hue > 1 ) hue-= 1;
			wxImage::HSVValue hsvcolor( hue, saturation, value );
			wxImage::RGBValue rgbvalue = wxImage::HSVtoRGB( hsvcolor );
			wxColour col( rgbvalue.red, rgbvalue.green, rgbvalue.blue );
			result.push_back( col );
    }
    return result;
}


wxImage BorderInvariantResizeImage(  const wxImage& image, int width, int height )
{
	if ( !image.IsOk() || (width == image.GetWidth() && height == image.GetHeight()) )
		return image;

	wxImage ret( width, height );
	Resizer data_resize( ret, image, false );
	data_resize();

	if ( image.HasAlpha() )
	{
		ret.InitAlpha();
		Resizer alpha_resize( ret, image, true );
		alpha_resize();
	}

	return ret;
}


wxColour GetColourFromUser(wxWindow *parent, const wxColour& colInit, const wxString& caption, const wxString& palette)
{
    wxColourData data;
    data = sett().GetCustomColors( palette );
    data.SetChooseFull(true);
    if ( colInit.Ok() )
    {
        data.SetColour((wxColour &)colInit); // const_cast
    }

    wxColour colRet;
    wxColourDialog dialog(parent, &data);
    if (!caption.empty())
        dialog.SetTitle(caption);
    if ( dialog.ShowModal() == wxID_OK )
    {
        colRet = dialog.GetColourData().GetColour();
    }
    //else: leave it invalid
    sett().SaveCustomColors( dialog.GetColourData(), palette );

    return colRet;
}

wxImage ReplaceChannelStatusColour( wxBitmap img, const wxColour& colour )
{
  wxImage ret = img.ConvertToImage();
  wxImage::HSVValue origcolour = wxImage::RGBtoHSV( wxImage::RGBValue::RGBValue( colour.Red(), colour.Green(), colour.Blue() ) );

  double bright = origcolour.value - 0.1*origcolour.value;
  bright = clamp( bright, 0.0, 1.0 );
  wxImage::HSVValue hsvdarker1( origcolour.hue, origcolour.saturation, bright );
  bright = origcolour.value - 0.5*origcolour.value;
  bright = clamp( bright, 0.0, 1.0 );
  wxImage::HSVValue hsvdarker2( origcolour.hue, origcolour.saturation, bright );

  wxImage::RGBValue rgbdarker1 = wxImage::HSVtoRGB( hsvdarker1 );
  wxImage::RGBValue rgbdarker2 = wxImage::HSVtoRGB( hsvdarker2 );


  ret.Replace( 164, 147, 0, rgbdarker2.red, rgbdarker2.green, rgbdarker2.blue );

  ret.Replace( 255, 228, 0, rgbdarker1.red, rgbdarker1.green, rgbdarker1.blue );

  ret.Replace( 255, 253, 234, colour.Red(), colour.Green(), colour.Blue() );

  return ret;

}

wxSize MakeFit(const wxSize &original, const wxSize &bounds)
{
  if( ( bounds.GetWidth() <= 0 ) || ( bounds.GetHeight() <= 0 ) ) return wxSize(0,0);
  int sizex = ( original.GetWidth() * bounds.GetHeight() ) / original.GetHeight();
  if( sizex <= bounds.GetWidth() )
  {
    return wxSize( sizex, bounds.GetHeight() );
  }
  else
  {
    int sizey = ( original.GetHeight() * bounds.GetWidth() ) / original.GetWidth();
    return wxSize( bounds.GetWidth(), sizey );
  }
}

void CopyToClipboard( const wxString& text )
{
    if ( wxTheClipboard->Open() ) {
        // This data objects are held by the clipboard,
        // so do not delete them in the app.
        wxTheClipboard->SetData( new wxTextDataObject( text ) );
        wxTheClipboard->Close();
    }
}

