#ifdef __GNUC__
#  pragma implementation "graph2D.H"
#endif

#include "graph2D.H"
#include <math.h>
#include <stdio.h>
#include <iostream.h>

const int max_pathlen = 100;	// do a stroke() after every 100 points to avoid PostScript interpreter memory overflow

greal selected_xticks[max_selected_xticks];
greal selected_xticks_uvalues[max_selected_xticks];
int N_selected_xticks = 0;
static int N_ticks = 0;

void Tgraph2D::rect(greal x1, greal y1, greal x2, greal y2)
{
	begin_line(x1,y1);
	vertex(x2,y1);
	vertex(x2,y2);
	vertex(x1,y2);
	vertex(x1,y1);
	end_line();
}

void Tgraph2D::curve(
	int n, const real x[], const real y[],
	greal x1, greal y1, greal x2, greal y2,			// rect where to draw
	greal xmin, greal xmax, greal ymin, greal ymax,	// wanted min,max of x,y
	int markertype, greal markersize
	)
{
//	cerr << "curve(xmin=" << xmin << ",xmax=" << xmax << ",ymin=" << ymin << ",ymax=" << ymax << ")\n";
	int i,cnt;
	if (n <= 0) return;
	// xplotted = x1 + (x2-x1)*(x-xmin)/(xmax-xmin) ==> xplotted = x1 - xmin*(x2-x1)/(xmax-xmin) + ((x2-x1)/(xmax-xmin))*x
	// yplotted = y1 + (y2-y1)*(y-ymin)/(ymax-ymin) ==> yplotted = y1 - ymin*(y2-y1)/(ymax-ymin) + ((y2-y1)/(ymax-ymin))*y
	// xplotted == ax + bx*x
	// yplotted == ay + by*y
	const real ax = x1 - xmin*(x2-x1)/(xmax-xmin);
	const real ay = y1 - ymin*(y2-y1)/(ymax-ymin);
	const real bx = (x2-x1)/(xmax-xmin);
	const real by = (y2-y1)/(ymax-ymin);
	if (getlinetype() != LINE_NONE) {
		i = 0;
		cnt = 0;
		while (i < n) {
			int j;
			for (j=i; (!finite(x[j]) || !finite(y[j])) && j<n; j++) /*comment(ax+bx*x[j], ay+by*y[j])*/;
			begin_line(ax+bx*x[j], ay+by*y[j]);
//			cerr << "- begin_line j=" << j << ",x=" << x[j] << ",y=" << y[j] << ", ay=" << ay << ",by=" << by << "\n";
			for (j++; j<n && finite(x[j]) && finite(y[j]); j++) {
				vertex(ax+bx*x[j], ay+by*y[j]);
//				cerr << "- vertex j=" << j << ",x=" << x[j] << ",y=" << y[j] << "\n";
				if (cnt++ % max_pathlen == max_pathlen-1) {end_line(); begin_line(ax+bx*x[j], ay+by*y[j]);}
			}
			end_line();
			i = j;
		}
	}
	if (markertype == 0 || markertype == 1 && getlinetype() == LINE_NONE) return;
	gsave();
	setlinetype(LINE_SOLID);
	setlinewidth(1);
	for (i=0; i<n; i++) if (finite(x[i]) && finite(y[i])) mark(ax+bx*x[i],ay+by*y[i],markertype,markersize);
	grestore();
}

void Tgraph2D::filledcurve(
	int n, const real x[], const real y[],
	greal x1, greal y1, greal x2, greal y2,			// rect where to draw
	greal xmin, greal xmax, greal ymin, greal ymax	// wanted min,max of x,y
	)
{
	int i,cnt;
	if (n <= 0) return;
	const real ax = x1 - xmin*(x2-x1)/(xmax-xmin);
	const real ay = y1 - ymin*(y2-y1)/(ymax-ymin);
	const real bx = (x2-x1)/(xmax-xmin);
	const real by = (y2-y1)/(ymax-ymin);
	i = 0;
	cnt = 0;
	const int max_pathlen_fill = polygons_always_convex() ? 1 : max_pathlen;
	while (i < n) {
		int j;
		for (j=i; (!finite(x[j]) || !finite(y[j])) && j<n; j++) /*comment(ax+bx*x[j], ay+by*y[j])*/;
		begin_fill(ax+bx*x[j], ay);
		vertex(ax+bx*x[j], ay+by*y[j]);
		for (j++; j<n && finite(x[j]) && finite(y[j]); j++) {
			vertex(ax+bx*x[j], ay+by*y[j]);
			if (cnt++ % max_pathlen_fill == max_pathlen_fill-1) {
				vertex(ax+bx*x[j],ay);
				end_fill();
				begin_fill(ax+bx*x[j], ay);
				vertex(ax+bx*x[j], ay+by*y[j]);
			}
		}
		vertex(ax+bx*x[j-1],ay);
		end_fill();
		i = j;
	}
}

static const int RainbowPalette[3][Ncolors] = {
	{126, 124, 120, 115, 111, 106, 102, 97, 93, 88, 84, 79, 75, 70, 66, 61,
	 57, 52, 48, 43, 39, 34, 30, 25, 21, 16, 12, 7, 3, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 5,
	 10, 14, 19, 23, 28, 32, 37, 41, 46, 50, 55, 59, 64, 68, 73, 77,
	 82, 86, 91, 95, 100, 104, 109, 113, 118, 123, 126, 132, 135, 141, 144, 150,
	 153, 159, 162, 168, 171, 177, 180, 186, 189, 195, 198, 204, 207, 213, 216, 222,
	 225, 231, 234, 240, 243, 249, 252, 255, 254, 255, 254, 255, 254, 255, 254, 255,
	 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255,
	 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255,
	 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 255, 255},
	{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 6, 11,
	 15, 20, 24, 29, 33, 38, 42, 47, 51, 56, 60, 65, 69, 74, 78, 83,
	 87, 92, 96, 101, 105, 110, 114, 119, 122, 128, 131, 137, 140, 146, 149, 155,
	 158, 164, 167, 173, 176, 182, 185, 191, 194, 200, 203, 209, 212, 218, 221, 227,
	 230, 236, 240, 245, 249, 254, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255,
	 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255,
	 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255,
	 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255,
	 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255,
	 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255,
	 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255,
	 254, 255, 254, 255, 254, 255, 254, 252, 247, 243, 238, 234, 229, 225, 220, 216,
	 211, 207, 202, 198, 193, 189, 184, 180, 175, 171, 166, 162, 157, 153, 148, 144,
	 139, 135, 131, 126, 122, 117, 113, 108, 104, 99, 95, 90, 86, 81, 77, 72,
	 68, 63, 59, 54, 50, 45, 41, 36, 32, 27, 23, 18, 14, 9, 5, 0},
	{255, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255,
	 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255,
	 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255,
	 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255,
	 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255, 254, 255,
	 254, 255, 254, 255, 254, 255, 250, 247, 241, 238, 232, 229, 223, 220, 214, 211,
	 205, 202, 196, 193, 187, 184, 178, 175, 169, 166, 160, 157, 151, 148, 142, 139,
	 133, 130, 125, 121, 116, 112, 107, 103, 98, 94, 89, 85, 80, 76, 71, 67,
	 62, 58, 53, 49, 44, 40, 35, 31, 26, 22, 17, 13, 8, 4, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}};

void Tgraph2D::setpalette(TPaletteType p)
{
	switch (p) {
	case PALETTE_RAINBOW:
		setpalette(RainbowPalette);
		break;
	case PALETTE_GRAYSCALE:
	{
		int pal[3][Ncolors];
		int i;
		for (i=0; i<Ncolors; i++) pal[0][i] = pal[1][i] = pal[2][i] = i;
		setpalette(pal);
	}
	break;
	}
}

void Tgraph2D::pcolor(int nx, const real x[],		// x-vector, length nx
					  int ny, const real y[],		// y-vector, length ny
					  const real z[],				// z-matrix, size nx x ny
					  greal zmin, greal zmax,		// found (or set) min/max of z-matrix
					  greal x1, greal y1, greal x2, greal y2,			// rect where to draw
					  greal xmin, greal xmax, greal ymin, greal ymax,	// wanted min,max of x,y
					  bool interp, bool ghosts_only
	)
{
	if (nx <= 1 || ny <= 1) return;
	const real ax = x1 - xmin*(x2-x1)/(xmax-xmin);
	const real ay = y1 - ymin*(y2-y1)/(ymax-ymin);
	const real bx = (x2-x1)/(xmax-xmin);
	const real by = (y2-y1)/(ymax-ymin);
	int i,j;
	if (zmax <= zmin) zmax = zmin + 1;
	// c = (z-zmin)/(zmax-zmin) ==> c = az + bz*z, az=-zmin/(zmax-zmin), bz=1/(zmax-zmin)
	greal dxleft,dxright,dyleft,dyright;
	const real az = -zmin/(zmax-zmin);
	const real bz = 1.0/(zmax-zmin);
	greal rgb[3];
	const bool grayscale = false;
	const real epsilon_x = 1e-5*(xmax-xmin);
	const real epsilon_y = 1e-5*(ymax-ymin);
	if (interp) {
		int *colmatrix = new int [nx*ny];
		for (i=0; i<nx*ny; i++)
			colmatrix[i] = finite(z[i]) ? limit(int((Ncolors - 1e-3)*(az+bz*z[i])),0,Ncolors-1) : -1;
		for (j=0; j<ny-1; j++) for (i=0; i<nx-1; i++) {
			int ninfinite = 0;
			if (colmatrix[i*ny+j] < 0) ninfinite++;
			if (colmatrix[(i+1)*ny+j] < 0) ninfinite++;
			if (colmatrix[(i+1)*ny+j+1] < 0) ninfinite++;
			if (colmatrix[i*ny+j+1] < 0) ninfinite++;
			if (ninfinite > 1) continue;
			if (x[i+1] == x[i] || y[j+1]==y[j]) continue;
			fillrrect_interp(ax+bx*x[i],ay+by*y[j],bx*(x[i+1]-x[i]),by*(y[j+1]-y[j]),
							 colmatrix[i*ny+j],colmatrix[(i+1)*ny+j],colmatrix[(i+1)*ny+j+1],colmatrix[i*ny+j+1]);
		}
		delete [] colmatrix;
	} else {
		// flat shading, no interpolation.
		greal *xleft = new greal [nx];
		greal *xright = new greal [nx];
		greal *yleft = new greal [ny];
		greal *yright = new greal [ny];
		for (i=0; i<nx; i++) {
			const greal xmid = ax+bx*x[i];
			if (i == 0) {
				dxleft = dxright = x[1] - x[0];
			} else if (i == nx-1) {
				dxleft = dxright = x[nx-1] - x[nx-2];
			} else {
				dxleft = x[i] - x[i-1];
				dxright = x[i+1] - x[i];
			}
			dxleft*= bx; dxright*= bx;
			xleft[i] = xmid - 0.5*dxleft;
			xright[i] = xmid + 0.5*dxright;
		}
		for (j=0; j<ny; j++) {
			const greal ymid = ay + by*y[j];
			if (j == 0) {
				dyleft = dyright = y[1] - y[0];
			} else if (j == ny-1) {
				dyleft = dyright = y[ny-1] - y[ny-2];
			} else {
				dyleft = y[j] - y[j-1];
				dyright = y[j+1] - y[j];
			}
			dyleft*= by; dyright*= by;
			yleft[j] = ymid - 0.5*dyleft;
			yright[j] = ymid + 0.5*dyright;
		}
		if (ghosts_only) {
			int *colmatrix = new int [nx*ny];
			for (i=0; i<nx*ny; i++)
				colmatrix[i] = finite(z[i]) ? limit(int((Ncolors - 1e-3)*(az+bz*z[i])),0,Ncolors-1) : -1;
			for (j=0; j<ny; j++) for (i=0; i<nx; i++) {
				if (!(j==0 || j==ny-1 || i==0 || i==nx-1)) continue;
				if (colmatrix[i*ny+j] < 0) continue;
				setcolor(colmatrix[i*ny+j]);
				fillrrect(xleft[i],yleft[j],xright[i]-xleft[i],yright[j]-yleft[j]);
			}
			delete [] colmatrix;
		} else {
			// Optimize by merging same-color rects together in x
			int *blocklen = new int [nx];
			int *blockstart = new int [nx];
			int *colors = new int [nx];
			for (j=0; j<ny; j++) {
				for (i=0; i<nx; i++)
					colors[i] = finite(z[i*ny+j]) ? limit(int((Ncolors - 1e-3)*(az+bz*z[i*ny+j])),0,Ncolors-1) : -1;
				int prevcolor = colors[0], nblocks=1;
				blocklen[0] = 1; blockstart[0] = 0;
				for (i=1; i<nx; i++)
					if (colors[i] == prevcolor) {
						blocklen[nblocks-1]++;
					} else {
						blocklen[nblocks] = 1;
						blockstart[nblocks] = i;
						nblocks++;
						prevcolor = colors[i];
					}
				for (i=0; i<nblocks; i++) {
					const int c = colors[blockstart[i]];
					if (c < 0) continue;	// NaN
					if (grayscale) {
						rgb[0] = rgb[1] = rgb[2] = c/greal(Ncolors-1);
						setcolor(rgb);
					} else {
						setcolor(c);
					}
					const greal dx = xright[blockstart[i]+blocklen[i]-1]-xleft[blockstart[i]];
					const greal dy = yright[j]-yleft[j];
					if (dx==0 || dy==0) continue;
					fillrrect(xleft[blockstart[i]]-epsilon_x, yleft[j]-epsilon_y,
							  dx+2*epsilon_x,dy+2*epsilon_y);
				}
//				cerr << " " << nblocks << "/" << nx;
			}
			delete [] colors;
			delete [] blockstart;
			delete [] blocklen;
		}
		delete [] yright;
		delete [] yleft;
		delete [] xright;
		delete [] xleft;
	}
}

int Tgraph2D::DrawAxis1(greal x1, greal y1,
						greal x2, greal y2,
						greal dx, greal dy,
						bool logaxis,
						real u1, real u2,
						int level,		// 1 for main ticks, 2 for finer ticks
						bool labels, bool houraxis, bool actually_draw)
{
	int nlabels = 0;
	if (u2 < u1) {real tmp=u1; u1=u2; u2=tmp;}
	if (u1 == u2 || (u2-u1 < 1e-10*fabs(u2))) u2 = u1 + (fabs(u1) > 1e-20 ? fabs(u1) : 1e-20);
	const double invlog10 = 1.0/log(10.0);
	if (actually_draw) line(x1,y1,x2,y2);
	const bool drawing_log_minor = (logaxis && level == 2);
	double du;
	if (drawing_log_minor)
		du = 0.1;
	else if (houraxis) {
#if 0
		du = pow(6.0,floor(log(u2-u1)/log(6.0)-0.9999999*(1-1)));
		if (!logaxis && level==1 && (u2-u1)/du <= 3.5) du*= 0.5;
#endif
		du = pow(60.0,floor(log(u2-u1)/log(60.0)-0.9999999*(level-1)));
		const double X = (u2-u1)/du;
		const double pw = pow(6.0,level-1);
		if (X < 3.5*pw) {
			if (X < 1.5*pw)
				du/= 6;
			else
				du*= 0.5;
		} else if (X > 10*pw) {
			if (X > 100*pw)
				du*= 60;
			else
				du*= 10;
		}
#if 0
		if ((u2-u1)/du <= 3.5) du*= 0.5;
		if ((u2-u1)/du <= 3.5) du/= 3;
		if ((u2-u1)/du <= 3.5) du*= 0.1;
#endif
#if 0
		if (level >= 2) {
			if (du>=1) {
				du = (u2-u1 > 6) ? 1 : 1.0/6.0;
			} else if (du >=1.0/60.0) {
				du = 1.0/60.0;
			} else {
				du = 1.0/360.0;
			}
		}
#endif
//		cerr << "houraxis: level=" << level << ", du=" << du << "\n";
	} else {
		du = pow(10.0,floor(log(u2-u1)*invlog10-0.9999999*(level-1)));
		if (!logaxis && level==1 && (u2-u1)/du <= 3.5) du*= 0.5;
	}
	double u;
	const double C = 1/(u2-u1);
	const double umin = u1 - fmod(u1,du);
	if (drawing_log_minor) {
		// ensure the modified tickmarks do not exceed the original bound
		const greal uint = floor(u2);
		const greal ufrac = u2 - uint;
		u2 = uint + (pow(10.0,ufrac) - 1)*0.1;
	}
	N_ticks = 0;
	for (u=umin; u<=u2; u+= du) if (u1 <= u && u <= u2) {
		greal uu = u;
		greal ratio = 1;
		if (drawing_log_minor) {
			const greal uint = floor(u);
			greal ufrac = u - uint;
			if (ufrac < 0.1) ufrac = 0.1;
			if (ufrac > 0.9) ufrac = 0.9;
			if (fabs(ufrac - 0.4) < 0.05) ratio = 1.7;		// slightly longer tick for "0.5"
			ufrac = log(1 + 10*ufrac)*invlog10;
			uu = uint + ufrac;
		}
		const greal t = (uu-u1)*C;
		const greal x = x1 + t*(x2-x1);
		const greal y = y1 + t*(y2-y1);
		if (actually_draw) line(x,y,x+ratio*dx,y+ratio*dy);
		N_ticks++;
	}
	if (labels && !drawing_log_minor) {
		const double plusratio = 1.3;
		const double minusratio = 1.3;
		const bool record_to_array = (level == 1);
		if (record_to_array) N_selected_xticks = 0;
		for (u=umin; u<=u2; u+= du) if (u1 <= u && u <= u2) {
			const greal t = (u-u1)*C;
			greal ratio;
			int Xoffset,Yoffset;
			if (fabs(dx) > fabs(dy)) {
				ratio = (dx > 0) ? plusratio : minusratio;
				Xoffset = (dx > 0) ? 1 : -1;
				Yoffset = 0;
			} else {
				ratio = (dy > 0) ? plusratio : minusratio;
				Xoffset = 0;
				Yoffset = (dy > 0) ? 1 : -1;
			}
			const greal x = x1 + t*(x2-x1) + ratio*dx;
			const greal y = y1 + t*(y2-y1) + ratio*dy;
			if (record_to_array && N_selected_xticks < max_selected_xticks) {
				selected_xticks[N_selected_xticks] = x;
				selected_xticks_uvalues[N_selected_xticks++] = u;
//				cerr << "selected_xticks[" << N_selected_xticks-1 << "] = " << x << "\n";
			}
			static char lab[80];
			if (logaxis) {
				if (-3 < u && u < 3) {
					sprintf(lab,"%1.5g",pow(10.0,u));
					if (actually_draw) text(x,y,lab,Xoffset,Yoffset);
				} else {
					sprintf(lab,"%1.5g",u);
					if (actually_draw)
						text_exponent(x,y,"10",lab,Xoffset,Yoffset);
				}
			} else {
				if (fabs(u) < 1e-7*(fabs(u1) > fabs(u2) ? fabs(u1) : fabs(u2))) {
					lab[0] = '0';
					lab[1] = '\0';
				} else {
					if (houraxis) {
						const bool drawseconds = fabs(u2-u1) < 20.0/60.0;
						int hh = int(floor(u));
						double mm = (u-hh)*60.0;
						if (drawseconds) {
							int iss = int(floor((mm - floor(mm))*60.0+0.5));
							int imm = int(floor(mm));
							if (iss == 60) {iss = 0; imm++;}
							if (imm == 60) {imm = 0; hh++;}
							sprintf(lab,"%.2d:%.2d:%.2d",hh,imm,iss);
						} else {
							if (int(mm+0.5)==60) {hh++; mm=0;}
							sprintf(lab,"%.2d:%.2d",hh,int(mm+0.5));
						}
					} else
						sprintf(lab,"%1.5g",u);
				}
				if (actually_draw) {
					text(x,y,lab,Xoffset,Yoffset);
				}
			}
			nlabels++;
		}
	}
	return nlabels;
}



void Tgraph2D::axis(greal x1, greal y1,
					greal x2, greal y2,
					greal dx, greal dy,
					bool logaxis,
					real u1, real u2, bool include_label, bool houraxis, TMinorTicks minorticks)
{
//	cerr << "axis(" << x1 << "," << y1 << "," << x2 << "," << y2 << "," << dx << "," << dy << "," << u1 << "," << u2 << "," << include_label << ")\n";
	const double ratio = 0.5;
	bool include_finer = false;
	if (include_label) {
		// Don't draw but count how many labels would become if drawn
		const int nlabels = DrawAxis1(x1,y1,x2,y2,dx,dy,logaxis,u1,u2,1,include_label,houraxis,false);
		if (nlabels < 2) {
			// Draw main ticks without label
			DrawAxis1(x1,y1,x2,y2,dx,dy,logaxis,u1,u2,1,false,houraxis,true);
			// Draw finer axis with label
			DrawAxis1(x1,y1,x2,y2,ratio*dx,ratio*dy,logaxis,u1,u2,2,true,houraxis,true);
		} else {
			// Draw main ticks with label
			DrawAxis1(x1,y1,x2,y2,dx,dy,logaxis,u1,u2,1,true,houraxis,true);
			// Draw finer axis without label
			switch (minorticks) {
			case MINORTICKS_FALSE: include_finer = false; break;
			case MINORTICKS_TRUE: include_finer = true; break;
			case MINORTICKS_AUTO: include_finer = logaxis || sqrt((x2-x1)*(x2-x1)+(y2-y1)*(y2-y1))/(N_ticks*10) > 2;
			}
			if (include_finer) DrawAxis1(x1,y1,x2,y2,ratio*dx,ratio*dy,logaxis,u1,u2,2,false,houraxis,true);
		}
	} else {
		// Just draw main and finer axes
		DrawAxis1(x1,y1,x2,y2,dx,dy,logaxis,u1,u2,1,false,houraxis,true);
		switch (minorticks) {
		case MINORTICKS_FALSE: include_finer = false; break;
		case MINORTICKS_TRUE: include_finer = true; break;
		case MINORTICKS_AUTO: include_finer = logaxis || sqrt((x2-x1)*(x2-x1)+(y2-y1)*(y2-y1))/(N_ticks*10) > 2;
		}
		if (include_finer) DrawAxis1(x1,y1,x2,y2,ratio*dx,ratio*dy,logaxis,u1,u2,2,false,houraxis,true);
	}
}

void Tgraph2D::mark(greal x, greal y, int type, greal size)
{
	switch (type) {
	case 0:
		return;		// do nothing
	case 1:			// single point
		point(x,y);
		break;
	case 2:			// plus sign
	{
		const greal dx = 0.55*size;
		line(x-dx,y,x+dx,y);
		line(x,y-dx,x,y+dx);
	}
	break;
	case 3:			// cross
	{
		const greal dx = 0.353553*1.1*size;
		line(x-dx,y-dx,x+dx,y+dx);
		line(x+dx,y-dx,x-dx,y+dx);
	}
	break;
	case 4:			// framed box
	{
		const greal dx = 0.4*size;
		rrect(x-dx,y-dx,2*dx,2*dx);
	}
	break;
	case 5:			// filled box
	{
		const greal dx = 0.4*size;
		fillrrect(x-dx,y-dx,2*dx,2*dx);
	}
	break;
	case 6:			// framed 45 deg tilted box
	{
		const greal dx = 0.4*1.41421*size;
		begin_line(x-dx,y);
		vertex(x,y-dx);
		vertex(x+dx,y);
		vertex(x,y+dx);
		vertex(x-dx,y);
		end_line();
	}
	break;
	case 7:			// filled 45 tilted box
	{
		const greal dx = 0.4*1.41421*size;
		begin_fill(x-dx,y);
		vertex(x,y-dx);
		vertex(x+dx,y);
		vertex(x,y+dx);
		vertex(x-dx,y);
		end_fill();
	}
	break;
	case 8:			// framed triangle
	{
		const greal r = 0.5*size;
		const greal sqrt3 = sqrt(3.0);
		begin_line(x,y+r);
		vertex(x-2*r/sqrt3,y-r/sqrt3);
		vertex(x+2*r/sqrt3,y-r/sqrt3);
		vertex(x,y+r);
		end_line();
	}
	break;
	case 9:			// filled triangle
	{
		const greal r = 0.5*size;
		const greal sqrt3 = sqrt(3.0);
		begin_fill(x,y+r);
		vertex(x-2*r/sqrt3,y-r/sqrt3);
		vertex(x+2*r/sqrt3,y-r/sqrt3);
		vertex(x,y+r);
		end_fill();
	}
	break;
	case 10:			// framed inverted triangle
	{
		const greal r = 0.5*size;
		const greal sqrt3 = sqrt(3.0);
		begin_line(x,y-r);
		vertex(x-2*r/sqrt3,y+r/sqrt3);
		vertex(x+2*r/sqrt3,y+r/sqrt3);
		vertex(x,y-r);
		end_line();
	}
	break;
	case 11:			// filled inverted triangle
	{
		const greal r = 0.5*size;
		const greal sqrt3 = sqrt(3.0);
		begin_fill(x,y-r);
		vertex(x-2*r/sqrt3,y+r/sqrt3);
		vertex(x+2*r/sqrt3,y+r/sqrt3);
		vertex(x,y-r);
		end_fill();
	}
	break;
	case 12:			// circle
	{
		circle(x,y,0.5*size);
	}
	break;
	case 13:			// filled circle
	{
		fillcircle(x,y,0.5*size);
	}
	break;
	default:
		cerr << "Tgraph2D::mark: unknown markertype " << type << "\n";
		break;
	}
}

