/* specter_MYSQL.c
 *
 * output plugin for logging data into a MySQL database
 * 
 * (C) 2004,2005 by Michal Kwiatkowski <ruby@joker.linuxstuff.pl>
 */

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 
 *  as published by the Free Software Foundation
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <specter/specter.h>
#include <conffile/conffile.h>
#include <mysql/mysql.h>
#include "sql.h"


/* our configuration directives */
static config_entry_t my_config[] = {
	{ .key = "db", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_MANDATORY },
	{ .key = "host", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_NONE },
	{ .key = "user", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_NONE },
	{ .key = "pass", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_NONE },
	{ .key = "table", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_MANDATORY },
	{ .key = "buffsize", .type = CONFIG_TYPE_MEM, .options = CONFIG_OPT_NONE, .u = { .value = 0 } },
	{ .key = "ssl_enable", .type = CONFIG_TYPE_BOOLEAN, .options = CONFIG_OPT_NONE, .u = { .value = 0 } },
	{ .key = "ssl_key", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_NONE },
	{ .key = "ssl_cert", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_NONE },
	{ .key = "ssl_ca", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_NONE },
	{ .key = "ssl_capath", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_NONE },
	{ .key = "ssl_cipher", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_NONE },
	{ .key = "port", .type = CONFIG_TYPE_INT, .options = CONFIG_OPT_NONE, .u = { .value = 0 } },
};

struct my_data {
	MYSQL *dbh;
	struct sql_field *f;
	char *buff_start;
	char *buff_cur;
	size_t length;
};


static void *specter_mysql_init(config_entry_t *ce)
{
	struct my_data *data;
	MYSQL_RES *result;
	MYSQL_FIELD *field;
	char **columns;
	int ctr;

	if ((data = malloc(sizeof(struct my_data))) == NULL) {
		specter_log(SPECTER_FATAL, "Couldn't allocate data: %s.\n",
				strerror(errno));
		return NULL;
	}

	memset(data, 0x0, sizeof(struct my_data));

	if ((data->dbh = mysql_init(NULL)) == NULL) {
		specter_log(SPECTER_FATAL, "Couldn't allocate memory for new MYSQL object.\n");
		goto out_free_data;
	}

	if (GET_CE(ce,6)->u.value) {
		char *key = GET_CE(ce,7)->u.string[0] ? GET_CE(ce,7)->u.string : NULL;
		char *cert = GET_CE(ce,8)->u.string[0] ? GET_CE(ce,8)->u.string : NULL;
		char *ca = GET_CE(ce,9)->u.string[0] ? GET_CE(ce,9)->u.string : NULL;
		char *capath = GET_CE(ce,10)->u.string[0] ? GET_CE(ce,10)->u.string : NULL;
		char *cipher = GET_CE(ce,11)->u.string[0] ? GET_CE(ce,11)->u.string : NULL;

		mysql_ssl_set(data->dbh, key, cert, ca, capath, cipher);
	}

	if (mysql_real_connect(data->dbh, GET_CE(ce,1)->u.string,
				GET_CE(ce,2)->u.string, GET_CE(ce,3)->u.string,
				GET_CE(ce,0)->u.string, GET_CE(ce,12)->u.value, NULL, 0) == NULL) {

		specter_log(SPECTER_FATAL, "Couldn't connect to database: %s.\n",
				mysql_error(data->dbh));
		goto out_free_data;
	}

	if ((result = mysql_list_fields(data->dbh, GET_CE(ce,4)->u.string, NULL)) == NULL) {
		specter_log(SPECTER_FATAL, "Couldn't get mysql columns: %s.\n",
				mysql_error(data->dbh));
		goto out_free_data;
	}

	if ((columns = malloc((mysql_num_fields(result) + 1) * sizeof(char *))) == NULL) {
		specter_log(SPECTER_FATAL, "Couldn't allocate data: %s.\n",
				strerror(errno));
		goto out_free_result;
	}

	for (ctr = 0; (field = mysql_fetch_field(result)) != NULL; ctr++)
		columns[ctr] = field->name;
	columns[ctr] = NULL;

	data->length = GET_CE(ce,5)->u.value;
	data->buff_cur = alloc_sql_insert(columns, GET_CE(ce,4)->u.string,
			&data->buff_start, &data->length, &data->f);
	/* alloc_sql_insert already printed error message */
	if (data->buff_cur == NULL)
		goto out_free_columns;

	/* all done */
	free(columns);
	mysql_free_result(result);
	return data;

out_free_columns:
	free(columns);

out_free_result:
	mysql_free_result(result);

out_free_data:
	free(data);

	return NULL;
}


static MYSQL *global_mysql_dbh;
static size_t specter_mysql_escape(char *dest, const char *src, size_t n)
{
	unsigned long length = strlen(src);

	if (n < length*2+1) /* will cause exit in fill_sql_insert() */
		return length*2+1;

	*dest++ = '\'';
#ifdef MYSQL_OLD
	length = mysql_escape_string(dest, src, length);
#else
	length = mysql_real_escape_string(global_mysql_dbh, dest, src, length);
#endif
	*(dest+length) = '\'';
	return length+2;
}


static int specter_mysql_output(config_entry_t *ce, void *data)
{
	struct my_data *md = data;
	char *end;

	global_mysql_dbh = md->dbh;
	if ((end = fill_sql_insert(md->f, md->buff_cur,
				md->length - (md->buff_cur - md->buff_start),
				&specter_mysql_escape)) == NULL)
		return -1;
	global_mysql_dbh = NULL;

	specter_log(SPECTER_DEBUG, "Calling MySQL: #%s#\n", md->buff_start);

	if (mysql_real_query(md->dbh, md->buff_start, end - md->buff_start) != 0) {
		specter_log(SPECTER_ERROR, "Couldn't insert data into database: %s\n",
					mysql_error(md->dbh));
		return -1;
	}

	return 0;
}


static void specter_mysql_fini(config_entry_t *ce, void *data)
{
	struct my_data *md = data;

	mysql_close(md->dbh);
	free_sql_insert(md->buff_start, md->f);
	free(data);
}


static specter_output_t mysql_op = {
	.name = "mysql",
	.ce_base = my_config,
	.ce_num = 13,
	.init = &specter_mysql_init,
	.fini = &specter_mysql_fini,
	.output = &specter_mysql_output
};

void _init(void) 
{
	if (register_output(&mysql_op, 0) == -1) {
		specter_log(SPECTER_FATAL, "Can't register.\n");
		exit(EXIT_FAILURE);
	}
}

