/*
/ Shapefiles.cpp
/ methods related to Shapefile loading and saving
/
/ version 1.2, 2008 October 9
/
/ Author: Sandro Furieri a-furieri@lqt.it
/
/ Copyright (C) 2008  Alessandro Furieri
/
/    This program is free software: you can redistribute it and/or modify
/    it under the terms of the GNU General Public License as published by
/    the Free Software Foundation, either version 3 of the License, or
/    (at your option) any later version.
/
/    This program is distributed in the hope that it will be useful,
/    but WITHOUT ANY WARRANTY; without even the implied warranty of
/    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/    GNU General Public License for more details.
/
/    You should have received a copy of the GNU General Public License
/    along with this program.  If not, see <http://www.gnu.org/licenses/>.
/
*/

#include "Classdef.h"

void MyFrame::CleanTxtTab(char *buf)
{
// well-formatting a string to be used as a Txt/Tab string
  char tmp[65536];
  char *in = tmp;
  char *out = buf;
  strcpy(tmp, buf);
  while (*in != '\0')
    {
      if (*in == '\t' || *in == '\r' || *in == '\n')
        {
          in++;
          *out++ = ' ';
      } else
        *out++ = *in++;
    }
  *out = '\0';
}

void MyFrame::CleanCsv(char *buf)
{
// well-formatting a string to be used as a Csv string
  char tmp[65536];
  char *in = tmp;
  char *out = buf;
  bool special = false;
  strcpy(tmp, buf);
  while (*in != '\0')
    {
      if (*in == ',' || *in == '\r' || *in == '\n')
        special = true;
      if (*in == '"')
        *out++ = '"';
      *out++ = *in++;
    }
  *out = '\0';
  if (special == true)
    {
      sprintf(tmp, "\"%s\"", buf);
      strcpy(buf, tmp);
    }
}

void MyFrame::CleanHtml(char *buf)
{
// well-formatting a string to be used as an Html string
  char tmp[65536];
  char *in = tmp;
  char *out = buf;
  strcpy(tmp, buf);
  while (*in != '\0')
    {
      if (*in == '<')
        {
          *out++ = '&';
          *out++ = 'l';
          *out++ = 't';
          *out++ = ';';
          in++;
          continue;
        }
      if (*in == '>')
        {
          *out++ = '&';
          *out++ = 'g';
          *out++ = 't';
          *out++ = ';';
          in++;
          continue;
        }
      if (*in == ' ')
        {
          *out++ = '&';
          *out++ = 'n';
          *out++ = 'b';
          *out++ = 's';
          *out++ = 'p';
          *out++ = ';';
          in++;
          continue;
        }
      if (*in == '"')
        {
          *out++ = '&';
          *out++ = 'q';
          *out++ = 'u';
          *out++ = 'o';
          *out++ = 't';
          *out++ = ';';
          in++;
          continue;
        }
      if (*in == '&')
        {
          *out++ = '&';
          *out++ = 'a';
          *out++ = 'm';
          *out++ = 'p';
          *out++ = ';';
          in++;
          continue;
        }
      *out++ = *in++;
    }
  *out = '\0';
}

bool MyFrame::TableAlreadyExists(wxString & name)
{
//
// checks if a table of this name already exists 
//
  char **results;
  int rows;
  int columns;
  int i;
  char *errMsg = NULL;
  bool already_exists = false;
  wxString sql =
    wxT("SELECT name FROM sqlite_master WHERE type = 'table' AND name LIKE '");
  sql += name;
  sql += wxT("'");
  int ret = sqlite3_get_table(SqliteHandle, sql.ToUTF8(), &results,
                              &rows, &columns, &errMsg);
  if (ret != SQLITE_OK)
    {
      wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                   wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
      sqlite3_free(errMsg);
      return false;
    }
  if (rows < 1)
    ;
  else
    {
      for (i = 1; i <= rows; i++)
        already_exists = true;
    }
  sqlite3_free_table(results);
  return already_exists;
}

bool MyFrame::SridNotExists(int srid)
{
//
// checks if a SRID value is a valid one
//
  char **results;
  int rows;
  int columns;
  int i;
  char *errMsg = NULL;
  bool constrained = false;
  bool not_exists = true;
  wxString RefSysName;
  char xsql[128];
  wxString sql =
    wxT
    ("SELECT name FROM sqlite_master WHERE type = 'table' AND name = 'spatial_ref_sys'");
  int ret = sqlite3_get_table(SqliteHandle, sql.ToUTF8(), &results,
                              &rows, &columns, &errMsg);
  if (ret != SQLITE_OK)
    {
      wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                   wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
      sqlite3_free(errMsg);
      return false;
    }
  if (rows < 1)
    ;
  else
    {
      for (i = 1; i <= rows; i++)
        constrained = true;
    }
  sqlite3_free_table(results);
  if (constrained == false)
    return false;
  sprintf(xsql, "SELECT ref_sys_name FROM spatial_ref_sys WHERE srid = %d",
          srid);
  ret =
    sqlite3_get_table(SqliteHandle, xsql, &results, &rows, &columns, &errMsg);
  if (ret != SQLITE_OK)
    {
      wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                   wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
      sqlite3_free(errMsg);
      return false;
    }
  if (rows < 1)
    ;
  else
    {
      for (i = 1; i <= rows; i++)
        not_exists = false;
    }
  sqlite3_free_table(results);
  return not_exists;
}

bool MyFrame::CheckMetadata()
{
//
// checking if METADATA tables are defined
//
  char **results;
  int rows;
  int columns;
  int i;
  char *errMsg = NULL;
  bool constrained = false;
  if (SpatiaLiteMetadata == false)
    return false;
  wxString sql =
    wxT
    ("SELECT name FROM sqlite_master WHERE type = 'table' AND name = 'geometry_columns'");
  int ret = sqlite3_get_table(SqliteHandle, sql.ToUTF8(), &results,
                              &rows, &columns, &errMsg);
  if (ret != SQLITE_OK)
    {
      wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                   wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
      sqlite3_free(errMsg);
      return false;
    }
  if (rows < 1)
    ;
  else
    {
      for (i = 1; i <= rows; i++)
        constrained = true;
    }
  sqlite3_free_table(results);
  return constrained;
}

void MyFrame::CleanSqlString(char *value)
{
//
// returns a well formatted TEXT value for SQL
// 1] strips trailing spaces
// 2] masks any ' inside the string, appending another '
//
  char new_value[1024];
  char *p;
  int len;
  int i;
  len = strlen(value);
  for (i = (len - 1); i >= 0; i--)
    {
      // stripping trailing spaces
      if (value[i] == ' ')
        value[i] = '\0';
      else
        break;
    }
  p = new_value;
  for (i = 0; i < len; i++)
    {
      if (value[i] == '\'')
        *(p++) = '\'';
      *(p++) = value[i];
    }
  *p = '\0';
  strcpy(value, new_value);
}

void MyFrame::LoadShapefile(wxString & path, wxString & table, int srid,
                            wxString & column, wxString & charset)
{
//
// loading a Shapefile as a new DB table
//
  sqlite3_stmt *stmt;
  int cnt;
  int col_cnt;
  int seed;
  int len;
  int dup;
  int idup;
  int current_row;
  int ret;
  int rows = 0;
  char *errMsg = NULL;
  char xtable[1024];
  char xcolumn[1024];
  char dummyName[4096];
  char sql[65536];
  char **col_name = NULL;
  unsigned char *blob;
  int blob_size;
  const char *geom_type = "UNKNOWN";
  wxString dummyStr;
  wxString msg;
  gaiaShapefilePtr shp = NULL;
  gaiaDbfFieldPtr dbf_field;
  bool metadata = CheckMetadata();
  bool sqlError = false;
//
// performing some checks before starting
//
  if (TableAlreadyExists(table) == true)
    {
      wxMessageBox(wxT("a table name '") + table + wxT("' already exists"),
                   wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
      return;
    }
  if (metadata == true)
    {
      if (SridNotExists(srid) == true)
        {
          wxMessageBox(wxT("invalid SRID value"), wxT("spatialite-gui"),
                       wxOK | wxICON_ERROR, this);
          return;
        }
    }
//
// initalizing the SHP struct
//
  shp = gaiaAllocShapefile();
  gaiaOpenShpRead(shp, path.ToUTF8(), charset.ToUTF8(), "UTF-8");
  if (!(shp->Valid))
    {
      wxString error = wxT("ERROR: invalid Shapefile\n\n");
      if (shp->LastError)
        error += wxString::FromUTF8(shp->LastError);
      gaiaFreeShapefile(shp);
      wxMessageBox(error, wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
      return;
    }
  ::wxBeginBusyCursor();
//
// checking for duplicate / illegal column names and antialising them 
//
  col_cnt = 0;
  dbf_field = shp->Dbf->First;
  while (dbf_field)
    {
      // counting DBF fields
      col_cnt++;
      dbf_field = dbf_field->Next;
    }
  col_name = (char **) malloc(sizeof(char *) * col_cnt);
  cnt = 0;
  seed = 0;
  dbf_field = shp->Dbf->First;
  while (dbf_field)
    {
      // preparing column names
      strcpy(dummyName, dbf_field->Name);
      dup = 0;
      for (idup = 0; idup < cnt; idup++)
        {
          if (strcasecmp(dummyName, *(col_name + idup)) == 0)
            dup = 1;
        }
      if (strcasecmp(dummyName, "PK_UID") == 0)
        dup = 1;
      if (strcasecmp(dummyName, column.ToUTF8()) == 0)
        dup = 1;
      if (dup)
        sprintf(dummyName, "COL_%d", seed++);
      len = strlen(dummyName);
      *(col_name + cnt) = (char *) malloc(len + 1);
      strcpy(*(col_name + cnt), dummyName);
      cnt++;
      dbf_field = dbf_field->Next;
    }
//
// starting a transaction
//
  ret = sqlite3_exec(SqliteHandle, "BEGIN", NULL, 0, &errMsg);
  if (ret != SQLITE_OK)
    {
      wxMessageBox(wxT("load shapefile error:") + wxString::FromUTF8(errMsg),
                   wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
      sqlite3_free(errMsg);
      sqlError = true;
      goto clean_up;
    }
//
// creating the Table 
//
  strcpy(xtable, table.ToUTF8());
  strcpy(xcolumn, column.ToUTF8());
  sprintf(sql, "CREATE TABLE \"%s\"", xtable);
  strcat(sql, " (\n\"PK_UID\" INTEGER PRIMARY KEY AUTOINCREMENT");
  cnt = 0;
  dbf_field = shp->Dbf->First;
  while (dbf_field)
    {
      strcat(sql, ",\n\"");
      strcat(sql, *(col_name + cnt));
      cnt++;
      switch (dbf_field->Type)
        {
          case 'C':
            strcat(sql, "\" TEXT");
            break;
          case 'N':
            if (dbf_field->Decimals)
              strcat(sql, "\" DOUBLE");
            else
              {
                if (dbf_field->Length <= 18)
                  strcat(sql, "\" INTEGER");
                else
                  strcat(sql, "\" DOUBLE");
              }
            break;
          case 'D':
            strcat(sql, "\" DOUBLE");
            break;
          case 'L':
            strcat(sql, "\" INTEGER");
            break;
        };
      dbf_field = dbf_field->Next;
    }
  if (metadata == true)
    strcat(sql, ")");
  else
    {
      strcat(sql, ",\n");
      strcat(sql, xcolumn);
      strcat(sql, " BLOB)");
    }
  ret = sqlite3_exec(SqliteHandle, sql, NULL, 0, &errMsg);
  if (ret != SQLITE_OK)
    {
      wxMessageBox(wxT("load shapefile error:") + wxString::FromUTF8(errMsg),
                   wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
      sqlite3_free(errMsg);
      sqlError = true;
      goto clean_up;
    }
  if (metadata)
    {
      // creating Geometry column 
      switch (shp->Shape)
        {
          case 1:
          case 11:
          case 21:
            geom_type = "POINT";
            break;
          case 8:
            geom_type = "MULTIPOINT";
            break;
          case 3:
          case 13:
          case 23:
            gaiaShpAnalyze(shp);
            if (shp->EffectiveType == GAIA_LINESTRING)
              geom_type = "LINESTRING";
            else
              geom_type = "MULTILINESTRING";
            break;
          case 5:
          case 15:
          case 25:
            gaiaShpAnalyze(shp);
            if (shp->EffectiveType == GAIA_POLYGON)
              geom_type = "POLYGON";
            else
              geom_type = "MULTIPOLYGON";
            break;
        };
      sprintf(sql, "SELECT AddGeometryColumn('%s', '%s', %d, '%s', 2)", xtable,
              xcolumn, srid, geom_type);
      ret = sqlite3_exec(SqliteHandle, sql, NULL, 0, &errMsg);
      if (ret != SQLITE_OK)
        {
          wxMessageBox(wxT("load shapefile error:") +
                       wxString::FromUTF8(errMsg), wxT("spatialite-gui"),
                       wxOK | wxICON_ERROR, this);
          sqlite3_free(errMsg);
          sqlError = true;
          goto clean_up;
        }
  } else
    {
      // no Metadata
      if (shp->Shape == 3 || shp->Shape == 13 || shp->Shape == 23
          || shp->Shape == 5 || shp->Shape == 15 || shp->Shape == 25)
        {
          // fixing anyway the Geometry type for LINESTRING/MULTILINESTRING or POLYGON/MULTIPOLYGON
          gaiaShpAnalyze(shp);
        }
    }
// preparing the INSERT INTO parameterized statement
  sprintf(sql, "INSERT INTO \"%s\" (\"PK_UID\",", xtable);
  cnt = 0;
  dbf_field = shp->Dbf->First;
  while (dbf_field)
    {
      // columns corresponding to some DBF attribute 
      strcat(sql, "\"");
      strcat(sql, *(col_name + cnt));
      cnt++;
      strcat(sql, "\",");
      dbf_field = dbf_field->Next;
    }
  strcat(sql, xcolumn);         // the GEOMETRY column
  strcat(sql, ")\nVALUES (? ");
  dbf_field = shp->Dbf->First;
  while (dbf_field)
    {
      // column values
      strcat(sql, ", ?");
      dbf_field = dbf_field->Next;
    }
  strcat(sql, ", ?)");          // the GEOMETRY column
  ret = sqlite3_prepare_v2(SqliteHandle, sql, strlen(sql), &stmt, NULL);
  if (ret != SQLITE_OK)
    {
      wxString err = wxString::FromUTF8(sqlite3_errmsg(SqliteHandle));
      wxMessageBox(wxT("load shapefile error: ") + err, wxT("spatialite-gui"),
                   wxOK | wxICON_ERROR, this);
      sqlError = true;
      goto clean_up;
    }
  current_row = 0;
  while (1)
    {
      //
      // inserting rows from shapefile 
      //
      ret = gaiaReadShpEntity(shp, current_row, srid);
      if (!ret)
        {
          if (!(shp->LastError))  // normal SHP EOF
            break;
          wxMessageBox(wxT("load shapefile error:") +
                       wxString::FromUTF8(shp->LastError),
                       wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
          sqlError = true;
          goto clean_up;
        }
      current_row++;
      // binding query params
      sqlite3_reset(stmt);
      sqlite3_clear_bindings(stmt);
      sqlite3_bind_int(stmt, 1, current_row);
      cnt = 0;
      dbf_field = shp->Dbf->First;
      while (dbf_field)
        {
          // column values
          if (!(dbf_field->Value))
            sqlite3_bind_null(stmt, cnt + 2);
          else
            {
              switch (dbf_field->Value->Type)
                {
                  case GAIA_INT_VALUE:
                    sqlite3_bind_int64(stmt, cnt + 2,
                                       dbf_field->Value->IntValue);
                    break;
                  case GAIA_DOUBLE_VALUE:
                    sqlite3_bind_double(stmt, cnt + 2,
                                        dbf_field->Value->DblValue);
                    break;
                  case GAIA_TEXT_VALUE:
                    sqlite3_bind_text(stmt, cnt + 2,
                                      dbf_field->Value->TxtValue,
                                      strlen(dbf_field->Value->
                                             TxtValue), SQLITE_STATIC);
                    break;
                  default:
                    sqlite3_bind_null(stmt, cnt + 2);;
                    break;
                };
            }
          cnt++;
          dbf_field = dbf_field->Next;
        }
      if (shp->Dbf->Geometry)
        {
          gaiaToSpatiaLiteBlobWkb(shp->Dbf->Geometry, &blob, &blob_size);
          sqlite3_bind_blob(stmt, cnt + 2, blob, blob_size, free);
      } else
        {
          /* handling a NULL-Geometry */
          sqlite3_bind_null(stmt, cnt + 2);
        }
      ret = sqlite3_step(stmt);
      if (ret == SQLITE_DONE || ret == SQLITE_ROW)
        ;
      else
        {
          wxString err = wxString::FromUTF8(sqlite3_errmsg(SqliteHandle));
          wxMessageBox(wxT("load shapefile error:") + err,
                       wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
          sqlite3_finalize(stmt);
          sqlError = true;
          goto clean_up;
        }
      rows++;
    }
  sqlite3_finalize(stmt);
clean_up:
  gaiaFreeShapefile(shp);
  if (col_name)
    {
      // releasing memory allocation for column names 
      for (cnt = 0; cnt < col_cnt; cnt++)
        free(*(col_name + cnt));
      free(col_name);
    }
  if (sqlError == true)
    {
      // some error occurred - ROLLBACK 
      ret = sqlite3_exec(SqliteHandle, "ROLLBACK", NULL, 0, &errMsg);
      if (ret != SQLITE_OK)
        {
          wxMessageBox(wxT("load shapefile error:") +
                       wxString::FromUTF8(errMsg), wxT("spatialite-gui"),
                       wxOK | wxICON_ERROR, this);
          sqlite3_free(errMsg);
        }
      ::wxEndBusyCursor();
      msg =
        wxT("Shapefile not loaded\n\n\na ROLLBACK was automatically performed");
      wxMessageBox(msg, wxT("spatialite-gui"), wxOK | wxICON_WARNING, this);
  } else
    {
      // ok - confirming pending transaction - COMMIT 
      ret = sqlite3_exec(SqliteHandle, "COMMIT", NULL, 0, &errMsg);
      if (ret != SQLITE_OK)
        {
          wxMessageBox(wxT("load shapefile error:") +
                       wxString::FromUTF8(errMsg), wxT("spatialite-gui"),
                       wxOK | wxICON_ERROR, this);
          sqlite3_free(errMsg);
          return;
        }
      ::wxEndBusyCursor();
      sprintf(dummyName, "Shapefile loaded\n\n%d inserted rows", rows);
      msg = wxString::FromUTF8(dummyName);
      wxMessageBox(msg, wxT("spatialite-gui"), wxOK | wxICON_INFORMATION, this);
      InitTableTree();
    }
}

gaiaDbfFieldPtr MyFrame::GetDbfField(gaiaDbfListPtr list, char *name)
{
//
// find a DBF attribute by name 
//
  gaiaDbfFieldPtr fld = list->First;
  while (fld)
    {
      if (strcasecmp(fld->Name, name) == 0)
        return fld;
      fld = fld->Next;
    }
  return NULL;
}

void MyFrame::DumpShapefile(wxString & path, wxString & table,
                            wxString & column, wxString & charset)
{
//
// dumping a  geometry table as Shapefile
//
  char **results;
  int rows;
  int columns;
  int i;
  char *errMsg = NULL;
  char *gtype;
  wxString geometryType;
  int shape = -1;
  char xtable[1024];
  char xcolumn[1024];
  char xpath[1024];
  char xsql[4096];
  sqlite3_stmt *stmt;
  int row1 = 0;
  int n_cols = 0;
  int offset = 0;
  int type;
  int multiple_entities = 0;
  const unsigned char *char_value;
  const void *blob_value;
  gaiaShapefilePtr shp = NULL;
  gaiaDbfListPtr dbf_export_list = NULL;
  gaiaDbfListPtr dbf_list = NULL;
  gaiaDbfListPtr dbf_write;
  gaiaDbfFieldPtr dbf_field;
  gaiaGeomCollPtr geom;
  int *max_length = NULL;
  int *sql_type = NULL;
  char dummy[1024];
  int len;
  wxString msg;
  wxString sql =
    wxT("SELECT type FROM geometry_columns WHERE f_table_name = '");
  sql += table;
  sql += wxT("' AND f_geometry_column = '");
  sql += column;
  sql += wxT("'");
  int ret = sqlite3_get_table(SqliteHandle, sql.ToUTF8(), &results,
                              &rows, &columns, &errMsg);
  if (ret != SQLITE_OK)
    {
      wxMessageBox(wxT("dump shapefile error:") + wxString::FromUTF8(errMsg),
                   wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
      sqlite3_free(errMsg);
      return;
    }
  if (rows < 1)
    ;
  else
    {
      for (i = 1; i <= rows; i++)
        {
          gtype = results[(i * columns) + 0];
          geometryType = wxString::FromUTF8(gtype);
        }
    }
  sqlite3_free_table(results);
  if (geometryType == wxT("POINT"))
    shape = GAIA_POINT;
  if (geometryType == wxT("MULTIPOINT"))
    shape = GAIA_MULTIPOINT;
  if (geometryType == wxT("LINESTRING")
      || geometryType == wxT("MULTILINESTRING"))
    shape = GAIA_LINESTRING;
  if (geometryType == wxT("POLYGON") || geometryType == wxT("MULTIPOLYGON"))
    shape = GAIA_POLYGON;
  if (shape < 0)
    {
      wxMessageBox(wxT("Unable to detect GeometryType for '") +
                   table + wxT(".") + column + wxT("'"), wxT("spatialite-gui"),
                   wxOK | wxICON_ERROR, this);
      return;
    }
//
// preparing SQL statement 
//
  strcpy(xtable, table.ToUTF8());
  strcpy(xcolumn, column.ToUTF8());
  sprintf(xsql, "SELECT * FROM \"%s\" WHERE GeometryType(\"%s\") = ", xtable,
          xcolumn);
  if (shape == GAIA_LINESTRING)
    {
      strcat(xsql, "'LINESTRING' OR GeometryType(\"");
      strcat(xsql, xcolumn);
      strcat(xsql, "\") = 'MULTILINESTRING'");
  } else if (shape == GAIA_POLYGON)
    {
      strcat(xsql, "'POLYGON' OR GeometryType(\"");
      strcat(xsql, xcolumn);
      strcat(xsql, "\") = 'MULTIPOLYGON'");
  } else if (shape == GAIA_MULTIPOINT)
    {
      strcat(xsql, "'POINT' OR GeometryType(\"");
      strcat(xsql, xcolumn);
      strcat(xsql, "\") = 'MULTIPOINT'");
  } else
    strcat(xsql, "'POINT'");
// fetching anyway NULL Geometries 
  strcat(xsql, " OR \"");
  strcat(xsql, xcolumn);
  strcat(xsql, "\" IS NULL");
//
// compiling SQL prepared statement 
//
  ret = sqlite3_prepare_v2(SqliteHandle, xsql, strlen(xsql), &stmt, NULL);
  if (ret != SQLITE_OK)
    goto sql_error;
  rows = 0;
  while (1)
    {
      //
      // Pass I - scrolling the result set to compute real DBF attributes' sizes and types 
      //
      ret = sqlite3_step(stmt);
      if (ret == SQLITE_DONE)
        break;                  // end of result set
      if (ret == SQLITE_ROW)
        {
          // processing a result set row 
          row1++;
          if (n_cols == 0)
            {
              // this one is the first row, so we are going to prepare the DBF Fields list 
              n_cols = sqlite3_column_count(stmt);
              dbf_export_list = gaiaAllocDbfList();
              max_length = (int *) malloc(sizeof(int) * n_cols);
              sql_type = (int *) malloc(sizeof(int) * n_cols);
              for (i = 0; i < n_cols; i++)
                {
                  // initializes the DBF export fields
                  strcpy(dummy, sqlite3_column_name(stmt, i));
                  gaiaAddDbfField(dbf_export_list, dummy, '\0', 0, 0, 0);
                  max_length[i] = 0;
                  sql_type[i] = SQLITE_NULL;
                }
            }
          for (i = 0; i < n_cols; i++)
            {
              // update the DBF export fields analyzing fetched data
              type = sqlite3_column_type(stmt, i);
              if (type == SQLITE_BLOB && strcasecmp((char *) xcolumn,
                                                    (char *)
                                                    sqlite3_column_name(stmt,
                                                                        i)) == 0
                  && shape == GAIA_POINT)
                {
                  // 
                  // we need to check if there is any MULTIPOINT,
                  // because shapefile handles simple-points shapes and multi-points shapes
                  // in a complete differet way
                  //
                  blob_value = sqlite3_column_blob(stmt, i);
                  len = sqlite3_column_bytes(stmt, i);
                  geom =
                    gaiaFromSpatiaLiteBlobWkb((unsigned char *) blob_value,
                                              len);
                  if (geom)
                    {
                      if (geom->FirstPoint != geom->LastPoint)
                        multiple_entities = 1;
                      gaiaFreeGeomColl(geom);
                    }
                }
              if (type == SQLITE_NULL || type == SQLITE_BLOB)
                continue;
              if (type == SQLITE_TEXT)
                {
                  char_value = sqlite3_column_text(stmt, i);
                  len = sqlite3_column_bytes(stmt, i);
                  sql_type[i] = SQLITE_TEXT;
                  if (len > max_length[i])
                    max_length[i] = len;
              } else if (type == SQLITE_FLOAT && sql_type[i] != SQLITE_TEXT)
                sql_type[i] = SQLITE_FLOAT; // promoting a numeric column to be DOUBLE
              else if (type == SQLITE_INTEGER
                       && (sql_type[i] == SQLITE_NULL
                           || sql_type[i] == SQLITE_INTEGER))
                sql_type[i] = SQLITE_INTEGER; // promoting a null column to be INTEGER
            }
      } else
        goto sql_error;
    }
  if (!row1)
    goto empty_result_set;
  i = 0;
  offset = 0;
  dbf_list = gaiaAllocDbfList();
  dbf_field = dbf_export_list->First;
  while (dbf_field)
    {
      // preparing the final DBF attribute list 
      if (sql_type[i] == SQLITE_NULL)
        {
          i++;
          dbf_field = dbf_field->Next;
          continue;
        }
      if (sql_type[i] == SQLITE_TEXT)
        {
          gaiaAddDbfField(dbf_list, dbf_field->Name, 'C', offset, max_length[i],
                          0);
          offset += max_length[i];
        }
      if (sql_type[i] == SQLITE_FLOAT)
        {
          gaiaAddDbfField(dbf_list, dbf_field->Name, 'N', offset, 24, 6);
          offset += 24;
        }
      if (sql_type[i] == SQLITE_INTEGER)
        {
          gaiaAddDbfField(dbf_list, dbf_field->Name, 'N', offset, 18, 0);
          offset += 18;
        }
      i++;
      dbf_field = dbf_field->Next;
    }
  free(max_length);
  free(sql_type);
  gaiaFreeDbfList(dbf_export_list);
// resetting SQLite query 
  ret = sqlite3_reset(stmt);
  if (ret != SQLITE_OK)
    goto sql_error;
// trying to open shapefile files 
  shp = gaiaAllocShapefile();
  strcpy(xpath, path.ToUTF8());
  gaiaOpenShpWrite(shp, xpath, shape, dbf_list, "UTF-8", charset.ToUTF8());
  if (!(shp->Valid))
    goto no_file;
  while (1)
    {
      // Pass II - scrolling the result set to dump data into shapefile 
      ret = sqlite3_step(stmt);
      if (ret == SQLITE_DONE)
        break;                  // end of result set
      if (ret == SQLITE_ROW)
        {
          rows++;
          geom = NULL;
          dbf_write = gaiaCloneDbfEntity(dbf_list);
          for (i = 0; i < n_cols; i++)
            {
              if (strcasecmp
                  ((char *) xcolumn,
                   (char *) sqlite3_column_name(stmt, i)) == 0)
                {
                  // this one is the internal BLOB encoded GEOMETRY to be exported 
                  if (sqlite3_column_type(stmt, i) != SQLITE_BLOB)
                    {
                      // this one is a NULL Geometry
                      dbf_write->Geometry = NULL;
                  } else
                    {
                      blob_value = sqlite3_column_blob(stmt, i);
                      len = sqlite3_column_bytes(stmt, i);
                      dbf_write->Geometry =
                        gaiaFromSpatiaLiteBlobWkb((unsigned char *) blob_value,
                                                  len);
                    }
                }
              strcpy(dummy, sqlite3_column_name(stmt, i));
              dbf_field = GetDbfField(dbf_write, dummy);
              if (!dbf_field)
                continue;
              if (sqlite3_column_type(stmt, i) == SQLITE_NULL)
                {
                  // handling NULL values
                  gaiaSetNullValue(dbf_field);
              } else
                {
                  switch (dbf_field->Type)
                    {
                      case 'N':
                        if (sqlite3_column_type(stmt, i) == SQLITE_INTEGER)
                          gaiaSetIntValue(dbf_field,
                                          sqlite3_column_int64(stmt, i));
                        else if (sqlite3_column_type(stmt, i) == SQLITE_FLOAT)
                          gaiaSetDoubleValue(dbf_field,
                                             sqlite3_column_double(stmt, i));
                        else
                          gaiaSetNullValue(dbf_field);
                        break;
                      case 'C':
                        if (sqlite3_column_type(stmt, i) == SQLITE_TEXT)
                          {
                            strcpy(dummy,
                                   (char *) sqlite3_column_text(stmt, i));
                            gaiaSetStrValue(dbf_field, dummy);
                        } else
                          gaiaSetNullValue(dbf_field);
                        break;
                    };
                }
            }
          if (!gaiaWriteShpEntity(shp, dbf_write))
            {
              wxMessageBox(wxT("Shapefile write error"), wxT("spatialite-gui"),
                           wxOK | wxICON_INFORMATION, this);
            }
          gaiaFreeDbfList(dbf_write);
      } else
        goto sql_error;
    }
  sqlite3_finalize(stmt);
  gaiaFlushShpHeaders(shp);
  gaiaFreeShapefile(shp);
  sprintf(dummy, "Exported %d rows into Shapefile", rows);
  msg = wxString::FromUTF8(dummy);
  wxMessageBox(msg, wxT("spatialite-gui"), wxOK | wxICON_INFORMATION, this);
  return;
sql_error:
//
// some SQL error occurred
//
  sqlite3_finalize(stmt);
  if (dbf_export_list);
  gaiaFreeDbfList(dbf_export_list);
  if (dbf_list);
  gaiaFreeDbfList(dbf_list);
  if (shp)
    gaiaFreeShapefile(shp);
  wxMessageBox(wxT("dump shapefile error:") +
               wxString::FromUTF8(sqlite3_errmsg(SqliteHandle)),
               wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
  return;
no_file:
//
// shapefile can't be created/opened
//
  if (dbf_export_list);
  gaiaFreeDbfList(dbf_export_list);
  if (dbf_list);
  gaiaFreeDbfList(dbf_list);
  if (shp)
    gaiaFreeShapefile(shp);
  wxMessageBox(wxT("ERROR: unable to open '") + path + wxT("' for writing"),
               wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
  return;
empty_result_set:
//
// the result set is empty - nothing to do 
//
  sqlite3_finalize(stmt);
  if (dbf_export_list);
  gaiaFreeDbfList(dbf_export_list);
  if (dbf_list);
  gaiaFreeDbfList(dbf_list);
  if (shp)
    gaiaFreeShapefile(shp);
  wxMessageBox(wxT
               ("The SQL SELECT returned an empty result set\n... there is nothing to export ..."),
               wxT("spatialite-gui"), wxOK | wxICON_WARNING, this);
}

void MyFrame::DumpTxtTab(wxString & path, wxString & table, wxString & charset)
{
//
// dumping a  table as Txt/Tab
//
  wxString sql;
  sqlite3_stmt *stmt;
  int ret;
  int rows = 0;
  int i;
  int n_cols;
  char xpath[1024];
  char dummy[65536];
  char outCs[128];
  char *pDummy;
  wxString msg;
  strcpy(outCs, charset.ToUTF8());
  strcpy(xpath, path.ToUTF8());
  FILE *out = fopen(xpath, "w");
  if (!out)
    goto no_file;
//
// preparing SQL statement 
//
  sql = wxT("SELECT * FROM \"");
  sql += table;
  sql += wxT("\"");
//
// compiling SQL prepared statement 
//
  ret = sqlite3_prepare_v2(SqliteHandle, sql.ToUTF8(), sql.Len(), &stmt, NULL);
  if (ret != SQLITE_OK)
    goto sql_error;
  rows = 0;
  while (1)
    {
      ret = sqlite3_step(stmt);
      if (ret == SQLITE_DONE)
        break;                  // end of result set
      if (ret == SQLITE_ROW)
        {
          n_cols = sqlite3_column_count(stmt);
          if (rows == 0)
            {
              // outputting the column titles
              for (i = 0; i < n_cols; i++)
                {
                  if (i == 0)
                    fprintf(out, "%s", sqlite3_column_name(stmt, i));
                  else
                    fprintf(out, "\t%s", sqlite3_column_name(stmt, i));
                }
              fprintf(out, "\n");
            }
          rows++;
          for (i = 0; i < n_cols; i++)
            {
              if (i > 0)
                fprintf(out, "\t");
              if (sqlite3_column_type(stmt, i) == SQLITE_INTEGER)
                fprintf(out, "%d", sqlite3_column_int(stmt, i));
              else if (sqlite3_column_type(stmt, i) == SQLITE_FLOAT)
                fprintf(out, "%1.6lf", sqlite3_column_double(stmt, i));
              else if (sqlite3_column_type(stmt, i) == SQLITE_TEXT)
                {
                  strcpy(dummy, (char *) sqlite3_column_text(stmt, i));
                  CleanTxtTab(dummy);
                  pDummy = dummy;
                  if (!gaiaConvertCharset(&pDummy, "UTF-8", outCs))
                    goto encoding_error;
                  fprintf(out, "%s", dummy);
                }
            }
          fprintf(out, "\n");
      } else
        goto sql_error;
    }
  sqlite3_finalize(stmt);
  fclose(out);
  sprintf(dummy, "Exported %d rows into Txt/Tab file", rows);
  msg = wxString::FromUTF8(dummy);
  wxMessageBox(msg, wxT("spatialite-gui"), wxOK | wxICON_INFORMATION, this);
  return;
sql_error:
//
// some SQL error occurred
//
  sqlite3_finalize(stmt);
  wxMessageBox(wxT("dump Txt/Tab error:") +
               wxString::FromUTF8(sqlite3_errmsg(SqliteHandle)),
               wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
  if (out)
    fclose(out);
  return;
encoding_error:
//
// some CHARSET converion occurred
//
  sqlite3_finalize(stmt);
  wxMessageBox(wxT("dump Txt/Tab: charset conversion reported an error"),
               wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
  if (out)
    fclose(out);
  return;
no_file:
//
// output file can't be created/opened
//
  wxMessageBox(wxT("ERROR: unable to open '") + path + wxT("' for writing"),
               wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
  return;
}

void MyFrame::DumpCsv(wxString & path, wxString & table, wxString & charset)
{
//
// dumping a  table as CSV
//
  wxString sql;
  sqlite3_stmt *stmt;
  int ret;
  int rows = 0;
  int i;
  int n_cols;
  char xpath[1024];
  char dummy[65536];
  char outCs[128];
  char *pDummy;
  wxString msg;
  strcpy(outCs, charset.ToUTF8());
  strcpy(xpath, path.ToUTF8());
  FILE *out = fopen(xpath, "w");
  if (!out)
    goto no_file;
//
// preparing SQL statement 
//
  sql = wxT("SELECT * FROM \"");
  sql += table;
  sql += wxT("\"");
//
// compiling SQL prepared statement 
//
  ret = sqlite3_prepare_v2(SqliteHandle, sql.ToUTF8(), sql.Len(), &stmt, NULL);
  if (ret != SQLITE_OK)
    goto sql_error;
  rows = 0;
  while (1)
    {
      ret = sqlite3_step(stmt);
      if (ret == SQLITE_DONE)
        break;                  // end of result set
      if (ret == SQLITE_ROW)
        {
          n_cols = sqlite3_column_count(stmt);
          if (rows == 0)
            {
              // outputting the column titles
              for (i = 0; i < n_cols; i++)
                {
                  if (i == 0)
                    {
                      strcpy(dummy, sqlite3_column_name(stmt, i));
                      CleanCsv(dummy);
                      fprintf(out, "%s", dummy);
                  } else
                    {
                      strcpy(dummy, sqlite3_column_name(stmt, i));
                      CleanCsv(dummy);
                      fprintf(out, ",%s", dummy);
                    }
                }
              fprintf(out, "\n");
            }
          rows++;
          for (i = 0; i < n_cols; i++)
            {
              if (i > 0)
                fprintf(out, ",");
              if (sqlite3_column_type(stmt, i) == SQLITE_INTEGER)
                fprintf(out, "%d", sqlite3_column_int(stmt, i));
              else if (sqlite3_column_type(stmt, i) == SQLITE_FLOAT)
                fprintf(out, "%1.6lf", sqlite3_column_double(stmt, i));
              else if (sqlite3_column_type(stmt, i) == SQLITE_TEXT)
                {
                  strcpy(dummy, (char *) sqlite3_column_text(stmt, i));
                  CleanCsv(dummy);
                  pDummy = dummy;
                  if (!gaiaConvertCharset
                      (&pDummy, (const char *) "UTF-8", outCs))
                    goto encoding_error;
                  fprintf(out, "%s", dummy);
                }
            }
          fprintf(out, "\n");
      } else
        goto sql_error;
    }
  sqlite3_finalize(stmt);
  fclose(out);
  sprintf(dummy, "Exported %d rows into CSV file", rows);
  msg = wxString::FromUTF8(dummy);
  wxMessageBox(msg, wxT("spatialite-gui"), wxOK | wxICON_INFORMATION, this);
  return;
sql_error:
//
// some SQL error occurred
//
  sqlite3_finalize(stmt);
  wxMessageBox(wxT("dump CSV error:") +
               wxString::FromUTF8(sqlite3_errmsg(SqliteHandle)),
               wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
  if (out)
    fclose(out);
  return;
encoding_error:
//
// some CHARSET converion occurred
//
  sqlite3_finalize(stmt);
  wxMessageBox(wxT("dump CSV: charset conversion reported an error"),
               wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
  if (out)
    fclose(out);
  return;
no_file:
//
// output file can't be created/opened
//
  wxMessageBox(wxT("ERROR: unable to open '") + path + wxT("' for writing"),
               wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
  return;
}

void MyFrame::DumpHtml(wxString & path, wxString & table, wxString & charset)
{
//
// dumping a  table as HTML
//
  wxString sql;
  sqlite3_stmt *stmt;
  int ret;
  int rows = 0;
  int i;
  int n_cols;
  char xpath[1024];
  char xtable[1024];
  char dummy[65536];
  char outCs[128];
  char *pDummy;
  wxString msg;
  strcpy(outCs, charset.ToUTF8());
  strcpy(xpath, path.ToUTF8());
  strcpy(xtable, table.ToUTF8());
  FILE *out = fopen(xpath, "w");
  if (!out)
    goto no_file;
  fprintf(out,
          "<!DOCTYPE html PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n");
  fprintf(out, "<html>\n\t<head>\n");
  fprintf(out,
          "\t\t<meta http-equiv=\"content-type\" content=\"text/html; charset=%s\">\n",
          outCs);
  fprintf(out, "\t\t<title>\nTable '%s': from SQLite/SpatiaLite DB '%s'\n",
          xtable, xpath);
  fprintf(out, "\t\t</title>\n");
  fprintf(out, "\t\t<style type=\"text/css\">\n");
  fprintf(out, "table { border: 1px; }\n");
  fprintf(out, "tr.t0 th { background-color: #c9c9df; }\n");
  fprintf(out, "tr.d0 td { background-color: #e0efe0; }\n");
  fprintf(out, "tr.d1 td { background-color: #d0d0df; }\n");
  fprintf(out, "\t\t</style>\n\t</head>\n\t<body>\n\t\t<table>\n");
//
// preparing SQL statement 
//
  sql = wxT("SELECT * FROM \"");
  sql += table;
  sql += wxT("\"");
//
// compiling SQL prepared statement 
//
  ret = sqlite3_prepare_v2(SqliteHandle, sql.ToUTF8(), sql.Len(), &stmt, NULL);
  if (ret != SQLITE_OK)
    goto sql_error;
  rows = 0;
  while (1)
    {
      ret = sqlite3_step(stmt);
      if (ret == SQLITE_DONE)
        break;                  // end of result set
      if (ret == SQLITE_ROW)
        {
          n_cols = sqlite3_column_count(stmt);
          if (rows == 0)
            {
              // outputting the column titles
              fprintf(out, "\t\t\t<tr class=\"t0\">\n");
              for (i = 0; i < n_cols; i++)
                {
                  strcpy(dummy, sqlite3_column_name(stmt, i));
                  CleanHtml(dummy);
                  fprintf(out, "\t\t\t\t<th>%s</th>\n", dummy);
                }
              fprintf(out, "\t\t\t</tr>\n");
            }
          rows++;
          fprintf(out, "\t\t\t<tr class=\"%s\">\n", (rows % 2) ? "d0" : "d1");
          for (i = 0; i < n_cols; i++)
            {
              if (sqlite3_column_type(stmt, i) == SQLITE_INTEGER)
                fprintf(out, "\t\t\t\t<td align=\"right\">%d</td>\n",
                        sqlite3_column_int(stmt, i));
              else if (sqlite3_column_type(stmt, i) == SQLITE_FLOAT)
                fprintf(out, "\t\t\t\t<td align=\"right\">%1.6lf</td>\n",
                        sqlite3_column_double(stmt, i));
              else if (sqlite3_column_type(stmt, i) == SQLITE_TEXT)
                {
                  strcpy(dummy, (char *) sqlite3_column_text(stmt, i));
                  CleanHtml(dummy);
                  pDummy = dummy;
                  if (!gaiaConvertCharset
                      (&pDummy, (const char *) "UTF-8", outCs))
                    goto encoding_error;
                  fprintf(out, "\t\t\t\t<td>%s</td>\n", dummy);
                }
            }
          fprintf(out, "\t\t\t</tr>\n");
      } else
        goto sql_error;
    }
  sqlite3_finalize(stmt);
  fprintf(out, "\t\t</table>\n\t</body>\n</html>\n");
  fclose(out);
  sprintf(dummy, "Exported %d rows into HTML file", rows);
  msg = wxString::FromUTF8(dummy);
  wxMessageBox(msg, wxT("spatialite-gui"), wxOK | wxICON_INFORMATION, this);
  return;
sql_error:
//
// some SQL error occurred
//
  sqlite3_finalize(stmt);
  wxMessageBox(wxT("dump HTML error:") +
               wxString::FromUTF8(sqlite3_errmsg(SqliteHandle)),
               wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
  if (out)
    fclose(out);
  return;
encoding_error:
//
// some CHARSET convertion occurred
//
  sqlite3_finalize(stmt);
  wxMessageBox(wxT("dump HTML: charset conversion reported an error"),
               wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
  if (out)
    fclose(out);
  return;
no_file:
//
// output file can't be created/opened
//
  wxMessageBox(wxT("ERROR: unable to open '") + path + wxT("' for writing"),
               wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
  return;
}
