/*
/ BlobExplorer.cpp
/ a dialog to explore a BLOB value
/
/ version 1.2, 2008 October 9
/
/ Author: Sandro Furieri a-furieri@lqt.it
/
/ Copyright (C) 2008  Alessandro Furieri
/
/    This program is free software: you can redistribute it and/or modify
/    it under the terms of the GNU General Public License as published by
/    the Free Software Foundation, either version 3 of the License, or
/    (at your option) any later version.
/
/    This program is distributed in the hope that it will be useful,
/    but WITHOUT ANY WARRANTY; without even the implied warranty of
/    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/    GNU General Public License for more details.
/
/    You should have received a copy of the GNU General Public License
/    along with this program.  If not, see <http://www.gnu.org/licenses/>.
/
*/

#include "Classdef.h"

#include "wx/mstream.h"
#include "wx/clipbrd.h"

BlobExplorerDialog::BlobExplorerDialog(MyFrame * parent, int blob_size,
                                       unsigned char *blob)
{
//
// constructor; just calls Create()
//
  Create(parent, blob_size, blob);
}

bool BlobExplorerDialog::Create(MyFrame * parent, int blob_size,
                                unsigned char *blob)
{
//
// creating the dialog
//
  MainFrame = parent;
  BlobSize = blob_size;
  Blob = blob;
  BlobType = gaiaGuessBlobType(Blob, BlobSize);
  Geometry = NULL;
  Image = NULL;
  if (BlobType == GAIA_GEOMETRY_BLOB)
    Geometry = gaiaFromSpatiaLiteBlobWkb(Blob, BlobSize);
  if (BlobType == GAIA_JPEG_BLOB || BlobType == GAIA_EXIF_BLOB
      || BlobType == GAIA_EXIF_GPS_BLOB || BlobType == GAIA_PNG_BLOB
      || BlobType == GAIA_GIF_BLOB)
    {
      ::wxInitAllImageHandlers();
      wxMemoryInputStream reader(Blob, BlobSize);
      Image = new wxImage(reader);
    }
  if (wxPropertySheetDialog::Create(parent, wxID_ANY, wxT("BLOB explorer")) ==
      false)
    return false;
  wxBookCtrlBase *book = GetBookCtrl();
// creates individual panels
  wxPanel *hexadecimal = CreateHexadecimalPage(book);
  book->AddPage(hexadecimal, wxT("Hexadecimal dump"), true);
  if (BlobType == GAIA_GEOMETRY_BLOB)
    {
      wxPanel *geometry = CreateGeometryPage(book);
      book->AddPage(geometry, wxT("Geometry explorer"), false);
    }
  if (BlobType == GAIA_JPEG_BLOB || BlobType == GAIA_EXIF_BLOB
      || BlobType == GAIA_EXIF_GPS_BLOB || BlobType == GAIA_PNG_BLOB
      || BlobType == GAIA_GIF_BLOB)
    {
      wxPanel *image = CreateImagePage(book);
      book->AddPage(image, wxT("Image"), false);
    }
  CreateButtons(wxOK);
  LayoutDialog();
// appends event handler for TAB/PAGE changing
  Connect(wxID_ANY, wxEVT_COMMAND_NOTEBOOK_PAGE_CHANGED,
          (wxObjectEventFunction) & BlobExplorerDialog::OnPageChanged);
// appends event handler for OK button
  Connect(wxID_OK, wxEVT_COMMAND_BUTTON_CLICKED,
          (wxObjectEventFunction) & BlobExplorerDialog::OnOk);
// centers the dialog window
  Centre();
  UpdateHexadecimalPage();
  return true;
}

wxPanel *BlobExplorerDialog::CreateHexadecimalPage(wxWindow * parent)
{
//
// creating the HEXADECIMAL page
//
  wxPanel *panel = new wxPanel(parent, ID_PANE_HEXADECIMAL);
  wxBoxSizer *topSizer = new wxBoxSizer(wxVERTICAL);
  panel->SetSizer(topSizer);
  wxBoxSizer *boxSizer = new wxBoxSizer(wxVERTICAL);
  topSizer->Add(boxSizer, 0, wxALIGN_CENTER | wxALL, 5);
// creating a control to show the hexadecimal dump
  wxBoxSizer *hexSizer = new wxBoxSizer(wxHORIZONTAL);
  boxSizer->Add(hexSizer, 0, wxALIGN_LEFT | wxALL, 0);
  MyHexList *hexCtrl = new MyHexList(this, Blob, BlobSize, panel,
                                     ID_HEX, wxDefaultPosition,
                                     wxSize(620, 320),
                                     wxLC_REPORT | wxLC_VIRTUAL);
  wxFont font(9, wxFONTFAMILY_MODERN, wxFONTSTYLE_NORMAL, wxFONTWEIGHT_NORMAL);
  hexCtrl->SetFont(font);
  wxListItem column0;
  hexCtrl->InsertColumn(0, wxT("Address"));
  hexCtrl->SetColumnWidth(0, 90);
  hexCtrl->InsertColumn(1, wxT("Hexadecimal"));
  hexCtrl->SetColumnWidth(1, 370);
  hexCtrl->InsertColumn(2, wxT("ASCII"));
  hexCtrl->SetColumnWidth(2, 130);
  hexSizer->Add(hexCtrl, 0, wxALIGN_CENTER_VERTICAL | wxALL, 5);
  panel->SetSizer(topSizer);
  topSizer->Fit(panel);
  return panel;
}

wxPanel *BlobExplorerDialog::CreateGeometryPage(wxWindow * parent)
{
//
// creating the GEOMETRY page
//
  wxPanel *panel = new wxPanel(parent, ID_PANE_HEXADECIMAL);
  wxBoxSizer *topSizer = new wxBoxSizer(wxVERTICAL);
  panel->SetSizer(topSizer);
  wxBoxSizer *boxSizer = new wxBoxSizer(wxVERTICAL);
  topSizer->Add(boxSizer, 0, wxALIGN_CENTER | wxALL, 5);
// creating a control to show the geometry as a text table
  wxBoxSizer *geomSizer = new wxBoxSizer(wxHORIZONTAL);
  boxSizer->Add(geomSizer, 0, wxALIGN_LEFT | wxALL, 0);
  wxTextCtrl *geomCtrl = new wxTextCtrl(panel, ID_GEOM_TABLE, wxT(""),
                                        wxDefaultPosition, wxSize(270,
                                                                  320),
                                        wxTE_MULTILINE | wxTE_RICH |
                                        wxTE_READONLY | wxHSCROLL);
  geomSizer->Add(geomCtrl, 0, wxALIGN_LEFT | wxALL, 5);
// creating a control to show the geometry in a graphical fashion
  wxStaticBox *exBox = new wxStaticBox(panel, wxID_STATIC,
                                       wxT("Geometry preview"),
                                       wxDefaultPosition, wxDefaultSize);
  wxBoxSizer *exampleSizer = new wxStaticBoxSizer(exBox, wxHORIZONTAL);
  geomSizer->Add(exampleSizer, 0, wxALIGN_CENTER_VERTICAL | wxALL, 5);
  DrawGeometry(300, 300);
  GraphicsGeometry *geomGraph = new GraphicsGeometry(this, panel, ID_GEOM_GRAPH,
                                                     GeomPreview, wxSize(300,
                                                                         300));
  exampleSizer->Add(geomGraph, 0, wxALIGN_CENTER_VERTICAL | wxALL, 0);
  panel->SetSizer(topSizer);
  topSizer->Fit(panel);
  return panel;
}

wxPanel *BlobExplorerDialog::CreateImagePage(wxWindow * parent)
{
//
// creating the IMAGE page
//
  wxPanel *panel = new wxPanel(parent, ID_PANE_IMAGE);
  wxBoxSizer *topSizer = new wxBoxSizer(wxVERTICAL);
  panel->SetSizer(topSizer);
  wxBoxSizer *boxSizer = new wxBoxSizer(wxVERTICAL);
  topSizer->Add(boxSizer, 0, wxALIGN_CENTER | wxALL, 5);
// creating a control to show the image title
  wxBoxSizer *imgSizer = new wxBoxSizer(wxVERTICAL);
  boxSizer->Add(imgSizer, 0, wxALIGN_TOP | wxALL, 0);
  wxStaticText *imageTitle = new wxStaticText(panel, ID_IMAGE_TITLE,
                                              wxT("Image"),
                                              wxDefaultPosition,
                                              wxSize(560,
                                                     10));
  imgSizer->Add(imageTitle, 0, wxALIGN_LEFT | wxALL, 5);
// creating a control to show the image
  wxStaticBox *exBox = new wxStaticBox(panel, ID_IMG_BOX,
                                       wxT("Image preview"),
                                       wxDefaultPosition, wxDefaultSize);
  wxBoxSizer *exampleSizer = new wxStaticBoxSizer(exBox, wxHORIZONTAL);
  imgSizer->Add(exampleSizer, 0, wxALIGN_CENTER_VERTICAL | wxALL, 5);
  ImageShow *imgShow = new ImageShow(this, panel, ID_IMAGE,
                                     wxBitmap(), wxSize(560, 300));
  exampleSizer->Add(imgShow, 0, wxALIGN_CENTER_VERTICAL | wxALL, 0);
  panel->SetSizer(topSizer);
  topSizer->Fit(panel);
  return panel;
}

void BlobExplorerDialog::OnPageChanged(wxNotebookEvent & event)
{
//
// TAB/PAGE selection changed
//
  switch (event.GetSelection())
    {
      case 0:
        UpdateHexadecimalPage();
        break;
      case 1:
        if (BlobType == GAIA_GEOMETRY_BLOB)
          UpdateGeometryPage();
        else
          UpdateImagePage();
        break;
    };
}

void BlobExplorerDialog::UpdateHexadecimalPage()
{
//
// updating the HEXADECIMAL page
//
  MyHexList *hexCtrl = (MyHexList *) FindWindow(ID_HEX);
  hexCtrl->EnsureVisible(0);
}

void BlobExplorerDialog::UpdateGeometryPage()
{
//
// updating the GEOMETRY page
//
  gaiaPointPtr pt;
  gaiaLinestringPtr ln;
  gaiaPolygonPtr pg;
  gaiaRingPtr rng;
  int points = 0;
  int linestrings = 0;
  int polygons = 0;
  int ib;
  wxString strValue;
  char dummy[1024];
  wxTextAttr attrBold(wxColour(0, 0, 0), wxColour(255, 255, 255),
                      wxFont(9, wxFONTFAMILY_MODERN, wxFONTSTYLE_NORMAL,
                             wxFONTWEIGHT_BOLD));
  wxTextAttr attrNorm(wxColour(0, 0, 0), wxColour(255, 255, 255),
                      wxFont(9, wxFONTFAMILY_MODERN, wxFONTSTYLE_NORMAL,
                             wxFONTWEIGHT_NORMAL));
  wxTextCtrl *geomCtrl = (wxTextCtrl *) FindWindow(ID_GEOM_TABLE);
  if (geomCtrl->GetValue().Len() < 1)
    {
      ::wxBeginBusyCursor();
      pt = Geometry->FirstPoint;
      while (pt)
        {
          // counting how many points are into this Geometry
          points++;
          pt = pt->Next;
        }
      ln = Geometry->FirstLinestring;
      while (ln)
        {
          // counting how many linestrings are into this Geometry
          linestrings++;
          ln = ln->Next;
        }
      pg = Geometry->FirstPolygon;
      while (pg)
        {
          // counting how many polygons are into this Geometry
          polygons++;
          pg = pg->Next;
        }
      // determining the Geometry type
      geomCtrl->SetDefaultStyle(attrNorm);
      geomCtrl->AppendText(wxT("SRID: "));
      geomCtrl->SetDefaultStyle(attrBold);
      sprintf(dummy, "%d", Geometry->Srid);
      strValue = wxString::FromUTF8(dummy);
      geomCtrl->AppendText(strValue);
      geomCtrl->SetDefaultStyle(attrNorm);
      geomCtrl->AppendText(wxT("\n\n"));
      strValue = wxT("UNKNOWN GEOMETRY TYPE");
      if (points == 1 && linestrings == 0 && polygons == 0)
        {
          if (Geometry->DeclaredType == GAIA_MULTIPOINT)
            strValue = wxT("MULTIPOINT");
          else if (Geometry->DeclaredType == GAIA_GEOMETRYCOLLECTION)
            strValue = wxT("GEOMETRYCOLLECTION");
          else
            strValue = wxT("POINT");
      } else if (points == 0 && linestrings == 1 && polygons == 0)
        {
          if (Geometry->DeclaredType == GAIA_MULTILINESTRING)
            strValue = wxT("MULTILINESTRING");
          else if (Geometry->DeclaredType == GAIA_GEOMETRYCOLLECTION)
            strValue = wxT("GEOMETRYCOLLECTION");
          else
            strValue = wxT("LINESTRING");
      } else if (points == 0 && linestrings == 0 && polygons == 1)
        {
          if (Geometry->DeclaredType == GAIA_MULTIPOLYGON)
            strValue = wxT("MULTIPOLYGON");
          else if (Geometry->DeclaredType == GAIA_GEOMETRYCOLLECTION)
            strValue = wxT("GEOMETRYCOLLECTION");
          else
            strValue = wxT("POLYGON");
      } else if (points > 1 && linestrings == 0 && polygons == 0)
        {
          if (Geometry->DeclaredType == GAIA_GEOMETRYCOLLECTION)
            strValue = wxT("GEOMETRYCOLLECTION");
          else
            strValue = wxT("MULTIPOINT");
      } else if (points == 0 && linestrings > 1 && polygons == 0)
        {
          if (Geometry->DeclaredType == GAIA_GEOMETRYCOLLECTION)
            strValue = wxT("GEOMETRYCOLLECTION");
          else
            strValue = wxT("MULTILINESTRING");
      } else if (points == 0 && linestrings == 0 && polygons > 1)
        {
          if (Geometry->DeclaredType == GAIA_GEOMETRYCOLLECTION)
            strValue = wxT("GEOMETRYCOLLECTION");
          else
            strValue = wxT("MULTIPOLYGON");
      } else
        strValue = wxT("GEOMETRYCOLLECTION");
      geomCtrl->SetDefaultStyle(attrNorm);
      geomCtrl->AppendText(wxT("Geometry type: "));
      geomCtrl->SetDefaultStyle(attrBold);
      geomCtrl->AppendText(strValue);
      geomCtrl->SetDefaultStyle(attrNorm);
      geomCtrl->AppendText(wxT("\n\n"));
      if (points)
        {
          // printing the Points list
          sprintf(dummy, "#%d POINT", points);
          strValue = wxString::FromUTF8(dummy);
          if (points > 1)
            strValue += wxT("s:");
          else
            strValue += wxT(":");
          geomCtrl->SetDefaultStyle(attrBold);
          geomCtrl->AppendText(strValue);
          geomCtrl->SetDefaultStyle(attrNorm);
          pt = Geometry->FirstPoint;
          points = 0;
          while (pt)
            {
              // printing each Point
              points++;
              sprintf(dummy, "\n  %d) ", points);
              strValue = wxString::FromUTF8(dummy);
              geomCtrl->SetDefaultStyle(attrBold);
              geomCtrl->AppendText(strValue);
              geomCtrl->SetDefaultStyle(attrNorm);
              sprintf(dummy, "%1.4lf  %1.4lf", pt->X, pt->Y);
              strValue = wxString::FromUTF8(dummy);
              geomCtrl->AppendText(strValue);
              pt = pt->Next;
            }
          geomCtrl->AppendText(wxT("\n\n\n"));
        }
      if (linestrings)
        {
          // printing the Linestrings list
          sprintf(dummy, "#%d LINESTRING", linestrings);
          strValue = wxString::FromUTF8(dummy);
          if (linestrings > 1)
            strValue += wxT("s:");
          else
            strValue += wxT(":");
          geomCtrl->SetDefaultStyle(attrBold);
          geomCtrl->AppendText(strValue);
          geomCtrl->SetDefaultStyle(attrNorm);
          ln = Geometry->FirstLinestring;
          linestrings = 0;
          while (ln)
            {
              // printing each Linestring
              linestrings++;
              sprintf(dummy, "\n  %d) ", linestrings);
              strValue = wxString::FromUTF8(dummy);
              geomCtrl->SetDefaultStyle(attrBold);
              geomCtrl->AppendText(strValue);
              geomCtrl->SetDefaultStyle(attrNorm);
              sprintf(dummy, "%d vertices", ln->Points);
              strValue = wxString::FromUTF8(dummy);
              geomCtrl->AppendText(strValue);
              ln = ln->Next;
            }
          geomCtrl->AppendText(wxT("\n\n\n"));
        }
      if (polygons)
        {
          // printing the Polygons list
          sprintf(dummy, "#%d POLYGON", polygons);
          strValue = wxString::FromUTF8(dummy);
          if (polygons > 1)
            strValue += wxT("s:");
          else
            strValue += wxT(":");
          geomCtrl->SetDefaultStyle(attrBold);
          geomCtrl->AppendText(strValue);
          geomCtrl->SetDefaultStyle(attrNorm);
          pg = Geometry->FirstPolygon;
          polygons = 0;
          while (pg)
            {
              // printing each Polygon
              polygons++;
              sprintf(dummy, "\n  %d)    exterior ring", polygons);
              strValue = wxString::FromUTF8(dummy);
              geomCtrl->SetDefaultStyle(attrBold);
              geomCtrl->AppendText(strValue);
              geomCtrl->SetDefaultStyle(attrNorm);
              rng = pg->Exterior;
              sprintf(dummy, ": %d vertices", rng->Points);
              strValue = wxString::FromUTF8(dummy);
              geomCtrl->AppendText(strValue);
              for (ib = 0; ib < pg->NumInteriors; ib++)
                {
                  // printing each interior ring
                  sprintf(dummy, "\n  %d.%d) ", polygons, ib + 1);
                  strValue = wxString::FromUTF8(dummy);
                  geomCtrl->SetDefaultStyle(attrBold);
                  geomCtrl->AppendText(strValue);
                  geomCtrl->SetDefaultStyle(attrNorm);
                  rng = pg->Interiors + ib;
                  sprintf(dummy, " interior ring: %d vertices", rng->Points);
                  strValue = wxString::FromUTF8(dummy);
                  geomCtrl->AppendText(strValue);
                }
              pg = pg->Next;
            }
          geomCtrl->AppendText(wxT("\n\n\n"));
        }
      ::wxEndBusyCursor();
    }
  GraphicsGeometry *geomGraph = (GraphicsGeometry *) FindWindow(ID_GEOM_GRAPH);
  geomGraph->SetBitmap(GeomPreview);
}

void BlobExplorerDialog::UpdateImagePage()
{
//
// updating the IMAGE page
//
  double horz;
  double vert;
  wxImage scaledImg;
  wxSize sz;
  wxSize box;
  int boxX;
  int boxY;
  int posX;
  int posY;
  char latlong[1024];
  char dummy[1024];
  wxString ll;
  wxString title = wxT("Invalid Image");
  wxStaticBox *imgBox = (wxStaticBox *) FindWindow(ID_IMG_BOX);
  ImageShow *imgShow = (ImageShow *) FindWindow(ID_IMAGE);
  wxStaticText *imageTitle = (wxStaticText *) FindWindow(ID_IMAGE_TITLE);
  if (Image)
    {
      ::wxBeginBusyCursor();
      if (Image->IsOk() == true)
        {
          horz = Image->GetWidth();
          vert = Image->GetHeight();
          sz = imgShow->GetSize();
          box = imgBox->GetSize();
          while (horz > sz.GetWidth() || vert > sz.GetHeight())
            {
              horz *= 0.9;
              vert *= 0.9;
            }
          if (horz == Image->GetWidth() && vert == Image->GetHeight())
            scaledImg = Image->Copy();
          else
            scaledImg =
              Image->Scale((int) horz, (int) vert, wxIMAGE_QUALITY_HIGH);
          wxBitmap bmp(scaledImg);
          imgBox->GetPosition(&boxX, &boxY);
          posX = (box.GetWidth() - (int) horz) / 2;
          posY = (box.GetHeight() - (int) vert) / 2;
          imgShow->SetSize(boxX + posX, boxY + posY, (int) horz, (int) vert);
          imgShow->SetBitmap(bmp);
          imgShow->Show(true);
          switch (BlobType)
            {
              case GAIA_JPEG_BLOB:
                sprintf(dummy,
                        "JPEG image     resolution: %d x %d          %d bytes",
                        Image->GetWidth(), Image->GetHeight(), BlobSize);
                title = wxString::FromUTF8(dummy);
                break;
              case GAIA_EXIF_BLOB:
                sprintf(dummy,
                        "EXIF image     resolution: %d x %d          %d bytes",
                        Image->GetWidth(), Image->GetHeight(), BlobSize);
                title = wxString::FromUTF8(dummy);
                break;
              case GAIA_EXIF_GPS_BLOB:
                if (gaiaGetGpsLatLong(Blob, BlobSize, latlong, 1024))
                  ll = wxString::FromUTF8(latlong);
                else
                  ll = wxT("NOT AVAILABLE");
                sprintf(dummy,
                        "EXIF-GPS image     resolution: %d x %d          %d bytes    GPS: ",
                        Image->GetWidth(), Image->GetHeight(), BlobSize);
                title = wxString::FromUTF8(dummy);
                title += ll;
                break;
              case GAIA_PNG_BLOB:
                sprintf(dummy,
                        "PNG image     resolution: %d x %d          %d bytes",
                        Image->GetWidth(), Image->GetHeight(), BlobSize);
                title = wxString::FromUTF8(dummy);
                break;
              case GAIA_GIF_BLOB:
                sprintf(dummy,
                        "GIF image     resolution: %d x %d          %d bytes",
                        Image->GetWidth(), Image->GetHeight(), BlobSize);
                title = wxString::FromUTF8(dummy);
                break;
            }
        }
      ::wxEndBusyCursor();
    }
  imageTitle->SetLabel(title);
}

void BlobExplorerDialog::OnOk(wxCommandEvent & event)
{
//
// all done: exiting
//
  wxDialog::EndModal(wxID_OK);
}

void BlobExplorerDialog::DrawGeometry(int horz, int vert)
{
//
// drawing graphic representation for current Geometry
//
  gaiaPointPtr pt;
  gaiaLinestringPtr ln;
  gaiaPolygonPtr pg;
  gaiaRingPtr rng;
  double minx;
  double miny;
  double maxx;
  double maxy;
  double ext_x;
  double ext_y;
  double cx;
  double cy;
  double pixelRatioX;
  double pixelRatioY;
  double pixelRatio;
  double span_x;
  double span_y;
  double baseX;
  double baseY;
  double x;
  double y;
  double xx;
  double yy;
  int iv;
  int ib;
  int pts;
  int *borders;
  wxPoint *points;
  GeomPreview.Create(horz, vert);
  wxMemoryDC dc(GeomPreview);
//
// background filling
//
  dc.SetBrush(wxBrush(wxColour(255, 255, 255)));
  dc.DrawRectangle(0, 0, horz, vert);
//
// prepearing the drawing pen and brush
//
  dc.SetBrush(wxBrush(wxColour(240, 240, 192)));
  dc.SetPen(wxPen(wxColour(64, 64, 192), 1));
//
// computing the pixel ratio, center position and so on
//
  minx = Geometry->MinX;
  miny = Geometry->MinY;
  maxx = Geometry->MaxX;
  maxy = Geometry->MaxY;
  ext_x = maxx - minx;
  ext_y = maxy - miny;
  if (ext_x < 1.0)
    ext_x = 1.0;
  if (ext_y < 1.0)
    ext_y = 1.0;
  minx = Geometry->MinX - (ext_x / 20.0);
  miny = Geometry->MinY - (ext_y / 20.0);
  maxx = Geometry->MaxX + (ext_x / 20.0);
  maxy = Geometry->MaxY + (ext_y / 20.0);
  ext_x = maxx - minx;
  ext_y = maxy - miny;
  cx = minx + (ext_x / 2.0);
  cy = miny + (ext_y / 2.0);
  pixelRatioX = ext_x / horz;
  pixelRatioY = ext_y / vert;
  if (pixelRatioX > pixelRatioY)
    pixelRatio = pixelRatioX;
  else
    pixelRatio = pixelRatioY;
//
// centering the Y axis
//
  span_y = vert * pixelRatio;
  baseY = cy - (span_y / 2.0);
//
// centering the X axis
//
  span_x = horz * pixelRatio;
  baseX = cx - (span_x / 2.0);
  pg = Geometry->FirstPolygon;
  while (pg)
    {
      //
      // drawing polygons
      //
      pts = pg->Exterior->Points;
      for (ib = 0; ib < pg->NumInteriors; ib++)
        {
          rng = pg->Interiors + ib;
          pts += rng->Points;
        }
      borders = new int[pg->NumInteriors + 1];
      points = new wxPoint[pts];
      pts = 0;
      rng = pg->Exterior;
      borders[0] = rng->Points;
      for (iv = 0; iv < rng->Points; iv++)
        {
          gaiaGetPoint(rng->Coords, iv, &x, &y);
          xx = (x - baseX) / pixelRatio;
          yy = (y - baseY) / pixelRatio;
          yy = vert - yy;
          points[pts].x = (int) xx;
          points[pts].y = (int) yy;
          pts++;
        }
      for (ib = 0; ib < pg->NumInteriors; ib++)
        {
          rng = pg->Interiors + ib;
          borders[1 + ib] = rng->Points;
          for (iv = 0; iv < rng->Points; iv++)
            {
              gaiaGetPoint(rng->Coords, iv, &x, &y);
              xx = (x - baseX) / pixelRatio;
              yy = (y - baseY) / pixelRatio;
              yy = vert - yy;
              points[pts].x = (int) xx;
              points[pts].y = (int) yy;
              pts++;
            }
        }
      dc.DrawPolyPolygon(pg->NumInteriors + 1, borders, points);
      delete[]points;
      delete[]borders;
      pg = pg->Next;
    }
  ln = Geometry->FirstLinestring;
  while (ln)
    {
      //
      // drawing linestrings
      //
      points = new wxPoint[ln->Points];
      for (iv = 0; iv < ln->Points; iv++)
        {
          gaiaGetPoint(ln->Coords, iv, &x, &y);
          xx = (x - baseX) / pixelRatio;
          yy = (y - baseY) / pixelRatio;
          yy = vert - yy;
          points[iv].x = (int) xx;
          points[iv].y = (int) yy;
        }
      dc.DrawLines(ln->Points, points);
      delete[]points;
      ln = ln->Next;
    }
  pt = Geometry->FirstPoint;
  while (pt)
    {
      //
      // drawing points
      //
      xx = (pt->X - baseX) / pixelRatio;
      yy = (pt->Y - baseY) / pixelRatio;
      yy = vert - yy;
      dc.DrawCircle((int) xx, (int) yy, 2);
      pt = pt->Next;
    }
}

GraphicsGeometry::GraphicsGeometry(BlobExplorerDialog * parent,
                                   wxWindow * panel, wxWindowID id,
                                   const wxBitmap & bmp,
                                   const wxSize & size):wxStaticBitmap(panel,
                                                                       id, bmp,
                                                                       wxDefaultPosition,
                                                                       size)
{
  Parent = parent;
}

ImageShow::ImageShow(BlobExplorerDialog * parent, wxWindow * panel,
                     wxWindowID id, const wxBitmap & bmp,
                     const wxSize & size):wxStaticBitmap(panel, id, bmp,
                                                         wxDefaultPosition,
                                                         size)
{
  Parent = parent;
// appends event handler 
  Connect(ID_IMAGE, wxEVT_RIGHT_DOWN,
          (wxObjectEventFunction) & ImageShow::OnRightClick);
  Connect(Image_Copy, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & ImageShow::OnCmdCopy);
}

void ImageShow::OnRightClick(wxMouseEvent & event)
{
//
// right click on the Image
//
  wxMenu *menu = new wxMenu();
  wxMenuItem *menuItem;
  wxImage *Image = Parent->GetImage();
  if (Image)
    {
      if (Image->IsOk() == true)
        {
          wxPoint pt = event.GetPosition();
          menuItem = new wxMenuItem(menu, Image_Copy, wxT("&Copy"));
          menu->Append(menuItem);
          PopupMenu(menu, pt);
        }
    }
}

void ImageShow::OnCmdCopy(wxCommandEvent & event)
{
//
// copying the Image into the clipboard
//
  wxImage *Image = Parent->GetImage();
  if (wxTheClipboard->Open())
    {
      wxTheClipboard->SetData(new wxBitmapDataObject(*Image));
      wxTheClipboard->Close();
    }
}

MyHexList::MyHexList(BlobExplorerDialog * parent, unsigned char *blob,
                     int blob_size, wxWindow * panel, wxWindowID id,
                     const wxPoint & pos, const wxSize & size,
                     long style):wxListCtrl(panel, id, pos, size, style)
{
// constructor - the blob hexadecimal dump
  int i = 0;
  int rows = 0;
  Parent = parent;
  Blob = blob;
  BlobSize = blob_size;
  while (i < BlobSize)
    {
      // counting how many rows are there
      rows++;
      i += 16;
    }
  SetItemCount(rows);
}

MyHexList::~MyHexList()
{
// does nothing
}

wxString MyHexList::OnGetItemText(long item, long column) const
{
// return a column value
  int i;
  int c;
  int base = item * 16;
  wxString value;
  char dummy[64];
  wxString hex;
  if (column == 0)
    {
      sprintf(dummy, "%08xd", base);
      value = wxString::FromUTF8(dummy);
  } else if (column == 1)
    {
      // prepearing the hex-dump
      c = 0;
      for (i = base; i < BlobSize; i++)
        {
          if (c >= 16)
            break;
          sprintf(dummy, "%02x", *(Blob + i));
          hex = wxString::FromUTF8(dummy);
          if (c == 8)
            value += wxT("   ");
          else
            value += wxT(" ");
          value += hex;
          c++;
        }
  } else
    {
      // prepearing the ascii dump
      c = 0;
      for (i = base; i < BlobSize; i++)
        {
          if (c >= 16)
            break;
          if (isprint(*(Blob + i)))
            {
              sprintf(dummy, "%c", *(Blob + i));
              hex = wxString::FromUTF8(dummy);
          } else
            hex = wxT(".");
          value += hex;
          c++;
        }
    }
  return value;
}
