-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (SparkLex.Lex)
procedure GetIdent (Token : out SPSymbols.SPTerminal)

-- Given that the current position in the line buffer, CurrLine, indexes a
-- letter, GetIdent forward scans the input line buffer to locate the end of
-- the identifier.
-- The extent of the identifier is determined by the
-- maximal length substring starting at the current position accepted by one
-- of the following two regular expressions :-
--    identifier ::= letter {[underline] letter_or_digit}
--    illegal_id ::= identifier non_id_separator {non_id_separator}
-- where
--    non_id_separator is any character which cannot legally separate an
--    identifier from the next token in the Ada Lexis.
-- The expression which scans the longest substring is the one that is chosen.
-- The value of Token is set according to which expression was used.
-- GetIdent selects the correct expression without backtracking.
is
   type IdState is (UnderlineLetterOrDigit, LetterOrDigitOnly, IllegalState);
   State : IdState;
   Ch    : Character;

   function IdSeparator (Ch : Character) return Boolean is
      Result : Boolean;
   begin
      Result := Ch = ' '
        or else Ch = End_Of_Line
        or else Ch = End_Of_Text
        or else Ch = Line_Too_Long
        or else Format_Effector (Ch => Ch)
        or else Simple_Delimiter (Ch => Ch);
      return Result;
   end IdSeparator;

begin
   LineManager.Accept_Char;  -- The first letter has already been parsed
   State := UnderlineLetterOrDigit;

   loop
      LineManager.Inspect_Char (Ch => Ch);
      exit when IdSeparator (Ch);
      LineManager.Accept_Char;
      case State is
         when UnderlineLetterOrDigit =>
            -- Modified following to eliminate reference to tilde and
            -- also to disallow reference to double underbar in proof context
            -- this second reverses an undocumented change in the tool
            if Letter_Or_Digit (Ch => Ch) then
               null;
            elsif Ch = '_' then
               State := LetterOrDigitOnly;
            else
               State := IllegalState;
            end if;
         when LetterOrDigitOnly =>
            if Letter_Or_Digit (Ch => Ch) then
               State := UnderlineLetterOrDigit;
            else
               State := IllegalState;
            end if;
         when IllegalState =>
            null;
      end case;
   end loop;

   --# assert State in IdState;
   case State is
      when UnderlineLetterOrDigit =>
         Token := SPSymbols.identifier;
      when LetterOrDigitOnly =>
         Token := SPSymbols.illegal_id;
      when IllegalState =>
         Token := SPSymbols.illegal_id;
   end case;
end GetIdent;
