-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure Wf_Subprogram_Body (Node      : in     STree.SyntaxNode;
                              Scope     : in out Dictionary.Scopes;
                              Next_Node :    out STree.SyntaxNode) is
   type Generic_Kinds is (Generic_Procedure, Generic_Function);

   Ident_Node, Spec_Node, Formal_Part_Node, Constraint_Node, Anno_Node, Main_Node, With_Node, Subprog_Implem_Node, End_Desig_Node :
     STree.SyntaxNode;
   Subprog_Sym                                                                                                                    :
     Dictionary.Symbol;
   Hidden                                                                                                                         :
     Hidden_Class;
   Is_Generic, First_Seen                                                                                                         :
     Boolean;
   Subprog_Scope                                                                                                                  :
     Dictionary.Scopes;
   Is_Overriding                                                                                                                  :
     Boolean := False;
   Scope_For_Formal_Part_Check                                                                                                    :
     Dictionary.Scopes;

   ------------------------------------------------------------------

   procedure ProcessPartitionAnnotation (Main_Node : in STree.SyntaxNode;
                                         Scope     : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Dictionary.Dict,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Main_Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Main_Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   ------------------------------------------------------------------

   procedure Shared_Variable_Check
     (Main_Program_Sym : in Dictionary.Symbol;
      Scope            : in Dictionary.Scopes;
      Error_Node_Pos   : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict            from *,
   --#                                         Main_Program_Sym &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Error_Node_Pos,
   --#                                         LexTokenManager.State,
   --#                                         Main_Program_Sym,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys;
   is
      Inherited_Package_It  : Dictionary.Iterator;
      Inherited_Package_Sym : Dictionary.Symbol;
      It                    : Dictionary.Iterator;
      Sym                   : Dictionary.Symbol;
      Type_Sym              : Dictionary.Symbol;

      ------------------------------------------------------------------

      procedure Check_Unprotected_Globals
        (Check_List                  : in Dictionary.Iterator;
         The_Thread                  : in Dictionary.Symbol;
         Annotations_Are_Well_Formed : in Boolean;
         Scope                       : in Dictionary.Scopes;
         Error_Node_Pos              : in LexTokenManager.Token_Position)
      --# global in     CommandLineData.Content;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dictionary.Dict            from *,
      --#                                         Check_List,
      --#                                         The_Thread &
      --#         ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from Annotations_Are_Well_Formed,
      --#                                         Check_List,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         Error_Node_Pos,
      --#                                         LexTokenManager.State,
      --#                                         Scope,
      --#                                         SPARK_IO.File_Sys,
      --#                                         The_Thread;
      is
         It           : Dictionary.Iterator;
         Sym          : Dictionary.Symbol;
         Other_Thread : Dictionary.Symbol;
      begin
         It := Check_List;
         while It /= Dictionary.NullIterator loop
            Sym := Dictionary.CurrentSymbol (It);
            if Sym /= Dictionary.GetNullVariable then
               if not Dictionary.GetOwnVariableProtected (Sym) then
                  Other_Thread := Dictionary.GetUnprotectedReference (Sym);
                  if Other_Thread /= Dictionary.NullSymbol then
                     -- This is non-protected global variable that is being
                     -- accessed by more than one thread of control.
                     ErrorHandler.Semantic_Error_Sym3
                       (Err_Num   => 938,
                        Reference => ErrorHandler.No_Reference,
                        Position  => Error_Node_Pos,
                        Sym       => Sym,
                        Sym2      => Other_Thread,
                        Sym3      => The_Thread,
                        Scope     => Scope);
                  else
                     -- Mark this global variable as being accessed by a thread.
                     Dictionary.SetUnprotectedReference (Sym, The_Thread);
                  end if;
               end if;
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
         if not Annotations_Are_Well_Formed then
            -- The thread has errors in the annotations and so the shared variable check
            -- may not be complete.
            ErrorHandler.Semantic_Warning_Sym (Err_Num  => 413,
                                               Position => Error_Node_Pos,
                                               Sym      => The_Thread,
                                               Scope    => Scope);
         end if;
      end Check_Unprotected_Globals;

   begin -- Shared_Variable_Check

      -- Look for access to unprotected globals by the main program
      Check_Unprotected_Globals
        (Check_List                  => Dictionary.FirstGlobalVariable (Dictionary.IsAbstract, Main_Program_Sym),
         The_Thread                  => Main_Program_Sym,
         Annotations_Are_Well_Formed => Dictionary.SubprogramSignatureIsWellformed (Dictionary.IsAbstract, Main_Program_Sym),
         Scope                       => Scope,
         Error_Node_Pos              => Error_Node_Pos);

      -- Look for access to unprotected globals by all tasks.
      Inherited_Package_It := Dictionary.FirstInheritsClause (Main_Program_Sym);
      while Inherited_Package_It /= Dictionary.NullIterator loop
         Inherited_Package_Sym := Dictionary.CurrentSymbol (Inherited_Package_It);
         It                    := Dictionary.FirstOwnTask (Inherited_Package_Sym);
         while It /= Dictionary.NullIterator loop
            Sym      := Dictionary.CurrentSymbol (It);
            Type_Sym := Dictionary.GetRootType (Dictionary.GetType (Sym));
            if Dictionary.IsDeclared (Type_Sym) then
               if Dictionary.UsesUnprotectedVariables (Type_Sym) then
                  Check_Unprotected_Globals
                    (Check_List                  => Dictionary.FirstGlobalVariable (Dictionary.IsAbstract, Type_Sym),
                     The_Thread                  => Sym,
                     Annotations_Are_Well_Formed => Dictionary.SubprogramSignatureIsWellformed (Dictionary.IsAbstract, Type_Sym),
                     Scope                       => Scope,
                     Error_Node_Pos              => Error_Node_Pos);
               end if;
            elsif not Dictionary.IsUnknownTypeMark (Type_Sym) then
               -- The task type is not available and hence we cannot perform
               -- the shared variable check for this task.
               ErrorHandler.Semantic_Warning_Sym (Err_Num  => 411,
                                                  Position => Error_Node_Pos,
                                                  Sym      => Type_Sym,
                                                  Scope    => Scope);
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
         Inherited_Package_It := Dictionary.NextSymbol (Inherited_Package_It);
      end loop;
   end Shared_Variable_Check;

   ------------------------------------------------------------------

   procedure Max_One_In_A_Queue_Check
     (Main_Program_Sym : in Dictionary.Symbol;
      Scope            : in Dictionary.Scopes;
      Error_Node_Pos   : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict            from *,
   --#                                         Main_Program_Sym &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Error_Node_Pos,
   --#                                         LexTokenManager.State,
   --#                                         Main_Program_Sym,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys;
   is
      Inherited_Package_It  : Dictionary.Iterator;
      Inherited_Package_Sym : Dictionary.Symbol;
      It                    : Dictionary.Iterator;
      Sym                   : Dictionary.Symbol;
      Type_Sym              : Dictionary.Symbol;

      ------------------------------------------------------------------

      procedure Check_Suspends_Items
        (Check_List     : in Dictionary.Iterator;
         The_Thread     : in Dictionary.Symbol;
         Scope          : in Dictionary.Scopes;
         Error_Node_Pos : in LexTokenManager.Token_Position)
      --# global in     CommandLineData.Content;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dictionary.Dict            from *,
      --#                                         Check_List,
      --#                                         The_Thread &
      --#         ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from Check_List,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         Error_Node_Pos,
      --#                                         LexTokenManager.State,
      --#                                         Scope,
      --#                                         SPARK_IO.File_Sys,
      --#                                         The_Thread;
      is
         It           : Dictionary.Iterator;
         Sym          : Dictionary.Symbol;
         Other_Thread : Dictionary.Symbol;
      begin
         It := Check_List;
         while It /= Dictionary.NullIterator loop
            Sym          := Dictionary.CurrentSymbol (It);
            Other_Thread := Dictionary.GetSuspendsReference (Sym);
            if Other_Thread /= Dictionary.NullSymbol then
               -- This is a suspendable entity that is being
               -- accessed by more than one thread of control.
               ErrorHandler.Semantic_Error_Sym3
                 (Err_Num   => 939,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Error_Node_Pos,
                  Sym       => Sym,
                  Sym2      => Other_Thread,
                  Sym3      => The_Thread,
                  Scope     => Scope);
            else
               -- Mark this suspends item as being accessed by a thread.
               Dictionary.SetSuspendsReference (Sym, The_Thread);
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
      end Check_Suspends_Items;

   begin -- Max_One_In_A_Queue_Check

      -- Look for suspendable entities in the main program
      Check_Suspends_Items
        (Check_List     => Dictionary.FirstSuspendsListItem (Main_Program_Sym),
         The_Thread     => Main_Program_Sym,
         Scope          => Scope,
         Error_Node_Pos => Error_Node_Pos);

      -- Look for suspendable entities in all the tasks.
      -- Note. interrupt handlers cannot call operations that suspend.
      Inherited_Package_It := Dictionary.FirstInheritsClause (Main_Program_Sym);
      while Inherited_Package_It /= Dictionary.NullIterator loop
         Inherited_Package_Sym := Dictionary.CurrentSymbol (Inherited_Package_It);
         It                    := Dictionary.FirstOwnTask (Inherited_Package_Sym);
         while It /= Dictionary.NullIterator loop
            Sym      := Dictionary.CurrentSymbol (It);
            Type_Sym := Dictionary.GetRootType (Dictionary.GetType (Sym));
            if Dictionary.IsDeclared (Type_Sym) then
               Check_Suspends_Items
                 (Check_List     => Dictionary.FirstSuspendsListItem (Type_Sym),
                  The_Thread     => Sym,
                  Scope          => Scope,
                  Error_Node_Pos => Error_Node_Pos);
            elsif not Dictionary.IsUnknownTypeMark (Type_Sym) then
               -- The task type is not available and hence we cannot perform
               -- the max-one-in-a-queue check for this task.
               ErrorHandler.Semantic_Warning_Sym (Err_Num  => 412,
                                                  Position => Error_Node_Pos,
                                                  Sym      => Type_Sym,
                                                  Scope    => Scope);
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
         Inherited_Package_It := Dictionary.NextSymbol (Inherited_Package_It);
      end loop;
   end Max_One_In_A_Queue_Check;

   ------------------------------------------------------------------

   procedure Wf_Main_Program
     (Node                 : in STree.SyntaxNode;
      Subprog_Sym          : in Dictionary.Symbol;
      Scope, Subprog_Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Dictionary.Dict,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         Subprog_Scope,
   --#                                         Subprog_Sym,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Subprog_Scope,
   --#                                         Subprog_Sym,
   --#                                         TheHeap;
   is
      Context_Node, Inherit_Node : STree.SyntaxNode;

      ------------------------------------------------------------------

      procedure Check_Program_Completeness
        (Node_Pos    : in LexTokenManager.Token_Position;
         Subprog_Sym : in Dictionary.Symbol;
         Scope       : in Dictionary.Scopes)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         LexTokenManager.State,
      --#                                         Node_Pos,
      --#                                         Scope,
      --#                                         SPARK_IO.File_Sys,
      --#                                         Subprog_Sym;
      is
         Inherit_It        : Dictionary.Iterator;
         Inherited_Package : Dictionary.Symbol;

         function Contains_Task (The_Package : Dictionary.Symbol) return Boolean
         --# global in Dictionary.Dict;
         is
         begin
            return not Dictionary.IsNullIterator (Dictionary.FirstOwnTask (The_Package));
         end Contains_Task;

         function Contains_Interrupt (The_Package : Dictionary.Symbol) return Boolean
         --# global in Dictionary.Dict;
         is
            Result : Boolean := False;
            It     : Dictionary.Iterator;
         begin
            It := Dictionary.FirstOwnVariable (The_Package);
            while not Dictionary.IsNullIterator (It) loop
               Result := Dictionary.GetHasInterruptProperty (Dictionary.CurrentSymbol (It));
               exit when Result;
               It := Dictionary.NextSymbol (It);
            end loop;
            return Result;
         end Contains_Interrupt;

      begin -- Check_Program_Completeness
         Inherit_It := Dictionary.FirstInheritsClause (Subprog_Sym);
         while not Dictionary.IsNullIterator (Inherit_It) loop
            Inherited_Package := Dictionary.CurrentSymbol (Inherit_It);
            if Contains_Task (The_Package => Inherited_Package)
              or else Contains_Interrupt (The_Package => Inherited_Package) then
               -- then it must also be WITHed to ensure program completeness
               if not Dictionary.IsWithed (Inherited_Package, Scope) then
                  ErrorHandler.Semantic_Error_Sym
                    (Err_Num   => 951,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Pos,
                     Sym       => Inherited_Package,
                     Scope     => Scope);
               end if;
            end if;
            Inherit_It := Dictionary.NextSymbol (Inherit_It);
         end loop;
      end Check_Program_Completeness;

   begin -- Wf_Main_Program

      -- ASSUME Node = main_program_declaration
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.main_program_declaration,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = main_program_declaration in Wf_Main_Program");

      if not Dictionary.MainProgramExists then
         Dictionary.AddMainProgram
           (Subprog_Sym,
            Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                 End_Position   => Node_Position (Node => Node)));

         Inherit_Node := Child_Node (Current_Node => Node);
         -- ASSUME Inherit_Node = inherit_clause OR main_program_annotation
         if Syntax_Node_Type (Node => Inherit_Node) = SPSymbols.inherit_clause then
            -- ASSUME Inherit_Node = inherit_clause
            Wf_Inherit_Clause (Inherit_Node, Subprog_Sym, Scope);
         elsif Syntax_Node_Type (Node => Inherit_Node) /= SPSymbols.main_program_annotation then
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Inherit_Node = inherit_clause OR main_program_annotation in Wf_Main_Program");
         end if;

         Context_Node := Child_Node (Current_Node => Parent_Node (Current_Node => Parent_Node (Current_Node => Node)));
         -- ASSUME Context_Node = context_clause OR library_unit
         if Syntax_Node_Type (Node => Context_Node) = SPSymbols.context_clause then
            -- ASSUME Context_Node = context_clause
            Wf_Context_Clause (Context_Node, Subprog_Sym, Subprog_Scope);
         elsif Syntax_Node_Type (Node => Context_Node) /= SPSymbols.library_unit then
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Context_Node = context_clause OR library_unit in Wf_Main_Program");
         end if;

         -- check here, in Ravencar, that all inherited packages with tasks/interrupts are also WITHed
         if Syntax_Node_Type (Node => Inherit_Node) = SPSymbols.inherit_clause and then CommandLineData.Ravenscar_Selected then
            Check_Program_Completeness
              (Node_Pos    => Node_Position (Node => Inherit_Node),
               Subprog_Sym => Subprog_Sym,
               Scope       => Subprog_Scope);
         end if;

         -- in Ravencar mode, a main program may have an addition partition flow analysis annotation
         ProcessPartitionAnnotation (Main_Node => Node,
                                     Scope     => Scope);

      else  -- Dictionary.MainProgramExists
         ErrorHandler.Semantic_Error
           (Err_Num   => 313,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => LexTokenManager.Null_String);
      end if;

   end Wf_Main_Program;

   ------------------------------------------------------------------

   function Requires_Second_Annotation (Subprog_Sym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
      Global_Var       : Dictionary.Symbol;
      Required         : Boolean;
      Global_Item      : Dictionary.Iterator;
      Enclosing_Region : Dictionary.Symbol;
   begin
      Required := False;
      if not Dictionary.IsGlobalScope (Dictionary.GetScope (Subprog_Sym)) then
         Enclosing_Region := Dictionary.GetRegion (Dictionary.GetScope (Subprog_Sym));
         if Dictionary.IsPackage (Enclosing_Region)
           or else (Dictionary.IsType (Enclosing_Region) and then Dictionary.TypeIsProtected (Enclosing_Region)) then
            Global_Item := Dictionary.FirstGlobalVariable (Dictionary.IsAbstract, Subprog_Sym);
            while Global_Item /= Dictionary.NullIterator loop
               Global_Var := Dictionary.CurrentSymbol (Global_Item);
               if Dictionary.IsRefinedOwnVariable (Global_Var) and then Dictionary.GetOwner (Global_Var) = Enclosing_Region then
                  Required := True;
                  exit;
               end if;
               Global_Item := Dictionary.NextSymbol (Global_Item);
            end loop;
         end if;
      end if;
      return Required;
   end Requires_Second_Annotation;

   ----------------------------------------------------------------------

   procedure Check_Function_Has_Return
     (Subprog_Node       : in STree.SyntaxNode;
      End_Desig_Node_Pos : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         End_Desig_Node_Pos,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Subprog_Node;
   is
      Next_Node : STree.SyntaxNode;
   begin
      -- ASSUME Subprog_Node = subprogram_implementation
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Subprog_Node) = SPSymbols.subprogram_implementation,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Subprog_Node = subprogram_implementation in Check_Function_Has_Return");

      Next_Node := Next_Sibling (Current_Node => Child_Node (Current_Node => Subprog_Node));
      -- Skip over declarative_part if there is one
      -- ASSUME Next_Node = declarative_part OR sequence_of_statements OR code_insertion OR hidden_part
      if Syntax_Node_Type (Node => Next_Node) = SPSymbols.declarative_part then
         -- ASSUME Next_Node = declarative_part
         Next_Node := Next_Sibling (Current_Node => Next_Node);
      elsif Syntax_Node_Type (Node => Next_Node) /= SPSymbols.sequence_of_statements
        and then Syntax_Node_Type (Node => Next_Node) /= SPSymbols.code_insertion
        and then Syntax_Node_Type (Node => Next_Node) /= SPSymbols.hidden_part then
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Next_Node = declarative_part OR sequence_of_statements OR code_insertion OR hidden_part in Check_Function_Has_Return");
      end if;
      -- ASSUME Next_Node = sequence_of_statements OR code_insertion OR hidden_part
      if Syntax_Node_Type (Node => Next_Node) = SPSymbols.sequence_of_statements then
         -- ASSUME Next_Node = sequence_of_statements
         Next_Node := Child_Node (Current_Node => Next_Node);

         -- Now we have a sequence_of_statements which can be reduced to:
         -- sequence_of_statements statement | statement ;
         -- (See SPARK.LLA)
         -- If the sequence_of_statements is a sequence_of_statements followed by
         -- a statement then skip to the statement (which will be the final statement
         -- in the subprogram).
         -- ASSUME Next_Node = sequence_of_statements OR statement
         if Syntax_Node_Type (Node => Next_Node) = SPSymbols.sequence_of_statements then
            -- ASSUME Next_Node = sequence_of_statements
            Next_Node := Next_Sibling (Current_Node => Next_Node);
         elsif Syntax_Node_Type (Node => Next_Node) /= SPSymbols.statement then
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Next_Node = sequence_of_statements OR statement in Check_Function_Has_Return");
         end if;
         -- ASSUME Next_Node = statement
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Next_Node) = SPSymbols.statement,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Next_Node = statement in Check_Function_Has_Return");

         -- The final statement in the subprogram should be a return statement, but we
         -- need to cater for labels because a statement can be reduced to:
         -- simple_statement | sequence_of_labels simple_statement ...
         -- (and a simple_statement can be reduced to a return_statement).

         -- The child node will either be a simple_statement or a sequence_of_labels
         Next_Node := Child_Node (Current_Node => Next_Node);
         -- Skip the label(s) if present.
         -- ASSUME Next_Node = sequence_of_labels OR simple_statement OR compound_statement OR
         --                    proof_statement OR justification_statement OR apragma
         if Syntax_Node_Type (Node => Next_Node) = SPSymbols.sequence_of_labels then
            -- ASSUME Next_Node = sequence_of_labels
            Next_Node := Next_Sibling (Next_Node);
         elsif Syntax_Node_Type (Node => Next_Node) /= SPSymbols.simple_statement
           and then Syntax_Node_Type (Node => Next_Node) /= SPSymbols.compound_statement
           and then Syntax_Node_Type (Node => Next_Node) /= SPSymbols.proof_statement
           and then Syntax_Node_Type (Node => Next_Node) /= SPSymbols.justification_statement
           and then Syntax_Node_Type (Node => Next_Node) /= SPSymbols.apragma then
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Next_Node = sequence_of_labels OR simple_statement OR compound_statement OR proof_statement OR justification_statement OR apragma in Check_Function_Has_Return");
         end if;
         -- ASSUME Next_Node = simple_statement OR compound_statement OR proof_statement OR justification_statement OR apragma
         -- Now we have reached the final statement in the subprogram. This should be
         -- a return statement.
         if Syntax_Node_Type (Node => Next_Node) = SPSymbols.simple_statement then
            -- ASSUME Next_Node = simple_statement
            if Syntax_Node_Type (Node => Child_Node (Current_Node => Next_Node)) = SPSymbols.null_statement
              or else Syntax_Node_Type (Node => Child_Node (Current_Node => Next_Node)) = SPSymbols.assignment_statement
              or else Syntax_Node_Type (Node => Child_Node (Current_Node => Next_Node)) = SPSymbols.procedure_call_statement
              or else Syntax_Node_Type (Node => Child_Node (Current_Node => Next_Node)) = SPSymbols.exit_statement
              or else Syntax_Node_Type (Node => Child_Node (Current_Node => Next_Node)) = SPSymbols.delay_statement then
               -- ASSUME Child_Node (Current_Node => Next_Node) = null_statement OR assignment_statement OR
               --                                                 procedure_call_statement OR exit_statement OR delay_statement
               ErrorHandler.Control_Flow_Error (Err_Type => ErrorHandler.Missing_Return,
                                                Position => End_Desig_Node_Pos);
            elsif Syntax_Node_Type (Node => Child_Node (Current_Node => Next_Node)) /= SPSymbols.return_statement then
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                  Msg     => "Expect Child_Node (Current_Node => Next_Node) = null_statement OR assignment_statement OR procedure_call_statement OR exit_statement OR return_statement OR delay_statement in Check_Function_Has_Return");
            end if;
         elsif Syntax_Node_Type (Node => Next_Node) = SPSymbols.compound_statement
           or else Syntax_Node_Type (Node => Next_Node) = SPSymbols.proof_statement
           or else Syntax_Node_Type (Node => Next_Node) = SPSymbols.justification_statement
           or else Syntax_Node_Type (Node => Next_Node) = SPSymbols.apragma then
            -- ASSUME Next_Node = compound_statement OR proof_statement OR justification_statement OR apragma
            ErrorHandler.Control_Flow_Error (Err_Type => ErrorHandler.Missing_Return,
                                             Position => End_Desig_Node_Pos);
         else
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Next_Node = simple_statement OR compound_statement OR proof_statement OR justification_statement OR apragma in Check_Function_Has_Return");
         end if;
      elsif Syntax_Node_Type (Node => Next_Node) /= SPSymbols.code_insertion
        and then Syntax_Node_Type (Node => Next_Node) /= SPSymbols.hidden_part then
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Next_Node = sequence_of_statements OR code_insertion OR hidden_part in Check_Function_Has_Return");
      end if;
   end Check_Function_Has_Return;

   ----------------------------------------------------------------------

   procedure Get_Anno_And_Con_Nodes
     (Node            : in     STree.SyntaxNode;
      Anno_Node       :    out STree.SyntaxNode;
      Constraint_Node :    out STree.SyntaxNode)
   --# global in STree.Table;
   --# derives Anno_Node,
   --#         Constraint_Node from Node,
   --#                              STree.Table;
   is
   begin
      -- ASSUME Node = procedure_annotation OR function_annotation
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.procedure_annotation
           or else Syntax_Node_Type (Node => Node) = SPSymbols.function_annotation,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = procedure_annotation OR function_annotation in Get_Anno_And_Con_Nodes");
      Constraint_Node := Child_Node (Current_Node => Node);
      -- ASSUME Constraint_Node = moded_global_definition OR dependency_relation OR declare_annotation OR
      --                          procedure_constraint OR function_constraint
      if Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.function_constraint
        or else Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.procedure_constraint then
         -- ASSUME Constraint_Node = function_constraint OR procedure_constraint
         Anno_Node := STree.NullNode; -- only a constraint found
      elsif Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.moded_global_definition
        or else Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.dependency_relation
        or else Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.declare_annotation then
         -- ASSUME Constraint_Node = moded_global_definition OR dependency_relation OR declare_annotation
         Anno_Node       := Node;
         Constraint_Node := Last_Sibling_Of (Start_Node => Constraint_Node);
      else
         Anno_Node       := STree.NullNode;
         Constraint_Node := STree.NullNode;
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Constraint_Node = moded_global_definition OR dependency_relation OR declare_annotation OR procedure_constraint OR function_constraint in Get_Anno_And_Con_Nodes");
      end if;
      -- ASSUME Anno_Node = procedure_annotation OR function_annotation OR NULL
      SystemErrors.RT_Assert
        (C       => Anno_Node = STree.NullNode
           or else Syntax_Node_Type (Node => Anno_Node) = SPSymbols.procedure_annotation
           or else Syntax_Node_Type (Node => Anno_Node) = SPSymbols.function_annotation,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Anno_Node = procedure_annotation OR function_annotation OR NULL in Get_Anno_And_Con_Nodes");
      -- ASSUME Constraint_Node = function_constraint OR procedure_constraint
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.function_constraint
           or else Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.procedure_constraint,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Constraint_Node = function_constraint OR procedure_constraint in Get_Anno_And_Con_Nodes");
   end Get_Anno_And_Con_Nodes;

   ----------------------------------------------------------------------

   procedure Check_Generic_Specification
     (Node        : in     STree.SyntaxNode;
      Kind        : in     Generic_Kinds;
      Scope       : in out Dictionary.Scopes;
      Subprog_Sym :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict            from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Kind,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         Kind,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Scope,
   --#         STree.Table,
   --#         Subprog_Sym                from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         Kind,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Kind,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
   is
      Ident_Node : STree.SyntaxNode;
      Ident_Str  : LexTokenManager.Lex_String;

      function Procedure_When_Function_Expected (Sym  : Dictionary.Symbol;
                                                 Kind : Generic_Kinds) return Boolean
      --# global in Dictionary.Dict;
      is
      begin
         return Kind = Generic_Procedure and then Dictionary.IsFunction (Sym);
      end Procedure_When_Function_Expected;

      ------------------

      function Function_When_Procedure_Expected (Sym  : Dictionary.Symbol;
                                                 Kind : Generic_Kinds) return Boolean
      --# global in Dictionary.Dict;
      is
      begin
         return Kind = Generic_Function and then Dictionary.IsProcedure (Sym);
      end Function_When_Procedure_Expected;

      ------------------

      procedure Check_Function_Return_Type (Sym              : in out Dictionary.Symbol;
                                            Return_Type_Node : in     STree.SyntaxNode)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --#        in out STree.Table;
      --# derives ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         LexTokenManager.State,
      --#                                         Return_Type_Node,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table,
      --#                                         Sym &
      --#         STree.Table,
      --#         Sym                        from CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Return_Type_Node,
      --#                                         STree.Table,
      --#                                         Sym;
      is
         Body_Return_Type_Sym : Dictionary.Symbol;
      begin
         -- ASSUME Return_Type_Node = type_mark
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Return_Type_Node) = SPSymbols.type_mark,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Return_Type_Node = type_mark in Check_Function_Return_Type");
         Wf_Type_Mark
           (Node          => Return_Type_Node,
            Current_Scope => Dictionary.LocalScope (Sym),
            Context       => Dictionary.ProgramContext,
            Type_Sym      => Body_Return_Type_Sym);
         if Dictionary.GetType (Sym) /= Body_Return_Type_Sym then
            ErrorHandler.Semantic_Error
              (Err_Num   => 22,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Return_Type_Node),
               Id_Str    => Dictionary.GetSimpleName (Sym));
            Sym := Dictionary.NullSymbol; -- signal error back to caller
         end if;
      end Check_Function_Return_Type;

   begin -- Check_Generic_Specification

      -- ASSUME Node = procedure_specification OR function_specification
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.procedure_specification
           or else Syntax_Node_Type (Node => Node) = SPSymbols.function_specification,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = procedure_specification OR function_specification in Check_Generic_Specification");

      Ident_Node := Child_Node (Current_Node => Child_Node (Current_Node => Node));
      -- ASSUME Ident_Node = identifier
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = Ident_Node = identifier in Check_Generic_Specification");
      Ident_Str := Node_Lex_String (Node => Ident_Node);

      Subprog_Sym := Dictionary.LookupItem (Name              => Ident_Str,
                                            Scope             => Scope,
                                            Context           => Dictionary.ProgramContext,
                                            Full_Package_Name => False);

      -- to be valid, Sym must be a generic unit of the right Kind which
      -- does not already have a body
      if Subprog_Sym = Dictionary.NullSymbol then
         -- no generic dec
         ErrorHandler.Semantic_Error
           (Err_Num   => 641,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Ident_Node),
            Id_Str    => Ident_Str);
      elsif not Dictionary.Is_Generic_Subprogram (The_Symbol => Subprog_Sym) then
         -- Sym is not a generic subprogram declaration
         ErrorHandler.Semantic_Error
           (Err_Num   => 642,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Ident_Node),
            Id_Str    => Ident_Str);
         Subprog_Sym := Dictionary.NullSymbol;
      elsif Procedure_When_Function_Expected (Sym  => Subprog_Sym,
                                              Kind => Kind) then
         -- wrong unit
         ErrorHandler.Semantic_Error
           (Err_Num   => 643,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Ident_Node),
            Id_Str    => Ident_Str);
         Subprog_Sym := Dictionary.NullSymbol;
      elsif Function_When_Procedure_Expected (Sym  => Subprog_Sym,
                                              Kind => Kind) then
         -- wrong unit
         ErrorHandler.Semantic_Error
           (Err_Num   => 644,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Ident_Node),
            Id_Str    => Ident_Str);
         Subprog_Sym := Dictionary.NullSymbol;
      elsif Dictionary.HasBody (Subprog_Sym) then
         -- already has body
         ErrorHandler.Semantic_Error
           (Err_Num   => 13,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Ident_Node),
            Id_Str    => Ident_Str);
         Subprog_Sym := Dictionary.NullSymbol;
      else
         -- OK so far, just check function return type if applicable
         STree.Set_Node_Lex_String (Sym  => Subprog_Sym,
                                    Node => Ident_Node);
         if Syntax_Node_Type (Node => Node) = SPSymbols.function_specification then
            Check_Function_Return_Type
              (Sym              => Subprog_Sym,
               Return_Type_Node => Last_Sibling_Of (Start_Node => Child_Node (Current_Node => Node)));
         end if;

         if Subprog_Sym /= Dictionary.NullSymbol then
            -- check above didn't find an error so add body
            Dictionary.AddBody
              (CompilationUnit => Subprog_Sym,
               Comp_Unit       => ContextManager.Ops.Current_Unit,
               TheBody         => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                       End_Position   => Node_Position (Node => Node)),
               Hidden          => False);
            -- an enter the local scope of the now legal body
            Scope := Dictionary.LocalScope (Subprog_Sym);
         end if;
      end if;
   end Check_Generic_Specification;

begin -- Wf_Subprogram_Body

   -- ASSUME Node = subprogram_body OR not_overriding_subprogram_body
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.subprogram_body
        or else Syntax_Node_Type (Node => Node) = SPSymbols.not_overriding_subprogram_body,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = subprogram_body OR not_overriding_subprogram_body in Wf_Subprogram_Body");

   Spec_Node := Child_Node (Current_Node => Node);
   -- ASSUME Spec_Node = overriding_indicator OR procedure_specification OR function_specification
   if Syntax_Node_Type (Node => Spec_Node) = SPSymbols.overriding_indicator then
      -- ASSUME Spec_Node = overriding_indicator
      if Syntax_Node_Type (Node => Child_Node (Current_Node => Spec_Node)) = SPSymbols.RWoverriding then
         Is_Overriding := True;
      end if;
      Spec_Node := Next_Sibling (Current_Node => Spec_Node);
   elsif Syntax_Node_Type (Node => Spec_Node) /= SPSymbols.procedure_specification
     and then Syntax_Node_Type (Node => Spec_Node) /= SPSymbols.function_specification then
      SystemErrors.Fatal_Error
        (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Spec_Node = overriding_indicator OR procedure_specification OR function_specification in Wf_Subprogram_Body");
   end if;
   -- ASSUME Spec_Node = procedure_specification OR function_specification

   Subprog_Implem_Node := Last_Sibling_Of (Start_Node => Spec_Node);
   -- ASSUME Subprog_Implem_Node = subprogram_implementation
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Subprog_Implem_Node) = SPSymbols.subprogram_implementation,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Subprog_Implem_Node = subprogram_implementation in Wf_Subprogram_Body");

   Hidden        := Body_Hidden_Class (Node => Subprog_Implem_Node);
   Is_Generic    := Syntax_Node_Type (Node => Parent_Node (Current_Node => Node)) = SPSymbols.generic_subprogram_body;
   Subprog_Scope := Scope;

   -- introduced variable below because the scope we use to check formal parts of
   -- generic bodies is not he same as we need to use for other subprograms.  The
   -- problem occurs when the formal part contains a type which is a generic formal
   -- parameter.  This is only visible in the local scope of the generic subprogram
   -- whereas types in the formal part of a normal subprogram will always be visible
   -- in the scope where the subprogram is being declared.  For all but generics,
   -- the new variable is set as below, for generics it is set to the local scope
   -- of the subprogram (further below).
   Scope_For_Formal_Part_Check := Scope;

   -- NOTE: Given Ada83 declaration order restrictions, I /think/ that we could always
   -- check formal parts in subprogram local scope rather than, as above, sometimes
   -- doing it the scope in which the subprogram is being declared.  With relaxed ordering
   -- there /might/ be a problem with subunits thus:
   -- spec
   -- stub
   -- declarations that the body can't see -- of course these can't exist in 83
   -- the body (here we might see the declarations we didn't ought to?)
   -- Anyway, I thought it best to leave the existing code alone and chnage the scope only
   -- for the generic case

   -- ASSUME Spec_Node = procedure_specification OR function_specification
   if Syntax_Node_Type (Node => Spec_Node) = SPSymbols.procedure_specification then
      -- ASSUME Spec_Node = procedure_specification
      if Is_Generic then
         First_Seen := False;
         Check_Generic_Specification
           (Node        => Spec_Node,
            Kind        => Generic_Procedure,
            Scope       => Subprog_Scope,
            Subprog_Sym => Subprog_Sym);
         Scope_For_Formal_Part_Check := Subprog_Scope; -- see comment above where var initialized
      else
         Wf_Procedure_Specification
           (Node        => Spec_Node,
            Hidden      => (Hidden = All_Hidden),
            Scope       => Subprog_Scope,
            Subprog_Sym => Subprog_Sym,
            First_Seen  => First_Seen);
      end if;
   elsif Syntax_Node_Type (Node => Spec_Node) = SPSymbols.function_specification then
      -- ASSUME Spec_Node = function_specification
      if Is_Generic then
         First_Seen := False;
         Check_Generic_Specification
           (Node        => Spec_Node,
            Kind        => Generic_Function,
            Scope       => Subprog_Scope,
            Subprog_Sym => Subprog_Sym);
         Scope_For_Formal_Part_Check := Subprog_Scope; -- see comment above where var initialized
      else
         Wf_Function_Specification
           (Node        => Spec_Node,
            Hidden      => (Hidden = All_Hidden),
            Scope       => Subprog_Scope,
            Subprog_Sym => Subprog_Sym,
            First_Seen  => First_Seen);
      end if;
   else
      Subprog_Sym := Dictionary.NullSymbol;
      First_Seen  := False;
      SystemErrors.Fatal_Error
        (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Spec_Node = procedure_specification OR function_specification in Wf_Subprogram_Body");
   end if;

   --# assert True;

   if Subprog_Sym /= Dictionary.NullSymbol then
      Main_Node := Parent_Node (Current_Node => Node);
      -- ASSUME Main_Node = proper_body OR protected_operation_item OR generic_subprogram_body OR main_program_declaration
      if Syntax_Node_Type (Node => Main_Node) = SPSymbols.main_program_declaration then
         -- ASSUME Main_Node = main_program_declaration
         Wf_Main_Program (Node          => Main_Node,
                          Subprog_Sym   => Subprog_Sym,
                          Scope         => Scope,
                          Subprog_Scope => Subprog_Scope);
      elsif Syntax_Node_Type (Node => Main_Node) = SPSymbols.proper_body then
         -- ASSUME Main_Node = proper_body
         -- check to look for WITH node in case of subunit
         With_Node := Parent_Node (Current_Node => Main_Node);
         -- ASSUME With_Node = subunit OR abody
         if Syntax_Node_Type (Node => With_Node) = SPSymbols.subunit then
            -- ASSUME With_Node = subunit
            -- there may be a WITH node to deal with
            With_Node :=
              Child_Node
              (Current_Node => Child_Node (Current_Node => Parent_Node (Current_Node => Parent_Node (Current_Node => With_Node))));
            -- ASSUME With_Node = subunit OR with_clause
            if Syntax_Node_Type (Node => With_Node) = SPSymbols.with_clause then
               -- ASSUME With_Node = with_clause
               Wf_Context_Clause (Parent_Node (Current_Node => With_Node), Subprog_Sym, Subprog_Scope);
            elsif Syntax_Node_Type (Node => With_Node) /= SPSymbols.subunit then
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                  Msg     => "Expect With_Node = subunit OR with_clause in Wf_Subprogram_Body");
            end if;
         elsif Syntax_Node_Type (Node => With_Node) /= SPSymbols.abody then
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect With_Node = subunit OR abody in Wf_Subprogram_Body");
         end if;
      elsif Syntax_Node_Type (Node => Main_Node) /= SPSymbols.protected_operation_item
        and then Syntax_Node_Type (Node => Main_Node) /= SPSymbols.generic_subprogram_body then
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Main_Node = proper_body OR protected_operation_item OR generic_subprogram_body OR main_program_declaration in Wf_Subprogram_Body");
      end if;

      Formal_Part_Node := Next_Sibling (Current_Node => Child_Node (Current_Node => Spec_Node));
      -- ASSUME Formal_Part_Node = formal_part OR type_mark OR NULL
      if Syntax_Node_Type (Node => Formal_Part_Node) = SPSymbols.formal_part then
         -- ASSUME Formal_Part_Node = formal_part
         Wf_Formal_Part
           (Node             => Formal_Part_Node,
            Current_Scope    => Scope_For_Formal_Part_Check,
            Subprog_Sym      => Subprog_Sym,
            First_Occurrence => First_Seen,
            Context          => Dictionary.ProgramContext);
      elsif Formal_Part_Node = STree.NullNode or else Syntax_Node_Type (Node => Formal_Part_Node) = SPSymbols.type_mark then
         -- ASSUME Formal_Part_Node = type_mark OR NULL
         if Dictionary.GetNumberOfSubprogramParameters (Subprog_Sym) /= 0 then
            ErrorHandler.Semantic_Error
              (Err_Num   => 152,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
         end if;
      else
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Formal_Part_Node = formal_part OR type_mark OR NULL in Wf_Subprogram_Body");
      end if;

      --# assert True;

      Get_Anno_And_Con_Nodes
        (Node            => Next_Sibling (Current_Node => Spec_Node),
         Anno_Node       => Anno_Node,
         Constraint_Node => Constraint_Node);
      -- ASSUME Anno_Node = procedure_annotation OR function_annotation OR NULL
      if Anno_Node = STree.NullNode then
         -- ASSUME Anno_Node = NULL
         if not (Syntax_Node_Type (Node => Parent_Node (Current_Node => Main_Node)) = SPSymbols.subunit) then
            if not First_Seen and then Requires_Second_Annotation (Subprog_Sym => Subprog_Sym) then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 87,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Spec_Node),
                  Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
               Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsRefined, Subprog_Sym);
            elsif First_Seen
              and then Syntax_Node_Type (Node => Spec_Node) = SPSymbols.procedure_specification
              and then (CommandLineData.Content.Language_Profile = CommandLineData.SPARK83
                          or else CommandLineData.Content.Do_Information_Flow) then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 154,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Spec_Node),
                  Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
               Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract, Subprog_Sym);
            end if;
         end if;
      elsif Syntax_Node_Type (Node => Anno_Node) = SPSymbols.procedure_annotation
        or else Syntax_Node_Type (Node => Anno_Node) = SPSymbols.function_annotation then
         -- ASSUME Anno_Node = procedure_annotation OR function_annotation
         if not (First_Seen or else Requires_Second_Annotation (Subprog_Sym => Subprog_Sym))
           or else Syntax_Node_Type (Node => Parent_Node (Current_Node => Main_Node)) = SPSymbols.subunit then
            -- annotation not required
            if Syntax_Node_Type (Node => Spec_Node) = SPSymbols.procedure_specification then
               -- ASSUME Spec_Node = procedure_specification
               ErrorHandler.Semantic_Error
                 (Err_Num   => 155,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Anno_Node),
                  Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
            elsif Syntax_Node_Type (Node => Spec_Node) = SPSymbols.function_specification then
               -- ASSUME Spec_Node = function_specification
               -- now distinguish between repeated anno and misplaced anno
               if Dictionary.IsNullIterator (Dictionary.FirstGlobalVariable (Dictionary.IsAbstract, Subprog_Sym)) then
                  -- misplaced anno
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 335,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Anno_Node),
                     Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
               else -- duplicated anno
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 336,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Anno_Node),
                     Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
               end if;
            end if;
         else -- annotation both present and required
            if Syntax_Node_Type (Node => Spec_Node) = SPSymbols.procedure_specification then
               -- ASSUME Spec_Node = procedure_specification
               Wf_Procedure_Annotation
                 (Node          => Anno_Node,
                  Current_Scope => Scope,
                  Subprog_Sym   => Subprog_Sym,
                  First_Seen    => First_Seen);
            elsif Syntax_Node_Type (Node => Spec_Node) = SPSymbols.function_specification then
               -- ASSUME Spec_Node = function_specification
               Wf_Function_Annotation
                 (Node          => Anno_Node,
                  Current_Scope => Scope,
                  Subprog_Sym   => Subprog_Sym,
                  First_Seen    => First_Seen);
            end if;
         end if;
      else
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Anno_Node = procedure_annotation OR function_annotation OR NULL in Wf_Subprogram_Body");
      end if;

      Scope     := Subprog_Scope;
      Next_Node := Spec_Node;

      --# assert True;

      -- clause for production of "full" dependency clause using modes
      if CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83
        and then Dictionary.IsProcedure (Subprog_Sym)
        and then not CommandLineData.Content.Do_Information_Flow
        and then not (Syntax_Node_Type (Node => Parent_Node (Current_Node => Main_Node)) = SPSymbols.subunit) then
         if First_Seen then
            CreateFullSubProgDependency (Node, Subprog_Sym, Dictionary.IsAbstract);
         elsif Requires_Second_Annotation (Subprog_Sym => Subprog_Sym) then
            CreateFullSubProgDependency (Node, Subprog_Sym, Dictionary.IsRefined);
         end if;
      end if;

      --# assert True;

      -- ASSUME Constraint_Node = procedure_constraint OR function_constraint
      if Syntax_Node_Type (Node => Child_Node (Current_Node => Constraint_Node)) = SPSymbols.precondition
        or else Syntax_Node_Type (Node => Child_Node (Current_Node => Constraint_Node)) = SPSymbols.postcondition
        or else Syntax_Node_Type (Node => Child_Node (Current_Node => Constraint_Node)) = SPSymbols.return_expression then
         -- ASSUME Child_Node (Current_Node => Constraint_Node) = precondition OR postcondition OR return_expression
         -- a constraint exists; should it? Check here
         if not (First_Seen
                   or else Requires_Second_Annotation (Subprog_Sym => Subprog_Sym)
                   or else HasParameterGlobalOrReturnOfLocalPrivateType (Subprog_Sym))
           or else Syntax_Node_Type (Node => Parent_Node (Current_Node => Main_Node)) = SPSymbols.subunit then
            -- annotation not required

            -- two possible errors: misplaced anno or duplicate anno
            if Dictionary.HasPrecondition (Dictionary.IsAbstract, Subprog_Sym)
              or else Dictionary.HasPostcondition (Dictionary.IsAbstract, Subprog_Sym) then
               -- illegal duplicate anno
               ErrorHandler.Semantic_Error
                 (Err_Num   => 343,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Constraint_Node),
                  Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
            else -- misplaced anno
               ErrorHandler.Semantic_Error
                 (Err_Num   => 342,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Constraint_Node),
                  Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
            end if;
         else -- annotation is required so continue
            if Syntax_Node_Type (Node => Spec_Node) = SPSymbols.procedure_specification then
               -- ASSUME Spec_Node = procedure_specification
               wf_procedure_constraint (Constraint_Node, Dictionary.LocalScope (Subprog_Sym), First_Seen);
            elsif Syntax_Node_Type (Node => Spec_Node) = SPSymbols.function_specification then
               -- ASSUME Spec_Node = function_specification
               wf_function_constraint (Constraint_Node, Dictionary.LocalScope (Subprog_Sym), First_Seen);
            end if;
         end if;
      elsif Child_Node (Current_Node => Constraint_Node) /= STree.NullNode then
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Child_Node (Current_Node => Constraint_Node)  =  precondition OR postcondition OR return_expression OR NULL in Wf_Subprogram_Body");
      end if;
   else
      Next_Node := STree.NullNode;
   end if;

   --# assert True;

   -- set up identifier for hidden part reporting
   Ident_Node := Child_Node (Current_Node => Child_Node (Current_Node => Spec_Node));
   -- ASSUME Ident_Node = identifier
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Ident_Node = identifier in Wf_Subprogram_Body");
   End_Desig_Node := Last_Sibling_Of (Start_Node => Child_Node (Current_Node => Subprog_Implem_Node));
   -- ASSUME End_Desig_Node = designator OR hidden_part
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => End_Desig_Node) = SPSymbols.designator
        or else Syntax_Node_Type (Node => End_Desig_Node) = SPSymbols.hidden_part,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect End_Desig_Node = designator OR hidden_part in Wf_Subprogram_Body");
   case Hidden is
      when All_Hidden =>
         ErrorHandler.Hidden_Text
           (Position => Node_Position (Node => End_Desig_Node),
            Unit_Str => Node_Lex_String (Node => Ident_Node),
            Unit_Typ => SPSymbols.subprogram_implementation);
      when Handler_Hidden =>
         ErrorHandler.Hidden_Handler
           (Position => Node_Position (Node => End_Desig_Node),
            Unit_Str => Node_Lex_String (Node => Ident_Node),
            Unit_Typ => SPSymbols.subprogram_implementation);
      when Not_Hidden =>
         null;
   end case;

   -- For SPARK 83 and 95:
   -- If a potentially inheritable subprogram of the same name exists then
   -- the new declaration is only legal if it successfully overrides it.
   -- This check is only required if the subprogram has not been previously declared
   -- because, if it has, the check will already have been done in the package spec

   -- For SPARK 2005:
   -- The check is required even if the subprogram has been previously declared
   -- as we need to verify that the overriding_indicator is correct.

   if First_Seen or else CommandLineData.Content.Language_Profile = CommandLineData.SPARK2005 then
      CheckNoOverloadingFromTaggedOps (Spec_Node, Subprog_Sym, Scope, Dictionary.IsRefined, Is_Overriding);
   end if;

   if Dictionary.IsMainProgram (Subprog_Sym) and then not Is_Generic and then CommandLineData.Ravenscar_Selected then
      Shared_Variable_Check
        (Main_Program_Sym => Subprog_Sym,
         Scope            => Subprog_Scope,
         Error_Node_Pos   => Node_Position (Node => Node));
      Max_One_In_A_Queue_Check
        (Main_Program_Sym => Subprog_Sym,
         Scope            => Subprog_Scope,
         Error_Node_Pos   => Node_Position (Node => Node));
   end if;

   -- Check that function ends with a return statement; this check was previously done
   -- at up_wf_subprogram body where any error detected was too late to stop flow analysis
   -- or VC generation
   if Syntax_Node_Type (Node => Spec_Node) = SPSymbols.function_specification then
      -- ASSUME Spec_Node = function_specification
      Check_Function_Has_Return
        (Subprog_Node       => Subprog_Implem_Node,
         End_Desig_Node_Pos => Node_Position (Node => End_Desig_Node));
   end if;
end Wf_Subprogram_Body;
