-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with SLI;

separate (Sem.CompUnit.Wf_Package_Specification.Wf_Anno)
procedure wf_init_spec
  (Node         : in STree.SyntaxNode;
   PackSym      : in Dictionary.Symbol;
   ScopeType    : in Enclosing_Scope_Types;
   CurrentScope : in Dictionary.Scopes) is

   -------------------------------------------------------------------------

   procedure DoLibraryPack (InitSpecNode : in STree.SyntaxNode;
                            CurrentScope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         InitSpecNode,
   --#                                         LexTokenManager.State,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         InitSpecNode,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   is
      IdentListNode, NextNode : STree.SyntaxNode;
      It                      : STree.Iterator;
      VarSym                  : Dictionary.Symbol;
   begin
      IdentListNode := Child_Node (InitSpecNode);

      It := Find_First_Node (Node_Kind    => SPSymbols.identifier,
                             From_Root    => IdentListNode,
                             In_Direction => STree.Down);

      while not STree.IsNull (It) loop
         NextNode := Get_Node (It => It);
         -- LastNode := NextNode;
         VarSym :=
           Dictionary.LookupImmediateScope
           (Name    => Node_Lex_String (Node => NextNode),
            Scope   => CurrentScope,
            Context => Dictionary.ProofContext);
         if VarSym = Dictionary.NullSymbol then   --undeclared
            ErrorHandler.Semantic_Error
              (Err_Num   => 139,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => NextNode),
               Id_Str    => Node_Lex_String (Node => NextNode));
         elsif Dictionary.IsOwnVariable (VarSym) then   --ok, as long as not moded
            if Dictionary.GetOwnVariableMode (VarSym) = Dictionary.DefaultMode then
               if not Dictionary.GetOwnVariableProtected (VarSym) then -- OK as long as not protected
                  STree.Set_Node_Lex_String (Sym  => VarSym,
                                             Node => NextNode);
                  Dictionary.AddInitializedOwnVariable
                    (VarSym,
                     Dictionary.Location'(Start_Position => Node_Position (Node => NextNode),
                                          End_Position   => Node_Position (Node => NextNode)));
               else
                  -- own protected are implicitly initialized
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 863,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => NextNode),
                     Id_Str    => Node_Lex_String (Node => NextNode));
               end if;
            else --has a mode
               ErrorHandler.Semantic_Error
                 (Err_Num   => 708,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => NextNode),
                  Id_Str    => Node_Lex_String (Node => NextNode));
            end if;
         else   --not an own variable
            ErrorHandler.Semantic_Error
              (Err_Num   => 174,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => NextNode),
               Id_Str    => Node_Lex_String (Node => NextNode));
         end if;
         It := STree.NextNode (It);
      end loop;
   end DoLibraryPack;

   -------------------------------------------------------------------------

   procedure DoEmbeddedPack
     (InitSpecNode : in STree.SyntaxNode;
      PackSym      : in Dictionary.Symbol;
      CurrentScope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         InitSpecNode,
   --#                                         LexTokenManager.State,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         InitSpecNode,
   --#                                         LexTokenManager.State,
   --#                                         PackSym,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   is
      IdentListNode, LastNode, NextNode : STree.SyntaxNode;
      NodeIt                            : STree.Iterator;
      VarSym                            : Dictionary.Symbol;
      It                                : Dictionary.Iterator;
   begin
      IdentListNode := Child_Node (InitSpecNode);
      LastNode      := IdentListNode;

      NodeIt := Find_First_Node (Node_Kind    => SPSymbols.identifier,
                                 From_Root    => IdentListNode,
                                 In_Direction => STree.Down);

      while not STree.IsNull (NodeIt) loop
         NextNode := Get_Node (It => NodeIt);
         LastNode := NextNode;
         VarSym   :=
           Dictionary.LookupImmediateScope
           (Name    => Node_Lex_String (Node => NextNode),
            Scope   => CurrentScope,
            Context => Dictionary.ProofContext);
         if VarSym = Dictionary.NullSymbol then --undeclared
            ErrorHandler.Semantic_Error
              (Err_Num   => 139,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => NextNode),
               Id_Str    => Node_Lex_String (Node => NextNode));
         elsif Dictionary.IsOwnVariable (VarSym) and then Dictionary.OwnVariableIsAnnounced (VarSym) then -- potentially ok
            if Dictionary.GetOwnVariableMode (VarSym) /= Dictionary.DefaultMode then
               -- it does have a mode which is not allowed
               ErrorHandler.Semantic_Error
                 (Err_Num   => 708,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => NextNode),
                  Id_Str    => Node_Lex_String (Node => NextNode));

            elsif not Dictionary.OwnVariableIsInitialized (Dictionary.GetSubject (VarSym)) then
               -- initialization not allowed because subject not initialized
               ErrorHandler.Semantic_Error
                 (Err_Num   => 78,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => NextNode),
                  Id_Str    => Node_Lex_String (Node => NextNode));

            elsif Dictionary.GetOwnVariableProtected (VarSym) then
               --  OK as long as not protected own protected are
               --  implicitly initialized
               ErrorHandler.Semantic_Error
                 (Err_Num   => 863,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => NextNode),
                  Id_Str    => Node_Lex_String (Node => NextNode));
            else
               STree.Set_Node_Lex_String (Sym  => VarSym,
                                          Node => NextNode);
               Dictionary.AddInitializedOwnVariable
                 (VarSym,
                  Dictionary.Location'(Start_Position => Node_Position (Node => NextNode),
                                       End_Position   => Node_Position (Node => NextNode)));

            end if;
         else   --not an announced own variable
            ErrorHandler.Semantic_Error
              (Err_Num   => 174,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => NextNode),
               Id_Str    => Node_Lex_String (Node => NextNode));
         end if;
         NodeIt := STree.NextNode (NodeIt);
      end loop;

      It := Dictionary.FirstOwnVariable (PackSym);
      while not Dictionary.IsNullIterator (It) loop
         VarSym := Dictionary.CurrentSymbol (It);
         if Dictionary.OwnVariableIsInitialized (Dictionary.GetSubject (VarSym))
           and then not Dictionary.OwnVariableIsInitialized (VarSym)
           and then Dictionary.GetOwnVariableMode (VarSym) = Dictionary.DefaultMode then
            ErrorHandler.Semantic_Error
              (Err_Num   => 79,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => LastNode),
               Id_Str    => Dictionary.GetSimpleName (VarSym));
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;
   end DoEmbeddedPack;

begin -- wf_init_spec
   if ScopeType = In_Package then
      DoEmbeddedPack (Node, PackSym, CurrentScope);
   else
      DoLibraryPack (Node, CurrentScope);
   end if;

   if ErrorHandler.Generate_SLI then
      SLI.Generate_Xref_Initializes (Comp_Unit  => ContextManager.Ops.Current_Unit,
                                     Parse_Tree => Node,
                                     Scope      => CurrentScope);
   end if;

end wf_init_spec;
