-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with LexTokenManager.Relation_Algebra.String;
with LexTokenManager.Seq_Algebra;

separate (Sem.CompUnit.Wf_Package_Specification)
procedure CheckModes (Node     : in STree.SyntaxNode;
                      Pack_Sym : in Dictionary.Symbol) is

   PrivTypeIt, SubprogIt, ParamIt    : Dictionary.Iterator;
   TypeSym, SubprogSym, ParamSym     : Dictionary.Symbol;
   Vis_Part_Rep_Node, Proc_Spec_Node : STree.SyntaxNode;
   SubprogramsToMark                 : Boolean;
   CurrentParamList                  : LexTokenManager.Seq_Algebra.Seq;
   The_Relation                      : LexTokenManager.Relation_Algebra.String.Relation;

   procedure Process_Procedure (Node       : in STree.SyntaxNode;
                                Param_List : in LexTokenManager.Seq_Algebra.Seq)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in     TheHeap;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Param_List,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   is
      It      : STree.Iterator;
      Id_Node : STree.SyntaxNode;
   begin
      -- ASSUME Node = formal_part
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.formal_part,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = formal_part in Process_Procedure");
      It := Find_First_Node (Node_Kind    => SPSymbols.identifier,
                             From_Root    => Node,
                             In_Direction => STree.Down);

      while not STree.IsNull (It) loop
         Id_Node := Get_Node (It => It);
         if LexTokenManager.Seq_Algebra.Is_Member
           (The_Heap    => TheHeap,
            S           => Param_List,
            Given_Value => Node_Lex_String (Node => Id_Node)) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 338,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Id_Node),
               Id_Str    => Node_Lex_String (Node => Id_Node));
         end if;
         It := STree.NextNode (It);
      end loop;
   end Process_Procedure;

begin -- CheckModes

   -- ASSUME Node = visible_part_rep
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.visible_part_rep,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = visible_part_rep in CheckModes");

   LexTokenManager.Relation_Algebra.String.Create_Relation (The_Heap => TheHeap,
                                                            R        => The_Relation);
   SubprogramsToMark := False;
   PrivTypeIt        := Dictionary.FirstPrivateType (Pack_Sym);
   while not Dictionary.IsNullIterator (PrivTypeIt) loop
      TypeSym := Dictionary.CurrentSymbol (PrivTypeIt);
      if Dictionary.IsDeclared (TypeSym) and then Dictionary.TypeIsScalar (TypeSym) then
         -- we have a scalar private type which may affect subprog params
         SubprogIt := Dictionary.FirstVisibleSubprogram (Pack_Sym);
         while not Dictionary.IsNullIterator (SubprogIt) loop
            SubprogSym := Dictionary.CurrentSymbol (SubprogIt);

            ParamIt := Dictionary.FirstSubprogramParameter (SubprogSym);
            while not Dictionary.IsNullIterator (ParamIt) loop
               ParamSym := Dictionary.CurrentSymbol (ParamIt);
               if Dictionary.GetType (ParamSym) = TypeSym
                 and then Dictionary.GetSubprogramParameterMode (ParamSym) = Dictionary.InOutMode
                 and then not Dictionary.IsImport (Dictionary.IsAbstract, SubprogSym, ParamSym) then
                  SubprogramsToMark := True;
                  LexTokenManager.Relation_Algebra.String.Insert_Pair
                    (The_Heap => TheHeap,
                     R        => The_Relation,
                     I        => Dictionary.GetSimpleName (SubprogSym),
                     J        => Dictionary.GetSimpleName (ParamSym));
               end if;
               ParamIt := Dictionary.NextSymbol (ParamIt);
            end loop;

            SubprogIt := Dictionary.NextSymbol (SubprogIt);
         end loop;
      end if;
      PrivTypeIt := Dictionary.NextSymbol (PrivTypeIt);
   end loop;

   -- At this point we have created in SubprogList a data structure listing
   -- all the procedures made illegal by the private types' full declarations
   -- and for each of them a list of affected parameters.  We now walk the
   -- syntax tree marking each parameter occurrence found.
   if SubprogramsToMark then
      Vis_Part_Rep_Node := Child_Node (Current_Node => Node);
      -- ASSUME Vis_Part_Rep_Node = visible_part_rep OR NULL
      while Vis_Part_Rep_Node /= STree.NullNode loop
         -- ASSUME Vis_Part_Rep_Node = visible_part_rep
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Vis_Part_Rep_Node) = SPSymbols.visible_part_rep,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Vis_Part_Rep_Node = visible_part_rep in CheckModes");

         Proc_Spec_Node := Next_Sibling (Current_Node => Vis_Part_Rep_Node);
         if Syntax_Node_Type (Node => Proc_Spec_Node) = SPSymbols.subprogram_declaration then
            -- ASSUME Proc_Spec_Node = subprogram_declaration
            Proc_Spec_Node := Child_Node (Current_Node => Proc_Spec_Node);
            if Syntax_Node_Type (Node => Proc_Spec_Node) = SPSymbols.overriding_indicator then
               -- ASSUME Proc_Spec_Node = overriding_indicator
               Proc_Spec_Node := Next_Sibling (Current_Node => Proc_Spec_Node);
            elsif Syntax_Node_Type (Node => Proc_Spec_Node) /= SPSymbols.function_specification
              and then Syntax_Node_Type (Node => Proc_Spec_Node) /= SPSymbols.procedure_specification
              and then Syntax_Node_Type (Node => Proc_Spec_Node) /= SPSymbols.proof_function_declaration then
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                  Msg     => "Expect Proc_Spec_Node = overriding_indicator OR procedure_specification OR function_specification OR proof_function_declaration in CheckModes");
            end if;
            -- ASSUME Proc_Spec_Node = procedure_specification OR function_specification OR proof_function_declaration
            if Syntax_Node_Type (Node => Proc_Spec_Node) = SPSymbols.procedure_specification then
               -- ASSUME Proc_Spec_Node = procedure_specification
               LexTokenManager.Relation_Algebra.String.Row_Extraction
                 (The_Heap    => TheHeap,
                  R           => The_Relation,
                  Given_Index => Node_Lex_String (Node => Child_Node (Current_Node => Child_Node (Current_Node => Proc_Spec_Node))),
                  S           => CurrentParamList);
               if not LexTokenManager.Seq_Algebra.Is_Null_Seq (S => CurrentParamList) then
                  Process_Procedure
                    (Node       => Next_Sibling (Current_Node => Child_Node (Current_Node => Proc_Spec_Node)),
                     Param_List => CurrentParamList);
               end if;
            elsif Syntax_Node_Type (Node => Proc_Spec_Node) /= SPSymbols.function_specification
              and then Syntax_Node_Type (Node => Proc_Spec_Node) /= SPSymbols.proof_function_declaration then
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                  Msg     => "Expect Proc_Spec_Node = procedure_specification OR function_specification OR proof_function_declaration in CheckModes");
            end if;
         end if;
         Vis_Part_Rep_Node := Child_Node (Current_Node => Vis_Part_Rep_Node);
      end loop;
   end if;
end CheckModes;
