-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

-- Overview:
-- Checks a Package Declaration for Sem on down pass through
-- TreeProcessor.  Starts at node package_declaration.  May directly raise
-- errors for: re-declaration of package identifier.  Other errors may be raised
-- indirectly by wf_package_specification, wf_inherit_clause and
-- wf_context_clause which are called from here.
--------------------------------------------------------------------------------

separate (Sem.CompUnit)
procedure Wf_Package_Declaration (Node          : in STree.SyntaxNode;
                                  Current_Scope : in Dictionary.Scopes) is
   type Enclosing_Scope_Types is (In_Library, In_Package, In_Procedure);
   Enclosing_Scope_Type                              : Enclosing_Scope_Types;
   Ident_Str                                         : LexTokenManager.Lex_String;
   Spec_Node, Context_Node, Inherit_Node, Ident_Node : STree.SyntaxNode;
   Pack_Sym                                          : Dictionary.Symbol;
   Child_Package_Declaration                         : Boolean;
   Private_Package_Declaration                       : Boolean;
   Valid_Name                                        : Boolean := True;

   ------------------------------

   procedure Find_Key_Nodes
     (Node                        : in     STree.SyntaxNode;
      Context_Node                :    out STree.SyntaxNode;
      Private_Package_Declaration :    out Boolean;
      Inherit_Node                :    out STree.SyntaxNode;
      Spec_Node                   :    out STree.SyntaxNode;
      Ident_Node                  :    out STree.SyntaxNode;
      Ident_Str                   :    out LexTokenManager.Lex_String;
      Child_Package_Declaration   :    out Boolean)
   --# global in STree.Table;
   --# derives Child_Package_Declaration,
   --#         Context_Node,
   --#         Ident_Node,
   --#         Ident_Str,
   --#         Inherit_Node,
   --#         Private_Package_Declaration,
   --#         Spec_Node                   from Node,
   --#                                          STree.Table;
   is
   begin
      -- ASSUME Node = package_declaration OR private_package_declaration
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.package_declaration
           or else Syntax_Node_Type (Node => Node) = SPSymbols.private_package_declaration,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = package_declaration OR private_package_declaration in Find_Key_Nodes");

      Context_Node := Parent_Node (Current_Node => Node);
      -- ASSUME Context_Node = library_unit OR initial_declarative_item_rep
      if Syntax_Node_Type (Node => Context_Node) = SPSymbols.library_unit then
         Context_Node := Child_Node (Current_Node => Parent_Node (Current_Node => Context_Node));
         -- ASSUME Context_Node = context_clause OR library_unit
         if Syntax_Node_Type (Node => Context_Node) = SPSymbols.library_unit then
            -- ASSUME Context_Node = library_unit
            Context_Node := STree.NullNode;
         elsif Syntax_Node_Type (Node => Context_Node) /= SPSymbols.context_clause then
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Context_Node = context_clause OR library_unit in Find_Key_Nodes");
         end if;
      elsif Syntax_Node_Type (Node => Context_Node) = SPSymbols.initial_declarative_item_rep then
         Context_Node := STree.NullNode;
      else
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Context_Node = library_unit OR initial_declarative_item_rep in Find_Key_Nodes");
      end if;
      -- ASSUME Context_Node = context_clause OR NULL
      SystemErrors.RT_Assert
        (C       => Context_Node = STree.NullNode or else Syntax_Node_Type (Node => Context_Node) = SPSymbols.context_clause,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Context_Node = context_clause OR NULL in Find_Key_Nodes");

      Private_Package_Declaration := Syntax_Node_Type (Node => Node) = SPSymbols.private_package_declaration;
      Inherit_Node                := Child_Node (Current_Node => Node);
      -- ASSUME Inherit_Node = inherit_clause OR package_specification
      if Syntax_Node_Type (Node => Inherit_Node) = SPSymbols.inherit_clause then
         -- ASSUME Inherit_Node = inherit_clause
         Spec_Node := Next_Sibling (Current_Node => Inherit_Node);
      elsif Syntax_Node_Type (Node => Inherit_Node) = SPSymbols.package_specification then
         -- ASSUME Inherit_Node = package_specification
         Spec_Node    := Inherit_Node;
         Inherit_Node := STree.NullNode;
      else
         Spec_Node    := STree.NullNode;
         Inherit_Node := STree.NullNode;
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Inherit_Node = inherit_clause OR package_specification in Find_Key_Nodes");
      end if;
      -- ASSUME Inherit_Node = inherit_clause OR NULL
      SystemErrors.RT_Assert
        (C       => Inherit_Node = STree.NullNode or else Syntax_Node_Type (Node => Inherit_Node) = SPSymbols.inherit_clause,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Inherit_Node = inherit_clause OR NULL in Find_Key_Nodes");
      -- ASSUME Spec_Node = package_specification
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Spec_Node) = SPSymbols.package_specification,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Spec_Node = package_specification in Find_Key_Nodes");

      Ident_Node := Child_Node (Current_Node => Child_Node (Current_Node => Spec_Node));
      -- ASSUME Ident_Node = dotted_simple_name OR identifier
      if Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier then
         -- ASSUME Ident_Node = identifier
         Child_Package_Declaration := False;
      elsif Syntax_Node_Type (Node => Ident_Node) = SPSymbols.dotted_simple_name then
         -- ASSUME Ident_Node = dotted_simple_name
         Child_Package_Declaration := True;
         Ident_Node                := Last_Child_Of (Start_Node => Ident_Node);
      else
         Child_Package_Declaration := False;
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Ident_Node = dotted_simple_name OR identifier in Find_Key_Nodes");
      end if;
      -- ASSUME Ident_Node = identifier
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Ident_Node = identifier in Find_Key_Nodes");
      Ident_Str := Node_Lex_String (Node => Ident_Node);
   end Find_Key_Nodes;

   ------------------------------

   procedure Find_Enclosing_Scope_Type (Scope                : in     Dictionary.Scopes;
                                        Enclosing_Scope_Type :    out Enclosing_Scope_Types)
   --# global in Dictionary.Dict;
   --# derives Enclosing_Scope_Type from Dictionary.Dict,
   --#                                   Scope;
   is
   begin
      if Dictionary.IsGlobalScope (Scope) then
         Enclosing_Scope_Type := In_Library;
      elsif Dictionary.IsPackage (Dictionary.GetRegion (Scope)) then
         Enclosing_Scope_Type := In_Package;
      else
         Enclosing_Scope_Type := In_Procedure;
      end if;
   end Find_Enclosing_Scope_Type;

   ------------------------------

   function Is_Not_Refinement_Announcement
     (Sym                  : Dictionary.Symbol;
      Enclosing_Scope_Type : Enclosing_Scope_Types)
     return                 Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return Enclosing_Scope_Type /= In_Package or else Dictionary.GetContext (Sym) /= Dictionary.ProofContext;
   end Is_Not_Refinement_Announcement;

begin -- Wf_Package_Declaration

   -- ASSUME Node = package_declaration OR private_package_declaration
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.package_declaration
        or else Syntax_Node_Type (Node => Node) = SPSymbols.private_package_declaration,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = package_declaration OR private_package_declaration in Wf_Package_Declaration");

   Find_Key_Nodes
     (Node                        => Node,
      Context_Node                => Context_Node,
      Private_Package_Declaration => Private_Package_Declaration,
      Inherit_Node                => Inherit_Node,
      Spec_Node                   => Spec_Node,
      Ident_Node                  => Ident_Node,
      Ident_Str                   => Ident_Str,
      Child_Package_Declaration   => Child_Package_Declaration);
   -- ASSUME Spec_Node = package_specification
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Spec_Node) = SPSymbols.package_specification,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Spec_Node = package_specification in Wf_Package_Declaration");
   -- ASSUME Ident_Node = identifier
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Ident_Node = identifier in Wf_Package_Declaration");

   -- tells us where package is being declared
   Find_Enclosing_Scope_Type (Scope                => Current_Scope,
                              Enclosing_Scope_Type => Enclosing_Scope_Type);
   if CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83
     and then Child_Package_Declaration
     and then Enclosing_Scope_Type = In_Library then
      AddChild (Ident_Node, Private_Package_Declaration, Current_Scope,
                -- to get
                Pack_Sym, Ident_Str);
      -- if Pack_Sym is null then something went wrong when we added the child so we need to supress
      -- any further analysis of the package specification
      Valid_Name := Pack_Sym /= Dictionary.NullSymbol;
   else
      if CommandLineData.Content.Language_Profile = CommandLineData.SPARK83 then
         -- check that syntax conforms
         if Child_Package_Declaration or else Private_Package_Declaration then
            ErrorHandler.Semantic_Error
              (Err_Num   => 610,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Ident_Node),
               Id_Str    => LexTokenManager.Null_String);
            Private_Package_Declaration := False;
         end if;
      elsif Child_Package_Declaration and then Enclosing_Scope_Type /= In_Library then
         ErrorHandler.Semantic_Error
           (Err_Num   => 614,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Ident_Node),
            Id_Str    => LexTokenManager.Null_String);
      end if;
      Pack_Sym := Dictionary.LookupItem (Name              => Ident_Str,
                                         Scope             => Current_Scope,
                                         Context           => Dictionary.ProofContext,
                                         Full_Package_Name => False);
      --# assert True;
      if Pack_Sym /= Dictionary.NullSymbol
        and then Is_Not_Refinement_Announcement (Sym                  => Pack_Sym,
                                                 Enclosing_Scope_Type => Enclosing_Scope_Type) then
         Valid_Name := False;
         ErrorHandler.Semantic_Error
           (Err_Num   => 10,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Ident_Node),
            Id_Str    => Ident_Str);
      else
         if Pack_Sym /= Dictionary.NullSymbol then
            STree.Set_Node_Lex_String (Sym  => Pack_Sym,
                                       Node => Ident_Node);
         end if;
         --# assert True;
         if Private_Package_Declaration then -- root level private package
            Dictionary.AddPrivatePackage
              (Name          => Ident_Str,
               Comp_Unit     => ContextManager.Ops.Current_Unit,
               Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Ident_Node),
                                                     End_Position   => Node_Position (Node => Ident_Node)),
               Scope         => Current_Scope,
               ThePackage    => Pack_Sym);
         else
            Dictionary.AddPackage
              (Name          => Ident_Str,
               Comp_Unit     => ContextManager.Ops.Current_Unit,
               Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Ident_Node),
                                                     End_Position   => Node_Position (Node => Ident_Node)),
               Scope         => Current_Scope,
               ThePackage    => Pack_Sym);
         end if;
      end if;
   end if;

   -- wff the package specification iff its declaration is valid
   if Valid_Name then
      -- ASSUME Inherit_Node = inherit_clause OR NULL
      if Syntax_Node_Type (Node => Inherit_Node) = SPSymbols.inherit_clause then
         -- ASSUME Inherit_Node = inherit_clause
         Dictionary.AddInheritsAnnotation
           (Pack_Sym,
            Dictionary.Location'(Start_Position => Node_Position (Node => Inherit_Node),
                                 End_Position   => Node_Position (Node => Inherit_Node)));
         Wf_Inherit_Clause (Node     => Inherit_Node,
                            Comp_Sym => Pack_Sym,
                            Scope    => Current_Scope);
      elsif Inherit_Node /= STree.NullNode then
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Inherit_Node = inherit_clause OR NULL in Wf_Package_Declaration");
      end if;

      --# assert True;

      -- ASSUME Context_Node = context_clause OR NULL
      if Syntax_Node_Type (Node => Context_Node) = SPSymbols.context_clause then
         -- ASSUME Context_Node = context_clause
         Wf_Context_Clause (Context_Node, Pack_Sym, Dictionary.VisibleScope (Pack_Sym));
      elsif Context_Node /= STree.NullNode then
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Context_Node = context_clause OR NULL in Wf_Package_Declaration");
      end if;

      --# assert True;

      Wf_Package_Specification (Node          => Spec_Node,
                                Ident_Str     => Ident_Str,
                                Pack_Sym      => Pack_Sym,
                                Current_Scope => Current_Scope);
   end if;
end Wf_Package_Declaration;
