-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with SLI;

separate (Sem.CompUnit)
procedure Wf_Justification_Statement (Node  : in STree.SyntaxNode;
                                      Scope : in Dictionary.Scopes) is

   procedure Wf_Start_Justification (Node  : in STree.SyntaxNode;
                                     Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Dictionary.Dict,
   --#         GlobalComponentData,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   is
      It : STree.Iterator;

      procedure Wf_Justification_Clause
        (Start_Line : in LexTokenManager.Line_Numbers;
         Node       : in STree.SyntaxNode;
         Scope      : in Dictionary.Scopes)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out GlobalComponentData;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --#        in out STree.Table;
      --#        in out TheHeap;
      --# derives Dictionary.Dict,
      --#         GlobalComponentData,
      --#         Statistics.TableUsage,
      --#         STree.Table,
      --#         TheHeap                    from *,
      --#                                         CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         GlobalComponentData,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         Scope,
      --#                                         STree.Table,
      --#                                         TheHeap &
      --#         ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         GlobalComponentData,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         Scope,
      --#                                         SPARK_IO.File_Sys,
      --#                                         Start_Line,
      --#                                         STree.Table,
      --#                                         TheHeap;
      is
         Error_Found                    : Boolean;
         Current_Node                   : STree.SyntaxNode;
         Err_Num_Node                   : STree.SyntaxNode;
         Valid                          : Boolean;
         Kind                           : ErrorHandler.Justification_Kinds;
         Val                            : Maths.Value;
         Maths_Valid                    : Maths.ErrorCode;
         Err_Num                        : Natural;
         Explanation                    : E_Strings.T;
         Identifiers                    : ErrorHandler.Justification_Identifiers;
         Maximum_Justifications_Reached : Boolean;

         procedure Check_Kind
           (Lex_String : in     LexTokenManager.Lex_String;
            Kind       :    out ErrorHandler.Justification_Kinds;
            Valid      :    out Boolean)
         --# global in LexTokenManager.State;
         --# derives Kind,
         --#         Valid from LexTokenManager.State,
         --#                    Lex_String;
         is
            Flow      : constant String := "FLOW_MESSAGE";
            Warn      : constant String := "WARNING_MESSAGE";
            Ex_String : E_Strings.T;
            Found     : Boolean;
            Start_Pos : E_Strings.Positions;
         begin
            -- The kind of message (Flow or Warning) is in the form of an identifier and therefore is extracted
            -- from the syntax tree as a Lex_String.  We first convert it to an Examiner_String

            -- Then we see if it a unique subset of either "Flow_Message" or "Warning_Message"
            -- Ignore case
            Ex_String := E_Strings.Upper_Case (E_Str => LexTokenManager.Lex_String_To_String (Lex_Str => Lex_String));
            -- Try "flow" first
            E_Strings.Find_Examiner_Sub_String
              (E_Str         => E_Strings.Copy_String (Str => Flow),
               Search_String => Ex_String,
               String_Found  => Found,
               String_Start  => Start_Pos);
            -- To get a match we need Found and Start_Pos = 1
            if Found and then Start_Pos = 1 then
               Kind  := ErrorHandler.Flow_Message;
               Valid := True;

            else
               -- Try "warn"
               E_Strings.Find_Examiner_Sub_String
                 (E_Str         => E_Strings.Copy_String (Str => Warn),
                  Search_String => Ex_String,
                  String_Found  => Found,
                  String_Start  => Start_Pos);
               -- To get a match we need Found and Start_Pos = 1
               if Found and then Start_Pos = 1 then
                  Kind  := ErrorHandler.Warning_Message;
                  Valid := True;
               else
                  Kind  := ErrorHandler.Flow_Message; -- not used, for DF purposes only
                  Valid := False;
               end if;
            end if;
         end Check_Kind;

         function Is_Disallowed_Warning (Kind    : ErrorHandler.Justification_Kinds;
                                         Err_Num : Natural) return Boolean is
         begin
            -- Initially only prohibit warnings generated by the justification system itself.
            -- Extend here as necessary.

            return Kind = ErrorHandler.Warning_Message and then (Err_Num = 120 or else Err_Num = 121 or else Err_Num = 122);
         end Is_Disallowed_Warning;

         procedure Check_Identifiers
           (Opt_Node    : in     STree.SyntaxNode;
            Identifiers :    out ErrorHandler.Justification_Identifiers;
            Valid       :    out Boolean)
         --# global in     CommandLineData.Content;
         --#        in     ContextManager.Ops.Unit_Stack;
         --#        in     Err_Num;
         --#        in     Kind;
         --#        in     LexTokenManager.State;
         --#        in     Scope;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.Error_Context;
         --#        in out GlobalComponentData;
         --#        in out SPARK_IO.File_Sys;
         --#        in out Statistics.TableUsage;
         --#        in out STree.Table;
         --#        in out TheHeap;
         --# derives Dictionary.Dict,
         --#         GlobalComponentData,
         --#         Statistics.TableUsage,
         --#         STree.Table,
         --#         TheHeap                    from *,
         --#                                         CommandLineData.Content,
         --#                                         ContextManager.Ops.Unit_Stack,
         --#                                         Dictionary.Dict,
         --#                                         GlobalComponentData,
         --#                                         LexTokenManager.State,
         --#                                         Opt_Node,
         --#                                         Scope,
         --#                                         STree.Table,
         --#                                         TheHeap &
         --#         ErrorHandler.Error_Context,
         --#         SPARK_IO.File_Sys          from CommandLineData.Content,
         --#                                         ContextManager.Ops.Unit_Stack,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         Err_Num,
         --#                                         GlobalComponentData,
         --#                                         Kind,
         --#                                         LexTokenManager.State,
         --#                                         Opt_Node,
         --#                                         Scope,
         --#                                         SPARK_IO.File_Sys,
         --#                                         STree.Table,
         --#                                         TheHeap &
         --#         Identifiers                from CommandLineData.Content,
         --#                                         ContextManager.Ops.Unit_Stack,
         --#                                         Dictionary.Dict,
         --#                                         GlobalComponentData,
         --#                                         LexTokenManager.State,
         --#                                         Opt_Node,
         --#                                         Scope,
         --#                                         STree.Table,
         --#                                         TheHeap &
         --#         Valid                      from CommandLineData.Content,
         --#                                         ContextManager.Ops.Unit_Stack,
         --#                                         Dictionary.Dict,
         --#                                         Err_Num,
         --#                                         GlobalComponentData,
         --#                                         Kind,
         --#                                         LexTokenManager.State,
         --#                                         Opt_Node,
         --#                                         Scope,
         --#                                         STree.Table,
         --#                                         TheHeap;
         is
            It                : STree.Iterator;
            Identifier_Count  : Natural := 0;
            Valid_Simple_Name : Boolean;
            Id_Str            : LexTokenManager.Lex_String;
            Sym               : Dictionary.Symbol;
            Name_Error        : Natural;

            procedure Process_Dotted_Simple_Name
              (DSN_Node : in     STree.SyntaxNode;
               Str      :    out LexTokenManager.Lex_String;
               Sym      :    out Dictionary.Symbol;
               Valid    :    out Boolean)
            --# global in     CommandLineData.Content;
            --#        in     ContextManager.Ops.Unit_Stack;
            --#        in     LexTokenManager.State;
            --#        in     Scope;
            --#        in out Dictionary.Dict;
            --#        in out ErrorHandler.Error_Context;
            --#        in out GlobalComponentData;
            --#        in out SPARK_IO.File_Sys;
            --#        in out Statistics.TableUsage;
            --#        in out STree.Table;
            --#        in out TheHeap;
            --# derives Dictionary.Dict,
            --#         GlobalComponentData,
            --#         Str,
            --#         STree.Table,
            --#         Sym,
            --#         TheHeap,
            --#         Valid                      from CommandLineData.Content,
            --#                                         ContextManager.Ops.Unit_Stack,
            --#                                         Dictionary.Dict,
            --#                                         DSN_Node,
            --#                                         GlobalComponentData,
            --#                                         LexTokenManager.State,
            --#                                         Scope,
            --#                                         STree.Table,
            --#                                         TheHeap &
            --#         ErrorHandler.Error_Context,
            --#         SPARK_IO.File_Sys          from CommandLineData.Content,
            --#                                         ContextManager.Ops.Unit_Stack,
            --#                                         Dictionary.Dict,
            --#                                         DSN_Node,
            --#                                         ErrorHandler.Error_Context,
            --#                                         GlobalComponentData,
            --#                                         LexTokenManager.State,
            --#                                         Scope,
            --#                                         SPARK_IO.File_Sys,
            --#                                         STree.Table,
            --#                                         TheHeap &
            --#         Statistics.TableUsage      from *,
            --#                                         CommandLineData.Content,
            --#                                         ContextManager.Ops.Unit_Stack,
            --#                                         Dictionary.Dict,
            --#                                         DSN_Node,
            --#                                         GlobalComponentData,
            --#                                         LexTokenManager.State,
            --#                                         Scope,
            --#                                         STree.Table,
            --#                                         TheHeap;
            is
               It                    : STree.Iterator;
               Dotted                : Boolean;
               Id_Node               : STree.SyntaxNode;
               PId_Str, Id_Str       : LexTokenManager.Lex_String;
               Local_Sym, Sym_So_Far : Dictionary.Symbol;

               function Selector_Allowed_For (Sym : Dictionary.Symbol) return Boolean
               --# global in Dictionary.Dict;
               is
               begin
                  return Dictionary.IsPackage (Sym)
                    or else (Dictionary.IsTypeMark (Sym)
                               and then (Dictionary.TypeIsRecord (Sym) or else Dictionary.TypeIsProtected (Sym)))
                    or else Dictionary.IsRecordComponent (Sym)
                    or else (Dictionary.IsObject (Sym) and then Dictionary.TypeIsRecord (Dictionary.GetType (Sym)))
                    or else (Dictionary.IsFunction (Sym) and then Dictionary.TypeIsRecord (Dictionary.GetType (Sym)))
                    or else (Dictionary.IsObject (Sym) and then Dictionary.IsProtectedType (Dictionary.GetType (Sym)));
               end Selector_Allowed_For;

            begin
               -- local grammar
               --
               -- dotted_simple_name         or     dotted_simple_name
               --         |                                 |
               --     identifier                    dotted_simple_name --- identifier
               --                                           |
               --                                       identifier

               Valid := True; -- default
                              -- See whether it is a simple identifier or not.  If it is we return a Lex_String and a Symbol
                              -- otherwise just a Symbol.  Dotted gets set True if we loop through >1 identifiers
               Dotted := False;

               -- Loop through identifiers.  Loop exits prematurely for simple identifier case
               It     := Find_First_Node (Node_Kind    => SPSymbols.identifier,
                                          From_Root    => DSN_Node,
                                          In_Direction => STree.Down);

               Id_Node := Get_Node (It);
               Id_Str := Node_Lex_String (Id_Node);
               PId_Str := LexTokenManager.Null_String;
               -- Debug.Print_Lex_Str ("String sought in Check_Identifiers is ", Id_Str);
               -- Note that the lookup uses Proof_Context because we may be trying to justify a flow error
               -- or warning involving an identifier that is not visible in Program_Context (eg an abstract
               -- own variable).
               Local_Sym := Dictionary.LookupItem (Name              => Id_Str,
                                                   Scope             => Scope,
                                                   Context           => Dictionary.ProofContext,
                                                   Full_Package_Name => False);
               loop
                  -- Debug.Print_Sym ("Local_Sym in loop in Check_Identifiers is ", Local_Sym);
                  -- any time we fail to find something it is an error failure
                  if Local_Sym = Dictionary.NullSymbol then
                     ErrorHandler.Semantic_Error2 (1, ErrorHandler.No_Reference, Node_Position (Id_Node), Id_Str, PId_Str);
                     Valid := False;
                     exit;
                  end if;
                  STree.Set_Node_Lex_String (Sym  => Local_Sym,
                                             Node => Id_Node);
                  -- set up next iteration
                  It := STree.NextNode (It);

                  exit when STree.IsNull (It);

                  -- If we get to here then there is more than one identifier

                  -- If there is more than one identifier and we are processing a record object or
                  -- record subcomponent, then there is
                  -- an extra step required:  we do not add symbols for all the components of records to the
                  -- dictionary all the time, but only where they are needed.  Therefore if we try and look
                  -- up R.F here (where R is record object) then the look up will fail because there is no
                  -- subcomponent symbol for R.F.  Therefore we must add the symbols now so that the
                  -- LookUpSelectedItem below will succeed.
                  if Dictionary.IsVariableOrSubcomponent (Local_Sym)
                    and then Dictionary.TypeIsRecord (Dictionary.GetType (Local_Sym)) then
                     -- Debug.Print_Msg ("Adding subcomponents in Wf_Justification_Statement", True);
                     AddRecordSubComponents
                       (RecordVarSym  => Local_Sym,
                        RecordTypeSym => Dictionary.GetType (Local_Sym),
                        ComponentData => GlobalComponentData);
                  end if;
                  -- end of sub component addition

                  -- Because there is more than identifier we save some context for next time round the loop
                  Dotted     := True;
                  PId_Str    := Id_Str;
                  Id_Node    := Get_Node (It);
                  Id_Str     := Node_Lex_String (Id_Node);
                  Sym_So_Far := Local_Sym;  -- needed for trapping P.P.P.P.X case later on

                  -- At this point we have a prefix in Local_Sym and we are about to process
                  -- a selector.  Local_Sym had better be the kind of thing that can have a
                  -- selector.
                  if not Selector_Allowed_For (Sym => Local_Sym) then
                     ErrorHandler.Semantic_Error_Sym (9, ErrorHandler.No_Reference, Node_Position (Id_Node), Local_Sym, Scope);

                     Local_Sym := Dictionary.NullSymbol;
                     exit;
                  end if;

                  -- Debug.Print_Lex_Str ("String sought in loop in Check_Identifiers is ", Id_Str);
                  -- Debug.Print_Sym ("             looking it up in ", Local_Sym);
                  -- Note that the lookup uses Proof_Context because we may be trying to justify a flow error
                  -- or warning involving an identifier that is not visible in Program_Context (eg an abstract
                  -- own variable).
                  Local_Sym :=
                    Dictionary.LookupSelectedItem
                    (Prefix   => Local_Sym,
                     Selector => Id_Str,
                     Scope    => Scope,
                     Context  => Dictionary.ProofContext);
                  -- check to see if we are getting the same symbol over and again
                  if Local_Sym = Sym_So_Far then            -- P.P.P.P.X case
                     Local_Sym := Dictionary.NullSymbol;  -- to cause "Not visible" error at top of loop
                  end if;

               end loop;

               -- return results
               if Dotted then
                  Str := LexTokenManager.Null_String;
               else
                  Str := Id_Str;
               end if;
               Sym       := Local_Sym;
            end Process_Dotted_Simple_Name;

            -- This function checks the number of names in the accept
            -- annotation against the error number. It returns the error message
            -- to report, or 0 if the clause is semantically correct.
            --
            -- Unfortunately Flow Error 10 can have either 0 or 1 identifiers.
            --
            -- Flow Errors 50 and 602 can have 1 or 2 identifiers, depending
            -- on whether the enclosing program unit is a function (1 identifier needed)
            -- or not (2 identifiers needed for procedures or task bodies).
            function Justification_Name_Length_Error (Enclosing_Region_Is_A_Function : in Boolean) return Natural
            --# global in Err_Num;
            --#        in Identifier_Count;
            --#        in Kind;
            is
               Ret_Val : Natural := 0;
            begin
               case Kind is
                  when ErrorHandler.Flow_Message =>
                     case Err_Num is
                        -- These flow errors require exactly zero names to be justified
                        when 22 | 40 | 41 =>
                           if Identifier_Count /= 0 then
                              Ret_Val := 124;
                           end if;
                           -- These flow errors require exactly two names to be justified
                        when 3 | 4 | 57 | 601 | 605 | 606 =>
                           if Identifier_Count /= 2 then
                              Ret_Val := 126;
                           end if;
                           -- Flow Error 10 (ineffective expression or statement) can require
                           -- either zero or one name
                        when 10 =>
                           if Identifier_Count = 0 or Identifier_Count = 1 then
                              Ret_Val := 0;
                           else
                              Ret_Val := 127;
                           end if;

                           -- Flow errors 50 and 602 can require one or two names,
                           -- depending on Enclosing_Region_Is_A_Function
                        when 50 | 602 =>
                           if Enclosing_Region_Is_A_Function then
                              -- function - 1 identifier needed
                              if Identifier_Count = 1 then
                                 Ret_Val := 0;
                              else
                                 Ret_Val := 125;
                              end if;
                           else
                              -- procedure or task body - 2 identifiers needed
                              if Identifier_Count = 2 then
                                 Ret_Val := 0;
                              else
                                 Ret_Val := 126;
                              end if;
                           end if;

                           -- All other flow errors require exactly one name
                        when others =>
                           if Identifier_Count /= 1 then
                              Ret_Val := 125;
                           end if;
                     end case;
                  when ErrorHandler.Warning_Message =>
                     case Err_Num is
                        -- The following warnings require exactly 1 name to be justified
                        when 1   |
                          5   |
                          9   |
                          10  |
                          12  |
                          13  |
                          169 |
                          311 |
                          312 |
                          313 |
                          314 |
                          350 |
                          351 |
                          391 |
                          392 |
                          393 |
                          394 |
                          395 |
                          396 |
                          397 |
                          398 |
                          400 |
                          403 |
                          410 |
                          411 |
                          412 |
                          413 =>
                           if Identifier_Count /= 1 then
                              Ret_Val := 125;
                           end if;

                           -- All other warnings require exactly zero names
                        when others =>
                           if Identifier_Count /= 0 then
                              Ret_Val := 124;
                           end if;
                     end case;
               end case;
               return Ret_Val;
            end Justification_Name_Length_Error;

         begin -- Check_Identifiers

            -- local grammar
            -- justification_opt        or       justification_opt
            --       |                                   |
            --    nothing                        dotted_simple_name_list
            --                                           |
            --                                   dotted_simple_name_list---dotted_simple_name
            --                                                 |
            --                                   dotted_simple_name
            --
            -- Rules:
            --   (1) Between 0 and ErrorHandler.Max_Justification_Identifier_Length identifiers found
            --   (2) Each identifier must be declared and visible in Scope
            --   (3) Identifiers (I) is populated with each legal identifier thus:
            --        (a) If the identifier has no dots in it, then we store the Lex_String AND the looked-up symbol
            --        (b) If it has dots then we store a null lex string and the looked-up symbol
            --            (this complexity is because we don't know whether warnings will be passed to the errohandler
            --             using, for example, Semantic_Warning or Semantic_Warning_Sym so we need to match either)

            -- Establish default result
            Identifiers := ErrorHandler.Null_Justification_Identifiers;
            Valid       := True;

            -- Iterate through dotted_simple_name nodes
            It := Find_First_Node (Node_Kind    => SPSymbols.dotted_simple_name,
                                   From_Root    => Opt_Node,
                                   In_Direction => STree.Down);
            if not STree.IsNull (It) then
               loop
                  Identifier_Count := Identifier_Count + 1;

                  -- We have a single dotted simple name to process.  If it is OK then it will go in the
                  -- identifier list at index position Identifier_Count
                  Process_Dotted_Simple_Name (DSN_Node => Get_Node (It),
                                              Str      => Id_Str,
                                              Sym      => Sym,
                                              Valid    => Valid_Simple_Name);
                  if Valid_Simple_Name then
                     if Identifier_Count in ErrorHandler.Justification_Identifier_Index then
                        Identifiers (Identifier_Count) :=
                          ErrorHandler.Justification_Identifier'(String_Form => Id_Str,
                                                                 Symbol_Form => Sym);
                     end if;
                  else
                     Valid := False; -- don't add clause at all if any part is malformed
                  end if;

                  if STree.IsNull (STree.NextNode (It)) then
                     exit;
                  end if;

                  It := STree.NextNode (It);
               end loop;
            end if;

            Name_Error :=
              Justification_Name_Length_Error
              (Enclosing_Region_Is_A_Function => Dictionary.IsFunction (Dictionary.GetRegion (Scope)));

            if Name_Error /= 0 then
               ErrorHandler.Semantic_Error
                 (Name_Error,
                  ErrorHandler.No_Reference,
                  Node_Position (Next_Sibling (Child_Node (Parent_Node (Opt_Node)))),
                  LexTokenManager.Null_String);
               Valid := False; -- don't add clause at all if any part is malformed
            end if;
         end Check_Identifiers;

         procedure Handle_Function_Return
           (Kind        : in     ErrorHandler.Justification_Kinds;
            Err_Num     : in     Natural;
            Identifiers : in out ErrorHandler.Justification_Identifiers)
         --# derives Identifiers from *,
         --#                          Err_Num,
         --#                          Kind;
         is

            function Message_Is_IFA (Err_Num : Natural) return Boolean is
            begin
               return Err_Num = 50 or else Err_Num = 602;
            end Message_Is_IFA;

         begin
            -- If the users has tried to justify an information flow error where the "export" is the function
            -- return result, then there will only be one variable name in the message (which will say, e.g.,
            -- "The function value is not derived from the imported value(s) of Y.") but the pattern matching
            -- in ErrorHandler.Justification.Check_Whether_Justified will still be expecting an two symbols, an
            -- export followed by an import.  The Examiner's flow analyser uses Null_Symbol to represent the
            -- function return value.  In this procedure we:
            --    (1) Detect cases where only one argument has been supplied for an IFA msg that needs two
            --    (2) Assume in that case that a function return in implicitly intended
            --    (3) Move the given variable to the second, import, slot.
            --    (4) Insert a null identifier in the first slot (this will match Null_Symbol).
            -- Note that this transformation is "safe" even if the user has simply forgotten a variable name
            -- because the transformed annotation will not pattern match any more than it would have before.
            -- e.g. User in intends "F, 50, X, Y" but types "F, 50, X" by mistake.  Transformation gives
            -- "F, 50, Null_Sym, X".  Neither original incorrect form nor transformed form will pattern match
            -- so behaviour is unaltered

            if Kind = ErrorHandler.Flow_Message and then Message_Is_IFA (Err_Num => Err_Num) then
               -- possibly something to do
               if Identifiers (2) = ErrorHandler.Null_Justification_Identifier then
                  -- only one identifier was supplied so transformation is needed
                  Identifiers (2) := Identifiers (1); -- move given variable to second place
                  Identifiers (1) := ErrorHandler.Null_Justification_Identifier; -- to match Null_Sym
               end if;
            end if;
         end Handle_Function_Return;

      begin --Wf_Justification_Clause
            -- grammar: justification_clause
            --               |
            --          identifier--- numeric_literal---justification_opt---string_literal
            --                                                 |
            --                                          dotted_simple_name_list (or null)
            --                                                 |
            --                                          dotted_simple_name_list---dotted_simple_name
            --                                                 |
            --                                          dotted_simple_name
         Error_Found := False;

         -- Check whether we are dealing with Flow_Message or Warning_Message --------
         Current_Node := Child_Node (Node);
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Current_Node) = SPSymbols.identifier,
            Sys_Err => SystemErrors.Assertion_Failure,
            Msg     => "Failed to find indentifier giving error kind in justification clause");
         Check_Kind (Lex_String => Node_Lex_String (Current_Node),
                     Kind       => Kind,
                     Valid      => Valid);

         if not Valid then
            Error_Found := True;
            ErrorHandler.Semantic_Error
              (121,
               ErrorHandler.No_Reference,
               Node_Position (Current_Node),
               LexTokenManager.Null_String);
         end if;

         -- Check error number ---------------------------------------------------------
         Current_Node := Next_Sibling (Current_Node);
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Current_Node) = SPSymbols.numeric_literal,
            Sys_Err => SystemErrors.Assertion_Failure,
            Msg     => "Failed to find error number in justification clause");
         -- Use existing wff to get literal
         -- local grammar
         -- numeric_literal
         --       |
         -- decimal_literal
         --       |
         -- integer_number
         Err_Num_Node := Child_Node (Child_Node (Current_Node));
         if Syntax_Node_Type (Err_Num_Node) = SPSymbols.integer_number then
            GetLiteralValue (Child_Node (Child_Node (Current_Node)), -- drill down to the integer_number
                             -- to get
                             Val);
            Maths.ValueToInteger (Val, Err_Num, Maths_Valid);

            Valid := Maths_Valid = Maths.NoError;
            if not Valid then
               Error_Found := True;
               Err_Num     := 0;
            end if;
         else -- wrong kind of number
            Error_Found := True;
            Err_Num     := 0;
         end if;
         -- We should have a valid positive integer value for Err_Num by here.  If not, raise error
         if Err_Num = 0 then
            ErrorHandler.Semantic_Error
              (122,
               ErrorHandler.No_Reference,
               Node_Position (Err_Num_Node),
               LexTokenManager.Null_String);

         elsif Is_Disallowed_Warning (Kind    => Kind,
                                      Err_Num => Err_Num) then
            -- we have a wellformed warning number but we may want to disallow certain warning numbers
            ErrorHandler.Semantic_Error
              (123,
               ErrorHandler.No_Reference,
               Node_Position (Err_Num_Node),
               LexTokenManager.Null_String);
            Error_Found := True;
         end if;

         -- Check identifiers --------- ---------------------------------------------------------
         Current_Node := Next_Sibling (Current_Node);
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Current_Node) = SPSymbols.justification_opt,
            Sys_Err => SystemErrors.Assertion_Failure,
            Msg     => "Failed to find identifier list location in justification clause");

         Check_Identifiers (Opt_Node    => Current_Node,
                            Identifiers => Identifiers,
                            Valid       => Valid);
         if not Valid then  -- I think this is clearer that Error_FOund := Error_Found or not Valid;
            Error_Found := True;
         end if;

         -- Check explanation --------- ---------------------------------------------------------
         Current_Node := Next_Sibling (Current_Node);
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Current_Node) = SPSymbols.justification_string,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Current_Node = justification_string in Wf_Justification_Clause");
         Explanation := E_Strings.Empty_String;
         while Syntax_Node_Type (Current_Node) = SPSymbols.justification_string loop
            Current_Node := Child_Node (Current_Node);
            SystemErrors.RT_Assert
              (C       => Syntax_Node_Type (Current_Node) = SPSymbols.string_literal,
               Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Current_Node = string_literal in Wf_Justification_Clause");
            if not E_Strings.Is_Empty (Explanation) then
               E_Strings.Append_Char (E_Str => Explanation,
                                      Ch    => ' ');
            end if;
            E_Strings.Append_Examiner_String
              (E_Str1 => Explanation,
               E_Str2 => LexTokenManager.Lex_String_To_String (Node_Lex_String (Current_Node)));
            Current_Node := Next_Sibling (Current_Node);
         end loop;

         -- Insert justification data in error handler data table
         if not Error_Found then

            -- See whether Identifiers needs transforming to handle IFA errors on function return
            Handle_Function_Return (Kind        => Kind,
                                    Err_Num     => Err_Num,
                                    Identifiers => Identifiers);

            -- Finally, add it to table of justification
            ErrorHandler.Start_Justification
              (Node_Position (Node),
               Start_Line,
               Kind,
               Err_Num,
               Identifiers,
               Explanation,
               -- to get
               Maximum_Justifications_Reached);
            if Maximum_Justifications_Reached then
               ErrorHandler.Semantic_Warning (122, Node_Position (Node), LexTokenManager.Null_String);
            end if;
         end if;
      end Wf_Justification_Clause;

   begin -- Wf_Start_Justification
      It := Find_First_Node (Node_Kind    => SPSymbols.justification_clause,
                             From_Root    => Node,
                             In_Direction => STree.Down);
      while not STree.IsNull (It) loop
         Wf_Justification_Clause (Start_Line => Node_Position (Node).Start_Line_No,
                                  Node       => Get_Node (It),
                                  Scope      => Scope);
         It := STree.NextNode (It);
      end loop;
   end Wf_Start_Justification;

   ----------------------------

   procedure Wf_End_Justification (Node : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   is
      Unmatched_End : Boolean;
   begin
      ErrorHandler.End_Justification (Node_Position (Node).Start_Line_No,
                                      -- to get
                                      Unmatched_End);
      if Unmatched_End then
         ErrorHandler.Semantic_Warning (120, Node_Position (Node), LexTokenManager.Null_String);
      end if;
   end Wf_End_Justification;

   ----------------------------

begin -- Wf_Justification_Statement
   if Syntax_Node_Type (Child_Node (Node)) = SPSymbols.start_justification then
      Wf_Start_Justification (Node  => Node,
                              Scope => Scope);
   elsif Syntax_Node_Type (Child_Node (Node)) = SPSymbols.end_justification then
      Wf_End_Justification (Node => Node);
   else
      SystemErrors.Fatal_Error
        (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Unexpected node type found in Wf_Justification_Statement");
   end if;

   if ErrorHandler.Generate_SLI then
      SLI.Generate_Xref_Justification (Comp_Unit  => ContextManager.Ops.Current_Unit,
                                       Parse_Tree => Node,
                                       Scope      => Scope);
   end if;
end Wf_Justification_Statement;
