-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with SLI;

separate (Sem.CompUnit)
procedure Wf_Generic_Declaration (Node          : in STree.SyntaxNode;
                                  Current_Scope : in Dictionary.Scopes) is
   Current_Node : STree.SyntaxNode;

   procedure Process_Generic_Formal_Part
     (Node                 : in     STree.SyntaxNode;
      Generic_Unit         : in     Dictionary.Symbol;
      Error_In_Declaration : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         Error_In_Declaration,
   --#         STree.Table                from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Generic_Unit,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Generic_Unit,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   is
      Object_Or_Type_Node : STree.SyntaxNode;
      Rep_Node            : STree.SyntaxNode;

      procedure Process_Formal_Object
        (Node                 : in     STree.SyntaxNode;
         Generic_Unit         : in     Dictionary.Symbol;
         Error_In_Declaration : in out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --#        in out STree.Table;
      --# derives Dictionary.Dict,
      --#         Error_In_Declaration       from *,
      --#                                         CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         Generic_Unit,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table &
      --#         ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         Generic_Unit,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table &
      --#         STree.Table                from *,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         Generic_Unit,
      --#                                         LexTokenManager.State,
      --#                                         Node;
      is
         Ident_List_Node : STree.SyntaxNode;
         Mode_Node       : STree.SyntaxNode;
         Type_Node       : STree.SyntaxNode;
         Formal_Type     : Dictionary.Symbol;

         procedure Process_Identifiers
           (Node                 : in     STree.SyntaxNode;
            Type_Sym             : in     Dictionary.Symbol;
            Generic_Unit         : in     Dictionary.Symbol;
            Error_In_Declaration : in out Boolean)
         --# global in     CommandLineData.Content;
         --#        in     ContextManager.Ops.Unit_Stack;
         --#        in     LexTokenManager.State;
         --#        in     STree.Table;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.Error_Context;
         --#        in out SPARK_IO.File_Sys;
         --# derives Dictionary.Dict,
         --#         Error_In_Declaration       from *,
         --#                                         ContextManager.Ops.Unit_Stack,
         --#                                         Dictionary.Dict,
         --#                                         Generic_Unit,
         --#                                         LexTokenManager.State,
         --#                                         Node,
         --#                                         STree.Table,
         --#                                         Type_Sym &
         --#         ErrorHandler.Error_Context,
         --#         SPARK_IO.File_Sys          from CommandLineData.Content,
         --#                                         ContextManager.Ops.Unit_Stack,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         Generic_Unit,
         --#                                         LexTokenManager.State,
         --#                                         Node,
         --#                                         SPARK_IO.File_Sys,
         --#                                         STree.Table,
         --#                                         Type_Sym;
         is
            It        : STree.Iterator;
            Next_Node : STree.SyntaxNode;
            Ident_Str : LexTokenManager.Lex_String;
            Sym       : Dictionary.Symbol;
         begin
            -- ASSUME Node = identifier_list
            SystemErrors.RT_Assert
              (C       => Syntax_Node_Type (Node => Node) = SPSymbols.identifier_list,
               Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Node = identifier_list in Process_Identifiers");

            It := Find_First_Node (Node_Kind    => SPSymbols.identifier,
                                   From_Root    => Node,
                                   In_Direction => STree.Down);
            while not STree.IsNull (It) loop
               Next_Node := Get_Node (It => It);
               -- ASSUME Next_Node = identifier
               Ident_Str := Node_Lex_String (Node => Next_Node);
               Sym       :=
                 Dictionary.LookupItem (Name              => Ident_Str,
                                        Scope             => Dictionary.GlobalScope,
                                        Context           => Dictionary.ProofContext,
                                        Full_Package_Name => False);
               if Sym = Dictionary.NullSymbol then
                  Dictionary.AddGenericFormalParameter
                    (Name                => Ident_Str,
                     Comp_Unit           => ContextManager.Ops.Current_Unit,
                     Declaration         => Dictionary.Location'(Start_Position => Node_Position (Node => Next_Node),
                                                                 End_Position   => Node_Position (Node => Next_Node)),
                     SubprogramOrPackage => Generic_Unit,
                     TypeMark            => Type_Sym,
                     Kind                => Dictionary.GenericObjectParameter);
               else
                  -- already exists
                  Error_In_Declaration := True;
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 10,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Next_Node),
                     Id_Str    => Ident_Str);
               end if;
               It := STree.NextNode (It);
            end loop;
         end Process_Identifiers;

      begin -- Process_Formal_Object

         -- ASSUME Node = formal_object_declaration
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Node) = SPSymbols.formal_object_declaration,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Node = formal_object_declaration in Process_Formal_Object");

         Ident_List_Node := Child_Node (Current_Node => Node);

         -- ASSUME Ident_List_Node = identifier_list
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Ident_List_Node) = SPSymbols.identifier_list,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Ident_List_Node = identifier_list in Process_Formal_Object");

         Mode_Node := Child_Node (Current_Node => Next_Sibling (Current_Node => Ident_List_Node));

         -- ASSUME Mode_Node = in_mode OR inout_mode OR out_mode OR NULL
         SystemErrors.RT_Assert
           (C       => Mode_Node = STree.NullNode
              or else Syntax_Node_Type (Node => Mode_Node) = SPSymbols.in_mode
              or else Syntax_Node_Type (Node => Mode_Node) = SPSymbols.inout_mode
              or else Syntax_Node_Type (Node => Mode_Node) = SPSymbols.out_mode,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Mode_Node = in_mode OR inout_mode OR out_mode OR NULL in Process_Formal_Object");

         Type_Node := Next_Sibling (Current_Node => Next_Sibling (Current_Node => Ident_List_Node));
         -- ASSUME Type_Node = type_mark
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Type_Node) = SPSymbols.type_mark,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Type_Node = type_mark in Process_Formal_Object");

         if Syntax_Node_Type (Node => Mode_Node) = SPSymbols.inout_mode
           or else Syntax_Node_Type (Node => Mode_Node) = SPSymbols.out_mode then
            Error_In_Declaration := True;
            ErrorHandler.Semantic_Error
              (Err_Num   => 639,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Ident_List_Node),
               Id_Str    => LexTokenManager.Null_String);
         end if;

         Wf_Type_Mark
           (Node          => Type_Node,
            Current_Scope => Dictionary.LocalScope (Generic_Unit),
            Context       => Dictionary.ProgramContext,
            Type_Sym      => Formal_Type);
         if Formal_Type = Dictionary.GetUnknownTypeMark then
            Error_In_Declaration := True;
         else -- valid type mark so
            Process_Identifiers
              (Node                 => Ident_List_Node,
               Type_Sym             => Formal_Type,
               Generic_Unit         => Generic_Unit,
               Error_In_Declaration => Error_In_Declaration);
         end if;
      end Process_Formal_Object;

      -------------

      procedure Process_Formal_Type
        (Node                 : in     STree.SyntaxNode;
         Generic_Unit         : in     Dictionary.Symbol;
         Error_In_Declaration : in out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.File_Heap;
      --#        in     ContextManager.Ops.Unit_Heap;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SLI.State;
      --#        in out SPARK_IO.File_Sys;
      --#        in out STree.Table;
      --# derives Dictionary.Dict,
      --#         Error_In_Declaration,
      --#         STree.Table                from *,
      --#                                         CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         Generic_Unit,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table &
      --#         ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         ContextManager.Ops.File_Heap,
      --#                                         ContextManager.Ops.Unit_Heap,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         Generic_Unit,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SLI.State,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table &
      --#         SLI.State                  from *,
      --#                                         CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Heap,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         Generic_Unit,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table;
      is
         Ident_Node             : STree.SyntaxNode;
         Type_Node              : STree.SyntaxNode;
         Ident_Str              : LexTokenManager.Lex_String;
         Declaration            : Dictionary.Location;
         Type_Mark              : Dictionary.Symbol;
         Error_In_Generic_Array : Boolean;
      begin
         -- ASSUME Node = formal_type_declaration
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Node) = SPSymbols.formal_type_declaration,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Node = formal_type_declaration in Process_Formal_Type");

         Ident_Node := Child_Node (Current_Node => Node);
         -- ASSUME Ident_Node = indentifier
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Ident_Node = indentifier in Process_Formal_Type");
         Ident_Str := Node_Lex_String (Node => Ident_Node);

         -- check whether name already in use
         if Dictionary.IsDefined (Name              => Ident_Str,
                                  Scope             => Dictionary.GlobalScope,
                                  Context           => Dictionary.ProofContext,
                                  Full_Package_Name => False) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 10,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Ident_Node),
               Id_Str    => Ident_Str);
            Error_In_Declaration := True;
         else -- OK to add
            Type_Node := Child_Node (Current_Node => Next_Sibling (Current_Node => Ident_Node));
            -- ASSUME Type_Node = formal_private_type_definition OR formal_discrete_type_definition OR
            --                    formal_signed_integer_type_definition OR formal_modular_type_definition OR
            --                    formal_floating_point_definition OR formal_ordinary_fixed_point_definition OR
            --                    formal_array_type_definition
            Declaration :=
              Dictionary.Location'
              (Start_Position => Node_Position (Node => Ident_Node),
               End_Position   => Node_Position (Node => Ident_Node));
            case Syntax_Node_Type (Node => Type_Node) is
               when SPSymbols.formal_private_type_definition =>
                  -- ASSUME Type_Node = formal_private_type_definition
                  Dictionary.AddGenericPrivateType
                    (Name        => Ident_Str,
                     Comp_Unit   => ContextManager.Ops.Current_Unit,
                     Declaration => Declaration,
                     Scope       => Dictionary.LocalScope (Generic_Unit),
                     Type_Sym    => Type_Mark);
                  if ErrorHandler.Generate_SLI then
                     SLI.Generate_Xref_Symbol
                       (Comp_Unit      => ContextManager.Ops.Current_Unit,
                        Parse_Tree     => Ident_Node,
                        Symbol         => Type_Mark,
                        Is_Declaration => True);
                  end if;
               when SPSymbols.formal_discrete_type_definition =>
                  -- ASSUME Type_Node = formal_discrete_type_definition
                  Dictionary.AddGenericOrderedDiscreteType
                    (Name        => Ident_Str,
                     Comp_Unit   => ContextManager.Ops.Current_Unit,
                     Declaration => Declaration,
                     Scope       => Dictionary.LocalScope (Generic_Unit),
                     Type_Sym    => Type_Mark);
                  if ErrorHandler.Generate_SLI then
                     SLI.Generate_Xref_Symbol
                       (Comp_Unit      => ContextManager.Ops.Current_Unit,
                        Parse_Tree     => Ident_Node,
                        Symbol         => Type_Mark,
                        Is_Declaration => True);
                  end if;
               when SPSymbols.formal_signed_integer_type_definition =>
                  -- ASSUME Type_Node = formal_signed_integer_type_definition
                  Dictionary.AddGenericIntegerType
                    (Name        => Ident_Str,
                     Comp_Unit   => ContextManager.Ops.Current_Unit,
                     Declaration => Declaration,
                     Scope       => Dictionary.LocalScope (Generic_Unit),
                     Type_Sym    => Type_Mark);
                  if ErrorHandler.Generate_SLI then
                     SLI.Generate_Xref_Symbol
                       (Comp_Unit      => ContextManager.Ops.Current_Unit,
                        Parse_Tree     => Ident_Node,
                        Symbol         => Type_Mark,
                        Is_Declaration => True);
                  end if;
               when SPSymbols.formal_modular_type_definition =>
                  -- ASSUME Type_Node = formal_modular_type_definition
                  Dictionary.AddGenericModularType
                    (Name        => Ident_Str,
                     Comp_Unit   => ContextManager.Ops.Current_Unit,
                     Declaration => Declaration,
                     Scope       => Dictionary.LocalScope (Generic_Unit),
                     Type_Sym    => Type_Mark);
                  if ErrorHandler.Generate_SLI then
                     SLI.Generate_Xref_Symbol
                       (Comp_Unit      => ContextManager.Ops.Current_Unit,
                        Parse_Tree     => Ident_Node,
                        Symbol         => Type_Mark,
                        Is_Declaration => True);
                  end if;
               when SPSymbols.formal_floating_point_definition =>
                  -- ASSUME Type_Node = formal_floating_point_definition
                  Dictionary.AddGenericFloatingPointType
                    (Name        => Ident_Str,
                     Comp_Unit   => ContextManager.Ops.Current_Unit,
                     Declaration => Declaration,
                     Scope       => Dictionary.LocalScope (Generic_Unit),
                     Type_Sym    => Type_Mark);
                  if ErrorHandler.Generate_SLI then
                     SLI.Generate_Xref_Symbol
                       (Comp_Unit      => ContextManager.Ops.Current_Unit,
                        Parse_Tree     => Ident_Node,
                        Symbol         => Type_Mark,
                        Is_Declaration => True);
                  end if;
               when SPSymbols.formal_ordinary_fixed_point_definition =>
                  -- ASSUME Type_Node = formal_ordinary_fixed_point_definition
                  Dictionary.AddGenericFixedPointType
                    (Name        => Ident_Str,
                     Comp_Unit   => ContextManager.Ops.Current_Unit,
                     Declaration => Declaration,
                     Scope       => Dictionary.LocalScope (Generic_Unit),
                     Type_Sym    => Type_Mark);
                  if ErrorHandler.Generate_SLI then
                     SLI.Generate_Xref_Symbol
                       (Comp_Unit      => ContextManager.Ops.Current_Unit,
                        Parse_Tree     => Ident_Node,
                        Symbol         => Type_Mark,
                        Is_Declaration => True);
                  end if;
               when SPSymbols.formal_array_type_definition =>
                  -- ASSUME Type_Node = formal_array_type_definition
                  Wf_Array_Type_Definition
                    (Node         => Child_Node (Current_Node => Type_Node),
                     Scope        => Dictionary.LocalScope (Generic_Unit),
                     Ident_Node   => Ident_Node,
                     Dec_Loc      => Node_Position (Node => Node),
                     Is_Generic   => True,
                     Errors_Found => Error_In_Generic_Array,
                     The_Array    => Type_Mark);
                  Error_In_Declaration := Error_In_Declaration or else Error_In_Generic_Array;
               when others =>
                  Type_Mark := Dictionary.GetUnknownTypeMark; -- just to avoid data flow error
                  SystemErrors.Fatal_Error
                    (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                     Msg     => "Expect Type_Node = formal_private_type_definition OR formal_discrete_type_definition OR " &
                       "formal_signed_integer_type_definition OR formal_modular_type_definition OR " &
                       "formal_floating_point_definition OR formal_ordinary_fixed_point_definition OR " &
                       "formal_array_type_definition in Process_Formal_Type");
            end case;
            Dictionary.AddGenericFormalParameter
              (Name                => LexTokenManager.Null_String,
               Comp_Unit           => ContextManager.Ops.Current_Unit,
               Declaration         => Declaration,
               SubprogramOrPackage => Generic_Unit,
               TypeMark            => Type_Mark,
               Kind                => Dictionary.GenericTypeParameter);
         end if;
      end Process_Formal_Type;

   begin -- Process_Generic_Formal_Part

      -- ASSUME Node = generic_formal_part
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.generic_formal_part,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = generic_formal_part in Process_Generic_Formal_Part");

      Rep_Node := Last_Child_Of (Start_Node => Node);
      while Next_Sibling (Current_Node => Rep_Node) /= STree.NullNode loop
         -- ASSUME Rep_Node = generic_formal_part_rep
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Rep_Node) = SPSymbols.generic_formal_part_rep,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Rep_Node = generic_formal_part_rep in Process_Generic_Formal_Part");
         Object_Or_Type_Node := Next_Sibling (Current_Node => Rep_Node);
         -- ASSUME Object_Or_Type_Node = formal_object_declaration OR formal_type_declaration
         case Syntax_Node_Type (Node => Object_Or_Type_Node) is
            when SPSymbols.formal_object_declaration =>
               -- ASSUME Object_Or_Type_Node = formal_object_declaration
               Process_Formal_Object
                 (Node                 => Object_Or_Type_Node,
                  Generic_Unit         => Generic_Unit,
                  Error_In_Declaration => Error_In_Declaration);
            when SPSymbols.formal_type_declaration =>
               -- ASSUME Object_Or_Type_Node = formal_type_declaration
               Process_Formal_Type
                 (Node                 => Object_Or_Type_Node,
                  Generic_Unit         => Generic_Unit,
                  Error_In_Declaration => Error_In_Declaration);
            when others =>
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                  Msg     => "Expect Object_Or_Type_Node = formal_object_declaration OR formal_type_declaration in Process_Generic_Formal_Part");
         end case;
         Rep_Node := Parent_Node (Current_Node => Rep_Node);
      end loop;
   end Process_Generic_Formal_Part;

   -----------------------------------------------------------------------------------

   procedure Process_Generic_Subprogram_Declaration (Node : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   is
      type Subprogram_Kinds is (A_Function, A_Procedure, Unknown);

      Formal_Part_Node            : STree.SyntaxNode;
      Subprogram_Declaration_Node : STree.SyntaxNode;
      Generic_Subprogam_Symbol    : Dictionary.Symbol;
      Subprogram_Kind             : Subprogram_Kinds;
      Error_In_Declaration        : Boolean := False;

      procedure Check_And_Add_Subprogram
        (Node            : in     STree.SyntaxNode;
         Subprog_Sym     :    out Dictionary.Symbol;
         Subprogram_Kind :    out Subprogram_Kinds)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dictionary.Dict,
      --#         Subprog_Sym                from ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table &
      --#         ErrorHandler.Error_Context from *,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table &
      --#         SPARK_IO.File_Sys          from *,
      --#                                         CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table &
      --#         Subprogram_Kind            from Node,
      --#                                         STree.Table;
      is
         Ident_Node            : STree.SyntaxNode;
         Ident_Str             : LexTokenManager.Lex_String;
         Subprog_Sym_Local     : Dictionary.Symbol;
         Subprogram_Kind_Local : Subprogram_Kinds;
      begin
         -- ASSUME Node = not_overriding_subprogram_declaration
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Node) = SPSymbols.not_overriding_subprogram_declaration,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Node = not_overriding_subprogram_declaration in Check_And_Add_Subprogram");

         Ident_Node := Child_Node (Current_Node => Node);
         -- ASSUME Ident_Node = function_specification OR procedure_specification
         if Syntax_Node_Type (Node => Ident_Node) = SPSymbols.function_specification then
            -- ASSUME Ident_Node = function_specification
            Subprogram_Kind_Local := A_Function;
         elsif Syntax_Node_Type (Node => Ident_Node) = SPSymbols.procedure_specification then
            -- ASSUME Ident_Node = procedure_specification
            Subprogram_Kind_Local := A_Procedure;
         else
            Subprogram_Kind_Local := Unknown;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Ident_Node = function_specification OR procedure_specification in Check_And_Add_Subprogram");
         end if;
         -- find identifier
         Ident_Node := Child_Node (Current_Node => Child_Node (Current_Node => Ident_Node));
         -- ASSUME Ident_Node = identifier
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect ident_Node = identifier in Check_And_Add_Subprogram");
         Ident_Str := Node_Lex_String (Node => Ident_Node);
         if Dictionary.IsDefined (Name              => Ident_Str,
                                  Scope             => Dictionary.GlobalScope,
                                  Context           => Dictionary.ProofContext,
                                  Full_Package_Name => False) then
            Subprog_Sym_Local := Dictionary.NullSymbol;
            ErrorHandler.Semantic_Error
              (Err_Num   => 10,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str    => Ident_Str);
         else
            Dictionary.AddSubprogram
              (Name          => Ident_Str,
               Comp_Unit     => ContextManager.Ops.Current_Unit,
               Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                     End_Position   => Node_Position (Node => Node)),
               Scope         => Dictionary.GlobalScope,
               Context       => Dictionary.ProgramContext,
               Subprogram    => Subprog_Sym_Local);
            Dictionary.SetSubprogramIsGeneric (Subprog_Sym_Local);
         end if;
         Subprog_Sym     := Subprog_Sym_Local;
         Subprogram_Kind := Subprogram_Kind_Local;
      end Check_And_Add_Subprogram;

      ---------------

      procedure Process_Formal_Part (Node           : in STree.SyntaxNode;
                                     The_Subprogram : in Dictionary.Symbol)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --#        in out STree.Table;
      --# derives Dictionary.Dict,
      --#         STree.Table                from CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table,
      --#                                         The_Subprogram &
      --#         ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table,
      --#                                         The_Subprogram;
      is
         Formal_Part_Node : STree.SyntaxNode;
      begin
         -- ASSUME Node = not_overriding_subprogram_declaration
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Node) = SPSymbols.not_overriding_subprogram_declaration,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Node = not_overriding_subprogram_declaration in Process_Formal_Part");

         Formal_Part_Node := Next_Sibling (Current_Node => Child_Node (Current_Node => Child_Node (Current_Node => Node)));
         -- ASSUME Formal_Part_Node = formal_part OR type_mark OR NULL
         if Syntax_Node_Type (Node => Formal_Part_Node) = SPSymbols.formal_part then
            -- ASSUME Formal_Part_Node = formal_part
            Wf_Formal_Part
              (Node             => Formal_Part_Node,
               Current_Scope    => Dictionary.LocalScope (The_Subprogram),
               Subprog_Sym      => The_Subprogram,
               First_Occurrence => True,
               Context          => Dictionary.ProgramContext);
         elsif Formal_Part_Node /= STree.NullNode
           and then Syntax_Node_Type (Node => Formal_Part_Node) /= SPSymbols.type_mark then
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Formal_Part_Node = formal_part OR type_mark OR NULL in Process_Formal_Part");
         end if;
      end Process_Formal_Part;

      ---------------

      procedure Process_Return
        (Node                 : in     STree.SyntaxNode;
         The_Function         : in     Dictionary.Symbol;
         Error_In_Declaration : in out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --#        in out STree.Table;
      --# derives Dictionary.Dict            from *,
      --#                                         CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table,
      --#                                         The_Function &
      --#         ErrorHandler.Error_Context from *,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table,
      --#                                         The_Function &
      --#         Error_In_Declaration,
      --#         STree.Table                from *,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table,
      --#                                         The_Function &
      --#         SPARK_IO.File_Sys          from *,
      --#                                         CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table,
      --#                                         The_Function;
      is
         Return_Node : STree.SyntaxNode;
         Return_Type : Dictionary.Symbol;
      begin

         -- ASSUME Node = not_overriding_subprogram_declaration
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Node) = SPSymbols.not_overriding_subprogram_declaration,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Node = not_overriding_subprogram_declaration in Process_Return");

         Return_Node := Last_Sibling_Of (Start_Node => Child_Node (Current_Node => Child_Node (Current_Node => Node)));
         -- ASSUME Return_Node = type_mark
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Return_Node) = SPSymbols.type_mark,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Return_Node = type_mark in Process_Return");
         Wf_Type_Mark
           (Node          => Return_Node,
            Current_Scope => Dictionary.LocalScope (The_Function),
            Context       => Dictionary.ProgramContext,
            Type_Sym      => Return_Type);

         -- report errors back to caller
         if Return_Type = Dictionary.GetUnknownTypeMark then
            Error_In_Declaration := True;
         end if;

         -- GenericTBD check for illegal return types here (e.g. unconstrained arrays)

         -- add return  type to Dictionary
         Dictionary.AddReturnType
           (TheFunction   => The_Function,
            TypeMark      => Return_Type,
            Comp_Unit     => ContextManager.Ops.Current_Unit,
            TypeReference => Dictionary.Location'(Start_Position => Node_Position (Node => Return_Node),
                                                  End_Position   => Node_Position (Node => Return_Node)));
      end Process_Return;

      ------------------------------

      procedure Process_Anno
        (Node                 : in     STree.SyntaxNode;
         The_Subprogram       : in     Dictionary.Symbol;
         Error_In_Declaration : in out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.File_Heap;
      --#        in     ContextManager.Ops.Unit_Heap;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in out AggregateStack.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out GlobalComponentData;
      --#        in out LexTokenManager.State;
      --#        in out SLI.State;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --#        in out STree.Table;
      --#        in out TheHeap;
      --# derives AggregateStack.State,
      --#         Dictionary.Dict,
      --#         GlobalComponentData,
      --#         LexTokenManager.State,
      --#         Statistics.TableUsage,
      --#         STree.Table,
      --#         TheHeap                    from *,
      --#                                         CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         GlobalComponentData,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table,
      --#                                         TheHeap,
      --#                                         The_Subprogram &
      --#         ErrorHandler.Error_Context,
      --#         SLI.State,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         ContextManager.Ops.File_Heap,
      --#                                         ContextManager.Ops.Unit_Heap,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         GlobalComponentData,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SLI.State,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table,
      --#                                         TheHeap,
      --#                                         The_Subprogram &
      --#         Error_In_Declaration       from *,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         Node,
      --#                                         STree.Table,
      --#                                         The_Subprogram;
      is
         Anno_Node       : STree.SyntaxNode;
         Global_Node     : STree.SyntaxNode;
         Constraint_Node : STree.SyntaxNode;

         ------------------------------------------------------------------------

         procedure Get_Anno_And_Con_Nodes
           (Node            : in     STree.SyntaxNode;
            Anno_Node       :    out STree.SyntaxNode;
            Constraint_Node :    out STree.SyntaxNode;
            Global_Node     :    out STree.SyntaxNode)
         --# global in STree.Table;
         --# derives Anno_Node,
         --#         Constraint_Node,
         --#         Global_Node     from Node,
         --#                              STree.Table;
         is
         begin
            -- ASSUME Node = procedure_annotation OR function_annotation
            SystemErrors.RT_Assert
              (C       => Syntax_Node_Type (Node => Node) = SPSymbols.procedure_annotation
                 or else Syntax_Node_Type (Node => Node) = SPSymbols.function_annotation,
               Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Node = procedure_annotation OR function_annotation in Get_Anno_And_Con_Nodes");
            Constraint_Node := Child_Node (Current_Node => Node);
            -- ASSUME Constraint_Node = moded_global_definition OR dependency_relation OR declare_annotation OR
            --                          procedure_constraint OR function_constraint
            if Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.function_constraint
              or else Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.procedure_constraint then
               -- ASSUME Constraint_Node = function_constraint OR procedure_constraint
               Global_Node := STree.NullNode;
               Anno_Node   := STree.NullNode; -- only a constraint found
            elsif Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.moded_global_definition
              or else Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.dependency_relation
              or else Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.declare_annotation then
               -- ASSUME Constraint_Node = moded_global_definition OR dependency_relation OR declare_annotation
               if Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.moded_global_definition then
                  Global_Node := Constraint_Node;
               else
                  Global_Node := STree.NullNode;
               end if;
               Anno_Node       := Node;
               Constraint_Node := Last_Sibling_Of (Start_Node => Constraint_Node);
            else
               Global_Node     := STree.NullNode;
               Anno_Node       := STree.NullNode;
               Constraint_Node := STree.NullNode;
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                  Msg     => "Expect Constraint_Node = moded_global_definition OR dependency_relation OR declare_annotation OR procedure_constraint OR function_constraint in Get_Anno_And_Con_Nodes");
            end if;
            -- ASSUME Anno_Node = procedure_annotation OR function_annotation OR NULL
            SystemErrors.RT_Assert
              (C       => Anno_Node = STree.NullNode
                 or else Syntax_Node_Type (Node => Anno_Node) = SPSymbols.procedure_annotation
                 or else Syntax_Node_Type (Node => Anno_Node) = SPSymbols.function_annotation,
               Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Anno_Node = procedure_annotation OR function_annotation OR NULL in Get_Anno_And_Con_Nodes");
            -- ASSUME Constraint_Node = function_constraint OR procedure_constraint
            SystemErrors.RT_Assert
              (C       => Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.function_constraint
                 or else Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.procedure_constraint,
               Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Constraint_Node = function_constraint OR procedure_constraint in Get_Anno_And_Con_Nodes");
            -- ASSUME Global_Node = moded_global_definition OR NULL
            SystemErrors.RT_Assert
              (C       => Global_Node = STree.NullNode
                 or else Syntax_Node_Type (Node => Global_Node) = SPSymbols.moded_global_definition,
               Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Global_Node = moded_global_definition OR NULL in Get_Anno_And_Con_Nodes");
         end Get_Anno_And_Con_Nodes;

      begin -- Process_Anno

         -- ASSUME Node = procedure_annotation OR function_annotation
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Node) = SPSymbols.procedure_annotation
              or else Syntax_Node_Type (Node => Node) = SPSymbols.function_annotation,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Node = procedure_annotation OR function_annotation in Process_Anno");

         Get_Anno_And_Con_Nodes
           (Node            => Node,
            Anno_Node       => Anno_Node,
            Constraint_Node => Constraint_Node,
            Global_Node     => Global_Node);

         -- ASSUME Global_Node = moded_global_definition OR NULL
         if Syntax_Node_Type (Node => Global_Node) = SPSymbols.moded_global_definition then
            -- ASSUME Global_Node = moded_global_definition
            -- initial design disallows generic subprograms from having global anno.
            ErrorHandler.Semantic_Error
              (Err_Num   => 638,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Child_Node (Current_Node => Node)),
               Id_Str    => LexTokenManager.Null_String);
            Error_In_Declaration := True;
         elsif Global_Node = STree.NullNode then
            -- ASSUME Global_Node = NULL
            if Dictionary.IsProcedure (The_Subprogram) then
               -- ASSUME Anno_Node = procedure_annotation OR function_annotation OR NULL
               if Syntax_Node_Type (Node => Anno_Node) = SPSymbols.procedure_annotation
                 or else Syntax_Node_Type (Node => Anno_Node) = SPSymbols.function_annotation then
                  -- ASSUME Anno_Node = procedure_annotation OR function_annotation
                  Wf_Procedure_Annotation
                    (Node          => Anno_Node,
                     Current_Scope => Dictionary.LocalScope (The_Subprogram),
                     Subprog_Sym   => The_Subprogram,
                     First_Seen    => True);
               elsif Anno_Node = STree.NullNode then
                  -- ASSUME Anno_Node = NULL
                  if CommandLineData.Content.Do_Information_Flow then
                     -- a derives is needed
                     Error_In_Declaration := True;
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 154,
                        Reference => ErrorHandler.No_Reference,
                        Position  => Node_Position (Node => Node),
                        Id_Str    => Dictionary.GetSimpleName (The_Subprogram));
                  end if;
               else
                  SystemErrors.Fatal_Error
                    (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                     Msg     => "Expect Anno_Node = procedure_annotation OR function_annotation OR NULL in Process_Anno");
               end if;
               if not CommandLineData.Content.Do_Information_Flow then
                  CreateFullSubProgDependency (Node, The_Subprogram, Dictionary.IsAbstract);
               end if;
            end if;
            -- ASSUME Constraint_Node = function_constraint OR procedure_constraint
            if Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.function_constraint
              or else Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.procedure_constraint then
               -- ASSUME Constraint_Node = function_constraint OR procedure_constraint
               wf_procedure_constraint (Constraint_Node, Dictionary.LocalScope (The_Subprogram), True);
            else
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                  Msg     => "Expect Constraint_Node = function_constraint OR procedure_constraint OR NULL in Process_Anno");
            end if;
         else
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Global_Node = moded_global_definition OR NULL in Process_Anno");
         end if;
      end Process_Anno;

   begin -- Process_Generic_Subprogram_Declaration

      -- ASSUME Node = generic_subprogram_declaration
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.generic_subprogram_declaration,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = generic_subprogram_declaration in Process_Generic_Subprogram_Declaration");

      Formal_Part_Node := Child_Node (Current_Node => Node);
      -- ASSUME Formal_Part_Node = generic_formal_part
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Formal_Part_Node) = SPSymbols.generic_formal_part,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Formal_Part_Node = generic_formal_part in Process_Generic_Subprogram_Declaration");

      Subprogram_Declaration_Node := Next_Sibling (Current_Node => Formal_Part_Node);
      -- ASSUME Subprogram_Declaration_Node = not_overriding_subprogram_declaration
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Subprogram_Declaration_Node) = SPSymbols.not_overriding_subprogram_declaration,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Subprogram_Declaration_Node = not_overriding_subprogram_declaration in Process_Generic_Subprogram_Declaration");

      -- First check the generic subprogram name for illegal redeclaration etc.  Add subprogram if OK.
      -- We need to do this before handling the generic formal part because we need somewhere in the
      -- Dictionary where we can attach the list of generic formal parameters.
      Check_And_Add_Subprogram
        (Node            => Subprogram_Declaration_Node,
         Subprog_Sym     => Generic_Subprogam_Symbol,
         Subprogram_Kind => Subprogram_Kind);

      if Generic_Subprogam_Symbol /= Dictionary.NullSymbol then
         Process_Generic_Formal_Part
           (Node                 => Formal_Part_Node,
            Generic_Unit         => Generic_Subprogam_Symbol,
            Error_In_Declaration => Error_In_Declaration);

         Process_Formal_Part (Node           => Subprogram_Declaration_Node,
                              The_Subprogram => Generic_Subprogam_Symbol);
         -- note that the above does not return an erro flag; however, it does set
         -- the subprogram signature as not well formed if there is an error in the
         -- formal part

         if Subprogram_Kind = A_Function then
            Process_Return
              (Node                 => Subprogram_Declaration_Node,
               The_Function         => Generic_Subprogam_Symbol,
               Error_In_Declaration => Error_In_Declaration);
            Process_Anno
              (Node                 => Next_Sibling (Current_Node => Child_Node (Current_Node => Subprogram_Declaration_Node)),
               The_Subprogram       => Generic_Subprogam_Symbol,
               Error_In_Declaration => Error_In_Declaration);
         elsif Subprogram_Kind = A_Procedure then
            Process_Anno
              (Node                 => Next_Sibling (Current_Node => Child_Node (Current_Node => Subprogram_Declaration_Node)),
               The_Subprogram       => Generic_Subprogam_Symbol,
               Error_In_Declaration => Error_In_Declaration);
         end if;

         -- PNA to do:
         --    store pre/post nodes

         -- If there are any errors in the generic declaration we mark the subprogram signature as
         -- not well formed.  We can then raise a suitable error when an attempt is made to instantiate
         -- a badly formed generic unit.  Similar to the way we refuse to flow analyse a call to a malformed
         -- subprogram.
         if Error_In_Declaration then
            Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract, Generic_Subprogam_Symbol);
         end if;
      end if;
   end Process_Generic_Subprogram_Declaration;

   -----------------------------------------------------------------------------------

   procedure Process_Generic_Package_Declaration (Node          : in STree.SyntaxNode;
                                                  Current_Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   is
      Private_Package_Declaration, Child_Package_Declaration, Error_In_Declaration : Boolean;
      Generic_Formal_Part_Node, Spec_Node, Ident_Node, Context_Node, Inherit_Node  : STree.SyntaxNode;
      Pack_Sym                                                                     : Dictionary.Symbol;
      Ident_Str                                                                    : LexTokenManager.Lex_String;

      ------------------------------

      procedure Find_Key_Nodes
        (Node                        : in     STree.SyntaxNode;
         Context_Node                :    out STree.SyntaxNode;
         Private_Package_Declaration :    out Boolean;
         Inherit_Node                :    out STree.SyntaxNode;
         Generic_Formal_Part_Node    :    out STree.SyntaxNode;
         Spec_Node                   :    out STree.SyntaxNode;
         Ident_Node                  :    out STree.SyntaxNode;
         Ident_Str                   :    out LexTokenManager.Lex_String;
         Child_Package_Declaration   :    out Boolean)
      --# global in STree.Table;
      --# derives Child_Package_Declaration,
      --#         Context_Node,
      --#         Generic_Formal_Part_Node,
      --#         Ident_Node,
      --#         Ident_Str,
      --#         Inherit_Node,
      --#         Private_Package_Declaration,
      --#         Spec_Node                   from Node,
      --#                                          STree.Table;
      is
      begin
         -- ASSUME Node = generic_package_declaration OR private_generic_package_declaration
         if Syntax_Node_Type (Node => Node) = SPSymbols.generic_package_declaration then
            -- ASSUME Node = generic_package_declaration
            Context_Node := Parent_Node (Current_Node => Node);
         elsif Syntax_Node_Type (Node => Node) = SPSymbols.private_generic_package_declaration then
            Context_Node := STree.NullNode;
         else
            Context_Node := STree.NullNode;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Node = generic_package_declaration OR private_generic_package_declaration in Find_Key_Nodes");
         end if;
         Context_Node := Child_Node (Current_Node => Parent_Node (Current_Node => Parent_Node (Current_Node => Context_Node)));
         -- ASSUME Context_Node = context_clause OR library_unit
         if Syntax_Node_Type (Node => Context_Node) = SPSymbols.library_unit then
            -- ASSUME Context_Node = library_unit
            Context_Node := STree.NullNode;
         elsif Syntax_Node_Type (Node => Context_Node) /= SPSymbols.context_clause then
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Context_Node = context_clause OR library_unit in Find_Key_Nodes");
         end if;
         -- ASSUME Context_Node = context_clause OR NULL
         SystemErrors.RT_Assert
           (C       => Context_Node = STree.NullNode or else Syntax_Node_Type (Node => Context_Node) = SPSymbols.context_clause,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Context_Node = context_clause OR NULL in Find_Key_Nodes");

         Private_Package_Declaration := Syntax_Node_Type (Node => Node) = SPSymbols.private_generic_package_declaration;
         Inherit_Node                := Child_Node (Current_Node => Node);
         -- ASSUME Inherit_Node = inherit_clause OR generic_formal_part
         if Syntax_Node_Type (Node => Inherit_Node) = SPSymbols.inherit_clause then
            -- ASSUME Inherit_Node = inherit_clause
            Generic_Formal_Part_Node := Next_Sibling (Current_Node => Inherit_Node);
         elsif Syntax_Node_Type (Node => Inherit_Node) = SPSymbols.generic_formal_part then
            -- ASSUME Inherit_Node = generic_formal_part
            Generic_Formal_Part_Node := Inherit_Node;
            Inherit_Node             := STree.NullNode;
         else
            Generic_Formal_Part_Node := STree.NullNode;
            Inherit_Node             := STree.NullNode;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Inherit_Node = inherit_clause OR generic_formal_part in Find_Key_Nodes");
         end if;
         -- ASSUME Inherit_Node = inherit_clause OR NULL
         SystemErrors.RT_Assert
           (C       => Inherit_Node = STree.NullNode or else Syntax_Node_Type (Node => Inherit_Node) = SPSymbols.inherit_clause,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Inherit_Node = inherit_clause OR NULL in Find_Key_Nodes");
         -- ASSUME Generic_Formal_Part_Node = generic_formal_part
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Generic_Formal_Part_Node) = SPSymbols.generic_formal_part,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Generic_Formal_Part_Node = generic_formal_part in Find_Key_Nodes");

         Spec_Node := Next_Sibling (Current_Node => Generic_Formal_Part_Node);
         -- ASSUME Spec_Node = package_specification
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Spec_Node) = SPSymbols.package_specification,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Spec_Node = package_specification in Find_Key_Nodes");

         Ident_Node := Child_Node (Current_Node => Child_Node (Current_Node => Spec_Node));
         -- ASSUME Ident_Node = dotted_simple_name OR identifier
         if Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier then
            -- ASSUME Ident_Node = identifier
            Child_Package_Declaration := False;
         elsif Syntax_Node_Type (Node => Ident_Node) = SPSymbols.dotted_simple_name then
            -- ASSUME Ident_Node = dotted_simple_name
            Child_Package_Declaration := True;
            Ident_Node                := Last_Child_Of (Start_Node => Ident_Node);
         else
            Child_Package_Declaration := False;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Ident_Node = dotted_simple_name OR identifier in Find_Key_Nodes");
         end if;
         -- ASSUME Ident_Node = identifier
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Ident_Node = identifier in Find_Key_Nodes");
         Ident_Str := Node_Lex_String (Node => Ident_Node);
      end Find_Key_Nodes;

      procedure Check_And_Add_Package
        (Ident_Node    : in     STree.SyntaxNode;
         Current_Scope : in     Dictionary.Scopes;
         Package_Sym   :    out Dictionary.Symbol)
      --# global in     Child_Package_Declaration;
      --#        in     CommandLineData.Content;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out Ident_Str;
      --#        in out Private_Package_Declaration;
      --#        in out SPARK_IO.File_Sys;
      --#        in out STree.Table;
      --# derives Dictionary.Dict,
      --#         Package_Sym                 from Child_Package_Declaration,
      --#                                          CommandLineData.Content,
      --#                                          ContextManager.Ops.Unit_Stack,
      --#                                          Current_Scope,
      --#                                          Dictionary.Dict,
      --#                                          Ident_Node,
      --#                                          Ident_Str,
      --#                                          LexTokenManager.State,
      --#                                          Private_Package_Declaration,
      --#                                          STree.Table &
      --#         ErrorHandler.Error_Context  from *,
      --#                                          Child_Package_Declaration,
      --#                                          CommandLineData.Content,
      --#                                          Current_Scope,
      --#                                          Dictionary.Dict,
      --#                                          Ident_Node,
      --#                                          Ident_Str,
      --#                                          LexTokenManager.State,
      --#                                          Private_Package_Declaration,
      --#                                          SPARK_IO.File_Sys,
      --#                                          STree.Table &
      --#         Ident_Str                   from *,
      --#                                          Child_Package_Declaration,
      --#                                          CommandLineData.Content,
      --#                                          Current_Scope,
      --#                                          Dictionary.Dict,
      --#                                          Ident_Node,
      --#                                          LexTokenManager.State,
      --#                                          STree.Table &
      --#         Private_Package_Declaration from *,
      --#                                          Child_Package_Declaration,
      --#                                          CommandLineData.Content &
      --#         SPARK_IO.File_Sys           from *,
      --#                                          Child_Package_Declaration,
      --#                                          CommandLineData.Content,
      --#                                          ContextManager.Ops.Unit_Stack,
      --#                                          Current_Scope,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.Error_Context,
      --#                                          Ident_Node,
      --#                                          Ident_Str,
      --#                                          LexTokenManager.State,
      --#                                          Private_Package_Declaration,
      --#                                          STree.Table &
      --#         STree.Table                 from *,
      --#                                          Child_Package_Declaration,
      --#                                          CommandLineData.Content,
      --#                                          Current_Scope,
      --#                                          Dictionary.Dict,
      --#                                          Ident_Node,
      --#                                          LexTokenManager.State,
      --#                                          Private_Package_Declaration;
      is
         Pack_Sym   : Dictionary.Symbol;
         Valid_Name : Boolean := True;
      begin
         -- ASSUME Ident_Node = identifier
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Ident_Node = identifier in Check_And_Add_Package");

         if CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83 and then Child_Package_Declaration then
            AddChild (Ident_Node, Private_Package_Declaration, Current_Scope,
                      -- to get
                      Pack_Sym, Ident_Str);
            -- if Pack_Sym is null then something went wrong when we added the child so we need to supress
            -- any further analysis of the package specification
            Valid_Name := Pack_Sym /= Dictionary.NullSymbol;
         else
            if CommandLineData.Content.Language_Profile = CommandLineData.SPARK83 then
               -- check that syntax conforms
               if Child_Package_Declaration or else Private_Package_Declaration then
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 610,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Ident_Node),
                     Id_Str    => LexTokenManager.Null_String);
                  Private_Package_Declaration := False;
               end if;
            end if;
            Pack_Sym := Dictionary.LookupItem (Name              => Ident_Str,
                                               Scope             => Current_Scope,
                                               Context           => Dictionary.ProofContext,
                                               Full_Package_Name => False);
            --# assert True;
            if Pack_Sym /= Dictionary.NullSymbol then
               Valid_Name := False;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 10,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Ident_Node),
                  Id_Str    => Ident_Str);
            else
               --# assert True;
               if Private_Package_Declaration then -- root level private package
                  Dictionary.AddPrivatePackage
                    (Name          => Ident_Str,
                     Comp_Unit     => ContextManager.Ops.Current_Unit,
                     Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Ident_Node),
                                                           End_Position   => Node_Position (Node => Ident_Node)),
                     Scope         => Current_Scope,
                     ThePackage    => Pack_Sym);
               else
                  Dictionary.AddPackage
                    (Name          => Ident_Str,
                     Comp_Unit     => ContextManager.Ops.Current_Unit,
                     Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Ident_Node),
                                                           End_Position   => Node_Position (Node => Ident_Node)),
                     Scope         => Current_Scope,
                     ThePackage    => Pack_Sym);
               end if;
            end if;
         end if;

         if Valid_Name then
            Package_Sym := Pack_Sym;
         else
            Package_Sym := Dictionary.NullSymbol;
         end if;

      end Check_And_Add_Package;

   begin -- Process_Generic_Package_Declaration

      -- ASSUME Node = generic_package_declaration OR private_generic_package_declaration
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.generic_package_declaration
           or else Syntax_Node_Type (Node => Node) = SPSymbols.private_generic_package_declaration,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = generic_package_declaration OR private_generic_package_declaration in Process_Generic_Package_Declaration");

      --# accept Flow, 10, "Required to initialize for flow analysis";
      Error_In_Declaration := False;
      --# end accept;

      Find_Key_Nodes
        (Node                        => Node,
         Context_Node                => Context_Node,
         Private_Package_Declaration => Private_Package_Declaration,
         Inherit_Node                => Inherit_Node,
         Generic_Formal_Part_Node    => Generic_Formal_Part_Node,
         Spec_Node                   => Spec_Node,
         Ident_Node                  => Ident_Node,
         Ident_Str                   => Ident_Str,
         Child_Package_Declaration   => Child_Package_Declaration);
      -- ASSUME Generic_Formal_Part_Node = generic_formal_part
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Generic_Formal_Part_Node) = SPSymbols.generic_formal_part,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Generic_Formal_Part_Node = generic_formal_part in Process_Generic_Package_Declaration");
      -- ASSUME Spec_Node = package_specification
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Spec_Node) = SPSymbols.package_specification,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Spec_Node = package_specification in Process_Generic_Package_Declaration");
      -- ASSUME Ident_Node = identifier
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Ident_Node = identifier in Process_Generic_Package_Declaration");

      if Dictionary.IsGlobalScope (Current_Scope) then
         -- First we need to check the generic package name for illegal redeclaration etc.
         --# accept Flow, 10, Private_Package_Declaration, "Code under development";
         Check_And_Add_Package (Ident_Node    => Ident_Node,
                                Current_Scope => Current_Scope,
                                Package_Sym   => Pack_Sym);
         --# end accept;

         if Pack_Sym /= Dictionary.NullSymbol then
            Dictionary.SetPackageIsGeneric (Pack_Sym);

            --# accept Flow, 10, Error_In_Declaration, "Code under development";
            Process_Generic_Formal_Part
              (Node                 => Generic_Formal_Part_Node,
               Generic_Unit         => Pack_Sym,
               Error_In_Declaration => Error_In_Declaration);
            --# end accept;

            -- ASSUME Inherit_Node = inherit_clause OR NULL
            if Syntax_Node_Type (Node => Inherit_Node) = SPSymbols.inherit_clause then
               -- ASSUME Inherit_Node = inherit_clause
               Dictionary.AddInheritsAnnotation
                 (Pack_Sym,
                  Dictionary.Location'(Start_Position => Node_Position (Node => Inherit_Node),
                                       End_Position   => Node_Position (Node => Inherit_Node)));
               Wf_Inherit_Clause (Node     => Inherit_Node,
                                  Comp_Sym => Pack_Sym,
                                  Scope    => Current_Scope);
            elsif Inherit_Node /= STree.NullNode then
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                  Msg     => "Expect Inherit_Node = inherit_clause OR NULL in Process_Generic_Formal_Part");
            end if;

            --# assert True;

            -- ASSUME Context_Node = context_clause OR NULL
            if Syntax_Node_Type (Node => Context_Node) = SPSymbols.context_clause then
               -- ASSUME Context_Node = context_clause
               Wf_Context_Clause (Context_Node, Pack_Sym, Dictionary.VisibleScope (Pack_Sym));
            elsif Context_Node /= STree.NullNode then
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                  Msg     => "Expect Context_Node = context_clause OR NULL in Process_Generic_Formal_Part");
            end if;

            --# assert True;

            Wf_Package_Specification
              (Node          => Spec_Node,
               Ident_Str     => Ident_Str,
               Pack_Sym      => Pack_Sym,
               Current_Scope => Current_Scope);
         end if;
      end if;
   end Process_Generic_Package_Declaration;

begin -- Wf_Generic_Declaration

   -- ASSUME Node = generic_declaration OR private_generic_package_declaration
   case CommandLineData.Content.Language_Profile is
      when CommandLineData.SPARK83 =>
         -- not allowed in SPARK 83
         ErrorHandler.Semantic_Error
           (Err_Num   => 637,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => LexTokenManager.Null_String);

      when CommandLineData.SPARK95 | CommandLineData.SPARK2005 =>
         if Syntax_Node_Type (Node => Node) = SPSymbols.private_generic_package_declaration then
            -- ASSUME Node = private_generic_package_declaration
            Current_Node := Node;
         elsif Syntax_Node_Type (Node => Node) = SPSymbols.generic_declaration then
            -- ASSUME Node = generic_declaration
            Current_Node := Child_Node (Current_Node => Node);
         else
            Current_Node := Node;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Node = generic_declaration OR private_generic_package_declaration in Wf_Generic_Declaration");
         end if;
         -- ASSUME Current_Node = generic_subprogram_declaration OR generic_package_declaration OR
         --                       private_generic_package_declaration
         if Syntax_Node_Type (Node => Current_Node) = SPSymbols.generic_subprogram_declaration then
            -- ASSUME Current_Node = generic_subprogram_declaration
            Process_Generic_Subprogram_Declaration (Node => Current_Node);
         elsif Syntax_Node_Type (Node => Current_Node) = SPSymbols.generic_package_declaration
           or else Syntax_Node_Type (Node => Current_Node) = SPSymbols.private_generic_package_declaration then
            -- ASSUME Current_Node = generic_package_declaration OR private_generic_package_declaration
            Process_Generic_Package_Declaration (Node          => Current_Node,
                                                 Current_Scope => Current_Scope);
         else
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Current_Node = generic_subprogram_declaration OR generic_package_declaration OR private_generic_package_declaration in Wf_Generic_Declaration");
         end if;
   end case;
end Wf_Generic_Declaration;
