-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with E_Strings;
with SLI;

separate (Sem.CompUnit)
procedure Wf_Body_Stub (Node  : in STree.SyntaxNode;
                        Scope : in Dictionary.Scopes) is
   Node_Type                                      : SPSymbols.SPSymbol;
   Ident_Node                                     : STree.SyntaxNode;
   Ident_Str                                      : LexTokenManager.Lex_String;
   Spec_Node                                      : STree.SyntaxNode;
   Formal_Node                                    : STree.SyntaxNode;
   Constraint_Node                                : STree.SyntaxNode;
   Pragma_Node                                    : STree.SyntaxNode;
   Anno_Node                                      : STree.SyntaxNode;
   Pack_Sym, Subprog_Sym, Protected_Sym, Task_Sym : Dictionary.Symbol;
   First_Seen                                     : Boolean;
   Scope_Local                                    : Dictionary.Scopes;
   Interfacing_Pragma_Found                       : Boolean := False;
   Other_Pragma_Found                             : Boolean := False;
   Unused                                         : Boolean;
   Valid_Annotation                               : Boolean := False; -- used for task type stubs
   Valid_Stub_Position                            : Boolean := True;
   Is_Overriding                                  : Boolean;

   ------------------------------------------------------------------

   procedure Check_Position (Node : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Valid_Stub_Position;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Valid_Stub_Position        from *,
   --#                                         Node,
   --#                                         STree.Table;
   is
      Outer_Node : STree.SyntaxNode;
   begin
      -- ASSUME Node = body_stub
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.body_stub,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = body_stub in Check_Position");
      Outer_Node := Parent_Node (Current_Node => Parent_Node (Current_Node => Parent_Node (Current_Node => Node)));
      while Syntax_Node_Type (Node => Outer_Node) = SPSymbols.later_declarative_item_rep loop
         -- ASSUME Outer_Node = later_declarative_item_rep
         Outer_Node := Parent_Node (Current_Node => Outer_Node);
      end loop;
      -- ASSUME Outer_Node = declarative_part
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Outer_Node) = SPSymbols.declarative_part,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Outer_Node = declarative_part in Check_Position");
      Outer_Node := Parent_Node (Current_Node => Parent_Node (Current_Node => Parent_Node (Current_Node => Outer_Node)));
      -- ASSUME Outer_Node = proper_body OR protected_operation_item OR generic_subprogram_body OR
      --                     main_program_declaration OR library_unit_body
      if Syntax_Node_Type (Node => Outer_Node) = SPSymbols.proper_body
        or else Syntax_Node_Type (Node => Outer_Node) = SPSymbols.protected_operation_item then
         -- ASSUME Outer_Node = proper_body OR protected_operation_item
         if Syntax_Node_Type (Node => Outer_Node) = SPSymbols.protected_operation_item then
            while Syntax_Node_Type (Node => Outer_Node) = SPSymbols.protected_operation_item loop
               -- ASSUME Outer_Node = protected_operation_item
               Outer_Node := Parent_Node (Current_Node => Outer_Node);
            end loop;
            -- ASSUME Outer_Node = protected_body
            SystemErrors.RT_Assert
              (C       => Syntax_Node_Type (Node => Outer_Node) = SPSymbols.protected_body,
               Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Outer_Node = protected_body in Check_Position");
            Outer_Node := Parent_Node (Current_Node => Outer_Node);
         end if;
         -- ASSUME Outer_Node = proper_body
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Outer_Node) = SPSymbols.proper_body,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Outer_Node = proper_body in Check_Position");
         Outer_Node := Parent_Node (Current_Node => Outer_Node);
         -- ASSUME Outer_Node = abody OR subunit
         if Syntax_Node_Type (Node => Outer_Node) = SPSymbols.abody then
            -- ASSUME Outer_Node = abody
            Valid_Stub_Position := False;
            ErrorHandler.Semantic_Error
              (Err_Num   => 61,
               Reference => 17,
               Position  => Node_Position (Node),
               Id_Str    => LexTokenManager.Null_String);
         elsif Syntax_Node_Type (Node => Outer_Node) /= SPSymbols.subunit then
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Outer_Node = abody OR subunit in Check_Position");
         end if;
      elsif Syntax_Node_Type (Node => Outer_Node) /= SPSymbols.generic_subprogram_body
        and then Syntax_Node_Type (Node => Outer_Node) /= SPSymbols.main_program_declaration
        and then Syntax_Node_Type (Node => Outer_Node) /= SPSymbols.library_unit_body then
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Outer_Node = proper_body OR protected_operation_item OR generic_subprogram_body OR main_program_declaration OR library_unit_body in Check_Position");
      end if;

   end Check_Position;

   ------------------------------------------------------------------------

   function Requires_Second_Annotation (Subprog_Sym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
      Global_Var       : Dictionary.Symbol;
      Required         : Boolean;
      Global_Item      : Dictionary.Iterator;
      Enclosing_Region : Dictionary.Symbol;
   begin
      Required := False;
      if not Dictionary.IsGlobalScope (Dictionary.GetScope (Subprog_Sym)) then
         Enclosing_Region := Dictionary.GetEnclosingCompilationUnit (Dictionary.GetScope (Subprog_Sym));
         if Dictionary.IsPackage (Enclosing_Region) then
            Global_Item := Dictionary.FirstGlobalVariable (Dictionary.IsAbstract, Subprog_Sym);
            while Global_Item /= Dictionary.NullIterator loop
               Global_Var := Dictionary.CurrentSymbol (Global_Item);
               if Dictionary.IsRefinedOwnVariable (Global_Var) and then Dictionary.GetOwner (Global_Var) = Enclosing_Region then
                  Required := True;
                  exit;
               end if;
               Global_Item := Dictionary.NextSymbol (Global_Item);
            end loop;
         end if;
      end if;
      return Required;
   end Requires_Second_Annotation;

   ------------------------------------------------------------------------

   procedure Get_Anno_And_Con_Nodes
     (Node            : in     STree.SyntaxNode;
      Anno_Node       :    out STree.SyntaxNode;
      Constraint_Node :    out STree.SyntaxNode)
   --# global in STree.Table;
   --# derives Anno_Node,
   --#         Constraint_Node from Node,
   --#                              STree.Table;
   is
   begin
      -- ASSUME Node = procedure_annotation OR function_annotation
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.procedure_annotation
           or else Syntax_Node_Type (Node => Node) = SPSymbols.function_annotation,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = procedure_annotation OR function_annotation in Get_Anno_And_Con_Nodes");
      Constraint_Node := Child_Node (Current_Node => Node);
      -- ASSUME Constraint_Node = moded_global_definition OR dependency_relation OR declare_annotation OR
      --                          procedure_constraint OR function_constraint
      if Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.function_constraint
        or else Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.procedure_constraint then
         -- ASSUME Constraint_Node = function_constraint OR procedure_constraint
         Anno_Node := STree.NullNode; -- only a constraint found
      elsif Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.moded_global_definition
        or else Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.dependency_relation
        or else Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.declare_annotation then
         -- ASSUME Constraint_Node = moded_global_definition OR dependency_relation OR declare_annotation
         Anno_Node       := Node;
         Constraint_Node := Last_Sibling_Of (Start_Node => Constraint_Node);
      else
         Anno_Node       := STree.NullNode;
         Constraint_Node := STree.NullNode;
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Constraint_Node = moded_global_definition OR dependency_relation OR declare_annotation OR procedure_constraint OR function_constraint in Get_Anno_And_Con_Nodes");
      end if;
      -- ASSUME Anno_Node = procedure_annotation OR function_annotation OR NULL
      SystemErrors.RT_Assert
        (C       => Anno_Node = STree.NullNode
           or else Syntax_Node_Type (Node => Anno_Node) = SPSymbols.procedure_annotation
           or else Syntax_Node_Type (Node => Anno_Node) = SPSymbols.function_annotation,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Anno_Node = procedure_annotation OR function_annotation OR NULL in Get_Anno_And_Con_Nodes");
      -- ASSUME Constraint_Node = function_constraint OR procedure_constraint
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.function_constraint
           or else Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.procedure_constraint,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Constraint_Node = function_constraint OR procedure_constraint in Get_Anno_And_Con_Nodes");
   end Get_Anno_And_Con_Nodes;

   ----------------------------------------------------------------------

   function Empty_Annotation (Node : STree.SyntaxNode) return Boolean
   --# global in STree.Table;
   is
      Current_Node : STree.SyntaxNode;
   begin
      -- ASSUME Node = procedure_annotation
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.procedure_annotation,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = procedure_annotation in Empty_Annotation");
      Current_Node := Child_Node (Current_Node => Node);
      -- ASSUME Current_Node = moded_global_definition OR dependency_relation OR declare_annotation OR procedure_constraint
      if Syntax_Node_Type (Node => Current_Node) = SPSymbols.procedure_constraint then
         Current_Node := Child_Node (Current_Node => Current_Node);
         -- ASSUME Current_Node = precondition OR postcondition OR NULL
         SystemErrors.RT_Assert
           (C       => Current_Node = STree.NullNode
              or else Syntax_Node_Type (Node => Current_Node) = SPSymbols.precondition
              or else Syntax_Node_Type (Node => Current_Node) = SPSymbols.postcondition,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Current_Node = precondition OR postcondition OR NULL in Empty_Annotation");
      elsif Syntax_Node_Type (Node => Current_Node) /= SPSymbols.moded_global_definition
        and then Syntax_Node_Type (Node => Current_Node) /= SPSymbols.dependency_relation
        and then Syntax_Node_Type (Node => Current_Node) /= SPSymbols.declare_annotation then
         Current_Node := STree.NullNode;
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Current_Node = moded_global_definition OR dependency_relation OR declare_annotation OR procedure_constraint in Empty_Annotation");
      end if;
      return Current_Node = STree.NullNode;
   end Empty_Annotation;

   ----------------------------------------------------------------------

   procedure Process_Annotation
     (Anno_Node        : in     STree.SyntaxNode;
      Scope            : in     Dictionary.Scopes;
      Task_Sym         : in     Dictionary.Symbol;
      Valid_Annotation : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         Anno_Node,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         Task_Sym,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from Anno_Node,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Task_Sym,
   --#                                         TheHeap &
   --#         Valid_Annotation           from *,
   --#                                         Anno_Node,
   --#                                         STree.Table;
   is
      Current_Node : STree.SyntaxNode;

      procedure Raise_Error (Node : in STree.SyntaxNode)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table;
      is
      begin
         ErrorHandler.Semantic_Error
           (Err_Num   => 990,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node),
            Id_Str    => LexTokenManager.Null_String);
      end Raise_Error;

   begin -- Process_Annotation

      -- ASSUME Anno_Node = procedure_annotation
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Anno_Node) = SPSymbols.procedure_annotation,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Anno_Node = procedure_annotation in Process_Annotation");

      Current_Node := Child_Node (Current_Node => Anno_Node);
      -- ASSUME Current_Node = moded_global_definition OR dependency_relation OR declare_annotation OR procedure_constraint
      -- to be legal, Current_Node must be a moded_global_definition
      if Syntax_Node_Type (Node => Current_Node) = SPSymbols.moded_global_definition then
         -- ASSUME Current_Node = moded_global_definition
         Current_Node := Last_Sibling_Of (Start_Node => Current_Node);
         -- ASSUME Current_Node = procedure_constraint
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Current_Node) = SPSymbols.procedure_constraint,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Current_Node = procedure_constraint in Process_Annotation");
         if Child_Node (Current_Node => Current_Node) = STree.NullNode then
            -- ASSUME Child_Node (Current_Node => Current_Node) = NULL
            Valid_Annotation := True;
            Wf_Procedure_Annotation (Node          => Anno_Node,
                                     Current_Scope => Scope,
                                     Subprog_Sym   => Task_Sym,
                                     First_Seen    => False);
         elsif Syntax_Node_Type (Node => Child_Node (Current_Node => Current_Node)) = SPSymbols.precondition
           or else Syntax_Node_Type (Node => Child_Node (Current_Node => Current_Node)) = SPSymbols.postcondition then
            -- ASSUME Child_Node (Current_Node => Current_Node) = precondition OR postcondition
            Raise_Error (Node => Current_Node);
         else
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Child_Node (Current_Node => Current_Node) = precondition OR postcondition OR NULL in Process_Annotation");
         end if;
      elsif Syntax_Node_Type (Node => Current_Node) = SPSymbols.dependency_relation
        or else Syntax_Node_Type (Node => Current_Node) = SPSymbols.declare_annotation
        or else Syntax_Node_Type (Node => Current_Node) = SPSymbols.procedure_constraint then
         Raise_Error (Node => Current_Node);
      else
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Current_Node = moded_global_definition OR dependency_relation OR declare_annotation OR procedure_constraint in Process_Annotation");
      end if;
   end Process_Annotation;

begin -- Wf_Body_Stub

   -- ASSUME Node = body_stub
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.body_stub,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = body_stub in Wf_Body_Stub");

   Pragma_Node := Last_Sibling_Of (Start_Node => Child_Node (Current_Node => Node));
   -- ASSUME Pragma_Node = apragma OR procedure_annotation OR function_annotation OR
   --                      dotted_simple_name OR task_stub OR protected_stub
   if Syntax_Node_Type (Node => Pragma_Node) = SPSymbols.apragma then
      -- ASSUME Pragma_Node = apragma
      if IsExternalInterface (Pragma_Node) then
         -- either Interface of Import correctly used for language variant
         Interfacing_Pragma_Found := True;
      else
         -- some other pragma found
         Other_Pragma_Found := True;
      end if;
   elsif Syntax_Node_Type (Node => Pragma_Node) = SPSymbols.procedure_annotation
     or else Syntax_Node_Type (Node => Pragma_Node) = SPSymbols.function_annotation
     or else Syntax_Node_Type (Node => Pragma_Node) = SPSymbols.dotted_simple_name
     or else Syntax_Node_Type (Node => Pragma_Node) = SPSymbols.task_stub
     or else Syntax_Node_Type (Node => Pragma_Node) = SPSymbols.protected_stub then
      -- ASSUME Pragma_Node = procedure_annotation OR function_annotation OR dotted_simple_name OR task_stub OR protected_stub
      Check_Position (Node => Node); -- also sets global Valid_Stub_Position
   else
      SystemErrors.Fatal_Error
        (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = apragma OR procedure_annotation OR function_annotation OR dotted_simple_name OR task_stub OR protected_stub in Wf_Body_Stub");
   end if;
   --# assert True;
   Node_Type := Syntax_Node_Type (Node => Child_Node (Current_Node => Node));
   -- ASSUME Node_Type = overriding_indicator OR procedure_specification OR function_specification OR
   --                    dotted_simple_name OR task_stub OR protected_stub
   if Node_Type = SPSymbols.dotted_simple_name then
      -- ASSUME Node_Type = dotted_simple_name
      Ident_Node := Child_Node (Current_Node => Child_Node (Current_Node => Node));
      -- ASSUME Ident_Node = dotted_simple_name OR identifier
      if Syntax_Node_Type (Node => Ident_Node) = SPSymbols.dotted_simple_name then
         -- ASSUME Ident_Node = dotted_simple_name
         ErrorHandler.Semantic_Error
           (Err_Num   => 613,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Ident_Node),
            Id_Str    => LexTokenManager.Null_String);
      elsif Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier then
         -- ASSUME Ident_Node = identifier
         Pack_Sym :=
           Dictionary.LookupImmediateScope
           (Name    => Node_Lex_String (Ident_Node),
            Scope   => Scope,
            Context => Dictionary.ProgramContext);
         if Pack_Sym = Dictionary.NullSymbol or else not Dictionary.IsPackage (Pack_Sym) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 11,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Ident_Node),
               Id_Str    => Node_Lex_String (Ident_Node));
         elsif Dictionary.HasBody (Pack_Sym) or else Dictionary.HasBodyStub (Pack_Sym) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 16,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Ident_Node),
               Id_Str    => Node_Lex_String (Ident_Node));
         elsif Dictionary.IsPackage (Dictionary.GetRegion (Scope))
           and then not Dictionary.IsEmbeddedPackage (Dictionary.GetRegion (Scope))
           and then Dictionary.LookupSelectedItem
           (Prefix   => Dictionary.GetRegion (Scope),
            Selector => Node_Lex_String (Ident_Node),
            Scope    => Dictionary.GlobalScope,
            Context  => Dictionary.ProofContext) /=
           Dictionary.NullSymbol then
            -- name exists as child
            ErrorHandler.Semantic_Error
              (Err_Num   => 10,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Ident_Node),
               Id_Str    => Node_Lex_String (Ident_Node));
         else
            STree.Set_Node_Lex_String (Sym  => Pack_Sym,
                                       Node => Ident_Node);
            if Valid_Stub_Position then
               Dictionary.AddBodyStub
                 (CompilationUnit => Pack_Sym,
                  Comp_Unit       => ContextManager.Ops.Current_Unit,
                  BodyStub        => Dictionary.Location'(Start_Position => Node_Position (Node),
                                                          End_Position   => Node_Position (Node)));
            end if;
         end if;
      else
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Ident_Node = dotted_simple_name OR identifier in Wf_Body_Stub");
      end if;
   elsif Node_Type = SPSymbols.procedure_specification
     or else (Node_Type = SPSymbols.overriding_indicator
                and then Syntax_Node_Type (Node => Next_Sibling (Current_Node => Child_Node (Current_Node => Node))) =
                SPSymbols.procedure_specification) then
      -- ASSUME Node_Type = procedure_specification OR overriding_indicator
      if Valid_Stub_Position then
         Scope_Local   := Scope;
         Is_Overriding := False;
         if Node_Type = SPSymbols.overriding_indicator then
            if Syntax_Node_Type (Node => Last_Child_Of (Start_Node => Node)) = SPSymbols.RWoverriding then
               Is_Overriding := True;
            end if;
            Spec_Node := Next_Sibling (Current_Node => Child_Node (Current_Node => Node));
         else
            Spec_Node := Child_Node (Current_Node => Node);
         end if;
         -- ASSUME Spec_Node = procedure_specification
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Spec_Node) = SPSymbols.procedure_specification,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Spec_Node = procedure_specification in Wf_Body_Stub");
         Get_Anno_And_Con_Nodes
           (Node            => Next_Sibling (Current_Node => Spec_Node),
            Anno_Node       => Anno_Node,
            Constraint_Node => Constraint_Node);
         -- ASSUME Anno_Node = procedure_annotation OR NULL
         SystemErrors.RT_Assert
           (C       => Anno_Node = STree.NullNode or else Syntax_Node_Type (Node => Anno_Node) = SPSymbols.procedure_annotation,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Anno_Node = procedure_annotation OR NULL in Wf_Body_Stub");
         -- ASSUME Constraint_Node = procedure_constraint
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.procedure_constraint,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Constraint_Node = procedure_constraint in Wf_Body_Stub");
         --# assert True;
         Formal_Node := Next_Sibling (Current_Node => Child_Node (Current_Node => Spec_Node));
         -- ASSUME Formal_Node = formal_part OR NULL
         SystemErrors.RT_Assert
           (C       => Formal_Node = STree.NullNode or else Syntax_Node_Type (Node => Formal_Node) = SPSymbols.formal_part,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Formal_Node = formal_part OR NULL in Wf_Body_Stub");
         Wf_Procedure_Specification
           (Node        => Spec_Node,
            Hidden      => False,
            Scope       => Scope_Local,
            Subprog_Sym => Subprog_Sym,
            First_Seen  => First_Seen);
         --# assert True;
         if Subprog_Sym /= Dictionary.NullSymbol then
            if Syntax_Node_Type (Node => Pragma_Node) = SPSymbols.apragma and then not First_Seen then -- illegal redeclaration
               ErrorHandler.A_Pragma
                 (Pragma_Name => Node_Lex_String (Child_Node (Current_Node => Pragma_Node)),
                  Position    => Node_Position (Pragma_Node));
               ErrorHandler.Semantic_Error
                 (Err_Num   => 10,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node),
                  Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
            else -- this else means we don't process illegal redeclarations further
               if First_Seen and then Other_Pragma_Found then -- only an interfacing pragma will do
                  case CommandLineData.Content.Language_Profile is
                     when CommandLineData.SPARK83 =>
                        ErrorHandler.Semantic_Error
                          (Err_Num   => 70,
                           Reference => 18,
                           Position  => Node_Position (Pragma_Node),
                           Id_Str    => LexTokenManager.Interface_Token);
                     when CommandLineData.SPARK95 | CommandLineData.SPARK2005 =>
                        ErrorHandler.Semantic_Error
                          (Err_Num   => 70,
                           Reference => 18,
                           Position  => Node_Position (Pragma_Node),
                           Id_Str    => LexTokenManager.Import_Token);
                  end case;
               end if; -- wrong pragma

               -- If we are here then we have either:
               --               a legal declaration using a correct interfacing pragma; or
               --               a declaration with the wrong pragma that we have reported; or
               --               a legal "is separate"
               -- In each case we can go on to check formal parts and annotations

               if Syntax_Node_Type (Node => Formal_Node) = SPSymbols.formal_part then
                  Wf_Formal_Part
                    (Node             => Formal_Node,
                     Current_Scope    => Scope_Local,
                     Subprog_Sym      => Subprog_Sym,
                     First_Occurrence => First_Seen,
                     Context          => Dictionary.ProgramContext);
               elsif Dictionary.GetNumberOfSubprogramParameters (Subprog_Sym) /= 0 then
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 152,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node),
                     Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
               end if; -- formal part to check

               --# assert True;
               if (First_Seen or else Requires_Second_Annotation (Subprog_Sym => Subprog_Sym))
                 and then Syntax_Node_Type (Node => Anno_Node) = SPSymbols.procedure_annotation then
                  -- ASSUME Anno_Node = procedure_annotation
                  Wf_Procedure_Annotation
                    (Node          => Anno_Node,
                     Current_Scope => Scope_Local,
                     Subprog_Sym   => Subprog_Sym,
                     First_Seen    => First_Seen);
               elsif not First_Seen
                 and then Requires_Second_Annotation (Subprog_Sym => Subprog_Sym)
                 and then Anno_Node = STree.NullNode then
                  -- ASSUME Anno_Node = NULL
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 87,
                     Reference => 16,
                     Position  => Node_Position (Spec_Node),
                     Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
                  Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsRefined, Subprog_Sym);
               elsif First_Seen
                 and then (CommandLineData.Content.Language_Profile = CommandLineData.SPARK83
                             or else CommandLineData.Content.Do_Information_Flow)
                 and then Anno_Node = STree.NullNode then
                  -- ASSUME Anno_Node = NULL
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 154,
                     Reference => 19,
                     Position  => Node_Position (Spec_Node),
                     Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
                  Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract, Subprog_Sym);
               elsif Syntax_Node_Type (Node => Anno_Node) = SPSymbols.procedure_annotation then
                  -- ASSUME Anno_Node = procedure_annotation
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 155,
                     Reference => 16,
                     Position  => Node_Position (Spec_Node),
                     Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
               end if;

               --# assert True;
               if Interfacing_Pragma_Found then
                  --# accept F, 10, Unused, "Export not required here";
                  wf_external_interface (Pragma_Node => Pragma_Node,
                                         Entity_Sym  => Subprog_Sym,
                                         Error_Found => Unused);
                  --# end accept;
               end if;

               --# assert True;
               -- clause to create full derives from modes
               if CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83
                 and then not CommandLineData.Content.Do_Information_Flow then
                  if First_Seen then
                     CreateFullSubProgDependency (Node, Subprog_Sym, Dictionary.IsAbstract);
                  elsif Requires_Second_Annotation (Subprog_Sym => Subprog_Sym) then
                     CreateFullSubProgDependency (Node, Subprog_Sym, Dictionary.IsRefined);
                  end if;
               end if;

               --# assert True;
               if Syntax_Node_Type (Node => Child_Node (Current_Node => Constraint_Node)) = SPSymbols.precondition
                 or else Syntax_Node_Type (Node => Child_Node (Current_Node => Constraint_Node)) = SPSymbols.postcondition then
                  -- ASSUME Child_Node (Current_Node => Constraint_Node = precondition OR postcondition
                  -- a pre/post exists. Should it?
                  -- checks to see if constraint found is allowed
                  if not (First_Seen
                            or else Requires_Second_Annotation (Subprog_Sym => Subprog_Sym)
                            or else HasParameterGlobalOrReturnOfLocalPrivateType (Subprog_Sym)) then
                     -- annotation not required
                     -- two possible errors: misplaced anno or duplicate anno
                     if Dictionary.HasPrecondition (Dictionary.IsAbstract, Subprog_Sym)
                       or else Dictionary.HasPostcondition (Dictionary.IsAbstract, Subprog_Sym) then -- illegal duplicate anno
                        ErrorHandler.Semantic_Error
                          (Err_Num   => 343,
                           Reference => ErrorHandler.No_Reference,
                           Position  => Node_Position (Constraint_Node),
                           Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
                     else -- misplaced anno
                        ErrorHandler.Semantic_Error
                          (Err_Num   => 342,
                           Reference => ErrorHandler.No_Reference,
                           Position  => Node_Position (Constraint_Node),
                           Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
                     end if;
                  else -- annotation is required so continue
                     wf_procedure_constraint (Constraint_Node, Dictionary.LocalScope (Subprog_Sym), First_Seen);
                  end if;
               elsif Child_Node (Current_Node => Constraint_Node) /= STree.NullNode then
                  SystemErrors.Fatal_Error
                    (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                     Msg     => "Expect Child_Node (Current_Node => Constraint_Node = precondition OR postcondition OR NULL in Wf_Body_Stub")
                    ;
               end if;
            end if; -- not an illegal redclaration
         end if; -- subprogsym not null

         -- Prior to SPARK 2005, the check is only required when the procedure
         -- has not been previously declared. For SPARK 2005, the check is
         -- always required as the overriding_indicator for a
         -- subprogram body stub may be incorrect.
         if First_Seen or else CommandLineData.Content.Language_Profile = CommandLineData.SPARK2005 then
            CheckNoOverloadingFromTaggedOps (Spec_Node, Subprog_Sym, Scope, Dictionary.IsRefined, Is_Overriding);
         end if;
      end if; -- don't add stub if position illegal

   elsif Node_Type = SPSymbols.function_specification
     or else (Node_Type = SPSymbols.overriding_indicator
                and then Syntax_Node_Type (Node => Next_Sibling (Current_Node => Child_Node (Current_Node => Node))) =
                SPSymbols.function_specification) then
      -- ASSUME Node_Type = function_specification OR overriding_indicator
      if Valid_Stub_Position then
         Scope_Local   := Scope;
         Is_Overriding := False;
         if Node_Type = SPSymbols.overriding_indicator then
            if Syntax_Node_Type (Node => Last_Child_Of (Start_Node => Node)) = SPSymbols.RWoverriding then
               Is_Overriding := True;
            end if;
            Spec_Node := Next_Sibling (Current_Node => Child_Node (Current_Node => Node));
         else
            Spec_Node := Child_Node (Current_Node => Node);
         end if;
         -- ASSUME Spec_Node = function_specification
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Spec_Node) = SPSymbols.function_specification,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Spec_Node = function_specification in Wf_Body_Stub");
         Get_Anno_And_Con_Nodes
           (Node            => Next_Sibling (Current_Node => Spec_Node),
            Anno_Node       => Anno_Node,
            Constraint_Node => Constraint_Node);
         -- ASSUME Anno_Node = function_annotation OR NULL
         SystemErrors.RT_Assert
           (C       => Anno_Node = STree.NullNode or else Syntax_Node_Type (Node => Anno_Node) = SPSymbols.function_annotation,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Anno_Node = function_annotation OR NULL in Wf_Body_Stub");
         -- ASSUME Constraint_Node = function_constraint
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.function_constraint,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Constraint_Node = function_constraint in Wf_Body_Stub");
         --# assert True;
         Formal_Node := Next_Sibling (Current_Node => Child_Node (Current_Node => Spec_Node));
         -- ASSUME Formal_Node = formal_part OR type_mark
         if Syntax_Node_Type (Node => Formal_Node) = SPSymbols.type_mark then
            Formal_Node := STree.NullNode;
         elsif Syntax_Node_Type (Node => Formal_Node) /= SPSymbols.formal_part then
            Formal_Node := STree.NullNode;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Formal_Node = formal_part OR type_mark in Wf_Body_Stub");
         end if;
         -- ASSUME Formal_Node = formal_part OR NULL
         SystemErrors.RT_Assert
           (C       => Formal_Node = STree.NullNode or else Syntax_Node_Type (Node => Formal_Node) = SPSymbols.formal_part,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Formal_Node = formal_part OR NULL in Wf_Body_Stub");
         Wf_Function_Specification
           (Node        => Spec_Node,
            Hidden      => False,
            Scope       => Scope_Local,
            Subprog_Sym => Subprog_Sym,
            First_Seen  => First_Seen);
         --# assert True;
         if Subprog_Sym /= Dictionary.NullSymbol then
            if Syntax_Node_Type (Node => Pragma_Node) = SPSymbols.apragma and then not First_Seen then -- illegal redeclaration
               ErrorHandler.A_Pragma
                 (Pragma_Name => Node_Lex_String (Child_Node (Current_Node => Pragma_Node)),
                  Position    => Node_Position (Pragma_Node));
               ErrorHandler.Semantic_Error
                 (Err_Num   => 10,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node),
                  Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
            else -- this else means we don't process illegal redeclarations further
               if First_Seen and Other_Pragma_Found then -- only an interfacing pragma will do
                  case CommandLineData.Content.Language_Profile is
                     when CommandLineData.SPARK83 =>
                        ErrorHandler.Semantic_Error
                          (Err_Num   => 70,
                           Reference => 18,
                           Position  => Node_Position (Pragma_Node),
                           Id_Str    => LexTokenManager.Interface_Token);
                     when CommandLineData.SPARK95 | CommandLineData.SPARK2005 =>
                        ErrorHandler.Semantic_Error
                          (Err_Num   => 70,
                           Reference => 18,
                           Position  => Node_Position (Pragma_Node),
                           Id_Str    => LexTokenManager.Import_Token);
                  end case;
               end if; -- wrong pragma

               -- If we are here then we have either:
               --               a legal declaration using a correct interfacing pragma; or
               --               a declaration with the wrong pragma that we have reported; or
               --               a legal "is separate"
               -- In each case we can go on to check formal parts and annotations

               if Syntax_Node_Type (Node => Formal_Node) = SPSymbols.formal_part then
                  Wf_Formal_Part
                    (Node             => Formal_Node,
                     Current_Scope    => Scope_Local,
                     Subprog_Sym      => Subprog_Sym,
                     First_Occurrence => First_Seen,
                     Context          => Dictionary.ProgramContext);
               elsif Dictionary.GetNumberOfSubprogramParameters (Subprog_Sym) /= 0 then
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 152,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node),
                     Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
               end if; -- formal part to check

               --# assert True;
               if (First_Seen or else Requires_Second_Annotation (Subprog_Sym => Subprog_Sym))
                 and then Syntax_Node_Type (Node => Anno_Node) = SPSymbols.function_annotation then
                  -- ASSUME Anno_Node = function_annotation
                  Wf_Function_Annotation
                    (Node          => Anno_Node,
                     Current_Scope => Scope_Local,
                     Subprog_Sym   => Subprog_Sym,
                     First_Seen    => First_Seen);
               elsif not First_Seen
                 and then Requires_Second_Annotation (Subprog_Sym => Subprog_Sym)
                 and then Anno_Node = STree.NullNode then
                  -- ASSUME Anno_Node = NULL
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 87,
                     Reference => 16,
                     Position  => Node_Position (Spec_Node),
                     Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
               elsif Syntax_Node_Type (Node => Anno_Node) = SPSymbols.function_annotation then
                  -- ASSUME Anno_Node = function_annotation
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 336,
                     Reference => 16,
                     Position  => Node_Position (Spec_Node),
                     Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
               end if;

               --# assert True;
               if Interfacing_Pragma_Found then
                  --# accept F, 10, Unused, "Export not required here";
                  wf_external_interface (Pragma_Node => Pragma_Node,
                                         Entity_Sym  => Subprog_Sym,
                                         Error_Found => Unused);
                  --# end accept;
               end if;

               --# assert True;
               if Syntax_Node_Type (Node => Child_Node (Current_Node => Constraint_Node)) = SPSymbols.precondition
                 or else Syntax_Node_Type (Node => Child_Node (Current_Node => Constraint_Node)) = SPSymbols.return_expression then
                  -- a pre/return exists. Should it?
                  -- checks to see if constraint found is allowed
                  if not (First_Seen
                            or else Requires_Second_Annotation (Subprog_Sym => Subprog_Sym)
                            or else HasParameterGlobalOrReturnOfLocalPrivateType (Subprog_Sym)) then
                     --annotation not required
                     --two possible errors: misplaced anno or duplicate anno
                     if Dictionary.HasPrecondition (Dictionary.IsAbstract, Subprog_Sym)
                       or else Dictionary.HasPostcondition (Dictionary.IsAbstract, Subprog_Sym) then -- illegal duplicate anno
                        ErrorHandler.Semantic_Error
                          (Err_Num   => 343,
                           Reference => ErrorHandler.No_Reference,
                           Position  => Node_Position (Constraint_Node),
                           Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
                     else -- misplaced anno
                        ErrorHandler.Semantic_Error
                          (Err_Num   => 342,
                           Reference => ErrorHandler.No_Reference,
                           Position  => Node_Position (Constraint_Node),
                           Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
                     end if;
                  else -- annotation is required so continue
                     wf_function_constraint (Constraint_Node, Dictionary.LocalScope (Subprog_Sym), First_Seen);
                  end if;
               elsif Child_Node (Current_Node => Constraint_Node) /= STree.NullNode then
                  SystemErrors.Fatal_Error
                    (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                     Msg     => "Expect Child_Node (Current_Node => Constraint_Node = precondition OR return_expression OR NULL in Wf_Body_Stub");
               end if;
            end if;  -- not an illegal redeclaration
         end if; -- not null symbol for function

         -- Prior to SPARK 2005, the check is only required when the procedure
         -- has not been previously declared. For SPARK 2005, the check is
         -- always required as the overriding_indicator for a
         -- subprogram body stub may be incorrect.
         if First_Seen or else CommandLineData.Content.Language_Profile = CommandLineData.SPARK2005 then
            CheckNoOverloadingFromTaggedOps (Spec_Node, Subprog_Sym, Scope, Dictionary.IsRefined, Is_Overriding);
         end if;
      end if; -- don't process at all if stub position illegal

   elsif Node_Type = SPSymbols.task_stub then
      -- ASSUME Node_Type = task_stub
      if CommandLineData.Ravenscar_Selected then
         if Valid_Stub_Position then
            Ident_Node := Child_Node (Current_Node => Child_Node (Current_Node => Node));
            -- ASSUME Ident_Node = identifier
            SystemErrors.RT_Assert
              (C       => Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier,
               Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Ident_Node = identifier in Wf_Body_Stub");
            Ident_Str := Node_Lex_String (Ident_Node);
            Task_Sym  :=
              Dictionary.LookupItem (Name              => Node_Lex_String (Ident_Node),
                                     Scope             => Scope,
                                     Context           => Dictionary.ProgramContext,
                                     Full_Package_Name => False);
            if Task_Sym = Dictionary.NullSymbol or else not Dictionary.IsTaskType (Task_Sym) then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 898,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Ident_Node),
                  Id_Str    => Node_Lex_String (Ident_Node));
            elsif Dictionary.HasBody (Task_Sym) or else Dictionary.HasBodyStub (Task_Sym) then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 899,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Ident_Node),
                  Id_Str    => Node_Lex_String (Ident_Node));
            else
               -- valid so far
               Dictionary.AddBodyStub
                 (CompilationUnit => Task_Sym,
                  Comp_Unit       => ContextManager.Ops.Current_Unit,
                  BodyStub        => Dictionary.Location'(Start_Position => Node_Position (Node),
                                                          End_Position   => Node_Position (Node)));
               -- check annotation
               Anno_Node := Next_Sibling (Current_Node => Ident_Node);
               -- ASSUME Anno_Node = procedure_annotation
               SystemErrors.RT_Assert
                 (C       => Syntax_Node_Type (Node => Anno_Node) = SPSymbols.procedure_annotation,
                  Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                  Msg     => "Expect Anno_Node = procedure_annotation in Wf_Body_Stub");
               if Requires_Second_Annotation (Subprog_Sym => Task_Sym) then
                  if Empty_Annotation (Node => Anno_Node) then
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 154,
                        Reference => ErrorHandler.No_Reference,
                        Position  => Node_Position (Node),
                        Id_Str    => Ident_Str);
                  else -- anno present and required
                     STree.Set_Node_Lex_String (Sym  => Task_Sym,
                                                Node => Ident_Node);
                     Process_Annotation
                       (Anno_Node        => Anno_Node,
                        Scope            => Scope,
                        Task_Sym         => Task_Sym,
                        Valid_Annotation => Valid_Annotation);
                  end if;
               else -- second anno not required
                  if Empty_Annotation (Node => Anno_Node) then
                     STree.Set_Node_Lex_String (Sym  => Task_Sym,
                                                Node => Ident_Node);
                  else
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 155,
                        Reference => ErrorHandler.No_Reference,
                        Position  => Node_Position (Anno_Node),
                        Id_Str    => Ident_Str);
                  end if;
               end if;

               -- Create "full" derives if annotation is present, valid and DFA is selected.
               -- We know we are in SPARK 95.
               if Valid_Annotation and then not CommandLineData.Content.Do_Information_Flow then
                  CreateFullSubProgDependency (Node, Task_Sym, Dictionary.IsRefined);
               end if;
            end if;
         end if; -- don't process stub if illegally positioned
      else -- illegal
         ErrorHandler.Semantic_Error
           (Err_Num   => 850,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node),
            Id_Str    => LexTokenManager.Null_String);
      end if;

   elsif Node_Type = SPSymbols.protected_stub then
      -- ASSUME Node_Type = protected_stub
      if CommandLineData.Ravenscar_Selected then
         if Valid_Stub_Position then
            Ident_Node := Child_Node (Current_Node => Child_Node (Current_Node => Node));
            -- ASSUME Ident_Node = identifier
            SystemErrors.RT_Assert
              (C       => Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier,
               Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Ident_Node = identifier in Wf_Body_Stub");
            Protected_Sym :=
              Dictionary.LookupItem (Name              => Node_Lex_String (Ident_Node),
                                     Scope             => Scope,
                                     Context           => Dictionary.ProgramContext,
                                     Full_Package_Name => False);
            if Protected_Sym = Dictionary.NullSymbol
              or else not (Dictionary.IsType (Protected_Sym) and then Dictionary.TypeIsProtected (Protected_Sym)) then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 898,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Ident_Node),
                  Id_Str    => Node_Lex_String (Ident_Node));
            elsif Dictionary.HasBody (Protected_Sym) or else Dictionary.HasBodyStub (Protected_Sym) then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 899,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Ident_Node),
                  Id_Str    => Node_Lex_String (Ident_Node));
            else
               STree.Set_Node_Lex_String (Sym  => Protected_Sym,
                                          Node => Ident_Node);
               Dictionary.AddBodyStub
                 (CompilationUnit => Protected_Sym,
                  Comp_Unit       => ContextManager.Ops.Current_Unit,
                  BodyStub        => Dictionary.Location'(Start_Position => Node_Position (Node),
                                                          End_Position   => Node_Position (Node)));
            end if;
         end if; -- don't process stub if illegally positioned
      else -- illegal
         ErrorHandler.Semantic_Error
           (Err_Num   => 850,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node),
            Id_Str    => LexTokenManager.Null_String);
      end if;
   else
      SystemErrors.Fatal_Error
        (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node_Type = overriding_indicator OR procedure_specification OR function_specification OR dotted_simple_name OR task_stub OR protected_stub in Wf_Body_Stub");
   end if;

   if ErrorHandler.Generate_SLI then
      SLI.Increment_Nb_Separates (Comp_Unit  => ContextManager.Ops.Current_Unit,
                                  Parse_Tree => Node);
   end if;
   --# accept F, 33, Unused, "Export not required here";
end Wf_Body_Stub;
