-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure up_wf_named_record_component_association
  (Node      : in     STree.SyntaxNode;
   Scope     : in     Dictionary.Scopes;
   EStack    : in out ExpStack.ExpStackType;
   HeapParam : in out Lists.List_Heap) is
   NameExp, FieldName, ExpResult : Exp_Record;
   ExpectedType                  : Dictionary.Symbol;
   ErrorFound                    : Boolean := False;

   function ExpressionLocation return  STree.SyntaxNode
   --# global in Node;
   --#        in STree.Table;
   --  pre Syntax_Node_Type (Node => Node) = SPSymbols.named_record_component_association or
   --      Syntax_Node_Type (Node => Node) = SPSymbols.annotation_named_record_component_association;
   is
      LocalNode : STree.SyntaxNode;
   begin
      LocalNode := Child_Node (Node);
      if Syntax_Node_Type (Node => LocalNode) /= SPSymbols.record_component_selector_name then
         LocalNode := Next_Sibling (LocalNode);
      end if;
      return Next_Sibling (LocalNode);
   end ExpressionLocation;

   --------------------------------------------

   procedure CheckRecordCompleteness (NameExp : in out Exp_Record)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     Node;
   --#        in     STree.Table;
   --#        in out ErrorFound;
   --#        in out ErrorHandler.Error_Context;
   --#        in out HeapParam;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorFound                 from *,
   --#                                         Dictionary.Dict,
   --#                                         HeapParam,
   --#                                         LexTokenManager.State,
   --#                                         NameExp &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         HeapParam,
   --#                                         LexTokenManager.State,
   --#                                         NameExp,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         HeapParam                  from *,
   --#                                         LexTokenManager.State,
   --#                                         NameExp &
   --#         NameExp                    from *;
   is
      FieldStr : LexTokenManager.Lex_String;
      ErrorPos : LexTokenManager.Token_Position;
      Ptr      : Lists.List;
   begin
      ErrorPos := Node_Position (Node => ExpressionLocation);
      for I in Positive range
        Dictionary.GetNumberOfComponents (NameExp.Other_Symbol) + 1 ..    -- ancestor field count
        Dictionary.GetNumberOfComponents (NameExp.Type_Symbol) loop       -- total field count

         FieldStr := Dictionary.GetSimpleName (Dictionary.GetRecordComponent (NameExp.Type_Symbol, I));
         if not Lists.Is_Member (Heap     => HeapParam,
                                 The_List => NameExp.Param_List,
                                 Str      => FieldStr) then
            ErrorFound := True;
            ErrorHandler.Semantic_Error
              (Err_Num   => 104,
               Reference => ErrorHandler.No_Reference,
               Position  => ErrorPos,
               Id_Str    => FieldStr);
         end if;
      end loop;
      Ptr := NameExp.Param_List;
      DisposeOfNameList (Ptr, HeapParam);
      NameExp.Param_List := Ptr;
   end CheckRecordCompleteness;

   -------------------------------------------------------------

begin
   -- TOS is the result of walking an expression to be associated with a record field name
   -- 2nd TOS is the field name in a parameter record
   -- 3rd TOS is the aggregate type with the ancestor type in its OtherSymbol field

   ExpStack.Pop (ExpResult, EStack);
   ExpStack.Pop (FieldName, EStack);
   ExpStack.Pop (NameExp, EStack);

   if FieldName.Other_Symbol = Dictionary.NullSymbol then
      null;

   else
      ExpectedType := Dictionary.GetType (FieldName.Other_Symbol);
      STree.AddNodeSymbol (Node, ExpectedType);
      AssignmentCheck (Node_Position (Node => ExpressionLocation), Scope, ExpectedType, ExpResult);
      NameExp.Is_Constant := NameExp.Is_Constant and ExpResult.Is_Constant;

      if Next_Sibling (Node) = STree.NullNode then
         --this is the last named association so we need to check that
         --all fields have been given a value
         CheckRecordCompleteness (NameExp);
      end if;
   end if;
   NameExp.Errors_In_Expression := ErrorFound or NameExp.Errors_In_Expression or ExpResult.Errors_In_Expression;
   ExpStack.Push (NameExp, EStack);
end up_wf_named_record_component_association;
